;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; GNU GENERAL PUBLIC LICENSE ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;  The Musical Chairs model - A model of the interaction between farming and herding
;;  Copyright (C) 2013 Andreas Angourakis (andros.spica@gmail.com)
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;;;;;;;;;;;;;;;;
;;; VARIABLES ;;;
;;;;;;;;;;;;;;;;;

globals
[  
  ;;; modified parameters
  intGrowthF intGrowthH extGrowthF extGrowthH 
  initH initF
  h_r_m_i intgH intgF
  
  ;;; sets of agents that exists at the end of each run
  settledFarmingAgents stableHerdingAgents
  
  ;;; unlucky agents, their respective number of helpers and their intensity
  unluckyF farmingSupport farmingIntensity
  unluckyH herdingSupport herdingIntensity
  
  ;;; variables used in resolve_conflict
  index_of_opportunity ratio_of_intensities 
  incentives_to_relinquish
  
  ;;; counters and final measures
  AREA
  dilemmaEvents oasisDegressionEvents herdingSuccessRatio
  farmingGrowth farmingDeterrence herdingGrowth herdingDeterrence farmingBalance herdingBalance
  farming_histo_intensity_0_025 farming_histo_intensity_025_05 farming_histo_intensity_05_075 farming_histo_intensity_075_1
  farming_histo_independence_0_025 farming_histo_independence_025_05 farming_histo_independence_05_075 farming_histo_independence_075_1
  herding_histo_intensity_0_025 herding_histo_intensity_025_05 herding_histo_intensity_05_075 herding_histo_intensity_075_1
  herding_histo_independence_0_025 herding_histo_independence_025_05 herding_histo_independence_05_075 herding_histo_independence_075_1
  mean_fint mean_find mean_hint mean_hind
]

;;; establishes the classes of agents and their variables

breed [ farming_agents farming_agent ]

breed [ herding_agents herder_agent ]

farming_agents-own [ intensity independence ]

herding_agents-own [ intensity independence ]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SETUP ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  
  clear-all
  
  ;;; setup parameters depending on the type of experiment
  ifelse (random_exp? = true) 
  [
    set intGrowthF random-float farming_intrinsic_growth_rate
    set extGrowthF random-float farming_extrinsic_growth_rate
    set intgF random-float farming_integration
    set intGrowthH random-float herding_intrinsic_growth_rate
    set extGrowthH random-float herding_extrinsic_growth_rate
    set intgH random-float herding_integration
    set initH random init_herding
    set initF random init_farming
    set h_r_m_i ((1 + (random-float 10) ) / (1 + (random-float 10) )) ;; set a random value between 0.1 and 10 (around 1)
  ]
  [ 
    set intGrowthF farming_intrinsic_growth_rate
    set extGrowthF farming_extrinsic_growth_rate
    set intgF farming_integration
    set intGrowthH herding_intrinsic_growth_rate
    set extGrowthH herding_extrinsic_growth_rate
    set intgH herding_integration
    set initH init_herding
    set initF init_farming
    set h_r_m_i herding_relative_max_intensity
  ]

  ;;; create agents according to the parameter setting (position is arbitrary and has no consequence)
  ask patch 0 0
  [
    sprout-herding_agents initH [ initialize-an-agent ]
  ]
  ask patch max-pxcor 0
  [ 
    sprout-farming_agents initF [ initialize-an-agent ]
  ]
  
  ;;; initialize visualization
  ask patches [ set pcolor yellow ]
  
  update_visualization
  
  set dilemmaEvents 0 ;reset dilemma_events counter
  set oasisDegressionEvents 0 ;reset oasis_degression_events counter
  
  reset-ticks
    
end

to initialize-an-agent
  
  set hidden? true
  ifelse (breed = farming_agents)
  [ set intensity random-float farming_max_intensity ]
  [ set intensity random-float (farming_max_intensity * h_r_m_i) ]
  set independence (random-float 1)
  
end

to go
  
  farming_expansion
  
  herding_expansion
  
  ;reset dilemma_events and oasis_degression_events counters
  set dilemmaEvents 0 
  set oasisDegressionEvents 0
  
  check_competition
  
  update_visualization
  
  tick
  
end

to farming_expansion
  
    ;;; set of farming_agents currently present
  set settledFarmingAgents turtle-set farming_agents
  
  ;;; reset counters
  set farmingGrowth 0
  set farmingDeterrence 0
  
  ;;; Intrinsic growth
  ask settledFarmingAgents
  [ if ( random-float 1 <= intGrowthF )
    [ 
      hatch 1
      set farmingGrowth farmingGrowth + 1
    ]
  ]
  
  ;;; Extrinsic growth
  ask patch max-pxcor 0
  [ sprout-farming_agents (round (extGrowthF * ( (count patches) - (count settledFarmingAgents) ) ) )
    [
      initialize-an-agent
      set farmingGrowth farmingGrowth + 1
    ]
  ]
  let notSettledFarmingAgents farming_agents with [not member? self settledFarmingAgents]
  
  ;;; Fit-to-maximum exclusion
  ask notSettledFarmingAgents
  [
    ;new farming_agents exit the territory in a random order whenever there is no more land to use
    if (count farming_agents > count patches) [ set farmingDeterrence farmingDeterrence + 1 die ]
  ]
  
  ;;; Density-dependent exclusion
  ask notSettledFarmingAgents
  [ 
    ;new farming_agents exit the territory with a probability proportional to the density of land currently used for agriculture (i.e. proxy of the quality of the remaining land)
    if (random-float 1 < ((count farming_agents) / (count patches) ) ) [ set farmingDeterrence farmingDeterrence + 1 die ]
  ]
  
  ;;; Volition-opportunity exclusion
  ask notSettledFarmingAgents
  [
    ;;; if the territory is saturated...
    if (count patches - count herding_agents < count farming_agents)
    [
      initialize-an-agent
      ;;; if its not possible to extend farming over herding of another group 
      if ( ( (count herding_agents) / (count patches) ) > independence)
      [ set farmingDeterrence farmingDeterrence + 1 die ]
    ]
  ]
   
end

to herding_expansion
  
  ;;; set of herding_agents currently present
  set stableHerdingAgents turtle-set herding_agents
  
  ;;; reset counters
  set herdingGrowth 0
  set herdingDeterrence 0
  
  ;;; Intrinsic growth
  ask stableHerdingAgents
  [ if ( random-float 1 <= intGrowthH )
    [ 
      hatch 1
      set herdingGrowth herdingGrowth + 1
    ]
  ]
  
  ;;; Extrinsic growth
  ask patch 0 0
  [ sprout-herding_agents (round (extGrowthH * ( (count patches) - (count stableHerdingAgents) ) ) )
    [ 
      initialize-an-agent
      set herdingGrowth herdingGrowth + 1
    ]
  ]
  
  let notStableHerdingAgents herding_agents with [not member? self stableHerdingAgents]
  
  ;;; Fit-to-maximum exclusion
  ask notStableHerdingAgents
  [ 
    ;new herding_agents exit the territory in a random order whenever there is no more land to use
    if (count herding_agents > count patches) [ set herdingDeterrence herdingDeterrence + 1 die ]
  ]
  
  ;;; Density-dependent exclusion
  ask notStableHerdingAgents
  [ 
    ;new herding_agents exit the territory with a probability proportional to the density of pastures currently in use (i.e. proxy of the quality of the remaining land)
    if (random-float 1 < ( (count herding_agents) / (count patches) ) ) [ set herdingDeterrence herdingDeterrence + 1 die ]
  ] 
  
end

to check_competition
  
  if ((count patches) - (count farming_agents) < count herding_agents) [ resolve_competition ]

end

to resolve_competition

  ;;; set competition conditions
  
  ;;;;; select one farming agent and its supporters, calculate the intensity of the farming land use involved in a land use unit
  set unluckyF one-of farming_agents
  set farmingSupport round (intgF * ((count farming_agents) - 1))
  let p 0
  if (farmingSupport > 0) [ set p sum [intensity] of n-of farmingSupport farming_agents with [self != unluckyF] ]
  set farmingIntensity ([intensity] of unluckyF + p )
  
  ;;;;; select one herding agent and its supporters, calculate the intensity of the herding land use to be involved in the same land use unit
  set unluckyH one-of herding_agents
  set herdingSupport round (intgH * ((count herding_agents) - 1))
  let q 0
  if (herdingSupport > 0) [ set q sum [intensity] of n-of herdingSupport herding_agents with [self != unluckyH] ]
  set herdingIntensity ([intensity] of unluckyH + q )

  ;;;;; calculate the ratio of intensities, the index of opportunity and the incentives for relinquish, all taken from the perspective of herding land use
  set ratio_of_intensities  (herdingIntensity /(farmingIntensity + herdingIntensity))
  set index_of_opportunity ((count farming_agents) / (count patches))
  set incentives_to_relinquish (1 - (ratio_of_intensities * index_of_opportunity))
  
  ;;; resolve the competitive situation
  ask unluckyH
  [
    ;;; Does the competitive situation evolves into a dilemma event?
    ifelse ( independence < incentives_to_relinquish)
    
    ;;; No. The herding agent exit the territory.
    [ set herdingDeterrence herdingDeterrence + 1 die ]
    
    ;;; Yes. A dilemma event is produced for there are two variants to be realized in a single land use unit.
    [ 
      set dilemmaEvents (dilemmaEvents + 1)
      
      ;;; Does the dilemma event evolves into a oasis degression event?
      ifelse (random-float 1 < ratio_of_intensities)
      
      ;;; Yes. The unlucky farming agent exits the territory.
      [
        ask unluckyF [ set farmingDeterrence farmingDeterrence + 1 die ]
        set oasisDegressionEvents (oasisDegressionEvents + 1)
      ]
      
      ;;; No. The unlucky herding agent exits the territory.
      [ set herdingDeterrence herdingDeterrence + 1 die ]
    ]    
  ]
  
  ;;; re-check the presence of competitive situations
  check_competition
  
end

to update_visualization
  
  update_patches
  
  set AREA count patches
  
  set farmingBalance (farmingGrowth - farmingDeterrence)
  set herdingBalance (herdingGrowth - herdingDeterrence)
  
  ifelse (dilemmaEvents > 0) [ set herdingSuccessRatio (oasisDegressionEvents / dilemmaEvents) ] [ set herdingSuccessRatio 0 ]

  set farming_histo_intensity_0_025 (count farming_agents with [intensity <= ( 0.25 * farming_max_intensity)])
  set farming_histo_intensity_025_05 (count farming_agents with [intensity <= ( 0.5 * farming_max_intensity)])
  set farming_histo_intensity_025_05 (farming_histo_intensity_025_05 - farming_histo_intensity_0_025)
  set farming_histo_intensity_05_075 (count farming_agents with [intensity <= ( 0.75 * farming_max_intensity)])
  set farming_histo_intensity_05_075 (farming_histo_intensity_05_075 - farming_histo_intensity_0_025 - farming_histo_intensity_025_05)
  set farming_histo_intensity_075_1 ((count farming_agents) - farming_histo_intensity_0_025 - farming_histo_intensity_025_05 - farming_histo_intensity_05_075)
  ifelse (count farming_agents > 0)
  [ set mean_fint ((farming_histo_intensity_0_025 * 1 / (count farming_agents)) + (farming_histo_intensity_025_05 * 2 / (count farming_agents)) + (farming_histo_intensity_05_075 * 3 / (count farming_agents)) + (farming_histo_intensity_075_1 * 4 / (count farming_agents))) ]
  [ set mean_fint 0 ]
  
  set farming_histo_independence_0_025 (count farming_agents with [independence <= 0.25])
  set farming_histo_independence_025_05 (count farming_agents with [independence <= 0.5])
  set farming_histo_independence_025_05 (farming_histo_independence_025_05 - farming_histo_independence_0_025)
  set farming_histo_independence_05_075 (count farming_agents with [independence <= 0.75])
  set farming_histo_independence_05_075 (farming_histo_independence_05_075 - farming_histo_independence_0_025 - farming_histo_independence_025_05)
  set farming_histo_independence_075_1 ((count farming_agents) - farming_histo_independence_0_025 - farming_histo_independence_025_05 - farming_histo_independence_05_075)  
  ifelse (count farming_agents > 0)
  [ set mean_find ((farming_histo_independence_0_025 * 1 / (count farming_agents)) + (farming_histo_independence_025_05 * 2 / (count farming_agents)) + (farming_histo_independence_05_075 * 3 / (count farming_agents)) + (farming_histo_independence_075_1 * 4 / (count farming_agents))) ] 
  [ set mean_find 0 ]
  
  set herding_histo_intensity_0_025 (count herding_agents with [intensity <= (0.25 * (h_r_m_i * farming_max_intensity))])
  set herding_histo_intensity_025_05 (count herding_agents with [intensity <= (0.5 * (h_r_m_i * farming_max_intensity))])
  set herding_histo_intensity_025_05 (herding_histo_intensity_025_05 - herding_histo_intensity_0_025)
  set herding_histo_intensity_05_075 (count herding_agents with [intensity <= (0.75 * (h_r_m_i * farming_max_intensity))])
  set herding_histo_intensity_05_075 (herding_histo_intensity_05_075 - herding_histo_intensity_0_025 - herding_histo_intensity_025_05)
  set herding_histo_intensity_075_1 ((count herding_agents) - herding_histo_intensity_0_025 - herding_histo_intensity_025_05 - herding_histo_intensity_05_075)
  ifelse (count herding_agents > 0)
  [ set mean_hint ((herding_histo_intensity_0_025 * 1 / (count herding_agents)) + (herding_histo_intensity_025_05 * 2 / (count herding_agents)) + (herding_histo_intensity_05_075 * 3 / (count herding_agents)) + (herding_histo_intensity_075_1 * 4 / (count herding_agents))) ]
  [ set mean_hint 0 ]
  
  set herding_histo_independence_0_025 (count herding_agents with [independence <= 0.25])
  set herding_histo_independence_025_05 (count herding_agents with [independence <= 0.5])
  set herding_histo_independence_025_05 (herding_histo_independence_025_05 - herding_histo_independence_0_025)
  set herding_histo_independence_05_075 (count herding_agents with [independence <= 0.75])
  set herding_histo_independence_05_075 (herding_histo_independence_05_075 - herding_histo_independence_0_025 - herding_histo_independence_025_05)
  set herding_histo_independence_075_1 ((count herding_agents) - herding_histo_independence_0_025 - herding_histo_independence_025_05 - herding_histo_independence_05_075)
  ifelse (count herding_agents > 0)
  [ set mean_hind ((herding_histo_independence_0_025 * 1 / (count herding_agents)) + (herding_histo_independence_025_05 * 2 / (count herding_agents)) + (herding_histo_independence_05_075 * 3 / (count herding_agents)) + (herding_histo_independence_075_1 * 4 / (count herding_agents))) ]
  [ set mean_hind 0 ]
  
end

to update_patches
  
  ifelse ( count farming_agents > (count patches with [pcolor = green]) )
  [
    let to-paint (count farming_agents - (count patches with [pcolor = green]))
    repeat to-paint
    [
      ifelse (count patches with [pcolor = green] < (max-pycor + 1) )
      [ ask one-of patches with [pxcor = max-pxcor] [ set pcolor green] ]
      [ ask one-of patches with [(pcolor = yellow) and (count neighbors with [pcolor = green] > 2)] [ set pcolor green ] ]
    ]
  ] 
  [
    let to-paint ((count patches with [pcolor = green]) - count farming_agents)
    repeat to-paint
    [
      ifelse (any? patches with [(pcolor = green) and (count neighbors with [pcolor = yellow] > 2)])
      [
        ask one-of patches with [(pcolor = green) and (count neighbors with [pcolor = yellow] > 2)]
        [ set pcolor yellow ]
      ]
      [ ask one-of patches with [pxcor = min-pxcor] [set pcolor yellow] ]
    ]
  ]    
  
end
@#$#@#$#@
GRAPHICS-WINDOW
750
10
1310
151
-1
-1
5.5
1
10
1
1
1
0
0
0
1
0
99
0
19
0
0
1
ticks
30.0

BUTTON
14
10
77
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
89
11
152
44
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
162
12
225
45
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
12
151
237
184
farming_intrinsic_growth_rate
farming_intrinsic_growth_rate
0
0.1
0.04
0.01
1
NIL
HORIZONTAL

SLIDER
12
334
237
367
herding_relative_max_intensity
herding_relative_max_intensity
0.1
10
1
0.1
1
NIL
HORIZONTAL

PLOT
288
10
745
160
land use
years
cells
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"herding" 1.0 0 -1184463 true "plot (count herding_agents)" "plot (count herding_agents)"
"farming" 1.0 0 -13840069 true "plot (count farming_agents)" "plot (count farming_agents)"

PLOT
1008
170
1237
290
herding independence
NIL
frequency
0.0
200.0
0.0
200.0
false
false
"set-plot-x-range 0 1" "set-plot-y-range -0.01 (count herding_agents)"
PENS
"independence" 1.0 1 -16777216 true "histogram [independence] of herding_agents\nset-histogram-num-bars 10" "histogram [independence] of herding_agents"

PLOT
288
161
745
355
events
years
events
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dilemma events" 1.0 0 -16777216 true "" "plot dilemmaEvents"
"oasis degression events" 1.0 0 -5825686 true "" "plot oasisDegressionEvents"

SLIDER
12
264
238
297
herding_extrinsic_growth_rate
herding_extrinsic_growth_rate
0
1
0
0.01
1
NIL
HORIZONTAL

SLIDER
12
386
237
419
farming_integration
farming_integration
0
1
0
0.01
1
NIL
HORIZONTAL

SLIDER
12
419
237
452
herding_integration
herding_integration
0
1
0
0.01
1
NIL
HORIZONTAL

SLIDER
12
184
237
217
farming_extrinsic_growth_rate
farming_extrinsic_growth_rate
0
1
0
0.01
1
NIL
HORIZONTAL

SLIDER
12
231
237
264
herding_intrinsic_growth_rate
herding_intrinsic_growth_rate
0
0.1
0.04
0.01
1
NIL
HORIZONTAL

PLOT
1008
293
1236
413
farming independence
NIL
frequency
0.0
10.0
0.0
10.0
true
false
"set-plot-x-range 0 1" "set-plot-y-range -0.01 (count farming_agents)"
PENS
"independence" 1.0 1 -16777216 true "histogram [independence] of farming_agents\nset-histogram-num-bars 10" "histogram [independence] of farming_agents"

SLIDER
12
301
238
334
farming_max_intensity
farming_max_intensity
0
100
100
1
1
NIL
HORIZONTAL

PLOT
762
171
922
291
herding intensity
NIL
frequency
0.0
10.0
0.0
10.0
false
false
"set-plot-x-range 0 (farming_max_intensity * h_r_m_i)\nset-histogram-num-bars 10" "set-plot-y-range -0.01 (count herding_agents)"
PENS
"default" 1.0 1 -2674135 true "histogram [intensity] of herding_agents" "histogram [intensity] of herding_agents"

PLOT
762
293
922
413
farming intensity
NIL
frequency
0.0
10.0
0.0
10.0
false
false
"set-plot-x-range 0 farming_max_intensity\nset-histogram-num-bars 10" "set-plot-y-range -0.01 (count farming_agents)"
PENS
"default" 1.0 1 -955883 true "histogram [intensity] of farming_agents" "histogram [intensity] of farming_agents"

PLOT
288
356
745
510
balance
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"farming agents" 1.0 0 -13791810 true "" "plot farmingBalance"
"herding agents" 1.0 0 -10873583 true "" "plot herdingBalance"
"0" 1.0 0 -16777216 true "" "plot 0"

INPUTBOX
13
84
121
144
init_farming
100
1
0
Number

INPUTBOX
118
84
227
144
init_herding
100
1
0
Number

SWITCH
56
51
187
84
random_exp?
random_exp?
1
1
-1000

MONITOR
237
147
304
184
NIL
intGrowthF
4
1
9

MONITOR
237
184
307
221
NIL
extGrowthF
4
1
9

MONITOR
237
227
306
264
NIL
intGrowthH
4
1
9

MONITOR
237
263
308
300
NIL
extGrowthH
4
1
9

MONITOR
237
332
287
369
NIL
h_r_m_i
4
1
9

MONITOR
237
381
287
418
NIL
intgF
4
1
9

MONITOR
237
418
287
455
NIL
intgH
4
1
9

MONITOR
688
105
745
142
patches
count patches
0
1
9

MONITOR
614
296
722
333
herding success rate
herdingSuccessRatio
4
1
9

MONITOR
749
434
841
471
NIL
farmingGrowth
2
1
9

MONITOR
845
434
946
471
NIL
farmingDeterrence
2
1
9

MONITOR
749
473
841
510
NIL
herdingGrowth
2
1
9

MONITOR
846
473
946
510
NIL
herdingDeterrence
2
1
9

MONITOR
688
68
738
105
farming
(count farming_agents)
0
1
9

MONITOR
926
206
1005
243
Index of selection
mean_hint
2
1
9

MONITOR
924
334
1003
371
Index of selection
mean_fint
2
1
9

MONITOR
1237
208
1319
245
Index of selection
mean_hind
2
1
9

MONITOR
1237
331
1320
368
Index of selection
mean_find
2
1
9

MONITOR
613
215
702
252
dilemma events
dilemmaEvents
0
1
9

MONITOR
613
256
738
293
oasis degression events
oasisDegressionEvents
0
1
9

MONITOR
939
434
1029
471
NIL
farmingBalance
2
1
9

MONITOR
939
473
1029
510
NIL
herdingBalance
2
1
9

@#$#@#$#@
## Information

This Agent-Based model intends to explore the conditions for the emergence and change of land use patterns in Central Asian oases and similar contexts. Land use pattern is conceptualized as the proportion between the area used for mobile livestock breeding (herding) and sedentary agriculture (farming), the main forms of livelihood from the Neolithic to the Industrial Revolution. We assume that these different forms of land use interact in recurrent competitive situations, given that the land useful for both activities at the same time is limited and there is a pressure to increase both land uses, due to demographic and/or economic growth.

Further information can be found in the ODD protocol and the following publication:

Angourakis et al. (2014) Land Use Patterns in Central Asia. Step 1: The Musical Chairs Model. Journal of Archaeological Method and Theory, Speacial Issue on Computer Simulation.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="ext0intg0_init100vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg0_init100vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg0_init200vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg0_init200vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg1_init100vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg1_init100vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg1_init200vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext0intg1_init200vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg0_init100vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg0_init100vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg0_init200vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg0_init200vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg1_init100vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg1_init100vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg1_init200vs100" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ext25intg1_init200vs200" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="randomized" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <metric>intGrowthF</metric>
    <metric>intGrowthH</metric>
    <metric>extGrowthF</metric>
    <metric>extGrowthH</metric>
    <metric>initH</metric>
    <metric>initF</metric>
    <metric>h_r_m_i</metric>
    <metric>intgH</metric>
    <metric>intgF</metric>
    <metric>AREA</metric>
    <metric>count farmers</metric>
    <metric>count herders</metric>
    <metric>dilemmaEvents</metric>
    <metric>oasisDegressionEvents</metric>
    <metric>herdingSuccessRatio</metric>
    <metric>farmingGrowth</metric>
    <metric>farmingDeterrence</metric>
    <metric>herdingGrowth</metric>
    <metric>herdingDeterrence</metric>
    <metric>farmingBalance</metric>
    <metric>herdingBalance</metric>
    <metric>farming_histo_intensity_0_025</metric>
    <metric>farming_histo_intensity_025_05</metric>
    <metric>farming_histo_intensity_05_075</metric>
    <metric>farming_histo_intensity_075_1</metric>
    <metric>farming_histo_independence_0_025</metric>
    <metric>farming_histo_independence_025_05</metric>
    <metric>farming_histo_independence_05_075</metric>
    <metric>farming_histo_independence_075_1</metric>
    <metric>herding_histo_intensity_0_025</metric>
    <metric>herding_histo_intensity_025_05</metric>
    <metric>herding_histo_intensity_05_075</metric>
    <metric>herding_histo_intensity_075_1</metric>
    <metric>herding_histo_independence_0_025</metric>
    <metric>herding_histo_independence_025_05</metric>
    <metric>herding_histo_independence_05_075</metric>
    <metric>herding_histo_independence_075_1</metric>
    <metric>mean_fint</metric>
    <metric>mean_find</metric>
    <metric>mean_hint</metric>
    <metric>mean_hind</metric>
    <enumeratedValueSet variable="farming_max_intensity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_relative_max_intensity">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_intrinsic_growth_rate">
      <value value="0.04"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_extrinsic_growth_rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_exp?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herding_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farming_integration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_herding">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init_farming">
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
