breed [ nodes node ]

nodes-own [ innate-leadership-score follower? leader? neutral? num-votes-received ]
links-own [ number-votes-given ]

globals [ total-num-votes total-num-leaders ]

to setup
  clear-all
  set-default-shape nodes "person"
  repeat num-nodes [
    ask patch random-xcor random-ycor [
      sprout-nodes 1 [
        
        if leadership-distribution = "Linear" [
          set innate-leadership-score random-float 10
        ]
        
        if leadership-distribution = "Normal" [
          set innate-leadership-score random-normal 5 2
        ]
        
        if leadership-distribution = "Poisson" [
          set innate-leadership-score random-poisson 5
        ]
                
        set follower? false
        set leader? false
        set neutral? true
        set num-votes-received 0
        set color gray - 1
      ]
    ]
  ]
  update-plots
end

to go
  if not any? nodes with [ neutral? = true ] [
    stop 
  ]

  ask nodes with [ not leader? ] [ 
    vote-for-leader
  ]
  ask nodes [
    transform
    move
  ]
  ask nodes [
    update-node-visuals
  ]
  ask links [
    update-link-visuals
  ]
  
  update-globals
  update-plots

  tick
end

to move
  ask nodes [
    ifelse neutral? [
      rt random 30
      lt random 30
      fd movement-rate / 1000
      ]
    [ rt random 30
      lt random 30
      fd group-movement-rate / 10000
      ]
  ]
end

to-report leadership-value
  report innate-leadership-score + count in-link-neighbors with [ follower? ] + floor (num-votes-received / 1000)
end

to vote-for-leader
  let my-candidate max-one-of (other nodes in-radius vision) [ leadership-value ]
  if my-candidate != NOBODY [
    if [ leadership-value ] of my-candidate > leadership-value [
      ask my-candidate [
        set num-votes-received num-votes-received + 1
      ]
      ifelse out-link-neighbor? my-candidate [
        ask out-link-to my-candidate [
          set number-votes-given number-votes-given + 1
        ]
      ]
      [ 
        ask my-out-links [
          die
        ]
        set follower? false
        set leader? false
        set neutral? true
        create-link-to my-candidate [ set number-votes-given 1 ]
      ]
    ]
  ]
end

to transform
  if num-votes-received > leader-threshold [
    set neutral? false
    set follower? false
    set leader? true
    
    ask my-out-links [
      die
    ]
    
  ;; this is sort of a confusing bit so I'll do some explaining
  ;; basically, the way the model chooses initial followers is through the properties
  ;; of the links (specifically links-own [ number-votes-given ]  
  ;; the following code examines the out-links from the leaders and finds the top followers
  
    let num-supporters count in-link-neighbors
    let num-followers ceiling (num-supporters / 5)
   
  ;; check if I'm becoming a leader for the first time 
  ;; if I don't have a lot of supporters, pick all my followers
  ;; if I do have followers, pick anyone with a high enough vote number
   
   ifelse not any? in-link-neighbors with [follower? = true] [
     ask max-n-of num-followers in-link-neighbors [ [ number-votes-given ] of out-link-to myself ] [
       set neutral? false
       set follower? true
       ]        
     ]
    [
     ask in-link-neighbors with [follower? = false] [
         let supporter-score [number-votes-given] of one-of my-out-links
         if supporter-score > follower-threshold [
           set neutral? false
           set follower? true
         ]
       ]
     ]    
  ]
end

to update-plots
  clear-plot
  ask nodes [ 
    plotxy innate-leadership-score leadership-value
  ]
end

to update-globals
  set total-num-votes sum [num-votes-received] of nodes
  set total-num-leaders count nodes with [ leader? = true ] 
end

to update-node-visuals
  ifelse show-num-votes? [ 
    set label word (num-votes-received) "     " 
    ]
  [ set label "" ]

  ifelse leader? [
    set size 1 + (num-votes-received / 5000)
    set color violet
  ]
  [ ifelse follower? [
    set color white
  ]
  [ ifelse neutral? [
    set color gray - 1 
  ]
  [ set color pink ]         ;this is the error flag, if there are any nodes that are pink - there is an error in code
  ]
]
end

to update-link-visuals
  set color scale-color gray number-votes-given -5 100
end
@#$#@#$#@
GRAPHICS-WINDOW
291
10
730
470
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks

SLIDER
35
40
252
73
num-nodes
num-nodes
0
100
100
1
1
NIL
HORIZONTAL

BUTTON
35
380
140
413
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
35
80
235
113
vision
vision
0
10
5
1
1
NIL
HORIZONTAL

SWITCH
35
335
190
368
show-num-votes?
show-num-votes?
1
1
-1000

BUTTON
35
425
110
458
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
745
100
807
145
Leaders
total-num-leaders
17
1
11

PLOT
745
220
1038
470
Leadership
Innate Leadership
Leadership Score
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 2 -16777216 true

SLIDER
35
160
207
193
follower-threshold
follower-threshold
0
100
10
1
1
NIL
HORIZONTAL

SLIDER
35
200
195
233
movement-rate
movement-rate
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
35
240
195
273
group-movement-rate
group-movement-rate
0
10
3
1
1
NIL
HORIZONTAL

CHOOSER
35
280
190
325
leadership-distribution
leadership-distribution
"Linear" "Normal" "Poisson"
0

SLIDER
35
120
207
153
leader-threshold
leader-threshold
0
1000
400
1
1
NIL
HORIZONTAL

MONITOR
745
160
847
205
Group Density
count nodes with [ follower? = true ] / \\n( count nodes with [ leader? = true ] )
17
1
11

BUTTON
120
425
200
458
Go Once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
WHAT IS IT?
------------
This model explores whether leadership is a learned trait or an innate trait. Agents begin with various distributions of leadership, and after a process of voting certain leaders are selected. The interesting result is that the top leaders are rarely the ones with the highest innate leadership values.

What is the role of a follower in the journey of a leader? Can we have "placebo" effects of leadership? What is the role of the availability of information in selecting our leaders?


HOW IT WORKS
------------
Before the model runs, each agent is assigned an "innate leadership score". This is used to compute a "leadership value" that takes into account how many followers an agent has, how long have they been remained a leader, etc...

In this model, agents move around and "vote" for the agent with the highest leadership score. Agents look around to find the agent with the highest leadership score that is within their "vision".

The model works in a network geography, where a link is formed between the leader and the follower each time a vote occurs.

Once the number of votes received exceeds a predetermined threshold, the agent changes color into purple turns into a leader. Along with that transformation, the agents that have voted the most for that leader turn into initial followers.

Just like how agents keep a tab of the number of votes that they have received, they also keep a tab of the number of votes that they give out. ( An important thing to notice is agents can only "remember" to vote for one leader. This means that the moment they change voting for someone, the count is effectively cleared. )

The model stops once there are no neutral agents left.


HOW TO USE IT
-------------

Adjust the NUM-NODES button to decide how many agents you want to see in the model.

The VISION button determines how far agents can see to find the agent with the highest leadership potential.

FOLLOWER-THRESHOLD determines how many votes that an agent needs to give for another agent to turn into a follower.

LEADER-THRESHOLD determines how many votes that an agent needs to receive from other agents to turn into a leader.

MOVEMENT-RATE determines how fast agents move around when they are neutral.

GROUP-MOVEMENT-RATE determines how fast agents move when are part of a group.

SHOW-NUM-VOTES? makes the number of votes visible for the agents on the view.

LEADERSHIP-DISTRIBUTION chooser decides whether innate leadership potential is distributed linearly, exponentially or with a poisson distribution.


THINGS TO NOTICE
----------------

Notice the difference that the model shows about the real leadership value. The model proposes that the external factors play a huge role in selecting leaders, and the power of the followers become much more clear.

Two interesting results from the plot is the variability from the tail, where it is difficult to differentiate the top leader from the start. More interestingly, every once in a while the group produces someone with a very low leadership potential as the top leader of the group.

The randomness of the spread of the agents play a huge role in this interesting distribution of leadership. At the end of the day, people choose leaders that are better than them. In one sense, leadership is a relative trait. Therefore, even though an agent may start off with an advantage in the beginning of the model, it really depends on whether agents are surrounded by other agents that have lower leadership potential than them.


THINGS TO TRY
-------------
Play around with the sliders. How does the model change behavior if followers have a low threshold to attach to leaders? Will more or less groups form?

What about vision? Will a high vision rating return more leaders or less leaders?

Try around adjusting the model using different combinations of the sliders.

Finally, look at the different distribution models. Does that affect the spread of final leadership score?


EXTENDING THE MODEL
-------------------
Election Campaigns: 

Right now, agents only look around and they all share the vision coefficient. What if we could make some of the lower candidates for leadership more visible by having them "votable" by everyone? In other words, an agent on the other end of the view could vote for this candidate if it wasn't directly exposed to someone else with a higher leadership score. That signifies what role advertisement campaigns play in elections, since exposure may have a huge say in who receives the top votes.

Rebellion:

Currently, the agents who desert a leader simply join other groups as if nothing happened. What if there were lasting effects of desertion? What could happen to the leaders that have no followers? Do they turn neutral, or do they remain as a leader with no followers?

RELATED MODELS
--------------
Rebellion, Preferential Attachment are both in the models library and they can be accessed with NetLogo version 4.1.2.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles with [ leader? = true ]</metric>
    <enumeratedValueSet variable="num-nodes">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="leader-threshold">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="movement-rate">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="follower-threshold">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-num-votes?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="group-movement-rate">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="leadership-distribution">
      <value value="&quot;Linear&quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="vision" first="1" step="1" last="10"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
