;;   Spatial-Social Interactions

breed       [ people person ]     ;; the agents that interact with the environment
breed       [ events event ]      ;; turtles used to highlight where the environment has an event
patches-own [ urban               ;; this patch is an urban spot
              rural               ;; this patch is rural
              eventspace          ;; this patch is eventspace (an event can occur here)
              event-number ]      ;; the event number which has happened
people-own  [ concern             ;; the concern level of the person
              sharing             ;; whether or not they are sharing an event (1 = sharing this tick, 2 = sharing next tick)
              sharing-event       ;; the event number they are sharing
              shared-events-list  ;; the list of recent events they have shared - prevents runaway feedback loops
              home-patch          ;; the original patch so the person can go home after a "layout"
              shares              ;; the number of times this person has shared a story
              events-seen ]       ;; the number of events this person has seen (regardless of concern level)
globals     [ decay               ;; how much concern levels decay per tick
              count-eventspace    ;; number of eventspace patches
              link-state          ;; whether or not the link button has been toggled
              event-counter ]     ;; a running count of how many events to create event-numbers
extensions  [ rnd ]               ;; use this extension for weighted selection

to startup
  ;; This is really for it to be pretty, since we have to reload it following the clear-all
  load-environment
  reset-ticks
end

to setup
  clear-all
  ;; reset the event couunter
  set event-counter 1

  ;; load in the transportation network to preload patches
  load-environment

  ;; Sprout people in the urban spaces
  ask n-of ((urban-ratio) * num-people) patches with [urban and not eventspace]  [
    sprout-people 1 [ prepare-person set home-patch patch-here ]
  ]

  ;; Sprout people in the rural spaces
  ask n-of ((1 - urban-ratio) * num-people) patches with [rural and not eventspace]  [
    sprout-people 1 [ prepare-person set home-patch patch-here ]
  ]

  ;; now let's make the network
  if any? people [ make-network ]

  ;; hide the links
  ask links [ hide-link ]
  set link-state false

  reset-ticks
end


to go
  ;; calculate an exponential factor from the attention-span in case the user changed it
  ;; 10 = no decay, otherwise do use an exponential decay amount to make this respond somewhat linearly
  ifelse attention-span = 10 [ set decay 1 ] [set decay 1 - 10 ^ (- (attention-span ^ 0.5))]

  ;; clear out the old event markers since they are old news
  ask events [ die ]

  ;; Let's go ahead and trigger events in some of the cells
  ask patches with [eventspace] [
    ;; reset the incidents from the last tick back to normal
    if pcolor = cyan [ set pcolor black ]
    ;; trigger patches randomly to have a news event... uniform probability per eventspace patch
    if random-float 1.0 < event-freq / count-eventspace [
      ;; set the event number for this patch and increment the counter
      set event-number event-counter
      set event-counter event-counter + 1
      ;; set the patch color to show an event occured here
      set pcolor cyan
    ]
  ]

  ;; Since the people are cycled through in an order we need to do two passes so people can
  ;; react to local shares, and then react to the other people.

  ;; First pass - Set sharing for people who finished the last tick with an event to share. Also react to
  ;; local events.
  ask people [
    ;; if they ended the last tick in active sharing state, they are now sharing that story in this tick,
    ;; even without a local event. set them sharing normally, and make their links grey
    ifelse sharing = 2 [ set sharing 1 ask my-out-links [set color grey ]]
      ;; otherwise just clear the sharing out in case they shared normally last tick
      [set sharing 0 set sharing-event 0 ]

    ;; create their newspaper from local events, and then update their status if there are any events
    let newspaper [event-number] of patches in-radius news-range with [pcolor = cyan]
    ;; we pass handle-news false so that it knows it's getting a local newspaper
    handle-news newspaper false
  ]

  ;; Second pass - People react to the remote events shared to them
  ask people [
    ;; if someone in their network has shared it with them, they see it and are concerned
    if sum [sharing] of in-link-neighbors > 0 [
      ;; let's check to make sure we haven't seen this news story before. Get the list of events being shared to us.
      let event-list [sharing-event] of in-link-neighbors with [sharing > 0]
      ;; we pass handle-news true so that it knows it's getting a shared event-list
      handle-news event-list true
    ]
    ;; Clean up - If there is any concern, decay it, threshold it, and update the status
    if concern > 0 [
      set concern concern * decay
      if concern < concern-threshold [ set color blue ]
      if concern > active-threshold  [ set color yellow ]
      if concern > concern-cap [ set concern concern-cap ]
    ]
  ]

  ;; Now create visual indicators for events that have happened so they can be found on the
  ;; display (since patches are just one pixel)
  ask patches with [ pcolor = cyan ] [
    sprout-events 1 [
      ;; make the display a circle the size of the news-range
      set shape "circle"
      set size news-range
      set color cyan
    ]
  ]

  tick

  ;; if we've hit the stopping criteria, stop.
  if any? people [ if mean [concern] of people > stop-at-mean-concern [ stop ] ]

end

;; this makes an existing person into a fresh new one, but preserves their network
to prepare-person
  set shape "person"
  set color blue
  set concern initial-concern
  set size 10
  set shared-events-list []
  set shares 0
  set events-seen 0
end

;; this takes care of picking an event we haven't responded to before and setting up the sharing
to handle-news [ event-list secondary-mode ]
  ;; find one event that hasn't been shared with us.
  let this-event find-not-in-list event-list shared-events-list
  ;; if we got an event number back, that means we haven't seen that event before
  if this-event > 0 [
    ;; make us more concerned
    set concern concern + event-severity
    set events-seen events-seen + 1
    ;; add it to our list
    set shared-events-list lput this-event shared-events-list
    ;; keep our memory limited to the last 20 news stories. old news may reverberate around the network but
    ;; maybe that's realistic if there are long cycles in it.
    if length shared-events-list > 20 [
      set shared-events-list but-first shared-events-list
    ]
    ifelse not secondary-mode [
      ;; if this is an event in my area, we just set sharing
      if concern > concern-threshold or concern > active-threshold [
        set sharing 1
        set sharing-event this-event
        set shares shares + 1
      ]
    ] [
      ;; for events that weren't local but were shared to them...
      ;; if they are concerned, they will share the remote event with their network
      if concern > active-threshold [
        ;; set sharing to 2 so we know in the next round to handle this sharing. News can only move one
        ;; link length per tick.
        set sharing 2
        set sharing-event this-event
        ;; increment the sharing count for this person
        set shares shares + 1
        ;; make the active sharers show their links to make it obvious the span of the remote sharing
        ask my-out-links [show-link set color yellow ]
        ;; make it so the link display toggle button knows there are links showing
        set link-state true
      ]
    ]
  ]
end


;; this is used to search for new news by finding is a newspaper has news (needle-list)
;; they haven't seen before (haystack)
to-report find-not-in-list [needle-list haystack]
  foreach needle-list [
    needle -> if not member? needle haystack [ report needle ]
  ]
  report 0
end

;; if the user wants to keep the network but do another run of the model...
to reset-concern
  ;; have the people reset
  ask people [
    prepare-person
  ]
  ;; hide any links that were showing for active sharers
  ask links [ hide-link ]

  ;; clear all the plots and reset the ticks so it's like a new run
  clear-all-plots
  reset-ticks
end

;; this toggles the display of the links for convenience
to link-toggle
  ifelse link-state [ set link-state false ask links [hide-link] ]
                    [ set link-state true  ask links [show-link] ]
end

;; load up the environment into the patches and handle missing files
to load-environment
  let filename ""
  ;; if the files exist let's use them
  if geography-type = "freight-network"    [ set filename "transport dataset.png" ]
  if geography-type = "shark-attack-data"  [ set filename "shark dataset.png" ]
  ifelse file-exists? filename [
    import-pcolors filename
  ] [
    user-message "The graphics file for the map couldn't be found. Please check to make sure you extracted them from the ZIP file and placed them in the same directory."
  ]
  ;; set up some colors the PNG files use when imported
  let color-rural 56.8
  let color-urban 25
  let color-eventspace 1
  ;; Have the patches set some properties based on their colors for easy logic checks
  ask patches [
    set rural      pcolor = color-rural
    set urban      pcolor = color-urban
    set eventspace pcolor <= color-eventspace
  ]
  ;; give the eventspace patches a location property for analysis purposes
  ask patches with [ eventspace ] [
    ifelse count patches in-radius 4 with [ urban ] > count patches in-radius 4 with [ rural ] [ set urban true ] [ set rural true ]
  ]
  ;; Count how many eventspace patches we have to scale the event frequency to the size of the map
  set count-eventspace count patches with [eventspace]

end

;; allows the user to flip the size of the people so the icons aren't as distracting
to tiny-toggle
  ifelse [size] of one-of people = 1
    [ask people [ set size 10 ]]
    [ask people [ set size 1 ]]
end

;; create the social network
to make-network
  if num-followed > 0 [
    if network-mode = "random-network" [
      ask people [
        ;; create random links within the distance found
        create-links-from n-of num-followed find-nearby false
      ]
    ]
    if network-mode = "scale-free" [
      make-scale-free
    ]
    if network-mode = "small-world" [
      make-small-world
    ]
    if network-mode = "triangles" [
      make-friends-of-friends
    ]
    if network-mode = "dual-scale-free" [
      make-twitter
    ]
  ]
end

to make-scale-free
  ;; preferrential attachment using Uri's netlogo lottery method. Depending on settings,
  ;; it will prefer local links.

  ;; use a boolean for speed in the loop

  ;; create one link to get the lottery started
  if num-followed > 0 and count links = 0 [ ask one-of people [ create-link-from one-of other people ] ]
  ;; begin the lottery!
  ask people [
    ;; create links to other people with probability of how many people they are linked to
    repeat num-followed [
      let candidate self
      ;; find a turtle that isn't me
      while [candidate = self] [
        ;; conduct the lottery
        set candidate find-scale-free
        ;; if the turtle who is possibly anywhere only has 1 link and is not local,
        ;; find a local link instead.
        if ([count my-out-links] of candidate <= 1 and distance candidate > link-distance ) [
          set candidate one-of find-nearby false
        ]
      ]
      create-link-from candidate
    ]
  ]

end

;; this implements Uri's method of building a scale-free network in NetLogo
to-report find-scale-free
  report [one-of both-ends] of one-of links
end

;; this is used during setup to find nearby people to connect to, making sure it's as least as big as the number to follow
to-report find-nearby [ one-link-only ]
  ;; have a variable ready to expand the scope of the search if we don't find enough links
  let focus 0
  ;; a place to store the agentset between runs so we can simply report it if the set is big enough
  let searchset no-turtles
  ;; until we find an agentset of at least count of required links, keep expanding the search radius
  loop [
    ifelse one-link-only
      [ set searchset other people with [ count my-in-links = 0 ] in-radius (link-distance + focus) ]
      [ set searchset other people in-radius (link-distance + focus) ]
    ifelse count searchset < num-followed [
      ;; we didn't find enough people, so we'll just expand the search area next time
      set focus focus + link-distance
    ][
      ;; we've got a big enough set, simply report it.
      report searchset
    ]
  ]
end


to make-twitter
  ;; inspired by http://www.mdpi.com/1099-4300/17/8/5848
  ;; has 4 different preferential attachment types that are formed probabilistically until
  ;; all of the people have a link

  ;; these are weights, not probabilities
  ;; first, the types proposed by the authors
  let pw 1.0   ;; new node created as followed
  let qw 2.0   ;; new node created as a follower   (they recommend q<p but that makes no sense to me)
  let rw 0.5   ;; add links between existing nodes (similar to rewires in smallworld)
  ;; this one I'm adding in because the network above has very low bidirectional links unless the weight
  ;; for r is enormous (they appear to have chosen equivalents of pw=1 and rw=30 roughly)
  let sw 1.0   ;; influential nodes follow other influential nodes that follow them

  ;; rebalance the weights into probabilities
  let tot pw + qw + sw + rw
  let p pw / tot
  let q qw / tot
  let r rw / tot
  let s sw / tot

  ;; make one link so the preferrential attachment doesn't fail
  if not any? links [ ask one-of people [ create-link-from one-of other people ] ]

  while [ count people with [ count my-links = 0 ] > 0 ] [
    let rand random-float 1.0
    if ( rand <= p ) [      ;; new node as followed
      let followed one-of people with [ count my-links = 0 ]
      ask one-of links [
        let follower one-of both-ends
        ask follower [ create-link-from followed ]
      ]
    ]
    if ( rand <= p + q and rand > p ) [    ;; new node as follower
      let follower one-of people with [ count my-links = 0 ]
      ask one-of links [
        let followed one-of both-ends
        ask follower [
          create-link-from followed
        ]
      ]
    ]
    if ( rand <= p + q + s and rand > p + q ) [     ;; make a link bidirectional
      let candidate nobody  ;; we need to find a link that isn't already bidirectional
      let attempts 0   ;; let's try 10 times to find one that isn't bidirectional before we give up
      while [candidate = nobody and attempts < 10 ] [
        ;; selected a link weighted against the count of links the end points have
        set candidate rnd:weighted-one-of links [ count [my-links] of end1 + count [my-links] of end2 ]
        if (link [who] of [end2] of candidate [who] of [end1] of candidate != nobody) [
          set candidate nobody
        ]
        set attempts attempts + 1
      ]
      ;; if we found someone
      if candidate != nobody [
        ask candidate  [
          ;; create a reversed link
          let far-end end2
          ask end1 [ create-link-from far-end ]
        ]
      ]
    ]
    if ( rand > p + q + s ) [     ;; existing node as follower of another existing node
      ask one-of links [
        let follower one-of both-ends
        let followed follower
        while [ follower = followed ] [
          ask one-of links [ set followed one-of both-ends ]
        ]
        ask follower [ create-link-from followed ]
      ]
    ]
  ]
end

;; make a small-world network taking into account the distance settings
to make-small-world
  ;; This is a little complicated, but it makes a big lattice loop out of the people with link distances
  ;; based on the UI setting. With long distances it acts geographically random, and with short
  ;; distances it snakes around the country in a loop from neighbor to neighbor.
  let current-person one-of people
  let person-list []
  let everyone-set people
  ;; where there are still people to link
  while [ count everyone-set > 0 ] [
    ;; starting point for distance search
    let focus 0
    ;; make our candidate list empty
    let candidate-set no-turtles
    ;; while the candidate list is empty
    while [count candidate-set = 0 ] [
      ;; find candidates that are within the link range
      set candidate-set everyone-set with [ distance current-person < link-distance + focus ]
      ;; in case we didn't find any, increase the search distance
      set focus focus + link-distance
    ]
    ;; now pick a random one out of the candidate list
    let next-person one-of candidate-set
    ;; take that person out of the everyone list because they are taken care of
    set everyone-set everyone-set with [ self != next-person ]
    ;; put that onto the ordered person-list for order of linkage
    set person-list lput next-person person-list
    ;; make the next person the current person for the loop
    set current-person next-person
  ]
  ;; borrows heavily from Uri's implementation from here on
  let n 0
  while [n < length person-list ]
  [
    ;; create 2 legs to each has 2 in and 2 out connections.
    ;; with a small ratio of rewirings the loop is unlikely to
    ;; be severed or cut people out of the network.
    let lattice-legs 2
    ;; have a counter to step down list
    let lattice 1
    repeat lattice-legs [
      make-edge item n person-list
                item ((n + lattice) mod length person-list) person-list
      set lattice lattice + 1
    ]
    set n n + 1
  ]

  ;; now rewire to the user specification
  let potential-edges links
  if any? potential-edges [
    ask n-of (rewire-ratio * count potential-edges) potential-edges [
      ;; "a" remains the same
      let node1 end1
      ;; if "a" is not connected to everybody
      if [ count link-neighbors ] of end1 < (count people - 1)
      [
        ;; find a node distinct from node1 and not already a neighbor of node1
        let node2 one-of people with [ (self != node1) and (not link-neighbor? node1) ]
        ;; wire the new edge
        ask node1 [ create-link-from node2 ]
        ;; remove the old edge
        die
      ]
    ]
  ]
end

;; this implements Uri's method of rewiring small world networks, modified for directed links
to make-edge [node1 node2]
  ask node1 [ create-link-from node2 ]
end

;; this implements Allen Downey's Friends of Friends network as an example of a
;; "triangle" network to boost clustering. Unlike the other networks here,
;; this is non-directional like Facebook.
to make-friends-of-friends
  ;; http://allendowney.blogspot.com/2016/09/its-small-world-scale-free-network.html
  let m count people
  ;; make the core a subset of the total, scaled here to the population
  let n log m 10 * 10
  let p 0.10

  ;; make the fully connected core
  ;; randomly select people to make up the core
  let core n-of n people
  ask core [
    let node1 self
    ask other core [
      create-link-with node1
    ]
  ]

  ;; walk through each and make links to every other
  ;; the people that don't have links are the "others"
  let others people with [ count my-links = 0 ]
  ;; now make the triangles
  ask others [
    let link-target one-of core
    let link-from self
    let friends [link-neighbors] of link-target
    ask friends [
      if ( random-float 1.0 < p ) [
        let fof link-neighbors
        ask link-from [ create-link-with one-of fof with [ self != link-from ] ]
      ]
    ]
    create-link-with link-target
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
196
10
904
347
-1
-1
1.0
1
10
1
1
1
0
0
0
1
0
699
-327
0
0
0
1
ticks
30.0

BUTTON
14
292
77
325
NIL
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
14
329
77
362
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
197
366
369
399
event-freq
event-freq
0
20
3.0
.1
1
NIL
HORIZONTAL

PLOT
908
256
1068
376
concern population counts
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"concerned" 1.0 0 -2674135 true "" "plot count people with [concern > concern-threshold]"
"active" 1.0 0 -4079321 true "" "plot count people with [concern > active-threshold]"

SLIDER
383
366
525
399
concern-threshold
concern-threshold
0
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
383
402
525
435
active-threshold
active-threshold
0
100
80.0
1
1
NIL
HORIZONTAL

SLIDER
529
402
670
435
concern-cap
concern-cap
0
200
200.0
1
1
NIL
HORIZONTAL

PLOT
908
133
1068
253
mean concern history
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean [concern] of people"
"pen-1" 1.0 0 -817084 true "" "plot mean [concern] of people with [urban]"
"pen-2" 1.0 0 -8732573 true "" "plot mean [concern] of people with [not urban]"

PLOT
908
10
1068
130
concern histogram
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "set-plot-y-range 0 1\nset-plot-x-range 0 round (max [concern] of people + 1)\nset-histogram-num-bars 15"
PENS
"default" 1.0 1 -16777216 true "" "histogram [concern] of people"
"pen-1" 1.0 0 -817084 true "" "histogram [concern] of people with [urban]"
"pen-2" 1.0 0 -8732573 true "" "histogram [concern] of people with [not urban]"

SLIDER
6
202
98
235
num-followed
num-followed
0
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
529
366
669
399
attention-span
attention-span
1
10
10.0
0.1
1
NIL
HORIZONTAL

SLIDER
197
401
369
434
event-severity
event-severity
0
10
10.0
.1
1
NIL
HORIZONTAL

SLIDER
197
437
369
470
news-range
news-range
0
20
10.0
1
1
NIL
HORIZONTAL

BUTTON
85
329
186
362
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
84
292
187
325
NIL
reset-concern
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
6
84
194
117
num-people
num-people
0
3000
1000.0
50
1
NIL
HORIZONTAL

SLIDER
6
238
194
271
initial-concern
initial-concern
0
100
10.0
1
1
NIL
HORIZONTAL

SLIDER
103
119
195
152
link-distance
link-distance
1
300
40.0
1
1
NIL
HORIZONTAL

CHOOSER
6
155
193
200
network-mode
network-mode
"random-network" "small-world" "scale-free" "triangles" "dual-scale-free"
2

TEXTBOX
492
349
563
367
Agent Control
11
0.0
1

TEXTBOX
250
350
323
368
Event Control
11
0.0
1

BUTTON
692
366
788
399
NIL
link-toggle
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
66
69
143
87
Social Network
11
0.0
1

PLOT
1071
133
1231
253
followers count histogram
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "set-plot-y-range 0 1\nset-plot-x-range 0 round (max [count my-out-links] of people + 1)"
PENS
"out links" 1.0 1 -16777216 true "" "histogram [count my-out-links] of people"
"pen-1" 1.0 0 -955883 true "" "histogram [count my-in-links] of people"

PLOT
1070
10
1230
130
histogram of link-length
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "set-plot-y-range 0 1\nset-plot-x-range 0 round (max [link-length] of links + 1)\nset-histogram-num-bars 15"
PENS
"default" 1.0 1 -16777216 true "" "histogram [link-length] of links"

BUTTON
693
402
788
435
NIL
tiny-toggle
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
792
366
892
399
layout
ask events [ die ]\nif [pcolor != white ] of one-of patches [ ask patches [ set pcolor white ] ]\nask people [ set shape \"circle\" set size 3 ]\nrepeat 10 [ layout-spring people links 0.2 5 5 ]
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
792
402
893
435
undo-layout
load-environment\nask people [ move-to home-patch set shape \"person\" ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
704
348
783
366
Display Options
11
0.0
1

TEXTBOX
807
348
889
366
Reveal Social
11
0.0
1

TEXTBOX
32
10
180
38
Event/Population Geography
11
0.0
1

CHOOSER
7
23
194
68
geography-type
geography-type
"freight-network" "shark-attack-data"
0

PLOT
1071
256
1231
376
events-seen histogram
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "set-plot-y-range 0 1\nset-plot-x-range 0 round (max [events-seen] of people + 1)\nset-histogram-num-bars 15"
PENS
"default" 1.0 1 -16777216 true "" "histogram [events-seen] of people"

SLIDER
6
119
100
152
urban-ratio
urban-ratio
0
1
0.2
.10
1
NIL
HORIZONTAL

SLIDER
14
385
186
418
stop-at-mean-concern
stop-at-mean-concern
0
200
150.0
1
1
NIL
HORIZONTAL

BUTTON
384
437
526
470
move-in-range
ask people [ let me self let targett min-one-of patches with [eventspace] [distance me] if distance targett > news-range [face targett fd distance targett - news-range + 1] ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
101
202
193
235
rewire-ratio
rewire-ratio
0
0.2
0.05
0.01
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

Social networks have a geographic distribution that may be distinct from the geographic distribution of news events that are shared among that network. As a result people can be exposed to events at a much higher frequency than would otherwise occur locally to them.
 
This model is motivated by the potential for future technology in freight transportation to alter the public reaction to accidents. There are currently ~11 fatalties per day in the United States resulting from large truck accidents. News stories for these events appear predominantly in the news of rural areas and rarely in metro areas since truck accidents are generally distributed across road miles instead of being distributed with the population. Changes in truck technology and social media sharing of these local events could increase the reaction to these incidents.

An even more extreme difference between event and population distribution can be tested using the Shark Attack environment.

## HOW IT WORKS

The environment is modeled in three components. The events take place on black patches ("eventspace"). Urban areas appear in orange patches, and rural areas are indicated in green. People are only spawned into the urban and rural areas. All other patches are ignored.

People respond to events that take place on black patches within a selectable news-range along with events shared with them by their social networks. Those events are uniformly distributed over the black patches with average of event-frequency events per tick. Each time a person sees an event their level of concern increases by the event-severity. With every tick their concern level is decreased in accordance with the attention span as well.

Social networks can be configured by several different methods depending on what aspects of the social network are interesting to test.

Based on their level of concern people may also react to events:

* If a person's concern level exceeds their concern threshold, they will share a local event with their outbound links. These people are shown in red. They remain passive about events that took place elsewhere.

* If their concern level exceeds their active threshold, they will share events they see regardless of being local or not. These people are displayed in yellow. When they are actively sharing a news story their links will flash yellow.

## HOW TO USE IT

Use the Setup button to set the population and network. Click on Go Once to run the model one "day" or Go to run it continuously. The Reset-concern button will reset all of the people, charts, and ticks but keep the existing network. 

All settings on the left column above the setup button only apply before setup and will not impact the model if changed between ticks.

The environment can either be the heavy freight transportation network or a map of shark attacks across the United States. 

Use num-people to set how many people to simulate. Urban-ratio splits how people are spawned between urban and rural locations with 1.0 meaning all people spawn into urban locations.
 
The network-mode has the following options:

* Random networks are constructed among random people within the link distance. This option is best to mimic word of mouth in short distances. Number of followers is distributed by the Poisson distribution with a uniform number followed at the selected number in the interface. This can easily create disconnected geographic regions.

* Scale-free networks randomly connect followers to influential people regardless of distance, but includes distance in preferences to linking to less influential people. Number followed is uniform at the selected number in the interface, and power-law for the followers.

* Small World builds a lattice network in a loop with selectable rewiring ratio in the interface. These networks have uniform distribution of connections with no rewiring and normally distributed with many rewirings.

* Triangles uses the Friends-of-Friends approach with a fully connected core of links with followers that also form "triangles" with friends. These links are non-directional. This network type aims to generate the clustering aspects of Facebook social network. The links tend to follow normal distributions unlike the other options and much higher clustering.

* The Dual Scale-free network connects directionally with both followers and followed in power law relationships. Influential links are preferrentially made bidirectional, and additional links are also built between influential nodes. This was proposed as a way to mimic the Twitter network.

Except for Triangle networks, link-distance controls how the social network is built. Distance influences how close of links the network tries to build in pixels (roughly 8.4 km per pixel). Very long distance settings result in networks that are not built in regard to geographic distance while short settings will prefer short link distances.

The remaining settings can be changed between ticks:

Stop-at-mean-concern will shut the model down at a pre-set mean level of concern to simplify tick-based experiments.

Event-frequency establishes how many total events on average occur per tick across all of the environment. Event-severity is how much concern each event adds to the people. News-range configures how big of a geographic area hears about a local event.

Concern-threshold, Active-threshold, and concern-cap all control how people respond to their concern level. The concern-cap limits the total amount of concern a person can have. By making the concern-cap below either the active or concern thresholds you can prevent people from triggering these states. Attention-span sets how quickly concern decays for people. Low numbers are very rapid decay of concern, and high numbers are very slow decay with 10 disabling the feature. Move-in-range takes relocates all people so they are within range of a location they can directly see an event.

Two display toggles are available. The link display can be toggled to be able to examine the created network with the link-toggle button. The tiny-toggle button changes the people shape between large and tiny sizes which may make links and other patterns easier to see.

The layout button removes the geographic network and orders the network by the links. In the case of more than one follower per person, this could just be a tight ball. Click the undo-layout button to return to the model run.

The first column of plots: two plots provide information on the distribution of concern and the mean concern across all people split by urban (orange) and rural (green). The last plot is a time-series of the count of each population of concern (red) and active (yellow) over the ticks.

In the second column of plots there are two static histograms of the connectivity of the network. In the first the distribution of link lengths of the network is shown, and in the second the number of out-bound links for each person is shown. These can be used to confirm the construction of the network.The final histogram shows the distribution of number of events seen by each person.


## THINGS TO NOTICE

* Social networks with short distance links mimic word-of-mouth. Information moves slowly and can get stuck in local areas. Rural areas tend to have limited impact in these cases until the network is highly active. 

* With attention spans in use, it's very possible to have long periods of mild concern levels that seem stable until a sudden onset of active sharing occurs without a change in any settings. This is very sensitive to network configuration and an event being visible to influential people (lots of followers).

* Showing the layout of the social network at the end of a run reveals patterns in concern that differ from the geography. Geographic neighbors can have very different concern levels and sharing activity.

## THINGS TO TRY

* Check out the shark attack data set to see interactions between coastal and inshore areas. How many news stories will the people of Omaha see?

* Try networks with low activation thresholds but short attention spans. They'll share anything they see, but not stay concerned for long. This might be useful for panic types of events.

* If you are using attention spans, try comparing the event frequency or severities which are necessary to activate the social media network to the setting which is necessary to quiet it.

* Check out the different social networks for how their behavior differs. Some have an even rate of increase in concern, while others slowly increase until a rapid onset. The Triangle and Dual Scale Free networks correspond better to intuition with attention spans in use.

## EXTENDING THE MODEL

The black patches could represent anything geographically distinct from population distribution which produces information that travels across a social network. This seems possibly useful for other interactions like natural disasters and other infrastructure like electricity grids. 

The split of urban and rural is not rigorously modeled. A better way to distribute people across these categories and their social network structures might improve the correspondence to reality. The method of color coding urban areas scaled to population isn't ideal either. In the NYC area some people may be isolated from observing any events because they end up signficantly offshore artificially. Some urban areas also overlap. A heatmap might work better, but introduces its own issues. 

The agents are only mildly responsive to their environment and all operate with the same rules and thresholds. They are also perfectly perceptive and don't exaust on the issue unlike what happens in the real world.

The events having a single magnitude doesn't represent the real world well. Some shark attacks are just surf boards being bitten and others are fatalities.

## NETLOGO FEATURES

import-pcolors is the most exotic element used in this ABM. This configures the environment and corrals the turtles to particular spawning locations. 

## RELATED MODELS

Language Diffusion inspired the network approach. Grand Canyon inspired the approach to handle the geographic elements. Preferrential Attachment heavily informed the scale-free network approach, and Small World heavily informed the small world approach.

## CREDITS AND REFERENCES

Thanks to the anonymous reviewers of Complexity Explorer for their valuable feedback to improve this model.

Scalefree Network and Small World Implementations heavily borrowed from Uri Wilensky. Friends of Friends implemented based on the model proposed by Allen Downey. The dual preferrential attachment is inspired by the Aparicio, Villzon-Terrazas, and Alvarez (2015) model of the Twitter network as published in the journal Entropy. 

GIS Data:

* Transportation Network: https://ops.fhwa.dot.gov/freight/freight_analysis/faf/

* Shark Data: ArcGIS Shark Attacks

* Cities (scaled using the Flannery Method) https://catalog.data.gov/dataset/usgs-small-scale-dataset-cities-and-towns-of-the-united-states-201403-shapefile

* Basemap: Google Maps
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
