turtles-own
  [ age                  ;; age group: Young, Auult, Mature, Elderly
    healthy?             ;; if true, not infected, sick or immune
    infected?            ;; if true, turtle is infected
    infected-time        ;; how long, in days, the turtle has been infectous
    sick?                ;; if true, the turtle is symptomatic
    sick-time            ;; how long, in days, the turtle has been symptomatic
    immune?              ;; if true, turtle is immune
    immune-time          ;; how long, in days, the turtle has been immune
    chance-infection     ;; / 100 = the probability of infection for given age
    chance-recover       ;; / 100 = the probability of recovery for given age
  ]

globals
  [ young                ;; number of young (constant set from interface % slider)
    adult                ;; number of adults
    mature               ;; number of martures
    elderly              ;; number of elderly
    initial-infected     ;; total number of initial people infected
    %infected            ;; what % of the population is infectious
    %sick                ;; what % of the population is symptomatic
    %immune              ;; what % of the population is immune
    young-sick           ;; total young getting sick
    young-deaths         ;; count of young who have died
    young-deaths%        ;; % of young who have died
    adult-sick           ;; total adults getting sick
    adult-deaths         ;; count of adults who have died
    adult-deaths%        ;; % of adults who have died
    mature-sick          ;; total mature getting sick
    mature-deaths        ;; count of mature who have died
    mature-deaths%       ;; % of mature who have died
    elderly-sick         ;; total elderly getting sick
    elderly-deaths       ;; count of elderly who have died
    elderly-deaths%      ;; % of elderly who have died
    deaths               ;; count of total deaths during run
    deaths%              ;; deaths as percent of original population
    mark                 ;; dummy for plotting year marks
  ]

to setup
  ;; check for error in setting % of population sliders
  if ( %young + %adult + %mature + %elderly) != 100 [
    print "POPULATION DISTRIBUTION ERROR"
    print %young + %adult + %mature + %elderly
    stop ]

  clear-all                      ;; The setup is divided into five procedures
    setup-constants
    init-values
    setup-turtles
    update-global-variables
    update-display
  reset-ticks
end

to setup-constants     ;; This sets up constants of the model
  set initial-infected initial-infected% * number-people / 100

  set young %young * number-people / 100       ;; convert % to numbers
  set adult %adult * number-people / 100
  set mature %mature * number-people / 100
  set elderly %elderly * number-people / 100
end

to init-values     ;; initialize values to 0
  set young-deaths 0
  set adult-deaths 0
  set mature-deaths 0
  set elderly-deaths 0
  set young-deaths% 0
  set adult-deaths% 0
  set mature-deaths% 0
  set elderly-deaths% 0
end

to setup-turtles
  create-turtles young                     ;; number-people set in interface
    [ setxy random-xcor random-ycor        ;; random location
      set size .9                          ;; easier to see
      set age 10                           ;; age = Young
      get-infection                        ;; set age-dependent parameters
      get-recover
      get-healthy ]                        ;; make every turtle healthy
  create-turtles  adult                    ;; number-people set in interface
    [ setxy random-xcor random-ycor        ;; random location
      set size .9                          ;; easier to see
      set age 30                           ;; age = Adult
      get-infection                        ;; set age-dependent parameters
      get-recover
      get-healthy ]                        ;; make every turtle healthy
  create-turtles mature                    ;; number-people set in interface
    [ setxy random-xcor random-ycor        ;; random location
      set size .9                          ;; easier to see
      set age 50                           ;; age = Mature
      get-infection                        ;; set age-dependent parameters
      get-recover
      get-healthy ]                        ;; make every turtle healthy
  create-turtles elderly                   ;; number-people set in interface
    [ setxy random-xcor random-ycor        ;; random location
      set size .9                          ;; easier to see
      set age  70                            ;; age = Elderly
      get-infection                        ;; set age-dependent parameters
      get-recover
      get-healthy ]                        ;; make every turtle healthy

  ask n-of initial-infected turtles                ;; set initial number of infected
    [ get-infected ]
end

to get-infection     ;; set each turtle's chance of infection for their age (interface)
  if age = 10 [ set chance-infection infection-young ]
  if age = 30 [ set chance-infection infection-adult ]
  if age = 50 [ set chance-infection infection-mature ]
  if age = 70 [ set chance-infection infection-elderly ]
end

to get-recover     ;; set each turtle's chance of recovery for their age (interface)
  if age = 10 [ set chance-recover recover-young ]
  if age = 30 [ set chance-recover recover-adult ]
  if age = 50 [ set chance-recover recover-mature ]
  if age = 70 [ set chance-recover recover-elderly ]
end

to get-healthy           ;; set healthy parameters
  set healthy? true
  set infected? false
  set infected-time 0
  set sick? false
  set sick-time 0
  set immune? false
  set immune-time 0
end

to get-infected              ;; set infected parameters
  set healthy? false
  set infected? true
  set infected-time 0        ;; infection duration set in interface
  set sick? false
  set sick-time 0
  set immune? false
  set immune-time 0
end

to get-sick                  ;; set sick parameters
  set healthy? false
  set infected? false
  set infected-time 0
  set sick? true
  set sick-time 0            ;; sickness duration set in interface
  if age = 10 [set young-sick young-sick + 1]   ;; collect data on sick distribution
  if age = 30 [set adult-sick adult-sick + 1]
  if age = 50 [set mature-sick mature-sick + 1]
  if age = 70 [set elderly-sick elderly-sick + 1]
  set immune? false
  set immune-time 0
end

to get-immune             ;; set immune parameters
  set healthy? false
  set infected? false
  set infected-time 0
  set sick? false
  set sick-time 0
  set immune? true
  set immune-time 0          ;; immunity duration set in interface
end

to go
  ask turtles
    [ get-older
    if infected-time > infection-duration [ get-sick ]
    if sick-time > sickness-duration  [ recover-or-die ]
    if immune-time > immunity-duration [ get-healthy ]
    move
    ]
  update-global-variables
  update-display

  ;; if there is no one infected or sick, introduce sick turtles to test herd immunity
  let ill count turtles with [infected?] + count turtles with [sick?]
  if ill < initial-infected [
      ask n-of (initial-infected - ill ) turtles  ;; reset initial number of infected
    [ get-infected ]
    ]

  ifelse ( ticks = 364 or ticks = 728 or ticks = 1092 )     ;; put year marker on plot
    [ set mark 10 ]
    [ set mark 0 ]

  if  ticks >= 1093  [ stop ]     ;; stop at 3 years (1092 days)
  tick
end

to update-global-variables
  set %infected 100 * count turtles with [ infected? ] / number-people
  set %sick 100 * count turtles with [ sick? ] / number-people
  set %immune 100 * count turtles with [ immune? ] / number-people
  set young-deaths%  100 * young-deaths / young
  set adult-deaths% 100 * adult-deaths / adult
  set mature-deaths% 100 * mature-deaths / mature
  set elderly-deaths% 100 * elderly-deaths / elderly
  set deaths% 100 * deaths / number-people

end

to update-display
  ask turtles                                      ;; set turtle to appropriate color:
    [ set color ifelse-value infected? [ yellow ]  ;;  infected = yellow
      [ ifelse-value sick? [ red ]                 ;;  sick = red
        [ ifelse-value immune? [green ] [ gray ]   ;;  immune = green
        ]                                          ;;  healthy = gray
      ]
    ]
end

to get-older     ;;infected-time, sick-time, immune-time are increased
  if infected? [ set infected-time infected-time + 1 ]
  if sick? [ set sick-time sick-time + 1 ]
  if immune? [ set immune-time immune-time + 1 ]
end

to move                        ;; turtles turn at random, move forward, then decide what to do
  rt random 100
  lt random 100
  ifelse not any? turtles-on patch-ahead 1 [ fd 1 ]  ;; nobody there, go ahead and move
                                                     ;; somebody there, social distancing important
    [ if random 100 > sd-effective                   ;; ignore sd, move anyway
       [ fd 1
         if any? turtles-here with [ sick? or infected? ]
                                                     ;; any infected or sick here?
          [ if any? turtles-here with [ healthy? ] [
                                                     ;; any healthy mixed with infected or sick?
                                                     ;; if so check chance of infection
            ask turtles-here with [ healthy? ] [ if random 100 < chance-infection [ get-infected ] ] ]
          ]
       ]
    ]
end

;; Once the turtle has been sick long enough, it
;; either recovers (and becomes immune) or it dies.
;; Deaths are counted but turtle is not removed so population is retained for run.
to recover-or-die
  ifelse random 100 < chance-recover [ get-immune ]

  [ if age = 10 [ set young-deaths young-deaths + 1 ]
    if age = 30 [set adult-deaths adult-deaths + 1]
    if age = 50 [set mature-deaths mature-deaths + 1]
    if age = 70 [set elderly-deaths elderly-deaths + 1]
    set deaths deaths + 1                        ;; count death but don't remove turtle
    get-healthy                                  ;; reset turtle in population
    ]
end
@#$#@#$#@
GRAPHICS-WINDOW
377
10
1195
829
-1
-1
10.0
1
10
1
1
1
0
1
1
1
-40
40
-40
40
1
1
1
ticks
30.0

BUTTON
160
10
224
43
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1227
22
1282
55
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
13
81
185
114
number-people
number-people
0
10000
5000.0
100
1
NIL
HORIZONTAL

SLIDER
13
164
185
197
sickness-duration
sickness-duration
0
28
21.0
7
1
days
HORIZONTAL

MONITOR
1382
326
1443
371
%sick
%sick
1
1
11

MONITOR
1456
326
1523
371
%immune
%immune
1
1
11

BUTTON
1320
21
1381
54
Go-Once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1536
326
1593
371
%dead
deaths%
1
1
11

PLOT
1226
69
1580
290
Illness Data
Days
%
0.0
1220.0
0.0
10.0
true
true
"" ""
PENS
"%sick" 1.0 0 -2674135 true "" "plot %sick"
"%infected" 1.0 0 -16777216 true "" "plot %infected"
"year mark" 1.0 0 -10899396 true "" "plot mark"

SLIDER
194
164
366
197
immunity-duration
immunity-duration
0
720
364.0
1
1
days
HORIZONTAL

MONITOR
1224
326
1281
371
Day
ticks
0
1
11

SLIDER
193
82
365
115
initial-infected%
initial-infected%
0
10
1.0
1
1
%
HORIZONTAL

SLIDER
19
239
179
272
infection-young
infection-young
0
100
5.0
1
1
%
HORIZONTAL

SLIDER
20
278
181
311
infection-adult
infection-adult
0
100
50.0
1
1
%
HORIZONTAL

SLIDER
20
357
182
390
infection-elderly
infection-elderly
0
100
80.0
1
1
%
HORIZONTAL

SLIDER
200
239
362
272
recover-young
recover-young
0
100
99.0
1
1
%
HORIZONTAL

SLIDER
202
278
364
311
recover-adult
recover-adult
0
100
95.0
1
1
%
HORIZONTAL

SLIDER
201
358
364
391
recover-elderly
recover-elderly
0
100
80.0
1
1
%
HORIZONTAL

MONITOR
1307
445
1392
490
young deaths
young-deaths
0
1
11

MONITOR
1308
603
1395
648
elderly deaths
elderly-deaths
0
1
11

MONITOR
1490
445
1606
490
young mortality %
100 * young-deaths / young-sick
0
1
11

MONITOR
1490
496
1607
541
adult mortality %
100 * adult-deaths / adult-sick
0
1
11

MONITOR
1492
603
1610
648
elderly mortality %
100 * elderly-deaths / elderly-sick
0
1
11

MONITOR
1306
496
1393
541
adult deaths
adult-deaths
0
1
11

SLIDER
194
123
366
156
infection-duration
infection-duration
0
10
7.0
1
1
days
HORIZONTAL

SLIDER
13
124
185
157
sd-effective
sd-effective
0
100
70.0
1
1
%
HORIZONTAL

MONITOR
1297
326
1368
371
%infected
%infected
1
1
11

SLIDER
20
317
182
350
infection-mature
infection-mature
0
100
50.0
1
1
%
HORIZONTAL

SLIDER
201
318
364
351
recover-mature
recover-mature
0
100
90.0
1
1
%
HORIZONTAL

MONITOR
1308
551
1393
596
mature deaths
mature-deaths
0
1
11

MONITOR
1492
549
1612
594
mature mortality %
100 * mature-deaths / mature-sick
0
1
11

SLIDER
52
480
176
513
%young
%young
0
50
20.0
1
1
NIL
HORIZONTAL

SLIDER
209
479
333
512
%adult
%adult
0
50
30.0
1
1
NIL
HORIZONTAL

SLIDER
52
522
176
555
%mature
%mature
0
50
25.0
1
1
NIL
HORIZONTAL

SLIDER
210
520
333
553
%elderly
%elderly
0
50
25.0
1
1
NIL
HORIZONTAL

MONITOR
1216
445
1302
490
young cases
young-sick
0
1
11

MONITOR
1217
498
1302
543
adult cases
adult-sick
0
1
11

MONITOR
1217
551
1303
596
matures cases
mature-sick
0
1
11

MONITOR
1217
603
1302
648
elderly cases
elderly-sick
0
1
11

TEXTBOX
149
61
299
79
GLOBAL Variables
11
0.0
1

TEXTBOX
130
219
280
237
AGE-DEPENDENT Variables
11
0.0
1

TEXTBOX
132
453
282
471
Population Distribution
11
0.0
1

TEXTBOX
1349
417
1499
435
Mortality Summary
11
0.0
1

TEXTBOX
1355
301
1505
319
Instantaneous Data
11
0.0
1

MONITOR
1400
445
1482
490
% all deaths
100 * young-deaths / deaths
0
1
11

MONITOR
1400
497
1482
542
% all deaths
100 * adult-deaths / deaths
0
1
11

MONITOR
1401
551
1483
596
% all deaths
100 * mature-deaths / deaths
0
1
11

MONITOR
1403
603
1485
648
% all deaths
100 * elderly-deaths / deaths
0
1
11

@#$#@#$#@
## WHAT IS IT?
The model is a visualization of the transmission of a virus through a human population and the potential impact of social distancing on person-to-person contagion. The model is loosely based on the Virus model of the Netlogo Models Library and a model by Anamaria Berea of the Covid-19 virus (see citations). The model provides the user with control over parameters frequently considered as factors in the development of a wide-scale (pandemic) infectious viral disease. The time scale is in DAYS reflecting the rapid spread of such a disease. The population is broadly broken into four segments: young, adult, mature, and elderly, the fractions chosen by the user. Each segment has its own chance of infection and chance of recovery. 

Global factors include duration of infection (infecious but asymptomatic), duration of sickness (infectious and symptomatic), and duration of immunity following recovery. Deaths are recorded but dead agents are returned to the population. 

"Social distancing" (SD) became an issue with Covid-19 and a parameter for SD effectiveness has been included. An agent has a probability of avoiding other turtles by not moving to a patch where others are located. If the agent does move to an occupied patch it may become infected based on its individual (age related) chance of becoming infected. If the agent is already infected or sick, it can infect others on the common patch.

## HOW TO USE IT

Procedural notes
- The units in the model are critical to proper use. 
- The internal time unit (tick) is 1 DAY. 
- The percentage sliders are in integer units. 
- The sum of the population percentages must equal 100 or the code stops and an error is reported in the Command Center.

--- User selected sliders (GLOBAL):
number-people: 0 - 10000 (deaths are replaced, initial population is retained for run)
initial-infected: 0 - 10% of number-people
sd-effective: 0 - 100% effectivness of social-distancing
infection-duration: asymptomatic sick DAYS (infectious)
sickness-duration: symptomatic sick DAYS (infectous)
immunity-duration: immune to infection DAYS


--- User selected sliders (AGE SPECIFIC):
The population is divided into four groups: youth, adult, mature, and elderly. The specific age of each group is not important since the agents do not age. The duration of each agent's infection, sickness and immunity do age and change according to the parameters set by the user.

Chance of contracting illness when in contact with an infections agent is user-selected over range 0 - 100%.
infection-young
infection-adult
infection-mature
infection-elderly

Chance of recovery following sickness-duration period is user selected over range 0 - 100%. Failure to recover is death. Dead agents ARE replaced.
recovery-young 
recovery-adult
recovery-mature
recovery-elderly


--- Run will stop at 1092 ticks (3 years, 1092 days)

--- The turtles in the display are color-coded as follows:
gray - healthy
yellow - infected
red - sick
green - immune


## CREDITS AND REFERENCES:
Model author: V. Alan Mode, alan MODE associates, Pleasanton, CA, kk6zl@comcast.net

Corvid-19 model:
Anamaria Berea, Associate Term Professor in Computational and Data Sciences, George Mason University, http://modelingcommons.org/browse/one_model/6227#model_tabs_browse_info

Netlogo Virus model:
Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
COPYRIGHT AND LICENSE
Copyright 1998 Uri Wilensky.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
