;; forecasting is done in R environment
;; Before using the model it is important to install NetLogo 5.3.1, R code above 3.2, extension R (if it is not present) and the newest rJava package in R
;; Configuration  on windows environment variables:
;; create new variable - name: R_HOME, path: C:\Program Files\R\R-3.3.3
;; create new variable - name: JRI_HOME, path: C:\Users\user\Documents\R\win-library\3.3\rJava\jri
;; Add to existing variable  - name: Path, path: C:\Program Files\R\R-3.3.3\bin\x64
;; More information can be found in: https://ccl.northwestern.edu/netlogo/docs/r.html
extensions [r]

globals [
  first_demand ;; create a first demand variable to generate initial inputs and prepare demand history for forecasting
  mean_demand ;; distributions are used from multiple product category, Therefore there is extreme variation in the demand. The mean_demand is used as a threshold to limit this variation.
        ]

breed [distributors distributor]
breed [households household]

distributors-own [
  order_to_supplier
  order_from_household
  total_order_from_household
  available_stock
  demand_history ;; a list of all previous demand history
  group_demand_history ;; list of lists of all distributor demand history
  forecasting ;; variable to save forecasting result with all statistics
  forecast ;; variable for final forecasted value
  forecasting_history ;;save forecasting values to history
  plot_forecast ;; save previous tick forecasting result for plotting with delay
  backlog ;; level of unfulfilled demand
  total_backlog ;; level of unfulfilled demand for all distributors
  unfulfilled_demand ;; the backlog level is added to unfulfilled demand and is necessary to deliver once available inventory is present
  safety_stock ;; safety stock level based on previous 7 tick demand average and safety_level set by slider
]

households-own [
  order_to_distributor
  distribution_type ;; randomly assigns distribution type to select a distribution function
  product_quantity ;; the distributions are generated from ratio of sales by day. Therefore, the generated ratio is multiplied by different product quantity, to simulate the variation of consumer shopping habits.
]

to setup
  __clear-all-and-reset-ticks
  ask patches [set pcolor white]
  ;; assign the initial demand of all household
  random-seed 100

  create-distributors Distributor_N [
    set size 5
    set color blue
    set label (word "Distr." [who] of self "     ")
    set forecast 0
    set forecasting_history []
  ]

  create-households Households_N [
   set size 1
   set color 3
   set order_to_distributor 0
   set distribution_type []
  ]

  set-default-shape distributors "house two story"
  set-default-shape households "person"

  ask turtles [find_patch]

end

to go
  reset_variables

  if ticks = 0 [ set_first__variables ]
  ifelse consumer_loyalty = FALSE
  [
    create-link
    interaction_supplier
  ]
  [
    if ticks = 0 [ create-link ]
    interaction_supplier
  ]

  receive_order_from_supplier
  select_demand_distribution
  interaction_household
  check_mean_demand
  update_demand_history
  set_group_history
  update_inventory

  if consumer_loyalty = FALSE [ ask links [die] ]

  tick
end

;; randomly assign all distributors and household to individual patch
to find_patch
  setxy random-xcor random-ycor
  while [any? other turtles-here]
      [find_patch]
end

;; save individual distributor sales history to a list of lists, where the first element in the index of the distributor
;; and the second is a list of demand history of that distributor
to set_group_history
  ask turtles with [ breed = distributors ]
  [
  set group_demand_history (list (list who) (list demand_history))
  ]
end

to set_first__variables
  select_demand_distribution
  ask turtles with [breed = households]
    [
      ;; the distribution functions are obtained from ratio of real-data. The purpose of this is to simulate seasonality of the real-world environment.
      ;; the ratio is then multiplied by the product quantity which household buy. This way the customer is also distinguished by number of people in the household.
      set product_quantity random (Max_product_quantity - 1) + 1
      set order_to_distributor round product_quantity * order_to_distributor
      if order_to_distributor < 0 [ set order_to_distributor (- order_to_distributor)]
      let demand []
      set demand order_to_distributor
      set demand round demand
      create-link
      ask link-neighbors with [ breed = distributors ] [
        set order_from_household demand
        set total_order_from_household (total_order_from_household + order_from_household)
      ]
    ]

  ask turtles with [ breed = distributors ] [
    set first_demand total_order_from_household
    set demand_history (list (first_demand) (first_demand) (first_demand) (first_demand) (first_demand))
    set group_demand_history (list (list who) (list demand_history))
    set available_stock first_demand * 3
     ]
end

;; function to get first n elements of a list
to-report take_first [n xs]
  report sublist xs 0 min list n (length xs)
end

;; the order from supplier is added to the available stock. The value is obtained from previous tick, therefore there is a delay of delivery of 1 day
to receive_order_from_supplier
  ask turtles with [ breed = distributors ]
  [
    set safety_stock demand_history
    set safety_stock reverse safety_stock
    set safety_stock filter [? != 0] take_first 7 safety_stock
    set safety_stock (mean safety_stock) * safety_level
    ifelse available_stock < safety_stock
    [ set available_stock (available_stock + order_to_supplier + safety_stock) ]
    [ set available_stock (available_stock + order_to_supplier) ]
  ]
end

;; the demand is forecasted based on individual history or group history of companies. The distributor is ordering supplies 1 tick before the household orders.
 to interaction_supplier
   ifelse Share_information = false
     ;; individual demand forecasting
     [
       let distributors_index n-values count  turtles with [breed = distributors] [?]
       ask turtles with [breed = distributors] [
         ;; need to iterate trough individual turtle, otherwise demand_history is added for all distributors not only 1 (list of lists problem)
         foreach distributors_index [

         ;; machine learning/neural network algorithm for individual forecasting
         ;; clear the R workspace
         r:clear
         ;; command to clear console when printing variables use r:eval "print(variable_name)" or show r:get "variable_name", but some formats are not supported by NetLogo
         r:eval "cat(rep('\n',50))"
         ;; load extreme machine learning package
         r:eval "library('elmNN')"
         r:eval " splitWithOverlap <- function(vec, seg.length) { overlap = seg.length - 1 ; starts = seq(1, length(vec), by=seg.length-overlap); ends   = starts + seg.length - 1 ;ends[ends > length(vec)] = length(vec) ;y <- lapply(1:length(starts), function(i) vec[starts[i]:ends[i]]) ;y <- y[unlist(lapply(y, length) == seg.length)] ;y <- as.data.frame(t(as.data.frame(y))) ;rownames(y) <- c() ;colnames(y)[length(colnames(y))] <- 'y' ;return(y) } "

         ;; put demand_history variable in to R environment and create a sequence column to identify the time series
         r:put "r_demand_history" [demand_history] of turtle ?
         r:eval "train <- splitWithOverlap(r_demand_history,4)"
         ;; in the beginning the neural network when facing with spikes over predicts. Manually we set the over predictions to the mean. The threshold is decided by input
         r:eval "mean_demand <- mean(r_demand_history)"
         r:put "mean_threshold" mean_threshold

         ;; train the extreme machine learning algorithm
         r:eval "model <- elmtrain(y ~ . , data=train, nhid=1, actfun='sig')"
         ;; create an empty variable for the next tick prediction
         r:eval "predict_df <- tail(r_demand_history,3)"
         r:eval "predict_df <- as.data.frame(t(as.data.frame(predict_df)))"
         r:eval "predict_df$y <- 0"
         r:eval "colnames(predict_df) <- c('v1','v2','v3','y')"

         ;; predict the next tick's demand
         r:eval "prediction <- predict.elmNN(model, newdata = predict_df)"
         r:eval "prediction <- as.numeric(prediction)"
         ;; check if the predicted demand is not above the mean with a certain threshold
         if mean_limit_status = true [ r:eval "if (prediction > (mean_demand * mean_threshold)) { prediction <- mean_demand * mean_threshold}" ]
         ;; save the demand to NetLogo variable
         set forecasting r:get "prediction"
         ;; save the forecasting results and prepare order from supplier
         set forecast round forecasting
         set order_to_supplier forecast
         if order_to_supplier < 0 [set order_to_supplier 0]
         ifelse order_to_supplier + unfulfilled_demand < available_stock
         [ set order_to_supplier 0 ]
         [ set order_to_supplier (order_to_supplier + unfulfilled_demand - available_stock) ]
         ]
       ]
     ]
      ;; collaborative demand forecasting (forecast with machine learning from all distributor information, not only individual)
      [
        ;; machine learning/neural network algorithm for individual forecasting
        ;; clear the R workspace
        r:clear
        ;; command to clear console when printing variables use r:eval "print(variable_name)" or show r:get "variable_name", but some formats are not supported by NetLogo
        r:eval "cat(rep('\n',50))"
        ;; load extreme machine learning package
        r:eval "library('elmNN')"
        r:eval "splitWithOverlap <- function(vec, seg.length) { overlap = seg.length - 1 ; starts = seq(1, length(vec), by=seg.length-overlap); ends   = starts + seg.length - 1 ;ends[ends > length(vec)] = length(vec) ;y <- lapply(1:length(starts), function(i) vec[starts[i]:ends[i]]) ;y <- y[unlist(lapply(y, length) == seg.length)] ;y <- as.data.frame(t(as.data.frame(y))) ;rownames(y) <- c() ;colnames(y)[length(colnames(y))] <- 'y' ;return(y) } "
        r:eval "splitWithOverlap_combine <- function(x, seg.length = 4) { ;for (i in 1:ncol(x)) { ;active_df <- x[i] ;active_df <- t(as.vector(active_df)) ;ts <- splitWithOverlap(active_df,seg.length) ;colnames(ts) <- paste(colnames(x)[i], colnames(ts), sep = '_') ;if (exists('ts_final') == FALSE) { ;ts_final <- ts ;} else { ;ts_final <- cbind(ts_final,ts) ;} ;} ;return(ts_final) ;}"
        r:eval "create_predict_df <- function(x) { ;for (i in 1:ncol(x)) { ;active_df <- x[i] ;active_df <- tail(active_df,3) ;active_df <- as.data.frame(t(as.vector(active_df))) ;colnames(active_df) <- c('V1','V2','V3') ;active_df$y <- 0 ;colnames(active_df) <- paste(rownames(active_df), colnames(active_df), sep = '_') ;if (exists('df_final') == FALSE) { ;df_final <- active_df ;} else { ;df_final <- cbind(df_final,active_df) ;} ;} ;return(df_final);}"

        (r:putagent "r_demand_history" turtles with [breed = distributors] "group_demand_history")

        r:eval "r_demand_history <- as.data.frame(matrix(data = as.vector(unlist(r_demand_history)), nrow = length(unlist(r_demand_history$group_demand_history[[1]])), ncol = length(r_demand_history[[1]])))"
        r:eval "colnames(r_demand_history) <- r_demand_history[1, ]"
        r:eval "r_demand_history <- r_demand_history[-1, ] "
        r:eval "row.names(r_demand_history) <- NULL"
        r:eval "colnames(r_demand_history) <- paste('Distributor', colnames(r_demand_history), sep = '_')"
        r:eval "column_names <- colnames(r_demand_history)"
        r:eval "train <- splitWithOverlap_combine(r_demand_history,4)"

        ;; in the beginning the neural network when facing with spikes over predicts. Manually we set the over predictions to the mean. The threshold is decided by input
        r:eval "mean_demand <- mean(colMeans(r_demand_history[1:(ncol(r_demand_history)-1)]))"
        r:put "mean_threshold" mean_threshold

        ;; train the extreme machine learning algorithm
        ;;create a variable of the distributor index
        let distributors_index n-values count  turtles with [breed = distributors] [?]

        foreach distributors_index [

        r:put "active_distributors_index" ?
        r:eval "active_distributors_index <- as.numeric(active_distributors_index)"
        r:eval "active_distributor <- paste0('Distributor_',active_distributors_index, '_y')"
        r:eval "formula <- paste(active_distributor, ' ~ ' ,'.')"
        r:eval "model <- elmtrain(as.formula(formula) , data=train, nhid=1, actfun='sig')"

        ;; create an empty variable for the next tick prediction
        r:eval "col_names <- colnames(train)"
        r:eval "predict_df <- create_predict_df(r_demand_history)"

        ;; predict the next tick's demand
        r:eval "prediction <- predict.elmNN(model, newdata = predict_df)"
        r:eval "prediction <- as.numeric(prediction)"
        ;; check if the predicted demand is not above the mean with a certain threshold
        if mean_limit_status = true [ r:eval "if (prediction > (mean_demand * mean_threshold)) { prediction <- mean_demand * mean_threshold}" ]

        ;; save the demand to NetLogo variable
        ask turtle ? [
        set forecasting r:get "prediction"
        ;; save the forecasting results and prepare order from supplier
        set forecast round forecasting
        set order_to_supplier forecast
        if order_to_supplier < 0 [set order_to_supplier 0]
        ifelse order_to_supplier + unfulfilled_demand < available_stock
          [ set order_to_supplier 0 ]
          [ set order_to_supplier (order_to_supplier + unfulfilled_demand - available_stock) ]
        ]
        ]
      ]
end

;; pareto distribution (pareto from R package: tdistrplus)
to-report random-pareto [alpha mm]
  report mm / ( random-float 1 ^ (1 / alpha) )
end

;; lognormal distribution (lnorm from R package: tdistrplus)
to-report  log-normal [mu sigma]
  let beta ln (1 + ((sigma ^ 2) / (mu ^ 2)))
  let x exp (random-normal (ln (mu) - (beta / 2)) sqrt beta)
  report x
end

;; log-logistics distribution (llogis from R package: tdistrplus)
to-report random-loglogistics [alpha beta]
    let r random-float 1
    let q r / (1 - r)
    let x alpha * exp (ln (q) / beta)
    report x
end

;; burr distribution (burr from R package: tdistrplus)
to-report random-burr [c k]
    let r random-float 1
    let q exp ( - ln(r) / k) - 1
    ;;if q < 0 [ set q (- q) ]
    let x exp ( ln(q) / c )
    report x
end

to select_demand_distribution
     ask turtles with [breed = households]
        [
     ;; the household decides how much to order based on distributions fitted on real-data.
     ;; if the market type is perfect competition it means that there are distributors who are not involved in sharing information
     ;; therefore some household randomly buy products from other distributors
     ;; if the market is oligopoly it means that all distributors from the market are involved in sharing information
     if market_type = "Oligopoly" [set distribution_type random (7 - 1) + 1]
     if market_type = "Perfect competition" [set distribution_type random (28 - 1) + 1]
     if distribution_type = 1  [ set order_to_distributor random-pareto 8.021208 10.829160  ]
     if distribution_type = 2  [ set order_to_distributor random-loglogistics 0.5279758 1.0251284 ]
     if distribution_type = 3  [ set order_to_distributor random-burr 0.8406488 2.1144042]
     if distribution_type = 4  [ set order_to_distributor random-burr 0.888256 1.807287]
     if distribution_type = 5  [ set order_to_distributor random-pareto 8.021208 10.829160 ]
     if distribution_type = 6  [ set order_to_distributor random-loglogistics 0.5594916 0.9898947 ]
     if distribution_type = 7  [ set order_to_distributor log-normal 0.5594916 0.9898947 ]
     if distribution_type > 7  [ set order_to_distributor 0 ]
   ]
end

to create-link
   ask turtles with [breed = households] [create-link-with one-of other turtles with [ breed = distributors ] ]
end

;;households randomly selects a distributor and sends him an order based on demand distribution type
 to interaction_household
   ask turtles with [breed = households]
   [
     ;; the distribution functions are obtained from ratio of real-data. The purpose of this is to simulate seasonality of the real-world environment.
     ;; the ratio is then multiplied by the product quantity which household buy. This way the customer is also distinguished by number of people in the household.
     set product_quantity random (Max_product_quantity - 1) + 1
     if order_to_distributor < 0 [ set order_to_distributor (- order_to_distributor)]
     set order_to_distributor round product_quantity * order_to_distributor
     let demand []
     set demand order_to_distributor
     set demand round demand

     ask link-neighbors with [ breed = distributors ] [

       set order_from_household demand
       set total_order_from_household (total_order_from_household + order_from_household)
     ]

   ]
 end

;; determines the mean demand
to check_mean_demand
  if mean_limit_status = True [
     ask turtles with [breed = distributors]
   [
     let distributors_index n-values count  turtles with [breed = distributors] [?]
     foreach distributors_index [
     r:put "mean_demand" [demand_history] of turtle ?
     r:eval "mean_demand <- mean(mean_demand)"
     r:eval "mean_demand <- as.numeric(mean_demand)"
     set mean_demand r:get "mean_demand"
     if mean_limit_status = true [ if total_order_from_household > mean_demand * mean_threshold [set total_order_from_household mean_demand * mean_threshold] ]
     ]
   ]
   ]
end

;; send orders to household by subtracting total order from household from available stock
 to update_inventory
   ask turtles with [breed = distributors]
     [
       ifelse unfulfilled_demand != 0
       [
         set available_stock (available_stock - total_order_from_household - unfulfilled_demand)
         set unfulfilled_demand 0
       ]
       [
         set available_stock (available_stock - total_order_from_household)
       ]

       ;; evaluate  the unfulfilled orders if the stock level was negative
       if available_stock < 0
      [
        set backlog (- available_stock)
        set total_backlog total_backlog + backlog
        set unfulfilled_demand total_backlog
      ]
     ]

 end

;; add a total demand variable to the list of demand history
 to update_demand_history
   ask turtles with [breed = distributors]
     [
       set demand_history lput total_order_from_household demand_history
       set forecasting_history lput forecast forecasting_history
       if ticks != 0 [
       set plot_forecast item (length(forecasting_history) - 2) forecasting_history
       ]
     ]

 end

;; reset save variables
 to reset_variables
   ask turtles with [ breed = distributors ]
   [
       set total_order_from_household 0
       set total_backlog 0

   ]
 end
@#$#@#$#@
GRAPHICS-WINDOW
317
7
940
651
16
16
18.6
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

SLIDER
25
10
311
43
Distributor_N
Distributor_N
2
5
2
1
1
NIL
HORIZONTAL

SLIDER
25
49
312
82
Households_N
Households_N
50
Distributor_N * 100
200
50
1
NIL
HORIZONTAL

SLIDER
25
89
313
122
Max_product_quantity
Max_product_quantity
1
10
10
1
1
NIL
HORIZONTAL

BUTTON
26
391
316
426
NIL
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
26
436
169
471
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
176
437
315
472
Step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
25
129
313
162
Share_information
Share_information
0
1
-1000

PLOT
955
12
1859
217
Stock level of supply chain network
Time
Stock level
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum([available_stock] of distributors)"
"pen-1" 1.0 0 -5298144 true "" "auto-plot-off\nplotxy 0 0\nplotxy 1000000000 0\nauto-plot-on"

PLOT
954
227
1860
430
Demand vs forecast
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Forecast" 1.0 0 -13791810 true "" "plot sum([plot_forecast] of distributors)"
"Demand" 1.0 0 -2674135 true "" "plot sum([total_order_from_household] of distributors)"
"axis = 0" 1.0 0 -13840069 true "" "auto-plot-off\nplotxy 0 0\nplotxy 1000000000 0\nauto-plot-on"

PLOT
955
445
1864
647
Backlog level
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum([total_backlog] of distributors)"

CHOOSER
25
170
313
215
market_type
market_type
"Oligopoly" "Perfect competition"
0

SLIDER
25
222
314
255
safety_level
safety_level
0
5
0
1
1
%
HORIZONTAL

MONITOR
179
484
313
529
Day demand
sum([total_order_from_household] of distributors)
0
1
11

MONITOR
27
483
169
528
Forecast
sum([forecast] of distributors)
0
1
11

MONITOR
25
593
314
638
Available stock of distributors
map [round ?] sort ([available_stock] of turtles with [ breed = distributors])
0
1
11

MONITOR
25
535
169
580
Order to supplier
sum([order_to_supplier] of distributors)
0
1
11

MONITOR
180
537
313
582
Available stock
sum([available_stock] of distributors)
0
1
11

SLIDER
26
261
313
294
mean_threshold
mean_threshold
1
10
5
1
1
NIL
HORIZONTAL

SWITCH
28
305
314
338
mean_limit_status
mean_limit_status
0
1
-1000

SWITCH
28
350
311
383
consumer_loyalty
consumer_loyalty
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

Supply chain resilience has gained more importance in recent years due to the trend for e-commerce. The industry has a tendency to sell products in low quantity, with high demand variation. To increase the resilience of the supply chain, demand forecasting is necessary. The model compares two approaches towards forecasting. One case focuses on forecasting only on individual distributor's history and another case considers the possibility to share information and forecast on all distributors' histories.

## HOW IT WORKS

In the beginining of every tick, household selects a distribution function and determines the ratio of order. The ratio informs the household to increase or lower their order. Than the ratio is multiplied by variation of product quantity (This approach is used to simulate different size of households). Every beginning of tick household sends an order to the distributor. The distributor fulfills the order the same tick and the stock level is updated. If the stock level is negative, the backlog is evaluated. Than distributors must order products from suppliers, however there is 1 tick delay for product delivery from supplier. Therefore, the demand is forecasted in advanced and supplier delivery are made in advanced. If there is a backlog, it is being added to the order to the supplier during the next tick, to increase the stock level.

## HOW TO USE IT

You can set the distributors quantity and household quantity. A limit is set of how much agents you can have to decrease the limit of hardware.

Share_information is used to change the forecasting approach. Forecast on individual company's history (off), forecast on group companies' history (on).

Market_type = oligopoly. Is used to simulate a small market where only few distributors exists, in this case the households always places orders.
Market_type = perfect competition. The market type has multiple distributors, which are not involved in the simulation. Than the households sometimes does not buy any products assuming that they buy from other distributors, which information we do not have.

Safety_level is used to determine the necessary stock level, which must be held to have a possibility to deal with drastic demand increase. The main idea of the forecasting approach is to have as little stock as possible. The level is set to a percentege of previous 7 day demand history.

mean_treshold. Due to the random distributions and product quantity, sometimes the households all order the maximum amount at these times the demand drastically spikes, which disrupts the results of the model. Therefore, there is set a limit of how many times the demand and forecasting results can increase when comparing with the mean demand.

consumer_loyalty. If consumer loyalty is turned on, consumers selects only 1 distributor in the setup stage and always stays loyal to it. Of the consumer loyalty is turned off, every tick the consumers selects different distributor.

## THINGS TO NOTICE

Real companies' sales histories are fitted on distributions with R package "tdistrplus".

The forecasting is done with a machine learning approach, Extreme learning machine (ELM). The high variation in demand variation is also necessary to limit the ELM ability to adapt to the variation precisely. The ELM algorithm has quick neural hidden layer computation capabilities, however in temrs of precision is weak. This particular algorithm has been chosen due to processing power limitations.

When using the behaviour space it is important to set parallel running process to 1, because the r code is used to suite linear programming and not parallel.

## THINGS TO TRY
The main idea is to compare individual demand forecasting vs group demand forecasting in different context. Its recommended to leave mean sliders as default and change only: Distributor_N, Clients_N, share_information, safety_level, market_Type and consumer_loyalty.

Distributor_N and Clients_N can help to simulate different market size (e.g. Belgium vs USA).

Share_information helps to compare individual and group demand forecasting in the same context.

Safty_level is used to help limit the stock level. When the safety_level changes its important to evaluate the backlog level.

Market_type compares market type with many sellers (perfect competition) vs few sellers (oligopoly).

consumer_loyalty. This parameter simulates loyalty discounts and benefits vs random consumer behavior depending on lower price, availability or other criteria.

## EXTENDING THE MODEL

The model can be extended by introducing more product type variation, evaluating the costs of warehousing. The machine learning approach could also be used not only to make predictions, but also choose the safety_level and other variables on its own.

## NETLOGO FEATURES

I have used extension R to implement machine learning. NetLogo does not have built in futures.

## RELATED MODELS

http://modelingcommons.org/browse/one_model/3378#model_tabs_browse_info

## CREDITS AND REFERENCES

Extension R: https://ccl.northwestern.edu/netlogo/docs/r.html

Prediction method is not accurate, but has a very fast hidden layer evaluation, which is suitable for simulations with less computation power: http://www.ntu.edu.sg/home/egbhuang/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

house two story
false
0
Polygon -7500403 true true 2 180 227 180 152 150 32 150
Rectangle -7500403 true true 270 75 285 255
Rectangle -7500403 true true 75 135 270 255
Rectangle -16777216 true false 124 195 187 256
Rectangle -16777216 true false 210 195 255 240
Rectangle -16777216 true false 90 150 135 180
Rectangle -16777216 true false 210 150 255 180
Line -16777216 false 270 135 270 255
Rectangle -7500403 true true 15 180 75 255
Polygon -7500403 true true 60 135 285 135 240 90 105 90
Line -16777216 false 75 135 75 180
Rectangle -16777216 true false 30 195 93 240
Line -16777216 false 60 135 285 135
Line -16777216 false 255 105 285 135
Line -16777216 false 0 180 75 180
Line -7500403 true 60 195 60 240
Line -7500403 true 154 195 154 255

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Low market size" repetitions="3" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = 365</exitCondition>
    <metric>sum([available_stock] of distributors)</metric>
    <metric>sum([forecast] of distributors)</metric>
    <metric>sum([total_order_from_household] of distributors)</metric>
    <metric>sum([total_backlog] of distributors)</metric>
    <enumeratedValueSet variable="consumer_loyalty">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Distributor_N">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Households_N">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Share_information">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean_treshold">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="market_type">
      <value value="&quot;Oligopoly&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="safety_level">
      <value value="0"/>
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_product_quantity">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean_limit_status">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="High market size" repetitions="3" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = 365</exitCondition>
    <metric>sum([available_stock] of distributors)</metric>
    <metric>sum([forecast] of distributors)</metric>
    <metric>sum([total_order_from_household] of distributors)</metric>
    <metric>sum([total_backlog] of distributors)</metric>
    <enumeratedValueSet variable="consumer_loyalty">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Distributor_N">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Households_N">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Share_information">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean_treshold">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="market_type">
      <value value="&quot;Oligopoly&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="safety_level">
      <value value="0"/>
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_product_quantity">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean_limit_status">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
