globals [
  cam-x
  cam-y
  cam-z

  cam-yaw
  cam-pitch

  cam-fov
]

patches-own [
  patch-xcor
  patch-ycor
  patch-zcor

  patch-xvel
  patch-yvel
  patch-zvel

  patch-hits

  patch-is-done
]

turtles-own [
  world-xcor
  world-ycor
  world-zcor
  world-size
]

;;; CAMERA ;;;

to reset-camera
  set cam-x 0
  set cam-y 0
  set cam-z 0
  set cam-yaw 0
  set cam-pitch 0
  reset-patches
end

to move-camera [ dist ]
  set cam-x (cam-x + dist * (sin cam-yaw) * (cos cam-pitch))
  set cam-y (cam-y + dist * (sin cam-pitch))
  set cam-z (cam-z + dist * (cos cam-yaw) * (cos cam-pitch))
  reset-patches
end

to move-camera-up [ dist ]
  set cam-y (cam-y + dist)
  reset-patches
end

to move-camera-sideways [ dist ]
  set cam-x (cam-x + dist * (sin (cam-yaw + 90)))
  set cam-z (cam-z + dist * (cos (cam-yaw + 90)))
  reset-patches
end

to-report loop-angle [ ang ]
  let looped-ang (((ang mod 360) + 3600) mod 360)
  if (looped-ang > 180)
  [ set looped-ang (looped-ang - 360) ]
  report looped-ang
end

to turn-camera-yaw [ ang ]
  set cam-yaw (loop-angle (cam-yaw + ang))
  reset-patches

end

to turn-camera-pitch [ ang ]
  set cam-pitch (loop-angle (cam-pitch + ang))
  reset-patches
end

;;; PATCHES ;;;

to reset-patches
  ; camera must be above ground
  set cam-y (abs cam-y)

  ; direction camera is facing
  let start-xvel ((sin cam-yaw) * (cos cam-pitch))
  let start-yvel (sin cam-pitch)
  let start-zvel ((cos cam-yaw) * (cos cam-pitch))

  ; amount that the camera must turn due to fov
  let delt (tan (cam-fov / 2))

  ; amount that each of the axis
  ; will turn due to change in yaw
  let yaw-xdelt (delt * (sin (90 + cam-yaw)))
  let yaw-ydelt (0)
  let yaw-zdelt (delt * (cos (90 + cam-yaw)))

  ; amount that each of the axis
  ; will turn due to change in pitch
  let pitch-xdelt (delt * (0 - sin cam-yaw) * (sin cam-pitch))
  let pitch-ydelt (delt * (cos cam-pitch))
  let pitch-zdelt (delt * (0 - cos cam-yaw) * (sin cam-pitch))

  ; amount that the patches scale themself too
  let div-cor ((max-pxcor + max-pycor) / 2)

  ask patches [
    ; move ray to camera
    set patch-xcor (cam-x)
    set patch-ycor (cam-y)
    set patch-zcor (cam-z)

    ; add yaw and pitch to ray
    set patch-xvel (start-xvel + (yaw-xdelt * pxcor + pitch-xdelt * pycor) / div-cor)
    set patch-yvel (start-yvel + (yaw-ydelt * pxcor + pitch-ydelt * pycor) / div-cor)
    set patch-zvel (start-zvel + (yaw-zdelt * pxcor + pitch-zdelt * pycor) / div-cor)

    ; reset hits
    set patch-hits (0)

    ; activate-patch
    set patch-is-done false

    ; render really quickly
    move-patch
    update-patch-color
  ]
end

to-report distance-xyz [ x y z ]
  ; report pythagorean theorem
  report sqrt (x * x + y * y + z * z)
end

to-report patch-distancexyz [ x y z ]
  ; take distance of each axis
  let xdist (x - patch-xcor)
  let ydist (y - patch-ycor)
  let zdist (z - patch-zcor)

  ; report relative distance
  report (distance-xyz xdist ydist zdist)
end

to-report turtle-distancexyz [ x y z ]
  ; take distance of each axis
  let xdist (x - world-xcor)
  let ydist (y - world-ycor)
  let zdist (z - world-zcor)

  ; report relative distance
  report (distance-xyz xdist ydist zdist)
end

to-report is-touching-turtle [ x y z ]
  ; take distance of each axis
  let xdist (x - world-xcor)
  let ydist (y - world-ycor)
  let zdist (z - world-zcor)

  ; test if two are touching without sqrt
  report ((xdist * xdist + ydist * ydist + zdist * zdist) <= (world-size * world-size))
end

to-report max-step-size [ x y z xv yv zv ]
  ; take distance of each axis
  let xoc (x - world-xcor)
  let yoc (y - world-ycor)
  let zoc (z - world-zcor)

  ; calculate roots
  let a (xv * xv + yv * yv + zv * zv)
  let b (2 * (xoc * xv + yoc * yv + zoc * zv))
  let c ((xoc * xoc + yoc * yoc + zoc * zoc) - (world-size * world-size))

  ; get square root part of equation
  let step-size (b * b - 4 * a * c)

  ; if root is negative return big number
  if (step-size < 0) [ report 65536 ]

  ; get entire equation
  set step-size ((0 - (b + (sqrt step-size))) / (2 * a))

  ; if answer is negative return really big number
  if (step-size < 0) [ report 65536 ]

  ; return answer
  report step-size
end

to normalize-patch
  ; get distance of velocity
  let mag (distance-xyz patch-xvel patch-yvel patch-zvel)

  ; scale velocity down
  set patch-xvel (patch-xvel / mag)
  set patch-yvel (patch-yvel / mag)
  set patch-zvel (patch-zvel / mag)
end

to move-patch
  if (not patch-is-done) [
    ; normalize the velocitys
    normalize-patch

    ; push data into local memory
    let local-xcor (patch-xcor)
    let local-ycor (patch-ycor)
    let local-zcor (patch-zcor)
    let local-xvel (patch-xvel)
    let local-yvel (patch-yvel)
    let local-zvel (patch-zvel)
    let local-hits (patch-hits)
    let local-is-done (false)

    ; get largest steps size possible
    let floor-distance (65535)

    ; if patch is moving down, limit step size
    if (local-yvel < 0) [
      set patch-is-done false
      let scaled-ycor (abs (local-ycor / local-yvel))
      set floor-distance scaled-ycor
    ]

    ; calculate the closest turtle to the ray
    let closest-turtle (min-one-of turtles [
      max-step-size local-xcor local-ycor local-zcor
                    local-xvel local-yvel local-zvel
    ])

    ; do reflection calculation on closest turtle
    ask closest-turtle [
      ; get the distance from the ray to the turtle
      let dist (max-step-size local-xcor local-ycor local-zcor local-xvel local-yvel local-zvel)

      ; check which is closer, the turtle or floor
      ifelse(dist < floor-distance) [
        ; move the ray to the surface of the circle
        set local-xcor (local-xcor + local-xvel * dist)
        set local-ycor (local-ycor + local-yvel * dist)
        set local-zcor (local-zcor + local-zvel * dist)

        ; calculate dot product of the
        let normal-dist (turtle-distancexyz local-xcor local-ycor local-zcor)
        let normal-x ((local-xcor - world-xcor) / normal-dist)
        let normal-y ((local-ycor - world-ycor) / normal-dist)
        let normal-z ((local-zcor - world-zcor) / normal-dist)
        let dot (normal-x * local-xvel + normal-y * local-yvel + normal-z * local-zvel)

        ; reflect the ray off the turtle
        set local-xvel (local-xvel - 2 * dot * normal-x)
        set local-yvel (local-yvel - 2 * dot * normal-y)
        set local-zvel (local-zvel - 2 * dot * normal-z)

        ; increase the amount of hits
        set local-hits (local-hits + 1)
      ] [
        ; only bounce off floor when moving down
        ifelse (local-yvel < 0) [
          ; move ray to the floor
          set local-xcor (local-xcor + local-xvel * floor-distance)
          set local-ycor (local-ycor + local-yvel * floor-distance)
          set local-zcor (local-zcor + local-zvel * floor-distance)

          ; bounce the ray off the floor
          set local-ycor (abs local-ycor)
          set local-yvel (abs local-yvel)

          ; floor patch settings
          let square-size 20
          let floor-hits 2.5
          let square-diff 0.5

          ; calculate if it is a bright or dark square
          if-else (
            ((local-xcor mod square-size) <= (square-size / 2)) !=
            ((local-zcor mod square-size) <= (square-size / 2)))
          [ set local-hits (local-hits + floor-hits - square-diff) ]
          [ set local-hits (local-hits + floor-hits + square-diff) ]
        ] [
          ; if everything is done, tell ray that it is done
          set local-is-done (true)
        ]
      ]
    ]

    ; push data into patch memory
    set patch-xcor (local-xcor)
    set patch-ycor (local-ycor)
    set patch-zcor (local-zcor)
    set patch-xvel (local-xvel)
    set patch-yvel (local-yvel)
    set patch-zvel (local-zvel)
    set patch-hits (local-hits)
    set patch-is-done (local-is-done)
  ]
end

to-report cap-rgb [ val ]
  ; cap value from 0 - 255
  if (val < 000) [ report 000 ]
  if (val > 255) [ report 255 ]
  report val
end

to update-patch-color
  ; copy amount of hits into local memory
  let local-hits (patch-hits)

  ; if patch is traveling down, simulate hit
  if (patch-yvel < 0) [ set local-hits (local-hits + 3) ]

  ; get angle of the ray for rainbow color
  let angle 270
  if (rainbow and ((patch-xvel != 0) or (patch-zvel != 0)))
  [ set angle ((atan patch-xvel patch-zvel) * (2)) ]

  ; get pitch of the ray
  let xz-mag (sqrt (patch-xvel * patch-xvel + patch-zvel * patch-zvel))
  let hit-scale (0.618 ^ local-hits)
  let sky-scale (0.25 + 1.6 * (cos (atan xz-mag (abs patch-yvel))))

  ; turn angles into rainbow and shadows
  let r (128 + 128 * (sin (angle + 000)))
  let g (128 + 128 * (sin (angle + 120)))
  let b (128 + 128 * (sin (angle + 240)))

  ; darken colors due to sky
  set r (r * sky-scale)
  set g (g * sky-scale)
  set b (b * sky-scale)

  ; average color with hits
  set r (r * hit-scale + circle-r * (1 - hit-scale))
  set g (g * hit-scale + circle-g * (1 - hit-scale))
  set b (b * hit-scale + circle-b * (1 - hit-scale))

  ; set color to the calculated color
  set pcolor (list (cap-rgb r) (cap-rgb g) (cap-rgb b))
end

to update-progress
  set render-progress (round (1000 * (count patches with [ patch-is-done ]) / (count patches)) / 10)
end

to turtle-init [ x y z in-size]
  set hidden? true
  set world-xcor x
  set world-ycor y
  set world-zcor z
  set world-size in-size
end

;;; TEST ;;;

to setup
  ca

  ; create the spheres
  cro 1 [ turtle-init 0   12  10  3 ]
  cro 1 [ turtle-init -48 12  10  3 ]
  cro 1 [ turtle-init -24 40  10  3 ]
  cro 1 [ turtle-init -24 16  16  6 ]
  cro 1 [ turtle-init 0   24  50  25 ]
  cro 1 [ turtle-init -48 24  50  24 ]
  cro 1 [ turtle-init 0   50  150 36 ]
  cro 1 [ turtle-init -72 50  150 36 ]
  reset-camera

  ; setup camera
  set cam-x -30
  set cam-y 42
  set cam-z -40
  set cam-pitch -15
  set cam-yaw 7
  set cam-fov 90
  set fov cam-fov

  reset-patches
end

to go
  ; if fov changes, draw new image
  if (fov != cam-fov) [
    set cam-fov (fov)
    reset-patches
  ]

  ; cycle through update groups
  ask patches [
    move-patch
    update-patch-color
  ]

  ; update progress bar
  update-progress

  display
end
@#$#@#$#@
GRAPHICS-WINDOW
7
10
738
502
-1
-1
3.0
1
10
1
1
1
0
0
0
1
-120
120
-80
80
1
1
0
ticks
60.0

BUTTON
774
49
949
82
Reset
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
774
86
949
119
Render
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1134
52
1320
85
Forwards
move-camera movement-speed
NIL
1
T
OBSERVER
NIL
W
NIL
NIL
1

BUTTON
1206
237
1262
270
Right
turn-camera-yaw 10\n
NIL
1
T
OBSERVER
NIL
L
NIL
NIL
1

BUTTON
1076
237
1131
270
Left
turn-camera-yaw -10
NIL
1
T
OBSERVER
NIL
J
NIL
NIL
1

BUTTON
1134
87
1320
120
Backwards
move-camera (0 - movement-speed)
NIL
1
T
OBSERVER
NIL
S
NIL
NIL
1

SLIDER
769
173
957
206
fov
fov
1
180
90.0
1
1
NIL
HORIZONTAL

BUTTON
1133
202
1204
235
Up
turn-camera-pitch 10\n
NIL
1
T
OBSERVER
NIL
I
NIL
NIL
1

BUTTON
1133
237
1204
270
Down
turn-camera-pitch -10\n
NIL
1
T
OBSERVER
NIL
K
NIL
NIL
1

BUTTON
1228
122
1320
155
Move Right
move-camera-sideways movement-speed
NIL
1
T
OBSERVER
NIL
D
NIL
NIL
1

BUTTON
1134
122
1226
155
Move Left
move-camera-sideways (0 - movement-speed)
NIL
1
T
OBSERVER
NIL
A
NIL
NIL
1

SLIDER
769
208
957
241
movement-speed
movement-speed
0
10
1.0
0.1
1
NIL
HORIZONTAL

BUTTON
1032
68
1121
101
Fly Up
move-camera-up movement-speed
NIL
1
T
OBSERVER
NIL
Q
NIL
NIL
1

BUTTON
1032
103
1121
136
Fly Down
move-camera-up (0 - movement-speed)
NIL
1
T
OBSERVER
NIL
E
NIL
NIL
1

MONITOR
769
317
831
362
NIL
cam-x
17
1
11

MONITOR
834
317
891
362
NIL
cam-y
17
1
11

MONITOR
894
317
955
362
NIL
cam-z
17
1
11

MONITOR
769
365
860
410
NIL
cam-pitch
17
1
11

MONITOR
863
365
955
410
NIL
cam-yaw
17
1
11

SWITCH
769
243
957
276
rainbow
rainbow
0
1
-1000

SLIDER
745
469
1325
502
render-progress
render-progress
0
100
100.0
1
1
% done
HORIZONTAL

TEXTBOX
772
24
998
62
=== Initialization ===
16
0.0
1

TEXTBOX
786
149
962
187
=== Settings ===
16
0.0
1

TEXTBOX
1076
24
1299
62
=== Movement ===
16
0.0
1

TEXTBOX
1094
177
1244
196
=== Camera ===
16
0.0
1

TEXTBOX
775
293
962
314
=== Debug Info ===
16
0.0
1

SLIDER
1089
319
1261
352
circle-r
circle-r
0
255
32.0
1
1
NIL
HORIZONTAL

SLIDER
1089
356
1261
389
circle-g
circle-g
0
255
32.0
1
1
NIL
HORIZONTAL

SLIDER
1089
393
1261
426
circle-b
circle-b
0
255
32.0
1
1
NIL
HORIZONTAL

TEXTBOX
1071
294
1301
312
=== Customization ===
16
0.0
1

TEXTBOX
879
442
1426
460
=== Richard's Progress Bar I Stole === 
16
0.0
1

@#$#@#$#@
## WHAT IS IT?

This is a ray tracer that is implemented completely within NetLogo. There is obviously no hardware acceleration in netlogo so the raytracing is done in software. 

## HOW IT WORKS

The ray tracing is done by having the light work its way backwards out of the camera. So you can simulate the reflections off of circles in a really realistic way. Many optimizaitons had to be made when creating this ray tracer.   

## HOW TO USE IT

Click the reset button, then click the render button to start the rendering process. 

It is best that you leave the render button running at all time as everything can be interacted with while it is running.

WASD is used to move the camera forwards and backwards. 
E moves you up
W moves you down
IJKL is used to move the camera around

You can try changing the movement speed if you feel like you are moving too slowly.

The FOV slider will let you change how much you can see, but it can create distortions in size around the edges if set too high.

Also change the rainbow setting to your liking.

If the rendering is too slow, or too pixelated, you can manually change the patch size and the program will scale accordingly.

## THINGS TO TRY

If you would like to try adding in another circle type this command

`cro 1 [ turtle-init (NEW-X) (NEW-Y) (NEW-Z) (NEW-SIZE) ]`

and move around to see it update

## EXTENDING THE MODEL

You can attempt to add new shapes, but circles demonstrate the way ray tracers work quite well.

## CREDITS AND REFERENCES

Richard Lee) for inventing the idea of a progress bar. really helpful
Ray Tracing in a Weekend) formula for reflections and formula for finding point of intersection
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

real car
true
0
Rectangle -7500403 true true 90 75 210 150
Polygon -7500403 true true 90 75 75 150 90 150 90 75
Polygon -7500403 true true 210 75 225 150 210 150 210 75
Rectangle -16777216 true false 105 90 195 120
Rectangle -7500403 true true 90 150 120 195
Rectangle -7500403 true true 180 150 210 195
Circle -7500403 true true 84 159 42
Circle -7500403 true true 174 159 42

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
