<?xml version="1.0" encoding="utf-8"?>
<model version="NetLogo 7.0.0" snapToGrid="false">
  <code><![CDATA[breed [ agents an-agent ]
breed [ cops cop ]

globals [
  k                   ; factor for determining arrest probability
  ;; threshold           ; by how much must G > N to make someone rebel?
]

agents-own [
  risk-aversion       ; R, fixed for the agent's lifetime, ranging from 0-1 (inclusive)
  perceived-hardship  ; H, also ranging from 0-1 (inclusive)
  active?             ; if true, then the agent is actively rebelling
  jail-term           ; how many turns in jail remain? (if 0, the agent is not in jail)
]

patches-own [
  neighborhood        ; surrounding patches within the vision radius
]

to setup
  clear-all

  ; set globals
  set k 2.3
  ;; set threshold 0.1

  ask patches [
    ; make background a slightly dark gray
    set pcolor gray - 1
    ; cache patch neighborhoods
    set neighborhood patches in-radius vision
  ]

  if initial-cop-density + initial-agent-density > 100 [
    user-message (word
      "The sum of INITIAL-COP-DENSITY and INITIAL-AGENT-DENSITY "
      "should not be greater than 100.")
    stop
  ]

  ; create cops
  create-cops round (initial-cop-density * .01 * count patches) [
    move-to one-of patches with [ not any? turtles-here ]
    display-cop
  ]

  ; create agents
  create-agents round (initial-agent-density * .01 * count patches) [
    move-to one-of patches with [ not any? turtles-here ]
    set heading 0
    set risk-aversion random-float 1.0
    set perceived-hardship max list 0 min list 1 random-float (1 - life-conditions)
    set active? false
    set jail-term 0
    display-agent
  ]

  ; start clock and plot initial state of system
  reset-ticks
end

to go
  ask turtles [
    ; Rule M: Move to a random site within your vision
    if (breed = agents and jail-term = 0) or breed = cops [ move ]
    ;   Rule A: Determine if each agent should be active or quiet
    if breed = agents and jail-term = 0 [ determine-behavior ]
    ;  Rule C: Cops arrest a random active agent within their radius
    if breed = cops [ enforce ]
  ]
  ; Jailed agents get their term reduced at the end of each clock tick
  ask agents [ if jail-term > 0 [ set jail-term jail-term - 1 ] ]
  ; update agent display
  ask agents [ display-agent ]
  ask cops [ display-cop ]
  ; advance clock and update plots
  tick
end

; AGENT AND COP BEHAVIOR

; move to an empty patch
to move ; turtle procedure
  if movement? or breed = cops [
    ; move to a patch in vision; candidate patches are
    ; empty or contain only jailed agents
    let targets neighborhood with [
      not any? cops-here and all? agents-here [ jail-term > 0 ]
    ]
    if any? targets [ move-to one-of targets ]
  ]
end

; AGENT BEHAVIOR

to determine-behavior
  set active? (grievance - risk-aversion * estimated-arrest-probability > threshold)
end

to-report grievance
  report perceived-hardship * (1 - government-legitimacy)
end

to-report estimated-arrest-probability
  let c count cops-on neighborhood
  let a 1 + count (agents-on neighborhood) with [ active? ]
  ; See Info tab for a discussion of the following formula
  report 1 - exp (- k * floor (c / a))
end

; COP BEHAVIOR

to enforce
  if any? (agents-on neighborhood) with [ active? ] [
    ; arrest suspect
    let suspect one-of (agents-on neighborhood) with [ active? ]
    move-to suspect  ; move to patch of the jailed agent
    ask suspect [
      set active? false
      set jail-term random max-jail-term
    ]
  ]
end

; VISUALIZATION OF AGENTS AND COPS

to display-agent  ; agent procedure
   display-agent-2d
end

to display-agent-2d  ; agent procedure
  set shape "circle"
  ifelse active?
    [ set color red ]
    [ ifelse jail-term > 0
        [ set color black + 3 ]
        [ set color scale-color green grievance 1.5 -0.5 ] ]
end

to display-cop
  set color cyan
  set shape "triangle"
end


; Copyright 2004 Uri Wilensky.
; See Info tab for full copyright and license.]]></code>
  <widgets>
    <view x="363" wrappingAllowedX="true" y="10" frameRate="30.0" minPycor="0" height="400" showTickCounter="true" patchSize="10.0" fontSize="10" wrappingAllowedY="true" width="400" tickCounterLabel="ticks" maxPycor="39" updateMode="1" maxPxcor="39" minPxcor="0"></view>
    <slider x="9" step="1" y="144" max="10" width="205" display="vision" height="50" min="1" direction="Horizontal" default="7.0" variable="vision" units="patches"></slider>
    <slider x="8" step="0.1" y="32" max="20.0" width="205" display="initial-cop-density" height="50" min="0.0" direction="Horizontal" default="4.0" variable="initial-cop-density" units="%"></slider>
    <slider x="8" step="1.0" y="87" max="100.0" width="205" display="initial-agent-density" height="50" min="0.0" direction="Horizontal" default="70.0" variable="initial-agent-density" units="%"></slider>
    <note x="9" y="10" backgroundDark="0" fontSize="11" width="90" markdown="false" height="18" textColorDark="-1" textColorLight="-16777216" backgroundLight="0">Initial settings</note>
    <plot x="361" autoPlotX="true" yMax="150.0" autoPlotY="true" yAxis="agents" y="417" xMin="0.0" height="200" legend="true" xMax="20.0" yMin="0.0" width="412" xAxis="time" display="All agent types">
      <setup></setup>
      <update></update>
      <pen interval="1.0" mode="0" display="quiet" color="-10899396" legend="true">
        <setup></setup>
        <update>plot count agents with [not active? and jail-term = 0]</update>
      </pen>
      <pen interval="1.0" mode="0" display="jailed" color="-16777216" legend="true">
        <setup></setup>
        <update><![CDATA[plot count agents with [jail-term > 0]]]></update>
      </pen>
      <pen interval="1.0" mode="0" display="active" color="-2674135" legend="true">
        <setup></setup>
        <update>plot count agents with [active?]</update>
      </pen>
      <pen interval="1.0" mode="0" display="ripe" color="-13345367" legend="true">
        <setup></setup>
        <update><![CDATA[plot count agents with [grievance > threshold and not active?]]]></update>
      </pen>
    </plot>
    <monitor x="218" precision="3" y="30" height="60" fontSize="11" width="100" display="# of cops">count cops</monitor>
    <monitor x="217" precision="3" y="88" height="60" fontSize="11" width="100" display="# of agents">count agents</monitor>
    <slider x="11" step="0.1" y="201" max="0.5" width="201" display="life-conditions" height="50" min="-0.5" direction="Horizontal" default="0.0" variable="life-conditions"></slider>
    <button x="14" y="263" height="40" disableUntilTicks="false" forever="false" kind="Observer" width="70">setup</button>
    <button x="85" y="263" height="40" disableUntilTicks="true" forever="true" kind="Observer" width="70">go</button>
    <slider x="10" step="0.01" y="370" max="1.0" width="200" display="government-legitimacy" height="50" min="0.0" direction="Horizontal" default="0.9" variable="government-legitimacy"></slider>
    <monitor x="213" precision="1" y="525" height="60" fontSize="11" width="100" display="jailed (black)"><![CDATA[count agents with [jail-term > 0]]]></monitor>
    <monitor x="112" precision="3" y="525" height="60" fontSize="11" width="100" display="active (red)">count agents with [active?]</monitor>
    <monitor x="9" precision="1" y="525" height="60" fontSize="11" width="100" display="quiet (green)">count agents with [not active? and jail-term = 0]</monitor>
    <switch x="10" y="480" height="40" on="false" variable="movement?" width="139" display="movement?"></switch>
    <slider x="10" step="1.0" y="425" max="50.0" width="200" display="max-jail-term" height="50" min="0.0" direction="Horizontal" default="20.0" variable="max-jail-term" units="ticks"></slider>
    <slider x="8" step="0.01" y="316" max="0.2" width="205" display="threshold" height="50" min="-0.2" direction="Horizontal" default="0.1" variable="threshold"></slider>
  </widgets>
  <info><![CDATA[## WHAT IS IT?

This project models the rebellion of a subjugated population against a central authority. It is is an adaptation of Joshua Epstein's model of civil violence (2002).

The population wanders around randomly. If their level of grievance against the central authority is high enough, and their perception of the risks involved is low enough, they openly rebel. A separate population of police officers ("cops"), acting on behalf of the central authority, seeks to suppress the rebellion. The cops wander around randomly and arrest people who are actively rebelling.

## HOW IT WORKS

Each "agent," or member of the general population, has an individual level of grievance toward the central authority. GRIEVANCE is based on the agent's PERCEIVED-HARDSHIP, which is assigned randomly at startup, and on GOVERNMENT-LEGITIMACY, which is global across agents and specified by a slider in the interface.

Each agent also calculates an individual risk of rebelling at the beginning of each turn.  This ESTIMATED-ARREST-PROBABILITY, is based on the number of cops and already rebelling agents within VISION patches, namely 1 - exp (- k * (C/A)<sub>v</sub>) --- where (C/A)<sub>v</sub> is the ratio of cops to active agents, and k is a constant set in "startup" to ensure a reasonable value when there is only one cop and one agent within a particular field of vision.  In our implementation, we changed one aspect of Epstein's description.  After dividing by C by A, we take the "floor" of the result (that is, round downwards to an integer).  Without this change, the model does not exhibit punctuated equilibrium.  The effect of the change is that if there are more rebels than cops in the neighborhood, the probability of arrest is zero, otherwise it is very nearly 1.0.  In other words, the rule could be written more simply as:

      ifelse a > c [ report 0 ] [ report 0.99 ]

An agent's NET-RISK is the product of the calculated ESTIMATED-ARREST-PROBABILITY and RISK-AVERSION randomly set for each agent at birth.

The model cycles through three different rules, known as M (move), A (agents) and C (cops):

- Movement rule M says that each cop and non-imprisoned agent moves to a random unoccupied site within VISION patches.

- Agent rule A says that if an agent's GRIEVANCE exceeds the NET-RISK by a small threshold (set by default to 0.1 in "setup"), the agent decides to rebel.  This is noted by changing the agent's border color from blue to red.  (It is also possible for the agent to move from rebellion to quiet, in which case the border color changes from red to blue.)

- Cop rule C says that each cop should look for active agents within VISION patches.  If at least one exists, the cop randomly selects one active agent and sends it to jail for a number of turns between 0 and MAX-JAIL-TERM (settable by a slider in the interface) and moves to the patch of the jailed agent. The patch of the jailed agent is considered unoccupied.

## HOW TO USE IT

Use the sliders to pick the initial settings for the model. INITIAL-COP-DENSITY and POPULATION-DENSITY respectively determine the density of cops and agents in the world. VISION determines the number of patches in each direction that agents and cops can see.

Click SETUP to initialize the population. Click GO to begin the simulation.

As the simulation runs, you can adjust GOVERNMENT-LEGITIMACY and MAX-JAIL-TERM. You can control the mobility of the agents through the MOVEMENT? toggle.

The visualization chooser allows you to choose between 2D visualization where the turtles are circles and triangles or 3D visualization where the agents have people shapes.

The color of the agents shows their private level of grievance. The darker the green, the higher the GRIEVANCE.  This color does not change throughout the run of a model (unless you change the GOVERNMENT-LEGITIMACY slider).

## THINGS TO NOTICE

Watch how agents exhibit deceptive behavior by changing their public state when cops are present, despite a constant level of private grievance. To see this press the WATCH ONE button, slow down the model, switch to the 3D view and zoom in on the selected agent. It may be easier to see this phenomenon if you change the sliders, especially reducing the VISION.

Watch the plot of active agents. Notice that it exhibits "punctuated equilibrium" --- periods of quiescence followed by periods of rebellion.

## THINGS TO TRY

Epstein claims that movement of the agents causes more frequent episodes of rebellion. Run the model with the MOVEMENT? switch on and off to see if you get this result.

Compare the effects of a gradual erosion in the perceived legitimacy of the central authority to a smaller, but immediate drop. For example, set GOVERNMENT-LEGITIMACY to 0.9 and reduce it to zero over 250 runs, plotting the legitimacy and the number of active agents. Now do another run, beginning with legitimacy at 0.9. This time, leave the legitimacy constant for 80 runs and then drop it to 0.7 at once. Observe the results. Try to explain them. Think of historical examples of this phenomena. What are the implications of this result for the strategy of revolutionary leaders?

Epstein quotes de Tocqueville: "It is not always when things are going from bad to worse that revolutions break out. On the contrary, it oftener happens that when a people that has put up with oppressive rule over a long period without protest suddenly finds the government relaxing its pressure, it takes up arms against it." Test de Tocqueville's observation in the model by comparing the effects of a gradual erosion in perceived legitimacy with a gradual erosion in the number of cops. Observe the results. Explain the difference.

## EXTENDING THE MODEL

Write a reporter procedure that reports `true` when there is a rebellion, `false` during quiescent periods.

Measure the intervals between episodes of rebellion. What is the distribution of these waiting times?

Try changing some of Epstein's assumptions as below. Do these changes create qualitative differences in results?

Change GOVERNMENT-LEGITIMACY such that it increases proportionally with the number of nearby jailed agents.

Change the model such that each agent's `grievance` is influenced by the value of other nearby agents.

## NETLOGO FEATURES

Note the use of a patch variable to store precomputed neighborhood agentsets.  This improves performance because the neighborhoods don't have to be calculated over and over again.

## CREDITS AND REFERENCES

This model, and the preceding explanation, is adapted from Joshua M. Epstein, "Modeling civil violence: An agent-based computational approach", Proceedings of the National Academy of Sciences, Vol. 99, Suppl. 3, May 14, 2002, and is available at
https://www.ncbi.nlm.nih.gov/pmc/articles/PMC128592/.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. (2004).  NetLogo Rebellion model.  http://ccl.northwestern.edu/netlogo/models/Rebellion.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2004 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

This model was created as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227.

<!-- 2004 -->]]></info>
  <turtleShapes>
    <shape name="default" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="5"></point>
        <point x="40" y="250"></point>
        <point x="150" y="205"></point>
        <point x="260" y="250"></point>
      </polygon>
    </shape>
    <shape name="airplane" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="0"></point>
        <point x="135" y="15"></point>
        <point x="120" y="60"></point>
        <point x="120" y="105"></point>
        <point x="15" y="165"></point>
        <point x="15" y="195"></point>
        <point x="120" y="180"></point>
        <point x="135" y="240"></point>
        <point x="105" y="270"></point>
        <point x="120" y="285"></point>
        <point x="150" y="270"></point>
        <point x="180" y="285"></point>
        <point x="210" y="270"></point>
        <point x="165" y="240"></point>
        <point x="180" y="180"></point>
        <point x="285" y="195"></point>
        <point x="285" y="165"></point>
        <point x="180" y="105"></point>
        <point x="180" y="60"></point>
        <point x="165" y="15"></point>
      </polygon>
    </shape>
    <shape name="arrow" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="0"></point>
        <point x="0" y="150"></point>
        <point x="105" y="150"></point>
        <point x="105" y="293"></point>
        <point x="195" y="293"></point>
        <point x="195" y="150"></point>
        <point x="300" y="150"></point>
      </polygon>
    </shape>
    <shape name="box" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="285"></point>
        <point x="285" y="225"></point>
        <point x="285" y="75"></point>
        <point x="150" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="135"></point>
        <point x="15" y="75"></point>
        <point x="150" y="15"></point>
        <point x="285" y="75"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="15" y="75"></point>
        <point x="15" y="225"></point>
        <point x="150" y="285"></point>
        <point x="150" y="135"></point>
      </polygon>
      <line endX="150" startY="285" marked="false" color="255" endY="135" startX="150"></line>
      <line endX="15" startY="135" marked="false" color="255" endY="75" startX="150"></line>
      <line endX="285" startY="135" marked="false" color="255" endY="75" startX="150"></line>
    </shape>
    <shape name="bug" rotatable="true" editableColorIndex="0">
      <circle x="96" y="182" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="110" y="127" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <circle x="110" y="75" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <line endX="80" startY="100" marked="true" color="-1920102913" endY="30" startX="150"></line>
      <line endX="220" startY="100" marked="true" color="-1920102913" endY="30" startX="150"></line>
    </shape>
    <shape name="butterfly" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="165"></point>
        <point x="209" y="199"></point>
        <point x="225" y="225"></point>
        <point x="225" y="255"></point>
        <point x="195" y="270"></point>
        <point x="165" y="255"></point>
        <point x="150" y="240"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="165"></point>
        <point x="89" y="198"></point>
        <point x="75" y="225"></point>
        <point x="75" y="255"></point>
        <point x="105" y="270"></point>
        <point x="135" y="255"></point>
        <point x="150" y="240"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="139" y="148"></point>
        <point x="100" y="105"></point>
        <point x="55" y="90"></point>
        <point x="25" y="90"></point>
        <point x="10" y="105"></point>
        <point x="10" y="135"></point>
        <point x="25" y="180"></point>
        <point x="40" y="195"></point>
        <point x="85" y="194"></point>
        <point x="139" y="163"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="162" y="150"></point>
        <point x="200" y="105"></point>
        <point x="245" y="90"></point>
        <point x="275" y="90"></point>
        <point x="290" y="105"></point>
        <point x="290" y="135"></point>
        <point x="275" y="180"></point>
        <point x="260" y="195"></point>
        <point x="215" y="195"></point>
        <point x="162" y="165"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="255"></point>
        <point x="135" y="225"></point>
        <point x="120" y="150"></point>
        <point x="135" y="120"></point>
        <point x="150" y="105"></point>
        <point x="165" y="120"></point>
        <point x="180" y="150"></point>
        <point x="165" y="225"></point>
      </polygon>
      <circle x="135" y="90" marked="false" color="255" diameter="30" filled="true"></circle>
      <line endX="195" startY="105" marked="false" color="255" endY="60" startX="150"></line>
      <line endX="105" startY="105" marked="false" color="255" endY="60" startX="150"></line>
    </shape>
    <shape name="car" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="300" y="180"></point>
        <point x="279" y="164"></point>
        <point x="261" y="144"></point>
        <point x="240" y="135"></point>
        <point x="226" y="132"></point>
        <point x="213" y="106"></point>
        <point x="203" y="84"></point>
        <point x="185" y="63"></point>
        <point x="159" y="50"></point>
        <point x="135" y="50"></point>
        <point x="75" y="60"></point>
        <point x="0" y="150"></point>
        <point x="0" y="165"></point>
        <point x="0" y="225"></point>
        <point x="300" y="225"></point>
        <point x="300" y="180"></point>
      </polygon>
      <circle x="180" y="180" marked="false" color="255" diameter="90" filled="true"></circle>
      <circle x="30" y="180" marked="false" color="255" diameter="90" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="162" y="80"></point>
        <point x="132" y="78"></point>
        <point x="134" y="135"></point>
        <point x="209" y="135"></point>
        <point x="194" y="105"></point>
        <point x="189" y="96"></point>
        <point x="180" y="89"></point>
      </polygon>
      <circle x="47" y="195" marked="true" color="-1920102913" diameter="58" filled="true"></circle>
      <circle x="195" y="195" marked="true" color="-1920102913" diameter="58" filled="true"></circle>
    </shape>
    <shape name="circle" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
    </shape>
    <shape name="circle 2" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
    </shape>
    <shape name="cow" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="200" y="193"></point>
        <point x="197" y="249"></point>
        <point x="179" y="249"></point>
        <point x="177" y="196"></point>
        <point x="166" y="187"></point>
        <point x="140" y="189"></point>
        <point x="93" y="191"></point>
        <point x="78" y="179"></point>
        <point x="72" y="211"></point>
        <point x="49" y="209"></point>
        <point x="48" y="181"></point>
        <point x="37" y="149"></point>
        <point x="25" y="120"></point>
        <point x="25" y="89"></point>
        <point x="45" y="72"></point>
        <point x="103" y="84"></point>
        <point x="179" y="75"></point>
        <point x="198" y="76"></point>
        <point x="252" y="64"></point>
        <point x="272" y="81"></point>
        <point x="293" y="103"></point>
        <point x="285" y="121"></point>
        <point x="255" y="121"></point>
        <point x="242" y="118"></point>
        <point x="224" y="167"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="73" y="210"></point>
        <point x="86" y="251"></point>
        <point x="62" y="249"></point>
        <point x="48" y="208"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="25" y="114"></point>
        <point x="16" y="195"></point>
        <point x="9" y="204"></point>
        <point x="23" y="213"></point>
        <point x="25" y="200"></point>
        <point x="39" y="123"></point>
      </polygon>
    </shape>
    <shape name="cylinder" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
    </shape>
    <shape name="dot" rotatable="false" editableColorIndex="0">
      <circle x="90" y="90" marked="true" color="-1920102913" diameter="120" filled="true"></circle>
    </shape>
    <shape name="face happy" rotatable="false" editableColorIndex="0">
      <circle x="8" y="8" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="255"></point>
        <point x="90" y="239"></point>
        <point x="62" y="213"></point>
        <point x="47" y="191"></point>
        <point x="67" y="179"></point>
        <point x="90" y="203"></point>
        <point x="109" y="218"></point>
        <point x="150" y="225"></point>
        <point x="192" y="218"></point>
        <point x="210" y="203"></point>
        <point x="227" y="181"></point>
        <point x="251" y="194"></point>
        <point x="236" y="217"></point>
        <point x="212" y="240"></point>
      </polygon>
    </shape>
    <shape name="face neutral" rotatable="false" editableColorIndex="0">
      <circle x="8" y="7" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <rectangle endX="240" startY="195" marked="false" color="255" endY="225" startX="60" filled="true"></rectangle>
    </shape>
    <shape name="face sad" rotatable="false" editableColorIndex="0">
      <circle x="8" y="8" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="168"></point>
        <point x="90" y="184"></point>
        <point x="62" y="210"></point>
        <point x="47" y="232"></point>
        <point x="67" y="244"></point>
        <point x="90" y="220"></point>
        <point x="109" y="205"></point>
        <point x="150" y="198"></point>
        <point x="192" y="205"></point>
        <point x="210" y="220"></point>
        <point x="227" y="242"></point>
        <point x="251" y="229"></point>
        <point x="236" y="206"></point>
        <point x="212" y="183"></point>
      </polygon>
    </shape>
    <shape name="fish" rotatable="false" editableColorIndex="0">
      <polygon color="-1" filled="true" marked="false">
        <point x="44" y="131"></point>
        <point x="21" y="87"></point>
        <point x="15" y="86"></point>
        <point x="0" y="120"></point>
        <point x="15" y="150"></point>
        <point x="0" y="180"></point>
        <point x="13" y="214"></point>
        <point x="20" y="212"></point>
        <point x="45" y="166"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="135" y="195"></point>
        <point x="119" y="235"></point>
        <point x="95" y="218"></point>
        <point x="76" y="210"></point>
        <point x="46" y="204"></point>
        <point x="60" y="165"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="75" y="45"></point>
        <point x="83" y="77"></point>
        <point x="71" y="103"></point>
        <point x="86" y="114"></point>
        <point x="166" y="78"></point>
        <point x="135" y="60"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="30" y="136"></point>
        <point x="151" y="77"></point>
        <point x="226" y="81"></point>
        <point x="280" y="119"></point>
        <point x="292" y="146"></point>
        <point x="292" y="160"></point>
        <point x="287" y="170"></point>
        <point x="270" y="195"></point>
        <point x="195" y="210"></point>
        <point x="151" y="212"></point>
        <point x="30" y="166"></point>
      </polygon>
      <circle x="215" y="106" marked="false" color="255" diameter="30" filled="true"></circle>
    </shape>
    <shape name="flag" rotatable="false" editableColorIndex="0">
      <rectangle endX="75" startY="15" marked="true" color="-1920102913" endY="300" startX="60" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="90" y="150"></point>
        <point x="270" y="90"></point>
        <point x="90" y="30"></point>
      </polygon>
      <line endX="90" startY="135" marked="true" color="-1920102913" endY="135" startX="75"></line>
      <line endX="90" startY="45" marked="true" color="-1920102913" endY="45" startX="75"></line>
    </shape>
    <shape name="flower" rotatable="false" editableColorIndex="0">
      <polygon color="1504722175" filled="true" marked="false">
        <point x="135" y="120"></point>
        <point x="165" y="165"></point>
        <point x="180" y="210"></point>
        <point x="180" y="240"></point>
        <point x="150" y="300"></point>
        <point x="165" y="300"></point>
        <point x="195" y="240"></point>
        <point x="195" y="195"></point>
        <point x="165" y="135"></point>
      </polygon>
      <circle x="85" y="132" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="130" y="147" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="192" y="85" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="85" y="40" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="177" y="40" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="177" y="132" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="70" y="85" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="130" y="25" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="96" y="51" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="113" y="68" marked="false" color="255" diameter="74" filled="true"></circle>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="189" y="233"></point>
        <point x="219" y="188"></point>
        <point x="249" y="173"></point>
        <point x="279" y="188"></point>
        <point x="234" y="218"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="180" y="255"></point>
        <point x="150" y="210"></point>
        <point x="105" y="210"></point>
        <point x="75" y="240"></point>
        <point x="135" y="240"></point>
      </polygon>
    </shape>
    <shape name="house" rotatable="false" editableColorIndex="0">
      <rectangle endX="255" startY="120" marked="true" color="-1920102913" endY="285" startX="45" filled="true"></rectangle>
      <rectangle endX="180" startY="210" marked="false" color="255" endY="285" startX="120" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="15" y="120"></point>
        <point x="150" y="15"></point>
        <point x="285" y="120"></point>
      </polygon>
      <line endX="270" startY="120" marked="false" color="255" endY="120" startX="30"></line>
    </shape>
    <shape name="leaf" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="210"></point>
        <point x="135" y="195"></point>
        <point x="120" y="210"></point>
        <point x="60" y="210"></point>
        <point x="30" y="195"></point>
        <point x="60" y="180"></point>
        <point x="60" y="165"></point>
        <point x="15" y="135"></point>
        <point x="30" y="120"></point>
        <point x="15" y="105"></point>
        <point x="40" y="104"></point>
        <point x="45" y="90"></point>
        <point x="60" y="90"></point>
        <point x="90" y="105"></point>
        <point x="105" y="120"></point>
        <point x="120" y="120"></point>
        <point x="105" y="60"></point>
        <point x="120" y="60"></point>
        <point x="135" y="30"></point>
        <point x="150" y="15"></point>
        <point x="165" y="30"></point>
        <point x="180" y="60"></point>
        <point x="195" y="60"></point>
        <point x="180" y="120"></point>
        <point x="195" y="120"></point>
        <point x="210" y="105"></point>
        <point x="240" y="90"></point>
        <point x="255" y="90"></point>
        <point x="263" y="104"></point>
        <point x="285" y="105"></point>
        <point x="270" y="120"></point>
        <point x="285" y="135"></point>
        <point x="240" y="165"></point>
        <point x="240" y="180"></point>
        <point x="270" y="195"></point>
        <point x="240" y="210"></point>
        <point x="180" y="210"></point>
        <point x="165" y="195"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="195"></point>
        <point x="135" y="240"></point>
        <point x="120" y="255"></point>
        <point x="105" y="255"></point>
        <point x="105" y="285"></point>
        <point x="135" y="285"></point>
        <point x="165" y="240"></point>
        <point x="165" y="195"></point>
      </polygon>
    </shape>
    <shape name="line" rotatable="true" editableColorIndex="0">
      <line endX="150" startY="0" marked="true" color="-1920102913" endY="300" startX="150"></line>
    </shape>
    <shape name="line half" rotatable="true" editableColorIndex="0">
      <line endX="150" startY="0" marked="true" color="-1920102913" endY="150" startX="150"></line>
    </shape>
    <shape name="pentagon" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="15"></point>
        <point x="15" y="120"></point>
        <point x="60" y="285"></point>
        <point x="240" y="285"></point>
        <point x="285" y="120"></point>
      </polygon>
    </shape>
    <shape name="person" rotatable="false" editableColorIndex="0">
      <circle x="110" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <rectangle endX="172" startY="79" marked="true" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="195" y="90"></point>
        <point x="240" y="150"></point>
        <point x="225" y="180"></point>
        <point x="165" y="105"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="60" y="150"></point>
        <point x="75" y="180"></point>
        <point x="135" y="105"></point>
      </polygon>
    </shape>
    <shape name="person active" rotatable="false" editableColorIndex="0">
      <polygon color="764264191" filled="true" marked="false">
        <point x="135" y="90"></point>
        <point x="150" y="105"></point>
        <point x="135" y="165"></point>
        <point x="150" y="180"></point>
        <point x="165" y="165"></point>
        <point x="150" y="105"></point>
        <point x="165" y="90"></point>
      </polygon>
      <polygon color="-684578305" filled="true" marked="false">
        <point x="195" y="135"></point>
        <point x="240" y="30"></point>
        <point x="210" y="15"></point>
        <point x="165" y="120"></point>
      </polygon>
      <circle x="110" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <rectangle endX="172" startY="79" marked="true" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="-684578305" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <polygon color="-684578305" filled="true" marked="false">
        <point x="105" y="135"></point>
        <point x="60" y="30"></point>
        <point x="90" y="15"></point>
        <point x="135" y="120"></point>
      </polygon>
      <polygon color="-1653716737" filled="true" marked="false">
        <point x="195" y="15"></point>
        <point x="270" y="60"></point>
        <point x="270" y="75"></point>
        <point x="195" y="30"></point>
      </polygon>
    </shape>
    <shape name="person jailed" rotatable="false" editableColorIndex="0">
      <circle x="110" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <rectangle endX="172" startY="79" marked="true" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="255" filled="true" marked="false">
        <point x="195" y="90"></point>
        <point x="210" y="150"></point>
        <point x="195" y="180"></point>
        <point x="165" y="105"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="90" y="150"></point>
        <point x="105" y="180"></point>
        <point x="135" y="105"></point>
      </polygon>
    </shape>
    <shape name="person quiet" rotatable="false" editableColorIndex="0">
      <polygon color="764264191" filled="true" marked="false">
        <point x="135" y="90"></point>
        <point x="150" y="105"></point>
        <point x="135" y="165"></point>
        <point x="150" y="180"></point>
        <point x="165" y="165"></point>
        <point x="150" y="105"></point>
        <point x="165" y="90"></point>
      </polygon>
      <polygon color="-303222273" filled="true" marked="false">
        <point x="195" y="90"></point>
        <point x="240" y="195"></point>
        <point x="210" y="210"></point>
        <point x="165" y="105"></point>
      </polygon>
      <circle x="110" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <rectangle endX="172" startY="79" marked="true" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="-303222273" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="100" y="210"></point>
        <point x="130" y="225"></point>
        <point x="145" y="165"></point>
        <point x="85" y="135"></point>
        <point x="63" y="189"></point>
      </polygon>
      <polygon color="764264191" filled="true" marked="false">
        <point x="90" y="210"></point>
        <point x="120" y="225"></point>
        <point x="135" y="165"></point>
        <point x="67" y="130"></point>
        <point x="53" y="189"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="120" y="224"></point>
        <point x="131" y="225"></point>
        <point x="124" y="210"></point>
      </polygon>
      <line endX="126" startY="168" marked="false" color="255" endY="225" startX="139"></line>
      <line endX="76" startY="167" marked="false" color="255" endY="136" startX="140"></line>
      <polygon color="-303222273" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="60" y="195"></point>
        <point x="90" y="210"></point>
        <point x="135" y="105"></point>
      </polygon>
    </shape>
    <shape name="person soldier" rotatable="false" editableColorIndex="10">
      <rectangle endX="172" startY="79" marked="false" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="878553599" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="60" y="195"></point>
        <point x="90" y="210"></point>
        <point x="135" y="105"></point>
      </polygon>
      <polygon color="878553599" filled="true" marked="true">
        <point x="195" y="90"></point>
        <point x="240" y="195"></point>
        <point x="210" y="210"></point>
        <point x="165" y="105"></point>
      </polygon>
      <circle x="110" y="5" marked="false" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="878553599" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <polygon color="-1653716737" filled="true" marked="false">
        <point x="120" y="90"></point>
        <point x="105" y="90"></point>
        <point x="180" y="195"></point>
        <point x="180" y="165"></point>
      </polygon>
      <line endX="139" startY="105" marked="false" color="-1653716737" endY="105" startX="109"></line>
      <line endX="151" startY="125" marked="false" color="-1653716737" endY="117" startX="122"></line>
      <line endX="159" startY="143" marked="false" color="-1653716737" endY="134" startX="137"></line>
      <line endX="181" startY="179" marked="false" color="-1653716737" endY="158" startX="158"></line>
      <line endX="169" startY="160" marked="false" color="-1653716737" endY="146" startX="146"></line>
      <rectangle endX="180" startY="193" marked="false" color="-1653716737" endY="201" startX="120" filled="true"></rectangle>
      <polygon color="-1653716737" filled="true" marked="false">
        <point x="122" y="4"></point>
        <point x="107" y="16"></point>
        <point x="102" y="39"></point>
        <point x="105" y="53"></point>
        <point x="148" y="34"></point>
        <point x="192" y="27"></point>
        <point x="189" y="17"></point>
        <point x="172" y="2"></point>
        <point x="145" y="0"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="183" y="90"></point>
        <point x="240" y="15"></point>
        <point x="247" y="22"></point>
        <point x="193" y="90"></point>
      </polygon>
      <rectangle endX="128" startY="187" marked="false" color="-1653716737" endY="208" startX="114" filled="true"></rectangle>
      <rectangle endX="191" startY="187" marked="false" color="-1653716737" endY="208" startX="177" filled="true"></rectangle>
    </shape>
    <shape name="plant" rotatable="false" editableColorIndex="0">
      <rectangle endX="165" startY="90" marked="true" color="-1920102913" endY="300" startX="135" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="255"></point>
        <point x="90" y="210"></point>
        <point x="45" y="195"></point>
        <point x="75" y="255"></point>
        <point x="135" y="285"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="255"></point>
        <point x="210" y="210"></point>
        <point x="255" y="195"></point>
        <point x="225" y="255"></point>
        <point x="165" y="285"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="180"></point>
        <point x="90" y="135"></point>
        <point x="45" y="120"></point>
        <point x="75" y="180"></point>
        <point x="135" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="180"></point>
        <point x="165" y="210"></point>
        <point x="225" y="180"></point>
        <point x="255" y="120"></point>
        <point x="210" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="105"></point>
        <point x="90" y="60"></point>
        <point x="45" y="45"></point>
        <point x="75" y="105"></point>
        <point x="135" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="105"></point>
        <point x="165" y="135"></point>
        <point x="225" y="105"></point>
        <point x="255" y="45"></point>
        <point x="210" y="60"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="90"></point>
        <point x="120" y="45"></point>
        <point x="150" y="15"></point>
        <point x="180" y="45"></point>
        <point x="165" y="90"></point>
      </polygon>
    </shape>
    <shape name="square" rotatable="false" editableColorIndex="0">
      <rectangle endX="270" startY="30" marked="true" color="-1920102913" endY="270" startX="30" filled="true"></rectangle>
    </shape>
    <shape name="square 2" rotatable="false" editableColorIndex="0">
      <rectangle endX="270" startY="30" marked="true" color="-1920102913" endY="270" startX="30" filled="true"></rectangle>
      <rectangle endX="240" startY="60" marked="false" color="255" endY="240" startX="60" filled="true"></rectangle>
    </shape>
    <shape name="star" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="151" y="1"></point>
        <point x="185" y="108"></point>
        <point x="298" y="108"></point>
        <point x="207" y="175"></point>
        <point x="242" y="282"></point>
        <point x="151" y="216"></point>
        <point x="59" y="282"></point>
        <point x="94" y="175"></point>
        <point x="3" y="108"></point>
        <point x="116" y="108"></point>
      </polygon>
    </shape>
    <shape name="target" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
      <circle x="60" y="60" marked="true" color="-1920102913" diameter="180" filled="true"></circle>
      <circle x="90" y="90" marked="false" color="255" diameter="120" filled="true"></circle>
      <circle x="120" y="120" marked="true" color="-1920102913" diameter="60" filled="true"></circle>
    </shape>
    <shape name="tree" rotatable="false" editableColorIndex="0">
      <circle x="118" y="3" marked="true" color="-1920102913" diameter="94" filled="true"></circle>
      <rectangle endX="180" startY="195" marked="false" color="-1653716737" endY="300" startX="120" filled="true"></rectangle>
      <circle x="65" y="21" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="116" y="41" marked="true" color="-1920102913" diameter="127" filled="true"></circle>
      <circle x="45" y="90" marked="true" color="-1920102913" diameter="120" filled="true"></circle>
      <circle x="104" y="74" marked="true" color="-1920102913" diameter="152" filled="true"></circle>
    </shape>
    <shape name="triangle" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="15"></point>
        <point x="0" y="270"></point>
        <point x="300" y="270"></point>
      </polygon>
    </shape>
    <shape name="triangle 2" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="30"></point>
        <point x="15" y="255"></point>
        <point x="285" y="255"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="151" y="99"></point>
        <point x="225" y="223"></point>
        <point x="75" y="224"></point>
      </polygon>
    </shape>
    <shape name="truck" rotatable="false" editableColorIndex="0">
      <rectangle endX="195" startY="45" marked="true" color="-1920102913" endY="187" startX="4" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="296" y="193"></point>
        <point x="296" y="150"></point>
        <point x="259" y="134"></point>
        <point x="244" y="104"></point>
        <point x="208" y="104"></point>
        <point x="207" y="194"></point>
      </polygon>
      <rectangle endX="195" startY="60" marked="false" color="-1" endY="105" startX="195" filled="true"></rectangle>
      <polygon color="255" filled="true" marked="false">
        <point x="238" y="112"></point>
        <point x="252" y="141"></point>
        <point x="219" y="141"></point>
        <point x="218" y="112"></point>
      </polygon>
      <circle x="234" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <rectangle endX="214" startY="185" marked="true" color="-1920102913" endY="194" startX="181" filled="true"></rectangle>
      <circle x="144" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <circle x="24" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <circle x="24" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
      <circle x="144" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
      <circle x="234" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
    </shape>
    <shape name="turtle" rotatable="true" editableColorIndex="0">
      <polygon color="1504722175" filled="true" marked="false">
        <point x="215" y="204"></point>
        <point x="240" y="233"></point>
        <point x="246" y="254"></point>
        <point x="228" y="266"></point>
        <point x="215" y="252"></point>
        <point x="193" y="210"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="195" y="90"></point>
        <point x="225" y="75"></point>
        <point x="245" y="75"></point>
        <point x="260" y="89"></point>
        <point x="269" y="108"></point>
        <point x="261" y="124"></point>
        <point x="240" y="105"></point>
        <point x="225" y="105"></point>
        <point x="210" y="105"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="75" y="75"></point>
        <point x="55" y="75"></point>
        <point x="40" y="89"></point>
        <point x="31" y="108"></point>
        <point x="39" y="124"></point>
        <point x="60" y="105"></point>
        <point x="75" y="105"></point>
        <point x="90" y="105"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="132" y="85"></point>
        <point x="134" y="64"></point>
        <point x="107" y="51"></point>
        <point x="108" y="17"></point>
        <point x="150" y="2"></point>
        <point x="192" y="18"></point>
        <point x="192" y="52"></point>
        <point x="169" y="65"></point>
        <point x="172" y="87"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="85" y="204"></point>
        <point x="60" y="233"></point>
        <point x="54" y="254"></point>
        <point x="72" y="266"></point>
        <point x="85" y="252"></point>
        <point x="107" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="119" y="75"></point>
        <point x="179" y="75"></point>
        <point x="209" y="101"></point>
        <point x="224" y="135"></point>
        <point x="220" y="225"></point>
        <point x="175" y="261"></point>
        <point x="128" y="261"></point>
        <point x="81" y="224"></point>
        <point x="74" y="135"></point>
        <point x="88" y="99"></point>
      </polygon>
    </shape>
    <shape name="wheel" rotatable="false" editableColorIndex="0">
      <circle x="3" y="3" marked="true" color="-1920102913" diameter="294" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
      <line endX="150" startY="285" marked="true" color="-1920102913" endY="15" startX="150"></line>
      <line endX="285" startY="150" marked="true" color="-1920102913" endY="150" startX="15"></line>
      <circle x="120" y="120" marked="true" color="-1920102913" diameter="60" filled="true"></circle>
      <line endX="79" startY="40" marked="true" color="-1920102913" endY="269" startX="216"></line>
      <line endX="269" startY="84" marked="true" color="-1920102913" endY="221" startX="40"></line>
      <line endX="269" startY="216" marked="true" color="-1920102913" endY="79" startX="40"></line>
      <line endX="221" startY="40" marked="true" color="-1920102913" endY="269" startX="84"></line>
    </shape>
    <shape name="x" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="270" y="75"></point>
        <point x="225" y="30"></point>
        <point x="30" y="225"></point>
        <point x="75" y="270"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="30" y="75"></point>
        <point x="75" y="30"></point>
        <point x="270" y="225"></point>
        <point x="225" y="270"></point>
      </polygon>
    </shape>
  </turtleShapes>
  <linkShapes>
    <shape name="default" curviness="0.0">
      <lines>
        <line x="-0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
        <line x="0.0" visible="true">
          <dash value="1.0"></dash>
          <dash value="0.0"></dash>
        </line>
        <line x="0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
      </lines>
      <indicator>
        <shape name="link direction" rotatable="true" editableColorIndex="0">
          <line endX="90" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
          <line endX="210" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
        </shape>
      </indicator>
    </shape>
  </linkShapes>
  <previewCommands>setup
repeat 5 [ go ]</previewCommands>
</model>
