globals [
  left-side                                 ;; left side of the membrane
  right-side                                ;; right side of the membrane
  split                                     ;; location of membrane
  pre-split                                 ;; a value for calculating the change in membrane location
  equilibrium                               ;; the difference in number of particles moving each direction across membrane
  tick-delta                                ;; how much we advance the tick counter this time through
  collisions        ;; list used to keep track of future collisions
  particle1         ;; first particle currently colliding
  particle2         ;; second particle currently colliding
]

breed [particles particle]
breed [membranes membrane]

turtles-own [
  speed
  mass
  energy
  old-pos
  new-pos
  part-type
]

to setup
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  clear-all
  reset-ticks
  set-default-shape particles "circle"
  create-container
  spawn-particles
  set split 0
  calculate-wall
  set particle1 nobody
  set particle2 nobody
  set collisions []
  ask particles [ check-for-wall-collision ]
  ask particles [ check-for-particle-collision ]
  my-update-plots
end

to create-container
  ask patches with [pycor = max-pycor or pycor = min-pycor] [              ;; sets the walls of the container
    set pcolor blue
  ]
  ask patches with [pxcor = max-pxcor or pxcor = min-pxcor] [
    set pcolor blue
  ]
  set left-side patches with [pxcor < 0]                                   ;; defines the left and right sides of the container equally
  set right-side patches with [pxcor > 0]
  set equilibrium 0                                                        ;; sets equilibrium to starting value (the middle of the container)
  set split 0                                                              ;; sets the starting location of the membrane
  ask patches with [pxcor = 0 and pycor != max-pycor and pycor != min-pycor and abs pycor mod 2 != 0] [
    sprout-membranes 1 [                                                   ;; create the membrane
      set shape "square"
      set color red
      set size 1.3
      set heading 90
    ]
  ]
end

to setup-particles                                                      ;; set variable values for each new particle
  set speed 0.5
  if part-type = "solvent" [
    set size 1
  ]
  if part-type = "solute" [
    set size 2
  ]
  set mass (size * size)
  set energy (0.5 * mass * speed * speed)
end

to-report overlapping?  ;; particle procedure
  ;; here, we use IN-RADIUS just for improved speed; the real testing
  ;; is done by DISTANCE
  report any? other turtles in-radius ((size + 2) / 2)
                              with [distance myself < (size + [size] of myself) / 2]
end

to-report wall-lapping?  ;; particle procedure
  report any? patches in-radius ((size + 2) / 2) with [pcolor = blue] or
  any? membranes in-radius ((size + 2) / 2)
end

to start-loc
  move-to one-of patches with [pcolor = black]                        ;; randomly distribute them throughout the world
    while [overlapping?] [
      move-to one-of patches with [pcolor = black]
    ]
    while [wall-lapping?] [
      move-to one-of patches with [pcolor = black]
    ]
end

to start-right-loc
  move-to one-of right-side with [pcolor = black]                        ;; randomly distribute them throughout the world
    while [overlapping?] [
      move-to one-of right-side with [pcolor = black]
    ]
    while [wall-lapping?] [
      move-to one-of right-side with [pcolor = black]
    ]
end

to start-left-loc
  move-to one-of left-side with [pcolor = black]                        ;; randomly distribute them throughout the world
    while [overlapping?] [
      move-to one-of left-side with [pcolor = black]
    ]
    while [wall-lapping?] [
      move-to one-of left-side with [pcolor = black]
    ]
end

to spawn-particles
  if solute = "Sugar" [                                                ;; checks to see the identity of the solute based on the chooser
    output-type "C12H22O11"                                            ;; write the chemical formula in the output area
    create-particles solute-left [                                     ;; create solute particles based on slider value
      set color white
      set shape "circle"
      set part-type "solute"                                        ;; define these as solute particles
      setup-particles
      start-left-loc
    ]
    create-particles solute-right [                                    ;; create solute particles based on slider value
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-right-loc
    ]
  ]

  if solute = "Sodium Chloride" [
    output-type "NaCl"
    create-particles solute-left [
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-left-loc
      ask patch-here [
        sprout-particles 1 [                                   ;; solutes that dissociate in water split into their respective ions
          set color white                                      ;; since NaCl splits into two ions, one new turtle is sprouted for each solute molecule
          set shape "circle"
          set part-type "solute"
          setup-particles
          fd 0.2
          ]
      ]
    ]
    create-particles solute-right [
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-right-loc
      ask patch-here [
        sprout-particles 1 [
          set color white
          set shape "circle"
          set part-type "solute"
          setup-particles
          fd 0.2
          ]
      ]
    ]
  ]

  if solute = "Magnesium Chloride" [
    output-type "MgCl2"
    create-particles solute-left [
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-left-loc
      ask patch-here [
        sprout-particles 2 [                                        ;; splits into 3 ions..so 2 new turtles are sprouted
          set color white
          set shape "circle"
          set part-type "solute"
          setup-particles
          fd 0.2
          ]
      ]
    ]
    create-particles solute-right [
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-right-loc
      ask patch-here [
        sprout-particles 2 [
          set color white
          set shape "circle"
          set part-type "solute"
          setup-particles
          fd 0.2
          ]
      ]
    ]
  ]

  if solute = "Aluminum Chloride" [
    output-type "AlCl3"
    create-particles solute-left [
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-left-loc
      ask patch-here [
        sprout-particles 3 [                                     ;; splits into 4 ions...so 3 new turtles are sprouted
          set color white
          set shape "circle"
          set part-type "solute"
          setup-particles
          fd 0.2
          ]
      ]
    ]
    create-particles solute-right [
      set color white
      set shape "circle"
      set part-type "solute"
      setup-particles
      start-right-loc
      ask patch-here [
        sprout-particles 3 [
          set color white
          set shape "circle"
          set part-type "solute"
          setup-particles
          fd 0.2
          ]
      ]
    ]
  ]
  create-particles (100 - count particles with [part-type = "solute" and xcor < split]) [                                               ;; creates 1000 water molecules
    set color blue + 2
    set part-type "solvent"                                             ;; define these as solvents
    setup-particles
    start-left-loc
  ]
  create-particles (100 - count particles with [part-type = "solute" and xcor > split]) [                                               ;; creates 1000 water molecules
    set color blue + 2
    set part-type "solvent"                                             ;; define these as solvents
    setup-particles
    start-right-loc
  ]
end

to particle-jump
  ask particles with [xcor >= pre-split and xcor <= split and part-type = "solutes"] [     ;; check for solutes in the way of a membrane jump
    set xcor (split - pre-split) + 1
;    recalculate-particles-that-just-collided
  ]
  ask particles with [xcor <= pre-split and xcor >= split and part-type = "solutes"] [
    set xcor (pre-split - split) + 1
;    recalculate-particles-that-just-collided
  ]
end

to calculate-wall
  set pre-split split                                                  ;; save location of the membrane before it moves
  let nudge ((equilibrium * -1) / 10)                                  ;; calculates the amount to move the membrane
  set split split + nudge                                              ;; set split to be the new location of the membrane
  particle-jump                                                        ;; move solutes in the way of the membrane jump
  ask membranes [
    set xcor xcor + nudge                                              ;; move membrane turtles according to nudge value
  ]
  
  set left-side patches with [pxcor < split]                           ;; redefine right and left sides
  set right-side patches with [pxcor > split]
  set equilibrium 0                               ;; reset equilibrium so we can keep track of what happens during the next tick
end

to go
  if count turtles-on right-side = 0 or count turtles-on left-side = 0 [                    ;; stops model if membrane reaches either edge
    user-message "The membrane has burst! Make sure you have some solute on both sides!"
    stop
  ]
  ask particles [ set old-pos xcor ]
  choose-next-collision

  ask particles with [ part-type = "solute" ] [      ;; bouncing for solutes hitting membrane
    if any? membranes in-cone 2 180 [
      set heading (- heading)
    ]
  ]

  ask particles [ jump speed * tick-delta ]
  perform-next-collision

  ask particles with [ part-type = "solvent" ] [
    if old-pos < split and xcor >= split [                      ;; if a solvent moves from the left of the membrane to the right of the membrane
      set equilibrium equilibrium + 1                           ;; add one to equilibrium
    ]
    if old-pos > split and xcor <= split [                      ;; if a solvent moves from the right of the membrane to the left of the membrane
      set equilibrium equilibrium - 1                           ;; subract one from equilibrium
    ]
  ]

  calculate-wall                                                ;; recalculate the new location of the wall
  tick-advance tick-delta
  recalculate-particles-that-just-collided
  my-update-plots
end

to recalculate-particles-that-just-collided
  ;; Since only collisions involving the particles that collided most recently could be affected,
  ;; we filter those out of collisions.  Then we recalculate all possible collisions for
  ;; the particles that collided last.  The ifelse statement is necessary because
  ;; particle2 can be either a particle or a string representing a wall.  If it is a
  ;; wall, we don't want to invalidate all collisions involving that wall (because the wall's
  ;; position wasn't affected, those collisions are still valid.
  ifelse is-turtle? particle2
    [
      set collisions filter [item 1 ? != particle1 and
                             item 2 ? != particle1 and
                             item 1 ? != particle2 and
                             item 2 ? != particle2]
                            collisions
      ask particle2 [ check-for-wall-collision ]
      ask particle2 [ check-for-particle-collision ]
    ]
    [
      set collisions filter [item 1 ? != particle1 and
                             item 2 ? != particle1]
                            collisions
    ]
  if particle1 != nobody [ ask particle1 [ check-for-wall-collision ] ]
  if particle1 != nobody [ ask particle1 [ check-for-particle-collision ] ]
  ;; Slight errors in floating point math can cause a collision that just
  ;; happened to be calculated as happening again a very tiny amount of
  ;; time into the future, so we remove any collisions that involves
  ;; the same two particles (or particle and wall) as last time.
  set collisions filter [item 1 ? != particle1 or
                         item 2 ? != particle2]
                        collisions
  ;; All done.
  set particle1 nobody
  set particle2 nobody
end

;; check-for-particle-collision is a particle procedure that determines the time it takes
;; to the collision between two particles (if one exists).  It solves for the time by representing
;; the equations of motion for distance, velocity, and time in a quadratic equation of the vector
;; components of the relative velocities and changes in position between the two particles and
;; solves for the time until the next collision
to check-for-particle-collision
  let my-x xcor
  let my-y ycor
  let my-particle-size size
  let my-x-speed speed * sin heading
  let my-y-speed speed * cos heading
  ask other particles
  [
    let dpx (xcor - my-x)   ;; relative distance between particles in the x direction
    let dpy (ycor - my-y)    ;; relative distance between particles in the y direction
    let x-speed (speed * sin heading) ;; speed of other particle in the x direction
    let y-speed (speed * cos heading) ;; speed of other particle in the x direction
    let dvx (x-speed - my-x-speed) ;; relative speed difference between particles in x direction
    let dvy (y-speed - my-y-speed) ;; relative speed difference between particles in y direction
    let sum-r (((my-particle-size) / 2 ) + (([size] of self) / 2 )) ;; sum of both particle radii

    ;; To figure out what the difference in position (P1) between two particles at a future
    ;; time (t) will be, one would need to know the current difference in position (P0) between the
    ;; two particles and the current difference in the velocity (V0) between the two particles.
    ;;
    ;; The equation that represents the relationship is:
    ;;   P1 = P0 + t * V0
    ;; we want find when in time (t), P1 would be equal to the sum of both the particle's radii
    ;; (sum-r).  When P1 is equal to is equal to sum-r, the particles will just be touching each
    ;; other at their edges (a single point of contact).
    ;;
    ;; Therefore we are looking for when:   sum-r =  P0 + t * V0
    ;;
    ;; This equation is not a simple linear equation, since P0 and V0 should both have x and y
    ;; components in their two dimensional vector representation (calculated as dpx, dpy, and
    ;; dvx, dvy).
    ;;
    ;; By squaring both sides of the equation, we get:
    ;;   (sum-r) * (sum-r) =  (P0 + t * V0) * (P0 + t * V0)
    ;; When expanded gives:
    ;;   (sum-r ^ 2) = (P0 ^ 2) + (t * PO * V0) + (t * PO * V0) + (t ^ 2 * VO ^ 2)
    ;; Which can be simplified to:
    ;;   0 = (P0 ^ 2) - (sum-r ^ 2) + (2 * PO * V0) * t + (VO ^ 2) * t ^ 2
    ;; Below, we will let p-squared represent:   (P0 ^ 2) - (sum-r ^ 2)
    ;; and pv represent: (2 * PO * V0)
    ;; and v-squared represent: (VO ^ 2)
    ;;
    ;;  then the equation will simplify to:     0 = p-squared + pv * t + v-squared * t^2

    let p-squared   ((dpx * dpx) + (dpy * dpy)) - (sum-r ^ 2)   ;; p-squared represents difference
    ;; of the square of the radii and the square of the initial positions

    let pv  (2 * ((dpx * dvx) + (dpy * dvy)))  ;; vector product of the position times the velocity
    let v-squared  ((dvx * dvx) + (dvy * dvy)) ;; the square of the difference in speeds
    ;; represented as the sum of the squares of the x-component
    ;; and y-component of relative speeds between the two particles

    ;; p-squared, pv, and v-squared are coefficients in the quadratic equation shown above that
    ;; represents how distance between the particles and relative velocity are related to the time,
    ;; t, at which they will next collide (or when their edges will just be touching)

    ;; Any quadratic equation that is a function of time (t) can be represented as:
    ;;   a*t*t + b*t + c = 0,
    ;; where a, b, and c are the coefficients of the three different terms, and has solutions for t
    ;; that can be found by using the quadratic formula.  The quadratic formula states that if a is
    ;; not 0, then there are two solutions for t, either real or complex.
    ;; t is equal to (b +/- sqrt (b^2 - 4*a*c)) / 2*a
    ;; the portion of this equation that is under a square root is referred to here
    ;; as the determinant, D1.   D1 is equal to (b^2 - 4*a*c)
    ;; and:   a = v-squared, b = pv, and c = p-squared.
    let D1 pv ^ 2 -  (4 * v-squared * p-squared)

    ;; the next test tells us that a collision will happen in the future if
    ;; the determinant, D1 is > 0,  since a positive determinant tells us that there is a
    ;; real solution for the quadratic equation.  Quadratic equations can have solutions
    ;; that are not real (they are square roots of negative numbers).  These are referred
    ;; to as imaginary numbers and for many real world systems that the equations represent
    ;; are not real world states the system can actually end up in.

    ;; Once we determine that a real solution exists, we want to take only one of the two
    ;; possible solutions to the quadratic equation, namely the smaller of the two the solutions:
    ;;  (b - sqrt (b^2 - 4*a*c)) / 2*a
    ;;  which is a solution that represents when the particles first touching on their edges.
    ;;  instead of (b + sqrt (b^2 - 4*a*c)) / 2*a
    ;;  which is a solution that represents a time after the particles have penetrated
    ;;  and are coming back out of each other and when they are just touching on their edges.

    let time-to-collision  -1

    if D1 > 0
      [ set time-to-collision (- pv - sqrt D1) / (2 * v-squared) ]        ;; solution for time step

    ;; if time-to-collision is still -1 there is no collision in the future - no valid solution
    ;; note:  negative values for time-to-collision represent where particles would collide
    ;; if allowed to move backward in time.
    ;; if time-to-collision is greater than 1, then we continue to advance the motion
    ;; of the particles along their current trajectories.  They do not collide yet.

    if time-to-collision > 0
    [
      ;; time-to-collision is relative (ie, a collision will occur one second from now)
      ;; We need to store the absolute time (ie, a collision will occur at time 48.5 seconds.
      ;; So, we add clock to time-to-collision when we store it.
      ;; The entry we add is a three element list of the time to collision and the colliding pair.
      set collisions fput (list (time-to-collision + ticks) self myself)
                          collisions
    ]
  ]
end


;; determines when a particle will hit any of the four walls or membrane
to check-for-wall-collision  ;; particle procedure
  ;; right & left walls
  let x-speed (speed * sin heading)
  if x-speed != 0
    [ ;; solve for how long it will take particle to reach right wall
      let right-interval (max-pxcor - 0.5 - xcor - size / 2) / x-speed
      if right-interval > 0
        [ assign-colliding-wall right-interval "right wall" ]
      ;; solve for time it will take particle to reach left wall
      let left-interval ((- max-pxcor) + 0.5 - xcor + size / 2) / x-speed
      if left-interval > 0
        [ assign-colliding-wall left-interval "left wall" ] ]

;  ;; membrane
;  if x-speed != 0 and part-type = "solute" and xcor < split
;    [ ;; solve for how long it will take particle to reach the membrane from the left
;      let left-interval2 (split - xcor - size / 2) / x-speed
;      if left-interval2 > 0
;        [ assign-colliding-wall left-interval2 "membrane-left" ] ]
;   if x-speed != 0 and part-type = "solute" and xcor < split
;    [ ;; solve for time it will take particle to reach the membrane from the right
;      let right-interval2 (split - xcor + size / 2) / x-speed
;      if right-interval2 > 0
;        [ assign-colliding-wall right-interval2 "membrane-right" ] ]

  ;; top & bottom walls
  let y-speed (speed * cos heading)
  if y-speed != 0
    [ ;; solve for time it will take particle to reach top wall
      let top-interval (max-pycor - 0.5 - ycor - size / 2) / y-speed
      if top-interval > 0
        [ assign-colliding-wall top-interval "top wall" ]
      ;; solve for time it will take particle to reach bottom wall
      let bottom-interval ((- max-pycor) + 0.5 - ycor + size / 2) / y-speed
      if bottom-interval > 0
        [ assign-colliding-wall bottom-interval "bottom wall" ] ]
end


to assign-colliding-wall [time-to-collision wall]  ;; particle procedure
  ;; this procedure is used by the check-for-wall-collision procedure
  ;; to assemble the correct particle-wall pair
  ;; time-to-collision is relative (ie, a collision will occur one second from now)
  ;; We need to store the absolute time (ie, a collision will occur at time 48.5 seconds.
  ;; So, we add clock to time-to-collision when we store it.
  let colliding-pair (list (time-to-collision + ticks) self wall)
  set collisions fput colliding-pair collisions
end

to choose-next-collision
  if collisions = [] [ stop ]
  ;; Sort the list of projected collisions between all the particles into an ordered list.
  ;; Take the smallest time-step from the list (which represents the next collision that will
  ;; happen in time).  Use this time step as the tick-delta for all the particles to move through
  let winner first collisions
  foreach collisions [ if first ? < first winner [ set winner ? ] ]
  ;; winner is now the collision that will occur next
  let dt item 0 winner
  ;; If the next collision is more than 1 in the future,
  ;; only advance the simulation one tick, for smoother animation.
  set tick-delta dt - ticks
  if tick-delta > 1
    [ set tick-delta 1
      set particle1 nobody
      set particle2 nobody
      stop ]
  set particle1 item 1 winner
  set particle2 item 2 winner
end


to perform-next-collision
  ;; deal with 3 possible cases:
  ;; 1) no collision at all
  if particle1 = nobody [ stop ]
  ;; 2) particle meets wall
  if is-string? particle2
    [ if particle2 = "left wall" or particle2 = "right wall" ;or particle2 = "membrane-left" or particle2 = "membrane-right"
        [ ask particle1 [ set heading (- heading) ]
          stop ]
      if particle2 = "top wall" or particle2 = "bottom wall"
        [ ask particle1 [ set heading 180 - heading ]
          stop ] ]
  ;; 3) particle meets particle
  ask particle1 [ collide-with particle2 ]
end


to collide-with [other-particle]  ;; particle procedure
  ;;; PHASE 1: initial setup
  ;; for convenience, grab some quantities from other-particle
  let mass2 [mass] of other-particle
  let speed2 [speed] of other-particle
  let heading2 [heading] of other-particle
  ;; modified so that theta is heading toward other particle
  let theta towards other-particle

  ;;; PHASE 2: convert velocities to theta-based vector representation
  ;; now convert my velocity from speed/heading representation to components
  ;; along theta and perpendicular to theta
  let v1t (speed * cos (theta - heading))
  let v1l (speed * sin (theta - heading))
  ;; do the same for other-particle
  let v2t (speed2 * cos (theta - heading2))
  let v2l (speed2 * sin (theta - heading2))

  ;;; PHASE 3: manipulate vectors to implement collision
  ;; compute the velocity of the system's center of mass along theta
  let vcm (((mass * v1t) + (mass2 * v2t)) / (mass + mass2) )
  ;; now compute the new velocity for each particle along direction theta.
  ;; velocity perpendicular to theta is unaffected by a collision along theta,
  ;; so the next two lines actually implement the collision itself, in the
  ;; sense that the effects of the collision are exactly the following changes
  ;; in particle velocity.
  set v1t (2 * vcm - v1t)
  set v2t (2 * vcm - v2t)

  ;;; PHASE 4: convert back to normal speed/heading
  ;; now convert my velocity vector into my new speed and heading
  set speed sqrt ((v1t * v1t) + (v1l * v1l))
  ;; if the magnitude of the velocity vector is 0, atan is undefined. but
  ;; speed will be 0, so heading is irrelevant anyway. therefore, in that
  ;; case we'll just leave it unmodified.
  set energy (0.5 * mass * speed ^ 2)

  if v1l != 0 or v1t != 0
    [ set heading (theta - (atan v1l v1t)) ]
  ;; and do the same for other-particle
  ask other-particle [
    set speed sqrt ((v2t ^ 2) + (v2l ^ 2))
    set energy (0.5 * mass * speed ^ 2)

    if v2l != 0 or v2t != 0
      [ set heading (theta - (atan v2l v2t)) ]
  ]

end

to my-update-plots
  set-current-plot "Water#"
  set-current-plot-pen "left"
  plot count particles with [pxcor < split and part-type = "solvent"]
  set-current-plot-pen "right"
  plot count particles with [pxcor > split and part-type = "solvent"]
end
@#$#@#$#@
GRAPHICS-WINDOW
275
10
1093
369
50
20
8.0
1
10
1
1
1
0
1
1
1
-50
50
-20
20
0
0
1
ticks
30.0

BUTTON
20
10
85
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
90
10
155
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
20
50
155
95
Solute
Solute
"Sugar" "Sodium Chloride" "Magnesium Chloride" "Aluminum Chloride"
0

SLIDER
20
100
145
133
solute-left
solute-left
1
30
20
1
1
NIL
HORIZONTAL

OUTPUT
160
50
270
95
12

MONITOR
20
140
100
185
Water # Left
count particles with [pxcor < split and part-type = \\"solvent\\"]
0
1
11

MONITOR
180
140
265
185
Water # Right
count particles with [pxcor > split and part-type = \\"solvent\\"]
0
1
11

PLOT
20
240
270
370
Water#
Ticks
Amount
0.0
10.0
50.0
150.0
true
true
"" ""
PENS
"left" 1.0 0 -16777216 true "" ""
"right" 1.0 0 -2674135 true "" ""

SLIDER
145
100
270
133
solute-right
solute-right
1
30
10
1
1
NIL
HORIZONTAL

MONITOR
105
165
175
210
Membrane
split
2
1
11

MONITOR
20
190
100
235
Solute # Left
count particles with [pxcor < split and part-type = \\"solute\\"]
17
1
11

MONITOR
180
190
265
235
Solute # Right
count particles with [pxcor > split and part-type = \\"solute\\"]
17
1
11

MONITOR
20
375
77
420
X-L
count particles with [part-type = \\"solute\\" and xcor < split] / count particles with [part-type = \\"solvent\\" and xcor < split]
2
1
11

MONITOR
80
375
137
420
X-R
count particles with [part-type = \\"solute\\" and xcor > split] / count particles with [ part-type = \\"solvent\\" and xcor > split]
2
1
11

@#$#@#$#@
## VERSION

$Id: Osmotic Pressure.nlogo 47208 2010-10-26 20:43:15Z nholbert $

## WHAT IS IT?

Osmotic Pressure is generally defined as the amount of pressure required to bring solvent movement across a semipermeable membrane to equilibrium. This model attempts to model an agent-based description of the movment of solution particles across a semipermeable membrane and illustrate the colligative nature of osmotic pressure.

## HOW IT WORKS

In this model blue patches represent a container divided by a semipermeable membrane (red square turtles).  Blue circles represent molecules of water and are randomly distributed throughout the container.  White turtles represent particles of added solute. The amount of solute defined by the sliders is placed on the appropriate side of the membrane. Solute particles cannot pass through the membrane.

As the model runs, particles move throughout the container according to kinetic molecular theory and NetLogo code first defined in the GasLab suite of models.  All particles move in a straight line until they collide with another particle, the wall, or the membrane (only insoluble particles). Particles collide with one another in an elastic collision. While water molecules (blue turtles) may pass through the membrane freely, solute particles (white turtles) are restricted to the side they are created on. As the model progresses, according to the phenomenon of osmosis, the solvent (water) should show a net movement towards the side of higher solute concentration.

## HOW TO USE IT

Initial settings:  
SOLUTE: Choose the solute to add to the solution.  The chemical formula of the solute will be printed in the output box to the right.  Each solute will act differently in solution depending on its bonding behavior.  
SOLUTE-LEFT: This number will determine the number of solute molecules added to the solution on the left side of the membrane.  Keep in mind that this number may be different from the total number of particles once the solute is dissolved.  
SOLUTE-RIGHT: This number will determine the number of solute molecules added to the solution on the right side of the membrane.  Keep in mind that this number may be different from the total number of particles once the solute is dissolved.

Buttons  
SETUP: Sets up the model  
GO: Runs the model

Monitors  
WATER # LEFT: Shows the number of water particles on the left side of the membrane.  
WATER # RIGHT: Shows the number of water particles on the right side of the membrane.  
SOLUTE LEFT: Shows the number of solute particles on the left side of the membrane.  
SOLUTE RIGHT: Shows the number of solute particles on the right side of the membrane.  
MEMBRANE: Shows the x-cor of the membrane. Note: The membrane moves based on the difference between the amount of particles moving accross the membrane in a given direction.

Plot  
WATER #: Plots the number of water molecules on the left and right side of the membrane over time (ticks).

## THINGS TO NOTICE

As the model runs more water molecules should end up on the side of the membrane with more solute particles. Instead, the movement of the membrane seems to be at chance.  
How does the membrane movement change when adding different solutes? Is there a pattern?  
What happens when adding Sodium Chloride?  How is this different than adding Sugar?

## THINGS TO TRY

Try adding different solutes.  Can you get change the number of solute molecules so that Sodium Chloride acts like Sugar?  
Try to figure out the mathematical relationship between membrane movement and the number of solute particles. Will this relationship depend on the type of solute added?  Why or why not?

## EXTENDING THE MODEL

Try making new solutes.  
Can you figure out an agent-based approach that explains why more solvent particles move to the side with a higher concentration of solute particles? Keep in mind, this solution should depend on the number of solute agents, and not on substance-specific properties.

## NETLOGO FEATURES

"Sub-breeds" are instantiated by defining a variable for every particle that defines it as either a "soluble" or "insoluble."

## RELATED MODELS

GasLab

## CREDITS AND REFERENCES

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="15000"/>
    <metric>split</metric>
    <metric>count particles with [part-type = "solute" and xcor &lt; split] / count particles with [part-type = "solvent" and xcor &lt; split]</metric>
    <metric>count particles with [part-type = "solute" and xcor &gt; split] / count particles with [part-type = "solvent" and xcor &gt; split]</metric>
    <enumeratedValueSet variable="Solute">
      <value value="&quot;Sugar&quot;"/>
      <value value="&quot;Sodium Chloride&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="solute-left">
      <value value="10"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="solute-right">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
