globals [
  minimum-separation
  max-separate-turn
  max-align-turn
  max-cohere-turn
  max-avoid-turn
  vision
  current-tolerance
  current-watch
]

turtles-own [
  surface-attitude
  embedded-attitude
  embedded-strength
  surface-similars
  nearest-surface-similar
  surface-differents
  nearest-surface-different
  embedded-similars
  nearest-embedded-similar
  embedded-differents
  nearest-embedded-different
  tolerance-ratio
]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; setup
;; sets up all the parameters and turtles for the model.  sets up the 
;; variables that aren't currently in use, but may become active later.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to setup
  ca
  crt population [
    setxy random-xcor random-ycor
    if display-type = "Surface"
    [
      set size 3
      ifelse possible-surface = 38
      [ set shape one-of shapes ]
      [ set shape one-of sublist shapes 0 possible-surface ]
      set surface-attitude shape
      if possible-embedded < 14
      [ set color one-of sublist base-colors 0 possible-embedded ]
      set embedded-attitude color
      set embedded-strength 1 / e ^ (15 / 4) + random-float e ^ (25 / 4)
      set tolerance-ratio 0
    ]
    if display-type = "Embedded"
    [
      set embedded-strength 1 / e ^ (15 / 4) + random-float e ^ (25 / 4)
      set size .4 * ln embedded-strength + 2.5
      set shape "circle"
      if possible-embedded < 14
      [ set color one-of sublist base-colors 0 possible-embedded ]
      set embedded-attitude color
      ifelse possible-surface = 38
      [ set surface-attitude one-of shapes ]
      [ set surface-attitude one-of sublist shapes 0 possible-surface ]
      set tolerance-ratio 0
    ]
    if display-type = "Both"
    [
      set embedded-strength 1 / e ^ (15 / 4) + random-float e ^ (25 / 4)
      set size .4 * ln embedded-strength + 2.5
      ifelse possible-surface = 38
      [ set shape one-of shapes ]
      [ set shape one-of sublist shapes 0 possible-surface ]
      if possible-embedded < 14
      [ set color one-of sublist base-colors 0 possible-embedded ]
      set embedded-attitude color
      set surface-attitude shape
      set tolerance-ratio 0
    ]
  ]
  
  ;; flocking details set based on current tolerance level
  set current-tolerance tolerance
  set minimum-separation (current-tolerance / 20 + 3)
  set max-align-turn ((100 - current-tolerance) / 5)
  set max-separate-turn (current-tolerance / 5)
  set max-cohere-turn ((100 - current-tolerance) / 5)
  set max-avoid-turn ((100 - current-tolerance) / 5)
  
  set vision 10
  
  set current-watch "" ;; turtle being watched (none at beginning)
  
  reset-ticks
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; go
;; default way to run the model.  modifies the tolerance if there's a change,
;; gives orders to the turtles based on the current display state.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to go
  if current-tolerance != tolerance
  [
    set current-tolerance tolerance
    set minimum-separation (current-tolerance / 20 + 6)
    set max-align-turn ((100 - current-tolerance) / 5)
    set max-separate-turn (current-tolerance / 5)
    set max-cohere-turn ((100 - current-tolerance) / 5)
    set max-avoid-turn ((100 - current-tolerance) / 5)
  ]
  ask turtles
  [
    if display-type = "Surface" 
    [ 
      surface-behavior
      surface-movement
    ]
    if display-type = "Embedded"
    [
      embedded-behavior
      embedded-movement
    ]
    if display-type = "Both"
    [
      embedded-behavior
      surface-behavior
      combined-movement
    ]
  ]
  repeat 5 [ ask turtles [ fd .1 ] display ]
  plot-ratio 
  tick
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; avoid-minority
;; makes a list of all turtles based on color and singles out those in the
;; smallest gorup.  tells the other turtles to behave normally towards each
;; other, but avoid the smallest group with 0% tolerance.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to avoid-minority
  let smallest first first sort-by [last ?1 < last ?2] filter [last ? > 0] n-values 140 [ (list ? count turtles with [ color = ? ] )]
  if current-tolerance != tolerance
  [
    set current-tolerance tolerance
    set minimum-separation (current-tolerance / 20 + 6)
    set max-align-turn ((100 - current-tolerance) / 5)
    set max-separate-turn (current-tolerance / 5)
    set max-cohere-turn ((100 - current-tolerance) / 5)
    set max-avoid-turn ((100 - current-tolerance) / 5)
  ]
  ask turtles
  [
    if display-type = "Surface" 
    [ 
      surface-behavior
      surface-movement
    ]
    if display-type = "Embedded"
    [
      embedded-behavior
      embedded-movement
    ]
    if display-type = "Both"
    [
      embedded-behavior
      surface-behavior
      combined-movement
    ]
  ]
  
  ask turtles with [color != smallest]
  [
    set current-tolerance 0
    set minimum-separation (current-tolerance / 20 + 6)
    set max-align-turn ((100 - current-tolerance) / 5)
    set max-separate-turn (current-tolerance / 5)
    set max-cohere-turn ((100 - current-tolerance) / 5)
    set max-avoid-turn ((100 - current-tolerance) / 5)
    if display-type = "Embedded"
    [
      if any? embedded-differents
      [
        set embedded-differents embedded-differents with [color = smallest]
        if any? embedded-differents
        [
          find-nearest-embedded-different
          if distance nearest-embedded-different < minimum-separation * 2
          [ avoid-embedded ]
        ]
      ]
    ]
    if display-type = "Both"
    [
      if any? embedded-differents
      [
        set embedded-differents embedded-differents with [color = smallest]
        if any? embedded-differents
        [
          find-nearest-embedded-different
          if distance nearest-embedded-different < minimum-separation * 2
          [ avoid-embedded ]
        ]
      ]
    ]
  ]
  repeat 5 [ ask turtles [ fd .1 ] display ]
  plot-ratio
  tick
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; avoid-majority
;; makes a list of all turtles based on color and singles out those in the
;; largest gorup.  tells the other turtles to behave normally towards each
;; other, but avoid the largest group with 0% tolerance.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to avoid-majority
  let largest first first sort-by [last ?1 > last ?2] filter [last ? > 0] n-values 140 [ (list ? count turtles with [ color = ? ] )]
  if current-tolerance != tolerance
  [
    set current-tolerance tolerance
    set minimum-separation (tolerance / 20 + 6)
    set max-align-turn ((100 - current-tolerance) / 5)
    set max-separate-turn (tolerance / 5)
    set max-cohere-turn ((100 - current-tolerance) / 5)
    set max-avoid-turn ((100 - current-tolerance) / 5)
    set vision 10
  ]
  ask turtles
  [
    if display-type = "Surface" 
    [ 
      surface-behavior
      surface-movement
    ]
    if display-type = "Embedded"
    [
      embedded-behavior
      embedded-movement
    ]
    if display-type = "Both"
    [
      embedded-behavior
      surface-behavior
      combined-movement
    ]
  ]
  
  ask turtles with [color != largest]
  [
    set current-tolerance 0
    set minimum-separation (current-tolerance / 20 + 6)
    set max-align-turn ((100 - current-tolerance) / 5)
    set max-separate-turn (current-tolerance / 5)
    set max-cohere-turn ((100 - current-tolerance) / 5)
    set max-avoid-turn ((100 - current-tolerance) / 5)
    if display-type = "Embedded"
    [
      if any? embedded-differents
      [
        set embedded-differents embedded-differents with [color = largest]
        if any? embedded-differents
        [
          find-nearest-embedded-different
          if distance nearest-embedded-different < minimum-separation * 2
          [ avoid-embedded ]
        ]
      ]
    ]
    if display-type = "Both"
    [
      if any? embedded-differents
      [
        set embedded-differents embedded-differents with [color = largest]
        if any? embedded-differents
        [
          find-nearest-embedded-different
          if distance nearest-embedded-different < minimum-separation * 2
          [ avoid-embedded ]
        ]
      ]
    ]
  ]
  repeat 5 [ ask turtles [ fd .1 ] display ]
  plot-ratio
  tick
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; surface-behavior
;; controls how the turtles behave in Surface mode.  they only keep track of
;; their surface attitude, represented by shape.  they have the chance to
;; randomly change on their own, or if they share a patch with a turtle of a
;; different shape, they have the chance to copy their surface attitude
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to surface-behavior
  let x random 100
  ifelse count other turtles-here != 0 and [surface-attitude] of one-of other turtles-here != [surface-attitude] of self
  [ 
    ifelse x < 10 [
      set surface-attitude [surface-attitude] of one-of other turtles-here
    ]
    [ 
      if x < 15
      [  
        ifelse possible-surface = 38
        [ set surface-attitude one-of shapes ]
        [ set surface-attitude one-of sublist shapes 0 possible-surface ]
      ]
    ]
  ]
  [ 
    if x < 55
    [  
      ifelse possible-surface = 38
      [ set surface-attitude one-of shapes ]
      [ set surface-attitude one-of sublist shapes 0 possible-surface ]
    ]
  ]
  set shape surface-attitude
  if display-type = "Surface" [ set size 3 ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; embedded-behavior
;; controls how the turtles behave in Embedded mode.  they only keep track of
;; their embedded attitude, represented by size and color.  they change based
;; on random interaction with other turtles.  they grow, shrink, or change
;; color based on chance and the color and size of the turtle they're with.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to embedded-behavior
  if count other turtles-here != 0
  [
    let t one-of other turtles-here
    let x random 100
    ifelse [embedded-attitude] of t = [embedded-attitude] of self
    [
      ifelse [size] of t - [size] of self > 2
      [
        ifelse x < 5
        [ set embedded-strength embedded-strength - .2 ]
        [
          if x < 10
          [ set embedded-strength embedded-strength + .1 ]
        ]
      ]
      [ if x < 5
        [ set embedded-strength embedded-strength + .1 ]
      ]
    ]
    [ 
      ifelse abs ([embedded-attitude] of self - [embedded-attitude] of t) < current-tolerance / 10
      [
        ifelse x < 5
        [
          set embedded-attitude [embedded-attitude] of t
          set embedded-strength embedded-strength + .4 * -1 ^ random 1
        ]
        [ 
          if x < 25
          [ set embedded-strength embedded-strength + .2 * -1 ^ random 1 ]
        ]
      ]
      [
        ifelse size < 2
        [
          ifelse x < 5
          [
            set embedded-attitude [embedded-attitude] of t
            set embedded-strength embedded-strength + .2
          ]
          [
            ifelse x < 20 [ set embedded-strength embedded-strength + .1 ] 
            [ if x < 42 [ set embedded-strength embedded-strength - .2 ] ] 
          ]
        ]
        [
          ifelse x < 1
          [
            set embedded-attitude [embedded-attitude] of t
            set embedded-strength embedded-strength + .8 * -1 ^ random 1
          ]
          [
            ifelse x < 31
            [
              set embedded-strength embedded-strength + .1
            ]
            [
              if x < 1 [ set embedded-strength embedded-strength - .4 ]
            ]
          ]
        ]
      ]
    ]
  ]
  
  if is-agentset? embedded-similars and is-agentset? embedded-differents and count embedded-differents != 0
  [ set tolerance-ratio count embedded-similars / count embedded-differents ]
  
  if embedded-strength < 1 / e ^ 4 [ set embedded-strength 1 / e ^ 4 ]
  if embedded-strength > e ^ 12 [ set embedded-strength e ^ 12 ]
  if display-type = "Embedded" [ set shape "circle" ]
  set size  .4 * ln embedded-strength + 2.5
  set color embedded-attitude
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; surface-movement
;; controls how the turtles move in Surface mode.  normally they move in a
;; straight line, but as tolerance gets lower, the chance that they will
;; follow flocking rules increases
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to surface-movement
  if random 100 < 100 - current-tolerance
  [
    find-surface-similars
    if any? surface-similars
    [ 
      find-nearest-surface-similar
      ifelse distance nearest-surface-similar < minimum-separation
      [ separate-surface ]
      [ 
        align-surface
        cohere-surface
      ]
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; embedded-movement
;; controls how the turtles move in Embedded mode.  turtles are always
;; affected by tolerance, but the flocking rules become more relaxed as level
;; of tolerance increases.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to embedded-movement
  find-embedded-similars
  if any? embedded-similars
  [ 
    find-nearest-embedded-similar
    ifelse distance nearest-embedded-similar < minimum-separation
    [ separate-embedded ]
    [ 
      align-embedded
      cohere-embedded
    ]
  ]
  
  find-embedded-differents
  if any? embedded-differents
  [
    find-nearest-embedded-different
    if distance nearest-embedded-different < minimum-separation * 2
    [ avoid-embedded ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; combined-movement
;; controls how the turtles move in Both mode.  surface and embedded behavior
;; are present, so rules for both modes are followed simultaneously.
;; surface rules are evaluated first because they are less important.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to combined-movement
  if random 100 < 100 - current-tolerance
  [
    find-surface-similars
    if any? surface-similars
    [ 
      find-nearest-surface-similar
      ifelse distance nearest-surface-similar < minimum-separation
      [ separate-surface ]
      [ 
        align-surface
        cohere-surface
      ]
    ]
  ]
  
  find-embedded-similars
  if any? embedded-similars
  [ 
    find-nearest-embedded-similar
    ifelse distance nearest-embedded-similar < minimum-separation
    [ separate-embedded ]
    [ 
      align-embedded
      cohere-embedded
    ]
  ]
  
  find-embedded-differents
  if any? embedded-differents
  [
    find-nearest-embedded-different
    if distance nearest-embedded-different < minimum-separation * 2
    [ avoid-embedded ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; flocking rules
;; there are two compies of almost every method below, one for surface, one
;; for embedded.  there are some differences, such as how turtles don't care
;; about finding turtles with different surface attitudes from themsevles,
;; because surface attitudes are more specific and diverse than embedded
;; attitudes.  Most of these are similar to the functions in the focking
;; model in the models library, the biggest difference is finding differents
;; and avoiding those differents.  These funcitons are similar to separating
;; from similars
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to find-surface-similars
  set surface-similars other turtles with [surface-attitude = [surface-attitude] of myself] in-radius vision
end

to find-embedded-similars
  set embedded-similars other turtles with
  [
    color - current-tolerance / 10 < [color] of myself and color + current-tolerance / 10 > [color] of myself
  ]
  in-radius vision
end

to find-embedded-differents
  set embedded-differents other turtles with
  [
    color - current-tolerance / 10 < [color] of myself and color + current-tolerance / 10 < [color] of myself
  ]
  in-radius vision
end

to find-nearest-surface-similar
  set nearest-surface-similar min-one-of surface-similars [distance myself]
end

to find-nearest-embedded-similar
  set nearest-embedded-similar min-one-of embedded-similars [distance myself]
end

to find-nearest-embedded-different
  set nearest-embedded-different min-one-of embedded-differents [distance myself]
end

to separate-surface
  turn-away ([heading] of nearest-surface-similar) max-separate-turn
end

to separate-embedded
  turn-away ([heading] of nearest-embedded-similar) max-separate-turn
end

to avoid-embedded
  turn-away ([heading] of nearest-embedded-different) max-avoid-turn
end

to align-surface
  turn-towards average-similar-surface-heading max-align-turn
end

to align-embedded
  turn-towards average-similar-embedded-heading max-align-turn
end

to-report average-similar-surface-heading
  let x-component sum [dx] of surface-similars
  let y-component sum [dy] of surface-similars
  ifelse x-component = 0 and y-component = 0
    [ report heading ]
    [ report atan x-component y-component ]
end 

to-report average-similar-embedded-heading
  let x-component sum [dx] of embedded-similars
  let y-component sum [dy] of embedded-similars
  ifelse x-component = 0 and y-component = 0
    [ report heading ]
    [ report atan x-component y-component ]
end 

to cohere-surface
  turn-towards average-heading-towards-surface-similars max-cohere-turn
end

to cohere-embedded
  turn-towards average-heading-towards-embedded-similars max-cohere-turn
end

to-report average-heading-towards-surface-similars
  let x-component mean [sin (towards myself + 180)] of surface-similars
  let y-component mean [cos (towards myself + 180)] of surface-similars
  ifelse x-component = 0 and y-component = 0
    [ report heading ]
    [ report atan x-component y-component ]
end

to-report average-heading-towards-embedded-similars
  let x-component mean [sin (towards myself + 180)] of embedded-similars
  let y-component mean [cos (towards myself + 180)] of embedded-similars
  ifelse x-component = 0 and y-component = 0
    [ report heading ]
    [ report atan x-component y-component ]
end 

to turn-towards [new-heading max-turn]
  turn-at-most (subtract-headings new-heading heading) max-turn
end 

to turn-away [new-heading max-turn]
  turn-at-most (subtract-headings heading new-heading) max-turn
end 

to turn-at-most [turn max-turn]
  ifelse abs turn > max-turn
    [ ifelse turn > 0
        [ rt max-turn ]
        [ lt max-turn ] ]
    [ rt turn ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; plot-ratio
;; plots the ratio of similars to differents the turtles observe when on
;; Embedded or Both mode
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to plot-ratio
  set-current-plot "local area ratio"
  let x 0
  ask turtles
  [ set x x + tolerance-ratio ]
  plot x / population
end
@#$#@#$#@
GRAPHICS-WINDOW
215
10
630
446
40
40
5.0
1
10
1
1
1
0
1
1
1
-40
40
-40
40
1
1
1
ticks
30.0

BUTTON
7
10
73
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7
50
73
84
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
81
10
208
55
display-type
display-type
"Surface" "Embedded" "Both"
2

SLIDER
6
213
209
246
population
population
1
300
150
1
1
people
HORIZONTAL

SLIDER
4
250
209
283
tolerance
tolerance
1
100
15
1
1
percent
HORIZONTAL

BUTTON
111
60
209
94
watch a turtle
set current-watch random population\nwatch turtle current-watch
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
7
89
104
134
watching turtle
current-watch
17
1
11

BUTTON
110
100
209
134
stop watching
reset-perspective\nset current-watch \"\"
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
636
158
901
299
embedded attitude stats
NIL
NIL
0.0
10.0
1.0
5.0
true
true
"" "if display-type = \"Surface\"\n[ clear-plot ]"
PENS
"mean" 1.0 0 -14985354 true "" "plot mean [size] of turtles"
"variance" 1.0 0 -10899396 true "" "plot variance [size] of turtles"

PLOT
636
306
901
447
local area ratio
NIL
NIL
0.0
10.0
0.0
0.1
true
true
"" ""
PENS
"sim/diff" 1.0 0 -14985354 true "" "if display-type = \"Surface\"\n[ clear-plot ]"
"tolerance" 1.0 0 -14439633 true "" "ifelse display-type = \"Surface\"\n[ clear-plot ]\n[ plot tolerance / 100 ]"

PLOT
635
10
900
151
turtles of each embedded attitude
NIL
NIL
0.0
1.0
0.0
1.0
true
false
"" "set-plot-x-range 0 (max [color] of turtles + 5)\nset-plot-pen-mode 1\nset-plot-pen-interval 10"
PENS
"default" 1.0 0 -16777216 true "" "histogram [color] of turtles"

TEXTBOX
908
10
1033
223
Color Code:\n5 = gray\n15 = red\n25 = orange\n35 = brown\n45 = yellow\n55 = green\n65 = lime\n75 = turqoise\n85 = cyan\n95 = sky\n105 = blue\n115 = violet\n125 = magenta\n135 = pink
11
0.0
1

TEXTBOX
906
305
1031
362
> 1 : More similar\nthan different\n< 1 : More different\nthan similar
11
0.0
1

BUTTON
105
329
208
363
avoid minority
avoid-minority
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
5
290
205
318
Only one forever button should be active at a time
11
0.0
1

TEXTBOX
6
323
102
369
Makes everyone avoid the smallest group
11
0.0
1

TEXTBOX
7
377
101
420
Makes everyone avoid the largest group
11
0.0
1

BUTTON
105
383
209
417
avoid majority
avoid-majority
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
138
210
171
possible-surface
possible-surface
2
38
38
1
1
choices
HORIZONTAL

SLIDER
5
175
210
208
possible-embedded
possible-embedded
2
14
14
1
1
choices
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

The majority of the material in social psychology rests on the concept of attitude and behavior.  Attitudes are feelings or beliefs about different things.
Surface attitudes are your favorite movie, least favorite book, etc.  They are easily changed, easily affected by others, and easily accessed for basic interactions, especially with people you don't know.
Embedded attitudes are things like polictial views, religious beliefs, etc.  They are deeply rooted, studies have shown they can even be genetic, they are usually inflexible, and come up in more meaningful interactions.  They define who your inner circle of friends are, and how you view strangers at a glance.
Embedded attitudes can change, but they don't change the same way as surface attitudes.  If someone interacts with a person of a similar attitude with a similar attitude strength, it's likely they will both feel that attitude grow stronger.  If two people of very different attitudes interact, it's possible the person with the weaker attitude will be encouraged and grow stronger, or be alienated by the extremeness of the other person, and mvove more neutral.  The other person will usually be largely unaffected.  When two people of different attitudes interact, they tend to feel more strongly about their own attitude afterwards, but they can also feel more neutral, or change their views to be more similar to the person they were interacting with.  There are people who change their attitude very often, but their attitude will be weak, which means they're easily swayed to another view.
For the most part, surface attitudes don't matter, so they don't have a huge affect on behavior, but people do have a tendency to meet people and start friendships with people based on surface attitudes.  Embedded attitudes definitely have more of an effect on behavior.  Often, people spend a majority of their time with people who have similar attitudes to their own, and sometimes even avoid people who have different attitudes.
The flip side of the coin is that behavior has an effect on attitude.  Spending time with people who have the same embedded attitude can lead to everyone liking the same music or food.  Meeting people becuase of surface attitude similarities can lead to more meaningful interactions that affect embedded attitudes.
All of this is governed by people's tolerance.  Tolerance can change very easily and vary often, even over the course of a day.  Tolerance involves how willing a person is to interact with people who don't share their attitudes, and thus how willing they are to consider changing their own attitude because of that interaction.  Tolerance isn't universal, between groups or within groups.  Certain groups are avoided or accpeted more than others, and certain people within each group are better at branching out than others.

## HOW IT WORKS

Surface attitudes are shown using the turtles shape, while embedded attitudes are displayed using the turtles color and size.  Basic flocking procedures are used to demonstrate low tolerance, where turtles of similar color group together, with the addition of an avoidance function to avoid turtles who are considered different.  The size of the turtles is on a logarithmic scale because larger changes are easier when the attitude is weaker than when they are stronger.
Changing size, shape, or color is all based on nested randomness.  One random number is chosen at the beginning of the behavior function because at most one action is possible per tick for each turtle.  Interactions are simulated by more than one turtle occupying the sme patch, and both have the chance to be affacted.
Tolerance affects embedded movement by restricting the movement parameters for flocking.  Surface movement is affected by controlling how often the flocking rules are implemented.  Higher tolerance, means running the rules almost every time.
When the chooser is set to Surface, only surface attitudes are displayed, and only surface behaviors are run.  The same goes for when the chooser is set to Embedded.  when it's set to Both, both surface and embedded attitudes are displayed, and the turtles movement is affected by both in each move.  Turtles first move according to surface rules, then to embedded rules, because the embedded ones are more important.
For avoid-minority and avoid-majority, the color with the smallest or largest number of turtles was avoided by running flocking rules as normal to the set tolerance, and then running them again before letting the turtle move, but this time only evauating according to a 0& tolerance to the chosen group.

## HOW TO USE IT

setup - setup the model according to the chooser and sliders in the Interface tab
go - run the model and update plots
display-type - allows user to choose whether they want to observe surface attitudes, embedded attitudes, or both, cna be changed at any time
watching turtle - monitor displays the who of the turtle being watched
watch a turtle - chooses a random turtle and watches it
stop watching - stops watching turtle being watched if there is one
possible-surface / possible-embedded - allows the user to limit the maximum possible number of attitudes a turtle can choose from, can be changed at any time
population - number of turtles to be present in this run of the model, can only be set once per build
tolerance - willingness of each turtle to interact with turtles of different attitudes
avoid minority - runs normally, except every turtle not in the smallest group avoids every turtle in the smallest group
avoid majority - same as avoid minority, but in respect to the largest group
turtles of each embedded attitude - a histogram of all the different possible attitudes in the current build that shows how many turtles are in each group
embedded attitude stats - shows the mean and variance of the sizes of all turtles in the model, disabled when size is not a factor, such as when only surface attitudes are being evaluated
local area ratio - displays the mean ratio of similar turtles to different turtles in the vicinity of each turtle and the current tolerance level, for context

## THINGS TO NOTICE

Run the model in different attitude modes to see how they run separately, then use both to see a more realistic representation of how the concept works.  Notice how, over time, the number of groups dwindles, and the range of sizes for embedded attitude gets smaller.  Notice how in surface behavior, little groups of turtles of the same shape will align and match each other for a short time, then change and find new partners, whereas in the embedded behavior, two turtles can stay together for an extremely long time if tolerance is low.  Notice how the model explodes when in Embedded or Both mode and changing from a low tolerance to a high tolerance.  Notice how avoiding the majority or minority affects the size of either group.

## THINGS TO TRY

Play around with different population sizes and tolerances, limit the number of possible attitues to chose from, and observe how changing these things affect the ratio of similar to different, the change in the mean size of the turtles, and how quickly a dominant group is found.  

## EXTENDING THE MODEL

Add a method that makes tolerance unique to each turtle, so they have the option to behave indpendently rather than all act the same way.

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

Flocking

## CREDITS AND REFERENCES

Attitudes, Behavior, and Social Context: The Role of Norms and Group Membership
 edited by Deborah J. Terry, Michael A. Hogg

Gilovich, T.,  Keltner, D., Chen,  S., & Nisbett, R. E..  (2012). Social Psychology  (3rd ed.).  New York: Norton.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
