;;;
;;; AGENTS DEFINITIONS
;;;


globals [
  involve-chance  ;; The chance out of 100 that an corporate volunteer will pass on institutional during one hour of exposure.
  support-show    ;; How long a person will be involved according support gives.
    
  slider-check-1    ;; Temporary variables for slider values, so that if sliders
  slider-check-2    ;;   are changed on the fly, the model will notice and
  slider-check-3    ;;   change volunteers's tendencies appropriately.
]

turtles-own [
  involved?          ;; If true, the person is involved.  It may be influenced or not.
  influenced?        ;; If true, the involving is influenced (and involved? must also be true).
  involve-length     ;; How long the person has been involved.
  enrolled?          ;; If true, the person is working for social cause.
  enroll-length      ;; How long the person has enrolled in social cause.
                     
                     
  work-exposure      ;; How long the person will work in social causes.
  supporting-tendency;; How likely the person is to support social causes.
  social-networks-use;; The percent chance a person uses social networks.
  
  partner            ;; The person that is our current partner (maybe influencer) in a work exposure.
]

;;;
;;; SETUP PROCEDURES
;;;

to setup
  __clear-all-and-reset-ticks
  setup-globals
  setup-people
  setup-plot
  update-plot
end

to setup-globals
  reset-ticks
  set involve-chance 30    ;; if you have social network using tendency,
                           ;; you have a 10% chance of being involved
  set support-show 30.0    ;; support show up 1 hour after working
  set slider-check-1 average-work-exposure
  set slider-check-2 average-supporting-tendency
  set slider-check-3 average-social-networks-use
end

;; Create carrying-capacity number of volunteers half are righty and half are lefty
;;   and some are involved in social causes.  Also assigns colors to people with the ASSIGN-COLORS routine.

to setup-people
  crt volunteers
    [ setxy random-xcor random-ycor
      set influenced? false
      set enrolled? false
      set partner nobody
      ifelse random 2 = 0
        [ set shape "person righty" ]
        [ set shape "person lefty" ]
      ;; 10% of the people start out involved.
      set involved? (who < volunteers * 0.10)
      if involved?
        [ set involve-length random-float support-show ]
      assign-work-exposure
      assign-supporting-tendency
      assign-social-networks-use
      assign-color ]
end

;; Different volunteers are displayed in 3 different colors depending on activities
;; green is institutional
;; blue is corporate
;; gray is influencer related to partner

to assign-color  ;; turtle procedure
  ifelse not involved?
    [ set color green ]
    [ ifelse influenced?
      [ set color gray ]
      [ set color blue ] ]
end

;; RANDOM-NEAR distribution implementing.

to assign-work-exposure  ;; turtle procedure
  set  work-exposure random-near average-work-exposure
end

to assign-supporting-tendency  ;; turtle procedure
  set supporting-tendency random-near average-supporting-tendency
end

to assign-social-networks-use  ;; turtle procedure
  set social-networks-use random-near average-social-networks-use
end

to-report random-near [center]  ;; turtle procedure
  let result 0
  repeat 40
    [ set result (result + random-float center) ]
  report result / 20
end

;;;
;;; GO PROCEDURES
;;;

to go
  if all? turtles [influenced?]
    [ stop ]
  check-sliders
  ask turtles
    [ if involved?
        [ set involve-length involve-length + 1 ]
      if enrolled?
        [ set enroll-length enroll-length + 1 ] ]
  ask turtles
    [ if not enrolled?
        [ move ] ]
  ;; Righties are always the ones to initiate influencing.
  ask turtles
    [ if not enrolled? and shape = "person righty" and (random-float 10.0 < supporting-tendency)
        [ enroll ] ]
  ask turtles [ disenroll ]
  ask turtles [ involve ]
  ask turtles [ assign-color ]
  tick
  update-plot
end

;; Each tick a check is made to see if sliders have been changed.
to check-sliders
  if (slider-check-1 != average-work-exposure)
    [ ask turtles [ assign-work-exposure ]
      set slider-check-1 average-work-exposure ]
  if (slider-check-2 != average-supporting-tendency)
    [ ask turtles [ assign-supporting-tendency ]
      set slider-check-2 average-supporting-tendency ]
  if (slider-check-3 != average-social-networks-use)
    [ ask turtles [ assign-social-networks-use ]
      set slider-check-3 average-social-networks-use ]
end

;; People move about at random.

to move  ;; turtle procedure
  rt random-float 360
  fd 1
end

;; Volunteers have the chance to participate in social activities.

to enroll  ;; turtle procedure -- righties only!
  let potential-partner one-of (turtles-at -1 0)
                          with [not enrolled? and shape = "person lefty"]
  if potential-partner != nobody
    [ if random-float 10.0 < [supporting-tendency] of potential-partner
      [ set partner potential-partner
        set enrolled? true
        ask partner [ set enrolled? true ]
        ask partner [ set partner myself ]
        move-to patch-here ;; move to center of patch
        move-to patch-here ;; partner moves to center of patch
        set pcolor gray - 3
        ask (patch-at -1 0) [ set pcolor gray - 3 ] ] ]
end

;; If two diferents kind of volunteers work together in social cause it's probabbly one of them be influenced, 
;; in order to work in social causes.

to disenroll  ;; turtle procedure
  if enrolled? and (shape = "person righty")
    [ if (enroll-length >  work-exposure) or
         ([enroll-length] of partner) > ([ work-exposure] of partner)
        [ set enrolled? false
          set enroll-length 0
          ask partner [ set enroll-length 0 ]
          set pcolor black
          ask (patch-at -1 0) [ set pcolor black ]
          ask partner [ set partner nobody ]
          ask partner [ set enrolled? false ]
          set partner nobody ] ]
end

;; Involving occurs if volunteer corporative is influenced.

to involve  ;; turtle procedure
  if enrolled? and involved? and not influenced?
    [ if random-float 11 > social-networks-use or
         random-float 11 > ([social-networks-use] of partner)
        [ if random-float 100 < involve-chance
            [ ask partner [ set involved? true ] ] ] ]
end
;;;
;;; PLOTTING PROCEDURES
to setup-plot
  set-current-plot "Populations"
  set-plot-y-range 0 (volunteers + 50)
end

to update-plot
  set-current-plot "Populations"
  set-current-plot-pen "Institutional"
  plot count turtles with [not involved?]
  set-current-plot-pen "Corporate"
  plot count turtles with [involved?] -
       count turtles with [influenced?]
end;
;;; MONITOR PROCEDURES
;;
to-report %involved
  ifelse any? turtles
    [ report (count turtles with [involved?] / count turtles) * 100 ]
    [ report 0 ]
end


; Copyright 2012 Yannick Patrick Carrasco Merma - San Ignacio de Loyola University. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
364
22
796
475
12
12
16.9
1
10
1
1
1
0
1
1
1
-12
12
-12
12
1
1
1
days
30.0

BUTTON
10
66
93
110
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
94
66
177
110
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
228
66
358
111
% involved
%involved
2
1
11

SLIDER
9
22
358
55
volunteers
volunteers
50
300
200
1
1
NIL
HORIZONTAL

SLIDER
4
123
357
156
average-work-exposure
average-work-exposure
1
21
10
1
1
hours
HORIZONTAL

SLIDER
190
163
357
196
average-social-networks-use
average-social-networks-use
0
5
3
1
1
NIL
HORIZONTAL

SLIDER
4
163
180
196
average-supporting-tendency
average-supporting-tendency
1
10
1
1
1
NIL
HORIZONTAL

PLOT
5
206
356
474
Populations
days
volunteers
0.0
52.0
0.0
350.0
true
true
"" ""
PENS
"Institutional" 1.0 0 -10899396 true "" ""
"Corporate" 1.0 0 -13345367 true "" ""

@#$#@#$#@
## WHAT IS IT?

This model contains the main principles related to model of Voluntarism from Social Media environment.

## HOW IT WORKS

We can assume the importance of using social media in order to promote and reenforce the social program values.

## HOW TO USE IT

The agents for this models are the Corporate Volunteers, the Social Volunteers and the RSE Environment designed for this propossal.

## THINGS TO NOTICE

/*This section could give some ideas of things for the user to notice while running the model.*/

## THINGS TO TRY

/*This section could give some ideas of things for the user to try to do (move sliders, switches, etc.) with the model.*/

## EXTENDING THE MODEL

/*This section could give some ideas of things to add or change in the procedures tab to make the model more complicated, detailed, accurate, etc.*/

## NETLOGO FEATURES

/*This section could point out any especially interesting or unusual features of NetLogo that the model makes use of, particularly in the Procedures tab.  It might also point out places where workarounds were needed because of missing features.*/

## RELATED MODELS

/*This section could give the names of models in the NetLogo Models Library or elsewhere which are of related interest.*/

## CREDITS AND REFERENCES

/*This section could contain a reference to the model's URL on the web if it has one, as well as any other necessary credits or references.*/

## COPYRIGHT NOTICE

Copyright 2012 Yannick Carrasco. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:  
a) this copyright notice is included.  
b) this model will not be redistributed for profit without permission from Uri Wilensky. Contact Uri Wilensky for appropriate licenses for redistribution for profit.

This model was created as part of the project: CONNECTED MATHEMATICS: MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL MODELS (OBPML).  The project gratefully acknowledges the support of the National Science Foundation (Applications of Advanced Technologies Program) -- grant numbers RED #9552950 and REC #9632612.

This model was converted to NetLogo as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227. Converted from StarLogoT to NetLogo, 2001.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

person lefty
false
0
Circle -7500403 true true 170 5 80
Polygon -7500403 true true 165 90 180 195 150 285 165 300 195 300 210 225 225 300 255 300 270 285 240 195 255 90
Rectangle -7500403 true true 187 79 232 94
Polygon -7500403 true true 255 90 300 150 285 180 225 105
Polygon -7500403 true true 165 90 120 150 135 180 195 105
Rectangle -7500403 true true 105 180 165 225
Circle -7500403 false true 120 165 30

person righty
false
0
Circle -7500403 true true 50 5 80
Polygon -7500403 true true 45 90 60 195 30 285 45 300 75 300 90 225 105 300 135 300 150 285 120 195 135 90
Rectangle -7500403 true true 67 79 112 94
Polygon -7500403 true true 135 90 180 150 165 180 105 105
Polygon -7500403 true true 45 90 0 150 15 180 75 105
Polygon -7500403 true true 105 15 150 45 105 45
Circle -7500403 true true 75 0 30

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
