globals [
  distances
  average-language
  most-distinctive
  distance-between-edges
  avg-intelligible-neighbors
  ;distinct-languages
  max-possible-distance
]

patches-own[
  language ;;list with three numbers
]

to-report distance-between-languages [lang1 lang2]
  let squares (map [(?2 - ?1) ^ 2] lang2 lang1)
  report sqrt (sum squares)
end 

to-report is-obstacle? [lang]
  ifelse (sum lang) < 0
  [report true]
  [report false]
end 

to-report compute-max-possible-distance
  report distance-between-languages (n-values number-of-attributes [0]) (n-values number-of-attributes [255])
end

to-report intelligible-dist ;computes maximum intelligible distance
  report ((threshold-of-intelligibility + 0.0001) / 100) * max-possible-distance
end

to-report compute-average-language
  let languages [language] of patches
  let simplified_mean mean (map mean languages)
  
  report first [language] of patches with-min [abs((mean language) - simplified_mean)]
end

to-report compute-distance-between-edges
  let max-language first [language] of patches with-max [mean language]
  let min-language first [language] of patches with-min [mean language]
  
  report distance-between-languages max-language min-language
end

to-report most-average [languages]
  let simplified map mean languages
  let sl map [abs((mean ?1) - mean simplified)] languages
  let min_sl min sl
  
  report item (position min_sl sl) languages 
end

to-report intelligible-to-average
  report (count patches with [distance-between-languages average-language language < intelligible-dist])/(count patches) * 100
end

to-report compute-most-distinctive
  report first [language] of patches with-max [distance-between-languages language average-language]  
  
end

to-report distinctiveness ;distinctiveness = distance between the average and the furthest edge of the distribution of languages
  report distance-between-languages most-distinctive average-language
end

to-report unintelligible? [a b] ;method written relatively late: many times in code I do the check directly instead of calling this function
  report ((distance-between-languages a b) > 
    intelligible-dist)
end

to-report distinct-languages ;this method runs very slowly and should be not be called every tick, particularly on a very non-uniform set of languages
  let languages [language] of patches
  let lang-list (list average-language)
  set languages (filter [unintelligible? average-language ?] languages)
  
  while [not empty? languages] [
    let new-lang most-average languages
    set lang-list fput new-lang lang-list
    set languages (filter [unintelligible? new-lang ?] languages) 
  ]
  
  report lang-list
  
end

to-report compute-avg-intelligible-neighbors
  report mean [count intelligible-neighbors] of patches
end

to update-metrics
  ;this procedure exists to avoid having multiple outside calls to an expensive reporter per tick
  set average-language compute-average-language 
  set most-distinctive compute-most-distinctive
  set distance-between-edges compute-distance-between-edges
  set avg-intelligible-neighbors compute-avg-intelligible-neighbors
end

to setup-uniform
  ca
  let starting-language (n-values number-of-attributes [random 255])
  ask patches [
    set language starting-language
    reset-patch-color
  ]
  set max-possible-distance compute-max-possible-distance
  update-metrics
  reset-ticks
end

to setup-random
  ca
  ask patches [
    set language (n-values number-of-attributes [random 255])
    reset-patch-color
  ]
  set max-possible-distance compute-max-possible-distance
  update-metrics
  reset-ticks
end

to setup-contact
  ca
  
  let lang_1 (n-values number-of-attributes [random 255])
  let lang_2 map [? + random contact-max-distance] lang_1
  let divider 0
  
  ask patches [
    ifelse pxcor > divider 
      [ set language lang_1]
      [ set language lang_2]
    reset-patch-color
  ]
  set max-possible-distance compute-max-possible-distance
  update-metrics
  reset-ticks
end

to spread-obstacle-with-mouse
  if mouse-down?
    [ ask patch mouse-xcor mouse-ycor
        [ set language (n-values number-of-attributes [-1])
          reset-patch-color
          ]
        display ]
end 

to go
  ask patches [
    mutate-self
    talk-to-neighbors
    reset-patch-color
  ]
  update-metrics
  tick
end 



;;
;;patch procedures
;;

to-report scaled-color [lang]
  let not-negative (max (list lang 0))
  let squished (min (list not-negative 255))
  report 0 + squished
end 

to-report intelligible-neighbors
  report neighbors with [not (unintelligible? language [language] of myself)]
end

to reset-patch-color
  if is-obstacle? language [
    set pcolor black
    stop]
  
  ifelse length language = 3 [
    let new-color map scaled-color language
    ;show language
    ;show new-color
    set pcolor new-color
    stop
  ]
  [
    set pcolor scaled-color mean language
    stop
  ]
end 

to mutate-self
  if is-obstacle? language [stop] ;;don't mutate obstacles
  let new-language (list)
  foreach language [
    set new-language lput (min list 255 (max list 0 (? + ((random-float (mutation-rate) ) * (one-of (list 1 -1)))))) new-language
  ]
  set language new-language
end 

to talk-to-neighbors
  let possible-neighbors patches with [distance myself <= speech-radius]
  let num-possible-neighbors (length sort possible-neighbors)
  ask n-of (min (list num-interactions num-possible-neighbors)) possible-neighbors [
    if is-obstacle? language or is-obstacle? [language] of myself [
      stop] ;;don't talk to obstacles
      talk-crawl myself
  ]
end 

to talk-average [target]
  if not (distance-between-languages language ([language] of target) < threshold-of-intelligibility) [stop] ;;don't  talk if the languages are too unrelated
  let mean-language mean (list language [language] of target)
  
  set language mean-language
  ask target [set language mean-language]
end 


;move-towards1 moves lang1 towards lang2
;move-towards2 moves lang2 towards lang1
;they're two different procedures because you can't return two values from a procedure
;without some kludge like bundling the values into a list, or having two almost identical procedures
to-report move-towards1 [lang1 lang2 influence]
  let new-lang1 (list)
  (foreach lang1 lang2 [
    ifelse (?1 > ?2) [
      set new-lang1 lput (max (list (?1 - influence) ?2 0)) new-lang1]
    [  
      set new-lang1 lput (min (list (?1 + influence) ?2 255)) new-lang1]
  ])
  report new-lang1
end

to-report move-towards2 [lang1 lang2 influence]
  let new-lang2 (list)
  (foreach lang1 lang2 [
    ifelse (?1 > ?2) [
      set new-lang2 lput (min (list (?2 + influence) ?1 255)) new-lang2]
    [  
      set new-lang2 lput (max (list (?2 - influence) ?1 0)) new-lang2]
  ])
  report new-lang2
end

to talk-crawl [target] 
  ;named talk-crawl because of a procedure I took out called talk-average
  ;this one, in contrast to that one, moves ("crawls") towards the target language instead of averaging their languages
  
  let target-language [language] of target
  let dist (distance-between-languages language target-language)
  if dist > intelligible-dist [stop] ;; don't talk if the languages are too unrelated
  let influence dist * (influence-factor / 100) ;divided by 100 so I get at states that don't behave like high influence-factor does
  
  set language move-towards1 language target-language influence
  ask target [set language move-towards2 ([language] of myself) language influence]
end 
@#$#@#$#@
GRAPHICS-WINDOW
203
10
642
470
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

SLIDER
15
299
187
332
num-interactions
num-interactions
0
8
8
1
1
NIL
HORIZONTAL

SLIDER
15
343
187
376
mutation-rate
mutation-rate
0
255
15
1
1
NIL
HORIZONTAL

SLIDER
15
391
187
424
speech-radius
speech-radius
0
5
1
1
1
NIL
HORIZONTAL

BUTTON
25
68
190
101
NIL
setup-uniform
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
26
29
187
62
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
25
114
191
147
NIL
setup-random
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
661
37
832
70
NIL
setup-contact
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
17
434
189
467
influence-factor
influence-factor
0
100
20
1
1
%
HORIZONTAL

SLIDER
15
476
216
509
threshold-of-intelligibility
threshold-of-intelligibility
0
100
95
1
1
%
HORIZONTAL

SLIDER
659
79
831
112
contact-max-distance
contact-max-distance
0
500
45
1
1
NIL
HORIZONTAL

BUTTON
659
123
833
156
NIL
spread-obstacle-with-mouse
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
16
262
188
295
number-of-attributes
number-of-attributes
0
5
3
1
1
NIL
HORIZONTAL

BUTTON
24
160
101
193
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
227
476
326
521
intelligible distance
intelligible-dist
1
1
11

PLOT
660
216
836
336
Average Over Time
Time
Language (Avg)
0.0
10.0
0.0
255.0
true
false
"" ""
PENS
"attr1" 1.0 0 -2674135 true "plot mean average-language" "plot item 0 average-language"
"attr2" 1.0 0 -10899396 true "" "plot item 1 average-language"
"attr3" 1.0 0 -13345367 true "" "plot item 2 average-language"
"mean" 1.0 0 -16777216 true "" "plot (mean average-language)"

MONITOR
661
167
838
212
average-language
map [precision ? 0] average-language
0
1
11

MONITOR
861
166
961
211
most-distinctive
map [precision ? 0] most-distinctive
1
1
11

MONITOR
980
167
1061
212
NIL
distinctiveness
1
1
11

PLOT
860
220
1060
370
Distinctiveness
Time
Distinctiveness
0.0
10.0
0.0
500.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "plot distinctiveness" "plot distinctiveness"

PLOT
861
11
1061
161
Most Distinctive
Time
Most Distinctive
0.0
10.0
0.0
255.0
true
false
"" ""
PENS
"pen-1" 1.0 0 -2674135 true "plot item 0 most-distinctive" "plot item 0 most-distinctive"
"pen-2" 1.0 0 -10899396 true "plot item 1 most-distinctive" "plot item 1 most-distinctive"
"pen-3" 1.0 0 -13345367 true "plot item 2 most-distinctive" "plot item 2 most-distinctive"
"pen-4" 1.0 0 -16777216 true "plot mean most-distinctive" "plot mean most-distinctive"

MONITOR
862
545
1067
590
intelligible-to-average
(count patches with [distance-between-languages average-language language < intelligible-dist])/(count patches) * 100
17
1
11

PLOT
862
385
1062
535
Intelligible to Average Language
Time
intelligible-to-average (%)
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot intelligible-to-average"

MONITOR
661
348
838
393
NIL
avg-intelligible-neighbors
2
1
11

PLOT
665
403
841
533
Intelligible Neighbors
Time
Neighbors
0.0
10.0
0.0
8.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "plot avg-intelligible-neighbors" "plot avg-intelligible-neighbors"

MONITOR
1074
168
1276
213
NIL
distance-between-edges
1
1
11

PLOT
1074
221
1274
371
Distance Between Edges
Time
Distance
0.0
10.0
0.0
500.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "plot distance-between-edges" "plot distance-between-edges"

@#$#@#$#@
## WHAT IS IT?

A model of language change over an area of space. Patches reprsent speakers, and their language is coded as their color.

## HOW IT WORKS

Each tick, each patch mutates its own language by MUTATION-RATE and tries to talk to NUM-INTERACTIONS other patches around it. If the language of a patch it tries to talk to is within THRESHOLD-OF-INTELLIGIBILITY percent of the maximum distance, it succeeds, and both patches move their language closer together by the lexical distance between them times INFLUENCE-FACTOR.

## HOW TO USE IT

"setup-uniform" makes all patches have the same random language. "setup-random" makes all patches have different random languages. "setup-contact" makes the left side of the landscape have one language and the right side have another; how far apart they can be is determined by the CONTACT-MAX-DISTANCE slider.

Other than that, play around with sliders and see what happens.

## THINGS TO NOTICE

If NUMBER-OF-ATTRIBUTES is not 3, the way the color is represented will change, and so the display will change pretty radically. I don't recommend moving this slider for this reason.

Many of the measures are dependent on THRESHOLD-OF-INTELLIGIBILITY. If any patch can understand any other patch, it's no surprise that every patch can understand all its neighbors.

The sides of the landscape do not wrap.

## THINGS TO TRY

Play with MUTATION-RATE and INFLUENCE-FACTOR and see how they interact. Also, the effect of NUM-INTERACTIONS and THRESHOLD-OF-INTELLIGIBILITY isn't very well represented by the charts but is fairly clear if you mess with them yourself.

Try playing with SPREAD-OBSTACLE-WITH-MOUSE. The black patches it produces do not talk to other languages and essentially act like walls. What happens when you isolate a small region? What happens when you divide the landscape in half? In quarters?

## RELATED MODELS

Although it's not related directly, the Hotellier's Law, Rumor Mill, and Voting models are all space-control models like this one. Also, there is a Language Change model in the model library but the implementation is radically different from this one.

## CREDITS AND REFERENCES

At the Modelling Commons: http://modelingcommons.org/account/models/1511

by Brian Bowyer for EECS 372, taught by Prof. Uri Wilensky
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="threshold-experiment-low" repetitions="4" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold-of-intelligibility" first="0" step="10" last="100"/>
  </experiment>
  <experiment name="influence-experiment" repetitions="2" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="influence-factor" first="0" step="5" last="100"/>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="threshold-of-intelligibility">
      <value value="25"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="mutation-experiment-low" repetitions="4" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <steppedValueSet variable="mutation-rate" first="0" step="10" last="100"/>
    <enumeratedValueSet variable="threshold-of-intelligibility">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="interactions-experiment-low" repetitions="4" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <steppedValueSet variable="num-interactions" first="1" step="1" last="5"/>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="threshold-of-intelligibility">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="mixed-experiment" repetitions="2" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="25"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="threshold-of-intelligibility">
      <value value="0"/>
      <value value="25"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="distinct-languages-mixed" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>most-distinctive</metric>
    <metric>length distinct-languages</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="50"/>
      <value value="150"/>
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="50"/>
      <value value="100"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold-of-intelligibility" first="0" step="20" last="100"/>
  </experiment>
  <experiment name="threshold-experiment-high" repetitions="4" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold-of-intelligibility" first="0" step="10" last="100"/>
  </experiment>
  <experiment name="mutation-experiment-high" repetitions="4" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <enumeratedValueSet variable="num-interactions">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <steppedValueSet variable="mutation-rate" first="0" step="10" last="100"/>
    <enumeratedValueSet variable="threshold-of-intelligibility">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="interactions-experiment-high" repetitions="4" runMetricsEveryStep="true">
    <setup>setup-uniform</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>distinctiveness</metric>
    <metric>intelligible-to-average</metric>
    <metric>average-language</metric>
    <metric>mean average-language</metric>
    <metric>most-distinctive</metric>
    <metric>mean most-distinctive</metric>
    <metric>avg-intelligible-neighbors</metric>
    <metric>distance-between-edges</metric>
    <steppedValueSet variable="num-interactions" first="1" step="1" last="5"/>
    <enumeratedValueSet variable="number-of-attributes">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speech-radius">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="influence-factor">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact-max-distance">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mutation-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="threshold-of-intelligibility">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
