patches-own [reactant-in product-in reactant product rxn-rate]
globals [centerline reactant-profile reactant-profile-turtle]


;;GLOBAL VARIABLES:
  ;;centerline = agentset of patches with pycor = 0
  ;;reactant-profile = ordered list of reactant concentrations along the longitudinal axis (centerline) of the current reactor
  ;;reactant-profile-turtle = ordered list of reactant concentrations along the longitudinal axis (centerline) of the imported turtle-based reactor


;;PATCH VARIABLES:
  ;;reactant,product = local concentration of reactant/product within the present patch
  ;;reactant-in,product-in = local concentration of reactant/product in the patch directly upstream (will become my concentration after convection)
  ;;rxn-rate = local change in reactant/product concentration, expressed in mol/L/s



to setup
  clear-all
  
  if fill-reactor?
    [ask patches [set reactant feed-concentration] 
     recolor]
  
  reset-ticks  
  
end




to go
;;1 tick = 1 second

  ask patches with [pxcor = min-pxcor] [feed-reactor]
  convective-transport
  diffusive-transport
  ask patches with [pxcor > min-pxcor] [run-kinetics]
  calculate-performance
  recolor
  update-plots
  tick
  
end




to feed-reactor  ;;Patch procedure
;;Ask patches to set their concentration to that of the feed.
  
    set reactant feed-concentration
    set product 0

end



to run-kinetics  ;;Patch procedure
;;Calculate instantaneous deterministic rxn rate at each patch according to local concentration
;;Ask each patch to add the quantity reacted per tick to their previous concentration 

   set rxn-rate (rate-constant * ( reactant ^ rxn-order )) ;mol reacted / L / s
   set reactant ( reactant - rxn-rate )
   set product ( product + rxn-rate )
   
end



to convective-transport
;;RUNS CONVECTION PROCESS FOR PATCHES  
  
;;Set ticks per convection step
  let steps-per-tick  (( feed-velocity * (world-width - 1)  ) / reactor-length)
  
  ;;Ask patches to undergo convection steps-per-tick times per tick
  let counter 0
  while [counter < steps-per-tick]
    [flow
      set counter (counter + 1)]
    
end




to flow
;;ACTUAL CONVECTIVE TRANSPORT OCCURS HERE
  ;;Fluid moves one patch per tick.  
  ;;Patches store the incoming reactant and product before setting their own in order to make sure that every patch inherits the current value of its upstream neighbor.
  
  ;;Store concentration of incoming fluid
  ask patches with [pxcor > min-pxcor] [
    set reactant-in [reactant] of patch-at -1 0
    set product-in [product] of patch-at -1 0]
  
  ;;Set new concentrations    
  ask patches with [pxcor > min-pxcor] [
    set reactant reactant-in
    set product product-in]
      
end





to diffusive-transport
;;DIFFUSIVE MECHANISM FOR THE PATCH-BASED MODEL
  
  ;;Ask patches to undergo diffusion "diffusivity" times per tick.  This keeps diffusion on the same time scale as convection, as they are both in units of "executions per tick".
  let counter 0
  while [counter < diffusivity]
    [diffuse reactant 1
      diffuse product 1
      set counter (counter + 1)]
  
end




to calculate-performance
;;CALCULATES PERFORMANCE METRICS FOR THE REACTOR
  
  ;;Defines "centerline" as patches lying along the longitudinal axis.  Convenient for calculating profiles
  set centerline patches with [pycor = 0]
  
  ;;Store reactant concentration profile
  set reactant-profile (map [[reactant] of ?] (sort-on [pxcor] centerline))   

end


to recolor
;;RECOLORS PATCHES AT EACH TICK. COLOR DEPENDS UPON VISUALIZATION MODE
  
  if visualization-mode = "reactant-profile"
    [ask patches [set pcolor scale-color blue reactant 0 (feed-concentration * 2)]] ;;Scales color between blue (high reactant concentration) and black (no/low reactant)
  
  if visualization-mode = "product-profile"
    [ask patches [set pcolor scale-color red product 0 (feed-concentration * 2)]] ;;Scales color between red (high product concentration) and black (no/low products)
  
  if visualization-mode = "blended-profile"
    [ask patches 
      [ifelse (reactant + product) = 0
        [set pcolor black ]                                                     ;;If patch is empty, set color black
        [let reactant-frac (reactant / (reactant + product))                                     ;;Reactant mole fraction 
          set pcolor rgb ((1 - reactant-frac) * 255) 0 (reactant-frac * 255)]]] ;;Scales color between blue (reactants) and red (products)
  
  
end




to import-turtle-model
;;FUNCTION IMPORTS PARAMETERS AND RESULTS FROM TURTLE-BASED MODEL
;;THEN SETS CURRENT MODEL TO THOSE OPERATING PARAMETERS 
  
  
  ;;Open file containing data from turtle-based model
  file-open "turtle-results.csv"
  
  ;;If file is at the end (i.e. already imported once), close it and reopen it.
  if file-at-end? 
    [file-close-all
      file-open "turtle-results.csv"]
  
  
  ;;Store each of the variables from the other model.  List order is known a priori, not ideal but... no tengo tiempo!
  
  let dim                 read-from-string (file-read-line)
  resize-world (item 0 dim) (item 1 dim) (item 2 dim) (item 3 dim)          ;;Set dimensions to those of turtle model
  set feed-velocity       read-from-string (file-read-line)                 ;;Sets feed-velocity to turtle-based model's value
  set diffusivity         (2 * round (read-from-string (file-read-line)))   ;;Sets diffusivity to turtle-based model's value, rounded because patch model takes integer values.  The 8x is an approximate scaling factor between the two mechanisms..
  set rate-constant       read-from-string (file-read-line)                 ;;Sets rate constant to turtle-based model's reaction-probability 
  let reaction-type       file-read-line                                    ;;Stores the reaction type (i.e. conc. ind./dep. or bimolecular) as string
  set reactant-profile-turtle read-from-string (file-read-line)                 ;;Import results (conc. profile) from turtle-based model
  
  
                                                                                ;;Set appropriate rxn order based on turtle model's mechanism
  import-rxn-order reaction-type
  
  ;;Close file, resetting line count.
  file-close-all
  
end



to import-rxn-order [reaction-type]
;;CONVERTS TURTLE-BASED MODEL MECHANISM TO RXN ORDER.  BASED ON INSIGHT FROM THIS MODEL!
  
  show reaction-type
  
  ifelse reaction-type = "Bimolecular" or reaction-type = "Concentration Independent"
    [set rxn-order 1]
    [set rxn-order 2]

end



to export-results
;;WRITE OPERATING PARAMETERS AND CONCENTRATION PROFILE TO A FILE FROM WHICH IT CAN BE IMPORTED TO TURTLE MODEL

;;Normalize reactant concentration profile 
  let raw-profile (map [([reactant] of ?)] (sort-on [pxcor] centerline))
  let normalized-profile (map [? / (item (world-width - 10) raw-profile)] raw-profile)
  
  
  ;;Create file (or delete old one and reopen)   
  ifelse file-exists? "patch-results.csv"
    [file-delete "patch-results.csv"
      file-open "patch-results.csv" ]
    [file-open "patch-results.csv"]
  
  ;;Write variables to file.  I've left out labels because I can't figure out how to make the import step recognize them.
  file-print (list min-pxcor max-pxcor min-pycor max-pycor)
  file-print feed-velocity
  file-print diffusivity
  file-print (rate-constant)  
  file-print normalized-profile     
  
  ;;Save and close file  
  file-flush
  file-close

end


to-report conversion
;;CALCULATES FRACTIONAL CONVERSION
  
  ;;Conversion is defined as: ((reactant-in - reactant-out) / reactant-in )
  let outlet (mean [reactant] of patches with [pxcor = max-pxcor])
  let inlet  (mean [reactant] of patches with [pxcor = min-pxcor])
  ifelse inlet > 0
    [report (( inlet - outlet ) / inlet)]
    [report 0]
    
end


to-report mean-sq-error [profile1 profile2]
;;REPORTS MEAN SQUARED ERROR BETWEEN PATCH AND TURTLE BASED MODELS

  ;;Turtle concentration profile is scaled to a feed of 1 before export.  Here it's rescaled based on the concentration at the point 10 patches from the end of the reactor.
  let scaled-patch-conc ([reactant] of (item (world-width - 10) (sort-on [pxcor] centerline)))
  let scaled-profile2 map [? * scaled-patch-conc ] profile2 
  
  ;;Filters out first ten and last five entries from profile.  These are the regions where noise has a strong influence.
  let filtered-profile1 sublist profile1 (10) ((length profile1) - 5)
  let filtered-profile2 sublist scaled-profile2 (10) ((length scaled-profile2) - 5)
  
  
  ;;Calculates sum of mean squared error between profiles
  let squared-errors (map [(?1 - ?2) ^ 2] filtered-profile1 filtered-profile2)
  
  ;;Report sum of squared error
  report (sum squared-errors)   

end






@#$#@#$#@
GRAPHICS-WINDOW
585
25
999
220
50
20
4.0
1
10
1
1
1
0
0
0
1
-50
50
-20
20
0
0
1
seconds
30.0

BUTTON
26
30
96
65
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
124
92
199
127
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
47
92
116
127
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
-149
46
18
66
SETUP
11
0.0
1

SLIDER
47
428
184
461
rxn-order
rxn-order
0
2
1
0.5
1
NIL
HORIZONTAL

INPUTBOX
47
464
185
524
rate-constant
0.01
1
0
Number

PLOT
299
350
582
564
Overall Conversion
Time
Conversion
0.0
1.0
0.0
1.0
true
false
"" "set-plot-y-range 0 1"
PENS
"Conversion" 1.0 0 -16777216 true "" "plot conversion"

MONITOR
395
299
485
345
Conversion
conversion
2
1
11

SLIDER
47
337
205
371
diffusivity
diffusivity
0
1000
0
1
1
times/sec
HORIZONTAL

CHOOSER
308
180
460
225
visualization-mode
visualization-mode
"reactant-profile" "product-profile" "blended-profile"
2

PLOT
589
350
891
565
Reactant Profile
Reactor Length
Reactant Concentration
0.0
10.0
0.0
10.0
false
true
"" "clear-plot\n   set-plot-x-range min-pxcor max-pxcor\n   set-plot-y-range 0 (1.5 * feed-concentration + 0.5 )\n   \n   ;;Iterate across length of the reactor, plotting concentrations from each model.\n   \n   if ticks > 0 [\n   \n   carefully [\n     foreach (sort-on [pxcor] centerline)  [\n         \n         ;;Plot concentration from active patch model\n         set-current-plot-pen \"Patch Model\"\n         plotxy ([pxcor] of ?) ([reactant] of ?)\n        \n         ;;Plot stored concentration profile from synced turtle model\n         set-current-plot-pen \"Turtle Model\"\n         let position-index (position ? (sort-on [pxcor] centerline))      ;;Store list position of current x-coordinate\n         let scaled-patch-conc ([reactant] of (item (world-width - 10) (sort-on [pxcor] centerline)))\n         let scaled-profile-turtle map [? * scaled-patch-conc ] reactant-profile-turtle ;;Turtle concentration profile is scaled to a feed of 1 before export.  Here it's rescaled.\n         if (item position-index scaled-profile-turtle) > 0\n           [plotxy ([pxcor] of ?) (item position-index scaled-profile-turtle)] ]\n\n    ] \n    \n    ;; If error is thrown, just plot patch model results.\n    [ foreach (sort-on [pxcor] centerline)  [\n         \n         ;;Plot concentration from active patch model\n         set-current-plot-pen \"Patch Model\"\n         plotxy ([pxcor] of ?) ([reactant] of ?) ]]]"
PENS
"Patch Model" 1.0 0 -16777216 true "" ""
"Turtle Model" 1.0 0 -2674135 true "" ""

SWITCH
100
30
229
63
fill-reactor?
fill-reactor?
0
1
-1000

PLOT
895
350
1197
565
Reaction Rate Profile
Reactor Length
Average Rate
0.0
10.0
0.0
1.0
true
false
"" "clear-plot\n   set-plot-x-range min-pxcor max-pxcor\n   carefully[\n     foreach sort-on [pxcor] centerline\n       [plotxy ([pxcor] of ?) ([rxn-rate] of ?)]][]"
PENS
"default" 1.0 0 -16777216 true "" ""

SLIDER
48
258
195
292
feed-velocity
feed-velocity
0
5
1
1
1
m/s
HORIZONTAL

SLIDER
307
233
457
266
reactor-length
reactor-length
1
100
100
1
1
m
HORIZONTAL

MONITOR
299
299
389
345
Space Time
reactor-length / feed-velocity
2
1
11

BUTTON
309
24
494
72
Import Turtle Model Params.
import-turtle-model
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
50
183
256
217
feed-concentration
feed-concentration
1
10
1
1
1
mol/L
HORIZONTAL

TEXTBOX
84
242
167
265
CONVECTION
11
0.0
1

TEXTBOX
87
319
159
338
DISPERSION
11
0.0
1

TEXTBOX
84
410
146
429
REACTION
11
0.0
1

TEXTBOX
73
167
203
187
FEED CONDITIONS
11
0.0
1

BUTTON
309
75
493
121
Export Parameters
export-results
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
523
117
596
146
REACTOR\nENTRANCE
11
0.0
1

MONITOR
593
573
714
618
MSE Between Profiles
mean-sq-error reactant-profile reactant-profile-turtle
4
1
11

TEXTBOX
727
580
839
608
Error exludes first 10 and last 5 points
11
0.0
1

@#$#@#$#@
## WHAT IS IT?

This is an agent-based approach to modeling continuous-flow chemical reactors.  By using simple agent-level rules to recreate a number of known reaction engineering phenomena, the model is designed to serve as an educational tool for chemical engineering students.

The scope of the model is analogous to that of the convection-dispersion-reaction system taught in undergraduate chemical engineering courses.  Namely, non-idealities due to fluid dynamics and bulk diffusion effects are rolled into a single diffusion mechanism called dispersion.

The model operates from the Eulerian reference frame, as all actions are assigned to stationary patches.  This model is designed to supplement another NetLogo model operating from the Lagrangian reference frame (see RELATED MODELS), but may be used independently if desired.



## HOW IT WORKS

The reactor is oriented such that the feed stream enters at the left boundary and leaves at the right boundary. The model consists of patches to which two properties are assigned: reactant and product concentrations.   

On setup, the far left column of patches sprout reactant at a user-specified concentration (number per patch).  If the FILL-REACTOR? option is switched on, all patches are filled with reactant in this manner.

A significant portion of the code was written for analysis purposes and is not crucial to the function of the reactor.  Consequently, all such code has been omitted from the pseudo-code below.



At each tick: 
	
	
	1. Feed enters the reactor.  Patches at the far left boundary are asked to set their reactant and product concentrations to that of the feed.
	2. Convection occurs.  Species flow as a result of each patch inheriting the concentrations of its upstream neighbor.  The FEED-VELOCITY is used to determine the number of convection steps per tick, and the single-step flow is repeated until this number of steps have been taken.
	3. Diffusion occurs.  Each patch is asked to diffuse 100% of its contents to surrounding patches.  This action is repeated DIFFUSIVITY times per step.  
	4. Reaction occurs.  The local rate is deterministically calculated by each patch.  The patch inserts its local concentrations into the specified rate equation, which is of the form:  rxn-rate = RATE-CONSTANT * (reactant) ^ RXN-ORDER.  This form is consistent with typical rate expressions for elementary reactions.
	5. Performance metrics are calculated. Inlet and outlet average concentrations are used to determine the conversion, while patches along the longitudinal axis are aggregated for plotting concentration profiles.




## HOW TO USE IT

The best way to the learn the model is to play around with parameters and view the results. The model is essentially 1-D, but has been constructed in 2-D so as to maximize interpretability.  To run the model:

	1. Select a desired FEED-CONCENTRATION. 

	2. Select a FEED-VELOCITY to set the convection rate.

	3. Select a DIFFUSIVITY to set the dispersion rate.

	4. Select a RATE-CONSTANT and REACTION-ORDER to set the rate law.

	5. Click Go

	OPTIONAL: To export these parameters to the turtle-based model, click the EXPORT PARAMETERS button, then click the IMPORT PATCH MODEL PARAMS. button on the turtle model's interface.  Similarly, to match a turtle-model export its parameters then click the IMPORT TURTLE MODEL PARAMS. button on this model's interface.	



## THINGS TO NOTICE

The model is able to capture dynamic behavior, including startup, with no lag time.  This is not typically coded into high level reactor models, and allows for some interesting analyses as well as potential for developing control strategies.

The model allows for direct input of reaction rate expressions.  To model a real reactor, known kinetics can be entered into the patch-based model.  The results can then be exported to the turtle-based model, where the DIFFUSIVITY is tuned to match the residence time distribution of a real tracer response experiment.

The stochastic reaction mechanisms of the turtle-based model can be mapped to real rate expressions here.  Allow the turtle-based model to reach steady state with the desired mechanism and a sufficiently wide sampling window, then export its parameters.  Import those parameters in the patch-based model and tune the rate law to match the reactant profile.  In general, it appears that a first order rate constant is exactly the same as a CONCENTRATION-INDEPENDENT REACTION-PROBABILITY.  Similarly, the CONCENTRATION-DEPENDENT rate mechanism from the turtle-based model produces a concentration profile that can be  mapped to a second order reaction in the patch-model. The BIMOLECULAR mechanism maps to a first order model, presumably because collision with a second reactant is a requirement but does not actively increase the probability of reaction. In the CONCENTRATION-DEPENDENT mechanism, the REACTION-PROBABILITY scales linearly with the number of collisions.


## THINGS TO TRY


Try obtaining a steady state in the turtle-based model and import the results to this model.  Can you map the turtle-based model's stochastic reaction mechanisms to the  deterministic rate laws in this model?

Try to achieve a perfectly mixed ideal CSTR.  Is this feasible in this model? Is it feasible in the turtle-based model?  If not, try to determine why the two models might differ in this regard.


## EXTENDING THE MODEL

Potential extensions could entail:

	1. Building a means to import a reaction network, complete with multiple products, reactants, and stoichiometries.
	2. Implementation of heat effects.  
	3. Addition of a mechanism that tracks residence time.


## RELATED MODELS

The accompanying turtle-based model is available on the NetLogo Modeling Commons. This model handles everything from the perspective of mobile reactants. :

http://modelingcommons.org/browse/one_model/4401#model_tabs_browse_info

  
## CREDITS AND REFERENCES

The model was created for EECS 472 by Sebastian Bernasek at Northwestern University. The latest version of this model is available on the NetLogo Modeling Commons:

http://modelingcommons.org/browse/one_model/4402#model_tabs_browse_info


@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

head-2
true
0
Polygon -16777216 true false 15 285 147 1 270 285
Polygon -1 true false 27 287 150 0 21 281 27 287
Polygon -1 true false 273 287 150 0 279 281 273 287

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

worm-head
true
0
Polygon -16777216 true false 30 150 150 0 255 150
Polygon -16777216 true false 30 150 150 300 255 150

worm-segment
true
0
Rectangle -16777216 true false 60 30 240 270

worm-segment-2
true
0
Circle -16777216 true false 15 15 270
Line -1 false 60 60 30 150
Polygon -1 true false 240 60 270 165 225 255 270 195 285 150 270 90
Polygon -1 true false 60 60 30 165 75 255 30 195 15 150 30 90

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0-RC4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="NumberFans" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>red-contentness</metric>
    <metric>blue-contentness</metric>
    <enumeratedValueSet variable="minimum-proximity">
      <value value="0.5"/>
      <value value="0.75"/>
      <value value="1"/>
      <value value="1.25"/>
      <value value="1.5"/>
      <value value="1.75"/>
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-blue-fans">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-red-fans">
      <value value="500"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
