globals [ maxmass minmass nofreeze BIAS<0 BIAS>0 ]
patches-own [ mass bias ]
breed [ antimatter antiparticle ]
breed [ normal-matter normal-particle ]

to setup
  clear-all
  ask patches [ set mass 1 set bias 0 ]
  create-antimatter population / (100 / initial-antimatter-percent) [
    set color green set shape "triangle" ]
  create-normal-matter population / (100 / (100 - initial-antimatter-percent)) [
    set color yellow set shape "triangle 2" ]
  ask turtles [ setxy random-xcor random-ycor ]
  mark-mass
  mark-bias
  reset-ticks
end

to go   reset-timer
  display-options
  mark-mass  ; Increase the mass in a radius around particles.
  mark-bias  ; Blueshifted area in front and redshifted area behind particles.
  mark-color ; Patches show shifting and mass at the same time.
  ask turtles [
    turnit ; Wiggle particles a little and then turn toward higher mass areas.
    moveit ; Move particles slower when others move in dissimilar directions.
  ]
  adjust-population ; Annihilation, quantum fluctuation, and particle creation.
  tick show timer
end


;+++++++++++++++++++++++++++++++++++++++++++++++++
; Marking routines
;+++++++++++++++++++++++++++++++++++++++++++++++++

To display-options
  ; Simple display options.
  ifelse pen? [ ask turtles [ pendown ]][ ask turtles [ penup ]]
  ifelse hide-particles? [ ask turtles [ hide-turtle ]][ ask turtles [ show-turtle ]]
end

to mark-mass
  ; Drop mass onto patch.
  ask patches [ set mass 1 ] ; Reset.
  ask turtles [
    ask patches in-radius radius-cutoff [
      set mass (mass + (1 / ((distance self ^ 2) + 1)))
    ]
  ]
end

to mark-bias
  ; Make redshift and blueshift areas.
  if any? turtles [
    ifelse relativistic-bias? [
      ask patches [ set bias 0 ]
      ask turtles [
        ; Blueshift area ahead of particle.
        ask patches in-cone radius-cutoff 180 [ set bias bias + 1 ] rt 180
        ; Redshift area behind particle.
        ask patches in-cone radius-cutoff 180 [ set bias bias - 1 ] rt 180
      ]
      set BIAS<0 (mean [ mass ] of patches with [ bias < 0 ])
      set BIAS>0 (mean [ mass ] of patches with [ bias > 0 ])
      ][ ; If no bias, patches are shaded violet.
      ask patches [set bias 0]
      set BIAS<0 0
      set BIAS>0 0
      ]
  ]
end

to mark-color
  ; Color is limited to a small set.
  if any? turtles [
    set maxmass max [ mass ] of patches
    set minmass min [ mass ] of patches - population / 2000 ; Make it brighter.
  ]
  ask patches [
    If bias > 0     [ set pcolor scale-color blue   mass (minmass) (maxmass) ]
    Ifelse bias < 0 [ set pcolor scale-color red    mass (minmass) (maxmass) ]
                    [ set pcolor scale-color violet mass (minmass) (maxmass) ]
  ]
end


;+++++++++++++++++++++++++++++++++++++++++++++++++
; Turing and moving routines
;+++++++++++++++++++++++++++++++++++++++++++++++++

to turnit  ; turtle procedure
  ; Wiggle it.
  rt random-float (7 - random-float 7)
  ; Turn it.
  let greatest-mass max-one-of patches in-cone 2 gravity-search-angle [ mass ]
  if greatest-mass != nobody [
    let oldheading heading ; Where were we going?
    face greatest-mass ; Head towards the greatest mass.
    let shift subtract-headings heading oldheading ; How much change was that?
    rt shift + 180 ; Turn around.
    let shifted one-of turtles in-cone (radius-cutoff * 2) 180 ; Picks a turtle.
    ifelse shifted != nobody
      [ ask shifted [ rt shift ]]        ; Conserve angular momentum
      [ ask one-of turtles [ rt shift ]] ; somewhere,
    lt 180 + shift                       ; and turn back.
  ]
end

to moveit ; turtle procedure
  ; Move particles less around particles with dissimilar headings.
  let similar 1 let all 1
  let vectors other turtles in-radius 3
  if vectors != nobody [
    if count vectors > 3 [ set vectors (n-of 3 vectors) ]
    ask vectors [ ; Only looking at up to three nearby particles.
      let distancenow  (1 / (1 + distance myself)) ; Smaller effect further away.
      let angle subtract-headings heading [ heading ] of myself
      ; Scale to speed between ~0 and 1.
      set similar (similar + ((((90 - (abs angle)) / -180) + 0.5)) * distancenow)
      set all (all + distancenow)
    ]
  ]
  fd (similar / all)
end


;+++++++++++++++++++++++++++++++++++++++++++++++++
; Adjusting the population routines
;+++++++++++++++++++++++++++++++++++++++++++++++++

to adjust-population
  set nofreeze population
  ask patches [ annihilate ]
  QF ; Comment this out if you don't want 5% random fluctuation.
  more less
end

to annihilate ; patch procedure
  ; Ask each patch if they have antimatter and normal-matter.
  let A one-of antimatter in-radius 2
  let C one-of normal-matter in-radius 2
  if A != nobody [
    if C != nobody [
      ask A [
        ask C [
        ; Ask if headings cancel each other out.
          if abs subtract-headings heading [ heading ] of myself > 150 [
            ; Then annihilate both.
            ask A [die] ask C [die]
            annihilate ; And repeat.
          ]
        ]
      ]
    ]
  ]
end

to QF
  ; 'Quantum fluctuation' to make a random particle.
  ; Useful to try to not get stuck at 100% antimatter or normal-matter.
  if any? turtles [
    if random 100 = 5 [
      ask one-of turtles [ die ]
	  ]
  ]	
end

to more
  ; Create a new particle when the population is below the initial population.
  if count turtles < population [
    set nofreeze nofreeze - 1
    if nofreeze > 0 [ ; Protects the program from looping too much.
      ask one-of patches [
        ; Puts more weight on spots with less mass.
        if random (mass + 1) = 1 [ make-particle ]
      ]
    more ; Repeat until the population is stable or nofreeze =< 0.
    ]
  ]
end

to make-particle  ; patch procedure
  ; Creates a new particle depending on the bias at the location.
  If bias > 0 [
    ; Antimatter bias in blueshifted areas.
    sprout-antimatter 1 [ set color green  set shape "triangle" ]]
  Ifelse bias < 0 [
    ; Normal-matter bias in redshifted areas.
    sprout-normal-matter 1 [ set color yellow set shape "triangle 2" ]]
    ; No bias if no shift or balanced shifts.
    [ make-random-particle ]
  set nofreeze population
  more ; Repeat.
end

to make-random-particle ; patch procedure
  ; 50% chance of creating either antimatter or normal-matter.
  ifelse random 2 = 1
    [ sprout-normal-matter 1 [ set color yellow set shape "triangle 2" ]]
    [ sprout-antimatter 1    [ set color green  set shape "triangle"   ]]
  more
end

to less
  ; Destroy a random particle if the population slider is lowered.
  if count turtles > population [
    ask one-of turtles [ die ] less ; Repeat until the population is stable.
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
260
115
760
616
-1
-1
12.0
1
8
1
1
1
0
1
1
1
-20
20
-20
20
1
1
1
ticks
30.0

SLIDER
260
80
425
113
population
population
5
2000
2000.0
5
1
NIL
HORIZONTAL

BUTTON
260
45
350
78
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
350
45
425
78
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
0
130
260
266
PLOT 2: Bias Count
time
% of patches
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Redshifted" 1.0 0 -2674135 true "" "plot 100 * (count patches with [bias < 0] / count patches)"
"Blueshifted" 1.0 0 -13345367 true "" "plot 100 * (count patches with [bias > 0] / count patches)"

PLOT
0
10
260
130
PLOT 1: Population
time
particles
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"normal-matter" 1.0 0 -12440034 true "" "plot count normal-matter"
"antimatter" 1.0 0 -10899396 true "" "plot count antimatter"

SLIDER
260
10
425
43
initial-antimatter-percent
initial-antimatter-percent
1
99
95.0
1
1
NIL
HORIZONTAL

SWITCH
425
10
585
43
relativistic-bias?
relativistic-bias?
0
1
-1000

PLOT
0
265
260
405
PLOT 3: Mass Distribution
time
average mass
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Blueshifted" 1.0 0 -14070903 true "" "plot BIAS>0"
"Redshifted" 1.0 0 -5298144 true "" "plot BIAS<0"

SWITCH
585
45
705
78
pen?
pen?
1
1
-1000

SWITCH
585
10
705
43
hide-particles?
hide-particles?
1
1
-1000

BUTTON
585
80
705
113
clear pen
clear-drawing
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
425
45
585
78
radius-cutoff
radius-cutoff
1
10
4.0
1
1
NIL
HORIZONTAL

BUTTON
0
405
260
438
NIL
 clear-all-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
425
80
585
113
gravity-search-angle
gravity-search-angle
60
180
90.0
3
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

At the beginning of the universe, antimatter and normal-matter were presumably created in equal amounts. It is an open question why we now have more of one type than the other. Theoretically, the two should have annihilated each other, so it's a mystery why we have any matter at all.

This model demonstrates the relativistic feedback loop that suppresses the creation of antimatter and leads to a normal-matter dominated universe regardless of initial conditions and spatial separation. The bias toward normal-matter is not apparent with small populations or individual particles; the effect is emergent with larger populations.  


## HOW IT WORKS

Particles mark their mass and relativistic bias, turn toward higher mass, and move. Patches test for annihilation, and then new particles are created based on the mass and bias at patches. The model accounts for the relativistic time dilation near mass, so there are less moments in a higher mass area and thus a lower probability for a particle to be randomly created there. 

The area in front of particles is blueshifted and the area behind particles is redshifted. Areas with more blueshift will create an antimatter particle when they are chosen, the areas with more redshift will create a normal-matter particle when they are chosen, and areas with no bias or equal bias have a 50/50 chance to create either type of particle. If the relativistic bias is turned off then random particles are created. 

Particles tend to cluster together making smaller but more populous blueshifted cores and larger but less populous redshifted fields. The greater number of particles created in the redshifted area then annihilate the particles from the blueshifted area until they dominate the field. We call the particles that inevitably dominate 'normal-matter.'


## HOW TO USE IT

INITIAL-ANTIMATTER-PERCENT changes the 50% antimatter and 50% normal-matter default setup. While the Big Bang model and total universe may be zero-sum, it is useful to experiment with the possibility that different regions emerged with different antimatter/normal-matter mixes. This slider has no effect after setup. 

SETUP creates a collection of randomly distributed normal-matter and antimatter particles. These toy particles are neutral in charge, flavor, and spin. They have equal mass and velocity. The particles are only differentiated by their origins, making them antimatter or normal-matter. Antimatter and normal-matter annihilate each other, while particles of the same type do not.

GO starts the simulation. Antimatter particles are green triangles with black lines. Normal-matter particles are yellow triangles with smaller black tringles.

POPULATION sets the amount of particles at setup and acts as a maximum while running. 

RELATIVISTIC-BIAS? allows the redshift and blueshift to influence particle generation. When *off* the type of particle will be randomly chosen when creating new particles.

RADIUS-CUTOFF affects three parts in the code: How far is mass marked? How far is bias marked? When a particle turns toward the highest mass nearby, how far behind a particle does it look when asking another particle to turn the complimenting amount to conserve angular momentum? Technically, each radius should propagate at a finite speed and have infinite range, so this setting helps offset the idiosyncrasies of the model.

GRAVITY-SEARCH-ANGLE sets the angle of the cone used to find the largest mass in front of the particle. It influences the shape of the clusters. 

HIDE-PARTICLES? helps you see the mass and bias distribution without the particles in the way. New particles are hidden only after they are created, so you can also see exactly where new particles are made with this turned *on*.  

PEN? and CLEAR-PEN help track where the particles and clusters are going. 

CLEAR-PLOTS resets the graphs. 


## THINGS TO NOTICE

The evolution of the population follows some basic patterns. 

**Stage one:** Annihilation reduces the population and triggers new particle creation.

**Stage two:** Antimatter creation is suppressed; normal-matter creation is promoted. 
A look at stage two in detail:

PLOT 1 shows the distribution of particles. Even if the antimatter count starts or veers higher than the normal-matter count, once the normal-matter domination process starts it rarely stops. This stage only starts if the population, radius-cutoff, and gravity-search-angle are high enough.
PLOT 2 shows the count of redshifted patches vs blueshifted patches. The particles turn toward each other, speed up toward a point, and then slow down as they approach the particles with dissimilar headings. So blueshifted areas are concentrated in a smaller area, and redshifted areas take up more space outside clusters.
PLOT 3 shows the average mass per patch (particles in the radius-cutoff weighted for distance). Because the particles converge and slow down, the blueshifted areas generally have more mass in them. Patches with more mass have less causal actions per moment.  

Less space and less actions in blueshifted areas leads to a natural bias toward matter made in redshifted areas. 

**Stage three:** Normal-matter dominates and complexity increases.

Due to frequent annihilations, the complexity of clusters seems to be limited until one type of matter dominates. Try waiting or skipping to a 95% normal-matter universe and experiment with the cut-off radius and gravity-search-angle. Particles will merge into clusters, but those clusters will spread out depending on their search angle and radius. For example, low radius-cutoff (set to 2) and 90 gravity-search-angle will make the particles form long lines and webs.

Sometimes clusters become more complex than a cluster with a blueshifted core: the cluster will have a half-blueshifted and half-redshifted semicircles. These clusters can be stable and move across the field. In an open universe, these clusters could seek other systems to integrated into. Sometimes antimatter systems will linger like this for a while until quantum fluctuations help introduce enough normal-matter to start the feedback cycle. Notice that the semicircles can appear to spin around and show the rotation of the cluster.

Finally consider the observation that the universe is expanding because of the redshifted photons we see. In this model, this makes sense because the blueshifted photons are hidden in clusters that are getting more condensed and complex. The universe is functionally expanding, but the operational size is stable. 


## THINGS TO TRY

Adjust the initial-antimatter-percent.

Notice that normal-matter domination is inevitable. The relativistic effects create a feedback loop so even if a model starts with 95% or more antimatter, normal-matter will dominate eventually. This is in sharp contrast to the conception that the initial conditions of the universe must have had slightly more normal-matter than antimatter to arrive at our current universe, and that the universe must have been compact to allow for similar conditions to prevail throughout space. 


## EXTENDING THE MODEL

**Standard model:** Try adding and differentiating particles. 

**Gravity:** The model uses different routines to turn and slow the particles. Other methods to code gravity work as well. To be more realistic, try making one that is not instantaneous and spreads through the entire field. 

**Resilient model:** The current model has particles created anywhere in the field. Another step could be to only make particles in far-from-equilibrium conditions, such as where blueshifted areas meet other blueshifted areas and redshifted areas meet other redshifted areas. This should help regulate big crunch and big tears in operational space.


## NETLOGO FEATURES

**Weighted Patches:** The RND extension allows for weighted randomness, but it does not work for patches. To pick a weighted patch, we pick one at random, ask ``if random mass + 1 = 1``, make a particle if it is true, and pick a different patch if it is not true. Thie code continues to look for a patch but stops making more particles if it loops too often (like in huge populations with a high radius-cutoff). 

Another workaround is using redshifted and blueshifted patches instead of using individual particles to influence the new particles. Particles work, but using patches saved us time and memory. 


## DEEPER THEORY

The model assumes particles have a quantum roll. Fermions have integer quantum rolls, so their signal does not diminish over space or time, but they have half-integer quantum spins causing their path to be curved in an electromagnetic field (which is not included in this toy model). Second, bosons have integer quantum spins so their path is not curved in an electromagnetic field, but this model proposes that bosons have a half-integer quantum roll which contributes to the diminished signal of bosons over space and time. 

What is quantum roll? Another way to look at it uses Turing machine/M-theory duality perspective. Vector-machines/strings in the bulk are normally too slow to move alone and too small to make a signal out from the noise of the bulk. But quantized arrays of similar vector-machines/strings can roll through the bulk faster and make a noticeable signal. The process is like a rolling paceline of bicycles (Belgian tourniquet), so the leader moves back to be replaced by a new leader, ensuring distributed work. So fermions, particles with integer rolls, have equal amounts of vector-machines coming in and out of the bulk. Bosons, particles with half-integer rolls, have an asymmetric amount of vector-machines/strings going into or out of the bulk. This model of antimatter suppression presents one prediction of quantum roll.


## RELATED MODELS

Gravity in this model is similar to dropping and turning toward a higher amount of chemicals in the Ants and Slime models. The process of moving was inspired by the Flocking models.


## CREDITS AND REFERENCES

* Coyle, Dylan. (2017). NetLogo Antimatter Suppression model. 

Created with NetLogo 6.0.1 :
* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
true
0
Polygon -7500403 true true 150 30 15 255 285 255
Line -16777216 false 150 30 150 255

triangle 2
true
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 129 225 253 75 254

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <metric>count antimatter</metric>
    <metric>count normal-matter</metric>
    <metric>count patches with [bias &gt; 0]</metric>
    <metric>count patches with [bias &lt; 0]</metric>
    <metric>count [ mass ] of patches with [ bias &gt; 0 ]</metric>
    <metric>count [ mass ] of patches with [ bias &lt; 0 ]</metric>
    <enumeratedValueSet variable="relativistic-bias?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-antimatter-percent">
      <value value="5"/>
      <value value="45"/>
      <value value="50"/>
      <value value="55"/>
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population">
      <value value="10"/>
      <value value="100"/>
      <value value="1000"/>
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-cutoff">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gravity-search-angle">
      <value value="90"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
