breed[pigs pig]
breed[fences fence]
breed[red-zones red-zone]

globals
[
  climate-daylight?
  climate-daylight-start
  climate-daylight-end
  climate-sunhours
  climate-sunhours-statistic
  climate-temperature
  climate-temperature-minimum-statistic
  climate-temperature-maximum-statistic

  schedule-feeding-time?
  schedule-feeding
  sound-rest
  sound-unrest


  time-day
  time-daytime-start
  time-daytime-duration
  time-hour
  time-minute
  time-month
  time-month-name
  time-month-range
  time-year
  time-cumulative-minutes
  time-section
]

patches-own
[
  patch-allowed-for-activities?
  patch-eating-line-occupied?
  patch-eating-line?
  patch-feeder?
  patch-fence?
  patch-slatted?
  patch-slatted-occupied?
  patch-strawed?
  patch-strawed-occupied?
]

pigs-own
[
  pig-dominant?
  pig-eating?
  pig-cumulative-resting-time-on-slatted-floor
  pig-happiness
  pig-has-lying-down-mate?
  pig-has-sleeping-mate?
  pig-lying-down?
  pig-nearest-playmate
  pig-playmates
  pig-playing?
  pig-reach-feeder?
  pig-schedule-urinate
  pig-schedule-urinate-time-hour-start
  pig-schedule-urinate-time-minute-start
  pig-sleeping?
  pig-speed
  pig-time-to-eat
  pig-time-to-lie-down
  pig-time-to-rest
  pig-time-to-sleep
  pig-time-to-walk-to-feeder
  pig-to-eat?
  pig-to-lie-down?
  pig-to-sleep?
  pig-urinating?
]

to initialise-schedule
  let t-hour time-daytime-start
  let t-minute 0
  let t list t-hour t-minute
  set schedule-feeding lput t []

  while[t-hour >= time-daytime-start and t-hour < time-daytime-start + time-daytime-duration]
  [
    set t-minute  (t-minute + pig-feeding-interval) mod 60
    set t-hour floor (t-hour + (pig-feeding-interval / 60))
    set t list t-hour t-minute
    set schedule-feeding lput t schedule-feeding
  ]
end

to initialise-pig-schedule-urinate
  let pig-urinate-interval (24 * 60 / pig-urinating-frequency) ;;minutes

  ask pigs
  [
    set pig-schedule-urinate-time-hour-start random 24
    set pig-schedule-urinate-time-minute-start random 60

    let t-hour pig-schedule-urinate-time-hour-start
    let t-minute pig-schedule-urinate-time-minute-start
    let t-list list t-hour t-minute
    let t (t-hour * 60) + t-minute

    set pig-schedule-urinate lput t-list []

    repeat (pig-urinating-frequency - 1)
    [
      set t t + pig-urinate-interval
      set t-minute  t  mod 60
      set t-hour floor (t / 60)
      set t-list list t-hour t-minute
      set pig-schedule-urinate lput t-list pig-schedule-urinate
    ]
  ]
end

to calculate-pig-resting-on-slatted-floor
  ask pigs with[pig-lying-down? or pig-sleeping?]
  [
    if([patch-slatted?]of patch-here = true)
    [
      set pig-cumulative-resting-time-on-slatted-floor pig-cumulative-resting-time-on-slatted-floor + 1
    ]
  ]
end

to define-time-section
  set time-section ifelse-value(time-hour >= time-daytime-start and time-hour < time-daytime-start + time-daytime-duration)
  [
    "day time"
  ]
  [
    "night time"
  ]
end

to setup
  ca
  initialise-plot-resting-pigs-time-on-slatted-floor
  initialise-plot-pig-cumulative-resting-time
  initialise-plot-pig-emotion

  initialise-time
  initialise-schedule
  define-time-section
  initialise-climate
  initialise-pig-housing
  initialise-pig
  initialise-pig-schedule-urinate
  calculate-dominance
  calculate-happiness
  color-pigs
  set time-cumulative-minutes 0

  reset-ticks
end

to generate-climate
  let i position time-month time-month-name
  let s item i climate-sunhours-statistic
  set climate-sunhours round (max list 0 random-normal first s last s)

  ifelse(climate-sunhours > 0)
  [
    set climate-daylight-start round (12 - round (climate-sunhours / 2))
    set climate-daylight-end climate-daylight-start + climate-sunhours
  ]
  [
    set climate-daylight-start 1e32
    set climate-daylight-end 1e32
  ]

  set climate-daylight? (time-hour >= climate-daylight-start and time-hour <= climate-daylight-end)

  let tmin item i climate-temperature-minimum-statistic
  let tmax item i climate-temperature-maximum-statistic

  set climate-temperature ifelse-value(climate-daylight?)
  [
    random-normal first tmax last tmax
  ]
  [
    random-normal first tmin last tmin
  ]

  if(heater?)
  [
    if(climate-temperature < temperature-threshold)
    [
      set climate-temperature climate-temperature + heating-temperature
    ]
  ]

end

to initialise-climate
  ;; based on Schipol weather station data of year 2014

  set climate-sunhours-statistic ;;daily sunshine duration each month (mean and sd) in hours
    [
      [1.787096774  1.705626363]  [3.285714286  2.665932439]  [6.44516129  3.174674651]  [5.793333333  3.645113246]  [6.832258065  4.786814327]  [7.9  4.120679556]
      [7.241935484  4.649499915]  [6.238709677  3.989960519]  [5.651724138  3.485441219]  [3.587096774  2.641305477]  [3.09  3.137795185]  [1.729032258  2.034894518]
    ]

  set climate-temperature-minimum-statistic ;;daily minimum temperature each month (mean and sd) in degree Celcius
  [
    [3.687096774  2.822025506]  [4.335714286  1.327825927]  [3.377419355  3.176445569]  [7.066666667  2.787080209]  [8.583870968  3.819868826]  [10.32666667  2.229880673]
    [14.95806452  2.720388966]  [12.11935484  2.995153792]  [10.93103448  2.197742908]  [10.6483871  1.897168235]  [5.75  3.158213157]  [2.5  3.744150994]
  ]

  set climate-temperature-maximum-statistic ;;daily maximum temperature each month (mean and sd) in degree Celcius
  [
    [8.206451613  2.309824738]  [9.6  1.731195263]  [13.50322581  3.658732373]  [16.5  3.127960666]  [17.80967742  3.794237283]  [20.52666667  2.752545322]
    [24.27096774  3.922345688]  [20.33870968  2.898468264]  [20.93448276  2.406579651]  [16.92903226  2.738879765]  [10.99666667  3.194875566]  [7.596774194  3.201196013]

  ]
end

to initialise-pig-housing
  ask patches
  [
    set patch-allowed-for-activities? false
    set patch-fence? false
    set patch-feeder? false
    set patch-slatted? false
    set patch-strawed? false
    set patch-eating-line? false
    set pcolor 41
  ]

  ask patches with[pxcor = min-pxcor or pxcor = max-pxcor or pycor = min-pycor or pycor = max-pycor]
  [
    set patch-fence? true
    set pcolor 65
  ]

  ask patches with
  [
    (pycor >= min-pycor + 1 and pycor <= min-pycor + 1) and
    not
    (
      (pxcor >= min-pxcor and pxcor <= min-pxcor)  or
      (pxcor >= max-pxcor and pxcor <= max-pxcor)
    )
  ]
  [
    set patch-feeder? true
    set pcolor blue
  ]

  ask patches with
  [
    (pycor = min-pycor + 2 and pycor <= min-pycor + 3) and
    not
    (
      (pxcor >= min-pxcor and pxcor <= min-pxcor)  or
      (pxcor >= max-pxcor and pxcor <= max-pxcor)
    )
  ]
  [
    set patch-eating-line? true
    set pcolor 40.75
  ]

  ask patches with
  [
    (pycor >= max-pycor - 2 and pycor <= max-pycor - 1) and
    not (patch-fence? or patch-feeder? or patch-eating-line?)
  ]
  [
    set patch-slatted? true
    set pcolor grey
  ]

  set-default-shape fences "dot"

  ask patches with[patch-fence?]
  [
    sprout-fences 1
    [
      set color 63
      set size 1
    ]
  ]

  ask patches with[not(patch-fence? or patch-feeder? or patch-slatted? or patch-eating-line?)]
  [
    set patch-strawed? true
  ]

  ask patches with[patch-eating-line?]
  [
    set patch-strawed? true
  ]

  ask patches
  [
    set patch-allowed-for-activities? (patch-strawed?  or patch-slatted?)
  ]

  ask patches with[not patch-allowed-for-activities?]
  [
    sprout-red-zones 1
    [
      ht

      set heading ifelse-value(xcor = min-pxcor)
      [
        90
      ]
      [
        ifelse-value(xcor = max-pxcor)
        [
          270
        ]
        [
          ifelse-value(ycor = max-pycor)
          [
            180
          ]
          [
            0
          ]
        ]
      ]
    ]
  ]
end

to initialise-pig
  set-default-shape pigs "pig"
  create-pigs pig-population
  [
    move-to one-of patches with[patch-strawed? and not patch-eating-line?]
    set size 2.7 ;; assumption: adult pig length = 2 m
    set pig-time-to-rest random pig-max-initial-resting-time
    set pig-schedule-urinate[]
    set pig-dominant? false
    set pig-urinating? false


    ;; to be reinitialised at night time
    set pig-reach-feeder? true
    set pig-eating? false
    set pig-to-eat? false
    set pig-playing? false
    set pig-lying-down? false
    set pig-to-lie-down? false
    set pig-has-lying-down-mate? false

    ;; to be reinitialised at day time
    set pig-to-sleep? true
    set pig-sleeping? false
    set pig-has-sleeping-mate? false

    set pig-cumulative-resting-time-on-slatted-floor 0
  ]
end

to initialise-time
  set time-year 1
  set time-month-range[[1 31] [32 59] [60 90] [91 120] [121 151] [152 181] [182 212] [213 243] [244 273] [274 304] [305 334] [335 365]]
  set time-month-name["jan" "feb" "mar" "apr" "may" "jun" "jul" "aug" "sep" "oct" "nov" "dec"]
  set time-month first time-month-name
  set time-day 1 ;;julian date
  set time-hour 0
  set time-minute 0
  set time-daytime-start 6
  set time-daytime-duration 12
end


to update-time
  set time-minute (time-minute + 1) mod 60
  if(time-minute = 0)
  [
    set time-hour (time-hour + 1) mod 24
    if(time-hour = 0)
    [
      set time-day (time-day + 1) mod 366

      if(time-day = 0)
      [
        set time-day 1
        set time-year time-year + 1
      ]

      foreach time-month-range
      [
        let i position ? time-month-range
        if(time-day >= first ? and time-day <= last ?)
        [
          set time-month item i time-month-name
        ]
      ]
    ]
  ]
  set time-cumulative-minutes time-cumulative-minutes + 1
end

to color-pigs
  let m max [pig-happiness]of pigs
  ask pigs
  [
    ;; not dominant pigs
    set color ifelse-value(not pig-dominant?)
    [
      ifelse-value(m > 0)
      [
        18 - (pig-happiness / m)
      ]
      [
        18 - random-float 1
      ]
    ]
    ;; dominant pigs
    [
      ifelse-value(m > 0)
      [
        23 - (pig-happiness / m)
      ]
      [
        23 - random-float 1
      ]
    ]
  ]
end

to calculate-speed
  let m max [pig-time-to-rest]of pigs
  let s (pig-base-speed / 60) ;;convert mph into cells per tick
  ask pigs
  [
    set pig-speed ifelse-value(not (pig-eating? or pig-lying-down? or pig-urinating? or pig-sleeping?))
    [
      ifelse-value(m > 0)
      [
        s + random-float (pig-time-to-rest / m);;cells per tick
      ]
      [
        s + random-float 1
      ]
    ]
    [
      0
    ]
  ]
end


to calculate-dominance
  ask pigs
  [
    set pig-dominant? false
  ]

  let n round ((pig-dominance-percentage / 100) * count pigs)
  let d max-n-of n pigs[pig-happiness]
  ask d
  [
    set pig-dominant? true
  ]
end

to calculate-happiness
  ask pigs
  [
    ifelse(time-cumulative-minutes > 0)
    [
      set pig-happiness max list 0 ((pig-time-to-rest - resting-time-threshold) / time-cumulative-minutes)
      ifelse(temperature-threshold > 0)
      [
        set pig-happiness pig-happiness * max list 0 (climate-temperature / temperature-threshold)
      ]
      [
        set pig-happiness 0
      ]
    ]
    [
      set pig-happiness 0
    ]
  ]
end

to feed
  ;; walking to feeder; i.e. eating line
  ask patches
  [
    set patch-eating-line-occupied? false
  ]

  ask pigs
  [
    set pig-to-eat? true
    set pig-eating? false
    set pig-reach-feeder? false
    set pig-time-to-walk-to-feeder 0
    set pig-time-to-eat 0
    set pig-playing? false
    set pig-to-lie-down? false
    set pig-lying-down? false
  ]
end

to walk-to-feeder
  ask pigs with[not pig-reach-feeder?]
  [
    repeat animation-frame-rate
    [
      if([patch-eating-line?]of patch-here = false and ([pycor]of patch-here > min([pycor]of patches with[patch-eating-line?])))
      [
        set heading ifelse-value(any? patches with[patch-eating-line? and not patch-eating-line-occupied?])
        [
          min list 215 max list 145
          towards one-of patches with[patch-eating-line? and not patch-eating-line-occupied?]
        ]
        [
          min list 215 max list 145
          towards one-of patches with[patch-eating-line?]
        ]
      ]

      ifelse([patch-eating-line?]of patch-here = true)
      [
        ask patch-here
        [
          set patch-eating-line-occupied? true
        ]
        set pig-speed 0
        set pig-reach-feeder? true
        set pig-eating? true
        set pig-time-to-rest max list 0 (pig-time-to-rest - pig-time-to-walk-to-feeder)
        set pig-time-to-walk-to-feeder 0
      ]
      [
        set pig-time-to-walk-to-feeder pig-time-to-walk-to-feeder + (1 / animation-frame-rate)
      ]

      fd pig-speed / animation-frame-rate
    ]
    display
  ]
end

to eat
  ask pigs with[pig-eating?]
  [
     set pig-speed 0

     ifelse (pig-time-to-eat < pig-eating-duration)
     [
       set pig-time-to-eat pig-time-to-eat + 1
       repeat animation-frame-rate
       [
         set heading 175 + random-float 10
         fd pig-speed / animation-frame-rate
       ]
       display
     ]
     [
       set pig-time-to-rest max list 0 (pig-time-to-rest - pig-time-to-eat)
       set pig-eating? false
       set pig-to-eat? false
       set pig-time-to-eat 0
       set heading 180 - heading ;;turn-away after eating
       avoid-forbidden-zone
     ]
  ]


end

to play-or-lie-down
  ask pigs with[not pig-to-eat? and not pig-lying-down?]
  [
     set pig-playing? (random-float 1 < pig-happiness)
     set pig-to-lie-down? not(pig-playing?)
  ]

  play
  lie-down
end

to avoid-forbidden-zone
  let zone-forbidden-neighbors red-zones in-radius sqrt 2
  let n count zone-forbidden-neighbors

  if(any? zone-forbidden-neighbors)
  [
    ifelse(n < 2)
    [
        let h [heading]of one-of zone-forbidden-neighbors

        if(h = 0 or h = 180)
        [
          set heading 180 - heading;;turn-away
        ]

        if(h = 90 or h = 270)
        [
          set heading (- heading);;turn-away
        ]
    ]
    [
      set heading heading + 180 ;;turn away
    ]
  ]
end


to play
  ask pigs with[pig-playing?]
  [
    ;; play

    repeat animation-frame-rate
    [
      flock
      avoid-forbidden-zone

      fd (pig-speed / animation-frame-rate)

      set pig-time-to-rest max list 0 (pig-time-to-rest - (1 / animation-frame-rate))
    ]
    display
    avoid-forbidden-zone
  ]
end

to lie-down-not-dominant-pigs
  ask pigs with[pig-to-lie-down? and not pig-lying-down? and not pig-dominant?]
  [
    let g pig-dominant?
    let p one-of patches with[(patch-strawed? and not patch-strawed-occupied?) or (patch-slatted? and not patch-slatted-occupied?)]
    let x [pxcor]of p
    let y [pycor]of p
    let d1 distancexy x y
    let t1 d1 * pig-speed ;;time to find place to lie down

    set pig-time-to-rest max list 0 (pig-time-to-rest - t1)

    move-to p

    ask p
    [
      if(patch-strawed?)
      [
        set patch-strawed-occupied? true
      ]
      if(patch-slatted?)
      [
        set patch-slatted-occupied? true
      ]
    ]

    if(climate-temperature <= temperature-threshold)
    [
      ifelse(any? other pigs with[pig-to-lie-down? and not pig-has-lying-down-mate?])
      [
        let m min-one-of other pigs with[pig-to-lie-down? and not pig-has-lying-down-mate?] [distance myself]
        let d2 distance m

        let t2 d2 * pig-speed ;;time to find place to lie down
        set pig-time-to-rest max list 0 (pig-time-to-rest - t2)

        ask patch-here
        [
          if(patch-strawed?)
          [
            set patch-strawed-occupied? false
          ]

          if(patch-slatted?)
          [
            set patch-slatted-occupied? false
          ]
        ]

        move-to m

        set pig-has-lying-down-mate? true

        ask m
        [
          set pig-has-lying-down-mate? true
        ]

        set pig-to-lie-down? false
        set pig-lying-down? true
        set pig-time-to-lie-down 0

      ]
      [
        set heading random-float 360
        set pig-to-lie-down? false
        set pig-lying-down? true
        set pig-time-to-lie-down 0
      ]
    ]
  ]
end

to lie-down-dominant-pigs
  ask pigs with[pig-to-lie-down? and not pig-lying-down? and pig-dominant?]
  [
    let g pig-dominant?
    let p one-of patches with[patch-strawed? and not patch-strawed-occupied?]
    let x [pxcor]of p
    let y [pycor]of p
    let d1 distancexy x y
    let t1 d1 * pig-speed ;;time to find place to lie down

    set pig-time-to-rest max list 0 (pig-time-to-rest - t1)

    move-to p

    ask p
    [
      set patch-strawed-occupied? true
    ]

    if(climate-temperature <= temperature-threshold)
    [
      ifelse(any? other pigs with[pig-to-lie-down? and not pig-has-lying-down-mate?])
      [
        let m min-one-of other pigs with[pig-to-lie-down? and not pig-has-lying-down-mate?] [distance myself]
        let d2 distance m

        let t2 d2 * pig-speed ;;time to find place to lie down
        set pig-time-to-rest max list 0 (pig-time-to-rest - t2)

        ask patch-here
        [
          set patch-strawed-occupied? false
        ]

        move-to m

        set pig-has-lying-down-mate? true

        ask m
        [
          set pig-has-lying-down-mate? true
        ]

        set pig-to-lie-down? false
        set pig-lying-down? true
        set pig-time-to-lie-down 0

      ]
      [
        set heading random-float 360
        set pig-to-lie-down? false
        set pig-lying-down? true
        set pig-time-to-lie-down 0
      ]
    ]


  ]
end

to lie-down
  ask patches
  [
    set patch-strawed-occupied? false
    set patch-slatted-occupied? false
  ]

  lie-down-dominant-pigs
  lie-down-not-dominant-pigs


  ask pigs with[pig-lying-down?]
  [
    let annoying-index 0
    if any? (pigs with[pig-playing?])
    [
      set annoying-index
        (count other pigs with[pig-playing?] in-radius sqrt 2) /
        (count other pigs with[pig-playing?] ); meters
    ]

    repeat animation-frame-rate
    [
      set pig-speed 0
      fd pig-speed / animation-frame-rate
      set pig-time-to-rest max list 0 (pig-time-to-rest + (1 / animation-frame-rate) - (annoying-index / animation-frame-rate))
    ]
    display
    ifelse(pig-time-to-lie-down < pig-resting-duration)
    [
      set pig-time-to-lie-down pig-time-to-lie-down + 1
    ]
    [
      set pig-lying-down? false
      set pig-has-lying-down-mate? false
    ]
  ]
end

to sleep-not-dominant-pigs
  ask pigs with[pig-to-sleep? and not pig-sleeping? and not pig-dominant?]
  [
    let g pig-dominant?
    let p one-of patches with[(patch-strawed? and not patch-strawed-occupied?) or (patch-slatted? and not patch-slatted-occupied?)]
    let x [pxcor]of p
    let y [pycor]of p
    let d1 distancexy x y
    let t1 d1 * pig-speed ;;time to find place to lie down

    set pig-time-to-rest max list 0 (pig-time-to-rest - t1)

    move-to p

    ask p
    [
      if(patch-strawed?)
      [
        set patch-strawed-occupied? true
      ]
      if(patch-slatted?)
      [
        set patch-slatted-occupied? true
      ]
    ]

    if(climate-temperature <= temperature-threshold)
    [
      ifelse(any? other pigs with[pig-to-sleep? and not pig-has-sleeping-mate?])
      [
        let m min-one-of other pigs with[pig-to-sleep? and not pig-has-sleeping-mate?] [distance myself]
        let d2 distance m

        let t2 d2 * pig-speed ;;time to find place to lie down
        set pig-time-to-rest max list 0 (pig-time-to-rest - t2)

        ask patch-here
        [
          if(patch-strawed?)
          [
            set patch-strawed-occupied? false
          ]

          if(patch-slatted?)
          [
            set patch-slatted-occupied? false
          ]
        ]

        move-to m

        set pig-has-sleeping-mate? true

        ask m
        [
          set pig-has-sleeping-mate? true
        ]

        set pig-to-sleep? false
        set pig-sleeping? true
        set pig-time-to-sleep 0

      ]
      [
        set heading random-float 360
        set pig-to-sleep? false
        set pig-sleeping? true
        set pig-time-to-sleep 0
      ]
    ]

  ]
end

to sleep-dominant-pigs
  ask pigs with[pig-to-sleep? and not pig-sleeping? and pig-dominant?]
  [
    let g pig-dominant?
    let p one-of patches with[patch-strawed? and not patch-strawed-occupied?]
    let x [pxcor]of p
    let y [pycor]of p
    let d1 distancexy x y
    let t1 d1 * pig-speed ;;time to find place to lie down

    set pig-time-to-rest max list 0 (pig-time-to-rest - t1)

    move-to p

    ask p
    [
      set patch-strawed-occupied? true
    ]

    if(climate-temperature <= temperature-threshold)
    [
      ifelse(any? other pigs with[pig-to-sleep? and not pig-has-sleeping-mate?])
      [
        let m min-one-of other pigs with[pig-to-sleep? and not pig-has-sleeping-mate?] [distance myself]
        let d2 distance m

        let t2 d2 * pig-speed ;;time to find place to lie down
        set pig-time-to-rest max list 0 (pig-time-to-rest - t2)

        ask patch-here
        [
          set patch-strawed-occupied? false
        ]

        move-to m

        set pig-has-sleeping-mate? true

        ask m
        [
          set pig-has-sleeping-mate? true
        ]

        set pig-to-sleep? false
        set pig-sleeping? true
        set pig-time-to-sleep 0

      ]
      [
        set heading random-float 360
        set pig-to-sleep? false
        set pig-sleeping? true
        set pig-time-to-sleep 0
      ]
    ]

  ]
end

to sleep
  ask patches
  [
    set patch-strawed-occupied? false
    set patch-slatted-occupied? false
  ]

  sleep-dominant-pigs
  sleep-not-dominant-pigs


  ask pigs with[pig-sleeping?]
  [
    repeat animation-frame-rate
    [
      set pig-speed 0
      fd pig-speed / animation-frame-rate
      set pig-time-to-rest max list 0 (pig-time-to-rest + (1 / animation-frame-rate))
    ]
    display
    ifelse(pig-time-to-sleep < pig-resting-duration)
    [
      set pig-time-to-sleep pig-time-to-sleep + 1
    ]
    [
      ifelse(time-section = "night time")
      [
        set pig-time-to-sleep pig-time-to-sleep + 1
      ]
      [
        set pig-sleeping? false
        set pig-has-sleeping-mate? false
      ]
    ]
  ]
end

to do-during-nighttime
  ;; reinitialise
  ask pigs
  [
    set pig-speed 0
    set pig-reach-feeder? true
    set pig-eating? false
    set pig-to-eat? false
    set pig-playing? false
    set pig-lying-down? false
    set pig-to-lie-down? false
    set pig-has-lying-down-mate? false

    set pig-to-sleep? true
  ]

  sleep
end

to urinate
  ask pigs
  [
    set pig-urinating? member? (list time-hour time-minute) pig-schedule-urinate
  ]

  ask pigs with[pig-urinating?]
  [
     ifelse any? other pigs-here
     [
       let distuberd-pigs other pigs-here
       ask distuberd-pigs
       [
         ifelse random-float 1 < .5
         [
           rt random-float 45
         ]
         [
           lt random-float 45
         ]
         repeat animation-frame-rate
         [

           fd pig-speed / animation-frame-rate
           avoid-forbidden-zone
           set pig-time-to-rest max list 0 (pig-time-to-rest - (1 / animation-frame-rate))
         ]
         display
       ]
     ]
     [
       repeat animation-frame-rate
       [
         set pig-time-to-rest max list 0 (pig-time-to-rest - (1 / animation-frame-rate))
       ]
       display
     ]
  ]
end

to do-during-daytime
  ask pigs
  [
    set pig-sleeping? false
    set pig-to-sleep? false
    set pig-has-sleeping-mate? false
  ]

  let t list time-hour time-minute
  set schedule-feeding-time? member? t schedule-feeding

  if(schedule-feeding-time?)
  [
    feed
  ]

  walk-to-feeder
  eat
  play-or-lie-down
end

to go
  calculate-pig-resting-on-slatted-floor
  define-time-section
  generate-climate
  calculate-speed
  urinate


  ifelse(time-section = "day time")
  [
    do-during-daytime
  ]
  [
    do-during-nighttime
  ]
  calculate-dominance
  calculate-happiness
  color-pigs
  plot-resting-pigs-time-on-slatted-floor
  plot-pig-cumulative-resting-time
  plot-pig-emotion
  update-time

  tick ;;per minute
end
;;

;; pig playing rule following flocking rule

to flock
  find-playmates
  if(any? pig-playmates)
  [
    find-nearest-playmate
    ifelse(distance pig-nearest-playmate < 1) ;;meters
    [
      separate
    ]
    [
      align
      cohere
    ]
  ]
end

to find-playmates
  let group pig-dominant?
  set pig-playmates other pigs with[pig-playing? and pig-dominant? = group] in-radius sqrt 8 ;;meters
end

to find-nearest-playmate
  set pig-nearest-playmate min-one-of pig-playmates [distance myself]
end

to separate
  turn-away ([heading] of pig-nearest-playmate) 1.5 ;;degrees
end

to align
  turn-towards average-playmate-heading 5 ;;degrees
end

to-report average-playmate-heading
  let x-component sum [dx] of pig-playmates
  let y-component sum [dy] of pig-playmates
  ifelse(x-component = 0 and y-component = 0)
  [
    report heading
  ]
  [
    report atan x-component y-component
  ]
end

to cohere
  turn-towards average-heading-towards-playmates 3 ;;degrees
end

to-report average-heading-towards-playmates
  let x-component mean [sin (towards myself + 180)] of pig-playmates
  let y-component mean [cos (towards myself + 180)] of pig-playmates
  ifelse(x-component = 0 and y-component = 0)
  [
    report heading
  ]
  [
    report atan x-component y-component
  ]
end

to turn-towards [new-heading max-turn]
  turn-at-most (subtract-headings new-heading heading) max-turn
end

to turn-away [new-heading max-turn]
  turn-at-most (subtract-headings heading new-heading) max-turn
end

to turn-at-most [turn max-turn]
  ifelse(abs turn > max-turn)
  [
    ifelse(turn > 0)
    [
      rt max-turn
    ]
    [
      lt max-turn
    ]
  ]
  [
    rt turn
  ]
end

;;
to initialise-plot-pig-cumulative-resting-time
  let plot-pen-color n-values pig-population [5 + ? * 10]
  let plot-pen-list
  [
    "pig 1" "pig 2" "pig 3" "pig 4" "pig 5" "pig 6" "pig 7" "pig 8" "pig 9" "pig 10"
    "pig 11" "pig 12" "pig 13" "pig 14" "pig 15" "pig 16" "pig 17" "pig 18" "pig 19" "pig 20"
  ]

  let plot-pen sublist plot-pen-list 0 pig-population
  set-current-plot "pig cumulative resting time"
  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    set-plot-pen-color white
  ]

  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    set-plot-pen-color item i plot-pen-color
  ]

end

to plot-pig-cumulative-resting-time
  let plot-pen-color n-values pig-population[5 + ? * 10]
  let plot-pen-list
  [
    "pig 1" "pig 2" "pig 3" "pig 4" "pig 5" "pig 6" "pig 7" "pig 8" "pig 9" "pig 10"
    "pig 11" "pig 12" "pig 13" "pig 14" "pig 15" "pig 16" "pig 17" "pig 18" "pig 19" "pig 20"
  ]
  let plot-pen sublist plot-pen-list 0 pig-population
  let p sort-on [who] pigs


  set-current-plot "pig cumulative resting time"

  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    plot [pig-time-to-rest]of item i p
  ]

end

to initialise-plot-resting-pigs-time-on-slatted-floor
  let plot-pen-color n-values pig-population [5 + ? * 10]
  let plot-pen-list
  [
    "pig 1" "pig 2" "pig 3" "pig 4" "pig 5" "pig 6" "pig 7" "pig 8" "pig 9" "pig 10"
    "pig 11" "pig 12" "pig 13" "pig 14" "pig 15" "pig 16" "pig 17" "pig 18" "pig 19" "pig 20"
  ]

  let plot-pen sublist plot-pen-list 0 pig-population
  set-current-plot "resting pig cumulative time on slatted floor"
  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    set-plot-pen-color white
  ]

  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    set-plot-pen-color item i plot-pen-color
  ]

end

to plot-resting-pigs-time-on-slatted-floor
  let plot-pen-color n-values pig-population[5 + ? * 10]
  let plot-pen-list
  [
    "pig 1" "pig 2" "pig 3" "pig 4" "pig 5" "pig 6" "pig 7" "pig 8" "pig 9" "pig 10"
    "pig 11" "pig 12" "pig 13" "pig 14" "pig 15" "pig 16" "pig 17" "pig 18" "pig 19" "pig 20"
  ]
  let plot-pen sublist plot-pen-list 0 pig-population
  let p sort-on [who] pigs


  set-current-plot "resting pig cumulative time on slatted floor"

  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    plot [pig-cumulative-resting-time-on-slatted-floor]of item i p
  ]

end

;;
to initialise-plot-pig-emotion
  let plot-pen-color n-values pig-population [5 + ? * 10]
  let plot-pen-list
  [
    "pig 1" "pig 2" "pig 3" "pig 4" "pig 5" "pig 6" "pig 7" "pig 8" "pig 9" "pig 10"
    "pig 11" "pig 12" "pig 13" "pig 14" "pig 15" "pig 16" "pig 17" "pig 18" "pig 19" "pig 20"
  ]

  let plot-pen sublist plot-pen-list 0 pig-population
  set-current-plot "pig emotion"
  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    set-plot-pen-color white
  ]

  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    set-plot-pen-color item i plot-pen-color
  ]
end

to plot-pig-emotion
  let plot-pen-color n-values pig-population[5 + ? * 10]
  let plot-pen-list
  [
    "pig 1" "pig 2" "pig 3" "pig 4" "pig 5" "pig 6" "pig 7" "pig 8" "pig 9" "pig 10"
    "pig 11" "pig 12" "pig 13" "pig 14" "pig 15" "pig 16" "pig 17" "pig 18" "pig 19" "pig 20"
  ]
  let plot-pen sublist plot-pen-list 0 pig-population
  let p sort-on [who] pigs


  set-current-plot "pig emotion"

  foreach plot-pen
  [
    let i position ? plot-pen
    set-current-plot-pen ?
    plot [pig-happiness]of item i p
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
286
10
730
268
-1
-1
20.7
1
10
1
1
1
0
0
0
1
0
20
0
10
0
0
1
ticks
30.0

BUTTON
12
10
75
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
83
10
146
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
448
272
498
317
hour
time-hour
0
1
11

MONITOR
394
272
444
317
day
time-day
0
1
11

MONITOR
339
272
391
317
month
time-month
0
1
11

MONITOR
286
272
336
317
year
time-year
0
1
11

PLOT
285
369
445
519
temperature
minute
deg C
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"temp" 1.0 0 -2674135 true "" "plot climate-temperature"

SLIDER
12
50
184
83
pig-population
pig-population
1
20
20
1
1
pigs
HORIZONTAL

MONITOR
501
272
558
317
minute
time-minute
0
1
11

SLIDER
12
89
239
122
pig-max-initial-resting-time
pig-max-initial-resting-time
0
1000
300
1
1
minutes
HORIZONTAL

SLIDER
12
128
216
161
pig-feeding-interval
pig-feeding-interval
0
500
90
10
1
minutes
HORIZONTAL

SLIDER
12
171
260
204
pig-eating-duration
pig-eating-duration
1
10
6
1
1
minutes per cycle
HORIZONTAL

SLIDER
11
264
183
297
pig-base-speed
pig-base-speed
5
25
20
1
1
mph
HORIZONTAL

PLOT
736
170
917
326
pig emotion
minutes
happiness
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"pig 1" 1.0 0 -1 true "" ""
"pig 2" 1.0 0 -1 true "" ""
"pig 3" 1.0 0 -1 true "" ""
"pig 4" 1.0 0 -1 true "" ""
"pig 5" 1.0 0 -1 true "" ""
"pig 6" 1.0 0 -1 true "" ""
"pig 7" 1.0 0 -1 true "" ""
"pig 8" 1.0 0 -1 true "" ""
"pig 9" 1.0 0 -1 true "" ""
"pig 10" 1.0 0 -1 true "" ""
"pig 11" 1.0 0 -1 true "" ""
"pig 12" 1.0 0 -1 true "" ""
"pig 13" 1.0 0 -1 true "" ""
"pig 14" 1.0 0 -1 true "" ""
"pig 15" 1.0 0 -1 true "" ""
"pig 16" 1.0 0 -1 true "" ""
"pig 17" 1.0 0 -1 true "" ""
"pig 18" 1.0 0 -1 true "" ""
"pig 19" 1.0 0 -1 true "" ""
"pig 20" 1.0 0 -1 true "" ""

SLIDER
11
470
282
503
animation-frame-rate
animation-frame-rate
1
500
500
1
1
frames per minute
HORIZONTAL

SLIDER
10
303
223
336
pig-dominance-percentage
pig-dominance-percentage
0
100
50
5
1
%
HORIZONTAL

MONITOR
562
273
630
318
time
time-section
0
1
11

SLIDER
10
342
268
375
pig-urinating-frequency
pig-urinating-frequency
0
10
8
1
1
times per day
HORIZONTAL

MONITOR
650
274
729
319
temp(deg C)
climate-temperature
1
1
11

PLOT
1127
11
1318
167
pig eating
minutes
%
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"eating" 1.0 0 -955883 true "" "plot 100 * (count pigs with[pig-eating?]) / (count pigs)"

PLOT
735
12
918
168
pig playing
minutes
%
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"playing" 1.0 0 -13345367 true "" "plot 100 * (count pigs with[pig-playing?]) / (count pigs)"

PLOT
922
11
1123
167
pig urinating
minutes
%
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"urinating" 1.0 0 -13840069 true "" "plot 100 * (count pigs with[pig-urinating?]) / (count pigs)"

SLIDER
11
384
222
417
temperature-threshold
temperature-threshold
1
20
19
1
1
deg C
HORIZONTAL

SLIDER
11
427
275
460
resting-time-threshold
resting-time-threshold
100
500
300
100
1
minutes per day
HORIZONTAL

SWITCH
283
330
386
363
heater?
heater?
1
1
-1000

SLIDER
389
330
570
363
heating-temperature
heating-temperature
0
100
20
.1
1
deg C
HORIZONTAL

PLOT
922
171
1123
326
pig lying down
minutes
%
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"lying down" 1.0 0 -10899396 true "" "plot 100 * (count pigs with[pig-lying-down?]) / count pigs"

SLIDER
11
213
269
246
pig-resting-duration
pig-resting-duration
0
500
100
50
1
minutes per cycle
HORIZONTAL

PLOT
1128
171
1320
325
pig sleeping
minutes
%
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"sleeping" 1.0 0 -16777216 true "" "plot 100 * (count pigs with[pig-sleeping?] / count pigs)"

PLOT
1021
329
1319
504
pig cumulative resting time
minutes
minutes
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"pig 1" 1.0 0 -1 true "" ""
"pig 2" 1.0 0 -1 true "" ""
"pig 3" 1.0 0 -1 true "" ""
"pig 4" 1.0 0 -1 true "" ""
"pig 5" 1.0 0 -1 true "" ""
"pig 6" 1.0 0 -1 true "" ""
"pig 7" 1.0 0 -1 true "" ""
"pig 8" 1.0 0 -1 true "" ""
"pig 9" 1.0 0 -1 true "" ""
"pig 10" 1.0 0 -1 true "" ""
"pig 11" 1.0 0 -1 true "" ""
"pig 12" 1.0 0 -1 true "" ""
"pig 13" 1.0 0 -1 true "" ""
"pig 14" 1.0 0 -1 true "" ""
"pig 15" 1.0 0 -1 true "" ""
"pig 16" 1.0 0 -1 true "" ""
"pig 17" 1.0 0 -1 true "" ""
"pig 18" 1.0 0 -1 true "" ""
"pig 19" 1.0 0 -1 true "" ""
"pig 20" 1.0 0 -1 true "" ""

PLOT
737
331
1016
505
resting pig cumulative time on slatted floor
minutes
minutes
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"pig 1" 1.0 0 -1 true "" ""
"pig 2" 1.0 0 -1 true "" ""
"pig 3" 1.0 0 -1 true "" ""
"pig 4" 1.0 0 -1 true "" ""
"pig 5" 1.0 0 -1 true "" ""
"pig 6" 1.0 0 -1 true "" ""
"pig 7" 1.0 0 -1 true "" ""
"pig 8" 1.0 0 -1 true "" ""
"pig 9" 1.0 0 -1 true "" ""
"pig 10" 1.0 0 -1 true "" ""
"pig 11" 1.0 0 -1 true "" ""
"pig 12" 1.0 0 -1 true "" ""
"pig 13" 1.0 0 -1 true "" ""
"pig 14" 1.0 0 -1 true "" ""
"pig 15" 1.0 0 -1 true "" ""
"pig 16" 1.0 0 -1 true "" ""
"pig 17" 1.0 0 -1 true "" ""
"pig 18" 1.0 0 -1 true "" ""
"pig 19" 1.0 0 -1 true "" ""
"pig 20" 1.0 0 -1 true "" ""

@#$#@#$#@
## WHAT IS IT?

LIFE OF PI(gs) model portrays the life of pigs in a pig-house in every minute, aiming at estimating the pigs' welfare.

## DEVELOPMENT STATUS

Prototype


## CREDITS AND REFERENCES

Desi Suyamto(1) & Aritta Suwarno(2)

(1) Forest2020 Collaborative Programme, Faculty of Forestry, Bogor Agricultural University, Indonesia

(2) Department of Environmental Sciences, Wageningen University & Research, the Netherlands
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

pig
true
0
Circle -7500403 true true 96 76 108
Circle -7500403 true true 72 104 156
Polygon -7500403 true true 221 149 195 101 106 99 80 148
Circle -7500403 true true 105 25 90
Circle -7500403 true true 129 9 42
Polygon -7500403 true true 210 30 195 60 180 45
Polygon -7500403 true true 90 30 105 60 120 45
Circle -7500403 true true 72 119 156
Polygon -7500403 true true 165 255 180 285 165 300 150 300 135 285 135 270 150 270 150 285 165 285 165 270 165 255

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
true
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
