;; ======================================== INTRODUCITON =============================================
;; ========================== DEFINITION OF PARAMETERS AND STATE VARIABLES ===========================
   ;; Original model from Martin et al 2012/13
   ;; Adjustments have been made to fit the model to southern elephant seal data  for female seals only.
   ;; Values selected as per DEBtool unless otherwise specified, the definition of parameters for the
   ;; individuals the notation follows the DEBtool-notation as far as possible. Deviation: rates are
   ;; indicated with "_rate"

   ;; The model description and sensitivity analysis of the baseline model is published in PLoS ONE,
   ;; Goedegebuure et al. (2018, DOI 10.1371/journal.pone.0194950) Modelling southern elephant seals
   ;; Mirounga leonina using an individual-based model coupled with a dynamic energy budget.
   ;; For which the actual model is available from http://modelingcommons.org/browse/one_model/5348

   ;; The model description and modifications from the baseline model that allow for the running
   ;; of the scenarios are to be published, but are as yet only available in Chapter 4 of my thesis:
   ;; Using the DEB-IBM to assess the drivers of the decreasing population of elephant seals at
   ;; Macquarie Island (submitted to external examiners May 3, 2018).

   ;; General model information and handbook are found at https://ccl.northwestern.edu/netlogo/docs/
;; ========================================== MODEL START ==============================================
;; Declarations
;extensions [            ;; extensions do not work in the online version of NetLogo - thus blocked out
;  profiler              ;; for profiling of the model, adjusted on the interface
;]
globals[                 ;; set global parameters for the model
  my-seed
  model_just_started
  max-ticks
  counting
  ;; - - - - - - - - - - - - - - - time management
  ;; ---------------- to keep track of the day, month, and year in the model
  day-of-year
  day-of-month
  month-of-year
  year
  pups-born     ;; for overall output

  ;; ------------------- for competition calculations
  population          ; P
  competition         ; Delta P
  carrying_capacity   ; K

  ;; - - - - - - - - - - - - - - - stages for IBM
  foetus        ;;0           ;; U_H < U_H^b                 ; not yet born
  pup           ;;1           ;; U_H > U_H^b & U_H < U_H^x   ; born and weaning
  juvenile      ;;2           ;; U_H > U_H^x & U_H < U_H^p   ; past weaning
  mature        ;;3           ;; U_H > U_H^p                 ; mature

  ;; - - - - - - - - - - - - - - - status for IBM
  mother-dependent  ;; 0 - foetus or pup
  fasting           ;; 1 - resting/moulting/lactating
  foraging          ;; 2

  ;; -------------- pup mortality
  pup-mortality

  ;============================================================================================= HYPOTHESIS TESTING
  ;---------- K for Density dependence
  k1-k2                           ; the difference between K1 and K2
  rate-of-decline                 ; the number of individuals that the population declines by per year
  t-for-k2                        ; new time to set next level of K
  change-time                     ; for periodic change in K
  capacity-change                 ; for changing K by some fraction
  change-factor                   ; 0/1 for positive or negative change in capacity

  ; -------------- yearling mortality
  percent-pup-affected            ; percentage of pups affected with energy change
  SA_change

  ; --------------- fecundity
  percent-mum-affected
  fec_change
  fecundity_changed_now?

  ;---------------- climate variation
  climate-change
  clim-var
  ;=============================================================================================
]
;; --------------------------------------patches-own-------------------------------------------------------------
;patches-own[
;    ;; for now this isn't actually used... competition has been implemented through K1 - K2
;  X        ; # / cm^2, prey density
;  d_X      ; change of prey density in time
;]
;; ---------------------------------------------------------------------------------------------------
turtles-own[
;  is_male?    ; 1 / 0 -- implement when males are included (for now running model with only females)
  age         ; age of seals (in days)
  stage       ; foetus, pup, juvenile or mature
  status      ; fasting, foraging or mother-dependent

  ;; - - - - - - - - - - - - - - - STATE VARIABLES - - - - - - - - - - - - - - - - - - - - - - - - - -
  L           ; cm, structural length
  Lmax        ; cm, maximum structural length
  dL          ; change of structural length in time

  U_H         ; t L^2, scaled maturity
  dU_H        ; change of scaled maturity in time

  U_E         ; t L^2, scaled reserves
  dU_E        ; change of scaled reserves in time
  e_scaled    ; - , scaled reserves per unit of structure
  l_scaled    ; - , scaled structural length

  U_R         ; t L^2, scaled energy in reproduction buffer (not standard DEB)
  dU_R        ; change of energy in reproduction buffer (reproduction rate)

  ;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  iv          ; to include individual variability for DEB-IBM parameters on a normal distribution
                 ; with a mean on the input paramater and a coefficent of variation equal to the cv

  g           ; - , energy investment ratio

  ;; - - - - - - - - - - - - - - - FLUXES (used by several submodels) - - - - - - - - - - - - - - - -
  S_A         ; assimilation flux
  S_C         ; mobilisation flux

  ;; - - - - - - - - - - - - - - - STANDARD DEB PARAMETERS - - - - - - - - - - - - - - - - - - - - -
  U_H^b       ; t L^2, scaled maturity at birth
  U_H^x       ; t L^2, scaled maturity at weaning
  U_H^p       ; t L^2, scaled maturity at puberty

  ;; - - - - - - - - - - - - - - - PREY DYNAMICS (only relevant if prey-dynamics not constant)
  f           ; - , scaled functional response (food availability) (value between 0 and 1; where 1 = max feeding)
  ;; for now the following isn't actually used... competition has been implemented below
;  K           ; # / cm^2, (half) saturation coefficient
;  J_XAm_rate  ; # / (cm^2 t), surface-area-specific maximum ingestion rate

  ;; - - - - - - - - - - - - - - - AGEING -- - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  r_rate          ; growth rate

  q_accel         ; - , ageing acceleration
  dq_accel        ; change of ageing acceleration in time
  h_rate          ; - , hazard rate
  dh_rate         ; change of hazard rate in time

  ;; - - - - - - - - REPRODUCTION / MOULTING
  my_ID                 ; collect ID of self
  offspring_ID          ; collect ID of offspring (-1 if none)
  mother_ID             ; collect ID of mother (-1 if none)
;  partner_ID           ; collect ID of potential partner for mating (-1 if none) ---- as no males yet
  cumulative_UE          ; initial cumulative energy required for allocation to pup
  cum_UE                 ; final cumulative energy required for allocation to pup ---- proportional to mother's size

  is_pregnant?          ; 1 / 0
  resting?              ; 1 / 0
  moulting?             ; 1 / 0
  haul-out?             ; 1 / 0
  breeding?             ; 1 / 0
  impregnated?          ; 1 / 0

  had-pup?              ; 1 / 0
  lactating?            ; 1 / 0

  tried-mating          ; count for trying to get pregnant
  time-since-mating     ; days for calculation to start conception
  time-since-suckling   ; days for calculation to weaning
  days-moulting         ; days for calculation of getting back to foraging
  days-foraging         ; days for calculation for resting time (females and young ones)
  days-resting          ; for resting calculation (females and young ones)

  puppy                 ; for plotting
  total-puppies         ; for death output

  die-now               ; 0 / 1 - for death in this timestep
  died-of-?    ; 0 / 1
  final-age             ; recording before death
  final-pups            ; recording before death
  breeding-age          ; recording before death
  reprod-period         ; recording before death
  fecundity             ; recording before death

  weaning_age           ; final age at weaning
  days-dead             ; for collecting age and other final data

  xx                    ; for testing for pup-affected//mum-affected
  pup-affected          ; 0 / 1 -- for affected with less energy intake
  mum-affected

  track-life            ; for tracking life span of turtles (life table)

]

;; ========================== SETUP PROCEDURE: SETTING INITIAL CONDITIONS ============================
to setup
  ca
  set my-seed new-seed                           ;; to make the model reproducible but keep the seed used random
;  set my-seed 1548502745                         ;; manually set seed
  output-write word "Generated seed: " my-seed   ;; print it in the output box on the interface
  random-seed my-seed
  set model_just_started true                    ;; for profiling

  reset-ticks
  set max-ticks max-years * 360
  set carrying_capacity max-pop-1

  ;------------------------------------ scenarios
  set k1-k2 max-pop-1 - max-pop-2                             ; the difference between k1 and k2
  set rate-of-decline k1-k2 / time-for-decline                ; the number of individuals that the population declines by per year
  set percent-pup-affected 0
  set fecundity_changed_now? 0

  set pup-mortality 0                                           ;; to set for start (change in do-time-management)
  if (pup-mort = "minimum") [ set pup-mortality min-pup-mort ]
  if (pup-mort = "mean")    [ set pup-mortality mean-pup-mort ] ;; mean based on McMahon et al 2000, 2003
  if (pup-mort = "maximum") [ set pup-mortality max-pup-mort ]

  ;; - - - - - - - - - - - - - - - time management
  set day-of-year 1                ;; start on Jan 1
  set day-of-month 1
  set month-of-year 1
  set year 0

  ;; - - - - - - - - stages for IBM
  set foetus 0       ;; U_H < U_H^b
  set pup 1          ;; U_H > U_H^b & U_H < U_H^x
  set juvenile 2     ;; U_H > U_H^x & U_H < U_H^p
  set mature 3       ;; U_H > U_H^p

  ;; - - - - - - - - status for IBM
  set mother-dependent 0  ;; foetus/pup
  set fasting 1
  set foraging 2

  ;; - - - - - - - - - - - - - - - create seal population as on interface
  crt start_population
  ask turtles [ individual-variability ] ;; setting the initial settings of seals

  set climate-change 0
  set clim-var 0

;  profiler:reset     ;; clear the profile data
end
;; ========================== SUBMODELS ==============================================================
;; -------------------------- INITIAL SETTINGS--------------------------------------------------------
to individual-variability
  ;; following Martin et al's model -- to apply individual variability - cv set to 0.02
  ;; iv applied to DEB and IBM parameters.
  set iv e ^ (random-normal 0 cv)

  ;; calculate competition based on the current number of individuals
  set competition (1 - f_scaled) * ( 1 - (start_population / (2 * carrying_capacity - start_population)))
  ;; to set effective food availability (f) from initial food availability (f_scaled)
  set f (f_scaled + competition) * iv
  if f > 1 [set f 1]

  set g g_init * iv

  ;; set scaled maturities
  set U_H^b (E_H^b_init / p_am) ;* iv        ;; scaled maturity at birth (cm^2 d)
  set U_H^x (E_H^x_init / p_am) ;* iv        ;; scaled maturity at weaning (cm^2 d)
  set U_H^p (E_H^p_init / p_am) ;* iv        ;; scaled maturity at puberty (cm^2 d)

  ;; set scaled length
  set Lmax L_w^m * shape_factor  ;; max struct length
  ;; so that L > L_w^x and L < L_w^m (size between weaner and full adult)
  set L ((L_w^x + random (L_w^m - L_w^x)) * shape_factor)

  ;; set initial reserve
  ;; these initial settings are balanced out by the second generation seals
  set l_scaled L / Lmax                     ;; scaled length -- resets in calc-dL
  set U_E ((L ^ 3) * (l_scaled / v_rate))   ;; as e < l means starvation model is implemented and e = v_rate * U_E / L ^ 3
  set U_H U_E / 2.87                        ;; as U_H / U_E = 2.145 @ x, 3.585 @ p. -- not starting with pups - thus excluded from average.


  set e_scaled (v_rate * (U_E / (L ^ 3)))   ;; resets every time step in calc-dU_E
  ;; sanity check
  if U_H < U_H^x  [ set U_H U_H^x ]         ;; make sure every individual to start with has enenough stored energy

  ;; set stages -- the model will only start with juveniles or adults (foetus/pup initiated from mother only)
  ifelse U_H < U_H^p
  [ set stage juvenile    ;; 2
    set age round (23 + random (1059 - 23))
    set h_rate random-float 1.0e-19
    set U_R U_H
  ]
  [ set stage mature      ;; 3
    set age round (1059 + random (5400 - 1059))
    set h_rate random-float 1.0e-09
    set U_R U_H
  ]

  ;; set reproduction and moutling settings -- for now no pregnancy in start of model
  set status foraging
  set my_ID who              ; collect own ID
  set offspring_ID -1        ; collect ID of offspring (-1 when no offspring)
  set mother_ID -1           ; collect ID of mother (-1 when no longer relying on mother)

  ;; calcualtion for the cumulative energy required to produce a healthy offspring that reaches the required size at birth (L_w^b)
  ;; -- based on the modified calculation in comments on the DEB book (p. 38).
  ;; This calculation is adjusted here to suit our species by requiring only 70%
  ;; of the mother's UR levels for the lower limit of energy needs. (Used as the mean in simulations.)

  set cumulative_UE (((L_w^b * shape_factor) ^ 3) * ((f_scaled + g) / v_rate) * (1 + (3 / 4) * ((L_w^b / L_w^m) / f_scaled)))
  set cum_UE cumulative_UE * l_scaled * 0.2

  set impregnated? 0         ; 1 / 0 conception
  set is_pregnant? 0         ; 1 / 0 implantation
  set breeding? 0            ; 1 / 0 start in september
  set had-pup? 0             ; 1 / 0
  set moulting? 0            ; 1 / 0 once a year
  set lactating? 0           ; 1 / 0 after birth
  set resting? 0             ; 1 / 0 after foraging for extensive period
  set haul-out? 0            ; 1 / 0 mid winter haul out for juvenile seals

  set tried-mating 0         ; count for trying to mate
  set time-since-mating 0    ; days for calculation to start conception
  set time-since-suckling 0  ; days for calculation to weaning
  set days-moulting 0        ; days for calculation of getting back to foraging
  set days-foraging 0        ; days for calculation for resting time (females and young ones)
  set days-resting 0         ; for resting calculation (females and young ones)

  set puppy 0                ; for plotting
  set total-puppies 0        ; for death output for individuals

  set die-now 0              ; recording before death
  set died-of-? 0
  set final-age 0            ; recording before death
  set final-pups -1          ; recording before death
  set breeding-age 0         ; recording before death
  set reprod-period 0        ; recording before death
  set fecundity -1           ; recording before death

  set weaning_age 0          ; track age at weaning
  set days-dead 0            ; to track dead individual parameters

  set xx 100                 ; so that no-one starts off with being affected

  ;----------------------------------------- yearling mortality and fecundity scenarios
  set pup-affected 0         ; 0 / 1
  set xx random-float 100    ; set as random value between 0 and 100 (%)
  if xx < percent-pup-affected [ set pup-affected 1 ]

  set mum-affected 0         ; 0 / 1
  set xx random-float 100
  if xx < percent-mum-affected [ set mum-affected 1 ]

  set track-life 0          ; set in time-management

end

;; ========================== GO PROCEDURE: RUNNING THE MODEL ========================================

to go
  tick
  ;; start ticks at beginning of procedure so that the whole thing happens on the correct day
  ;; see Railsback and Grimm (2011) for more detaied information on why this should be first.

;  if ((model_just_started = true) and (use_profiler? = true)) [
;    profiler:start                        ;; start profiling
;    set model_just_started false          ;; reset
;  ]

  ;; for days of month, year etc. also adds day to the age of the turtles
  ;; also includes change for scenario implementations
  do-time-management

  ask turtles with [ die-now = 0 ] [ ;reset change each day
    set dL 0
    set dU_E 0
    set dU_H 0
    set dU_R 0
    set cum_UE cumulative_UE * l_scaled * 0.2
  ]

  if fecundity_changed_now? = 1 [
    ask turtles [
      if mum-affected > 0 [
        if increase? = "yes" [ set cum_UE cum_UE * (1 + fec_change) ]
        if increase? = "no"  [ set cum_UE cum_UE * (1 - fec_change) ]
      ]
    ]
  ]

  set population (count turtles) ;with [stage > 1]) ; not mother dependent...
  ask turtles [
    ifelse population < 1.9 * carrying_capacity
    [ set competition (1 - f_scaled) * ( 1 - (population / (2 * carrying_capacity - population)))
      set f (f_scaled + competition) * iv
      if climate-change = 1 [ set f f * clim-var ]
    ]
    [ set competition (1 - f_scaled) * ( 1 - (population / (carrying_capacity / 10) ) )
      set f (f_scaled + competition) * iv
      if climate-change = 1 [ set f f * clim-var ]
    ]
    if f > 1 [ set f 1 ]
  ]

;  if climate-change = 1 [
;    ask turtles [
;      set f (f_scaled + competition) * iv * clim-var
;      if f > 1 [set f 1]
;    ]
;  ]

  ;; go through daily life of seals - only for independent seals: status = 0 for mother-dependent seals (foetus/pup)
  ;; U_E, U_H, and L calculations only applied to independent seals - for pups this is implemented
  ;; in update-offspring-energy. Individuals calculate the change in their state var based on the current conditions
  ask turtles with [ die-now = 0 and status > 0 ][ check-status ]
  ask turtles with [ die-now = 0 and status > 0 ][ check-breeding ]

  ask turtles with [ die-now = 0 and status > 0 ][ calc-dU_E ]   ; calculate change in scaled reserve
  ask turtles with [die-now = 0 and status > 0]  [ calc-dU_H-R ] ; calc change in scaled maturity and reproductive buffer

  ask turtles with [ die-now = 0 and status > 0 ][ calc-dL ]     ; calculate change in structural length

  ;; aging is applied to all seals from the day that they were born - ageing here isn't a daily increase in age,
  ;; but deterioration of structure (acceleration and hazard rates)
  ask turtles with [ die-now = 0 and stage > 0 ][ calc-ageing ] ; apply aging to turtles

  ;; apply updated variables - must be implemented last as dL includes a stavation sub model which could change
  ;; the values in U_H, U_R and dL depending on activities.
  ask turtles with [ die-now = 0 and status > 0 ][ update ]

  ;; once adults and juveniles have updated their settings apply applicable changes to offspring
  ask turtles with [ die-now = 0 and is_pregnant? = 1 ][ update-offspring-energy ]
  ask turtles with [ die-now = 0 and stage = 1 ][ pup-feeding ]

  ;; apply mortality (ageing accelerations; h_rate from calc-ageing)
  ask turtles with [ die-now = 0 and stage > 0 ][ apply-ageing ]

  ;; end model if no seals or no time left.
  if count turtles <= 20 or ticks = max-ticks [
;    if (use_profiler? = true) [
;      profiler:stop                                               ;; stop profiling
;      print profiler:report                                       ;; view the results
;    ]
    if count turtles <= 20 [ print " stopped as no seals left" ]
    if ticks = max-ticks [ print " stopped as max time reached" ]
    stop                                                          ;; stop running the model
  ]
  ;; sanity check
  ;; to aviod completely crazy populations - stop the model when too many seals
  if count turtles > 50 * start_population [
;    if (use_profiler? = true) [
;      profiler:stop                                               ;; stop profiling
;      print profiler:report                                       ;; view the results
;    ]
    print "stopped as too many seals"
    stop
  ]
  ;; then the plots are updated - as when no turtles left, the plots show errors.
                          ;;as per below
  ask turtles with [die-now = 1 and days-dead = 0][
    set final-age age / 360
    set final-pups total-puppies
    set reprod-period round (final-age - breeding-age)
    if reprod-period = 0 [ set reprod-period 1]
    set fecundity final-pups / round reprod-period
    set days-dead 1
  ]

;  ask turtles [if stage > 1 and f > 1 [set f 1]]

  do-plots
  ask turtles with [ die-now = 1 and days-dead > 1 ][ die ] ;; plots have been updated - so seals can now die (and be removed from the model)

end

;;===================================================================================================
;; ----------------- RESERVE DYNAMICS ----------------------------------------------------------------
;; change in reserves: determined by the difference between assimilation (feeding; S_A) and mobilization
;; (energy use; S_C) fluxes. when food-dynamics are constant f = the value of f_scaled set in the user interface

to calc-dU_E
;  set f (f_scaled + competition) * iv;; competition calculated at start of timestep
;  if f > 1 [set f 1]                                 ;; cannot have access to more than 100% food

  set e_scaled (v_rate * (U_E / (L ^ 3)))  ;; calculate the reserve energy --> usable energy saved over time,
                                           ;;this changes as U_E and L change over time (de/dt = (f-e) v_rate / L)

  ;; calculate S_C depending on foraging or fasting is implemented
  if status = foraging [ set S_C (L ^ 2 * ((g * e_scaled) / (g + e_scaled)) * (1 + ((L * k_M_rate) / v_rate ))) ]
  if status = fasting [
    set f 0 ; fasting, thus no intake
    set S_C (((k_m_rate * g * kap) / v_rate) * (L ^ 3))
  ]

  if age < 360 and status = foraging [
    set xx random-float 100
    if xx < 80 [ set f 0.2 * f ]
  ]

  ifelse is_pregnant? = 1 and [stage] of turtle offspring_ID = 0  ;; != nobody has been removed from here.
  [ set S_A (f * ((L ^ 2) + ([L ^ 2] of turtle offspring_ID))) ]  ;; upregulation of intake during pregnancy
  [ set S_A (f * (L ^ 2)) ]

  if age < (age_for_pup_energy_change * 360) [
    if pup-affected = 1 [ set S_A S_A * (1 - SA_change) ]
  ]

  set dU_E (S_A - S_C)
end

to calc-dU_H-R ;;-----------------------------------------------------------
  ifelse U_H > U_H^p
;  ifelse stage = mature
  [ set dU_R ((1 - kap) * S_C - k_J_rate * U_H^p)  ; adult
    set dU_H 0
  ]; -----------------------------------------------------
  [ set dU_H ((1 - kap) * S_C - k_J_rate * U_H)    ; juvenile
    set dU_R 0
  ]

end


;; ----------------- DYNAMICS OF STRUCTURAL LENGTH----------------------------------------------------
;; the following procedure calculates change in structural length, if growth is negative the individual
;; does not have enough energy to pay somatic maintenance and the starvation submodel is run where growth
;; is set to 0 and individuals divert enough energy from development (for juveniles) or reproduction
;; (for adults) to pay maintenance costs
to calc-dL

  ;; non starvation conditions:
  if L < Lmax [
    set dL ((1 / 3) * (((v_rate /(g * L ^ 2)) * S_C) - k_M_rate * L))
    if dL < 0 [ set dL 0 ]
  ]

  ;; if scaled energy per unit of structure is less than the scaled length --> use starvation strategy
  set l_scaled L / Lmax    ;; scaled length
  if e_scaled < l_scaled [ ;; modified following section 4.2.3 of DEB-IBM user manual (Martin et al)
    set dL 0               ;; cannot have negative growth so set growth to zero
    set S_C ((k_m_rate * g * kap)/ v_rate) * (L ^ 3)

    ;; check to see if the turtle is mature to see where remainig energy is diverted to
;    ifelse U_H > U_H^p
    ifelse stage = mature ;---------------------------------------------------------------------------------
    [ set dU_R (1 - kap) * S_C - k_J_rate * U_H^p - kap * L ^ 2 * (l_scaled - e_scaled) ;; mature (dU_R)
      set dU_H 0
    ]
    [ set dU_H (1 - kap) * S_C - k_J_rate * U_H^p - kap * L ^ 2 * (l_scaled - e_scaled)  ;; immature (dU_H)
      set dU_R 0
    ]
    ;; adjusted reserves as S_C changed
    set dU_E (S_A - S_C)
    if e_scaled <= 0 [ type turtle who type " with stage = " type stage print " died as e < 0 " ;; sanity check
      if offspring_ID > -1 [
        ask turtle offspring_ID [
          if stage = 0 [
            set die-now 1
            set died-of-? 2
          ]
          if stage = 1 [
            set mother_ID -1
            set stage 2 ;immediate juvenile as it needs to fend for itself
            set status 1 ; fasting as left on land
            set resting? 0
            set moulting? 1 ; so that pup stays on land for ~50 days
            set f (f_scaled + competition) * iv
            if f > 1 [set f 1]
            ;; as offspring update happens after adults - go through this now -
            ;; but in this case only the check-status as energy already updated in update-offspring-energy
            check-status
            type turtle who type " juv as mum died. Age: " print [age] of turtle who
          ]
        ]
      ]
      set die-now 1
      set died-of-? 1
    ]
  ]
end
;; ----------------- AGEING --------------------------------------------------------------------------
;; the following procedure calculates the change in damage enducing compounds of an individual
to calc-ageing
  ;; h_a not dependent on food density - thus affects the max life span.
  ;; s_G dependent on food density
  ;; h_a = max life span of ~25 years.
  ;; s_G from Matlab code at T, h_a manually selected based on max age.

  ;; calculates the rate of growth, scaled acceleration change & scaled hazard rate
  set r_rate (3 / L) * dL
  set dq_accel (((q_accel * (L ^ 3 / Lmax ^ 3) * s_G) + h_a ) * e_scaled * ((v_rate / L) - r_rate) - (r_rate * q_accel))
  set dh_rate (q_accel - (r_rate * h_rate))

  ; apply effects
  set q_accel q_accel + dq_accel
  set h_rate h_rate + dh_rate
end

to apply-ageing
  ;; calculate chance of death
  let mortality-chance random-float (mortality-float * iv)
  if iv < 0.95 [ set mortality-chance mortality-chance / 10]

  if mortality-chance < h_rate [
    if offspring_ID > -1 and turtle offspring_ID != nobody [
      ; print " mother died of old age" ;; sanity check
      ask turtle offspring_ID [
        if stage = 0 [
          set die-now 1
          set died-of-? 2
        ]
        if stage = 1 [
          set mother_ID -1
          set stage 2             ;; immediate juvenile as it needs to fend for itself
          set status 1            ;; fasting as left on land
          set resting? 0
          set moulting? 1         ;; so that pup stays on land for ~50 days
          set f (f_scaled + competition) * iv
          if f > 1 [set f 1]
          ;; as offspring update happens after adults - go through this now -
          ;; but here only the check-status as energy already updated in update-offspring-energy
          check-status
        ]
      ]
    ]
    set die-now 1
    set died-of-? 3
  ]

  ;; apply pup mortality to pups within their first year of life
  ;; calculation for mortality: (1/360) * (1-0.716) = 7.89e-4
  ;; based on McMahon et al 2000, 2003 for a
  ;; survival rate of 71.6% as pups weigh (over) the required amount
  if pup_mort? [
  if age < 360 [                              ;; yearling
    let chance random-float 1
    if chance < pup-mortality [
      if offspring_ID > -1 [
        print "error - yearling has a pup?"
        ask turtle offspring_ID [
            set die-now 1
            set died-of-? 4
          ]
        ]
        if mother_ID > -1 [                   ;; advice mother of death of pup if applicable
          ask turtle mother_ID [
            set offspring_ID -1
            set lactating? 0
            set is_pregnant? 0
            set impregnated? 0
            set time-since-suckling 0
            ; not changing breeding or mating status as it could be that the mother has been impregnated again
          ]
        ]
        set die-now 1
        set died-of-? 5
      ]
    ]
  ]
end

; ============================ UPDATE ===============================================================
to update

  if (dU_R <= 0 and dU_H > 0) [ ;; somatic growth till age 6, start breeding at age 3 (Desprez et al 2014)
    if age > 360 [              ;; but should really save energy for self in first year...
      set dU_R (dU_H * 0.6)     ; 60% to dU_R
      set dU_H (dU_H - dU_R)
    ]
  ]

  set U_E U_E + dU_E
  set U_H U_H + dU_H
  if stage = juvenile and U_H > U_H^p [ ;; if they reach puberty levels - move extras into U_R
    set dU_R dU_R + (U_H - U_H^p)
    set U_H U_H^p
  ]

  set U_R U_R + dU_R
  set L L + dL
  if (breeding-age < 1 and U_R > cum_UE) [ set breeding-age age / 360 ]


  if stage = 2 and age < 360 [
    if U_H < 0.92 * U_H^x [
      set die-now 1
      set died-of-? 1
    ]
  ]

  ;; kill off seals with energy levels too low
  if U_R < 0 or U_H < 0 or e_scaled < 0 [
    if mother_ID > -1 [ print "energy levels too low to continue pregancy"
      ask turtle mother_ID [
        set offspring_ID -1
        set breeding? 0
        set lactating? 0
        set is_pregnant? 0
        set impregnated? 0

        set tried-mating 0
        set time-since-mating 0
        set time-since-suckling 0
      ]
    ]
    if offspring_ID > -1 [
      ifelse [stage] of turtle offspring_ID = 0
      [ ask turtle offspring_ID [        ;; foetus
          set die-now 1
          set died-of-? 2
        ]
      ]

      [ ask turtle offspring_ID [        ;; pup
          set mother_ID -1
          set stage 2           ;;immediate juvenile as it needs to fend for itself
          set status 1          ;; fasting as left on land
          set resting? 0
          set moulting? 1       ;; so that pup stays on land for ~50 days
          set f (f_scaled + competition) * iv
          if f > 1 [set f 1]
          ;; as offspring update happens after adults - go through this now
          check-status
          calc-dU_E
          calc-dU_H-R
          calc-dL
          calc-ageing
          update
        ]
      ]
    ]
    set die-now 1
    set died-of-? 1
  ]
  ;; update stages (birth and weaning to juvenile to be done in different section later)
  if U_H >= U_H^p and stage = juvenile [ set stage mature ]

end
;=======================================================================================
to check-status
  let max-moulting-time 0 ;; max moulting time set following Carrick et al 1962
  let max-resting-time 0  ;; max resting  time set following Carrick et al 1962
  let max-foraging-time 0 ;; max foraging time set following Carrick et al 1962

  ;; setting the correct limits for each stage - applies only to juveniles and adults
  ;; iv (individual variability) applied to durations
  if stage = 2 [
    ;; Carrick et al (1962) states that moulting for juveniles could be up to 4 weeks.
    set max-moulting-time moult-duration-juv * iv
    set max-resting-time resting-duration-juv * iv
    set max-foraging-time forage-duration-adult * iv
    ;; However, juveniles cannot stay out for too long in their first year,
    ;; and newly weaned pups need to stay on land for ~5-6 weeks (here 50 days).
    ;; pup resting after weaning period is included on the interface.
    if age < 360 [
      set max-moulting-time moult-duration-pup * iv
      set max-foraging-time forage-duration-juv * iv
    ]
  ]
  ;; Female southern elephant seals come on land twice per year. Once for breeding where they come
  ;; on land, give birth approx. 8 days later, then have a suckling pup for 23 days after which
  ;; they return to the water. During this time the seals do not enter the water and loose approx. 35%
  ;; of their body weight over three weeks (Desprez et al 2014). The second time that the seals come on
  ;; land is approx. 70 days after they have returned to the water following breeding (Hindell et al 1994)
  ;; (98 days acording to Carrick et al 1962) for moulting. The adult female seals stay on land for
  ;; approx. 30 days, however, they might enter the water for a quick feed off shore (Hindell et al 1994).
  ;; In this model this is ignored, for simplicity.
  if stage = 3 [
    ;; Adult moulting 30 - 50 days for females and males, respectively (Hindell et al 1994)
    ;; foraging 70 days after breeding season for moulting by breeding adults - (Hindell et al 1994)
    set max-moulting-time moult-duration-adult * iv
    set max-resting-time resting-duration-adult * iv
    set max-foraging-time forage-duration-adult * iv
  ]

  ;;========================================= FORAGING ================================================
  ;;-------------------------------------------------------
  ;; Do this before the setting of fasting periods as otherwise they miss a day.The options for
  ;; counting the days that they are fasting are either time-since-breeding-start,
  ;; time-after-breeding-end or days-moulting.
  ;; Applied to either foraging or fasting status.
  if status = foraging [
    set days-foraging days-foraging + 1
    if days-foraging > max-foraging-time [
      set days-foraging 0
      set status fasting
      set resting? 1
    ]
  ]

  ;;========================================== FASTING =================================================
  ;; Keeping track of days out at sea, particularly for young seals.
  ;; If they are out for too long, they'll come back onto land for a short while to rest.
  ;; fasting for both moulting and resting - resting includes the period where mum gives birth to pup.
  ;; lactating duration dealt with in update-offspring-energy

  if status = fasting [
    if moulting? = 1 [ ;;----------------------- moulting
      set days-moulting days-moulting + 1
      if days-moulting > max-moulting-time [
        set days-moulting 0
        set moulting? 0
        set resting? 0
        set status foraging
        set days-foraging 0
      ]
    ]

    if resting? = 1 [ ;;------------------------ resting
      set days-resting days-resting + 1
      if days-resting > max-resting-time [
        set days-resting 0
        set resting? 0
        set status foraging
        set days-foraging 0
      ]
    ]

    if haul-out? = 1 [ ;;----------------------- mid winter haul out -- Hindell & Burton 1988
      set days-resting days-resting + 1
      if days-resting > (mwho-juv * iv) [ ;; as set on interface
        set days-resting 0
        set haul-out? 0
        set status foraging
        set days-foraging 0
      ]
    ]
  ]
  ;; --------- implement annual moulting for females
  if month-of-year = 12 [
    if day-of-month > 1 and day-of-month < 7 [
      if age > 360 [                                  ;;as newly weaned pups have already moulted by now
        set status fasting
        set days-foraging 0
        set moulting? 1
        set days-moulting 0
      ]
    ]
  ]

  ;; Juveniles need to have a mid-winter haul out - which is implemented here. Duration is set on the interface
  ;; following Carrick et al 1962. This haul out could be anywhere from April to July, but for now set for
  ;; only July.
  ;;-------------implement midwinter haulout
  if month-of-year = 7 [                  ;; july
    if day-of-month > 5 and day-of-month < 15 [
      if stage = 2 [                        ;; juvenile
        if status  = foraging [             ;; foraging
          set status fasting                ;; now fasting during haul out
          set haul-out? 1
          set resting? 0
          set moulting? 0
          set days-resting 0
        ]
      ]
    ]
  ]
end

;;====================================== BREEDING ================================================
to check-breeding
  ;; start in reverse chronological order so not everything happens in the same time step!
  ;; breeding is set to 1 in september - which is when the females come onto land  to get impregnated.
  ;; This is only applied if the seal wasn't already impregnated while suckling it's previous pup.
  ;; As ses technically only breed every two years the breeding like this should be applied if
  ;; breeding is unsuccesful during weaning of previous pup.
  if breeding? = 1 [
    if offspring_ID > -1 and turtle offspring_ID != nobody [ ;;-------------------------------
      if ([stage] of turtle offspring_ID = 0)
      and [age] of turtle offspring_ID >= breeding-duration
      [
        give-birth
        set time-since-mating 0
      ]
      ;; preparation for birth on land - 8 days prior following Carrick et al // Deprez et al?
      if time-since-mating = (breeding-duration + diapause - 8)[
        set status fasting
        set resting? 1
        set days-foraging 0
        set days-resting 0
      ]
    ]
    ;; time for pregnancy to start - foetus created following Laws 1984 in Knox 2007
    if time-since-mating = diapause and is_pregnant? = 0 [
      ifelse offspring_ID > -1 [
        ifelse turtle offspring_ID = nobody
        [ set offspring_ID 0
          set lactating? 0
          set is_pregnant? 0
          set impregnated? 0
          set time-since-suckling 0
          set status foraging
          set days-foraging 0

          set puppy 0
        ]
        [ if [age] of turtle offspring_ID > 90 [
          ask turtle offspring_ID [
            set f (f_scaled + competition) * iv
            if f > 1 [set f 1]
            set stage juvenile
            set status fasting
            set moulting? 1          ;; so that pup stays on land for 50 days
            set days-moulting 0
            set mother_ID -1
          ]
          set offspring_ID -1
          set lactating? 0
          set is_pregnant? 0
          set impregnated? 0
          set time-since-suckling 0
          set status foraging
          set days-foraging 0

          set puppy 0
          ]
        ]
      ]
      [ do-pregnancy ]
    ]

    if impregnated? = 1 or is_pregnant? = 1[                         ; add a day to the pregnancy
      set time-since-mating time-since-mating + 1
    ]
  ]

  ;;seals not breeding yet - but it's the right time so could be:
  if lactating? = 0 [                            ;; not lactating
    if impregnated? = 0  and is_pregnant? = 0 [  ;; not pregnant from this season
      if tried-mating = 0[                       ;; hasn't tried to get pregnant yet
        if month-of-year = 10 [                  ;; it is the right month
          if day-of-month < 5 [                  ;; first week of the month
            if status = foraging [               ;; wasn't on land already
              set status fasting                 ;; adjusting settings
              set resting? 1
              set days-foraging 0
              set days-resting 0
            ]
          ]
        ]
      ]
    ]
  ]

  if impregnated? = 0  and is_pregnant? = 0 and status = fasting [ ;; get pregnant
    if month-of-year = 10 [
      if tried-mating < 7 [
        get-impregnated
      ]
    ]
  ]

  ;; Included a cycle where lactating mothers breed with invisible males while they're on land with a pup.
  ;; Mothers come on land about 8 days before giving birth. They then give birth and stay on land for
  ;; 23 days with the pups while they suckle. During this time the females are getting ready for mating
  ;; again - and although they generally (at Macca) only breed every other year, it'd be good to get
  ;; this going - during that time. So after about 19 days the female can get impregnated again.
  ;; This leaves 4 days for trying to get pregnant - if it doesn't work, they won't breed this year.
  ;; Following Laws 1984, Hindell et al 1994, Hindell & Slip 1997.
  if lactating? = 1 [
    if impregnated? = 0 and is_pregnant? = 0 [
      if month-of-year = 10 [
        if (time-since-suckling >= 19) [ get-impregnated ]
      ]
    ]
  ]

  if month-of-year = 9 [                                             ;; check month for start breeding
    if breeding? = 0 [                                               ;; check if not yet breeding
      if is_pregnant? = 0 and impregnated? = 0 [ set breeding? 1 ]   ;; check that not already pregnant or impregnated
    ]
  ]

  if month-of-year = 11 and day-of-month = 1 [                       ;; failed breeders - going back to foraging
    if breeding? = 1 [
      if is_pregnant? = 0 and impregnated? = 0 [
        set breeding? 0
        set status foraging
        set resting? 0
        set time-since-mating 0
        set days-foraging 0
        set tried-mating 0
      ]
    ]
  ]

end
;;================================================
to get-impregnated
  ;; mating success following Fedal et al 1996, Deprez et al 2014
  ;; 3 yo @ 0.98, 4 yo @ 0.21, 5 yo @ 0.15, 6+ @ 0.75
  ;; no breeding senecense for seals - following Hindell and Little 1988
  let prob_mating_fail 0
  set tried-mating tried-mating + 1

  ifelse age < (6 * 360)
  [ ifelse age < (5 * 360)
    [ ifelse age < (4 * 360)
      [ set prob_mating_fail prob_mate_fail_3yo ]  ;; < 4 yo
      [ set prob_mating_fail prob_mate_fail_4yo ]  ;; 4-5 yo
    ]
    [ set prob_mating_fail prob_mate_fail_5yo ]    ;; 5-6 yo
  ]
  [ set prob_mating_fail prob_mate_fail_6plus ]    ;; > 6 yo

  let cum-chance random-float 1                      ;; value 0 - 1
  if cum-chance > prob_mating_fail [
    set impregnated? 1
    set time-since-mating 1
    set tried-mating 0
    set breeding? 1
  ]

end
;;================================================
to do-pregnancy
  set is_pregnant? 1
  ;; NOTE:
        ;; original model for daphina spp. sets UR to zero after reproduction/hatching of offspring,
        ;; however, this CANNOT be done for seals as during reproductive time the seals need to fast
        ;; and take energy from UR to support themselves. Hence the check of UR levels before reproduction,
        ;; as personal survival is more important to the mothers than giving birth/producing offspring.

        ;; from section 7.7 in the DEB book and the associated comments we know the calculation for
        ;; cummulative energy invested in the pups. calculations are available for both foetal and
        ;; lactation periods. Thus we can make sure that a seal only goes ahead with the pregnancy if
        ;; they have enough energy to support the foetus through to birth.

  if U_R < cum_UE [
    set is_pregnant? 0
    set impregnated? 0
    set time-since-mating 0
    set status foraging
    set days-foraging 1
    set breeding? 0
    set puppy 0
  ]

  ;; creating a new turtle/individual......
  if is_pregnant? = 1 [
    hatch 1 [
      let hatch_ID who
      set mother_ID my_ID
      ask turtle mother_ID [ set offspring_ID hatch_ID ]        ;; connect mother and young
      set offspring_ID -1
      set my_ID who

      set f (f_scaled + competition) * iv
      if f > 1 [set f 1]

      set age 1
      set stage foetus
      set status mother-dependent

      set-foetus-variables
    ]
  ]
end
;;====================================
to give-birth
  set breeding? 0
  set is_pregnant? 0
  set impregnated? 0
  set time-since-mating 0
  set resting? 1
  set lactating? 1
  set time-since-suckling 0
  set status fasting

  if turtle offspring_ID = nobody [
    set puppy 0
    set total-puppies total-puppies - 1
    set resting? 0
    set lactating? 0
    set status foraging
    set breeding? 0
    set pups-born pups-born - 1
  ]
  if turtle offspring_ID != nobody [
    ask turtle offspring_ID [
      set h_rate random-float 1.0e-19
      set stage pup
      set age 1
      set weaning_age 0

      set xx random-float 100
      if xx < percent-pup-affected [ set pup-affected 1 ]
    ]
    set puppy 1
    set total-puppies total-puppies + 1
    set pups-born pups-born + 1
  ]

end
;;===============================================================================================================
to update-offspring-energy
  if offspring_ID > 0 [
    ;; separating stages of offspring between foetus and pups
    if [stage] of turtle offspring_ID = foetus [
      ask turtle offspring_ID [

        ;; following the additonal DEB information - "Comments on DEB book" - page 152, 501

        ;; adjusted the values here that aren't directly deived from Matlab - this makes a difference and allows
        ;; the seals grow at an accurate rate. Calculations for length following the fast foetal development should
        ;; be L(t) = t * v_rate / 3, but then the seals take 900-odd days till birth. Adjusting to match 217 days the
        ;; calculation becomes 25.85 = 217 * v_rate / 0.545 where 25.85 = L_w^b * shape-factor
        ;; thus:
        let r_B v_rate / (0.545 * Lmax)     ;; adjusted growth calculation for fast foetal development - birth @ U_H^b

        set dL r_B * Lmax
        if pup-affected = 1 [ set dL dL * (1 - (SA_change / 2)) ]
        set L L + dL

        ;; The following calcualtions approximately follow the DEB book (p. 283). Reserve flux from mother
        ;; to foetus bypasses assimilatory system of the foetus - thus dU_E = S_A
        set dU_E [e_scaled] of turtle mother_ID * L ^ 2 * kap_F                     ;; adjusted kap_F and modified formula
        if pup-affected = 1 [ set dU_E dU_E * (1 - SA_change) ]

        set dU_H dU_E

        set U_E U_E + dU_E
        set U_H U_H + dU_H

      ]
      ;; update mother's reproduction buffer
      set U_R U_R - ([dU_E * kap_F] of turtle offspring_ID)
    ]
  ]
end

to pup-feeding

  ifelse mother_ID > -1 [ ;;--------------- sanity check
    set f f_scaled * 2 * iv                            ;; higher fat content in milk (16 - 55% Hindell et al 1994)
    set e_scaled (v_rate * (U_E / (L ^ 3)))

    ;; test for proportional growth of pup and energy supply by mum
    set S_A (f * L ^ 2) / kap_L
    if pup-affected = 1 [ set S_A S_A * (1 - SA_change) ]

    set S_C 3 * (L ^ 2 * ((g * e_scaled) / (g + e_scaled)) * (1 + ((L * k_M_rate) / v_rate ))) ;; increased the energy mobilisation of pups

    set dU_E (S_A - S_C)
    set dU_H ((1 - kap_L) * S_C - (k_J_rate * U_H))
    set dL (((v_rate /(g * L ^ 2)) * S_C) - (k_M_rate * L))  ;; adjused for increase rate of growth (*3)

    set L L + dL
    set U_E U_E + dU_E
    set U_H U_H + dU_H
    set weaning_age weaning_age + 1
    set l_scaled L / Lmax

    ask turtle mother_ID [
      set U_R U_R - [ S_A * kap_L ] of turtle offspring_ID;--------see comments on DEB book (Kooijman 2010) - page 38
      set time-since-suckling time-since-suckling + 1
    ]
  ]
  [ ;; there's no mum -- they should go into fasting for some time before heading to the water for feeding
    set f (f_scaled + competition) * iv
    if f > 1 [set f 1]
    set stage juvenile
    set status fasting
    set moulting? 1          ;; so that pup stays on land for 50 days
    set days-moulting 0
  ]

  if age > weaning-duration
  [ ask turtle mother_ID [
      ;; update mother's details
      set offspring_ID -1
      set lactating? 0
      set is_pregnant? 0
      set time-since-suckling 0
      set status foraging
      set days-foraging 0

      set puppy 0
    ]
    ;; update pup details
    set f (f_scaled + competition) * iv
    if f > 1 [set f 1]
;    set f 0.5
    set stage juvenile
    set status fasting
    set moulting? 1          ;; so that pup stays on land for 50 days
    set days-moulting 0
    set mother_ID -1
  ]

end
;;===============================================================================================================
; -------------------------- TIME MANAGEMENT --------------------------------------------------------------------
to do-time-management
  ;; model works on a 360 day year, where each month has 30 days.
  ask turtles [ set age age + 1 ]
  ask turtles with [ die-now > 0 ][ set days-dead days-dead + 1 ]

  set day-of-year day-of-year + 1
  set day-of-month day-of-month + 1

  if day-of-month > 30 [
    set day-of-month 1
    set month-of-year month-of-year + 1
  ]

  if month-of-year > 12 or day-of-year > 360 [
    set day-of-year 1
    set day-of-month 1
    set month-of-year 1

    set year year + 1
    ask turtles with [U_H > U_H^b] [ set had-pup? 0 ]
  ]

  if ticks = (clear-plots * 360) [ clear-all-plots ]

;-------------------------------------- IMPLEMENT SCENARIOS
  ;--------------------------------------------------------DENSITY DEPENDENCE
  if food_change? [
  ifelse max-pop-2 < max-pop-1
  ;K2 < K1 - - population decrease
  [ if (ticks >= implement-change-at * 360)                      ;; K2 is smaller so at this time the population starts declining
    and (carrying_capacity > max-pop-2)[                    ;; assuming (check here) that the population hasn't already reached K2
      let t (ticks / 360)                                   ;; setting t at this time

      if t = implement-change-at [                               ;; if t (now) is when the population should start declining
        set carrying_capacity carrying_capacity - (rate-of-decline * period-in-K)   ;; set the new K at the decline calculated for a x year period (set on interface)
        set t-for-k2 t + period-in-K                                                ;; and reset the time for the next change 10 years further
      ]
      if t = t-for-k2 [                                     ;; if t (now) equals time for the next change
        set carrying_capacity carrying_capacity - (rate-of-decline * period-in-K)   ;; set the new K at the decline calculated for a ten year period
        set t-for-k2 t + period-in-K                                                ;; and reset the time for the next change 10 years further
      ]
    ]
  ]

  ;K2 > K1 - - population increase
  [ if ticks = implement-change-at * 360 [
      set carrying_capacity max-pop-2   ;; as the poplation increases slowly - this can be a one off change.
      ]
    ]
  ]

;------------------------------------------------------------- CLIMATE VARIATION
  if decadal_change? [
    let t (ticks / 360)

    if t = implement-change-at or t = t-for-K2 [
      if change_freq = "ENSO"    [ set change-time 7 ]
      if change_freq = "SAM"     [ set change-time 4 ]
      if change_freq = "decadal" [ set change-time 10 ]
      if change_freq = "random"  [ set change-time 15 ]

      set climate-change 1
      set t-for-K2 t + change-time
    ]

    if climate-change = 1 [
      set clim-var ((rv * (sin ((360 / change-time) * year))) + 1)
    ]
  ]

  ;------------------------------------------------------------- IMPLEMENT CHANGE IN PUP ENERGY INTAKE
  if pup_energy_change? [
    let t (ticks / 360)
    if t = implement-change-at [
;      if percent_of_pup_change = "five"   [ set percent-pup-affected 5 ]  ; for 5% of pups affected
;      if percent_of_pup_change = "ten"    [ set percent-pup-affected 10 ] ; for 10% of pups affected
      if percent_of_pup_change = "twenty" [ set percent-pup-affected 20 ] ; for 20% of pups affected
;      if percent_of_pup_change = "thirty" [ set percent-pup-affected 30 ] ; for 30% of pups affected
;      if percent_of_pup_change = "fourty" [ set percent-pup-affected 40 ] ; for 40% of pups affected
      if percent_of_pup_change = "fifty"  [ set percent-pup-affected 50 ] ; for 50% of pups affected
;      if percent_of_pup_change = "sixty"  [ set percent-pup-affected 60 ] ; for 60% of pups affected
      if percent_of_pup_change = "eighty" [ set percent-pup-affected 80 ] ; for 40% of pups affected

;      if percent_of_energy_change = "five"   [ set SA_change 0.05 ] ; for 5% less energy intake
;      if percent_of_energy_change = "ten"    [ set SA_change 0.10 ] ; for 10% less energy intake
      if percent_of_energy_change = "twenty" [ set SA_change 0.20 ] ; for 20% less energy intake
;      if percent_of_energy_change = "thirty" [ set SA_change 0.30 ] ; for 30% less energy intake
;      if percent_of_energy_change = "fourty" [ set SA_change 0.40 ] ; for 40% less energy intake
      if percent_of_energy_change = "fifty"  [ set SA_change 0.50 ] ; for 50% less energy intake
;      if percent_of_energy_change = "sixty"  [ set SA_change 0.60 ] ; for 60% less energy intake
      if percent_of_energy_change = "eighty" [ set SA_change 0.80 ] ; for 80% less energy intake
    ]
  ]

  ;----------------------------------------------------------- IMPLEMENT CHANGE IN MUMs FECUNDITY // CUM REP THRESHOLD
  if mother_fecundity_change? [
    let t (ticks / 360)
    if t = implement-change-at [
      ;if percent_of_mum_change = "five"    [ set percent-mum-affected 5]
      ;if percent_of_mum_change = "ten"     [ set percent-mum-affected 10]
      if percent_of_mum_change = "twenty"  [ set percent-mum-affected 20]
      if percent_of_mum_change = "fifty"   [ set percent-mum-affected 50]
      if percent_of_mum_change = "eighty"  [ set percent-mum-affected 80]
      ;if percent_of_mum_change = "hundred" [ set percent-mum-affected 100]

      ;if percent_of_fec_change = "five"    [ set fec_change 0.05 ]
      ;if percent_of_fec_change = "ten"     [ set fec_change 0.10 ]
      ;if percent_of_fec_change = "fifteen" [ set fec_change 0.15 ]
      if percent_of_fec_change = "twenty"  [ set fec_change 0.20 ]
      if percent_of_fec_change = "fifty"   [ set fec_change 0.50 ]
      if percent_of_fec_change = "eighty"  [ set fec_change 0.80 ]

      set fecundity_changed_now? 1
    ]
  ]

  ;; ==============================================================TRACK FOR LIFE TABLES
  if ticks > 56 * 360 and ticks < 57 * 360 [
    ask turtles with [stage = 1 and age < 10] [set track-life 1]
  ]

  if ticks > 90 * 360 and ticks < 91 * 360 [
    ask turtles with [stage = 1 and age < 10] [set track-life 1]
  ]

end
;;===============================================================================================================
to set-foetus-variables
  ;; - - - - - - - - - - - - - - - STATE VARIABLES - - - - - - - - - - - - - - - - - - - - - - - - - -
  set L 0.0001
  set dL 0
  set U_H 0.0001
  set dU_H 0

  set U_E 0.0001
  set dU_E 0
  set e_scaled [e_scaled] of turtle mother_ID
  set l_scaled L / Lmax

  set U_R 0
  set dU_R 0

  ;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set iv e ^ (random-normal 0 cv)

;  set J_XAm_rate   J_XAm_rate_init * iv
  set g g_init * iv

  set U_H^b (E_H^b_init / p_am) ;* iv        ;; scaled maturity at birth (cm^2 d)
  set U_H^x (E_H^x_init / p_am) ;* iv        ;; scaled maturity at weaning (cm^2 d)
  set U_H^p (E_H^p_init / p_am) ;* iv        ;; scaled maturity at puberty (cm^2 d)
  ;; - - - - - - - - - - - - - - - AGEING -- - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set r_rate 0
  set q_accel 0
  set dq_accel 0
  set h_rate 0
  set dh_rate 0

  ;; - - - - - - - - REPRODUCTION / MOULTING
  set cumulative_UE (((L_w^b * shape_factor) ^ 3) * ((f_scaled + g) / v_rate) * (1 + (3 / 4) * ((L_w^b / L_w^m) / f_scaled)))
  set cum_UE cumulative_UE * l_scaled * 0.2

  set is_pregnant? 0
  set resting? 0
  set moulting? 0
  set breeding? 0
  set impregnated? 0

  set had-pup? 0
  set lactating? 0

  set tried-mating 0
  set time-since-mating 0
  set time-since-suckling 0
  set days-moulting 0
  set days-foraging 0
  set days-resting 0

  set puppy 0
  set total-puppies 0

  set die-now 0
  set died-of-? 0
  set final-age 0
  set final-pups -1
  set breeding-age 0
  set reprod-period 0
  set fecundity -1

  set weaning_age 0
  set days-dead 0

  set pup-affected 0
  set mum-affected 0
  set xx random-float 100
  if xx < percent-pup-affected [ set pup-affected 1 ]
  set xx random-float 100
  if xx < percent-mum-affected [ set mum-affected 1 ]

  set track-life 0

end


;;===============================================================================================================
;; ----------------- PLOTS ----------------------------------------------------------------------------
to do-plots
;;   -------------------------------------- population plotting
  set-current-plot "total population numbers" ;;--------------------- total numbers
  set-current-plot-pen "adult"
  plot (count turtles with [stage = 3])
  set-current-plot-pen "juvenile"
  plot (count turtles with [stage = 2 and age > 360])
  set-current-plot-pen "pup"
  plot (count turtles with [stage = 1])
  set-current-plot-pen "embryo"
  plot (count turtles with [stage = 0])
  set-current-plot-pen "yearling"
  plot (count turtles with [age < 360 and stage > 0])
  set-current-plot-pen "population"
  plot (count turtles with [stage > 1])

  set-current-plot "population dynamics" ;; ------------------------- proportion
  set-current-plot-pen "0.5"
  plot 0.5
  set-current-plot-pen "adult"
  plot ((count turtles with [stage = 3]) / (count turtles + 0.001))
  set-current-plot-pen "juvenile"
  plot ((count turtles with [stage = 2]) / (count turtles + 0.001))
  set-current-plot-pen "pup"
  plot ((count turtles with [stage = 1]) / (count turtles + 0.001))
  set-current-plot-pen "embryo"
  plot ((count turtles with [stage = 0]) / (count turtles + 0.001))
  set-current-plot-pen "yearling"
  plot ((count turtles with [age < 360 and stage > 0]) / (count turtles + 0.001))

  set-current-plot "count" ;; ---------------------- total pups and breeders
  set-current-plot-pen "pups"
  plot (sum [puppy] of turtles )
  set-current-plot-pen "breeders"
  plot (count turtles with [U_R > cum_UE])  ; calculation for breeding energy levels
  set-current-plot-pen "total"
  plot count turtles
  set-current-plot-pen "K-pop"
  plot count turtles with [stage > 1]
  set-current-plot-pen "K"
  plot carrying_capacity

  set-current-plot "test pups survival" ;------------------------------------------- pup survival
  set-current-plot-pen "embryo"
  plot count turtles with [stage = 0]
  set-current-plot-pen "pup"
  plot count turtles with [stage = 1]
  set-current-plot-pen "yearling"
  plot count turtles with [stage = 2 and age < 350];-------------------------------------------------------------------- analysis

  set-current-plot "competition combined"
  set-current-plot-pen "comb"
  if any? turtles with [stage > 1 and age > 360 and status = 2 and f > 0][;-------------------------------------------------------------------- analysis
    plot mean [f] of turtles with [stage > 1 and f > 0]
  ]

  ;;;  ;; ----------------------------------------------- growth actual length
;;;  ;; to plot scaled length - remove "/ shape_factor" from plot and
;;;  ;; add "* shape_factor" to the set levels
  set-current-plot "L"
  if any? turtles with [stage > 1][ ;; set levels
    set-current-plot-pen "Lmax"
    plot mean [Lmax / shape_factor * f_scaled] of turtles
    set-current-plot-pen "Lp"
    plot mean [L_w^p] of turtles
    set-current-plot-pen "Lx"
    plot mean [L_w^x] of turtles
    set-current-plot-pen "Lb"
    plot mean [L_w^b] of turtles
  ]
  set-current-plot-pen "mean ad"        ;; adults
  ifelse any? turtles with [stage = 3]
  [ plot mean [L / shape_factor] of turtles with [stage = 3] ][ plot 0 ]
  set-current-plot-pen "mean juv"       ;; juveniles
  ifelse any? turtles with [stage = 2]
  [ plot mean [L / shape_factor] of turtles with [stage = 2] ][ plot 0 ]
   set-current-plot-pen "mean pup"       ;; pups
  ifelse any? turtles with [stage = 1]
  [ plot mean [L / shape_factor] of turtles with [stage = 1] ][ plot 0 ]
  set-current-plot-pen "mean embryo"  ;; embryos
  ifelse any? turtles with [stage = 0]
  [ plot mean [L / shape_factor] of turtles with [stage = 0] ][ plot 0 ]
  set-current-plot-pen "min ad"        ;; adults
  ifelse any? turtles with [stage = 3]
  [ plot min [L / shape_factor] of turtles with [stage = 3] ][ plot 0 ]
  set-current-plot-pen "min juv"       ;; juveniles
  ifelse any? turtles with [stage = 2]
  [ plot min [L / shape_factor] of turtles with [stage = 2] ][ plot 0 ]
  set-current-plot-pen "max ad"        ;; adults
  ifelse any? turtles with [stage = 3]
  [ plot max [L / shape_factor] of turtles with [stage = 3] ][ plot 0 ]
  set-current-plot-pen "max juv"       ;; juveniles
  ifelse any? turtles with [stage = 2]
  [ plot max [L / shape_factor] of turtles with [stage = 2] ][ plot 0 ]

  ; ------------------------------------------- energy level plotting
  set-current-plot "total mean Us - adult" ;; ----------------------adult energy
  if any? turtles with [stage = 3][
    set-current-plot-pen "0"
    plot 0
    set-current-plot-pen "U_H^p"
    plot (mean [U_H^p] of turtles)

    set-current-plot-pen "U_H"
    plot (mean [U_H] of turtles with [stage = 3])
    set-current-plot-pen "U_R-high"
    ifelse any? turtles with [stage = 3 and iv > 0.95]
    [ plot (mean [U_R] of turtles with [stage = 3 and iv > 0.95])][plot 0 ]
    set-current-plot-pen "U_R-low"
    ifelse any? turtles with [stage = 3 and iv < 0.95]
    [ plot (mean [U_R] of turtles with [stage = 3 and iv < 0.95])][plot 0 ]
    set-current-plot-pen "cum_UE"
    plot (mean [cum_UE] of turtles with [stage = 3])
  ]

  set-current-plot "total mean Us - juv" ;; -------------------------juvenile energy
  if any? turtles with [stage = 2][
    set-current-plot-pen "0"
    plot 0
    set-current-plot-pen "U_H^x"
    plot (mean [U_H^x] of turtles)
    set-current-plot-pen "U_H^p"
    plot (mean [U_H^p] of turtles)

    set-current-plot-pen "U_H"
    plot (mean [U_H] of turtles with [stage = 2])
    set-current-plot-pen "U_R-high"
    ifelse any? turtles with [stage = 2 and iv > 0.95]
    [ plot (mean [U_R] of turtles with [stage = 2 and iv > 0.95])][plot 0 ]
    set-current-plot-pen "U_R-low"
    ifelse any? turtles with [stage = 2 and iv < 0.95]
    [ plot (mean [U_R] of turtles with [stage = 2 and iv < 0.95])][plot 0 ]
    set-current-plot-pen "cum_UE"
    plot (mean [cum_UE] of turtles with [stage = 2])
  ]

  set-current-plot "pregnancy ages";;-------------------------------- age at pregancy
  set-current-plot-pen "10y"
  set-plot-pen-color 127
  plot 10
  set-current-plot-pen "4y"
  plot 4
  set-current-plot-pen "5y"
  plot 5
  set-current-plot-pen "max preg"
  ifelse any? turtles with [stage = 3 and offspring_ID > -1]
  [ plot (max [age] of turtles with [stage = 3 and offspring_ID > -1] / 360) ][ plot 0 ]
  set-current-plot-pen "mean preg" ;;;----------------------------------------------------------------- re-written below 15/02
  ifelse any? turtles with [stage = 3 and offspring_ID > -1]
  [ plot (mean [age] of turtles with [stage = 3 and offspring_ID > -1] / 360) ][ plot 0 ]
  set-current-plot-pen "min preg"
  ifelse any? turtles with [stage = 3 and offspring_ID > -1]
  [ plot (min [age] of turtles with [stage = 3 and offspring_ID > -1] / 360) ][ plot 0 ]
  set-current-plot-pen "mean preg" ;;;----------------------------------------------------------------- 15/02
  ifelse any? turtles with [offspring_ID > -1]
  [ plot (mean [age] of turtles with [offspring_ID > -1] / 360) ][ plot 0 ]
  set-current-plot-pen "min preg"
  ifelse any? turtles with [offspring_ID > -1]
  [ plot (min [age] of turtles with [offspring_ID > -1] / 360) ][ plot 0 ] ;-------------------------------------------------------------------- analysis
  set-current-plot-pen "first-preg";------------------------------------------ added 15/02
  ifelse any? turtles with [offspring_ID > -1 and total-puppies < 2]
  [ plot (mean [age] of turtles with [offspring_ID > -1 and total-puppies < 2] / 360) ][ plot 0 ]

  set-current-plot "fecundity"
  ifelse any? turtles with [fecundity > -1]
  [ set-current-plot-pen "max"
    plot max [fecundity] of turtles with [fecundity > -1];-------------------------------------------------------------------- analysis
    set-current-plot-pen "mean"
    plot mean [fecundity] of turtles with [fecundity > -1]
    set-current-plot-pen "min"
    plot min [fecundity] of turtles with [fecundity > -1]
  ]
  [ set-current-plot-pen "max"
    plot 0
    set-current-plot-pen "mean"
    plot 0
    set-current-plot-pen "min"
    plot 0
  ]

  set-current-plot "min fecundity"
  if any? turtles with [fecundity > -1]
  [ plot min [fecundity] of turtles with [fecundity > -1]]

  set-current-plot "mean fecundity"
  if any? turtles with [fecundity > -1]
  [ plot mean [fecundity] of turtles with [fecundity > -1]]

  set-current-plot "max fecundity"
  if any? turtles with [fecundity > -1]
  [ plot max [fecundity] of turtles with [fecundity > -1]]

  set-current-plot "max pups over max reproductive period"
  ifelse any? turtles with [reprod-period > 0]
  [ set-current-plot-pen "pups"
    plot max [final-pups] of turtles with [reprod-period > 0]
    set-current-plot-pen "period"
    plot max [reprod-period] of turtles with [final-pups = max [final-pups] of turtles with [reprod-period > 0]]
  ]
  [ set-current-plot-pen "pups"
    plot 0
    set-current-plot-pen "period"
    plot 0
  ]

  set-current-plot "final pups for seals"
  set-current-plot-pen "default"
  ifelse any? turtles with [final-pups > 0]
  [ plot max [final-pups] of turtles with [final-pups > 0] ][plot 0]
  set-current-plot-pen "only breeders - max"
  if any? turtles with [final-pups > 0][ plot max [final-pups] of turtles with [final-pups > 0]];-------------------------------------------------------------------- analysis
  set-current-plot-pen "only breeders - mean"
  if any? turtles with [final-pups > 0][ plot mean[final-pups] of turtles with [final-pups > 0]]
  set-current-plot-pen "mum-aff = 1"
  if any? turtles with [final-pups > 0 and mum-affected = 1][ plot max [final-pups] of turtles with [final-pups > 0 and mum-affected = 1]];-------------------------------------------------------------------- analysis
  set-current-plot-pen "mum-aff = 0"
  if any? turtles with [final-pups > 0 and mum-affected = 0][ plot max [final-pups] of turtles with [final-pups > 0 and mum-affected = 0]];-------------------------------------------------------------------- analysis

  set-current-plot "final age"
  if any? turtles with [final-age > 0]
  [ set-current-plot-pen "default"
    plot mean [final-age] of turtles with [final-age > 0]
    set-current-plot-pen "24"
    plot 24
  ]

  set-current-plot "seals older than 22 yo";;----------------------- count old seals
  set-current-plot-pen "count"
  plot count turtles with [age > (22 * 360)]

  set-current-plot "pup affected plot"
  set-current-plot-pen "default"
  plot ((count turtles with [pup-affected = 1 and age < (360 * 4) ]) / (count turtles with [age < (360 * 4)] + 0.001))
  set-current-plot-pen "0.5"
  plot 0.5

  set-current-plot "lifetable"
  set-current-plot-pen "default"
  ifelse any? turtles with [track-life = 1]
  [plot count turtles with [track-life = 1]]
  [plot 0]
  set-current-plot-pen "pen-1"
  ifelse any? turtles with [track-life = 1]
  [plot sum [puppy] of turtles with [track-life = 1]]
  [plot 0]

;;;;============================================================================================================================
;;;;============================================================================================================================
;;  ;plots for tracking of individual seal - intake, storage------------------------ individual plots
  if (turtle track-turtle != nobody) [

    set-current-plot "f of seal"
    set-current-plot-pen "default"
    plot [f] of turtle track-turtle
    set-current-plot-pen "1"
    plot 1

    set-current-plot "Seal energy use"
    set-current-plot-pen "U_H"
    plot ([U_H] of turtle track-turtle)
    set-current-plot-pen "U_R"
    plot ([U_R] of turtle track-turtle)
    set-current-plot-pen "U_H^b"
    plot ([U_H^b] of turtle track-turtle)
    set-current-plot-pen "U_H^x"
    plot ([U_H^x] of turtle track-turtle)
    set-current-plot-pen "U_H^p"
    plot ([U_H^p] of turtle track-turtle)
    set-current-plot-pen "breeding level"
    plot ([cum_UE] of turtle track-turtle)
    set-current-plot-pen "0"
    plot 0

    set-current-plot "seal energy intake"
    set-current-plot-pen "default"
    plot ([U_E] of turtle track-turtle)

    set-current-plot "seal size"
    plot [L / shape_factor] of turtle track-turtle


    set-current-plot "g and e" ;------------------------------------------- g e
    set-current-plot-pen "ge"
    plot [(g * e_scaled) / (g + e_scaled)] of turtle track-turtle
    set-current-plot-pen "l"
    plot [l_scaled] of turtle track-turtle
    set-current-plot-pen "e"
    plot [e_scaled] of turtle track-turtle


    set-current-plot "pregnant?"
    set-current-plot-pen "preg"
    ifelse [impregnated?] of turtle track-turtle = 1
    [plot 1][plot 0]
    set-current-plot-pen "age"
    ifelse [age] of turtle track-turtle = 1 * 360
    [ plot 1.5 ]
    [ ifelse [age] of turtle track-turtle = 2 * 360
      [ plot 1.5]
      [ ifelse [age] of turtle track-turtle = 3 * 360
        [ plot 1.5 ]
        [ ifelse [age] of turtle track-turtle = 4 * 360
          [ plot 1.5 ]
          [ ifelse [age] of turtle track-turtle = 5 * 360
            [ plot 1.5 ]
            [ ifelse [age] of turtle track-turtle = 6 * 360
              [ plot 1.5]
              [ ifelse [age] of turtle track-turtle = 7 * 360
                [ plot 1.5 ]
                [ ifelse [age] of turtle track-turtle = 8 * 360
                  [ plot 1.5 ]
                    [ ifelse [age] of turtle track-turtle = 9 * 360
                    [ plot 1.5 ]
                    [ ifelse [age] of turtle track-turtle = 10 * 360
                      [ plot 1.5]
                      [ ifelse [age] of turtle track-turtle = 15 * 360
                        [ set-plot-pen-color 105
                          plot 1.5]
                        [ plot 0]
                      ]
                    ]
                  ]
                ]
              ]
            ]
          ]
        ]
      ]
    ]
  ]
;
;;
;;  ;;==============================================================================================================
;;  ;;================================================================================= EXTRA PLOTS=================
;;;
;;;    set-current-plot "U of preg"
;;;  if any? turtles with [offspring_ID > -1]
;;;  [ set-current-plot-pen "0"
;;;    plot 0
;;;    set-current-plot-pen "U_H^p"
;;;    plot (mean [U_H^p] of turtles)
;;;    set-current-plot-pen "U_H^x"
;;;    plot (mean [U_H^x] of turtles)
;;;    set-current-plot-pen "U_H"
;;;    plot (mean [U_H] of turtles with [offspring_ID > -1])
;;;    set-current-plot-pen "U_R"
;;;    plot (mean [U_R] of turtles with [offspring_ID > -1])
;;;    set-current-plot-pen "cum_UE"
;;;    plot (mean [cum_UE] of turtles with [offspring_ID > -1])
;;;  ]
;;
;;;  set-current-plot "dead seals"
;;;  plot count turtles with [die-now = 1]
;;;

;
;;
;;;
;;;  set-current-plot "breeding age"
;;;  ifelse any? turtles with [breeding-age > 0]
;;;  [ set-current-plot-pen "max"
;;;    plot max [breeding-age] of turtles with [breeding-age > 0]
;;;    set-current-plot-pen "mean"
;;;    plot mean [breeding-age] of turtles with [breeding-age > 0]
;;;    set-current-plot-pen "min"
;;;    plot min [breeding-age] of turtles with [breeding-age > 0]
;;;  ]
;;;  [ set-current-plot-pen "max" plot 0
;;;    set-current-plot-pen "mean" plot 0
;;;    set-current-plot-pen "min" plot 0]
;;
;;;  set-current-plot "reproductive period"
;;;  ifelse any? turtles with [reprod-period > 0]
;;;  [ plot max [reprod-period] of turtles with [reprod-period > 0]][plot 0]
;;
;;
;;;  set-current-plot "weaning"
;;;  ifelse any? turtles with [stage = 1]
;;;  [ plot max [weaning_age] of turtles with [stage = 1]]
;;;  [ plot 0]
;;;
;;  set-current-plot "total mean Us - pup" ;; -------------------------pup energy
;;  if any? turtles with [stage = 1][
;;    set-current-plot-pen "U_H^b"
;;    plot (mean [U_H^b] of turtles)
;;    set-current-plot-pen "U_H^x"
;;    plot (mean [U_H^x] of turtles)
;;    set-current-plot-pen "U_H 1"
;;    plot (min [U_H] of turtles with [stage = 1])
;;    set-current-plot-pen "U_H 2"
;;    plot (mean [U_H] of turtles with [stage = 1])
;;    set-current-plot-pen "U_H 3"
;;    plot (max [U_H] of turtles with [stage = 1])
;;  ]
;;;
;;;  set-current-plot "total mean Us - embryo" ;;--------------------- embryo enery
;;;  if any? turtles with [stage = 0][
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;    set-current-plot-pen "U_H^b"
;;;    plot (mean [U_H^b] of turtles)
;;;    set-current-plot-pen "U_H^x"
;;;    plot (mean [U_H^x] of turtles)
;;;    set-current-plot-pen "U_H 1"
;;;    plot (min [U_H] of turtles with [stage = 0])
;;;    set-current-plot-pen "U_H 2"
;;;    plot (mean [U_H] of turtles with [stage = 0])
;;;    set-current-plot-pen "U_H 3"
;;;    plot (max [U_H] of turtles with [stage = 0])
;;;
;;;  ]
;;;
;;;  set-current-plot "Seals fasting" ;;---------------------------- total juveniles fasting
;;;  set-current-plot-pen "juvenile"
;;;  plot count turtles with [age > 360 and stage < 3 and status = fasting and haul-out? = 0]
;;;  set-current-plot-pen "juv-mwho"
;;;  plot count turtles with [age > 360 and stage < 3 and status = fasting and haul-out? = 1]
;;;  set-current-plot-pen "yearling"
;;;  plot count turtles with [age < 361 and status = fasting and haul-out? = 0]
;;;  set-current-plot-pen "year-mwho"
;;;  plot count turtles with [age < 361 and stage < 3 and status = fasting and haul-out? = 1]
;;;
;;;  set-current-plot "U_R test" ;;------------------------------- reproduction buffer levels seals
;;;  set-current-plot-pen "preg min"
;;;  ifelse any? turtles with [is_pregnant? = 1]
;;;  [ plot min [U_R] of turtles with [is_pregnant? = 1]][ plot 0 ]
;;;  set-current-plot-pen "preg mean"
;;;  ifelse any? turtles with [is_pregnant? = 1]
;;;  [ plot mean [U_R] of turtles with [is_pregnant? = 1]][ plot 0 ]
;;;;  set-current-plot-pen "preg max"
;;;;  ifelse any? turtles with [is_pregnant? = 1]
;;;;  [ plot max [U_R] of turtles with [is_pregnant? = 1]][ plot 0 ]
;;;;  set-current-plot-pen "non-preg min"
;;;;  ifelse any? turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]
;;;;  [plot min [U_R] of turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]][plot 0]
;;;;  set-current-plot-pen "non-preg mean"
;;;;  ifelse any? turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]
;;;;  [plot mean [U_R] of turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]] [plot 0]
;;;;  set-current-plot-pen "non-preg max"
;;;;  ifelse any? turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]
;;;;  [plot max [U_R] of turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]][plot 0]
;;;;  set-current-plot-pen "cum_UE"
;;;;  plot mean [cum_UE] of turtles
;;;  set-current-plot-pen "0.5 of cum_UE"
;;;  plot (mean [0.5 * cum_UE] of turtles)
;;
;;
;;
;;
;;
;;
;;;  set-current-plot "pup e-scaled"
;;;  set-current-plot-pen "pup"
;;;  ifelse any? turtles with [stage = 1]
;;;  [plot mean [e_scaled] of turtles with [stage = 1]][plot 0]
;;;  set-current-plot-pen "adult"
;;;  ifelse any? turtles with [stage = 3]
;;;  [plot mean [e_scaled] of turtles with [stage = 3]][plot 0]
;;;  set-current-plot-pen "1"
;;;  plot 1
;;;
;;
;;;
;;;  set-current-plot "embryo - pup L"
;;;  set-current-plot-pen "embryo"  ;; embryos
;;;  ifelse any? turtles with [stage = 0]
;;;  [ plot max [L / shape_factor] of turtles with [stage = 0] ][ plot 0 ]
;;;  set-current-plot-pen "pup"       ;; pups
;;;  ifelse any? turtles with [stage = 1]
;;;  [ plot max [L / shape_factor] of turtles with [stage = 1] ][ plot 0 ]
;;;  set-current-plot-pen "1"
;;;  plot mean [L_w^x] of turtles
;;;  set-current-plot-pen "0"
;;;  plot mean [L_w^b] of turtles
;;;
;;
;;;  set-current-plot "SC_pup"
;;;  set-current-plot-pen "default"
;;;  ifelse any? turtles with [stage = 1]
;;;  [plot mean [S_C] of turtles with [stage = 1]][plot 0]
;;
;;;  ; ---------------------------------- energy level plotting
;;;  ; --------------------------------------------- delta energy plotting
;;;  set-current-plot "total deltas - adult" ;; adults
;;;  if any? turtles with [stage = 3][
;;;    set-current-plot-pen "dU_E"
;;;    plot (mean [dU_E] of turtles  with [stage = 3])
;;;    set-current-plot-pen "dU_H"
;;;    plot (mean [dU_H] of turtles  with [stage = 3])
;;;    set-current-plot-pen "dU_R"
;;;    plot (mean [dU_R] of turtles  with [stage = 3])
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;  ]
;;;
;;;  set-current-plot "total deltas - juv" ;; juveniles
;;;  if any? turtles with [stage = 2][
;;;    set-current-plot-pen "dU_E"
;;;    plot (mean [dU_E] of turtles  with [stage = 2])
;;;    set-current-plot-pen "dU_H"
;;;    plot (mean [dU_H] of turtles  with [stage = 2])
;;;    set-current-plot-pen "dU_R"
;;;    plot (mean [dU_R] of turtles  with [stage = 2])
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;  ]
;;;  set-current-plot "total deltas - pup" ;; pups
;;;  if any? turtles with [stage = 1][
;;;;    set-current-plot-pen "dU_E"
;;;;    plot (mean [dU_E] of turtles  with [stage = 1])
;;;    set-current-plot-pen "dU_H"
;;;    plot (mean [dU_H] of turtles  with [stage = 1])
;;;    set-current-plot-pen "dU_R"
;;;    plot (mean [dU_R] of turtles  with [stage = 1])
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;  ]
;;;
;;;  set-current-plot "dU_E pup"
;;;  if any? turtles with [stage = 1][
;;;    set-current-plot-pen "default"
;;;    plot (mean [dU_E] of turtles  with [stage = 1])
;;;  ]
;;;;
;;;  set-current-plot "total deltas - embryo" ;; embryos
;;;  if any? turtles with [stage = 0][
;;;    set-current-plot-pen "dU_E"
;;;    plot (mean [dU_E] of turtles  with [stage = 0])
;;;    set-current-plot-pen "dU_H"
;;;    plot (mean [dU_H] of turtles  with [stage = 0])
;;;    set-current-plot-pen "dU_E w kap_F"
;;;    plot (mean [dU_E * kap_F] of turtles  with [stage = 0])
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;  ]
;;
;;;  ;;----------------------------------------------- delta growth plotting
;;;  set-current-plot "dL" ;-------------------------------------------- delta growth
;;;  ifelse any? turtles with [stage = 3] ;; adults
;;;  [ set-current-plot-pen "adult"
;;;    plot (mean [dL] of turtles with [stage = 3])
;;;  ]
;;;  [ set-current-plot-pen "adult"
;;;    plot 0
;;;  ]
;;;  ifelse any? turtles with [stage = 2] ;; juveniles
;;;  [ set-current-plot-pen "juv"
;;;    plot (mean [dL] of turtles with [stage = 2])
;;;  ]
;;;  [ set-current-plot-pen "juv"
;;;    plot 0
;;;  ]
;;;;  ifelse any? turtles with [stage = 1] ;; pups
;;;;  [ set-current-plot-pen "pups"
;;;;    plot (mean [dL] of turtles with [stage = 1])
;;;;  ]
;;;;  [ set-current-plot-pen "pups"
;;;;    plot 0
;;;;  ]
;;;
;;;;  set-current-plot "dL foetus + pups" ;----------------------------- delta growth foetus/pups
;;;;  ifelse any? turtles with [stage = 1] ;; pups
;;;;  [ set-current-plot-pen "pups"
;;;;    plot (mean [dL] of turtles with [stage = 1])
;;;;  ]
;;;;  [ set-current-plot-pen "pups"
;;;;    plot 0
;;;;  ]
;;;;  ifelse any? turtles with [stage = 0] ;; foetus
;;;;  [ set-current-plot-pen "embryo"
;;;;    plot (mean [dL] of turtles with [stage = 0])
;;;;  ]
;;;;  [ set-current-plot-pen "embryo"
;;;;    plot 0
;;;;  ]
;;;
;;;  if any? turtles with [e_scaled > l_scaled][;;------------------non-starvation mode
;;;;    1)
;;;    set-current-plot "normal deltas" ;--------------------------------- normal delta
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;    set-current-plot-pen "dU_E"
;;;    plot (mean [dU_E] of turtles with [e_scaled > l_scaled])
;;;    set-current-plot-pen "dU_H"
;;;    plot (mean [dU_H] of turtles with [e_scaled > l_scaled])
;;;    set-current-plot-pen "dU_R"
;;;    plot (mean [dU_R] of turtles with [e_scaled > l_scaled])
;;;
;;;;    2)
;;;    set-current-plot "normal Us" ;-------------------------------------- normal U
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;    set-current-plot-pen "U_E"
;;;    plot (mean [U_E] of turtles with [e_scaled > l_scaled])
;;;    set-current-plot-pen "U_H"
;;;    plot (mean [U_H] of turtles with [e_scaled > l_scaled])
;;;    set-current-plot-pen "U_R"
;;;    plot (mean [U_R] of turtles with [e_scaled > l_scaled])
;;;  ]
;;;  ;--------------------------------- starvation condition plots
;;;  if any? turtles with [e_scaled < l_scaled][
;;;;    type " count turtles with e_scaled < l_scaled " print count turtles with [e_scaled < l_scaled]
;;;;    1)
;;;    set-current-plot "starvation deltas" ;------------------------------ starvation delta
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;    set-current-plot-pen "dU_E"
;;;    plot (mean [dU_E] of turtles with [e_scaled < l_scaled])
;;;    set-current-plot-pen "dU_H"
;;;    plot (mean [dU_H] of turtles with [e_scaled < l_scaled])
;;;    set-current-plot-pen "dU_R"
;;;    plot (mean [dU_R] of turtles with [e_scaled < l_scaled])
;;;;    2)
;;;    set-current-plot "starvation Us" ;---------------------------------- starvation Us
;;;    set-current-plot-pen "0"
;;;    plot 0
;;;    set-current-plot-pen "U_E"
;;;    plot (mean [U_E] of turtles with [e_scaled < l_scaled])
;;;    set-current-plot-pen "U_H"
;;;    plot (mean [U_H] of turtles with [e_scaled < l_scaled])
;;;    set-current-plot-pen "U_R"
;;;    plot (mean [U_R] of turtles with [e_scaled < l_scaled])
;;;  ]
;;
;;  ;------------------------------ general plots
;;
;;;  set-current-plot "S_C" ;---------------------------------------------- SC
;;;  set-current-plot-pen "0"
;;;  plot 0
;;;  set-current-plot-pen "S_A"
;;;  plot (mean [S_A] of turtles with [stage > 1])
;;;  set-current-plot-pen "S_C"
;;;  plot (mean [S_C] of turtles with [stage > 1])
;;
;;
;;;  ;;-------------------------pregnancy/non-pregnancy U_R checks
;;;  set-current-plot "U_R of pregnant seals"
;;;  set-current-plot-pen "mean U_R"
;;;  ifelse any? turtles with [is_pregnant? = 1]
;;;  [ plot mean [U_R] of turtles with [is_pregnant? = 1]][ plot 0 ]
;;;  set-current-plot-pen "min U_R"
;;;  ifelse any? turtles with [is_pregnant? = 1]
;;;  [ plot min [U_R] of turtles with [is_pregnant? = 1]][ plot 0 ]
;;;  set-current-plot-pen "cum_UE"
;;;  plot mean [cum_UE] of turtles
;;;
;;;  set-current-plot "U_R of non pregnant seals"
;;;  set-current-plot-pen "mean U_R"
;;;  plot mean [U_R] of turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]
;;;  set-current-plot-pen "max U_R"
;;;  plot max [U_R] of turtles with [is_pregnant? = 0 and age > 4 * 360 and age < 15 * 360]
;;;  set-current-plot-pen "cum_UE"
;;;  plot mean [cum_UE] of turtles
;;
;;  ;;------------------------------------------------------- births
;;;  set-current-plot "prop succesful births to breeders" ;--------------------- births of breeders
;;;  set-current-plot-pen "pen-1"
;;;  plot 0.25
;;;  set-current-plot-pen "pen-2"
;;;  plot 0.5
;;;  set-current-plot-pen "pen-3"
;;;  plot 0.75
;;;  set-current-plot-pen "pen-4"
;;;  plot 1
;;;  if count turtles with [stage = 1] / (0.00000000000001 + count turtles with [U_R > cum_UE]) < 2 [
;;;    set-current-plot-pen "default"
;;;    plot (count turtles with [stage = 1] / (0.00000000000001 + count turtles with [U_R > cum_UE]))
;;;  ]
;;;  if count turtles with [stage = 1] / (0.00000000000001 + count turtles with [U_R > cum_UE and age < (15 * 360)]) < 2 [
;;;    set-current-plot-pen "pen-5"
;;;    plot (count turtles with [stage = 1] / (0.00000000000001 + count turtles with [U_R > cum_UE and age < (15 * 360)]))
;;;  ]
;;;
  ;;  set-current-plot "pop dyn" ;; ------------------------- proportion based on age
;;;  set-current-plot-pen "0.5"
;;;  plot 0.5
;;;  set-current-plot-pen "6+"
;;;  plot ((count turtles with [stage > 1 and age > 360 * 6]) / (count turtles + 0.001))
;;;  set-current-plot-pen "pup-6"
;;;  plot ((count turtles with [stage  > 1 and age < 360 * 6]) / (count turtles + 0.001))
;;;  set-current-plot-pen "pup"
;;;  plot ((count turtles with [stage = 1]) / (count turtles + 0.001))
;;;  set-current-plot-pen "embryo"
;;;  plot ((count turtles with [stage = 0]) / (count turtles + 0.001))
;;;
;
;;  set-current-plot "fec_by_age"
;;  if ticks > 50 * 360 [
;;  if any? turtles with [total-puppies > 0][
;;
;;    ifelse any? turtles with [total-puppies > 0 and age > 3 * 360 and age < 4  * 360] [
;;      set-current-plot-pen "pen-2"
;;      plot mean [total-puppies / ((3 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 3 * 360 and age < 4 * 360]
;;    ][set-current-plot-pen "pen-2" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 4 * 360 and age < 5 * 360] [
;;      set-current-plot-pen "pen-3"
;;      plot mean [total-puppies / ((4 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 4 * 360 and age < 5 * 360]
;;    ][set-current-plot-pen "pen-3" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 5 * 360 and age < 6 * 360] [
;;      set-current-plot-pen "pen-4"
;;      plot mean [total-puppies / ((5 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 5 * 360 and age < 6 * 360]
;;    ][set-current-plot-pen "pen-4" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age < 6 * 360 and age < 7 * 360] [
;;      set-current-plot-pen "pen-5"
;;      plot mean [total-puppies / ((6 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 6 * 360 and age < 7 * 360]
;;    ][set-current-plot-pen "pen-5" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 7 * 360 and age < 8  * 360] [
;;      set-current-plot-pen "pen-6"
;;      plot mean [total-puppies / ((7 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 7 * 360 and age < 8 * 360]
;;    ][set-current-plot-pen "pen-6" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 8 * 360 and age < 9 * 360] [
;;      set-current-plot-pen "pen-7"
;;      plot mean [total-puppies / ((8 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 8 * 360 and age < 9 * 360]
;;    ][set-current-plot-pen "pen-7" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 9 * 360 and age < 10 * 360] [
;;      set-current-plot-pen "pen-8"
;;      plot mean [total-puppies / ((9 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 9 * 360 and age < 10 * 360]
;;    ][set-current-plot-pen "pen-8" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 10 * 360 and age < 11 * 360] [
;;      set-current-plot-pen "pen-9"
;;      plot mean [total-puppies / ((10 / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 10 * 360 and age < 11 * 360]
;;    ][set-current-plot-pen "pen-9" plot 0]
;;    ifelse any? turtles with [total-puppies > 0 and age > 11 * 360] [
;;      set-current-plot-pen "pen-10"
;;      plot mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies > 0 and age > 11 * 360]
;;    ][set-current-plot-pen "pen-10" plot 0]
;;
;;  ]
;;  ]
;;
;;  set-current-plot "rate_of_survival"
;;  set-current-plot-pen "pen-1"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age < 1 * 360 and stage > 0] / count turtles with [die-now = 0 and age < 1 * 360 and stage > 0]
;;  set-current-plot-pen "pen-2"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 1 * 360 and age < 2 * 360] / count turtles with [die-now = 0 and age > 1 * 360 and age < 2 * 360]
;;  set-current-plot-pen "pen-3"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 2 * 360 and age < 3 * 360] / count turtles with [die-now = 0 and age > 2 * 360 and age < 3 * 360]
;;  set-current-plot-pen "pen-4"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 3 * 360 and age < 4 * 360] / count turtles with [die-now = 0 and age > 3 * 360 and age < 4 * 360]
;;  set-current-plot-pen "pen-5"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 4 * 360 and age < 5 * 360] / count turtles with [die-now = 0 and age > 4 * 360 and age < 5 * 360]
;;  set-current-plot-pen "pen-6"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 5 * 360 and age < 6 * 360] / count turtles with [die-now = 0 and age > 5 * 360 and age < 6 * 360]
;;  set-current-plot-pen "pen-7"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 6 * 360 and age < 7 * 360] / count turtles with [die-now = 0 and age > 6 * 360 and age < 7 * 360]
;;  set-current-plot-pen "pen-8"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 7 * 360 and age < 8 * 360] / count turtles with [die-now = 0 and age > 7 * 360 and age < 8 * 360]
;;  set-current-plot-pen "pen-9"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 8 * 360 and age < 9 * 360] / count turtles with [die-now = 0 and age > 8 * 360 and age < 9 * 360]
;;  set-current-plot-pen "pen-10"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 19 * 360 and age < 10 * 360] / count turtles with [die-now = 0 and age > 9 * 360 and age < 10 * 360]
;;  set-current-plot-pen "pen-11"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 10 * 360 and age < 11 * 360] / count turtles with [die-now = 0 and age > 10 * 360 and age < 11 * 360]
;;  set-current-plot-pen "pen-12"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 11 * 360 and age < 12 * 360] / count turtles with [die-now = 0 and age > 11 * 360 and age < 12 * 360]
;;  set-current-plot-pen "pen-13"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 12 * 360 and age < 13 * 360] / count turtles with [die-now = 0 and age > 12 * 360 and age < 13 * 360]
;;  set-current-plot-pen "pen-14"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 13 * 360 and age < 14 * 360] / count turtles with [die-now = 0 and age > 13 * 360 and age < 14 * 360]
;;  set-current-plot-pen "pen-15"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 14 * 360 and age < 15 * 360] / count turtles with [die-now = 0 and age > 14 * 360 and age < 15 * 360]
;;  set-current-plot-pen "pen-16"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 15 * 360 and age < 16 * 360] / count turtles with [die-now = 0 and age > 15 * 360 and age < 16 * 360]
;;  set-current-plot-pen "pen-17"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 16 * 360 and age < 17 * 360] / count turtles with [die-now = 0 and age > 16 * 360 and age < 17 * 360]
;;  set-current-plot-pen "pen-18"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 17 * 360 and age < 18 * 360] / count turtles with [die-now = 0 and age > 17 * 360 and age < 18 * 360]
;;  set-current-plot-pen "pen-19"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 18 * 360 and age < 19 * 360] / count turtles with [die-now = 0 and age > 18 * 360 and age < 19 * 360]
;;
;;  ifelse any? turtles with [age > 19 * 360 and age < 20 * 360][
;;  set-current-plot-pen "pen-20"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 19 * 360 and age < 20 * 360] / count turtles with [die-now = 0 and age > 19 * 360 and age < 20 * 360]
;;    ][set-current-plot-pen "pen-20" plot 0]
;;  ifelse any? turtles with [age > 20 * 360 and age < 21 * 360][
;;  set-current-plot-pen "pen-21"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 20 * 360 and age < 21 * 360] / count turtles with [die-now = 0 and age > 20 * 360 and age < 21 * 360]
;;    ][set-current-plot-pen "pen-21" plot 0]
;;  ifelse any? turtles with [age > 21 * 360 and age < 22 * 360][
;;  set-current-plot-pen "pen-22"
;;  plot count turtles with [die-now = 1 and days-dead = 1 and age > 21 * 360 and age < 22 * 360] / count turtles with [die-now = 0 and age > 21 * 360 and age < 22 * 360]
;;  ][set-current-plot-pen "pen-22" plot 0]
;;    ifelse any? turtles with [age > 22 * 360 and age < 23 * 360][
;;    set-current-plot-pen "pen-23"
;;    plot count turtles with [die-now = 1 and days-dead = 1 and age > 22 * 360 and age < 23 * 360] / count turtles with [die-now = 0 and age > 22 * 360 and age < 23 * 360]
;;  ][set-current-plot-pen "pen-23" plot 0]
;;  ifelse any? turtles with [age > 23 * 360 and age < 24 * 360][
;;    set-current-plot-pen "pen-24"
;;    plot count turtles with [die-now = 1 and days-dead = 1 and age > 23 * 360 and age < 24 * 360] / count turtles with [die-now = 0 and age > 23 * 360 and age < 24 * 360]
;;  ][set-current-plot-pen "pen-24" plot 0]
;;  ifelse any? turtles with [age > 24 * 360][
;;    set-current-plot-pen "pen-25"
;;    plot count turtles with [die-now = 1 and days-dead = 1 and age > 24 * 360 and age < 25 * 360] / count turtles with [die-now = 0 and age > 24 * 360 and age < 25 * 360]
;;  ][set-current-plot-pen "pen-25" plot 0]
;;  ifelse any? turtles with [age > 25 * 360][
;;    set-current-plot-pen "pen-26"
;;    plot count turtles with [die-now = 1 and days-dead = 1 and age > 25 * 360] / count turtles with [die-now = 0 and age > 25 * 360]
;;  ][set-current-plot-pen "pen-26" plot 0]
;
;
;
;
;;;  set-current-plot "death reason"
;;;  if any? turtles with [die-now = 1 and days-dead = 1] [
;;;    set-current-plot-pen "1 - low energy"
;;;    ifelse any? turtles with [die-now = 1 and days-dead = 1 and died-of-? = 1]
;;;    [ plot ((count turtles with [die-now = 1 and days-dead = 1 and died-of-? = 1]) / (count turtles with [die-now = 1 and days-dead = 1 ] + 0.001))]
;;;    [ plot 0 ]
;;;    set-current-plot-pen "2 - foetus w mum"
;;;    ifelse any? turtles with [die-now = 1 and days-dead = 1 and died-of-? = 2]
;;;    [ plot ((count turtles with [die-now = 1 and days-dead = 1 and died-of-? = 2])/ (count turtles with [die-now = 1 and days-dead = 1 ] + 0.001))]
;;;    [ plot 0 ]
;;;    set-current-plot-pen "3 - old age"
;;;    ifelse any? turtles with [die-now = 1 and days-dead = 1 and died-of-? = 3]
;;;    [ plot ((count turtles with [die-now = 1 and days-dead = 1 and died-of-? = 3])/ (count turtles with [die-now = 1 and days-dead = 1 ] + 0.001))]
;;;    [ plot 0 ]
;;;    set-current-plot-pen "4 - error"
;;;    ifelse any? turtles with [die-now = 1 and days-dead = 1 and died-of-? = 4]
;;;    [ plot ((count turtles with [die-now = 1 and days-dead = 1 and died-of-? = 4])/ (count turtles with [die-now = 1 and days-dead = 1 ] + 0.001))]
;;;    [ plot 0 ]
;;;    set-current-plot-pen "5 - pup mort"
;;;    ifelse any? turtles with [die-now = 1 and days-dead = 1 and died-of-? = 5]
;;;    [ plot ((count turtles with [die-now = 1 and days-dead = 1 and died-of-? = 5])/ (count turtles with [die-now = 1 and days-dead = 1 ] + 0.001))]
;;;    [ plot 0 ]
;;;  ]
;;
;
;;
;;;  set-current-plot "total mean Us - yearling" ;; -------------------------yearling energy
;;;  if any? turtles with [stage = 2 and age < 350][
;;;    set-current-plot-pen "U_H^p"
;;;    plot (mean [U_H^p] of turtles)
;;;    set-current-plot-pen "U_H^x"
;;;    plot (mean [U_H^x] of turtles)
;;;    set-current-plot-pen "U_H^b"
;;;    plot (mean [U_H^b] of turtles)
;;;    set-current-plot-pen "max UH"
;;;    plot (max [U_H] of turtles with [stage = 2 and age < 350])
;;;    set-current-plot-pen "mean UH"
;;;    plot (mean [U_H] of turtles with [stage = 2 and age < 350])
;;;    set-current-plot-pen "min UH"
;;;    plot (min [U_H] of turtles with [stage = 2 and age < 350])
;;;  ]
;;
;;set-current-plot "competition"
;;  set-current-plot-pen "comb"
;;  if any? turtles with [stage > 1 and age > 360 and status = 2 and f > 0][;-------------------------------------------------------------------- analysis
;;    plot mean [f] of turtles with [stage > 1 and f > 0]
;;  ]
;;  set-current-plot-pen "max juv"
;;  if any? turtles with [stage = 2 and status = 2 and f > 0 and age > 360][
;;    plot max [f] of turtles with [stage = 2 and f > 0] ; status = 2, only while foraging
;;  ]
;;  set-current-plot-pen "mean juv"
;;  if any? turtles with [stage = 2 and status = 2 and f > 0 and age > 360][
;;    plot mean [f] of turtles with [stage = 2 and f > 0] ; status = 2, only while foraging
;;  ]
;;  set-current-plot-pen "min juv"
;;  if any? turtles with [stage = 2 and status = 2 and f > 0 and age > 360][
;;    plot min [f] of turtles with [stage = 2 and f > 0] ; status = 2, only while foraging
;;  ]
;;  set-current-plot-pen "max adult"
;;  if any? turtles with [stage = 3 and status = 2 and f > 0][
;;    plot max [f] of turtles with [stage = 3 and f > 0] ; status = 2, only while foraging
;;  ]
;;  set-current-plot-pen "mean adult"
;;  if any? turtles with [stage = 3 and status = 2 and f > 0][
;;    plot mean [f] of turtles with [stage = 3 and f > 0] ; status = 2, only while foraging
;;  ]
;;  set-current-plot-pen "min adult"
;;  if any? turtles with [stage = 3 and status = 2 and f > 0][
;;    plot min [f] of turtles with [stage = 3 and f > 0] ; status = 2, only while foraging
;;  ]
;
;
;;
;;;  set-current-plot "total count"
;;;  set-current-plot-pen "old"
;;;  plot count turtles with [stage > 1]
;;;  set-current-plot-pen "young"
;;;  plot count turtles with [stage = 1]
;;;
;
;;
;
;
;
;;;;


;
;;
;  set-current-plot "L for export"
;  if any? turtles with [stage > 1][ ;; set levels
;    set-current-plot-pen "Lmax"
;    plot mean [Lmax / shape_factor * f_scaled] of turtles
;    set-current-plot-pen "Lp"
;    plot mean [L_w^p] of turtles
;    set-current-plot-pen "Lx"
;    plot mean [L_w^x] of turtles
;    set-current-plot-pen "Lb"
;    plot mean [L_w^b] of turtles
;  ]
;  set-current-plot-pen "max"
;  ifelse any? turtles
;  [ plot (max [L / shape_factor] of turtles)][ plot 0 ]
;  set-current-plot-pen "mean adult"
;  ifelse any? turtles  with [stage = 3]
;  [ plot (mean [L / shape_factor] of turtles with [stage = 3]) ][ plot 0 ]
;  set-current-plot-pen "min adult"
;  ifelse any? turtles  with [stage = 3]
;  [ plot (min [L / shape_factor] of turtles with [stage = 3]) ][ plot 0 ]
;  set-current-plot-pen "mean juv"
;  ifelse any? turtles  with [stage = 2]
;  [ plot (mean [L / shape_factor] of turtles with [stage = 2]) ][ plot 0]
;  set-current-plot-pen "min juv"
;  ifelse any? turtles  with [stage = 2]
;  [ plot (min [L / shape_factor] of turtles with [stage = 2]) ][ plot 0 ]
;  set-current-plot-pen "max pup"
;  ifelse any? turtles  with [stage = 2]
;  [ plot (mean [L / shape_factor] of turtles with [stage = 2]) ][ plot 0]
;  set-current-plot-pen "min pup"
;  ifelse any? turtles  with [stage = 2]
;  [ plot (min [L / shape_factor] of turtles with [stage = 2]) ][ plot 0 ]
;;
;;;  set-current-plot "L of embryo"
;;;  if any? turtles with [stage = 0][
;;;    set-current-plot-pen "Lx"
;;;    plot mean [L_w^x] of turtles
;;;    set-current-plot-pen "Lb"
;;;    plot mean [L_w^b] of turtles
;;;    set-current-plot-pen "L 1"
;;;    plot min [L / shape_factor] of turtles with [stage = 0]
;;;    set-current-plot-pen "L 2"
;;;    plot mean [L / shape_factor] of turtles with [stage = 0]
;;;    set-current-plot-pen "L 3"
;;;    plot max [L / shape_factor] of turtles with [stage = 0]
;;;  ]
;;
;;
;;;
;;;;;
;;;;;  ;--------------------------------------------------------------------------------
;;;;;  ;--------------------------------------------------------------------------------
;;;;;
;
;;;
;;;
;;;
;;;;  ;;----------------------------------------------------------------------------------------------------
;;;;  ;;---------------------------------------------------------------------------------------------------
;;;;;
;;;  ;---------------------------------- age plotting
;;  set-current-plot "age" ;----------------------------------------------- age
;;  set-current-plot-pen "max"
;;;  plot (max [age] of turtles / 360)
;;;  set-current-plot-pen "mean adult"
;;;  ifelse any? turtles  with [stage = 3]
;;;  [ plot (mean [age] of turtles with [stage = 3] / 360) ][ plot 0 ];-------------------------------------------------------------------- analysis
;;;  set-current-plot-pen "min adult"
;;;  ifelse any? turtles  with [stage = 3]
;;;  [ plot (min [age] of turtles with [stage = 3] / 360) ][ plot 0 ];-------------------------------------------------------------------- analysis
;;;  set-current-plot-pen "mean juv"
;;;  ifelse any? turtles  with [stage = 2]
;;;  [ plot (mean [age] of turtles with [stage = 2] / 360) ][ plot 0]
;;;  set-current-plot-pen "min juv"
;;;  ifelse any? turtles  with [stage = 2]
;;;  [ plot (min [age] of turtles with [stage = 2] / 360) ][ plot 0 ]
;;  set-current-plot-pen "max pup"
;;  ifelse any? turtles  with [stage = 1]
;;  [ plot (mean [age] of turtles with [stage = 1] / 360) ][ plot 0]
;;  set-current-plot-pen "min pup"
;;  ifelse any? turtles  with [stage = 1]
;;  [ plot (min [age] of turtles with [stage = 1] / 360) ][ plot 0 ]
;;;  set-current-plot-pen "23"
;;;  plot 23
;;;  set-current-plot-pen "15"
;;;  plot 15
;;;  set-current-plot-pen "7"
;;;  plot 7
;;;  set-current-plot-pen "4"
;;;  plot 4
;;;  set-current-plot-pen "1"
;;;  plot 1
;;;
;;;;
;
;;;;
;;;;;  set-current-plot "lactating and pregnant?"
;;;;;  ifelse any? turtles with [lactating? = 1 and impregnated? = 1]
;;;;;  [ set-current-plot-pen "yes"
;;;;;    plot 1
;;;;;    plot-pen-down
;;;;;    set-plot-pen-color 127
;;;;;    set-plot-pen-mode 1
;;;;;  ]
;;;;;  [ set-current-plot-pen "yes"
;;;;;    plot 0.5
;;;;;    plot-pen-up
;;;;;    set-plot-pen-color 7
;;;;;  ]
;;;
;;;;



end
@#$#@#$#@
GRAPHICS-WINDOW
1797
4092
1976
4272
-1
-1
171.0
1
10
1
1
1
0
1
1
1
0
0
0
0
1
1
1
ticks
30.0

SLIDER
556
874
673
907
f_scaled
f_scaled
0
1
0.935
.001
1
NIL
HORIZONTAL

BUTTON
152
882
218
915
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
394
879
460
912
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
223
882
289
915
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
718
29
1258
256
total population numbers
NIL
NIL
0.0
10.0
0.0
2.0
true
true
"" ""
PENS
"adult" 1.0 0 -2674135 true "" ""
"juvenile" 1.0 0 -612749 true "" ""
"pup" 1.0 0 -10649926 true "" ""
"embryo" 1.0 0 -6759204 true "" ""
"yearling" 1.0 0 -7500403 true "" ""
"population" 1.0 0 -16777216 true "" ""

MONITOR
476
866
545
911
total days
ticks
0
1
11

PLOT
1562
3304
1815
3496
juv e distribution
NIL
NIL
0.5
1.2
0.0
10.0
true
false
"" ""
PENS
"default" 0.01 1 -16777216 true "" ";histogram [e_scaled] of turtles with [U_H > U_H^b and U_H < U_H^p]"

PLOT
1707
3289
1971
3482
adult e distribution
NIL
NIL
0.5
1.2
0.0
10.0
true
false
"" ""
PENS
"e" 0.001 1 -14439633 true "" ";histogram [e_scaled] of turtles with [U_H >= U_H^p]"

INPUTBOX
299
754
388
814
cv
0.05
1
0
Number

INPUTBOX
128
688
226
748
shape_factor
0.235
1
0
Number

PLOT
283
3744
576
3932
starvation Us
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_E" 1.0 0 -16777216 true "" ""
"U_H" 1.0 0 -5298144 true "" ""
"U_R" 1.0 0 -1264960 true "" ""
"0" 1.0 0 -8990512 true "" ""

PLOT
283
3547
578
3743
starvation deltas
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dU_E" 1.0 0 -16777216 true "" ""
"dU_H" 1.0 0 -5298144 true "" ""
"dU_R" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -11221820 true "" ""

PLOT
13
3744
284
3930
normal Us
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_E" 1.0 0 -16777216 true "" ""
"U_H" 1.0 0 -5298144 true "" ""
"U_R" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -6759204 true "" ""

PLOT
9
3547
286
3741
normal deltas
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dU_E" 1.0 0 -16777216 true "" ""
"dU_H" 1.0 0 -5298144 true "" ""
"dU_R" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -8990512 true "" ""

PLOT
1169
3539
1492
3686
dL
NIL
cm
0.0
10.0
0.0
0.0
true
true
"" ""
PENS
"pups" 1.0 0 -14454117 true "" ""
"juv" 1.0 0 -955883 true "" ""
"adult" 1.0 0 -2674135 true "" ""

PLOT
42
2014
630
2143
g and e
NIL
NIL
0.0
10.0
0.6
0.8
true
true
"" ""
PENS
"e" 1.0 0 -14454117 true "" ""
"l" 1.0 0 -4699768 true "" ""
"ge" 1.0 0 -14439633 true "" ""

MONITOR
594
1412
662
1457
ge/g+e
[(g * e_scaled) / (g + e_scaled)] of turtle track-turtle
2
1
11

INPUTBOX
346
917
460
977
start_population
250.0
1
0
Number

MONITOR
375
979
460
1024
total pop
count turtles
0
1
11

PLOT
1096
3105
1256
3319
S_C
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"S_C" 1.0 0 -13345367 true "" ""
"0" 1.0 0 -16777216 true "" ""
"S_A" 1.0 2 -2064490 true "" ""

MONITOR
1139
3256
1203
3301
S_C
mean [S_C] of turtles
2
1
11

MONITOR
594
1457
662
1502
e_scaled
[e_scaled] of turtle track-turtle
2
1
11

TEXTBOX
1418
3127
1571
3169
if e_scaled < l_scaled then starvation model will be implemented.
11
0.0
1

MONITOR
1263
3105
1402
3150
adults in starvation mode
count turtles with [e_scaled < l_scaled and stage = 3]
0
1
11

MONITOR
1263
3150
1400
3195
juv in starvation mode
count turtles with [e_scaled < l_scaled and stage = 2]
0
1
11

PLOT
1234
519
1673
759
total mean Us - juv
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_H" 1.0 0 -2674135 true "" ""
"U_R-high" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -8990512 true "" ""
"U_H^p" 1.0 0 -16777216 true "" ""
"U_H^x" 1.0 0 -16777216 true "" ""
"U_R-low" 1.0 0 -14439633 true "" ""
"cum_UE" 1.0 0 -4079321 true "" ""

PLOT
1512
3767
1762
3887
total deltas - adult
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dU_E" 1.0 2 -16777216 true "" ""
"dU_H" 1.0 0 -2674135 true "" ""
"dU_R" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -8990512 true "" ""

PLOT
1559
3120
1816
3304
juv l distribution
NIL
NIL
0.5
1.2
0.0
10.0
true
false
"" ""
PENS
"default" 0.01 1 -16777216 true "" ";histogram [l_scaled] of turtles with [U_H < U_H^p]"

PLOT
723
519
1231
758
L
NIL
NIL
0.0
10.0
20.0
40.0
true
true
"" ""
PENS
"mean ad" 1.0 0 -5298144 true "" ""
"mean juv" 1.0 0 -612749 true "" ""
"mean pup" 1.0 0 -10649926 true "" ""
"mean embryo" 1.0 0 -11221820 true "" ""
"Lmax" 1.0 0 -16777216 true "" ""
"Lp" 1.0 0 -16777216 true "" ""
"Lx" 1.0 0 -16777216 true "" ""
"Lb" 1.0 0 -16777216 true "" ""
"max ad" 1.0 0 -8053223 true "" ""
"min ad" 1.0 0 -1604481 true "" ""
"max juv" 1.0 0 -14070903 true "" ""
"min juv" 1.0 0 -8020277 true "" ""

MONITOR
547
914
632
959
NIL
day-of-year
0
1
11

MONITOR
576
962
670
1007
NIL
day-of-month
0
1
11

MONITOR
475
962
573
1007
NIL
month-of-year
0
1
11

MONITOR
476
914
546
959
NIL
year
0
1
11

PLOT
1707
3106
1971
3291
adult l distribution
NIL
NIL
0.5
1.2
0.0
10.0
true
false
"" ""
PENS
"default" 0.001 1 -14439633 true "" ";histogram [l_scaled] of turtles with [U_H >= U_H^p]"

PLOT
1698
28
2051
258
count
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"total" 1.0 0 -16777216 true "" ""
"K-pop" 1.0 0 -7500403 true "" ""
"breeders" 1.0 0 -5298144 true "" ""
"pups" 1.0 0 -13345367 true "" ""
"K" 1.0 0 -15637942 true "" ""

OUTPUT
602
3685
852
3725
7

SWITCH
35
880
146
913
use_profiler?
use_profiler?
1
1
-1000

SLIDER
198
429
352
462
prob_mate_fail_6plus
prob_mate_fail_6plus
0
1
0.75
0.01
1
NIL
HORIZONTAL

INPUTBOX
32
515
127
575
L_w^b
110.0
1
0
Number

TEXTBOX
34
757
228
831
Standard seal parameters at birth (b),\nweaning (x), puberty (p) and ultimate (m)\nL_w = physical length (cm)\nE_H = maturity (J)\nshape_factor (del_M) converts physical\nlength to structural length (L)
10
0.0
1

INPUTBOX
32
575
127
635
L_w^x
125.0
1
0
Number

INPUTBOX
32
630
127
690
L_w^p
180.0
1
0
Number

INPUTBOX
32
689
127
749
L_w^m
280.0
1
0
Number

INPUTBOX
128
514
226
574
E_H^b_init
2.81E7
1
0
Number

INPUTBOX
128
574
226
634
E_H^x_init
6.5E7
1
0
Number

INPUTBOX
128
629
226
689
E_H^p_init
1.45E8
1
0
Number

INPUTBOX
244
514
343
574
p_am
968.2785
1
0
Number

TEXTBOX
440
680
685
779
p_am      ; max specific assimilation rate (J/d cm^2)\ng            ; - , energy investment ratio \nv_rate     ; cm /t , energy conductance (velocity)\nkap         ; - , allocation fraction to soma\nkap_F     ; - , foetal allocation \nkap_L      ; -, lactation allocation \nk_M_rate ; 1/t, somatic maintenance rate coefficient\nk_J_rate  ; 1/t, maturity maintenance rate coefficient
10
0.0
1

INPUTBOX
244
570
343
630
g_init
0.7138
1
0
Number

INPUTBOX
244
629
343
689
v_rate
0.065
1
0
Number

INPUTBOX
244
687
343
747
kap
0.74
1
0
Number

INPUTBOX
344
629
431
689
k_M_rate
0.0014
1
0
Number

INPUTBOX
344
688
431
748
k_J_rate
0.002
1
0
Number

TEXTBOX
440
780
676
838
h_a   ; t^-2, Weibull ageing acceleration\ns_G   ; - , Gompertz stress coefficient\nmortality-float sets a random value for ageing limit (smaller = younger deaths)
10
0.0
1

INPUTBOX
442
515
597
575
h_a
6.0E-10
1
0
Number

INPUTBOX
442
574
597
634
s_G
0.1
1
0
Number

TEXTBOX
606
3518
863
3558
J_XAm_rate  ; surface-area-specific maximum ingestion rate # / (cm^2 t)\nF_m             ; max specific searching rate (1/d cm^2)
10
0.0
1

INPUTBOX
602
3555
757
3615
J_XAm_rate_init
0.0023
1
0
Number

INPUTBOX
602
3615
757
3675
F_m
6.5
1
0
Number

TEXTBOX
762
3559
912
3615
<< These will only be used if there is a variable prey field included. Used for calculation of K (half saturation constant)
11
114.0
1

INPUTBOX
40
208
195
268
moult-duration-juv
26.0
1
0
Number

INPUTBOX
40
269
195
329
resting-duration-juv
2.0
1
0
Number

INPUTBOX
39
332
194
392
forage-duration-juv
60.0
1
0
Number

INPUTBOX
204
208
359
268
moult-duration-adult
30.0
1
0
Number

INPUTBOX
204
269
359
329
resting-duration-adult
1.0
1
0
Number

INPUTBOX
204
332
359
392
forage-duration-adult
98.0
1
0
Number

INPUTBOX
360
208
515
268
diapause
120.0
1
0
Number

INPUTBOX
360
332
515
392
weaning-duration
23.0
1
0
Number

INPUTBOX
360
269
515
329
breeding-duration
217.0
1
0
Number

INPUTBOX
524
208
687
268
moult-duration-pup
50.0
1
0
Number

PLOT
1509
3907
1888
4126
total deltas - pup
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dU_E" 1.0 0 -16777216 true "" ""
"dU_H" 1.0 0 -2674135 true "" ""
"dU_R" 1.0 0 -2064490 true "" ""
"0" 1.0 0 -11221820 true "" ""

PLOT
1268
3364
1520
3534
total deltas - embryo
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dU_E" 1.0 0 -16777216 true "" ""
"dU_H" 1.0 0 -2674135 true "" ""
"dU_E w kap_F" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -8990512 true "" ""

PLOT
24
2847
376
3087
total mean Us - pup
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_H^x" 1.0 0 -16777216 true "" ""
"U_H^b" 1.0 0 -16777216 true "" ""
"U_H 1" 1.0 2 -8053223 true "" ""
"U_H 2" 1.0 2 -2139308 true "" ""
"U_H 3" 1.0 2 -1069655 true "" ""

PLOT
36
3344
305
3528
total mean Us - embryo
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"0" 1.0 0 -8990512 true "" ""
"U_H^x" 1.0 0 -16777216 true "" ""
"U_H^b" 1.0 0 -16777216 true "" ""
"U_H 1" 1.0 0 -5298144 true "" ""
"U_H 2" 1.0 0 -2139308 true "" ""
"U_H 3" 1.0 0 -1069655 true "" ""

INPUTBOX
344
570
431
630
kap_L
0.02
1
0
Number

INPUTBOX
344
515
431
575
kap_F
0.725
1
0
Number

SLIDER
39
396
193
429
prob_mate_fail_3yo
prob_mate_fail_3yo
0
1
0.98
0.01
1
NIL
HORIZONTAL

SLIDER
39
429
193
462
prob_mate_fail_4yo
prob_mate_fail_4yo
0
1
0.21
0.01
1
NIL
HORIZONTAL

SLIDER
198
396
352
429
prob_mate_fail_5yo
prob_mate_fail_5yo
0
1
0.15
0.01
1
NIL
HORIZONTAL

PLOT
826
2924
1140
3082
age
NIL
NIL
0.0
0.0
0.0
0.0
true
true
"" ""
PENS
"max" 1.0 0 -10873583 true "" ""
"mean adult" 1.0 0 -3844592 true "" ""
"min adult" 1.0 0 -817084 true "" ""
"mean juv" 1.0 0 -14439633 true "" ""
"min juv" 1.0 0 -8732573 true "" ""
"max pup" 1.0 0 -14454117 true "" ""
"min pup" 1.0 0 -10649926 true "" ""
"23" 1.0 0 -16777216 true "" ""
"15" 1.0 0 -16777216 true "" ""
"7" 1.0 0 -16777216 true "" ""
"4" 1.0 0 -16777216 true "" ""
"1" 1.0 0 -16777216 true "" ""

PLOT
1670
519
2052
759
total mean Us - adult
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_H" 1.0 0 -2674135 true "" ""
"U_R-high" 1.0 0 -1664597 true "" ""
"0" 1.0 0 -11221820 true "" ""
"U_H^p" 1.0 0 -16777216 true "" ""
"U_R-low" 1.0 0 -14439633 true "" ""
"cum_UE" 1.0 0 -4079321 true "" ""

SLIDER
442
637
596
670
mortality-float
mortality-float
0
0.2
0.013
0.001
1
NIL
HORIZONTAL

PLOT
1513
3620
1761
3767
total deltas - juv
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"dU_E" 1.0 2 -16777216 true "" ""
"dU_H" 1.0 0 -5298144 true "" ""
"dU_R" 1.0 0 -2064490 true "" ""
"0" 1.0 0 -11221820 true "" ""

PLOT
1258
28
1698
257
population dynamics
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"adult" 1.0 0 -5298144 true "" ""
"juvenile" 1.0 0 -955883 true "" ""
"pup" 1.0 0 -13345367 true "" ""
"embryo" 1.0 0 -8990512 true "" ""
"0.5" 1.0 0 -3026479 true "" ""
"yearling" 1.0 0 -11053225 true "" ""

PLOT
1667
3490
1973
3713
prop succesful births to breeders
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"pen-1" 1.0 0 -4539718 true "" ""
"pen-2" 1.0 0 -3026479 true "" ""
"pen-3" 1.0 0 -3026479 true "" ""
"pen-4" 1.0 0 -3026479 true "" ""
"pen-5" 1.0 0 -1604481 true "" ""

PLOT
1010
256
1524
486
test pups survival
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"embryo" 1.0 0 -6759204 true "" ""
"pup" 1.0 0 -14070903 true "" ""
"yearling" 1.0 2 -612749 true "" ""

PLOT
42
1864
685
2012
Seal energy use
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_H" 1.0 0 -2674135 true "" ""
"U_R" 1.0 0 -14070903 true "" ""
"U_H^p" 1.0 0 -612749 true "" ""
"U_H^x" 1.0 0 -8330359 true "" ""
"U_H^b" 1.0 0 -1264960 true "" ""
"breeding level" 1.0 0 -5987164 true "" ""
"0" 1.0 0 -16777216 true "" ""

PLOT
42
1577
581
1697
Seal energy intake
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

TEXTBOX
1739
3202
1906
3222
plotting blocked out within plot
11
0.0
1

TEXTBOX
1754
3394
1921
3414
plotting blocked out within plot
11
0.0
1

SLIDER
525
374
688
407
mean-pup-mort
mean-pup-mort
0
0.002
9.456E-4
0.000001
1
NIL
HORIZONTAL

PLOT
612
3940
947
4270
Seals fasting
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"yearling" 1.0 0 -13791810 true "" ""
"juvenile" 1.0 0 -955883 true "" ""
"juv-mwho" 1.0 0 -3508570 true "" ""
"year-mwho" 1.0 0 -10141563 true "" ""

INPUTBOX
524
269
686
329
mwho-juv
15.0
1
0
Number

TEXTBOX
570
303
675
322
mid-winter-haul-out
10
0.0
1

PLOT
729
790
987
1085
pregnancy ages
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"max preg" 1.0 0 -15040220 true "" ""
"mean preg" 1.0 2 -13840069 true "" ""
"min preg" 1.0 0 -8330359 true "" ""
"25y" 1.0 0 -9276814 true "" ""
"20y" 1.0 0 -7500403 true "" ""
"15y" 1.0 0 -2674135 true "" ""
"10y" 1.0 0 -4539718 true "" ""
"4y" 1.0 0 -14985354 true "" ""
"first-preg" 1.0 0 -955883 true "" ""
"5y" 1.0 0 -16777216 true "" ""

PLOT
1072
1219
1425
1515
seals older than 22 yo
NIL
NIL
0.0
10.0
0.0
5.0
true
false
"" ""
PENS
"count" 1.0 0 -16777216 true "" ""

MONITOR
1106
1239
1156
1284
>22yo
count turtles with [age > (22 * 360)]
0
1
11

PLOT
915
3702
1222
3932
U_R of pregnant seals
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"mean U_R" 1.0 0 -4699768 true "" ""
"min U_R" 1.0 0 -865067 true "" ""
"cumm_UE" 1.0 0 -612749 true "" ""

PLOT
596
3704
912
3934
U_R of non pregnant seals
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"mean U_R" 1.0 0 -7500403 true "" ""
"cumm_UE" 1.0 0 -612749 true "" ""
"max U_R" 1.0 0 -11033397 true "" ""

INPUTBOX
248
917
343
977
clear-plots
50.0
1
0
Number

TEXTBOX
43
168
702
187
====================================================================================
11
0.0
1

TEXTBOX
40
183
266
202
IBM settings - references are found in code
11
2.0
1

TEXTBOX
33
480
697
499
====================================================================================
11
0.0
1

TEXTBOX
34
496
277
515
DEB parameters - obtained through DEBtool
11
2.0
1

TEXTBOX
823
16
1910
35
============================================================ POPULATION DYNAMICS ============================================================
11
0.0
1

TEXTBOX
838
496
1933
525
============================================================ ENERGY & SIZE ============================================================
11
0.0
1

TEXTBOX
707
12
725
1370
|| || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || || ||
11
0.0
1

TEXTBOX
734
765
1703
785
=================================================== REPRODUCTIVE RELATED =====================================================
11
0.0
1

TEXTBOX
33
842
691
861
====================================================================================
11
0.0
1

TEXTBOX
39
860
411
879
Setup and general model settings and outcomes
11
2.0
1

TEXTBOX
27
1327
692
1346
====================================================================================
11
0.0
1

TEXTBOX
17
2616
1829
2636
==============================================================================================================EXTRA PLOTS ===========================================================================================
11
0.0
1

PLOT
9
3940
607
4275
U_R test
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"preg min" 1.0 0 -1191199 true "" ""
"preg mean" 1.0 0 -1664597 true "" ""
"preg max" 1.0 0 -7858858 true "" ""
"non-preg min" 1.0 0 -2754856 true "" ""
"non-preg mean" 1.0 0 -8732573 true "" ""
"non-preg max" 1.0 0 -12087248 true "" ""
"cumm_UE" 1.0 0 -955883 true "" ""
"0.5 of cumm_UE" 1.0 0 -4079321 true "" ""

PLOT
454
3109
804
3339
total count
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"old" 1.0 0 -16777216 true "" ""
"young" 1.0 0 -7500403 true "" ""

MONITOR
293
869
385
914
seed
my-seed
0
1
11

PLOT
1666
3717
1979
3870
embryo - pup L
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"embryo" 1.0 0 -14070903 true "" ""
"pup" 1.0 0 -3425830 true "" ""
"0" 1.0 0 -4528153 true "" ""
"1" 1.0 0 -1069655 true "" ""

PLOT
985
3316
1255
3526
pup e-scaled
NIL
NIL
0.0
10.0
0.0
2.0
true
false
"" ""
PENS
"pup" 1.0 0 -16777216 true "" ""
"adult" 1.0 0 -7500403 true "" ""
"1" 1.0 0 -2674135 true "" ""

PLOT
1246
3906
1512
4126
dU_E pup
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
825
3105
1098
3319
SC_pup
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

SLIDER
525
338
689
371
max-pup-mort
max-pup-mort
0
0.002
0.00128
0.000001
1
NIL
HORIZONTAL

SLIDER
525
409
689
442
min-pup-mort
min-pup-mort
0
0.002
6.11E-4
0.000001
1
NIL
HORIZONTAL

CHOOSER
365
433
518
478
pup-mort
pup-mort
"minimum" "mean" "maximum" "test"
1

PLOT
964
3979
1217
4133
dead seals
NIL
NIL
0.0
10.0
0.0
5.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"1" 1.0 0 -7500403 true "" "plot 1"

PLOT
1724
792
2044
1202
final pups for seals
NIL
NIL
0.0
10.0
0.0
5.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"only breeders - max" 1.0 2 -14454117 true "" ""
"only breeders - mean" 1.0 0 -2674135 true "" ""
"mum-aff = 1" 1.0 2 -5825686 true "" ""
"mum-aff = 0" 1.0 2 -955883 true "" ""

PLOT
965
4136
1217
4273
breeding age
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"max" 1.0 0 -16777216 true "" ""
"mean" 1.0 0 -7500403 true "" ""
"min" 1.0 0 -3026479 true "" ""

PLOT
1219
4134
1578
4272
reproductive period
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
988
790
1377
1082
max pups over max reproductive period
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"period" 1.0 2 -11053225 true "" ""
"pups" 1.0 1 -955883 true "" ""

PLOT
1378
790
1723
1082
fecundity
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"max" 1.0 2 -5298144 true "" ""
"mean" 1.0 2 -13345367 true "" ""
"min" 1.0 2 -11085214 true "" ""

INPUTBOX
30
1169
121
1229
max-pop-1
1000.0
1
0
Number

PLOT
1644
2759
1804
2879
competition
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"1" 1.0 0 -16777216 true "" "plot 1"
"0.8" 1.0 0 -16777216 true "" "plot 0.8"
"0.5" 1.0 0 -16777216 true "" "plot 0.5"
"max adult" 1.0 0 -10141563 true "" ""
"mean adult" 1.0 0 -6917194 true "" ""
"min adult" 1.0 0 -3425830 true "" ""
"max juv" 1.0 0 -3844592 true "" ""
"mean juv" 1.0 0 -817084 true "" ""
"min juv" 1.0 0 -408670 true "" ""

TEXTBOX
29
1110
243
1129
Density dependence
11
13.0
1

INPUTBOX
28
1232
120
1292
max-pop-2
700.0
1
0
Number

INPUTBOX
214
1230
339
1290
implement-change-at
70.0
1
0
Number

MONITOR
79
1054
162
1103
K
carrying_capacity
0
1
12

TEXTBOX
270
817
420
835
cv = set individual variability
11
0.0
1

INPUTBOX
124
1169
211
1229
time-for-decline
60.0
1
0
Number

TEXTBOX
157
1199
197
1218
(years)
11
0.0
1

MONITOR
160
982
244
1027
pop tracked
count turtles with [stage > 1]
0
1
11

TEXTBOX
94
989
161
1021
(turtles with [stage > 1])
11
0.0
1

INPUTBOX
124
1232
212
1292
period-in-K
2.0
1
0
Number

SWITCH
1339
3254
1495
3287
change-pup-mort
change-pup-mort
1
1
-1000

CHOOSER
1338
3289
1495
3334
new-pup-mortality
new-pup-mortality
"minimum" "mean" "maximum" "test"
2

MONITOR
287
980
366
1025
latest who
max [who] of turtles
0
1
11

INPUTBOX
592
1352
670
1412
track-turtle
6000.0
1
0
Number

PLOT
1539
3500
1699
3620
lactating and pregnant?
NIL
NIL
0.0
10.0
0.0
1.5
true
false
"" ""
PENS
"yes" 1.0 0 -465430 true "" ""

PLOT
42
1697
581
1861
pregnant?
NIL
NIL
0.0
10.0
0.0
1.5
true
false
"" ""
PENS
"preg" 1.0 0 -16777216 true "" ""
"age" 1.0 0 -5825686 true "" ""

PLOT
380
2849
820
3079
pop dyn
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"6+" 1.0 0 -8053223 true "" ""
"pup-6" 1.0 0 -3844592 true "" ""
"pup" 1.0 0 -14730904 true "" ""
"embryo" 1.0 0 -11221820 true "" ""
"0.5" 1.0 0 -4539718 true "" ""

PLOT
303
3345
510
3479
weaning
NIL
NIL
0.0
10.0
0.0
30.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"23 d" 1.0 0 -7500403 true "" "plot 23"

PLOT
42
1342
580
1462
f of seal
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"1" 1.0 0 -7500403 true "" ""

MONITOR
594
1502
662
1547
cv
[iv] of turtle track-turtle
5
1
11

MONITOR
594
1548
663
1593
age
[age / 360] of turtle track-turtle
2
1
11

PLOT
718
257
1012
485
ind var
NIL
NIL
0.8
1.2
0.0
10.0
true
true
"" ""
PENS
"juv" 0.001 1 -9276814 true "" "histogram [iv] of turtles with [stage = 2]"
"ad" 0.001 1 -4699768 true "" "histogram [iv] of turtles with [stage = 3]"
"pup" 0.001 1 -14070903 true "" "histogram [iv] of turtles with [stage = 1]"
"emb" 0.001 1 -8275240 true "" "histogram [iv] of turtles with [stage = 0]"

PLOT
1428
1219
1781
1516
old juveniles
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"old juveniles" 1.0 0 -16777216 true "" "plot count turtles with [stage = 2 and age > 360 * 15]"

TEXTBOX
817
1207
1799
1227
=========================================================== AGE RELATED ========================================================
11
0.0
1

PLOT
844
3534
1161
3687
U of preg
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"U_H^p" 1.0 0 -16777216 true "" ""
"U_H^x" 1.0 0 -16777216 true "" ""
"0" 1.0 0 -6759204 true "" ""
"U_H" 1.0 0 -2674135 true "" ""
"U_R" 1.0 0 -4699768 true "" ""
"cumm_UE" 1.0 0 -4079321 true "" ""

PLOT
122
3112
454
3232
total puppies
NIL
NIL
0.0
15.0
0.0
15.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "histogram [total-puppies] of turtles with [total-puppies > 0]"

PLOT
509
3344
781
3513
L of embryo
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Lb" 1.0 0 -16777216 true "" ""
"Lx" 1.0 0 -16777216 true "" ""
"L 1" 1.0 0 -13345367 true "" ""
"L 2" 1.0 0 -10649926 true "" ""
"L 3" 1.0 0 -5325092 true "" ""

PLOT
122
2682
400
2802
L for export
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"max" 1.0 0 -8053223 true "" ""
"mean adult" 1.0 0 -3844592 true "" ""
"min adult" 1.0 0 -955883 true "" ""
"mean juv" 1.0 0 -12087248 true "" ""
"min juv" 1.0 0 -11085214 true "" ""
"max pup" 1.0 0 -14070903 true "" ""
"min pup" 1.0 0 -10649926 true "" ""
"Lmax" 1.0 0 -16777216 true "" ""
"Lp" 1.0 0 -16777216 true "" ""
"Lx" 1.0 0 -16777216 true "" ""
"Lb" 1.0 0 -16777216 true "" ""

PLOT
716
1218
1070
1514
final age
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"24" 1.0 0 -7500403 true "" ""

PLOT
42
1459
580
1579
seal size
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
1396
1082
1725
1202
min fecundity
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -13840069 true "" ""

PLOT
1050
1083
1398
1203
mean fecundity
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -14070903 true "" ""

PLOT
728
1084
1052
1204
max fecundity
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -5298144 true "" ""

SWITCH
216
1085
353
1118
decadal_change?
decadal_change?
1
1
-1000

SWITCH
30
1132
182
1165
food_change?
food_change?
1
1
-1000

CHOOSER
216
1120
354
1165
change_freq
change_freq
"SAM" "ENSO" "decadal" "random"
1

TEXTBOX
358
1102
525
1122
Change in pup energy intake
11
13.0
1

SWITCH
357
1120
516
1153
pup_energy_change?
pup_energy_change?
1
1
-1000

TEXTBOX
267
1262
304
1282
(years)
11
0.0
1

CHOOSER
359
1159
516
1204
percent_of_pup_change
percent_of_pup_change
"five" "ten" "twenty" "thirty" "fourty" "fifty" "sixty" "eighty"
5

CHOOSER
359
1207
516
1252
percent_of_energy_change
percent_of_energy_change
"five" "ten" "twenty" "thirty" "fourty" "fifty" "sixty" "eighty"
5

INPUTBOX
163
917
247
977
max-years
170.0
1
0
Number

SLIDER
359
1256
515
1289
age_for_pup_energy_change
age_for_pup_energy_change
0
4
1.0
0.1
1
NIL
HORIZONTAL

PLOT
997
1550
1291
1759
pup affected plot
NIL
NIL
0.0
1.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" ""
"0.5" 1.0 0 -7500403 true "" ""

SWITCH
364
396
518
429
pup_mort?
pup_mort?
0
1
-1000

PLOT
274
3165
621
3310
total mean Us - yearling
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"U_H^p" 1.0 0 -11053225 true "" ""
"U_H^x" 1.0 0 -11053225 true "" ""
"U_H^b" 1.0 0 -11053225 true "" ""
"max UH" 1.0 2 -7858858 true "" ""
"mean UH" 1.0 0 -4699768 true "" ""
"min UH" 1.0 2 -3508570 true "" ""

PLOT
503
2640
886
2844
death reason
NIL
NIL
0.0
0.0
0.0
0.0
true
true
"" ""
PENS
"1 - low energy" 1.0 0 -5298144 true "" ""
"2 - foetus w mum" 1.0 0 -955883 true "" ""
"3 - old age" 1.0 2 -12087248 true "" ""
"4 - error" 1.0 2 -3508570 true "" ""
"5 - pup mort" 1.0 0 -11033397 true "" ""

TEXTBOX
534
1080
631
1108
Change fecundity
11
13.0
1

SWITCH
524
1108
680
1141
mother_fecundity_change?
mother_fecundity_change?
1
1
-1000

CHOOSER
525
1145
680
1190
percent_of_mum_change
percent_of_mum_change
"five" "ten" "twenty" "fifty" "eighty" "hundred"
4

CHOOSER
524
1244
682
1289
percent_of_fec_change
percent_of_fec_change
"five" "ten" "twenty" "fifty" "eighty"
4

CHOOSER
525
1193
680
1238
increase?
increase?
"yes" "no"
0

MONITOR
594
1600
660
1645
mum-aff?
[mum-affected] of turtle track-turtle
0
1
11

INPUTBOX
216
1167
355
1227
rv
0.744
1
0
Number

PLOT
702
1552
994
1761
climate change?
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot climate-change"

PLOT
1524
256
2052
482
competition combined
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"comb" 1.0 0 -8053223 true "" ""
"1" 1.0 0 -16777216 true "" "plot 1"
"0.8" 1.0 0 -16777216 true "" "plot 0.8"

PLOT
702
1768
998
1972
clim-var
NIL
NIL
0.0
10.0
0.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot clim-var - 1"

PLOT
1187
2647
1405
2767
fec_by_age
NIL
NIL
0.0
10.0
0.0
0.5
true
true
"" ""
PENS
"pen-1" 1.0 0 -7500403 true "" ""
"pen-2" 1.0 0 -2674135 true "" ""
"pen-3" 1.0 0 -955883 true "" ""
"pen-4" 1.0 0 -1184463 true "" ""
"pen-5" 1.0 0 -10899396 true "" ""
"pen-6" 1.0 0 -13840069 true "" ""
"pen-7" 1.0 0 -14835848 true "" ""
"pen-8" 1.0 0 -11221820 true "" ""
"pen-9" 1.0 0 -13791810 true "" ""
"pen-10" 1.0 0 -13345367 true "" ""
"pen-11" 1.0 0 -8630108 true "" ""
"pen-12" 1.0 0 -5825686 true "" ""
"pen-13" 1.0 0 -2064490 true "" ""

PLOT
1187
2770
1405
2890
rate_of_survival
NIL
NIL
0.0
10.0
0.0
0.001
true
true
"" ""
PENS
"pen-1" 1.0 0 -2674135 true "" ""
"pen-2" 1.0 0 -955883 true "" ""
"pen-3" 1.0 0 -6459832 true "" ""
"pen-4" 1.0 0 -1184463 true "" ""
"pen-5" 1.0 0 -10899396 true "" ""
"pen-6" 1.0 0 -13840069 true "" ""
"pen-7" 1.0 0 -14835848 true "" ""
"pen-8" 1.0 0 -11221820 true "" ""
"pen-9" 1.0 0 -13791810 true "" ""
"pen-10" 1.0 0 -13345367 true "" ""
"pen-11" 1.0 0 -8630108 true "" ""
"pen-12" 1.0 0 -5825686 true "" ""
"pen-13" 1.0 0 -2064490 true "" ""
"pen-14" 1.0 0 -4539718 true "" ""
"pen-15" 1.0 0 -16777216 true "" ""
"pen-16" 1.0 0 -16777216 true "" ""
"pen-17" 1.0 0 -16777216 true "" ""
"pen-18" 1.0 0 -16777216 true "" ""
"pen-19" 1.0 0 -16777216 true "" ""
"pen-20" 1.0 0 -16777216 true "" ""
"pen-21" 1.0 0 -16777216 true "" ""
"pen-22" 1.0 0 -16777216 true "" ""
"pen-23" 1.0 0 -16777216 true "" ""
"pen-24" 1.0 0 -16777216 true "" ""
"pen-25" 1.0 0 -16777216 true "" ""
"pen-26" 1.0 0 -16777216 true "" ""
"pen-27" 1.0 0 -16777216 true "" ""

PLOT
1297
1552
1845
1984
lifetable
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""
"pen-1" 1.0 0 -2139308 true "" ""

TEXTBOX
55
12
638
168
NB. \nUse ctrl + '-' to zoom out of the interface; take care scrolling over the interface with the mouse pointer as the sliders of the model are sensitive to movement. \nInformation on the model is provided in the Info tab, and comments are included in the code.\nPlots can be edited in the 'do-plots' section of the code although some histograms are edited in the plot directly on the interface.
16
14.0
1

TEXTBOX
220
1067
387
1087
Climate variation
11
13.0
1

TEXTBOX
722
1530
1848
1549
============================================================== SCENARIOS AND LIFE TABLE ==============================================================
11
0.0
1

@#$#@#$#@
# MODEL DESCRIPTION

This model is an extension of the DEB-IBM (Individual-based model incorporated with a dynamic energy budget) for southern elephant seals presented in Goedegebuure et al. (2018 PLOS ONE; Modelling southern elephant seals Mirounga leonina using an individual-based model coupled with a dynamic energy budget; DOI 10.1371/journal.pone.0194950; http://modelingcommons.org/browse/one_model/5348). The IBM data is derived from longitudinal studies on Macquarie Island. DEB parameters are derived from Kooijman’s toolbox, DEBtool (http://www.debtheory.org/; more detail provided in associated manuscript). The full description of the model is presented in Chapter 4  ("Using the DEB-IBM to assess the drivers of the decreasing population of elephant seals at Macquarie Island") of my thesis: "Improving representations of higher trophic-level species in models: Using individual-based modelling and dynamic energy budget theory to project population trajectories of southern elephant seals". This has been sent for external review on May 3, 2018. 

## USER MANUAL

To learn the basics of modelling with NetLogo, please see the NetLogo general model information and handbook at https://ccl.northwestern.edu/netlogo/docs/; specific information on individual-based modelling in NetLogo can also be found in the book “Agent-based and individuals-based modelling. A practical introduction” by Railsback and Grimm (2011); an introduction to dynamic energy budget theory is found in the book “Dynamic energy budget theory for metabolic organisation” by Kooijman (2010).

Once NetLogo has been opened, three tabs can be seen; Interface, Info and Code. The main tab used for running of the models, is the Interface. Here all global parameters are set, and outcomes visualised. The Info tab can include information on the model, its use, or important notes. The Code tab contains the procedures implemented in the model (detailed explanations of these procedures can be found in the ODD description of the model). 
Using the DEB-IBM can be done at two levels. Minimal modelling experience is required for the first level of interaction, which is through the interface only. A deeper understanding of modelling and the programming language are required when alterations are made to the code as well as the interface. This may be required, for example, when the model is to be adjusted to fit different species, or when the model (currently non-spatial) is enhanced to become spatially explicit. 

The first thing to note when opening the model, is that it is visually quite large; you can zoom in (Ctrl, +) and out (Ctrl, -) to get to the optimal viewing of the model on your monitor. The buttons (purple) are clickable modelling actions; inputs on the interface (green) are global and used in the code. These can be in the form of set values, variable sliders or switches (on or off). All model outputs visualised on the interface are within yellow frames and can be in the form of plots or values.  
**Please note that the sliders and switches on the interface are sensitive to scrolling with the mouse.** If you need to scroll through the interface (to get from top to bottom) please use the bar on the side of the screen to avoid inadvertently modifying set parameter values. 

The standard term for individual agents in NetLogo is ‘turtles’ – this can be changed to whatever name you would like it to be in the initialisation section of the code (see the NetLogo manual; Breeds). For simplicity sake we have chosen not to do so. We do however recommend it when a model is created for multiple species which follow different procedures.  


### RUNNING THE MODEL

The most important buttons on the interface are the ‘setup’ and ‘go’ buttons. The ‘setup’ button initialises the model – here individuals (and their environment) are created, the ‘go’ button runs the model. This button is set to continue running the model until a ‘stop-command’ is reached in the code (e.g. if max-years is reached, or if there are no more individuals left). Alternatively there is a ‘go-once’ button, which allows you to run the model on step at a time. Run the model with its initial settings to run the ‘baseline model’ of Goedegebuure et al. (2018). 

Under the ‘setup and general model settings and outcomes’ heading on the interface, there are five inputs to the model which can be easily adjusted. The starting population can be modified to increase or decrease the initial population modelled. The time the model runs for (in modelled years) can be set in ‘max-years’. This can be coupled with the run in period ‘clear-plots’ which clears plots after a set time period (years). The parameter ‘cv’ sets the individual variability. The parameter ‘f_scaled’ sets the functional response f for individuals; this is equal to the initial food availability and can be used to easily run simple food-related scenarios. In the model this is converted to effective food availability, which takes into account competition between the individuals in the population.

### SPEED

You can speed up the program by deactivating the "view updates" option on the Interace tab.

## EXPORT DATA

A simple method of getting model results from NetLogo is through immediate plotting on the interface. Plots can be in the form of graphs, histograms, or scatterplots. The plots are set up in the code (see the NetLogo manual; Plotting), and are updated in the very last procedure of each time-step. Output from the model can be exported as either image (.png file, for plots or the interface) or ‘comma-separated values’ file (.csv for plots and the whole model ‘world’). This can be done manually from the interface (right-click, ‘export…’) or through a line in the code (see the NetLogo manual; ‘export-’). It should be noted however, that when a filename already exists, it will be over-written. This can be avoided by creating unique names for each of the files which include the current date, for example. 

An alternative method for collecting the model outputs - and this can be done for numerous model runs - is through the ‘BehaviourSpace’ option built in under ‘Tools’ in the model. This creates a popup where you can run multiple scenarios. The number of runs are adjusted through editing of the scenario and changing the number of repetitions. The results are automatically exported to a .csv file of your choosing, with the properties entered under ‘Measure runs using these reporters’. For a detailed explanation of BehaviourSpace, please see the NetLogo handbook at https://ccl.northwestern.edu/netlogo/docs/ and select the BehaviourSpace link of the left of the page, under Features. 


## References highlighted in model code
Carrick, R., S. Csordas, S. E. Ingham, and K. Keith. 1962. Studies on the southern elephant seal, Mirounga leonina (L.). III. The annual cycle in relation to age and sex. Wildlife Research 7:119-160.

Desprez, M., R. Harcourt, M. A. Hindell, S. Cubaynes, O. Gimenez, and C. R. McMahon. 2014. Age-specific cost of first reproduction in female southern elephant seals. Biology Letters 10:20140264.

Fedak, M., P. Lovell, and B. McConnell. 1996. MAMVIS: A marine mammal behaviour visualization system. The Journal of Visualization and Computer Animation 7:141-147.

Grimm, V., U. Berger, F. Bastiansen, S. Eliassen, V. Ginot, J. Giske, J. Goss-Custard, T. Grand, S. K. Heinz, and G. Huse. 2006. A standard protocol for describing individual-based and agent-based models. Ecological Modelling 198:115-126.

Grimm, V., U. Berger, D. L. DeAngelis, J. G. Polhill, J. Giske, and S. F. Railsback. 2010. The ODD protocol: a review and first update. Ecological Modelling 221:2760-2768.

Hindell, M., and D. Slip. 1997. The importance of being fat: maternal expenditure in the southern elephant seal Mirounga leonina. Pages 72-77 in Marine mammal research in the Southern Hemisphere.

Hindell, M. A., and G. J. Little. 1988. Longevity, fertility and philopatry of two female southern elephant seals (Mirounga leonina) at Macquarie Island. Marine Mammal Science 4:168-171.

Hindell, M. A., D. J. Slip, and H. R. Burton. 1994. Body mass loss of moulting female southern elephant seals, Mirounga leonina, at Macquarie Island. Polar Biology 14:275-278.

Kooijman, S. A. L. M. 2010a. Comments on Dynamic energy budget theory for metabolic 
organisation. Cambridge University Press, New York.

Kooijman, S. A. L. M. 2010b. Dynamic energy budget theory for metabolic organisation. 3rd edition. Cambridge University Press, New York.
Laws, R. M. 1984. Antarctic ecology.

Martin, B. T., T. Jager, R. M. Nisbet, T. G. Preuss, and V. Grimm. 2013. Predicting population dynamics from the properties of individuals: a cross-level test of Dynamic Energy Budget theory. The American Naturalist 181:506-519.

Martin, B. T., E. I. Zimmer, V. Grimm, and T. Jager. 2012. Dynamic Energy Budget theory meets individual-based modelling: a generic and accessible implementation. Methods in Ecology and Evolution 3:445-449.

McMahon, C. R., H. R. Burton, and M. N. Bester. 2000. Weaning mass and the future survival of juvenile southern elephant seals, Mirounga leonina, at Macquarie Island. Antarctic Science 12:149-153.

McMahon, C. R., H. R. Burton, and M. N. Bester. 2003. A demographic comparison of two southern elephant seal populations. Journal of Animal Ecology 72:61-74.

Railsback, S. F., and V. Grimm. 2011. Agent-based and individual-based modeling: a practical introduction. Princeton University Press.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

seal
true
0
Polygon -6459832 true false 45 44 45 55 44 62 71 77 77 93 84 116 103 143 74 211 89 205 91 209 115 183 126 161 138 229 153 211 164 178 227 201 249 257 263 230 250 197 274 229 273 218 248 180 231 144 207 114 108 57 97 41 68 41

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Pup_energy_change" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage =  0]</metric>
    <metric>count turtles with [ stage =  1]</metric>
    <metric>count turtles with [age &lt; 360 and stage &gt; 0]</metric>
    <metric>count turtles with [ stage =  2]</metric>
    <metric>count turtles with [ stage =  3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;twenty&quot;"/>
      <value value="&quot;fifty&quot;"/>
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;decadal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="170"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Climate_SAM_2017-11_v2" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage =  0]</metric>
    <metric>count turtles with [ stage =  1]</metric>
    <metric>count turtles with [age &lt; 360 and stage &gt; 0]</metric>
    <metric>count turtles with [ stage =  2]</metric>
    <metric>count turtles with [ stage =  3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <enumeratedValueSet variable="rv">
      <value value="0.95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;SAM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="8650"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="170"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Climate_ENSO_2017-11" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage =  0]</metric>
    <metric>count turtles with [ stage =  1]</metric>
    <metric>count turtles with [age &lt; 360 and stage &gt; 0]</metric>
    <metric>count turtles with [ stage =  2]</metric>
    <metric>count turtles with [ stage =  3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <enumeratedValueSet variable="rv">
      <value value="0.55"/>
      <value value="0.6"/>
      <value value="0.65"/>
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;ENSO&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="8650"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="170"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="fecundity_change_v2" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage =  0]</metric>
    <metric>count turtles with [ stage =  1]</metric>
    <metric>count turtles with [age &lt; 360 and stage &gt; 0]</metric>
    <metric>count turtles with [ stage =  2]</metric>
    <metric>count turtles with [ stage =  3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <enumeratedValueSet variable="rv">
      <value value="0.55"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;twenty&quot;"/>
      <value value="&quot;fifty&quot;"/>
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="8650"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;no&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;ENSO&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="170"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;twenty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Climate_ENSO_2017-12" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage =  0]</metric>
    <metric>count turtles with [ stage =  1]</metric>
    <metric>count turtles with [age &lt; 360 and stage &gt; 0]</metric>
    <metric>count turtles with [ stage =  2]</metric>
    <metric>count turtles with [ stage =  3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0 and age &lt; 3]</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0 and age &gt;= 3 and age &lt; 5]</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0 and age &gt;= 5 and age &lt; 8]</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0 and age &gt;= 8]</metric>
    <metric>mean [fecundity] of turtles with [fecundity &gt; 0 and age &lt; 3]</metric>
    <metric>mean [fecundity] of turtles with [fecundity &gt; 0 and age &gt;= 3 and age &lt; 5]</metric>
    <metric>mean [fecundity] of turtles with [fecundity &gt; 0 and age &gt;= 5 and age &lt; 8]</metric>
    <metric>mean [fecundity] of turtles with [fecundity &gt; 0 and age &gt;= 8]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &lt; 1]</metric>
    <metric>count turtles with [die-now = 0 and age &lt; 1]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt;= 1 and age &lt; 3]</metric>
    <metric>count turtles with [die-now = 0 and age &gt;= 1 and age &lt; 3]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt;= 3 and age &lt; 6]</metric>
    <metric>count turtles with [die-now = 0 and age &gt;= 3 and age &lt; 6]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt;= 6 and age &lt; 9]</metric>
    <metric>count turtles with [die-now = 0 and age &gt;= 6 and age &lt; 9]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt;= 9 and age &lt; 15]</metric>
    <metric>count turtles with [die-now = 0 and age &gt;= 9 and age &lt; 15]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt;= 15]</metric>
    <metric>count turtles with [die-now = 0 and age &gt;= 15]</metric>
    <enumeratedValueSet variable="rv">
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;ENSO&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="8650"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="170"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Ch4_Hyp_Base_2017-12_agestructs" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage =  0]</metric>
    <metric>count turtles with [ stage =  1]</metric>
    <metric>count turtles with [age &lt; 360 and stage &gt; 0]</metric>
    <metric>count turtles with [ stage =  2]</metric>
    <metric>count turtles with [ stage =  3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &lt; 2 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &lt; 1 * 360] / count turtles with [die-now = 0 and age &lt; 1 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 1 * 360 and age &lt; 2 * 360] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 2 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 2 * 360 and age &lt; 3 * 360] / count turtles with [die-now = 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 3 * 360 and age &lt; 4 * 360] / count turtles with [die-now = 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 4 * 360 and age &lt; 5 * 360] / count turtles with [die-now = 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 5 * 360 and age &lt; 6 * 360] / count turtles with [die-now = 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 6 * 360 and age &lt; 7 * 360] / count turtles with [die-now = 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 7 * 360 and age &lt; 8 * 360] / count turtles with [die-now = 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 8 * 360 and age &lt; 9 * 360] / count turtles with [die-now = 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 9 * 360 and age &lt; 10 * 360] / count turtles with [die-now = 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 10 * 360 and age &lt; 11 * 360] / count turtles with [die-now = 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 11 * 360 and age &lt; 12 * 360] / count turtles with [die-now = 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 12 * 360 and age &lt; 13 * 360] / count turtles with [die-now = 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 13 * 360 and age &lt; 14 * 360] / count turtles with [die-now = 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 14 * 360 and age &lt; 15 * 360] / count turtles with [die-now = 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 15 * 360 and age &lt; 16 * 360] / count turtles with [die-now = 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 16 * 360 and age &lt; 17 * 360] / count turtles with [die-now = 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 17 * 360 and age &lt; 18 * 360] / count turtles with [die-now = 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 18 * 360 and age &lt; 19 * 360] / count turtles with [die-now = 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 19 * 360 and age &lt; 20 * 360] / count turtles with [die-now = 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 20 * 360 and age &lt; 21 * 360] / count turtles with [die-now = 0 and age &gt; 20 * 360 and age &lt; 21 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 21 * 360 and age &lt; 22 * 360] / count turtles with [die-now = 0 and age &gt; 21 * 360 and age &lt; 22 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 22 * 360 and age &lt; 23 * 360] / count turtles with [die-now = 0 and age &gt; 22 * 360 and age &lt; 23 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 23 * 360 ] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 23 * 360]</metric>
    <enumeratedValueSet variable="rv">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;fifty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;twenty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;ENSO&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="170"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Ch4_Hyp_Base_2017-12_K1K2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage = 0]</metric>
    <metric>count turtles with [ stage = 1]</metric>
    <metric>count turtles with [ stage &gt; 0 and age &lt; 360]</metric>
    <metric>count turtles with [ stage = 2 and age &gt; 360]</metric>
    <metric>count turtles with [ stage = 3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &lt; 2 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &lt; 1 * 360] / count turtles with [die-now = 0 and age &lt; 1 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 1 * 360 and age &lt; 2 * 360] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 2 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 2 * 360 and age &lt; 3 * 360] / count turtles with [die-now = 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 3 * 360 and age &lt; 4 * 360] / count turtles with [die-now = 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 4 * 360 and age &lt; 5 * 360] / count turtles with [die-now = 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 5 * 360 and age &lt; 6 * 360] / count turtles with [die-now = 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 6 * 360 and age &lt; 7 * 360] / count turtles with [die-now = 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 7 * 360 and age &lt; 8 * 360] / count turtles with [die-now = 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 8 * 360 and age &lt; 9 * 360] / count turtles with [die-now = 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 9 * 360 and age &lt; 10 * 360] / count turtles with [die-now = 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 10 * 360 and age &lt; 11 * 360] / count turtles with [die-now = 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 11 * 360 and age &lt; 12 * 360] / count turtles with [die-now = 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 12 * 360 and age &lt; 13 * 360] / count turtles with [die-now = 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 13 * 360 and age &lt; 14 * 360] / count turtles with [die-now = 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 14 * 360 and age &lt; 15 * 360] / count turtles with [die-now = 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 15 * 360 and age &lt; 16 * 360] / count turtles with [die-now = 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 16 * 360 and age &lt; 17 * 360] / count turtles with [die-now = 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 17 * 360 and age &lt; 18 * 360] / count turtles with [die-now = 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 18 * 360 and age &lt; 19 * 360] / count turtles with [die-now = 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 19 * 360 and age &lt; 20 * 360] / count turtles with [die-now = 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 20 * 360 and age &lt; 21 * 360] / count turtles with [die-now = 0 and age &gt; 20 * 360 and age &lt; 21 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 21 * 360 and age &lt; 22 * 360] / count turtles with [die-now = 0 and age &gt; 21 * 360 and age &lt; 22 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 22 * 360 and age &lt; 23 * 360] / count turtles with [die-now = 0 and age &gt; 22 * 360 and age &lt; 23 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 23 * 360 ] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 23 * 360]</metric>
    <enumeratedValueSet variable="rv">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;fifty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;twenty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;ENSO&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="130"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Ch4_Hyp_Base_2017-12_SAM" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage = 0]</metric>
    <metric>count turtles with [ stage = 1]</metric>
    <metric>count turtles with [ stage &gt; 0 and age &lt; 360]</metric>
    <metric>count turtles with [ stage = 2 and age &gt; 360]</metric>
    <metric>count turtles with [ stage = 3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &lt; 2 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &lt; 1 * 360] / count turtles with [die-now = 0 and age &lt; 1 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 1 * 360 and age &lt; 2 * 360] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 2 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 2 * 360 and age &lt; 3 * 360] / count turtles with [die-now = 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 3 * 360 and age &lt; 4 * 360] / count turtles with [die-now = 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 4 * 360 and age &lt; 5 * 360] / count turtles with [die-now = 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 5 * 360 and age &lt; 6 * 360] / count turtles with [die-now = 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 6 * 360 and age &lt; 7 * 360] / count turtles with [die-now = 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 7 * 360 and age &lt; 8 * 360] / count turtles with [die-now = 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 8 * 360 and age &lt; 9 * 360] / count turtles with [die-now = 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 9 * 360 and age &lt; 10 * 360] / count turtles with [die-now = 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 10 * 360 and age &lt; 11 * 360] / count turtles with [die-now = 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 11 * 360 and age &lt; 12 * 360] / count turtles with [die-now = 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 12 * 360 and age &lt; 13 * 360] / count turtles with [die-now = 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 13 * 360 and age &lt; 14 * 360] / count turtles with [die-now = 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 14 * 360 and age &lt; 15 * 360] / count turtles with [die-now = 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 15 * 360 and age &lt; 16 * 360] / count turtles with [die-now = 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 16 * 360 and age &lt; 17 * 360] / count turtles with [die-now = 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 17 * 360 and age &lt; 18 * 360] / count turtles with [die-now = 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 18 * 360 and age &lt; 19 * 360] / count turtles with [die-now = 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 19 * 360 and age &lt; 20 * 360] / count turtles with [die-now = 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 20 * 360 and age &lt; 21 * 360] / count turtles with [die-now = 0 and age &gt; 20 * 360 and age &lt; 21 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 21 * 360 and age &lt; 22 * 360] / count turtles with [die-now = 0 and age &gt; 21 * 360 and age &lt; 22 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 22 * 360 and age &lt; 23 * 360] / count turtles with [die-now = 0 and age &gt; 22 * 360 and age &lt; 23 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 23 * 360 ] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 23 * 360]</metric>
    <enumeratedValueSet variable="rv">
      <value value="0.9"/>
      <value value="0.925"/>
      <value value="0.95"/>
      <value value="0.975"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;SAM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="8650"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="130"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Ch4_Hyp_Base_2017-12_ENSO" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage = 0]</metric>
    <metric>count turtles with [ stage = 1]</metric>
    <metric>count turtles with [ stage &gt; 0 and age &lt; 360]</metric>
    <metric>count turtles with [ stage = 2 and age &gt; 360]</metric>
    <metric>count turtles with [ stage = 3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &lt; 2 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>mean [total-puppies / ((age / 360) - breeding-age)] of turtles with [total-puppies &gt; 0 and age &gt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &lt; 1 * 360] / count turtles with [die-now = 0 and age &lt; 1 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 1 * 360 and age &lt; 2 * 360] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 2 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 2 * 360 and age &lt; 3 * 360] / count turtles with [die-now = 0 and age &gt; 2 * 360 and age &lt; 3 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 3 * 360 and age &lt; 4 * 360] / count turtles with [die-now = 0 and age &gt; 3 * 360 and age &lt; 4 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 4 * 360 and age &lt; 5 * 360] / count turtles with [die-now = 0 and age &gt; 4 * 360 and age &lt; 5 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 5 * 360 and age &lt; 6 * 360] / count turtles with [die-now = 0 and age &gt; 5 * 360 and age &lt; 6 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 6 * 360 and age &lt; 7 * 360] / count turtles with [die-now = 0 and age &gt; 6 * 360 and age &lt; 7 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 7 * 360 and age &lt; 8 * 360] / count turtles with [die-now = 0 and age &gt; 7 * 360 and age &lt; 8 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 8 * 360 and age &lt; 9 * 360] / count turtles with [die-now = 0 and age &gt; 8 * 360 and age &lt; 9 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 9 * 360 and age &lt; 10 * 360] / count turtles with [die-now = 0 and age &gt; 9 * 360 and age &lt; 10 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 10 * 360 and age &lt; 11 * 360] / count turtles with [die-now = 0 and age &gt; 10 * 360 and age &lt; 11 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 11 * 360 and age &lt; 12 * 360] / count turtles with [die-now = 0 and age &gt; 11 * 360 and age &lt; 12 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 12 * 360 and age &lt; 13 * 360] / count turtles with [die-now = 0 and age &gt; 12 * 360 and age &lt; 13 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 13 * 360 and age &lt; 14 * 360] / count turtles with [die-now = 0 and age &gt; 13 * 360 and age &lt; 14 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 14 * 360 and age &lt; 15 * 360] / count turtles with [die-now = 0 and age &gt; 14 * 360 and age &lt; 15 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 15 * 360 and age &lt; 16 * 360] / count turtles with [die-now = 0 and age &gt; 15 * 360 and age &lt; 16 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 16 * 360 and age &lt; 17 * 360] / count turtles with [die-now = 0 and age &gt; 16 * 360 and age &lt; 17 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 17 * 360 and age &lt; 18 * 360] / count turtles with [die-now = 0 and age &gt; 17 * 360 and age &lt; 18 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 18 * 360 and age &lt; 19 * 360] / count turtles with [die-now = 0 and age &gt; 18 * 360 and age &lt; 19 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 19 * 360 and age &lt; 20 * 360] / count turtles with [die-now = 0 and age &gt; 19 * 360 and age &lt; 20 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 20 * 360 and age &lt; 21 * 360] / count turtles with [die-now = 0 and age &gt; 20 * 360 and age &lt; 21 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 21 * 360 and age &lt; 22 * 360] / count turtles with [die-now = 0 and age &gt; 21 * 360 and age &lt; 22 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 22 * 360 and age &lt; 23 * 360] / count turtles with [die-now = 0 and age &gt; 22 * 360 and age &lt; 23 * 360]</metric>
    <metric>count turtles with [die-now = 1 and days-dead = 1 and age &gt; 23 * 360 ] / count turtles with [die-now = 0 and age &gt; 1 * 360 and age &lt; 23 * 360]</metric>
    <enumeratedValueSet variable="rv">
      <value value="0.71"/>
      <value value="0.72"/>
      <value value="0.73"/>
      <value value="0.74"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_freq">
      <value value="&quot;ENSO&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="8650"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6plus">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="selected_cumm_UE">
      <value value="&quot;prop_cumm_UE&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_3yo">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change-pup-mort">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^x">
      <value value="125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cv">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_mort?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clear-plots">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-years">
      <value value="130"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup-mort">
      <value value="&quot;mean&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_5yo">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^b_init">
      <value value="28100000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_J_rate">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_fec_change">
      <value value="&quot;eighty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-pup">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^p">
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pup-mort">
      <value value="0.00128"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-pup-mort">
      <value value="9.456E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-pup-mort">
      <value value="6.11E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="h_a">
      <value value="6.0E-10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Ch4_Hyp_Base_2018-01_K1K2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>min [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>min [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 3] / 360</metric>
    <metric>mean [age] of turtles with [stage = 2] / 360</metric>
    <metric>max [final-age] of turtles</metric>
    <metric>max [final-pups] of turtles</metric>
    <metric>max [fecundity] of turtles with [fecundity &gt; 0]</metric>
    <metric>max [L] of turtles</metric>
    <metric>mean [L] of turtles with [stage =  3]</metric>
    <metric>mean [L] of turtles with [stage =  2]</metric>
    <metric>count turtles with [stage &gt; 1]</metric>
    <metric>count turtles with [ stage = 0]</metric>
    <metric>count turtles with [ stage = 1]</metric>
    <metric>count turtles with [ stage &gt; 0 and age &lt; 360]</metric>
    <metric>count turtles with [ stage = 2 and age &gt; 360]</metric>
    <metric>count turtles with [ stage = 3]</metric>
    <metric>mean [f] of turtles with [stage &gt; 1 and age &gt; 360  and status = 2]</metric>
    <metric>mean [age] of turtles with [offspring_ID &gt; -1] / 360</metric>
    <metric>carrying_capacity</metric>
    <metric>my-seed</metric>
    <metric>count turtles with [track-life = 1]</metric>
    <metric>sum [puppy] of turtles with [track-life = 1]</metric>
    <enumeratedValueSet variable="rv">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortality-float">
      <value value="0.013"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="age_for_pup_energy_change">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weaning-duration">
      <value value="23"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="J_XAm_rate_init">
      <value value="0.0023"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_energy_change">
      <value value="&quot;fifty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="period-in-K">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k_M_rate">
      <value value="0.0014"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^x_init">
      <value value="65000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="F_m">
      <value value="6.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="implement-change-at">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p_am">
      <value value="968.2785"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="use_profiler?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start_population">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^b">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="g_init">
      <value value="0.7138"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_H^p_init">
      <value value="145000000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food_change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mother_fecundity_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-adult">
      <value value="98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_L">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_4yo">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-1">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-pup-mortality">
      <value value="&quot;maximum&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap_F">
      <value value="0.725"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="L_w^m">
      <value value="280"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diapause">
      <value value="120"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decadal_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_mum_change">
      <value value="&quot;hundred&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-adult">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pup_energy_change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="v_rate">
      <value value="0.065"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-juv">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="track-turtle">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="f_scaled">
      <value value="0.935"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mwho-juv">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moult-duration-juv">
      <value value="26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shape_factor">
      <value value="0.235"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="increase?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="s_G">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resting-duration-adult">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-pop-2">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_of_pup_change">
      <value value="&quot;twenty&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="time-for-decline">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kap">
      <value value="0.74"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="forage-duration-juv">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="breeding-duration">
      <value value="217"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="prob_mate_fail_6p                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 