extensions [nw cf]

breed [students student]
breed [organizations organization]
undirected-link-breed [friendships friendship]

globals [
  all-types
]

students-own [
  freshman?    ;to distinguish between freshmen and upperclassmen
  group-types-wanted
  group
  total-groups-recruited-for
  happy?
]

organizations-own [
  name
  group-types
  drops-in-year  ;counts how many students drop the group in a given year
]

;; To setup the model
to setup
  clear-all
  set all-types  ["cultural" "engineering" "greek" "business"]
  ;25% students are freshmen, rest upperclassmen
  initialize-orgs
  initialize-network
  initialize-freshmen
  colorize
  layout
  reset-ticks
end

;; The go procedure that runs until all freshmen are in a group or have exhausted their recruitment limits
;; Calls next year procedure if 'years-roll-on' switch is enabled
to go
  ; continue if there exists freshmen with no group or max-recruitment-limit not reached
  if count students with [freshman? = true and group = nobody and total-groups-recruited-for < student-recruit-limit] = 0 [
    tabulate-happiness
    tick
    reset-drops-in-yr
    ifelse years-roll-on
    [ next-year ]
    [ stop ]
  ]
  ask students[
    let new-friend nobody
    ifelse freshman?
    [
      ;; 50% chance of meeting freshman vs upperclassmen (as opposed to doing it randomly, would be 25 - 75 based on population)
      ifelse random 10 < 5
      [ set new-friend one-of other students with [freshman? = true] ]
      [ set new-friend one-of students with [freshman? = false]]
    ]
    [
      ; upperclassmen meet new ppl w/ only 50% prbability since they are less likely to mingle
      if random 100 < 50 [
        ;; 80% chance of meeting upperclassmen vs freshmen
        ifelse random 10 < 8
        [ set new-friend one-of other students with [freshman? = false] ]
        [ set new-friend one-of students with [freshman? = true] ]
      ]
    ]
    if new-friend != nobody [
      ; If not already friends, then:
      if not member? new-friend friendship-neighbors [
        create-friendship-with new-friend
      ]
      ; if still looking for a group, and new friend is in a group
      if [group] of new-friend != nobody and group = nobody[
        let potential-new-group [group] of new-friend
        if recruit? potential-new-group [
          set group potential-new-group
        ]
      ]
    ]

  ]
  colorize
  layout
end

;; To initialize the hidden organizations, with their names and group-types
to initialize-orgs
  create-organizations 1 [
    set name "ISBE"
    set hidden? true
    set group-types ["business"]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "Sig Nu"
    set hidden? true
    set group-types ["greek"]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "EPIC"
    set hidden? true
    set group-types ["business" ]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "MCSA"
    set hidden? true
    set group-types ["cultural"]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "Alpha Phi"
    set hidden? true
    set group-types ["greek"]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "IEEE"
    set hidden? true
    set group-types ["engineering"]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "Formula"
    set hidden? true
    set group-types ["engineering"]
    set drops-in-year 0
  ]
  create-organizations 1 [
    set name "SASA"
    set hidden? true
    set group-types ["cultural"]
    set drops-in-year 0
  ]
end

;; To initialize the initial network of Northwestern upperclassmen with preferrential attachment using built-in Networks extension
to initialize-network
  ;each upperclassman has at least 5 friends
  nw:generate-preferential-attachment students friendships (num-students * 3 / 4) 5
  ask students [
    set freshman? false
    set group-types-wanted get-two-types
    ; put student in a group that is of one of the types he wants
    set group one-of organizations with [ member? (item 0 [group-types-wanted] of myself) group-types or member? (item 1 [group-types-wanted] of myself) group-types ]
    set happy? true
    set total-groups-recruited-for 1
  ]
end

;; The recruit procedure, that returns true if the student is accepted in org, false otherwise
;; Calls the recruit-freshman? or recruit-upperclassman? procedure depending on the year of the student
;; student procedure
to-report recruit? [ org ]
  ifelse freshman?
  [ report recruit-freshman? org ]
  [ report recruit-upperclassman? org ]
end

;; Freshman recruit procedure, returns true if the student is accepted in org, false otherwise
;; student procedure
to-report recruit-freshman? [ org ]
  ; can only try to recruit if not exceeded limit
  let total-members-in-org count students with [group != nobody and [name] of group = [name] of org]
  if total-members-in-org >= group-recruitment-limit [
    report false
  ]
  ifelse total-groups-recruited-for < student-recruit-limit [
    set total-groups-recruited-for total-groups-recruited-for + 1
    let x 0 ;x represents # of group types wanted by student that are also in the group type of the selected organization
    let y 0 ;y represents total # of group types wanted by student
    foreach group-types-wanted [ g-type ->
      if member? g-type [group-types] of org [
        set x x + 1
      ]
      set y y + 1
    ]
    ;;once student has recruited for the max number of groups, he has a 50% chance of joining the group if there is no fit- desperation
    if total-groups-recruited-for = student-recruit-limit and x = 0[
      set x 1
      set y 2
    ]
    ifelse random 100 < x / y * 100
    [ report true ]
    [ report false ]
  ]
  [ report false ]
end

;; Upperclassman recruit procedure, returns true if the student is accepted in org, false otherwise
;; student procedure
to-report recruit-upperclassman? [ org ]
  let num-friends-in-org count friendship-neighbors with [group != nobody and [name] of group = [name] of org]
  let total-members-in-org count students with [group != nobody and [name] of group = [name] of org]
  if total-members-in-org >= group-recruitment-limit [
    report false
  ]
  if total-groups-recruited-for > student-recruit-limit [
    report false
  ]
  ;if you have an intersecting interest, you will join the group as an upperclassmen with 75% probability; or if you know more than threshold% of the group
  foreach group-types-wanted [ g-type ->
    if member? g-type [group-types] of org and random 100 < 75[
      report true
    ]
  ]
  ifelse num-friends-in-org / total-members-in-org * 100 > group-friendship-threshold-%
  [ report true ]
  [ report false ]

end

;; Initializes a new class of freshmen, with no friends or organization
to initialize-freshmen
  create-students num-students / 4 [
    set freshman? true
    set happy? false
    set total-groups-recruited-for 0
    set group nobody
    set group-types-wanted get-two-types
  ]
end

;; Sets the color and shape of the students based on their year and group affiliations
to colorize
  ask students [
    ifelse freshman?
    [ set shape "default" ]
    [ set shape "circle" ]
  ]
  ask students with [group = nobody ] [
    set color white
  ]
  ask students with [group != nobody] [
    (cf:ifelse
      [name] of group = "ISBE" [ set color red ]
      [name] of group = "Sig Nu" [ set color green ]
      [name] of group = "EPIC" [ set color violet  ]
      [name] of group = "MCSA" [ set color yellow  ]
      [name] of group = "Alpha Phi" [ set color pink  ]
      [name] of group = "IEEE" [ set color cyan  ]
      [name] of group = "Formula" [ set color orange  ]
      [name] of group = "SASA" [ set color brown  ]
    )
  ]
end

;; Sets the happy? parameter of each student as true/false at the end of the year
to tabulate-happiness
  ask students [
    ifelse group = nobody
    [
      ;;unhappy if in no group
      set happy? false
    ]
    [
      ;unhappy if in a group with no interests in common
      let interests-in-common 0
      foreach group-types-wanted [ g-type ->
        if member? g-type [group-types] of group [
          set interests-in-common interests-in-common + 1
        ]
      ]
      ifelse interests-in-common = 0
      [
        ; if no interests in common, can still be happy if many friends in the group
        let num-friends-in-same-group count friendship-neighbors with [group != nobody and [name] of group = [name] of [group] of myself]
        let total-members-my-org count students with [group != nobody and [name] of group = [name] of [group] of myself]
        ifelse num-friends-in-same-group / total-members-my-org * 100 < group-friendship-threshold-%
        [ set happy? false ]
        [ set happy? true ]
      ]
      [ set happy? true ]
    ]
  ]
end

;; Network visualization procedure
to layout
  repeat 30 [
    layout-spring students with [freshman? = false]  friendships 0.2 5 1
    layout-circle students with [freshman? = true] 10
    display
  ]
end

;; Called at the end of the year- seniors graduate, freshmen become upperclassmen, students drop groups if unhappy, and new class of freshmen initialized
to next-year
  ;;33% of upperclassmen graduate (upperclasmen = soph, jun, sen)- only seniors graduate
  ask n-of (25 / 100 * num-students) students with [freshman? = false] [
    die
  ]

  ask students with [freshman? = true] [
    set freshman? false
  ]

  ask students [
    if happy? = false [
      if group != nobody [
        ask group [
          set drops-in-year drops-in-year + 1
        ]
        set group nobody
      ]
    ]
  ]
  initialize-freshmen
  colorize
  layout
end

;; Resets drops-in-year of each organization to 0 so can be retabulated for the following year
to reset-drops-in-yr
  ask organizations [
    set drops-in-year 0
  ]
end

;; Returns a list of 2 group-types based on the relative distribution of interests (sliders)
to-report get-two-types
  ;The 2 types may be the same, just means that the particular student is very keen on that type of group
  let type1 get-one-type
  let type2 get-one-type
  report (list type1 type2)
end

;; Returns 1 group type based on the relative distribution of interests (sliders)
to-report get-one-type
  let type-probabilities (list cultural-interest engineering-interest greek-interest business-interest)
  report random-weighted all-types type-probabilities
end


;; Returns one element from values list based on the weighted probabilities list weights
;; Source: http://netlogo-users.18673.x6.nabble.com/Using-list-distribution-as-a-probability-function-td4867966.html
to-report random-weighted [values weights]
  let selector (random-float sum weights)
  let running-sum 0
  (foreach values weights [ [a b] ->
    set running-sum (running-sum + b)
    if (running-sum > selector) [
      report a
    ]
  ])
end


;; UNIT TESTS

;; Unit test for random-weighted
;; Takes in an input of 3 weights, reports the distribution of 100 actual values produced by calling the random-weighted function, which should reflect the inputed weights
;; If the sum of input weights is 100, output values will be approximately the same as input, otherwise output will reflect the distribution in a population of 100
to-report test-random-weighted [ weights ]
  let out []
  repeat 100 [
   set out lput (random-weighted [ "a" "b" "c"] weights) out
  ]
  let num-a 0
  let num-b 0
  let num-c 0
  (foreach out [ x ->
    (cf:ifelse
      x = "a" [ set num-a num-a + 1 ]
      x = "b" [ set num-b num-b + 1 ]
      x = "c" [ set num-c num-c + 1 ]
    )])
  report (list num-a num-b num-c)
end

;; Unit test #1 for recruitment
;; Tests that both freshmen and upperclassmen who have exceeded their recruitment limit do not join the group
;; Reports true if the recruitment procedures are working accordingly, false otherwise
to-report test-recruitment-1
  clear-all
  set all-types  ["cultural" "engineering" "greek" "business"]
  set student-recruit-limit 10
  ;; all students initialized having exceeded their recruitment limit, with overlapping interest with the group
  create-students 50 [
    set label "test"
    set freshman? true
    set happy? false
    set total-groups-recruited-for 20
    set group nobody
    set group-types-wanted [ "engineering" "greek" ]
  ]
  create-students 50 [
    set label "test"
    set freshman? false
    set happy? false
    set total-groups-recruited-for 20
    set group nobody
    set group-types-wanted [ "engineering" "greek" ]
  ]
  ;; No students in the group initially, so group recruitment limit will not be triggered
  create-organizations 1 [
    set name "test-group"
    set hidden? true
    set group-types ["engineering"]
    set drops-in-year 0
  ]
  ;; if any student gets in, means the recruit function is not working
  let rep true
  ask students with [label = "test"] [
     if (recruit? (one-of organizations with [name = "test-group"])) = true
    [ set rep false ]
  ]
  report rep
end

;; Unit test #2 for recruitment
;; Tests that both freshmen and upperclassmen are rejected by a group that has reached its recruitment capacity
;; Reports true if the recruitment procedures are working accordingly, false otherwise
to-report test-recruitment-2
  clear-all
  set all-types  ["cultural" "engineering" "greek" "business"]
  set group-recruitment-limit 10
  create-organizations 1 [
    set name "test-group"
    set hidden? true
    set group-types ["engineering"]
    set drops-in-year 0
  ]
  ;; initialize the group with 10 members, the limit
  create-students 10 [
    set label "test"
    set freshman? false
    set happy? true
    set total-groups-recruited-for 1
    set group one-of organizations with [name = "test-group"]
    set group-types-wanted get-two-types
  ]
  create-students 50 [
    set label "test"
    set freshman? true
    set happy? false
    set total-groups-recruited-for 0
    set group nobody
    set group-types-wanted get-two-types
  ]
  create-students 50 [
    set label "test"
    set freshman? false
    set happy? false
    set total-groups-recruited-for 0
    set group nobody
    set group-types-wanted get-two-types
  ]
  ;; if any student gets in, means the recruit function is not working
  let rep true
  ask students with [label = "test"] [
     if (recruit? (one-of organizations with [name = "test-group"])) = true
    [ set rep false ]
  ]
  report rep
end

;; Unit test #3 for recruitment
;; Tests that a freshman with no interest in a group does not join the group
;; Reports true if the recruitment procedures are working accordingly, false otherwise
to-report test-recruitment-3
  clear-all
  set all-types  ["cultural" "engineering" "greek" "business"]
  create-students 1 [
    set label "test"
    set freshman? true
    set happy? false
    set total-groups-recruited-for 0
    set group nobody
    set group-types-wanted ["cultural" "greek"]
  ]
  create-organizations 1 [
    set name "test-group"
    set hidden? true
    set group-types ["engineering"]
    set drops-in-year 0
  ]
  ;; if this student gets in, means the recruit function is not working
  let rep false
  ask one-of students with [label = "test"] [
     ifelse (recruit? (one-of organizations with [name = "test-group"])) = true
    [ set rep false ]
    [ set rep true]
  ]
  report rep
end

;; Unit test #4 for recruitment
;; Confirms that upperclassmen join group if they are friends with more than group-friendship-threshold-% of the members
;; Reports true if the recruitment procedures are working accordingly, false otherwise
to-report test-recruitment-4
  clear-all
  set all-types  ["cultural" "engineering" "greek" "business"]
  set group-friendship-threshold-% 25
  let temp group-recruitment-limit
  set group-recruitment-limit 200
  create-organizations 1 [
    set name "test-group"
    set hidden? true
    set group-types ["engineering"]
    set drops-in-year 0
  ]
  ;; create upperclassman with no overlapping interest
  create-students 1 [
    set label "test"
    set freshman? false
    set happy? false
    set total-groups-recruited-for 0
    set group nobody
    set group-types-wanted ["cultural" "greek"]
  ]
  ;; 30 students in the group who are friends with the test student
  create-students 30 [
    set freshman? false
    set happy? false
    set total-groups-recruited-for 1
    set group one-of organizations with [name = "test-group"]
    create-friendship-with one-of students with [label = "test"]
  ]
  ;; 70 students in the group who are not friends with the test student
  create-students 70 [
    set freshman? false
    set happy? false
    set total-groups-recruited-for 1
    set group one-of organizations with [name = "test-group"]
  ]
  ;; if this student does not get in, means the recruit function is not working, since he is frinds with more than threshold% of students in the group
  let rep false
  ask one-of students with [label = "test"] [
    print recruit? (one-of organizations with [name = "test-group"])
     if (recruit? (one-of organizations with [name = "test-group"])) = true
    [ set rep true ]
  ]
  set group-recruitment-limit temp
  report rep
end
@#$#@#$#@
GRAPHICS-WINDOW
780
10
1191
422
-1
-1
12.212121212121213
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
years
30.0

SLIDER
45
20
245
53
num-students
num-students
0
100
100.0
1
1
NIL
HORIZONTAL

BUTTON
25
365
91
398
NIL
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
97
365
160
398
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
830
425
1146
585
Students without group
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Freshmen" 1.0 0 -2674135 true "" "plot count students with [freshman? = true and group = nobody]"
"Upperclassmen" 1.0 0 -16777216 true "" "plot count students with [freshman? = false and group = nobody]"

SLIDER
45
120
245
153
student-recruit-limit
student-recruit-limit
1
15
7.0
1
1
NIL
HORIZONTAL

BUTTON
164
366
254
399
NIL
next-year
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
315
50
765
390
Group Membership
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"ISBE" 1.0 0 -2674135 true "" "plot count students with [group != nobody and [name] of group = \"ISBE\"]"
"Sig Nu" 1.0 0 -13840069 true "" "plot count students with [group != nobody and [name] of group = \"Sig Nu\"]"
"EPIC" 1.0 0 -8630108 true "" "plot count students with [group != nobody and [name] of group = \"EPIC\"]"
"MCSA" 1.0 0 -1184463 true "" "plot count students with [group != nobody and [name] of group = \"MCSA\"]"
"Alpha Phi" 1.0 0 -2064490 true "" "plot count students with [group != nobody and [name] of group = \"Alpha Phi\"]"
"Formula" 1.0 0 -955883 true "" "plot count students with [group != nobody and [name] of group = \"Formula\"]\n"
"SASA" 1.0 0 -6459832 true "" "plot count students with [group != nobody and [name] of group = \"SASA\"]"
"IEEE" 1.0 0 -11221820 true "" "plot count students with [group != nobody and [name] of group = \"IEEE\"]"
"No group" 1.0 0 -16777216 true "" "plot count students with [group = nobody]\n"

PLOT
5
425
300
585
Student Happiness %
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Freshmen" 1.0 0 -2674135 true "" "plot (count students with [freshman? = true and happy? = true]) / (count students with [freshman? = true]) * 100"
"Upperclassmen" 1.0 0 -16777216 true "" "plot (count students with [freshman? = false and happy? = true]) / (count students with [freshman? = false]) * 100"

SLIDER
45
165
245
198
group-recruitment-limit
group-recruitment-limit
0
100
25.0
1
1
NIL
HORIZONTAL

SLIDER
45
215
245
248
group-friendship-threshold-%
group-friendship-threshold-%
0
100
25.0
1
1
NIL
HORIZONTAL

PLOT
315
425
765
585
Group Deactivations
NIL
NIL
0.0
10.0
0.0
8.0
true
true
"" ""
PENS
"ISBE" 1.0 0 -2674135 true "" "plot [drops-in-year] of one-of organizations with [name = \"ISBE\"]"
"Sig Nu" 1.0 0 -13840069 true "" "plot [drops-in-year] of one-of organizations with [name = \"Sig Nu\"]"
"EPIC" 1.0 0 -8630108 true "" "plot [drops-in-year] of one-of organizations with [name = \"EPIC\"]"
"MCSA" 1.0 0 -1184463 true "" "plot [drops-in-year] of one-of organizations with [name = \"MCSA\"]"
"Alpha Phi" 1.0 0 -2064490 true "" "plot [drops-in-year] of one-of organizations with [name = \"Alpha Phi\"]"
"Formula" 1.0 0 -955883 true "" "plot [drops-in-year] of one-of organizations with [name = \"Formula\"]"
"SASA" 1.0 0 -6459832 true "" "plot [drops-in-year] of one-of organizations with [name = \"SASA\"]"
"IEEE" 1.0 0 -11221820 true "" "plot [drops-in-year] of one-of organizations with [name = \"IEEE\"]"

SWITCH
45
70
245
103
years-roll-on
years-roll-on
0
1
-1000

SLIDER
5
265
155
298
cultural-interest
cultural-interest
0
10
2.0
1
1
NIL
HORIZONTAL

SLIDER
160
320
310
353
greek-interest
greek-interest
0
10
8.0
1
1
NIL
HORIZONTAL

SLIDER
160
265
308
298
engineering-interest
engineering-interest
0
10
4.0
1
1
NIL
HORIZONTAL

SLIDER
5
320
155
353
business-interest
business-interest
0
10
7.0
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model shows the integration of incoming freshmen into existing Northwestern social and pre-professional student groups, and shows how this network evolves over different freshmen batches. 

## HOW IT WORKS

Incoming freshmen (25% of num-students) are displayed as default turtles, while the upperclassmen (75% of num-students) are displayed as circles. Each student can only be a part of 1 student group, and is colored based on the group they are a part of. This is simplistic but is realistic in that students always identify strongly with one particular group even if they are a part of multiple groups. There are white links between friends.

On setup, the initial Northwestern upperclassmen network is initialized. Upperclassman are assigned groups according to the relative distribution of interests. Freshmen are initialized with no friends and no groups. They are randomly assigned 2 group types depending on the distribution of the interests, which will drive their recruitment strategies.

On each tick, each freshman meets someone. There is a 50-50 chance of meeting another freshman vs meeting an upperclassman, in spite of the 25-75 population split. This is done because freshmen are more likely to socialize with and meet other students in their year, than in other years. If they are not already friends, the two become friends. If the new friend is in a group, the freshman tries to recruit for it if he is not yet in a group himself. This stops when each freshman is in a student group, or has reached the recruitment limit. 
Similarly, upperclassmen also meet people and recruit, but are less likely to meet students on a given tick, as they are less likely to mingle than freshmen. 

Recruitment Rules:
If there is an intersection between the interests of the potential freshman recruit and the student organization, means there is a good fit, and the student has a 50% chance of getting accepted into the organization. If the student has reached the limit of the number of groups he can recruit for, there is a higher likelihood that he joins this organization out of desperation. This has been modeled as a 50% acceptance rate even if there are no overlapping interests. Students that have not been able to join any organization after reaching the recruitment limit are left group-less.
Upperclassmen join groups slightly differently- if there is an overlapping interest, there is a 75% chance of getting entry into the organization. If there is no intersecting interest, the upperclassman will still join the group if he knows GROUP-FRIENDSHIP-THRESHOLD% of the members. 

Once all freshmen are in a group or have reached their recruitment limit, the overall happiness of the students is tabulated. Happiness is defined as being in a group where there is a good fit, either having an overlapping interest or being friends with GROUP-FRIENDSHIP-THRESHOLD-% of the members.

This process is then repeated for the next year, with seniors graduating, freshmen becoming upperclassmen, students deactivating from organizations, and a new batch of freshmen getting initialized. 

## HOW TO USE IT

Use the NUM-STUDENTS slider to set the total number of students, the STUDENT-RECRUIT-LIMIT and the GROUP-RECRUITMENT-LIMIT sliders to set the student and group recruitment limits respectively. As explained in the previous section, the GROUP-FRIENDSHIP-THRESHOLD-% slider drives upperclassmen recruitment strategies and the happiness of students. Use the CULTURAL-INTEREST, BUSINESS-INTEREST, GREEK-INTEREST and ENGINEERING-INTEREST sliders to set the relative distribution of interests among the student population. 

Click SETUP, and then GO to run the model. Click NEXT-YEAR to initalize the network for the following year. Click GO again to run the model for the following year. If the YEARS-ROLL-ON switch is on, the model will keep running, initializing the next year automatically. In this case, you just need to click GO once, and see the model run! Click GO again to stop the model from running further.

## THINGS TO NOTICE

With the default values for the interest sliders (CULTURAL-INTEREST = 2, ENGINEERING-INTEREST = 4, BUSINESS-INTEREST = 7, GREEK-INTEREST = 8), which is an approximation of the interests distribution at Northwestern, it is interesting to see how the cultural groups tend to die out pretty quick with these simple rules. It makes sense that the recruitment levels are low given that the interest is low, which primarily drives student recruitment strategies, at least for freshmen. But the reason they actually die out instead of just staying at a low membership, is that upperclassmen deactivate from these groups as they see that only few of their friends are in these groups, which is like a domino effect that in turn sees their friends drop the groups too. 

## THINGS TO TRY

It is interesting to tweak with one of the interest sliders keeping the others constant (at say 5). As this slider is at a low value, the associated groups die out pretty quick, as discussed in the previous section. For values near 5, the outcome is pretty random, with groups dying out based on the friendships that evolve. For values near 10, the membership of these groups pretty much reaches the group recruitment limit, as expected. 


## EXTENDING THE MODEL

This model has tremendous scope for extension. Students can be made more complex, with properties that more accurately refelct their interests than just group types wanted. Organizations can be made more interesting by ramping up recruitment efforts as their membership goes low. These changes will drastically affect recruitment strategies, which is the crux of this model. More groups can be added with more comprehensive group types, with groups taking on multiple group types, since often groups are not purely 'business' or purely 'engineering' groups. 

## NETLOGO FEATURES

The built-in Networks extension is leveraged to initialize the upperclassmen friendships, with prefferential attachment, and each upperclassman having at least 5 friends.
 
## RELATED MODELS

Team Assembly (in sample models)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="cultural-interest-experiment" repetitions="40" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <exitCondition>count students with [group != nobody and member? "cultural" [group-types] of group] = 0</exitCondition>
    <metric>count students with [group != nobody and member? "cultural" [group-types] of group]</metric>
    <enumeratedValueSet variable="greek-interest">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="engineering-interest">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="group-friendship-threshold-%">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="years-roll-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="student-recruit-limit">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="group-recruitment-limit">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="business-interest">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-students">
      <value value="100"/>
    </enumeratedValueSet>
    <steppedValueSet variable="cultural-interest" first="1" step="1" last="10"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
