globals [
counter  ;; set to stop running at 500 tick intervals
]
turtles-own [energy   ;; for keeping track of when the turtle is ready to reproduce or die
             age      ;; to track life expectancy
]


to setup
  clear-all
  setup-turtles
  reset-ticks
end

to setup-turtles
  create-turtles number
  ask turtles [
    set shape "turtle"
    setxy random-xcor random-ycor
    set energy 10
    set age 0                        ;; age starts at 0
    ifelse share?
    [ set color yellow ]             ;; sharing turtles set to yellow
    [ set color orange ]             ;; non-sharing turtles set to orange
  ]
end

to go
  if all? turtles [energy < 0] [stop]
  set counter counter + 1
  grow-resources
  move-turtles
  eat-resources
  reproduce
  check-death
  tick
  if counter = 500 [
    set counter 0
    stop
  ]
end

to grow-resources
  ask patches [
    if pcolor = black [
      if random-float 1000 < replenish-rate
        [ set pcolor green ]
    ]
  ]
end

to move-turtles
  ask turtles [
    right random 360
    forward 1
    set energy energy - 1
    set age age + 1
  ]
end

to eat-resources

  ask turtles [
    if pcolor = green [
      set pcolor black
      set energy (energy + (energy-from-resources))]

    ifelse share?
      [ set energy (sum [energy] of turtles-here / count turtles-here)
        set color yellow ]                     ;;all energy shared among all turtles occupying same patch
      [set color orange ]                      ;; otherwise keeps energy the same

    if count turtles-here > 1
    [ set color cyan ]
  ]
end

to reproduce
  ask turtles [
    if energy > birth-energy [
      set energy energy - birth-energy
      hatch 1 [
        set energy birth-energy
        set age 0
      ]
    ]
  ]
end

to check-death
  ask turtles [
    if energy <= 0 [ die ]
    if age > oldage [die]  ;; to effects of sharing across life expectancies
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
35
165
264
395
-1
-1
6.7
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
20
50
90
83
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
210
50
280
83
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
10
420
105
465
Turtles
count turtles
17
1
11

MONITOR
10
465
105
510
Resources
count patches with [pcolor = green]
17
1
11

SWITCH
105
50
195
83
share?
share?
1
1
-1000

PLOT
310
10
605
210
Totals
time
total
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"turtles" 1.0 0 -4079321 true "" "plot count turtles"
"resources" 1.0 0 -13840069 true "" "plot count patches with [pcolor = green]"

SLIDER
5
10
295
43
number
number
1
2000
350.0
1
1
NIL
HORIZONTAL

SLIDER
155
90
295
123
energy-from-resources
energy-from-resources
1
100
51.0
1
1
NIL
HORIZONTAL

SLIDER
5
90
145
123
birth-energy
birth-energy
1
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
155
125
295
158
replenish-rate
replenish-rate
1
40
21.0
1
1
NIL
HORIZONTAL

PLOT
310
210
605
385
Mean
time
mean
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"age" 1.0 0 -16777216 true "" "plot mean [age] of turtles"
"energy" 1.0 0 -2674135 true "" "plot mean [energy] of turtles"

PLOT
310
385
605
535
Age
age
turtles
0.0
10.0
0.0
10.0
true
false
"" "set-plot-y-range 0 1\nset-plot-x-range 0 max [age] of turtles + 1"
PENS
"age" 1.0 1 -16777216 true "" "histogram [age] of turtles"

SLIDER
5
125
145
158
oldage
oldage
10
500
261.0
1
1
NIL
HORIZONTAL

MONITOR
200
445
295
490
Median Age
median [age] of turtles
17
1
11

MONITOR
200
400
295
445
Mean Age
mean [age] of turtles
2
1
11

MONITOR
200
490
295
535
Oldest
max [age] of turtles
2
1
11

MONITOR
105
400
200
445
Mean Energy
mean [energy] of turtles
2
1
11

MONITOR
105
445
200
490
Median Energy
Median [energy] of turtles
2
1
11

MONITOR
105
490
200
535
Max Energy
Max [energy] of turtles
2
1
11

@#$#@#$#@
## WHAT IS IT?

This is an extension from the model included in the tutorial in the **NetLogo User Manual** (as of November, 2018). The tutorial provides a simple demonstration of a limited ecosystem in which turtles consume resources, reproduce, and die. Sustainable systems result in oscillating patterns of turtle populations and resources.

This extension adds an additional variable: sharing. In the tutorial, turtles never share resources and when more than one turtle occupies the same patch, the first turtle on the patch consumes all of the resources (if there are any). In this extension, turtles can be instructed to share _all of their resources_ with turtles who occupy the same patch.

This model is designed to evaluate the effects of "selfishness" versus sharing on the population, age, and energy of turtles in a simple ecosystem.


## HOW IT WORKS

Each patch randomly produces energy.  When ENERGY is available, the patch turns green.  The amount of ENERGY the patch produces is determined by the ENERGY-FROM-RESOURCES slider.  The rate that patches produce energy is determined by the REPLENISH-RATE slider.

A number of turtles determined by the NUMBER slider are placed randomly on the starting screen and move in random directions, one patch at a time.  Each turtle starts with 10 ENERGY, and increases ENERGY by moving onto patches with energy (i.e., green patches).  Turles lose 1 unit of ENERGY each time they move. Turtles who reach a certain amount of ENERGY, set by the BIRTH-ENERGY slider, hatch a new turtle.  The turtle 'giving birth' loses an amount of ENERGY equal to the BIRTH-ENERGY, and the hatchling starts with the same amount of ENERGY.

Turtles and hatchlings start with AGE at 0.  Age advances by 1 unit each tick.  A turtle dies when its ENERGY reaches 0 or below, or when AGE is greater than OLDAGE, a number determined by the OLDAGE slider.

A SHARE? switch determines the color of the turtles and how resources are consumed or shared _when turtles occupy the same patch_.  

  * When SHARE? is set to _off_, turtles never share ENERGY and when more than one turtle occupies a green patch, one of the turtles consumes all of the available ENERGY from the patch. 

  * When SHARE? is set to _on_, turtles who occupy the same patch (green or black) share all of their energy equally between each other. 

  * To aid visual analysis, the colors of the turtles who do not occupy a patch with other turtles is set to orange when SHARE? is _off_ and yellow when SHARE? is _on_. Regardless of the SHARE? setting, turtles who occcupy a patch with other turtles are set to cyan.

## HOW TO USE IT

SETUP clears all settings, creates turtles, and resets the ticks to 0.

GO starts the simulation. 

Users can examine the affects of changing the following variables:

  * Change the **number of turtles** using the NUMBER slider.

  * Change the **amount of energy each green patch provides** by using the ENERGY-FROM-RESOURCES slider.

  * Change the **rate at which resources are replenished** by using the REPLENISH RATE slider.

  * Change the amount of energy required to **give birth** to a new turtle by using the BIRTH-ENERGY slider.

  * Change the **behavior of the turtles** by switching the SHARING? switch on or off.

Watch the effects of your manipulations using one or more of the many monitors and graphs provided.  These include

  * Monitors on the top of the screen that show the total number of turtles alive at a given moment, the total number of patches with available resources, the mean age of the turtles, the median age of the turtles, and highest age of the turtles at a given time, and the mean amount of energy, median amount of energy, and maximum amount of energy of the turtles at a given time.

  * To the right of the sliders on the bottom of the screen, three figures plot the total number of turtles and the total number of patches with resources, the mean age and the mean energy of the turtles, and the median age and median energy of the turtles.  At the ar right, a histogram shows the distribution of the age of the turtles.

## THINGS TO NOTICE

The model is set up to allow the user to quickly see the effects of sharing and not-sharing on turtle population, age, and energy.

Look closely at the effects of sharing across different initial conditions.  What is the effect on age?  Is it the same effect for energy?  What about population? 

A nice feature of the model is that sharing can be turned on and off as the model runs, allowing for an easy comparison of the effects during the same run.  For anyone wishing to demonstrate a reversal design, this model provides a "hands-on" demonstration. 

## THINGS TO TRY

Run the simulation with the SHARING? switch set to off for awhile.  Then flip the switch to on for about the same number of ticks.  Then back to off.  And so on.  Watch the figures closely, especially the mean age and energy and the median age and energy figures.

Start with all the sliders set to near the middle (the default setting).

Run the next simulation with all the sliders set to near minimum, say around 10 (with OLDAGE 20).

Then run the next smulation with all the sliders at maximum.  

After testing out these settings, have fun.  Be sure to see how changing the BIRTH-ENERGY slider affects the outcomes.

## EXTENDING THE MODEL

There are many ways to extend this model.  A few suggestions include the following:

  * Rather than turning sharing on and off for all turtles, create a breed of turtle that shares and another that does not.

  * In this model, when sharing occurs, all of the energy is summed and then divided equally among all the turtles on the same patch.  What effects would other methods of sharing have (e.g., giving energy only to the neediest, etc.)?

  * Create a breed of turtles that takes all the energy from other turtles occupying the same patch.

  * Extend the reach of sharing to neighboring turtles.


## NETLOGO FEATURES

The code for sharing is below:

`to eat-resources

  ask turtles [
    if pcolor = green [
      set pcolor black
      set energy (energy + (energy-from-resources))]

    ifelse share?
      [ set energy (sum [energy] of turtles-here / count turtles-here)
        set color yellow ]                     ;;all energy shared among all turtles occupying same patch
      [set color orange ]                      ;; otherwise keeps energy the same

    if count turtles-here > 1
    [ set color cyan ]
  ]
end`

## RELATED MODELS

Rabbits Grass Weeds and Wolf Sheep Predation are two interacting ecosystem models similar to this one.  See also the Altruism and Cooperate.


## CREDITS AND REFERENCES

If you mention the NetLogo software in a publication, we ask that you include the citation as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
