; INDISIM-Denitrification - NetLogo Implementation
; Araujo-Granda P.A., Gras A., Ginovart M. and Moulton V. - 2019

;______________________________ Global Variables __________________________________________________________________________________
extensions [matrix]

globals
[
  world                                                                      ; culture medium continuos size [ litres ]
  steptime                                                                   ; step time [h]
  time_now                                                                   ; simulated time [hours]
  gbacteria gdioxide gbicarbonate gn2o gno2- gno gn2 gsuccinate gamonium gnitrate goxygen ; global variables for graphical and numerical outputs
  output_file                                                               ; global variables for outputs files
  DST DSTb DSTr DSTm                                                        ; Variables for General Standard Deviation
  succinatemedium amoniummedium oxygenmedium nitratemedium nitritemedium monoxidemedium dinitrogenmedium nitrogenmedium dioxidemedium bicarbonatemedium
  Suc Amo Oxy Nrt Nit N2O
  succinate-input amonium-input nitrate-input
  xsc xam xox xntr xnr xmn xdn xn2 xco2 xhco3 bacteriaout xbacteria
  d
  untri
  isuccinate iamonium ioxygen initrate initrite imonoxide idinitrogen initrogen idioxide ibicarbonate ibacteria icbacteria inbacteria
  fsuccinate famonium foxygen fnitrate fnitrite fmonoxide fdinitrogen fnitrogen fdioxide fbicarbonate fbacteria fcbacteria fnbacteria
  insuccinate inamonium innitrate
  outsuccinate outamonium outoxygen outnitrate outnitrite outmonoxide outdinitrogen outnitrogen outdioxide outbicarbonate outbacteria outcbacteria outnbacteria
  c-bacteria n-bacteria c-succinate n-amonium n-nitrate n-nitrite n-monoxide n-dinitrogen n-nitrogen c-co2 c-hco3
  n-obtained n-delivered c-obtained c-delivered
  %_e_c %_e_n yc/c
  scm oxm ntrm
  q
  sc-mant2 sc-mant3 sc-mant4
  ini_biomass rep_biomass fmol_small fmol_big deq_small deq_big

  molecular-weight e-eeq/mol dGpc
  A1 fs1 fe1 yc/c1
  A2 fs2 fe2 yc/c2
  A3 fs3 fe3 yc/c3
  A4 fs4 fe4 yc/c4
  A5 fs5 fe5 yc/c5
  A6 fs6 fe6 yc/c6
  donor acceptor1 acceptor2 acceptor3 acceptor4 acceptor5 acceptor6 b-biomass
  pathway1 pathway2 pathway3 pathway4 pathway5 pathway6
  auto_consum_NO auto_consum_N2O auto_consum_NO2-
  rx_mant-aero rx_mant_anox1 rx_mant_anox2 rx_mant_anox3 rx_mant_anox4
  gamma_cells
  u_succ u_nh4+ u_no3-a u_no3-x u_no2- u_no u_n2o u_o2
  u u0 u1 u3 umax

  ; for Mahcine learning matrix procedure

  biomass_matrix GRI_biomass
  nitrate_matrix GRI_nitrate
  nitrite_matrix GRI_nitrite
  nitrous_oxide_matrix GRI_nitrous_oxide
  Fitness

  ]

; ____________________________ Microorganism variables ____________________________________________________________________________

breed [bacteria bacterium]                                                   ; microorganism definition
bacteria-own
[
  biomass                                                                    ; microorganism biomass [ mmol ]
  biomass-reproduction                                                       ; reproduction treshold biomass [ mmol ]
  ac_mn ac_dn
  R                                                                          ; reproductions counter
  m_aero s_aero_1 s_aero_2                                                   ; aerobic phase counter
  m_anox_1 m_anox_2 m_anox_3 m_anox_4 s_anox_1 s_anox_2 s_anox_3 s_anox_4    ; anaerobic reactions counters
  sc-useful am-useful ox-useful ntr-useful nr-useful mn-useful dn-useful     ; usefuls nutrients variables
  bm
]

; ____________________________ Culture Medium variables ____________________________________________________________________________

patches-own
  [
  succinate-medium                     ; culture medium variable for the electron donor and C-Source in aerobic and anaerobic phases [ mmol ]
  amonium-medium                       ; culture medium variable for the N-Source in aerobic and anaerobic phases [ mmol ]
  oxygen-medium                        ; culture medium variable for the electron aceptor in aerobic phase [ mmol ]
  nitrate-medium                       ; culture medium variable for the electron aceptor in anoxic reaction 1 [ mmol ]
  nitrite-medium                       ; culture medium variable for the electron aceptor in anoxic reaction 2 [ mmol ]
  monoxide-medium                      ; culture medium variable for the electron aceptor in anoxic reaction 3 [ mmol ]
  dinitrogen-medium                    ; culture medium variable for the electron aceptor in anoxic reaction 4 [ mmol ]
  dioxide nitrogen bicarbonate         ; culture medium variables for the denitrification products [ mmol ]
  ]

; ____________________________ Virtual Bioreactor Setting ____________________________________________________________________________

to setup
  clear-all                      ; clear all variables, graphical and numerical outputs
  reset-ticks                    ; reset tick counter
  setup-thermodynamics           ; setup thermodynamic coefficients accord to McCarty 2007, TEEM2
  setup-model                    ; setup general model parameters to obtain the starting points accord to Felgate's (2012) experiment
  setup-time                     ; scaled time setting
  setup-medium                   ; formulation of the initial culture medium
  setup-bacteria                 ; formulation of initial bacterial inoculum
  setup-monitors                 ; monitors setting
  if Output_file? = true
  [
  setup-output                   ; setup output file
  write-outputfile               ; write initial values on output file
  ]
  setup-balance                  ; mass balance setup
  set u0 gbacteria
  set u3 0
  setup-machine.learning
end
;___________________________________ Initial data according to Felgate's Experiments _____________________________________________________

to setup-model
  set world (world-width * world-height)                                                  ; Differential adjustment of reactor volume
  set DSTb 0.15                                                                           ; Standard Deviation for initial biomass distribution
  set DSTr 0.15                                                                           ; Standard Deviation for biomass reproduction
  set DST 0.05                                                                            ; Standard Deviation for general model varaibles
  set DSTm 0.25                                                                           ; Strandard Deviation for culture medium availability
  set nsc 0.28 set nam 0.84 set nx 0.79 set nntr 0.63 set nnr 0.77 set nmn 1 set ndn 0.49 ; Setup of culture medium nutrients availability obtanied from Dab

  if Denitrifying_bacterium = "P._denitrificans"
  [
  set deq_small abs random-normal (0.5) (DST * 0.5)                                       ; define the minimun bacterium size Bergey's reference taking into account the spherical shape
  set deq_big abs random-normal (0.9) (DST * 0.9)                                         ; define the maximun bacterium size Bergey's reference taking into account the spherical shape
  set fmol_small (((deq_small * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight    ; small bacteria expresed in fmol
  set fmol_big (((deq_big * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight        ; big bacteria expresed in fmol
  set ini_biomass 0.75 * fmol_big                                                         ; [fmol] initial biomass for each bacterium
  set rep_biomass 0.75 * fmol_big                                                         ; [fmol] reproduction treshold biomass for each bacterium

  if Experiment = "Succ_High/Nitrate_Low"                                                   ; setup for experiment C-source sufficient / electron-acceptor limited
  [
  Set Succinate 20 Set Nitrate 4.99825                                                    ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate High/Nitrate low)
  let paracoccus abs random-normal (0.0033683) (DSTr * 0.0033683)                         ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate High/Nitrate low)
  set Microrganism round ((paracoccus * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  if Experiment = "Succ_Low/Nitrate_High"                                                 ; setup for experiment C-source limited / electron-accpetor suficient
  [
  Set Succinate 5 set Nitrate 21.60951                                                    ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate Low/Nitrate high)
  let paracoccus abs random-normal (0.0668478) (DSTr * 0.0668478)                         ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate low/Nitrate high)
  set Microrganism round ((paracoccus * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  ]

  if Denitrifying_bacterium = "A._xylosoxidans"
  [
  set deq_small abs random-normal (0.63) (DST * 0.63)                                        ; define the minimun bacterium size Bergey's reference taking into account the spherical shape
  set deq_big abs random-normal (1.40) (DST * 1.40)                                          ; define the maximun bacterium size Bergey's reference taking into account the spherical shape
  set fmol_small (((deq_small * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight       ; small bacteria expresed in fmol
  set fmol_big (((deq_big * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight           ; big bacteria expresed in fmol
  set ini_biomass 0.75 * fmol_big                                                            ; [fmol] initial biomass for each bacterium
  set rep_biomass 0.75 * fmol_big                                                            ; [fmol] reproduction treshold biomass for each bacteirum

  if Experiment = "Succ_High/Nitrate_Low"                                                    ; setup for experiment C-source sufficient / electron-acceptor limited
  [
  Set Succinate 20 Set Nitrate 5.15375                                                       ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate High/Nitrate low)
  let achromobacter abs random-normal (0.07695) (DSTr * 0.07695)                             ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate High/Nitrate low)
  set Microrganism round ((achromobacter * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  if Experiment = "Succ_Low/Nitrate_High"                                                    ; setup for experiment C-source limited / electron-accpetor suficient
  [
  Set Succinate 5 set Nitrate 21.74694                                                       ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate Low/Nitrate high)
  let achromobacter abs random-normal (0.05182) (DSTr * 0.05182)                             ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate low/Nitrate high)
  set Microrganism round ((achromobacter * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  ]
end

; ___________________________________ Proceeding to Scaled time ____________________________________________________________________________

to setup-time
  set steptime (Min/steptime / 60)       ; User define the number of minute for each step time using a slider labeled "Min_per_steptime"
end

; ___________________________________ Proceeding to Formulation of Initial culture medium __________________________________________________

to setup-medium
    if Denitrifying_bacterium = "P._denitrificans"
    [
     if Experiment = "Succ_High/Nitrate_Low"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.0255) (DST * 0.0255)                 ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.003003178) (DST * 0.003003178)       ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     if Experiment = "Succ_Low/Nitrate_High"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.011163522) (DST * 0.011163522)       ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.000028413312) (DST * 0.000028413312) ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     ]

    if Denitrifying_bacterium = "A._xylosoxidans"
    [
     if Experiment = "Succ_High/Nitrate_Low"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.36863) (DST * 0.36863)               ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.00006263) (DST * 0.00006263)         ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     if Experiment = "Succ_Low/Nitrate_High"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.00765) (DST * 0.00765)               ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.00001818) (DST * 0.00001818)         ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     ]

   ask patches
   [
     set succinate-medium abs random-normal (Suc) (DST * Suc)            ; Assign the succinate concentration to each cell according to the model's standar deviation
     set amonium-medium abs random-normal (Amo) (DST * Amo)              ; Assign the amonium concentration to each cell according to the model's standar deviation
     set oxygen-medium  abs random-normal (Oxy) (DST * Oxy)              ; Assign the oxygen concentration to each cell according to the model's standar deviation
     set nitrate-medium abs random-normal (Nrt) (DST * Nrt)              ; Assign the nitrate concentration to each cell according to the model's standar deviation
     set nitrite-medium abs random-normal (Nit) (DST * Nit)              ; Assign the nitrite concentration to each cell according to the model's standar deviation
     set dinitrogen-medium abs random-normal (N2O) (DST * N2O)           ; Assign the nitrous oxide concentration to each cell according to the model's standar deviation
     set monoxide-medium 0                                                ; Nitric Oxide initial concentration accord to Felgate's (2012) experiment [mM]
     set nitrogen 0                                                       ; Nitrogen gas (N2) initial concentration accord to Felgate's (2012) experiment [mM]
     set dioxide 0                                                        ; Carbon Dioxide initial concentration accord to Felgate's (2012) experiment [mM]
     set bicarbonate 0                                                    ; Bicarbonate initial concentration [mM]
     set pcolor scale-color blue succinate-medium 0 Succinate
   ]

end

; ____________________________ Proceeding to Formulation the Initial bacterial inoculum __________________________________________________

to setup-bacteria
 create-bacteria Microrganism                                                    ; User define the amount of initial active bacteria using a slider labaled Microrganism
 [
   if Denitrifying_bacterium = "A._xylosoxidans" [set shape "cylinder"]          ; Achrobobacter xylosoxidans is a rod bacteria
   if Denitrifying_bacterium = "P._denitrificans" [set shape "circle"]           ; Paracoccus denitrificans is a spherical bacteria
   set size 0.5                                                                  ; This size is for visual propouse only
   set color green                                                               ; This color is for visual propouse only
   setxy random-xcor random-ycor                                                 ; Random inicial position on culture medium
   set biomass abs random-normal (ini_biomass) (DSTb * ini_biomass)              ; Initial biomass for each microorganism [fmol] according to model's standar deviation
   set biomass-reproduction abs random-normal (rep_biomass) (DSTr * rep_biomass) ; reproduction treshold biomass [fmol] accord to INDISIM general procedures.
 ]
end

; ____________________________ Proceeding to monitors setting __________________________________________________________________________

to setup-monitors
  set time_now ticks * steptime                                                                          ; Simulated time [ h ]
  set gbacteria ((sum [biomass] of bacteria / world) / 1000) * molecular-weight                          ; Biomass concentration [ mg/ml ]
  set gsuccinate (sum[succinate-medium]of patches / world)                                               ; Succinate concentration [ mM ]
  set gamonium (sum [amonium-medium] of patches / world)                                                 ; Amonium concentration [ mM ]
  set goxygen (sum [oxygen-medium] of patches / world)                                                   ; Oxygen concentration [ mM ]
  set gnitrate (sum [nitrate-medium] of patches / world)                                                 ; Nitrate concentration [ mM ]
  set gno2- (sum [nitrite-medium] of patches / world)                                                    ; Nitrite concentration [ mM ]
  set gno (sum [monoxide-medium] of patches / world)                                                     ; Nitric Oxide concentration [ mM ]
  if Experiment = "Succ_Low/Nitrate_High"[set gn2o (sum [dinitrogen-medium] of patches / world) * 1000]  ; Nitrous Oxide concentration [ uM ]
  if Experiment = "Succ_High/Nitrate_Low"[set gn2o (sum [dinitrogen-medium] of patches / world)]         ; Nitrous Oxide concentration [ mM ]
  set gdioxide (sum [dioxide] of patches / world)                                                        ; Carbon dioxide concentration [ mM ]
  set gn2 (sum [nitrogen] of patches / world)                                                            ; Nitrogen concentration [ mM ]
  set gbicarbonate (sum [bicarbonate] of patches / world)                                                ; Bicarbonate concentration [ mM ]

  reactor_balance
end

; ____________________________ Proceeding to setup output file __________________________________________________________________________

to setup-output
  file-close
  if Experiment = "Succ_Low/Nitrate_High"
  [
  if Denitrifying_bacterium = "P._denitrificans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_PD_I_Denitrification_V10_ScLw_NtHg_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  if Denitrifying_bacterium = "A._xylosoxidans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_AX_I_Denitrification_V10_ScLw_NtHg_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  ]
  if Experiment = "Succ_High/Nitrate_Low"
  [
  if Denitrifying_bacterium = "P._denitrificans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_PD_I_Denitrification_V10_ScHg_NtLw_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  if Denitrifying_bacterium = "A._xylosoxidans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_AX_I_Denitrification_V10_ScHg_NtLw_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  ]
  carefully
  [file-delete output_file] [ ]
  file-open output_file
  file-type "Time_[h] "                                                  ; Simulation time labed in outputfile unit hours
  file-type "Succinate_[mM] "                                            ; Succinate concentration labed in outputfile unit milimolar
  file-type "NH4+_[mM] "                                                 ; Amonium concentration labed in outputfile unit milimolar
  file-type "O2_[mg/l] "                                                 ; Oxygen concentration labed in outputfile unit mg/ml
  file-type "NO3-_[mM] "                                                 ; Nitrate concentration labed in outputfile unit milimolar
  file-type "NO2-_[mM] "                                                 ; Nitrite concentration labed in outputfile unit milimolar
  file-type "NO_[mM] "                                                   ; Nitric Oxide concentration labed in outputfile unit milimolar
  if Experiment = "Succ_Low/Nitrate_High" [file-type "N2O_[uM] "]        ; Dinitrogen concentration labed in outputfile unit micromolar
  if Experiment = "Succ_High/Nitrate_Low" [file-type "N2O_[mM] "]        ; Dinitrogen concentration labed in outputfile unit milimolar
  file-type "N2_[mM] "                                                   ; Nitrogen concentration labed in outputfile unit milimolar
  file-type "CO2_[mM] "                                                  ; Carbon dioxide concentration labed in outputfile unit milimolar
  file-type "Biomass_[mg/ml] "                                           ; Biomass concentration labed in outputfile unit miligrams / mililiter
  file-type "Individuals "                                               ; Individuals into bioreactor labed in outputfile
  file-type "d "                                                         ; Reactor dilution ratio labed in outputfile units (h^-1)
  file-type "Sc_In_[mmol] "                                              ; Inflow Succinate labed in outputfile units milimoles
  file-type "NH4+_In_[mmol] "                                            ; Inflow Amonium labed in outputfile units milimoles
  file-type "NO3-_In_[mmol] "                                            ; Inflow Nitrate labed in outputfile units milimoles
  file-type "Sc_Out_[mmol] "                                             ; Outflow Succinate labed in outputfile units milimoles
  file-type "NH4+_Out_[mmol] "                                           ; Outflow amonium labed in outputfile units milimoles
  file-type "NO3-_Out_[mmol] "                                           ; Outflow  nitrate labed in outputfile units milimoles
  file-type "NO2-_Out_[mmol] "                                           ; Outflow nitrite labed in outputfile units milimoles
  file-type "NO_Out_[mmol] "                                             ; Outflow Nitric Oxide labed in outputfile units milimoles
  if Experiment = "Succ_Low/Nitrate_High" [file-type "N2O_Out_[umol] "]  ; Outflow Nitrous Oxide labed in outputfile units micromoles
  if Experiment = "Succ_High/Nitrate_Low" [file-type "N2O_Out_[mmol] "]  ; Outflow Nitrous oxide labed in outputfile units milimoles
  file-type "N2_Out_[mmol] "                                             ; Outflow Nitrogen gas labed in outputfile units milimoles
  file-type "Ind_Out "                                                   ; Outflow Microorganims labed in outputfile
  file-type "HCO3_[mM] "                                                 ; Bicarbonate concentration labed in outputfile units milimolar
  file-type "HCO3_Out_[mmol] "                                           ; Outflow Bicarbonate labed in outputfile units milimoles
  file-type "CO2_Out_[mmol] "                                            ; Outflow Carbon dioxide labed in outputfile units milimoles
  file-print " "
end

; ____________________________ Proceeding to write values on output file __________________________________________________________________________

to write-outputfile
  setup-monitors
  file-type time_now                                            ; Outputfile Simulated time [h]
  file-type " "
  file-type gsuccinate                                          ; Outputfile Bioreactor Succinate concentration [mM]
  file-type " "
  file-type gamonium                                            ; Outputfile Bioreactor Amonium concentration [mM]
  file-type " "
  file-type goxygen * 32                                        ; Outputfile Bioreactor Oxygen concentration [mg/l]
  file-type " "
  file-type gnitrate                                            ; Outputfile Bioreactor Nitrate concentration [mM]
  file-type " "
  file-type gno2-                                               ; Outputfile Bioreactor Nitrite concentration [mM]
  file-type " "
  file-type gno                                                 ; Outputfile Bioreactor Nitric Oxide concentration [mM]
  file-type " "
  file-type gn2o                                                ; Outputfile Bioreactor Nitrous Oxide concentration [mM][uM]
  file-type " "
  file-type gn2                                                 ; Outputfile Bioreactor Nitrogen gas concentration [mM]
  file-type " "
  file-type gdioxide                                            ; Outputfile Bioreactor Carbon Dioxide concentration [mM]
  file-type " "
  file-type gbacteria                                           ; Outputfile Bioreactor Biomass concentration [mg/ml]
  file-type " "
  file-type count bacteria                                      ; Outputfile Bioreactor microorganism count [individuals]
  file-type " "
  file-type d                                                   ; Outputfile Bioreactor Dilution ratio [h^1]
  file-type " "
  file-type succinate-input                                     ; Outputfile Inflow Succinate quantity [mmol]
  file-type " "
  file-type amonium-input                                       ; Outputfile Inflow Amonioum quantity [mmol]
  file-type " "
  file-type nitrate-input                                       ; Outputfile Inflow Nitrate quantity [mmol]
  file-type " "
  file-type xsc                                                 ; Outputfile Outflow Succinate quantity [mmol]
  file-type " "
  file-type xam                                                 ; Outputfile Outflow Amonium quantity [mmol]
  file-type " "
  file-type xntr                                                ; Outputfile Outflow Nitrate quantity [mmol]
  file-type " "
  file-type xnr                                                 ; Outputfile Outflow Nitrite quantity [mmol]
  file-type " "
  file-type xmn                                                 ; Outputfile Outflow Nitric oxide quantity [mmol]
  file-type " "
  if Experiment = "Succ_Low/Nitrate_High"[file-type xdn * 1000] ; Outputfile Outflow Nitrous oxide quantity [umol]
  if Experiment = "Succ_High/Nitrate_Low"[file-type xdn]        ; Outputfile Outflow Nitrous oxide quantity [mmol]
  file-type " "
  file-type xn2                                                 ; Outputfile Outflow Nitrogen gas quantity [mmol]
  file-type " "
  file-type bacteriaout                                         ; Outputfile Outflow microrganism quantity [individuals]
  file-type " "
  file-type gbicarbonate                                        ; Outputfile Bioreactor Bicarbonate concentration [mM]
  file-type " "
  file-type xhco3                                               ; Outputfile Outflow bicarbonate quantity [mmol]
  file-type " "
  file-type xco2                                                ; Outputfile Outflow Carbon Dioxide quantity [mmol]
  file-type " "
  file-print " "
end

; ____________________________ General mass balance proceedings _____________________________________________________________________________

to setup-balance
  set ibacteria (sum [biomass] of bacteria)                                                ; Initial Biomass [mmol]
  set isuccinate (sum[succinate-medium]of patches)                                         ; Initial Succinate [mmol]
  set iamonium (sum [amonium-medium] of patches)                                           ; Initial Amonium [mmol]
  set ioxygen (sum [oxygen-medium] of patches)                                             ; Initial Oxygen [mmol]
  set initrate (sum [nitrate-medium] of patches)                                           ; Initial Nitrate [mmol]
  set initrite (sum [nitrite-medium] of patches)                                           ; Initial Nitrite [mmol]
  set imonoxide (sum [monoxide-medium] of patches)                                         ; Initial Monoxide [mmol]
  set idinitrogen (sum [dinitrogen-medium] of patches)                                     ; Initial Dinitrogen [mmol]
  set idioxide (sum [dioxide] of patches)                                                  ; Initial Carbon dioxide [mmol]
  set initrogen (sum [nitrogen] of patches)                                                ; Initial Nitrogen [mmol]
  set ibicarbonate (sum [bicarbonate] of patches)                                          ; Initial Bicarbonate [mmol]
end

to reactor_balance
  set fbacteria (gbacteria * 1000 * world / molecular-weight)                              ; Final Biomass [mmol]
  set fsuccinate (gsuccinate * world)                                                      ; Final Succinate [mmol]
  set famonium (gamonium * world)                                                          ; Final Amonium [mmol]
  set foxygen (goxygen * world)                                                            ; Final Oxygen [mmol]
  set fnitrate (gnitrate * world)                                                          ; Final Nitrate [mmol]
  set fnitrite (gno2- * world)                                                             ; Final Nitrite [mmol]
  set fmonoxide (gno * world)                                                              ; Final Monoxide [mmol]
  if Experiment = "Succ_Low/Nitrate_High"[set fdinitrogen ((gn2o * world) / 1000)]         ; Final Dinitrogen [mmol]
  if Experiment = "Succ_High/Nitrate_Low"[set fdinitrogen (gn2o * world)]                  ; FInal Dinitrogen [mmol]
  set fdioxide (gdioxide * world)                                                          ; Final Carbon dioxide [mmol]
  set fnitrogen (gn2 * world)                                                              ; Final Nitrogen [mmol]
  set fbicarbonate (gbicarbonate * world)                                                  ; Final Bicarbonate [mmol]
end

to inflow_balance
  set insuccinate (insuccinate + succinate-input)                                          ; Inflow Succinate [mmol]
  set inamonium (inamonium + amonium-input)                                                ; Inflow Amonium [mmol]
  set innitrate (innitrate + nitrate-input)                                                ; Inflow Nitrate [mmol]
end

to outflow_balance
  set outsuccinate (outsuccinate + xsc)                                                    ; OutFlow Succinate [mmol]
  set outamonium (outamonium + xam)                                                        ; OutFlow Amonium [mmol]
  set outoxygen (outoxygen + xox)                                                          ; OutFlow Oxygen [mmol]
  set outnitrate (outnitrate + xntr)                                                       ; Outflow Nitrtate [mmol]
  set outnitrite (outnitrite + xnr)                                                        ; Outflow Nitrite [mmol]
  set outmonoxide (outmonoxide + xmn)                                                      ; Outflow Monoxide [mmol]
  set outdinitrogen (outdinitrogen + xdn)                                                  ; Outflow N2O [mmol]
  set outnitrogen (outnitrogen + xn2)                                                      ; Outflow N2 [mmol]
  set outdioxide (outdioxide + xco2)                                                       ; Outflow CO2 [mmol]
  set outbicarbonate (outbicarbonate + xhco3)                                              ; outflow HCO3 [mmol]
  set outbacteria (outbacteria + xbacteria)                                                ; OutFlow Baacteria [mmol]
end

to general_mass_balance
  set icbacteria (ibacteria * carbon * 12.011)                                             ; Initial C-Mic [mg]
  set inbacteria (ibacteria * b-nitrogen * 14)                                             ; Initial N-Mic [mg]
  set fcbacteria (fbacteria * carbon * 12.011)                                             ; Final C-Mic [mg]
  set fnbacteria (fbacteria * b-nitrogen * 14)                                             ; Final N-Mic [mg]
  set outcbacteria (outbacteria * carbon * 12.011)                                         ; OutFlow C-Mic [mg]
  set outnbacteria (outbacteria * b-nitrogen * 14)                                         ; OutFlow N-Mic [mg]

  set c-bacteria (fcbacteria - icbacteria) - (0 - outcbacteria)                            ; Microorganism Carbon balance
  set n-bacteria (fnbacteria - inbacteria) - (0 - outnbacteria)                            ; Microorganims Nitrogen balance
  set c-succinate ((fsuccinate - isuccinate) - (insuccinate - outsuccinate)) * 4 * 12.011  ; Succinate carbon balance
  set n-amonium ((famonium - iamonium) - (inamonium - outamonium)) * 1 * 14                ; Amonium nitrogen balance
  set n-nitrate ((fnitrate - initrate) - (innitrate - outnitrate)) * 1 * 14                ; Nitrate nitrogen balance
  set n-nitrite ((fnitrite - initrite) - (0 - outnitrite)) * 1 * 14                        ; Nitrite nitrogen balance
  set n-monoxide ((fmonoxide - imonoxide) - (0 - outmonoxide)) * 1 * 14                    ; Nitric oxide nitrogen balance
  set n-dinitrogen ((fdinitrogen - idinitrogen) - (0 - outdinitrogen)) * 2 * 14            ; Nitrous oxide nitrogen balance
  set n-nitrogen ((fnitrogen - initrogen) - (0 - outnitrogen)) * 2 * 14                    ; Nitrogen gas nitrogen balance
  set c-co2 ((fdioxide - idioxide) - (0 - outdioxide)) * 1 * 12.011                        ; Carbon dixoide carbon balance
  set c-hco3 ((fbicarbonate - ibicarbonate) - (0 - outbicarbonate)) * 1 * 12.011           ; Bicarbonate carbon balance

  set n-obtained abs (n-nitrite + n-monoxide + n-dinitrogen + n-nitrogen + n-bacteria)     ; Total nitrogen obtained
  set n-delivered abs (n-amonium + n-nitrate)                                              ; Total nitrogen delivered
  set c-obtained abs (c-co2 + c-hco3 + c-bacteria)                                         ; Total carbon obtained
  set c-delivered abs (c-succinate)                                                        ; Total carbon delivered

  set %_e_c abs ((c-delivered - c-obtained) / c-delivered ) * 100                          ; Carbon general mass balance porcentual error
  set %_e_n abs ((n-delivered - n-obtained) / n-delivered ) * 100                          ; Nitrogen general mass balance porcentual error
  set yc/c abs (c-bacteria / c-succinate)                                                  ; Population Yield growth
end

; ____________________________ Run Proceeding to Virtual Bioreactor __________________________________________________________________________

to go
  tick                                                                 ; Start tick counter
  stir                                                                 ; Observer procedure to stablish quantities for outflow in batch and continuos culture.
  outflow                                                              ; Observer and patch procedure to make outflow from bioreactor in continuos culture with or without culture medium homogenization
  inflow                                                               ; Observer procedure to simulate input nutrients to bioreactor according to dilution ratio with or without culture medium homogenization
  maintenance-requirements                                             ; Observer procedure to calculate maintenance requirements for aerobic and anaerobic phases.

  ask bacteria
  [
    uptake                                                             ; Agent procedure, each time step the bacteria establish their uptake capacities of all nutrients
    ifelse oxygen-medium >= abs random-normal NOX (DSTr * NOX)          ; Agent procedure, according to oxygen local concentration runs different metabolic pathways
    [mant-aerobic][mant-anoxic1]
    auto_consum
    metabolism_end                                                     ; Agent procedure, after cellular maintenance and biomass synthesis the uptaked nutrient quantities are updated
    bipartition                                                        ; Agent procedure, after execute metabolic pathways if the biomass reached the treshold valeu microorganism reproduce by bipartition
  ]

  ask bacteria [ move ]                                                ; Agent procedure, effect produced by the bioreactor agitation

  do-plotting                                                          ; Observer procedure, to setup graphical outputs
  if Output_file? = true
  [
;    if
;    time_now = 5 or
;    time_now = 6 or
;    time_now = 24 or
;    time_now = 24.5 or
;    time_now = 25 or
;    time_now = 30 or
;    time_now = 48 or
;    time_now = 54 or
;    time_now = 72 or
;    time_now = 78 or
;    time_now = 97 or
;    time_now = 99 or
;    time_now = 102 or
;    time_now = 120
;   [
      write-outputfile
      ]
 ; ]                            ; Observer procedure, write-out the all simulated data to output file.

  general_mass_balance
  to-umax

  get_data_machine.learning

  if time_now >= Simulation_Time [machine.learning file-close stop]                     ; Stop simulation if time is over and close output-file
  if (sum [succinate-medium] of patches / world) < 0 [machine.learning file-close stop] ; Stop simulation if the succinate global concentration is lower to zero and close output-file
  if (sum [amonium-medium] of patches / world) < 0 [machine.learning file-close stop]   ; Stop simulation if the amonium global concentration is lower to zero and close output-file
  if (sum [oxygen-medium] of patches / world) < 0 [machine.learning file-close stop]    ; Stop simulation if the oxygen global concentration is lower to zero and close output-file
  if (sum [nitrate-medium] of patches / world) < 0 [machine.learning file-close stop]   ; Stop simulation if the nitrate global concentration is lower to zero and close output-file
end

; ____________________________ Proceeding to simulate bioreactor agitation over the culture medium ____________________________________________________

to stir
  set succinatemedium (sum [succinate-medium] of patches)   ; Total Succinate in culture medium [mmol]
  set amoniummedium (sum [amonium-medium] of patches)       ; Total Amonium in culture medium [mmol]
  set oxygenmedium (sum [oxygen-medium] of patches)         ; Total Oxygen in culture medium [mmol]
  set nitratemedium (sum [nitrate-medium] of patches)       ; Total Nitrate in culture medium [mmol]
  set nitritemedium (sum [nitrite-medium] of patches)       ; Total Nitrite in culture medium [mmol]
  set monoxidemedium (sum [monoxide-medium] of patches)     ; Total Nitric oxide in culture medium [mmol]
  set dinitrogenmedium (sum [dinitrogen-medium] of patches) ; Total Nitrous oxide in culture medium [mmol]
  set nitrogenmedium (sum [nitrogen] of patches)            ; Total nitrogen gas in culture medium [mmol]
  set dioxidemedium (sum [dioxide] of patches)              ; Total carbon dioxide in culture medium [mmol]
  set bicarbonatemedium (sum [bicarbonate] of patches)      ; Total bicarbonate in culture medium [mmol]

  ifelse time_now >= Time_Shutdown_O2                       ; if simulated time is lower than time_shutdown_O2
  [set d abs random-normal (di) (DST * di)]                 ; Bioreactor works in continuos culture according to dilution ratio
  [
    set d 0                                                 ; the bioreactor works in batch conditions
    if oxygenmedium > 0.2360 * world
    [set oxygenmedium 0.2360 * world]                       ; and the oxygen keep saturated conditions
  ]
  set xsc d * succinatemedium * steptime                    ; succinate outflow according to dilution ratio [mmol]
  set xam d * amoniummedium * steptime                      ; amonium outflow according to dilution ratio [mmol]
  set xox d * oxygenmedium * steptime                       ; oxygen outflow according to dilution ration [mmol]
  set xntr d * nitratemedium * steptime                     ; nitrate outflow according to dilution ration [mmol]
  set xnr d * nitritemedium * steptime                      ; nitrite outflow according to dilution ration [mmol]
  set xmn d * monoxidemedium * steptime                     ; nitric oxide outflow according to dilution ration [mmol]
  set xdn d * dinitrogenmedium * steptime                   ; nitrous oxide outflow according to dilution ration [mmol]
  set xn2 d * nitrogenmedium * steptime                     ; nitrogen gas outflow according to dilution ration [mmol]
  set xco2 d * dioxidemedium * steptime                     ; carbon dioxide outflow according to dilution ration [mmol]
  set xhco3 d * bicarbonatemedium * steptime                ; bicarbonate outflow according to dilution ration [mmol]
  if xmn >= (3.27 * world) [set xmn (3.27 * world)]         ; if nitric oxide quantity is higher than maximun saturated conditions outflow only the saturated conditions
  if xdn >= (58.27 * world) [set xdn (58.27 * world)]       ; if nitrous oxide quantity is higher than maximun saturated conditions outflow only the saturated conditions
  if xn2 >= (104.91 * world) [set xn2 (104.91 * world)]     ; if nitrogen gas quantity is higher than maximun saturated conditions outflow only the saturated conditions
  if xco2 >= (8009.05 * world) [set xco2 (8009.05 * world)] ; if carbon dioxide quantity is higher than maximun saturated conditions outflow only the saturated conditions

  let bactout (sum [biomass] of bacteria)                   ; Total biomass in culture medium [mmol]
  let xbact (d * bactout * steptime)                        ; biomass outflow according to dilution ratio [mmol]
  let prom (count bacteria / bactout)                       ; average individual per biomass in outflow [individual ml /mg]
  set bacteriaout round (xbact * prom)                      ; Microorganism quantity to outflow [individuals]
  let monp (bactout / count bacteria)                       ; average biomass per individual in outflow [mg / ml individual]
  set xbacteria (monp * bacteriaout)                        ; Microorganism biomass concentration to outflow [mmol]

  outflow_balance
end

; ____________________________ Proceeding to simulate reactor OutFlow ________________________________________________________________________________________

to outflow
  set q random 2
  ifelse q = 0 [
   ask patches [
    set succinate-medium (succinatemedium - xsc) / world      ; The succinate outflow quantity is taken from global quantity and then homogenization occurs.
    set amonium-medium (amoniummedium - xam) / world          ; The amonium outflow quantity is taken from global quantity and then homogenization occurs.
    set oxygen-medium (oxygenmedium) / world                  ; Agitation and homogenization of oxygen
    set nitrate-medium (nitratemedium - xntr) / world         ; The nitrate outflow quantity is taken from global quantity and then homogenization occurs.
    set nitrite-medium (nitritemedium - xnr) / world          ; The nitrite outflow quantity is taken from global quantity and then homogenization occurs.
    set monoxide-medium (monoxidemedium - xmn) / world        ; The nitric oxide outflow quantity is taken from global quantity and then homogenization occurs.
    set dinitrogen-medium (dinitrogenmedium - xdn) / world    ; The nitrous oxide outflow quantity is taken from global quantity and then homogenization occurs.
    set nitrogen (nitrogenmedium - xn2) / world               ; The nitrogen gas outflow quantity is taken from global quantity and then homogenization occurs.
    set dioxide (dioxidemedium - xco2 ) / world               ; The carbon dioxide outflow quantity is taken from global quantity and then homogenization occurs.
    set bicarbonate (bicarbonatemedium - xhco3) / world       ; The bicarbonate outflow quantity is taken from global quantity and then homogenization occurs.
    set pcolor scale-color blue succinate-medium 0 Succinate
    ]
  ]
  [
    ask patches [
    set succinate-medium succinate-medium - (xsc / world)     ; The succinate outflow concentration is taken from local quantity without homogenization.
    set amonium-medium amonium-medium - (xam / world)         ; The amonium outflow concentration is taken from local quantity without homogenization.
    set oxygen-medium (oxygenmedium) / world                  ; Agitation and homogenization of oxygen
    set nitrate-medium nitrate-medium - (xntr / world)        ; The nitrate outflow concentration is taken from local quantity without homogenization.
    set nitrite-medium nitrite-medium - (xnr / world)         ; The nitrite outflow concentration is taken from local quantity without homogenization.
    set monoxide-medium monoxide-medium - (xmn / world)       ; The nitric oxide outflow concentration is taken from local quantity without homogenization.
    set dinitrogen-medium dinitrogen-medium - (xdn / world)   ; The nitrous oxide outflow concentration is taken from local quantity without homogenization.
    set nitrogen nitrogen - (xn2 / world)                     ; The nitrogen gas outflow concentration is taken from local quantity without homogenization.
    set dioxide dioxide - (xco2 / world)                      ; The carbon dioxide outflow concentration is taken from local quantity without homogenization.
    set bicarbonate bicarbonate - (xhco3 / world)             ; The bicarbonate outflow concentration is taken from local quantity without homogenization.
    set pcolor scale-color blue succinate-medium 0 Succinate
    ]
  ]
  ask n-of bacteriaout bacteria [ die ]                       ; Choose random bacteria and outflow accord to reactor dilution ratio
end

; __________________________________ Proceeding to simulate reactor Inflow ____________________________________________________________________________________

to inflow
  set succinate-input d * world * (isuccinate / world) * steptime                        ; Succinate inflow to continuos culture according to the slider labed succinate [mmol]
  set amonium-input d * world * (iamonium / world) * steptime                            ; Amonium inflow to continuos culture according to the slider labed amonium [mmol]
  set nitrate-input d * world * (initrate / world) * steptime                            ; Nitrate inflow to continuos culture according to the slider labed nitrate [mmol]

  set succinatemedium (sum [succinate-medium] of patches) + (succinate-input) ; Total Succinate in culture medium after inflow [mmol]
  set amoniummedium (sum [amonium-medium] of patches) + (amonium-input)       ; Total Amonium in culture medium after inflow [mmol]
  set nitratemedium (sum [nitrate-medium] of patches) + (nitrate-input)       ; Total Nitrate in culture medium after inflow [mmol]
  if time_now <= Time_Shutdown_O2 [set oxygenmedium 0.2360 * world]

  inflow_balance

  ifelse q = 0
  [ ask patches
   [ set succinate-medium succinatemedium / world                                ; The succinate inflow quantity is added to global quantity and then homogenization occurs.
     set amonium-medium amoniummedium / world                                    ; The amonium inflow quantity is added to global quantity and then homogenization occurs.
     set oxygen-medium oxygenmedium / world                                      ; The oxygen inflow quantity is added to global quantity and then homogenization occurs.
     set nitrate-medium nitratemedium / world                                    ; The nitrate inflow quantity is added to global quantity and then homogenization occurs.
     set pcolor scale-color blue succinate-medium 0 Succinate
   ]
  ]
  [ ask patches
   [ set succinate-medium succinate-medium + (succinate-input / world)           ; The succinate inflow concentration is added to local quantity without homogenization.
     set amonium-medium amonium-medium + (amonium-input / world)                 ; The amonium inflow concentration is added to local quantity without homogenization.
     set oxygen-medium oxygenmedium / world
     set nitrate-medium nitrate-medium + (nitrate-input / world)                 ; The nitrate inflow concentration is added to local quantity without homogenization.
     set pcolor scale-color blue succinate-medium 0 Succinate
    ]
  ]
end

; ____________________________ Maintenance Requirements for aerobic and anaerobic phases __________________________________________________________________________

to maintenance-requirements

ifelse time_now <= Time_Shutdown_O2
  [                                                                                       ; 1/14 (C4H4O4)2- + 1/4 O2 -> 1/7 CO2 + 1/7 HCO3- + 1/14 H2O
    set scm (energy_maintenance * steptime  * carbon / 4)                                 ; [mol succinate / molBiomass]
    set oxm (scm * abs (matrix:get rx_mant-aero 0 0) / abs (matrix:get rx_mant-aero 0 8)) ; [mol oxygen / molBiomass ]
  ]
  [                                                                                          ; 1/14 (C4H4O4)2- + 1/2 NO3- -> 1/7 CO2 + 1/7 HCO3- + 1/2 NO2- + 1/14 H2O
    set scm (energy_maintenance * steptime * carbon / 4)                                     ; [mol succinate / molBiomass ]
    set ntrm (scm * abs (matrix:get rx_mant_anox1 0 0) / abs (matrix:get rx_mant_anox1 0 8)) ; [mol nitrate / molBiomass ]
  ]
end

; ____________________________ Uptake proceeding according the core model INDISM __________________________________________________________________________

to uptake
  let uptsc (abs (random-normal (uSc) (DSTr * uSc)) * biomass * carbon * steptime)              ; [mmo] succinate by biomass uptake accord to INDISIM
  let uptam (abs (random-normal (uam) (DSTr * uam)) * biomass * carbon * steptime)              ; [mmol] amonium by biomass uptake accord to INDISIM
  let uptox (abs (random-normal (uox) (DSTr * uox)) * biomass * carbon * steptime)              ; [mmol] oxygen by biomass uptake accord to INDISIM
  ifelse time_now <= Time_Shutdown_O2 [set untri uNO3-a][set untri uNO3-x]                      ; The microorganism takes different uptake coefficient for aerobic and anaerobic phase.
  let uptntr (abs (random-normal (untri) (DSTr * untri)) * biomass * carbon * steptime)         ; [mmol] nitrate by biomass uptake accord to INDISIM
  let uptnr (abs (random-normal (uNO2-) (DSTr * uNO2-)) * biomass * carbon * steptime)          ; [mmol] nitrite by biomass uptake accord to INDISIM
  let uptmn (abs (random-normal (uNO) (DSTr * uNO)) * biomass * carbon * steptime)              ; [mmol] nitric oxide by biomass uptake accord to INDISIM
  let uptdn (abs (random-normal (uN2O) (DSTr * uN2O)) * biomass * carbon * steptime)            ; [mmol] nitrous oxide by biomass uptake accord to INDISIM

  let sc-available (abs (random-normal (nsc) (DSTm * nsc)) * succinate-medium * k_avail)  ; INDISIM lineal model to determine succinate availability in culture medium [mmol]
  let am-available (abs (random-normal (nam) (DSTm * nam)) * amonium-medium * k_avail)    ; INDISIM lineal model to determine amonium availability in culture medium [mmol]
  let ox-available (abs (random-normal (nx) (DSTm * nx)) * oxygen-medium * k_avail)       ; INDISIM lineal model to determine oxygen availability in culture medium [mmol]
  let ntr-available (abs (random-normal (nntr) (DSTm * nntr)) * nitrate-medium * k_avail) ; INDISIM lineal model to determine nitrate availability in culture medium [mmol]
  let nr-available (abs (random-normal (nnr) (DSTm * nnr)) * nitrite-medium * k_avail)    ; INDISIM lineal model to determine nitrite availability in culture medium [mmol]
  let mn-available (abs (random-normal (nmn) (DSTm * nmn)) * monoxide-medium * k_avail)   ; INDISIM lineal model to determine nitric oxide availability in culture medium [mmol]
  let dn-available (abs (random-normal (ndn) (DSTm * ndn)) * dinitrogen-medium * k_avail) ; INDISIM lineal model to determine nitrous oxide availability in culture medium [mmol]

  ifelse sc-available <= uptsc [set sc-useful sc-available] [set sc-useful uptsc]       ; For succinate comparation between uptake by biomass and availability and takes the lowest value.
  ifelse am-available <= uptam [set am-useful am-available] [set am-useful uptam]       ; For amonium comparation between uptake by biomass and availability and takes the lowest value.
  ifelse ox-available <= uptox [set ox-useful ox-available] [set ox-useful uptox]       ; For oxygen comparation between uptake by biomass and availability and takes the lowest value.
  ifelse ntr-available <= uptntr [set ntr-useful ntr-available] [set ntr-useful uptntr] ; For nitrate comparation between uptake by biomass and availability and takes the lowest value.
  ifelse nr-available <= uptnr [set nr-useful nr-available] [set nr-useful uptnr]       ; For nitrite comparation between uptake by biomass and availability and takes the lowest value.
  ifelse mn-available <= uptmn [set mn-useful mn-available] [set mn-useful uptmn]       ; For nitric oxide comparation between uptake by biomass and availability and takes the lowest value.
  ifelse dn-available <= uptdn [set dn-useful dn-available] [set dn-useful uptdn]       ; For nitrous oxide comparation between uptake by biomass and availability and takes the lowest value.

  set succinate-medium (succinate-medium - sc-useful)                                   ; Update local quantity of succinate
  set amonium-medium (amonium-medium - am-useful)                                       ; Update local quantity of amonium
  set oxygen-medium (oxygen-medium - ox-useful)                                         ; Update local quantity of oxygen
  set nitrate-medium (nitrate-medium - ntr-useful)                                      ; Update local quantity of nitrate
  set nitrite-medium (nitrite-medium - nr-useful)                                       ; Update local quantity of nitrite
  set monoxide-medium (monoxide-medium - mn-useful)                                     ; Update local quantity of nitric oxide
  set dinitrogen-medium (dinitrogen-medium - dn-useful)                                 ; Update local quantity of nitrous oxide

end

; ____________________________ Aerobic phase proceeding according to TEEM2 __________________________________________________________________________
; ____________________________ Aerobic Maintenance __________________________________________________________________________________________________

to mant-aerobic
  let sc-mant (scm * biomass)                                            ; Succinate celullar maintenance requirements by biomass [mmol]
  let ox-mant (oxm * biomass)                                            ; Oxygen cellular maintenance requirements by biomass [mmol]

  let y min (list sc-mant sc-useful)                                     ; takes the minimum value between maintenance requirements and uptake quantity for Succinate [mmol]
  let z min (list ox-mant ox-useful)                                     ; takes the minimum value between maintenance requirements and uptake quantity for Oxygen [mmol]

  let y1 (y / abs (matrix:get rx_mant-aero 0 8))                         ; succinate quantity by stoichometric coefficient
  let z1 (z / abs (matrix:get rx_mant-aero 0 0))                         ; oxygen quantity by stoichiometric coefficient

  let w1 min (list y1 z1)                                                ; limiting nutrient for aerobic maintenance

  let ndioxide (abs (matrix:get rx_mant-aero 0 3) * w1)                  ; Carbon dioxide generation by cellular maintenance [mmol]
  let nbicarbonate (abs (matrix:get rx_mant-aero 0 5) * w1)              ; Bicarbonate generation by cellular maintenance [mmol]
  set dioxide (dioxide + ndioxide)                                       ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                           ; Bicarbonate local quantity update

  set sc-useful (sc-useful - abs (matrix:get rx_mant-aero 0 8) * w1)     ; Update of Succinate uptaked  [mmol]
  set ox-useful (ox-useful - abs (matrix:get rx_mant-aero 0 0) * w1)     ; Update of Oxygen uptaked [mmol]
  set m_aero m_aero + 1                                                  ; counter to maintenance reaction on aerobic phase

  if (sc-useful > 0) and (am-useful > 0) and (ox-useful > 0) [aerobic1]  ; if uptaken updated quantities are higher than zero the microorganism could execute first aerobic parthway
  if (sc-useful > 0) and (am-useful > 0) and (ntr-useful > 0) [aerobic2] ; if uptaken updated quantities are higher than zero the microorganism could execute second aerobic parthway
end

; ______________________________ Aerobic Sinthesis ______________________________________________________________________________________________________

to aerobic1
  let a (sc-useful / abs (matrix:get pathway1 0 8))                      ; uptaken succinate updated divided by its stoichiometric coefficient
  let b (am-useful / abs (matrix:get pathway1 0 4))                      ; uptaken amonium updated divided by its stoichiometric coefficient
  let c (ox-useful / abs (matrix:get pathway1 0 0))                      ; uptaken oxygen updated divided by its stoichiometric coefficient

  let x1 min (list a b c)                                                ; limiting nutrient for pathway 1 in aerobic phase

  let nbiomass abs (matrix:get pathway1 0 6) * x1                        ; Biomass generation by aerobic pathway 1 [mmol]
  let ndioxide abs (matrix:get pathway1 0 3) * x1                        ; Carbon dioxide generation by aerobic pathway 1 [mmol]
  let nbicarbonate abs (matrix:get pathway1 0 5) * x1                    ; Bicarbonate generation by aerobic pathway 1 [mmol]
  set biomass (biomass + nbiomass)                                       ; Individual biomass update
  set dioxide (dioxide + ndioxide)                                       ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                           ; Bicarbonate local quantity update

  set sc-useful (sc-useful - abs (matrix:get pathway1 0 8) * x1)         ; update electron donor bacterium quantity
  set am-useful (am-useful - abs (matrix:get pathway1 0 4) * x1)         ; update N-source bacterium quantity
  set ox-useful (ox-useful - abs (matrix:get pathway1 0 0) * x1)         ; update electron aceptor bacterium quantity
  set s_aero_1 s_aero_1 + 1                                              ; counter to pathway 1 on aerobic phase

  if (sc-useful > 0) and (am-useful > 0) and (ntr-useful > 0) [aerobic2] ; if uptaken updated quantities are higher than zero the microorganism could execute second aerobic parthway
end

to aerobic2
  let a (sc-useful / abs (matrix:get pathway2 0 8))                     ; uptaken succinate updated divided by its stoichiometric coefficient
  let b (am-useful / abs (matrix:get pathway2 0 4))                     ; uptaken amonium updated divided by its stoichiometric coefficient
  let c (ntr-useful / abs (matrix:get pathway2 0 0))                    ; uptaken nitrate updated divided by its stoichiometric coefficient

  let x2 min (list a b c)                                               ; limiting nutrient for pathway 2 in aerobic phase

  let nbiomass abs (matrix:get pathway2 0 6) * x2                       ; Biomass generation by aerobic pathway 2 [mmol]
  let ndioxide abs (matrix:get pathway2 0 3) * x2                       ; Carbon dioxide generation by aerobic pathway 2 [mmol]
  let namonium abs (matrix:get pathway2 0 9) * x2                       ; Amonium generation by aerobic pathway 2 [mmol]
  let nbicarbonate abs (matrix:get pathway2 0 5) * x2                   ; Bicarbonate generation by aerobic pathway 1 [mmol]
  set biomass (biomass + nbiomass)                                      ; Individual biomass update
  set dioxide (dioxide + ndioxide)                                      ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                          ; Bicarbonate local quantity update
  set amonium-medium (amonium-medium + namonium)                        ; Amonium uptake quantity update

  set sc-useful (sc-useful - abs (matrix:get pathway2 0 8) * x2)        ; update electron donor bacterium quantity
  set am-useful (am-useful - abs (matrix:get pathway2 0 4) * x2)        ; update N-source bacterium quantity
  set ntr-useful (ntr-useful - abs (matrix:get pathway2 0 0) * x2)      ; update electron aceptor bacterium quantity
  set s_aero_2 s_aero_2 + 1                                             ; counter to pathway 2 on aerobic phase
end

; ____________________________ Anaerobic phase proceeding ________________________________________________________________________________________________________
; _____________________ Anoxics reactions are execute according to Reduction Order _______________________________________________________________________________
; _____________________________Cellular maintenance to anoxic reaction 1 _________________________________________________________________________________________

to mant-anoxic1

  let sc-mant (scm * biomass)                    ; Succinate celullar maintenance requirements by biomass [mmol]
  let ntr-mant (ntrm * biomass)                  ; Nitrate celullar maintenance requirements by biomass [mmol]

  let y min (list sc-mant sc-useful)             ; takes the minimum value between maintenance requirements and uptake quantity for Succinate [mmol]
  let z min (list ntr-mant ntr-useful)           ; takes the minimum value between maintenance requirements and uptake quantity for Nitrate [mmol]

  let y1 (y / abs (matrix:get rx_mant_anox1 0 8))                                            ; Succinate quantity by stoichometric coefficient
  let z1 (z / abs (matrix:get rx_mant_anox1 0 0))                                            ; Nitrate quantity by stoichometric coefficient
  let w1 min (list y1 z1)                                                                    ; limiting nutrient for first anaerobic maintenance

  let ndioxide abs (matrix:get rx_mant_anox1 0 3) * w1                                       ; Carbon dioxide generation by cellular maintenance [mmol]
  let nbicarbonate abs (matrix:get rx_mant_anox1 0 5) * w1                                   ; Bicarbonate generation by cellular maintenance [mmol]
  let nnitrite abs (matrix:get rx_mant_anox1 0 9) * w1                                       ; Nitrite generation by cellular maintenance [mmol]
  set dioxide (dioxide + ndioxide)                                                           ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                                               ; Bicarbonate local quantity update

  if ntr-mant > ntr-useful [set sc-mant2 sc-mant - abs (matrix:get rx_mant_anox1 0 8) * w1 ] ; Maintenance control

  set nr-useful (nr-useful + nnitrite)                                                       ; Uptaken nitrite updated
  set sc-useful (sc-useful - abs (matrix:get rx_mant_anox1 0 8) * w1)                        ; Update of Succinate uptaked  [mmol]
  set ntr-useful (ntr-useful - abs (matrix:get rx_mant_anox1 0 0) * w1)                      ; Update of Nitrate uptaked  [mmol]

  set m_anox_1 m_anox_1 + 1                                                                  ; counter to first maintenance reaction on anaerobic phase

  if sc-useful <= 0 [stop]                      ; if the succinate uptaken is lower than zero stop the microorganism metabolism until the next step time
  ifelse sc-mant2 > 0 [mant-anoxic2] [anoxic1]  ; if maintenance is fullfiled go to biomass synthesis procedures
end

; _____________________________Cellular maintenance to anoxic reaction 2 _________________________________________________________________________________________

to mant-anoxic2

  let nr-mant2 (sc-mant2 * abs (matrix:get rx_mant_anox2 0 0) / abs (matrix:get rx_mant_anox2 0 8)) ; nitrite maintenance requirements from succinate maintenance requirements

  let y min (list sc-mant2 sc-useful)               ; takes the minimum value between maintenance requirements and remaining uptaken quantity for Succinate [mmol]
  let z min (list nr-mant2 nr-useful)               ; takes the minimum value between maintenance requirements and uptake quantity for nitrite [mmol]

  let y1 (y / abs (matrix:get rx_mant_anox2 0 8))                                             ; Succinate quantity by stoichometric coefficient
  let z1 (z / abs (matrix:get rx_mant_anox2 0 0))                                             ; Nitrite quantity by stoichometric coefficient
  let w2 min (list y1 z1)                                                                     ; limiting nutrient for second anaerobic maintenance

  let ndioxide abs (matrix:get rx_mant_anox2 0 3) * w2                                        ; Carbon dioxide generation by cellular maintenance [mmol]
  let nbicarbonate abs (matrix:get rx_mant_anox2 0 5) * w2                                    ; Bicarbonate generation by cellular maintenance [mmol]
  let nmonoxide abs (matrix:get rx_mant_anox2 0 9) * w2                                       ; Nitric oxide generation by cellular maintenance [mmol]
  set dioxide (dioxide + ndioxide)                                                            ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                                                ; Bicarbonate local quantity update
  set mn-useful (mn-useful + nmonoxide)                                                       ; Nitric oxide generation is added to their useful quantity

  if  nr-mant2 > nr-useful [set sc-mant3 sc-mant2 - abs (matrix:get rx_mant_anox2 0 8) * w2 ] ; Maintenance control

  set sc-useful (sc-useful - w2 * abs (matrix:get rx_mant_anox2 0 8))                         ; Update of Succinate uptaken  [mmol]
  set nr-useful (nr-useful - w2 * abs (matrix:get rx_mant_anox2 0 0))                         ; Update of Nitrite uptaken  [mmol]

  set m_anox_2 m_anox_2 + 1                         ; counter to second maintenance reaction on anaerobic phase

  if sc-useful <= 0 [stop]                          ; if the succinate uptaken is lower than zero stop the microorganism metabolism until the next step time
  ifelse sc-mant3 > 0 [mant-anoxic3] [anoxic2]      ; if maintenance is fullfiled go to biomass synthesis procedures
end

; _____________________________Cellular maintenance to anoxic reaction 3 _________________________________________________________________________________________

to mant-anoxic3

  let mn-mant3 (sc-mant3 * abs (matrix:get rx_mant_anox3 0 0) / abs (matrix:get rx_mant_anox3 0 8)) ; nitric oxide maintenance requirements from succinate maintenance requirements

  let y min (list sc-mant3 sc-useful)                     ; takes the minimum value between maintenance requirements and remaining uptaken quantity for Succinate [mmol]
  let z min (list mn-mant3 mn-useful)                     ; takes the minimum value between maintenance requirements and uptake quantity for nitric oxide [mmol]

  let y1 (y / abs (matrix:get rx_mant_anox3 0 8))                                             ; Succinate quantity by stoichometric coefficient
  let z1 (z / abs (matrix:get rx_mant_anox3 0 0))                                             ; Nitric oxide quantity by stoichometric coefficient
  let w3 min (list y1 z1)                                                                     ; limiting nutrient for third anaerobic maintenance

  let ndioxide abs (matrix:get rx_mant_anox3 0 3) * w3                                        ; Carbon dioxide generation by cellular maintenance [mmol]
  let nbicarbonate abs (matrix:get rx_mant_anox3 0 5) * w3                                    ; Bicarbonate generation by cellular maintenance [mmol]
  let ndinitrogen abs (matrix:get rx_mant_anox3 0 9) * w3                                     ; Nitrous oxide generation by cellular maintenance [mmol]
  set dioxide (dioxide + ndioxide)                                                            ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                                                ; Bicarbonate local quantity update
  set dn-useful (dn-useful + ndinitrogen)                                                     ; Nitrous oxide generation is added to their useful quantity

  if mn-mant3 >= mn-useful [set sc-mant4 sc-mant3 - abs (matrix:get rx_mant_anox3 0 8) * w3 ] ; Maintenance control

  set sc-useful (sc-useful - w3 * abs (matrix:get rx_mant_anox3 0 8))                         ; Update of Succinate uptaken  [mmol]
  set mn-useful (mn-useful - w3 * abs (matrix:get rx_mant_anox3 0 0))                         ; Update of Nitric oxide uptaken  [mmol]

  set m_anox_3 m_anox_3 + 1                               ; counter to third maintenance reaction on anaerobic phase

  if sc-useful <= 0 [stop]                                ; if the succinate uptaken is lower than zero stop the microorganism metabolism until the next step time
  ifelse sc-mant4 > 0 [mant-anoxic4] [anoxic3]            ; if maintenance is fullfiled go to biomass synthesis procedures
end

; _____________________________Cellular maintenance to anoxic reaction 4 _________________________________________________________________________________________

to mant-anoxic4

  let dn-mant (sc-mant4 * abs (matrix:get rx_mant_anox4 0 0) / abs (matrix:get rx_mant_anox4 0 8)) ; nitrous oxide maintenance requirements from succinate maintenance requirements

  let y min (list sc-mant4 sc-useful)          ; takes the minimum value between maintenance requirements and remaining uptaken quantity for Succinate [mmol]
  let z min (list dn-mant dn-useful)           ; takes the minimum value between maintenance requirements and uptake quantity for nitrous oxide [mmol]

  let y1 (y / abs (matrix:get rx_mant_anox4 0 8))                     ; Succinate quantity by stoichometric coefficient
  let z1 (z / abs (matrix:get rx_mant_anox4 0 0))                     ; Nitrous oxide quantity by stoichometric coefficient
  let w4 min (list y1 z1)                                             ; limiting nutrient for fourth anaerobic maintenance

  let ndioxide abs (matrix:get rx_mant_anox4 0 3) * w4                ; Carbon dioxide generation by cellular maintenance [mmol]
  let nbicarbonate abs (matrix:get rx_mant_anox4 0 5) * w4            ; Bicarbonate generation by cellular maintenance [mmol]
  let nnitrogen abs (matrix:get rx_mant_anox4 0 9) * w4               ; Nitrogen gas generation by cellular maintenance [mmol]
  set dioxide (dioxide + ndioxide)                                    ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                        ; Bicarbonate local quantity update
  set nitrogen nitrogen + nnitrogen                                   ; Nitrogen gas local quantity update

  set sc-useful (sc-useful - w4 * abs (matrix:get rx_mant_anox4 0 8)) ; Update of Succinate uptaken  [mmol]
  set dn-useful (dn-useful - w4 * abs (matrix:get rx_mant_anox4 0 0)) ; Update of Nitrous oxide uptaken  [mmol]

  set m_anox_4 m_anox_4 + 1                                           ; counter to fourth maintenance reaction on anaerobic phase

  ifelse (sc-useful > 0) and (am-useful > 0) and (dn-useful > 0) [anoxic4][stop] ; if maintenance is fullfiled go to biomass synthesis procedures
end

; _____________________ Anoxics reactions are execute according to Reduction Order _______________________________________________________________________________
; _____________________________ Biomass Synthesis to anoxic reaction 1 ___________________________________________________________________________________________

to anoxic1
  let a (sc-useful / abs (matrix:get pathway3 0 8))                    ; uptaken succinate updated divided by its stoichiometric coefficient
  let b (am-useful / abs (matrix:get pathway3 0 4))                    ; uptaken amonium updated divided by its stoichiometric coefficient
  let c (ntr-useful / abs (matrix:get pathway3 0 0))                   ; uptaken nitrate updated divided by its stoichiometric coefficient

  let x1 min (list a b c)                                              ; limiting nutrient for enzimatic reaction 1 on pathway 3 in anaerobic phase

  let nbiomass abs (matrix:get pathway3 0 6) * x1                      ; Biomass generation for enzimatic reaction 1 on pathway 3 [mmol]
  let ndioxide abs (matrix:get pathway3 0 3) * x1                      ; Carbon dioxide generation for enzimatic reaction 1 on pathway 3 [mmol]
  let nnitrite abs (matrix:get pathway3 0 9) * x1                      ; Nitrite generation for enzimatic reaction 1 on pathway 3 [mmol]
  let nbicarbonate abs (matrix:get pathway3 0 5) * x1                  ; Bicarbonate generation for enzimatic reaction 1 on pathway 3 [mmol]
  set biomass (biomass + nbiomass)                                     ; Individual biomass update
  set dioxide (dioxide + ndioxide)                                     ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                         ; Bicarbonate local quantity update
  set nr-useful (nr-useful + nnitrite)                                 ; Update of nitrite uptaken

  set sc-useful (sc-useful - abs (matrix:get pathway3 0 8) * x1)       ; update electron donor bacterium quantity
  set am-useful (am-useful - abs (matrix:get pathway3 0 4) * x1)       ; update N-source bacterium quantity
  set ntr-useful (ntr-useful - abs (matrix:get pathway3 0 0) * x1)     ; update electron aceptor bacterium quantity
  set s_anox_1 s_anox_1 + 1                                            ; counter to first biomass synthesis reaction on anaerobic phase

  if (sc-useful > 0) and (am-useful > 0) and (nr-useful > 0) [anoxic2] ; if the succinate, amonium and nitrite quatities are enough go to enzimatic reaction 2
  if (sc-useful > 0) and (am-useful > 0) and (mn-useful > 0) [anoxic3] ; if the succinate, amonium and nitric oxide quatities are enough go to enzimatic reaction 3
  if (sc-useful > 0) and (am-useful > 0) and (dn-useful > 0) [anoxic4] ; if the succinate, amonium and nitrous oxide quatities are enough go to enzimatic reaction 4
end

; _____________________________ Biomass Synthesis to anoxic reaction 2 ___________________________________________________________________________________________

to anoxic2
  let a (sc-useful / abs (matrix:get pathway4 0 8))                    ; uptaken succinate updated divided by its stoichiometric coefficient
  let b (am-useful / abs (matrix:get pathway4 0 4))                    ; uptaken amonium updated divided by its stoichiometric coefficient
  let c (nr-useful / abs (matrix:get pathway4 0 0))                    ; uptaken nitrite updated divided by its stoichiometric coefficient

  let x2 min (list a b c)                                              ; limiting nutrient for enzimatic reaction 2 on pathway 3 in anaerobic phase

  let nbiomass abs (matrix:get pathway4 0 6) * x2                      ; Biomass generation for enzimatic reaction 2 on pathway 3 [mmol]
  let ndioxide abs (matrix:get pathway4 0 3) * x2                      ; Carbon dioxide generation for enzimatic reaction 2 on pathway 3 [mmol]
  let nmonoxide abs (matrix:get pathway4 0 9) * x2                     ; Nitric oxide generation for enzimatic reaction 2 on pathway 3 [mmol]
  let nbicarbonate abs (matrix:get pathway4 0 5) * x2                  ; Bicarbonate generation for enzimatic reaction 2 on pathway 3 [mmol]
  set biomass (biomass + nbiomass)                                     ; Individual biomass update
  set dioxide (dioxide + ndioxide)                                     ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                         ; Bicarbonate local quantity update
  set mn-useful (mn-useful + nmonoxide)                                ; Update of nitric oxide uptaken

  set sc-useful (sc-useful - abs (matrix:get pathway4 0 8) * x2)       ; update electron donor bacterium quantity
  set am-useful (am-useful - abs (matrix:get pathway4 0 4) * x2)       ; update N-source bacterium quantity
  set nr-useful (nr-useful - abs (matrix:get pathway4 0 0) * x2)       ; update electron aceptor bacterium quantity
  set s_anox_2 s_anox_2 + 1                                            ; counter to second biomass synthesis reaction on anaerobic phase

  if (sc-useful > 0) and (am-useful > 0) and (mn-useful > 0)[anoxic3]  ; if the succinate, amonium and nitric oxide quatities are enough go to enzimatic reaction 3
  if (sc-useful > 0) and (am-useful > 0) and (dn-useful > 0)[anoxic4]  ; if the succinate, amonium and nitrous oxide quatities are enough go to enzimatic reaction 4
end

; _____________________________ Biomass Synthesis to anoxic reaction 3 ___________________________________________________________________________________________

to anoxic3
  let a (sc-useful / abs (matrix:get pathway5 0 8))                    ; uptaken succinate updated divided by its stoichiometric coefficient
  let b (am-useful / abs (matrix:get pathway5 0 4))                    ; uptaken amonium updated divided by its stoichiometric coefficient
  let c (mn-useful / abs (matrix:get pathway5 0 0))                    ; uptaken nitric oxide updated divided by its stoichiometric coefficient

  let x3 min (list a b c)                                              ; limiting nutrient for enzimatic reaction 3 on pathway 3 in anaerobic phase

  let nbiomass abs (matrix:get pathway5 0 6) * x3                      ; Biomass generation for enzimatic reaction 3 on pathway 3 [mmol]
  let ndioxide abs (matrix:get pathway5 0 3) * x3                      ; Carbon dioxide generation for enzimatic reaction 3 on pathway 3 [mmol]
  let ndinitrogen abs (matrix:get pathway5 0 9) * x3                   ; Nitrous oxide generation for enzimatic reaction 3 on pathway 3 [mmol]
  let nbicarbonate abs (matrix:get pathway5 0 5) * x3                  ; Bicarbonate generation for enzimatic reaction 3 on pathway 3 [mmol]
  set biomass (biomass + nbiomass)                                     ; Individual biomass update
  set dioxide (dioxide + ndioxide)                                     ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                         ; Bicarbonate local quantity update
  set dn-useful (dn-useful + ndinitrogen)                              ; Update of nitrous oxide uptaken

  set sc-useful (sc-useful - abs (matrix:get pathway5 0 8) * x3)       ; update electron donor bacterium quantity
  set am-useful (am-useful - abs (matrix:get pathway5 0 4) * x3)       ; update N-source bacterium quantity
  set mn-useful (mn-useful - abs (matrix:get pathway5 0 0) * x3)       ; update electron aceptor bacterium quantity
  set s_anox_3 s_anox_3 + 1                                            ; counter to third biomass synthesis reaction on anaerobic phase

  if (sc-useful > 0) and (am-useful > 0) and (dn-useful > 0)[anoxic4]  ; if the succinate, amonium and nitrous oxide quatities are enough go to enzimatic reaction 4
end

; _____________________________ Biomass Synthesis to anoxic reaction 4 ___________________________________________________________________________________________

to anoxic4
  let a (sc-useful / abs (matrix:get pathway6 0 8))                  ; uptaken succinate updated divided by its stoichiometric coefficient
  let b (am-useful / abs (matrix:get pathway6 0 4))                  ; uptaken amonium updated divided by its stoichiometric coefficient
  let c (dn-useful / abs (matrix:get pathway6 0 0))                  ; uptaken nitrous oxide updated divided by its stoichiometric coefficient

  let x4 min (list a b c)                                            ; limiting nutrient for enzimatic reaction 4 on pathway 3 in anaerobic phase

  let nbiomass abs (matrix:get pathway6 0 6) * x4                    ; Biomass generation for enzimatic reaction 4 on pathway 3 [mmol]
  let ndioxide abs (matrix:get pathway6 0 3) * x4                    ; Carbon dioxide generation for enzimatic reaction 4 on pathway 3 [mmol]
  let nnitrogen abs (matrix:get pathway6 0 9) * x4                   ; Nitrogen gas generation for enzimatic reaction 4 on pathway 3 [mmol]
  let nbicarbonate abs (matrix:get pathway6 0 5) * x4                ; Bicarbonate generation for enzimatic reaction 4 on pathway 3 [mmol]
  set biomass (biomass + nbiomass)                                   ; Individual biomass update
  set dioxide (dioxide + ndioxide)                                   ; Carbon dioxide local quantity update
  set bicarbonate (bicarbonate + nbicarbonate)                       ; Bicarbonate local quantity update
  set nitrogen (nitrogen + nnitrogen)                                ; Nitrogen gas local quantity update

  set sc-useful (sc-useful - abs (matrix:get pathway6 0 8) * x4)     ; update electron donor bacterium quantity
  set am-useful (am-useful - abs (matrix:get pathway6 0 4) * x4)     ; update N-source bacterium quantity
  set dn-useful (dn-useful - abs (matrix:get pathway6 0 0) * x4)     ; update electron aceptor bacterium quantity
  set s_anox_4 s_anox_4 + 1                                          ; counter to fourth biomass synthesis reaction on anaerobic phase
end

; ____________________________ Procedure before bipartition _______________________________________________________________________________________________

to auto_consum
 if oxygen-medium <= abs random-normal NOX (DSTr * NOX)
  [
  set bm (abs random-normal self_consumption_energy (DSTr * self_consumption_energy)) * steptime * biomass

  if mn-useful > 0
  [
    let mnc bm / abs (matrix:get auto_consum_NO 0 6)               ; [mol electron acceptor NO]
    let mnc1 min (list mnc mn-useful)
    let ndinitrogen abs (matrix:get auto_consum_NO 0 9) * mnc1
    let namonium abs (matrix:get auto_consum_NO 0 4) * mnc1
    let ndioxide abs (matrix:get auto_consum_NO 0 3) * mnc1
    let nbicarbonate abs (matrix:get auto_consum_NO 0 5) * mnc1
    let nbiomass abs (matrix:get auto_consum_NO 0 6) * mnc1

    set amonium-medium (amonium-medium + namonium)
    set dioxide (dioxide + ndioxide)
    set bicarbonate (bicarbonate + nbicarbonate)
    set dn-useful (dn-useful + ndinitrogen)
    set biomass (biomass - nbiomass)
    set bm bm - nbiomass

    set ac_mn ac_mn + 1

    set mn-useful (mn-useful - mnc1)
   ]
  if (dn-useful > 0 and bm > 0)
  [
    let dnc bm * abs (matrix:get auto_consum_N2O 0 0) / abs (matrix:get auto_consum_N2O 0 6)                     ; [mol electron acceptor N2O]
    let dnc1 min (list dnc dn-useful)
    let nnitrogen abs (matrix:get auto_consum_N2O 0 9) * dnc1 / abs (matrix:get auto_consum_N2O 0 0)
    let namonium abs (matrix:get auto_consum_N2O 0 4) * dnc1 / abs (matrix:get auto_consum_N2O 0 0)
    let ndioxide abs (matrix:get auto_consum_N2O 0 3) * dnc1 / abs (matrix:get auto_consum_N2O 0 0)
    let nbicarbonate abs (matrix:get auto_consum_N2O 0 5) * dnc1 / abs (matrix:get auto_consum_N2O 0 0)
    let nbiomass abs (matrix:get auto_consum_N2O 0 6) * dnc1 / abs (matrix:get auto_consum_N2O 0 0)

    set amonium-medium (amonium-medium + namonium)
    set dioxide (dioxide + ndioxide)
    set bicarbonate (bicarbonate + nbicarbonate)
    set nitrogen (nitrogen + nnitrogen)
    set biomass (biomass - nbiomass)
    set bm bm - nbiomass
    set ac_dn ac_dn + 1

    set dn-useful (dn-useful - dnc1)

   ]
  ]
end

to metabolism_end
  ifelse sc-useful > 0 [set succinate-medium (succinate-medium + sc-useful) set sc-useful 0][set sc-useful 0]   ; Release to culture medium the quantity not used
  ifelse am-useful > 0 [set amonium-medium (amonium-medium + am-useful) set am-useful 0][set am-useful 0]       ; Release to culture medium the quantity not used
  ifelse ox-useful > 0 [set oxygen-medium (oxygen-medium + ox-useful) set ox-useful 0][set ox-useful 0]         ; Release to culture medium the quantity not used
  ifelse ntr-useful > 0 [set nitrate-medium (nitrate-medium + ntr-useful) set ntr-useful 0][set ntr-useful 0]   ; Release to culture medium the quantity not used
  ifelse nr-useful > 0 [set nitrite-medium (nitrite-medium + nr-useful) set nr-useful 0][set nr-useful 0]       ; Release to culture medium the quantity not used
  ifelse mn-useful > 0 [set monoxide-medium (monoxide-medium + mn-useful) set mn-useful 0][set mn-useful 0]
  ifelse dn-useful > 0 [set dinitrogen-medium (dinitrogen-medium + dn-useful) set dn-useful 0][set dn-useful 0]
  set sc-mant2 0 set sc-mant3 0 set sc-mant4 0                                                                  ; Update to zero succinate internal maintenance requirements
end

; ____________________________ Bipartition Procedure _______________________________________________________________________________________________

to bipartition
  if biomass >= biomass-reproduction                                ; cell division happens when a treshold value is reached
  [
    let division-proportioning random-normal 0.50 (DSTr * 0.50)     ; new bacteria differ randomly in their cell-biomass
    let cell-biomass-1 biomass * division-proportioning             ; original biomass is not splitted equally into two halfs
    let cell-biomass-2 biomass - cell-biomass-1
    set biomass cell-biomass-1                                      ; biomass for original microorganims
    set biomass-reproduction abs random-normal (rep_biomass) (DSTr * rep_biomass)   ; setup new reproduction treshold for the original microorganism
    hatch 1                                                         ; born new microorganism
    [
      set biomass cell-biomass-2                                    ; biomass for newest microorganims
      ifelse time_now > Time_Shutdown_O2 [set color blue][set color green]
      set R 0 set m_aero 0 set s_aero_1 0 set s_aero_2 0            ; Reset all counters for new microorganism
      set m_anox_1 0 set m_anox_2 0 set m_anox_3 0 set m_anox_4 0   ; Reset all counters for new microorganism
      set s_anox_1 0 set s_anox_2 0 set s_anox_3 0 set s_anox_4 0   ; Reset all counters for new microorganism
      set ac_mn 0 set ac_dn 0 set bm 0
      set biomass-reproduction abs random-normal (rep_biomass) (DSTr * rep_biomass) ; setup the reproduction treshold for the newest microorganism
    ]
    set R R + 1                                                     ; Counter for bipartitions
  ]
end

; ____________________________ Proceeding to simulate bioreactor agitation over the microorganism ____________________________________________________

to move
  if q = 0
  [ setxy random-xcor random-ycor ] ; Randomly change position because bioreactor agitation
end

; ____________________________ Proceeding to setup and write bioreactor graphical outputs ____________________________________________________

to do-plotting
  setup-monitors

  set-current-plot "Biomass"
  set-current-plot-pen "Biomass"
  plotxy time_now gbacteria

  set-current-plot "Products-CO2"
  set-current-plot-pen "CO2"
  plotxy time_now gdioxide

  set-current-plot "Products-N2O"
  set-current-plot-pen "N2O"
  plotxy time_now gn2o

  set-current-plot "Products - NO2-"
  set-current-plot-pen "NO2-"
  plotxy time_now gno2-

  set-current-plot "Products - NO"
  set-current-plot-pen "NO"
  plotxy time_now gno

  set-current-plot "Products - N2"
  set-current-plot-pen "N2"
  plotxy time_now gn2

  set-current-plot "Culture Medium Nutrients"
  set-current-plot-pen "Succinate"
  plotxy time_now gsuccinate
  set-current-plot-pen "Amonium"
  plotxy time_now gamonium

  set-current-plot "Nitrate"
  set-current-plot-pen "nitrate"
  plotxy time_now gnitrate

  set-current-plot "Oxygen"
  set-current-plot-pen "oxygen"
  plotxy time_now goxygen * 32

  set-current-plot "Bicarbonate"
  set-current-plot-pen "bicarbonate"
  plotxy time_now gbicarbonate

  set-current-plot "Bacterial-biomass-distribution"
  set-current-plot-pen "biomass-histogram"
  histogram [biomass] of bacteria
  set-plot-x-range fmol_small fmol_big
  if Denitrifying_bacterium = "P._denitrificans" [set-plot-pen-interval 0.1]
  if Denitrifying_bacterium = "A._xylosoxidans" [set-plot-pen-interval 0.3]


  set-current-plot "Metabolic_reactions"
  set-current-plot-pen "m_aero"
  plotxy time_now mean [m_aero] of bacteria
  set-current-plot-pen "s_aero_1"
  plotxy time_now mean [s_aero_1] of bacteria
  set-current-plot-pen "s_aero_2"
  plotxy time_now mean [s_aero_2] of bacteria
  set-current-plot-pen "m_anox_1"
  plotxy time_now mean [m_anox_1] of bacteria
  set-current-plot-pen "m_anox_2"
  plotxy time_now mean [m_anox_2] of bacteria
  set-current-plot-pen "m_anox_3"
  plotxy time_now mean [m_anox_3] of bacteria
  set-current-plot-pen "m_anox_4"
  plotxy time_now mean [m_anox_4] of bacteria
  set-current-plot-pen "s_anox_1"
  plotxy time_now mean [s_anox_1] of bacteria
  set-current-plot-pen "s_anox_2"
  plotxy time_now mean [s_anox_2] of bacteria
  set-current-plot-pen "s_anox_3"
  plotxy time_now mean [s_anox_3] of bacteria
  set-current-plot-pen "s_anox_4"
  plotxy time_now mean [s_anox_4] of bacteria
  set-current-plot-pen "R"
  plotxy time_now mean [R] of bacteria
  set-current-plot-pen "ac_mn"
  plotxy time_now mean [ac_mn] of bacteria
  set-current-plot-pen "ac_dn"
  plotxy time_now mean [ac_dn] of bacteria
end

to setup-thermodynamics

if Denitrifying_bacterium = "P._denitrificans" [set carbon 3 set hydrogen 5.4 set oxygen 1.45 set b-nitrogen 0.75]
if Denitrifying_bacterium = "A._xylosoxidans" [set carbon 5 set hydrogen 9 set oxygen 2.5 set b-nitrogen 1]

 set molecular-weight 12.011 * carbon + 1.008 * hydrogen + 16 * oxygen + 14 * b-nitrogen
 set e-eeq/mol (4 * carbon + hydrogen - 2 * oxygen - 3 * b-nitrogen)
 let b-co2 ((carbon - b-nitrogen) / e-eeq/mol)
 let b-hco3 (b-nitrogen / e-eeq/mol)
 let b-nh4 (b-nitrogen / e-eeq/mol)
 let b-bio (1 / e-eeq/mol)
 let b-h2o ((2 * carbon - oxygen + b-nitrogen) / e-eeq/mol)
 set dGpc (3.324 * molecular-weight / e-eeq/mol)
 set gamma_cells e-eeq/mol / carbon

 set A1 -1 * ((-29.09 / e1) + (30.9 / e1) + (dGpc / e1)) / (e1 * (-78.72 - 29.09))
 set fs1 (1 / (1 + A1)) set fe1 (A1 / (1 + A1)) set yc/c1 (3.5 / gamma_cells) * fs1

 set A2 -1 * ((-29.09 / e2) + (30.9 / e2) + (dGpc / e2)) / (e2 * (-35.11 - 29.09))
 set fs2 (1 / (1 + A2)) set fe2 (A2 / (1 + A2)) set yc/c2 (3.5 / gamma_cells) * fs2

 set A3 -1 * ((-29.09 / e3) + (30.9 / e3) + (dGpc / e3)) / (e3 * (-41.65 - 29.09))
 set fs3 (1 / (1 + A3)) set fe3 (A3 / (1 + A3)) set yc/c3 (3.5 / gamma_cells) * fs3

 set A4 -1 * ((-29.09 / e4) + (30.9 / e4) + (dGpc / e4)) / (e4 * (-33.718 - 29.09))
 set fs4 (1 / (1 + A4)) set fe4 (A4 / (1 + A4)) set yc/c4 (3.5 / gamma_cells) * fs4

 set A5 -1 * ((-29.09 / e5) + (30.9 / e5) + (dGpc / e5)) / (e5 * (-115.829 - 29.09))
 set fs5 (1 / (1 + A5)) set fe5 (A5 / (1 + A5)) set yc/c5 (3.5 / gamma_cells) * fs5

 set A6 -1 * ((-29.09 / e6) + (30.9 / e6) + (dGpc / e6)) / (e6 * (-133.469 - 29.09))
 set fs6 (1 / (1 + A6)) set fe6 (A6 / (1 + A6)) set yc/c6 (3.5 / gamma_cells) * fs6

 set donor matrix:from-row-list [[0 "d-H+" "d-e-" "d-co2" 0 "d-hco3" 0 "d-h2o" "d-succ" 0]]
 matrix:set donor 0 1 (1)
 matrix:set donor 0 2 (1)
 matrix:set donor 0 3 (1 / 7)
 matrix:set donor 0 5 (1 / 7)
 matrix:set donor 0 7 (-3 / 7)
 matrix:set donor 0 8 (-1 / 14)

 set b-biomass matrix:from-row-list [[0 "b-H+" "b-e-" "b-co2" "b-nh4" "b-hco3" "b-bio" "b-h2o" 0 0]]
 matrix:set b-biomass 0 1 (1)
 matrix:set b-biomass 0 2 (1)
 matrix:set b-biomass 0 3 (b-co2)
 matrix:set b-biomass 0 4 (b-nh4)
 matrix:set b-biomass 0 5 (b-hco3)
 matrix:set b-biomass 0 6 (-1 * b-bio)
 matrix:set b-biomass 0 7 (-1 * b-h2o)

 set acceptor1 matrix:from-row-list [["a1-o2" "a1-H+" "a1-e-" 0 0 0 0 "a1-h2o" 0 0]]
 matrix:set acceptor1 0 0 (1 / 4)
 matrix:set acceptor1 0 1 (1)
 matrix:set acceptor1 0 2 (1)
 matrix:set acceptor1 0 7 (-1 / 2)

 set acceptor2 matrix:from-row-list [["a2-no3-" "a2-H+" "a2-e-" 0 0 0 0 "a2-h2o" 0 "a2-nh4"]]
 matrix:set acceptor2 0 0 (1 / 8)
 matrix:set acceptor2 0 1 (5 / 4)
 matrix:set acceptor2 0 2 (1)
 matrix:set acceptor2 0 7 (-3 / 8)
 matrix:set acceptor2 0 9 (-1 / 8)

 set acceptor3 matrix:from-row-list [["a3-no3-" "a3-H+" "a3-e-" 0 0 0 0 "a3-h2o" 0 "a3-no2-"]]
 matrix:set acceptor3 0 0 (1 / 2)
 matrix:set acceptor3 0 1 (1)
 matrix:set acceptor3 0 2 (1)
 matrix:set acceptor3 0 7 (-1 / 2)
 matrix:set acceptor3 0 9 (-1 / 2)

 set acceptor4 matrix:from-row-list [["a4-no2-" "a4-H+" "a4-e-" 0 0 0 0 "a4-h2o" 0 "a4-no"]]
 matrix:set acceptor4 0 0 (1)
 matrix:set acceptor4 0 1 (2)
 matrix:set acceptor4 0 2 (1)
 matrix:set acceptor4 0 7 (-1)
 matrix:set acceptor4 0 9 (-1)

 set acceptor5 matrix:from-row-list [["a5-no" "a5-H+" "a5-e-" 0 0 0 0 "a5-h2o" 0 "a5-n2o"]]
 matrix:set acceptor5 0 0 (1)
 matrix:set acceptor5 0 1 (1)
 matrix:set acceptor5 0 2 (1)
 matrix:set acceptor5 0 7 (-1 / 2)
 matrix:set acceptor5 0 9 (-1 / 2)

 set acceptor6 matrix:from-row-list [["a6-n2o" "a6-H+" "a6-e-" 0 0 0 0 "a6-h2o" 0 "a6-n2"]]
 matrix:set acceptor6 0 0 (1 / 2)
 matrix:set acceptor6 0 1 (1)
 matrix:set acceptor6 0 2 (1)
 matrix:set acceptor6 0 7 (-1 / 2)
 matrix:set acceptor6 0 9 (-1 / 2)

 set pathway1 ((fe1 matrix:* acceptor1) matrix:+ (fs1 matrix:* b-biomass ) matrix:- (donor))
 set pathway2 ((fe2 matrix:* acceptor2) matrix:+ (fs2 matrix:* b-biomass ) matrix:- (donor))
 set pathway3 ((fe3 matrix:* acceptor3) matrix:+ (fs3 matrix:* b-biomass ) matrix:- (donor))
 set pathway4 ((fe4 matrix:* acceptor4) matrix:+ (fs4 matrix:* b-biomass ) matrix:- (donor))
 set pathway5 ((fe5 matrix:* acceptor5) matrix:+ (fs5 matrix:* b-biomass ) matrix:- (donor))
 set pathway6 ((fe6 matrix:* acceptor6) matrix:+ (fs6 matrix:* b-biomass ) matrix:- (donor))

 set auto_consum_NO acceptor5 matrix:- b-biomass
 set auto_consum_N2O acceptor6 matrix:- b-biomass
 set auto_consum_NO2- acceptor4 matrix:- b-biomass

 set rx_mant-aero acceptor1 matrix:- donor
 set rx_mant_anox1 acceptor3 matrix:- donor
 set rx_mant_anox2 acceptor4 matrix:- donor
 set rx_mant_anox3 acceptor5 matrix:- donor
 set rx_mant_anox4 acceptor6 matrix:- donor

 maximun_uptake

 ifelse (matrix:get pathway1 0 3) > 0
    [output-print "Beware - CO2 is reagent on pathway1, please review e1 value" stop]
    [ifelse (matrix:get pathway2 0 3) > 0
      [output-print "Beware - CO2 is reagent on pathway2, please review e2 value" stop]
      [ifelse (matrix:get pathway3 0 3) > 0
        [output-print "Beware - CO2 is reagent on pathway3, please review e3 value" stop]
        [ifelse (matrix:get pathway4 0 3) > 0
          [output-print "Beware - CO2 is reagent on pathway4, please review e4 value" stop]
          [ifelse (matrix:get pathway5 0 3) > 0
            [output-print "Beware - CO2 is reagent on pathway5, please review e5 value" stop]
            [ifelse (matrix:get pathway6 0 3) > 0
              [output-print "Beware - CO2 is reagent on pathway6, please review e6 value" stop]
              [output]
    ]]]]]

end

to maximun_uptake
  set u_succ u_max * abs mean (list (matrix:get pathway1 0 8 / matrix:get pathway1 0 6) (matrix:get pathway2 0 8 / matrix:get pathway2 0 6) (matrix:get pathway3 0 8 / matrix:get pathway3 0 6) (matrix:get pathway4 0 8 / matrix:get pathway4 0 6) (matrix:get pathway5 0 8 / matrix:get pathway5 0 6) (matrix:get pathway6 0 8 / matrix:get pathway6 0 6)) / carbon
  set u_nh4+ u_max * abs mean (list (matrix:get pathway1 0 4 / matrix:get pathway1 0 6) (matrix:get pathway2 0 4 / matrix:get pathway2 0 6) (matrix:get pathway3 0 4 / matrix:get pathway3 0 6) (matrix:get pathway4 0 4 / matrix:get pathway4 0 6) (matrix:get pathway5 0 4 / matrix:get pathway5 0 6) (matrix:get pathway6 0 4 / matrix:get pathway6 0 6)) / carbon
  set u_o2 u_max * abs (matrix:get pathway1 0 0 / matrix:get pathway1 0 6) / carbon
  set u_no3-a u_max * abs (matrix:get pathway2 0 0 / matrix:get pathway2 0 6) / carbon
  set u_no3-x (u_max * abs (matrix:get pathway3 0 0 / matrix:get pathway3 0 6) / carbon) / 4
  set u_no2- (u_max * abs (matrix:get pathway4 0 0 / matrix:get pathway4 0 6) / carbon) / 4
  set u_no (u_max * abs (matrix:get pathway5 0 0 / matrix:get pathway5 0 6) / carbon) / 4
  set u_n2o (u_max * abs (matrix:get pathway6 0 0 / matrix:get pathway6 0 6) / carbon) / 4
end

to output

output-print "Adjusted Metabolic Pathways by TEEM2 (McCarty, 2007) : "
output-type "Aerobic 1 : "
output-type precision (matrix:get pathway1 0 8 / matrix:get pathway1 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get pathway1 0 4 / matrix:get pathway1 0 8) 3 output-type " NH4+ + "
output-type precision (matrix:get pathway1 0 0 / matrix:get pathway1 0 8) 3 output-type " O2 --> "
output-type precision (matrix:get pathway1 0 6 / matrix:get pathway1 0 8) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get pathway1 0 3 / matrix:get pathway1 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get pathway1 0 5 / matrix:get pathway1 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get pathway1 0 7 / matrix:get pathway1 0 8) 3 output-print " H2O"

output-type "Aerobic 2 : "
output-type precision (matrix:get pathway2 0 8 / matrix:get pathway2 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get pathway2 0 4 / matrix:get pathway2 0 8) 3 output-type " NH4+ + "
output-type precision (matrix:get pathway2 0 0 / matrix:get pathway2 0 8) 3 output-type " NO3- + "
output-type precision (matrix:get pathway2 0 1 / matrix:get pathway2 0 8) 3 output-type " H+ + "
output-type precision (matrix:get pathway2 0 7 / matrix:get pathway2 0 8) 3 output-type " H2O --> "
output-type precision (matrix:get pathway2 0 6 / matrix:get pathway2 0 8) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get pathway2 0 3 / matrix:get pathway2 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get pathway2 0 5 / matrix:get pathway2 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get pathway2 0 9 / matrix:get pathway2 0 8) 3 output-print " NH4+"

output-type " Anoxic 1 : "
output-type precision (matrix:get pathway3 0 8 / matrix:get pathway3 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get pathway3 0 4 / matrix:get pathway3 0 8) 3 output-type " NH4+ + "
output-type precision (matrix:get pathway3 0 0 / matrix:get pathway3 0 8) 3 output-type " NO3- --> "
output-type precision (matrix:get pathway3 0 6 / matrix:get pathway3 0 8) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get pathway3 0 9 / matrix:get pathway3 0 8) 3 output-type " NO2- + "
output-type precision (matrix:get pathway3 0 3 / matrix:get pathway3 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get pathway3 0 5 / matrix:get pathway3 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get pathway3 0 7 / matrix:get pathway3 0 8) 3 output-print " H2O"

output-type " Anoxic 2 : "
output-type precision (matrix:get pathway4 0 8 / matrix:get pathway4 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get pathway4 0 4 / matrix:get pathway4 0 8) 3 output-type " NH4+ + "
output-type precision (matrix:get pathway4 0 0 / matrix:get pathway4 0 8) 3 output-type " NO2- + "
output-type precision (matrix:get pathway4 0 1 / matrix:get pathway4 0 8) 3 output-type " H+ --> "
output-type precision (matrix:get pathway4 0 6 / matrix:get pathway4 0 8) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get pathway4 0 9 / matrix:get pathway4 0 8) 3 output-type " NO + "
output-type precision (matrix:get pathway4 0 3 / matrix:get pathway4 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get pathway4 0 5 / matrix:get pathway4 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get pathway4 0 7 / matrix:get pathway4 0 8) 3 output-print " H2O"

output-type " Anoxic 3 : "
output-type precision (matrix:get pathway5 0 8 / matrix:get pathway5 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get pathway5 0 4 / matrix:get pathway5 0 8) 3 output-type " NH4+ + "
output-type precision (matrix:get pathway5 0 0 / matrix:get pathway5 0 8) 3 output-type " NO --> "
output-type precision (matrix:get pathway5 0 6 / matrix:get pathway5 0 8) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get pathway5 0 9 / matrix:get pathway5 0 8) 3 output-type " N2O + "
output-type precision (matrix:get pathway5 0 3 / matrix:get pathway5 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get pathway5 0 5 / matrix:get pathway5 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get pathway5 0 7 / matrix:get pathway5 0 8) 3 output-print " H2O"

output-type " Anoxic 4 : "
output-type precision (matrix:get pathway6 0 8 / matrix:get pathway6 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get pathway6 0 4 / matrix:get pathway6 0 8) 3 output-type " NH4+ + "
output-type precision (matrix:get pathway6 0 0 / matrix:get pathway6 0 8) 3 output-type " N2O --> "
output-type precision (matrix:get pathway6 0 6 / matrix:get pathway6 0 8) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get pathway6 0 9 / matrix:get pathway6 0 8) 3 output-type " N2 + "
output-type precision (matrix:get pathway6 0 3 / matrix:get pathway6 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get pathway6 0 5 / matrix:get pathway6 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get pathway6 0 7 / matrix:get pathway6 0 8) 3 output-print " H2O"

output-print "Adjusted Biomass Degradation Pathways by TEEM2 (McCarty, 2007) : "
output-type "Degradation of NO2-: "
output-type precision (matrix:get auto_consum_NO2- 0 6 / matrix:get auto_consum_NO2- 0 6) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get auto_consum_NO2- 0 1 / matrix:get auto_consum_NO2- 0 6) 3 output-type " H+ + "
output-type precision (matrix:get auto_consum_NO2- 0 0 / matrix:get auto_consum_NO2- 0 6) 3 output-type " NO2- --> "
output-type precision (matrix:get auto_consum_NO2- 0 3 / matrix:get auto_consum_NO2- 0 6) 3 output-type " CO2 + "
output-type precision (matrix:get auto_consum_NO2- 0 4 / matrix:get auto_consum_NO2- 0 6) 3 output-type " NH4+ + "
output-type precision (matrix:get auto_consum_NO2- 0 5 / matrix:get auto_consum_NO2- 0 6) 3 output-type " HCO3- + "
output-type precision (matrix:get auto_consum_NO2- 0 9 / matrix:get auto_consum_NO2- 0 6) 3 output-type " NO + "
output-type precision (matrix:get auto_consum_NO2- 0 7 / matrix:get auto_consum_NO2- 0 6) 3 output-print " H2O"

output-type "Degradation of NO  : "
output-type precision (matrix:get auto_consum_NO 0 6 / matrix:get auto_consum_NO 0 6) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get auto_consum_NO 0 0 / matrix:get auto_consum_NO 0 6) 3 output-type " NO --> "
output-type precision (matrix:get auto_consum_NO 0 3 / matrix:get auto_consum_NO 0 6) 3 output-type " CO2 + "
output-type precision (matrix:get auto_consum_NO 0 4 / matrix:get auto_consum_NO 0 6) 3 output-type " NH4+ + "
output-type precision (matrix:get auto_consum_NO 0 5 / matrix:get auto_consum_NO 0 6) 3 output-type " HCO3- + "
output-type precision (matrix:get auto_consum_NO 0 9 / matrix:get auto_consum_NO 0 6) 3 output-type " N2O + "
output-type precision (matrix:get auto_consum_NO 0 7 / matrix:get auto_consum_NO 0 6) 3 output-print " H2O"

output-type "Degradation of N2O : "
output-type precision (matrix:get auto_consum_N2O 0 6 / matrix:get auto_consum_N2O 0 6) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get auto_consum_N2O 0 0 / matrix:get auto_consum_N2O 0 6) 3 output-type " N2O --> "
output-type precision (matrix:get auto_consum_N2O 0 3 / matrix:get auto_consum_N2O 0 6) 3 output-type " CO2 + "
output-type precision (matrix:get auto_consum_N2O 0 4 / matrix:get auto_consum_N2O 0 6) 3 output-type " NH4+ + "
output-type precision (matrix:get auto_consum_N2O 0 5 / matrix:get auto_consum_N2O 0 6) 3 output-type " HCO3- + "
output-type precision (matrix:get auto_consum_N2O 0 9 / matrix:get auto_consum_N2O 0 6) 3 output-type " N2 + "
output-type precision (matrix:get auto_consum_N2O 0 7 / matrix:get auto_consum_N2O 0 6) 3 output-print " H2O"

output-print "Adjusted cellular Maintenance reactions by TEEM2 (McCarty, 2007) : "
output-type "Aerobic Manintenance  : "
output-type precision (matrix:get rx_mant-aero 0 8 / matrix:get rx_mant-aero 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get rx_mant-aero 0 0 / matrix:get rx_mant-aero 0 8) 3 output-type " O2 --> "
output-type precision (matrix:get rx_mant-aero 0 3 / matrix:get rx_mant-aero 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get rx_mant-aero 0 5 / matrix:get rx_mant-aero 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get rx_mant-aero 0 7 / matrix:get rx_mant-aero 0 8) 3 output-print " H2O"

output-type "Anoxic1 Manintenance  : "
output-type precision (matrix:get rx_mant_anox1 0 8 / matrix:get rx_mant_anox1 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get rx_mant_anox1 0 0 / matrix:get rx_mant_anox1 0 8) 3 output-type " NO3- --> "
output-type precision (matrix:get rx_mant_anox1 0 3 / matrix:get rx_mant_anox1 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get rx_mant_anox1 0 5 / matrix:get rx_mant_anox1 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get rx_mant_anox1 0 9 / matrix:get rx_mant_anox1 0 8) 3 output-type " NO2- + "
output-type precision (matrix:get rx_mant_anox1 0 7 / matrix:get rx_mant_anox1 0 8) 3 output-print " H2O"

output-type "Anoxic2 Manintenance  : "
output-type precision (matrix:get rx_mant_anox2 0 8 / matrix:get rx_mant_anox2 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get rx_mant_anox2 0 0 / matrix:get rx_mant_anox2 0 8) 3 output-type " NO2- + "
output-type precision (matrix:get rx_mant_anox2 0 1 / matrix:get rx_mant_anox2 0 8) 3 output-type " H+ --> "
output-type precision (matrix:get rx_mant_anox2 0 3 / matrix:get rx_mant_anox2 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get rx_mant_anox2 0 5 / matrix:get rx_mant_anox2 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get rx_mant_anox2 0 9 / matrix:get rx_mant_anox2 0 8) 3 output-type " NO + "
output-type precision (matrix:get rx_mant_anox2 0 7 / matrix:get rx_mant_anox2 0 8) 3 output-print " H2O"

output-type "Anoxic3 Manintenance  : "
output-type precision (matrix:get rx_mant_anox3 0 8 / matrix:get rx_mant_anox3 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get rx_mant_anox3 0 0 / matrix:get rx_mant_anox3 0 8) 3 output-type " NO --> "
output-type precision (matrix:get rx_mant_anox3 0 3 / matrix:get rx_mant_anox3 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get rx_mant_anox3 0 5 / matrix:get rx_mant_anox3 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get rx_mant_anox3 0 9 / matrix:get rx_mant_anox3 0 8) 3 output-type " N2O + "
output-type precision (matrix:get rx_mant_anox3 0 7 / matrix:get rx_mant_anox3 0 8) 3 output-print " H2O"

output-type "Anoxic4 Manintenance  : "
output-type precision (matrix:get rx_mant_anox4 0 8 / matrix:get rx_mant_anox4 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get rx_mant_anox4 0 0 / matrix:get rx_mant_anox4 0 8) 3 output-type " N2O --> "
output-type precision (matrix:get rx_mant_anox4 0 3 / matrix:get rx_mant_anox4 0 8) 3 output-type " CO2 + "
output-type precision (matrix:get rx_mant_anox4 0 5 / matrix:get rx_mant_anox4 0 8) 3 output-type " HCO3- + "
output-type precision (matrix:get rx_mant_anox4 0 9 / matrix:get rx_mant_anox4 0 8) 3 output-type " N2 + "
output-type precision (matrix:get rx_mant_anox4 0 7 / matrix:get rx_mant_anox4 0 8) 3 output-print " H2O"

output-type "Biomass half reaction : "
output-type precision (matrix:get b-biomass 0 3) 3 output-type " CO2 + "
output-type precision (matrix:get b-biomass 0 5) 3 output-type " HCO3- + "
output-type precision (matrix:get b-biomass 0 4) 3 output-type " NH4+ + "
output-type precision (matrix:get b-biomass 0 1) 3 output-type " H+ + "
output-type precision (matrix:get b-biomass 0 2) 3 output-type " e- --> "
output-type precision (matrix:get b-biomass 0 6) 3 output-type " C'" output-type carbon output-type "'H'" output-type hydrogen output-type "'O'" output-type oxygen output-type "'N'" output-type b-nitrogen output-type " + "
output-type precision (matrix:get b-biomass 0 7) 3 output-print " H2O"

output-type "Donor : "
output-type precision (matrix:get donor 0 3) 3 output-type " CO2 + "
output-type precision (matrix:get donor 0 5) 3 output-type " HCO3- + "
output-type precision (matrix:get donor 0 1) 3 output-type " H+ + "
output-type precision (matrix:get donor 0 2) 3 output-type " e- --> "
output-type precision (matrix:get donor 0 8) 3 output-type " (C4H4O4)2- + "
output-type precision (matrix:get donor 0 7) 3 output-print " H2O"

output-type "Acceptor1 : "
output-type precision (matrix:get acceptor1 0 0) 3 output-type " O2 + "
output-type precision (matrix:get acceptor1 0 1) 3 output-type " H+ + "
output-type precision (matrix:get acceptor1 0 2) 3 output-type " e- --> "
output-type precision (matrix:get acceptor1 0 7) 3 output-print " H2O"

output-type "Acceptor2 : "
output-type precision (matrix:get acceptor2 0 0) 4 output-type " NO3- + "
output-type precision (matrix:get acceptor2 0 1) 4 output-type " H+ + "
output-type precision (matrix:get acceptor2 0 2) 4 output-type " e- --> "
output-type precision (matrix:get acceptor2 0 9) 4 output-type " NH4+ + "
output-type precision (matrix:get acceptor2 0 7) 4 output-print " H2O"

output-type "Acceptor3 : "
output-type precision (matrix:get acceptor3 0 0) 4 output-type " NO3- + "
output-type precision (matrix:get acceptor3 0 1) 4 output-type " H+ + "
output-type precision (matrix:get acceptor3 0 2) 4 output-type " e- --> "
output-type precision (matrix:get acceptor3 0 9) 4 output-type " NO2- + "
output-type precision (matrix:get acceptor3 0 7) 4 output-print " H2O"

output-type "Acceptor4 : "
output-type precision (matrix:get acceptor4 0 0) 4 output-type " NO2- + "
output-type precision (matrix:get acceptor4 0 1) 4 output-type " H+ + "
output-type precision (matrix:get acceptor4 0 2) 4 output-type " e- --> "
output-type precision (matrix:get acceptor4 0 9) 4 output-type " NO + "
output-type precision (matrix:get acceptor4 0 7) 4 output-print " H2O"

output-type "Acceptor5 : "
output-type precision (matrix:get acceptor5 0 0) 4 output-type " NO + "
output-type precision (matrix:get acceptor5 0 1) 4 output-type " H+ + "
output-type precision (matrix:get acceptor5 0 2) 4 output-type " e- --> "
output-type precision (matrix:get acceptor5 0 9) 4 output-type " N2O + "
output-type precision (matrix:get acceptor5 0 7) 4 output-print " H2O"

output-type "Acceptor6 : "
output-type precision (matrix:get acceptor6 0 0) 4 output-type " N2O + "
output-type precision (matrix:get acceptor6 0 1) 4 output-type " H+ + "
output-type precision (matrix:get acceptor6 0 2) 4 output-type " e- --> "
output-type precision (matrix:get acceptor6 0 9) 4 output-type " N2 + "
output-type precision (matrix:get acceptor6 0 7) 4 output-print " H2O"


  if Output-data? = true
  [
  if Denitrifying_bacterium = "P._denitrificans" [export-output (word (word behaviorspace-run-number"_"molecular-weight)"_I_Denitrification_V10_P.denitrificans_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  if Denitrifying_bacterium = "A._xylosoxidans" [export-output (word (word behaviorspace-run-number"_"molecular-weight)"_I_Denitrification_V10_A.xylosoxidans_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  ]

end

to to-umax
  set u1 gbacteria
  let DBM ((u1 - u0) / u0)
  set u DBM / steptime
  set u0 u1
  ifelse u > u3 [set umax u set u3 umax][set umax u3 set u3 umax]
end

; ---------------------------------------- Data Analysis GRI ---------------------------------------------------------------------------------------------------------------------------------------

to setup-machine.learning

if Denitrifying_bacterium = "P._denitrificans" [

if Experiment = "Succ_Low/Nitrate_High"[

let column_matrix_biomass 17 let row_matrix_biomass 6
set biomass_matrix matrix:make-constant row_matrix_biomass column_matrix_biomass 0

matrix:set biomass_matrix 0 0 (0) matrix:set biomass_matrix 2 0 (0.0668478) matrix:set biomass_matrix 1 0 (gbacteria)
matrix:set biomass_matrix 0 1 (4) matrix:set biomass_matrix 2 1 (0.153156667)
matrix:set biomass_matrix 0 2 (7) matrix:set biomass_matrix 2 2 (0.270845867)
matrix:set biomass_matrix 0 3 (24) matrix:set biomass_matrix 2 3 (0.378976933)
matrix:set biomass_matrix 0 4 (24.5) matrix:set biomass_matrix 2 4 (0.382777067)
matrix:set biomass_matrix 0 5 (28) matrix:set biomass_matrix 2 5 (0.3668856)
matrix:set biomass_matrix 0 6 (30) matrix:set biomass_matrix 2 6 (0.3259478)
matrix:set biomass_matrix 0 7 (47) matrix:set biomass_matrix 2 7 (0.202443467)
matrix:set biomass_matrix 0 8 (50) matrix:set biomass_matrix 2 8 (0.222480533)
matrix:set biomass_matrix 0 9 (53) matrix:set biomass_matrix 2 9 (0.2103892)
matrix:set biomass_matrix 0 10 (71) matrix:set biomass_matrix 2 10 (0.206589067)
matrix:set biomass_matrix 0 11 (74) matrix:set biomass_matrix 2 11 (0.213671133)
matrix:set biomass_matrix 0 12 (77) matrix:set biomass_matrix 2 12 (0.203998067)
matrix:set biomass_matrix 0 13 (95) matrix:set biomass_matrix 2 13 (0.2109074)
matrix:set biomass_matrix 0 14 (98) matrix:set biomass_matrix 2 14 (0.193115867)
matrix:set biomass_matrix 0 15 (102) matrix:set biomass_matrix 2 15 (0.182060933)
matrix:set biomass_matrix 0 16 (119) matrix:set biomass_matrix 2 16 (0.1694514)

let column_matrix_nitrate 17 let row_matrix_nitrate 6
set nitrate_matrix matrix:make-constant row_matrix_nitrate column_matrix_nitrate 0

matrix:set nitrate_matrix 0 0 (0) matrix:set nitrate_matrix 2 0 (21.60951) matrix:set nitrate_matrix 1 0 (gnitrate)
matrix:set nitrate_matrix 0 1 (4) matrix:set nitrate_matrix 2 1 (16.42224)
matrix:set nitrate_matrix 0 2 (7) matrix:set nitrate_matrix 2 2 (19.884335)
matrix:set nitrate_matrix 0 3 (24) matrix:set nitrate_matrix 2 3 (14.25933)
matrix:set nitrate_matrix 0 4 (24.5) matrix:set nitrate_matrix 2 4 (21.196305)
matrix:set nitrate_matrix 0 5 (28) matrix:set nitrate_matrix 2 5 (15.784935)
matrix:set nitrate_matrix 0 6 (30) matrix:set nitrate_matrix 2 6 (19.45065)
matrix:set nitrate_matrix 0 7 (47) matrix:set nitrate_matrix 2 7 (17.16744)
matrix:set nitrate_matrix 0 8 (50) matrix:set nitrate_matrix 2 8 (10.15038)
matrix:set nitrate_matrix 0 9 (53) matrix:set nitrate_matrix 2 9 (15.426435)
matrix:set nitrate_matrix 0 10 (71) matrix:set nitrate_matrix 2 10 (7.95066)
matrix:set nitrate_matrix 0 11 (74) matrix:set nitrate_matrix 2 11 (12.17988)
matrix:set nitrate_matrix 0 12 (77) matrix:set nitrate_matrix 2 12 (10.57218)
matrix:set nitrate_matrix 0 13 (96) matrix:set nitrate_matrix 2 13 (7.759995)
matrix:set nitrate_matrix 0 14 (99) matrix:set nitrate_matrix 2 14 (13.250715)
matrix:set nitrate_matrix 0 15 (102) matrix:set nitrate_matrix 2 15 (11.33085)
matrix:set nitrate_matrix 0 16 (119) matrix:set nitrate_matrix 2 16 (13.93161)

let column_matrix_nitrite 17 let row_matrix_nitrite 6
set nitrite_matrix matrix:make-constant row_matrix_nitrite column_matrix_nitrite 0

matrix:set nitrite_matrix 0 0 (0) matrix:set nitrite_matrix 2 0 (0.011163522) matrix:set nitrite_matrix 1 0 (gno2-)
matrix:set nitrite_matrix 0 1 (3) matrix:set nitrite_matrix 2 1 (0.030220126)
matrix:set nitrite_matrix 0 2 (6) matrix:set nitrite_matrix 2 2 (0.00827044)
matrix:set nitrite_matrix 0 3 (24) matrix:set nitrite_matrix 2 3 (0.011981132)
matrix:set nitrite_matrix 0 4 (24.5) matrix:set nitrite_matrix 2 4 (0.01827044)
matrix:set nitrite_matrix 0 5 (27) matrix:set nitrite_matrix 2 5 (0.029622642)
matrix:set nitrite_matrix 0 6 (30) matrix:set nitrite_matrix 2 6 (0.017295597)
matrix:set nitrite_matrix 0 7 (49) matrix:set nitrite_matrix 2 7 (0.009968553)
matrix:set nitrite_matrix 0 8 (52) matrix:set nitrite_matrix 2 8 (0.006320755)
matrix:set nitrite_matrix 0 9 (55) matrix:set nitrite_matrix 2 9 (0.006540881)
matrix:set nitrite_matrix 0 10 (72) matrix:set nitrite_matrix 2 10 (0.005408805)
matrix:set nitrite_matrix 0 11 (74.5) matrix:set nitrite_matrix 2 11 (0.00663522)
matrix:set nitrite_matrix 0 12 (78.5) matrix:set nitrite_matrix 2 12 (0.005220126)
matrix:set nitrite_matrix 0 13 (96.5) matrix:set nitrite_matrix 2 13 (0.002924528)
matrix:set nitrite_matrix 0 14 (99) matrix:set nitrite_matrix 2 14 (0.005754717)
matrix:set nitrite_matrix 0 15 (102) matrix:set nitrite_matrix 2 15 (0.005377358)
matrix:set nitrite_matrix 0 16 (119) matrix:set nitrite_matrix 2 16 (0.005691824)

let column_matrix_nitrous_oxide 15 let row_matrix_nitrous_oxide 6
set nitrous_oxide_matrix matrix:make-constant row_matrix_nitrous_oxide column_matrix_nitrous_oxide 0

matrix:set nitrous_oxide_matrix 0 0 (0) matrix:set nitrous_oxide_matrix 2 0 (0.028413312) matrix:set nitrous_oxide_matrix 1 0 (gn2o)
matrix:set nitrous_oxide_matrix 0 1 (6) matrix:set nitrous_oxide_matrix 2 1 (0.034785811)
matrix:set nitrous_oxide_matrix 0 2 (24) matrix:set nitrous_oxide_matrix 2 2 (0.034617738)
matrix:set nitrous_oxide_matrix 0 3 (24.5) matrix:set nitrous_oxide_matrix 2 3 (0.036901924)
matrix:set nitrous_oxide_matrix 0 4 (27) matrix:set nitrous_oxide_matrix 2 4 (0.215676568)
matrix:set nitrous_oxide_matrix 0 5 (49) matrix:set nitrous_oxide_matrix 2 5 (0.506300774)
matrix:set nitrous_oxide_matrix 0 6 (52) matrix:set nitrous_oxide_matrix 2 6 (0.495535019)
matrix:set nitrous_oxide_matrix 0 7 (55) matrix:set nitrous_oxide_matrix 2 7 (0.544601612)
matrix:set nitrous_oxide_matrix 0 8 (72) matrix:set nitrous_oxide_matrix 2 8 (0.58388104)
matrix:set nitrous_oxide_matrix 0 9 (74.5) matrix:set nitrous_oxide_matrix 2 9 (0.582994072)
matrix:set nitrous_oxide_matrix 0 10 (78.5) matrix:set nitrous_oxide_matrix 2 10 (0.606296092)
matrix:set nitrous_oxide_matrix 0 11 (96.5) matrix:set nitrous_oxide_matrix 2 11 (0.53099924)
matrix:set nitrous_oxide_matrix 0 12 (99) matrix:set nitrous_oxide_matrix 2 12 (0.504279414)
matrix:set nitrous_oxide_matrix 0 13 (102) matrix:set nitrous_oxide_matrix 2 13 (0.58082208)
matrix:set nitrous_oxide_matrix 0 14 (119.5) matrix:set nitrous_oxide_matrix 2 14 (0.509617501)

] ; end procedure succinate low P.denitrificans

if Experiment = "Succ_High/Nitrate_Low"[

let column_matrix_biomass 17 let row_matrix_biomass 6
set biomass_matrix matrix:make-constant row_matrix_biomass column_matrix_biomass 0

matrix:set biomass_matrix 0 0 (0) matrix:set biomass_matrix 2 0 (0.0033683) matrix:set biomass_matrix 1 0 (gbacteria)
matrix:set biomass_matrix 0 1 (3) matrix:set biomass_matrix 2 1 (0.033683)
matrix:set biomass_matrix 0 2 (6) matrix:set biomass_matrix 2 2 (0.186552)
matrix:set biomass_matrix 0 3 (24) matrix:set biomass_matrix 2 3 (0.650341)
matrix:set biomass_matrix 0 4 (24.5) matrix:set biomass_matrix 2 4 (0.507836)
matrix:set biomass_matrix 0 5 (27) matrix:set biomass_matrix 2 5 (1.075265)
matrix:set biomass_matrix 0 6 (30) matrix:set biomass_matrix 2 6 (0.8221243)
matrix:set biomass_matrix 0 7 (48) matrix:set biomass_matrix 2 7 (0.4399518)
matrix:set biomass_matrix 0 8 (51) matrix:set biomass_matrix 2 8 (0.598521)
matrix:set biomass_matrix 0 9 (53) matrix:set biomass_matrix 2 9 (0.4161146)
matrix:set biomass_matrix 0 10 (72) matrix:set biomass_matrix 2 10 (0.4000504)
matrix:set biomass_matrix 0 11 (75) matrix:set biomass_matrix 2 11 (0.342012)
matrix:set biomass_matrix 0 12 (79) matrix:set biomass_matrix 2 12 (0.240963)
matrix:set biomass_matrix 0 13 (96) matrix:set biomass_matrix 2 13 (0.2842327)
matrix:set biomass_matrix 0 14 (99) matrix:set biomass_matrix 2 14 (0.2635047)
matrix:set biomass_matrix 0 15 (102) matrix:set biomass_matrix 2 15 (0.2492542)
matrix:set biomass_matrix 0 16 (120) matrix:set biomass_matrix 2 16 (0.2629865)

let column_matrix_nitrate 17 let row_matrix_nitrate 6
set nitrate_matrix matrix:make-constant row_matrix_nitrate column_matrix_nitrate 0

matrix:set nitrate_matrix 0 0 (0) matrix:set nitrate_matrix 2 0 (4.99825) matrix:set nitrate_matrix 1 0 (gnitrate)
matrix:set nitrate_matrix 0 1 (3) matrix:set nitrate_matrix 2 1 (5.0055)
matrix:set nitrate_matrix 0 2 (6) matrix:set nitrate_matrix 2 2 (5.1415)
matrix:set nitrate_matrix 0 3 (24) matrix:set nitrate_matrix 2 3 (4.30925)
matrix:set nitrate_matrix 0 4 (24.5) matrix:set nitrate_matrix 2 4 (4.33675)
matrix:set nitrate_matrix 0 5 (27) matrix:set nitrate_matrix 2 5 (1.5245)
matrix:set nitrate_matrix 0 6 (30) matrix:set nitrate_matrix 2 6 (0.1645)
matrix:set nitrate_matrix 0 7 (48) matrix:set nitrate_matrix 2 7 (0.0155)
matrix:set nitrate_matrix 0 8 (51) matrix:set nitrate_matrix 2 8 (0.0065)
matrix:set nitrate_matrix 0 9 (53) matrix:set nitrate_matrix 2 9 (0.00325)
matrix:set nitrate_matrix 0 10 (72) matrix:set nitrate_matrix 2 10 (0.004)
matrix:set nitrate_matrix 0 11 (75) matrix:set nitrate_matrix 2 11 (0.008)
matrix:set nitrate_matrix 0 12 (79) matrix:set nitrate_matrix 2 12 (0.003)
matrix:set nitrate_matrix 0 13 (96) matrix:set nitrate_matrix 2 13 (0.0025)
matrix:set nitrate_matrix 0 14 (99) matrix:set nitrate_matrix 2 14 (0.0035)
matrix:set nitrate_matrix 0 15 (102) matrix:set nitrate_matrix 2 15 (0.006)
matrix:set nitrate_matrix 0 16 (120) matrix:set nitrate_matrix 2 16 (0.005)

let column_matrix_nitrite 17 let row_matrix_nitrite 6
set nitrite_matrix matrix:make-constant row_matrix_nitrite column_matrix_nitrite 0

matrix:set nitrite_matrix 0 0 (0) matrix:set nitrite_matrix 2 0 (0.0255) matrix:set nitrite_matrix 1 0 (gno2-)
matrix:set nitrite_matrix 0 1 (3) matrix:set nitrite_matrix 2 1 (0.02075)
matrix:set nitrite_matrix 0 2 (6) matrix:set nitrite_matrix 2 2 (0.03875)
matrix:set nitrite_matrix 0 3 (24) matrix:set nitrite_matrix 2 3 (0.1715)
matrix:set nitrite_matrix 0 4 (24.5) matrix:set nitrite_matrix 2 4 (0.091)
matrix:set nitrite_matrix 0 5 (27) matrix:set nitrite_matrix 2 5 (0.19415)
matrix:set nitrite_matrix 0 6 (30) matrix:set nitrite_matrix 2 6 (0.193)
matrix:set nitrite_matrix 0 7 (48) matrix:set nitrite_matrix 2 7 (0.02275)
matrix:set nitrite_matrix 0 8 (51) matrix:set nitrite_matrix 2 8 (0.014)
matrix:set nitrite_matrix 0 9 (53) matrix:set nitrite_matrix 2 9 (0.014)
matrix:set nitrite_matrix 0 10 (72) matrix:set nitrite_matrix 2 10 (0.0125)
matrix:set nitrite_matrix 0 11 (75) matrix:set nitrite_matrix 2 11 (0.025)
matrix:set nitrite_matrix 0 12 (79) matrix:set nitrite_matrix 2 12 (0.00975)
matrix:set nitrite_matrix 0 13 (96) matrix:set nitrite_matrix 2 13 (0.01375)
matrix:set nitrite_matrix 0 14 (99) matrix:set nitrite_matrix 2 14 (0.0135)
matrix:set nitrite_matrix 0 15 (102) matrix:set nitrite_matrix 2 15 (0.01825)
matrix:set nitrite_matrix 0 16 (120) matrix:set nitrite_matrix 2 16 (0.01725)

let column_matrix_nitrous_oxide 13 let row_matrix_nitrous_oxide 6
set nitrous_oxide_matrix matrix:make-constant row_matrix_nitrous_oxide column_matrix_nitrous_oxide 0

matrix:set nitrous_oxide_matrix 0 0 (0) matrix:set nitrous_oxide_matrix 2 0 (0.003003178) matrix:set nitrous_oxide_matrix 1 0 (gn2o)
matrix:set nitrous_oxide_matrix 0 1 (6) matrix:set nitrous_oxide_matrix 2 1 (0.003003178)
matrix:set nitrous_oxide_matrix 0 2 (24) matrix:set nitrous_oxide_matrix 2 2 (0.000886128)
matrix:set nitrous_oxide_matrix 0 3 (27) matrix:set nitrous_oxide_matrix 2 3 (0.005377157)
matrix:set nitrous_oxide_matrix 0 4 (30) matrix:set nitrous_oxide_matrix 2 4 (0.002591133)
matrix:set nitrous_oxide_matrix 0 5 (48) matrix:set nitrous_oxide_matrix 2 5 (0.000710695)
matrix:set nitrous_oxide_matrix 0 6 (53) matrix:set nitrous_oxide_matrix 2 6 (0.000244818)
matrix:set nitrous_oxide_matrix 0 7 (72) matrix:set nitrous_oxide_matrix 2 7 (0.001205919)
matrix:set nitrous_oxide_matrix 0 8 (79) matrix:set nitrous_oxide_matrix 2 8 (0.000914869)
matrix:set nitrous_oxide_matrix 0 9 (96) matrix:set nitrous_oxide_matrix 2 9 (0.000611436)
matrix:set nitrous_oxide_matrix 0 10 (99) matrix:set nitrous_oxide_matrix 2 10 (0.001057633)
matrix:set nitrous_oxide_matrix 0 11 (102) matrix:set nitrous_oxide_matrix 2 11 (0.001152196)
matrix:set nitrous_oxide_matrix 0 12 (120) matrix:set nitrous_oxide_matrix 2 12 (0.00058742)

] ; End procedure succinate high P.denitrificans

] ; End bacterium procedure P.denitrificans

if Denitrifying_bacterium = "A._xylosoxidans" [

if Experiment = "Succ_Low/Nitrate_High"[

let column_matrix_biomass 11 let row_matrix_biomass 6
set biomass_matrix matrix:make-constant row_matrix_biomass column_matrix_biomass 0

matrix:set biomass_matrix 0 0 (0) matrix:set biomass_matrix 2 0 (0.05182) matrix:set biomass_matrix 1 0 (gbacteria)
matrix:set biomass_matrix 0 1 (6) matrix:set biomass_matrix 2 1 (0.20210)
matrix:set biomass_matrix 0 2 (24) matrix:set biomass_matrix 2 2 (0.27983)
matrix:set biomass_matrix 0 3 (25) matrix:set biomass_matrix 2 3 (0.25392)
matrix:set biomass_matrix 0 4 (30) matrix:set biomass_matrix 2 4 (0.24692)
matrix:set biomass_matrix 0 5 (48) matrix:set biomass_matrix 2 5 (0.15935)
matrix:set biomass_matrix 0 6 (54) matrix:set biomass_matrix 2 6 (0.14561)
matrix:set biomass_matrix 0 7 (78) matrix:set biomass_matrix 2 7 (0.11996)
matrix:set biomass_matrix 0 8 (97) matrix:set biomass_matrix 2 8 (0.10830)
matrix:set biomass_matrix 0 9 (102) matrix:set biomass_matrix 2 9 (0.11323)
matrix:set biomass_matrix 0 10 (120) matrix:set biomass_matrix 2 10 (0.10830)

let column_matrix_nitrate 12 let row_matrix_nitrate 6
set nitrate_matrix matrix:make-constant row_matrix_nitrate column_matrix_nitrate 0

matrix:set nitrate_matrix 0 0 (0) matrix:set nitrate_matrix 2 0 (21.74694) matrix:set nitrate_matrix 1 0 (gnitrate)
matrix:set nitrate_matrix 0 1 (6) matrix:set nitrate_matrix 2 1 (19.54182)
matrix:set nitrate_matrix 0 2 (24) matrix:set nitrate_matrix 2 2 (21.63708)
matrix:set nitrate_matrix 0 3 (24.5) matrix:set nitrate_matrix 2 3 (19.77804)
matrix:set nitrate_matrix 0 4 (30) matrix:set nitrate_matrix 2 4 (8.36640)
matrix:set nitrate_matrix 0 5 (48) matrix:set nitrate_matrix 2 5 (6.47461)
matrix:set nitrate_matrix 0 6 (54) matrix:set nitrate_matrix 2 6 (2.98806)
matrix:set nitrate_matrix 0 7 (72) matrix:set nitrate_matrix 2 7 (3.13756)
matrix:set nitrate_matrix 0 8 (78) matrix:set nitrate_matrix 2 8 (5.93996)
matrix:set nitrate_matrix 0 9 (97) matrix:set nitrate_matrix 2 9 (4.58357)
matrix:set nitrate_matrix 0 10 (102) matrix:set nitrate_matrix 2 10 (6.00180)
matrix:set nitrate_matrix 0 11 (120) matrix:set nitrate_matrix 2 11 (7.00376)

let column_matrix_nitrite 13 let row_matrix_nitrite 6
set nitrite_matrix matrix:make-constant row_matrix_nitrite column_matrix_nitrite 0

matrix:set nitrite_matrix 0 0 (0) matrix:set nitrite_matrix 2 0 (0.00765) matrix:set nitrite_matrix 1 0 (gno2-)
matrix:set nitrite_matrix 0 1 (5) matrix:set nitrite_matrix 2 1 (0.03032)
matrix:set nitrite_matrix 0 2 (24) matrix:set nitrite_matrix 2 2 (0.01729)
matrix:set nitrite_matrix 0 3 (25) matrix:set nitrite_matrix 2 3 (0.02565)
matrix:set nitrite_matrix 0 4 (30) matrix:set nitrite_matrix 2 4 (1.72077)
matrix:set nitrite_matrix 0 5 (48) matrix:set nitrite_matrix 2 5 (4.54305)
matrix:set nitrite_matrix 0 6 (54) matrix:set nitrite_matrix 2 6 (7.66966)
matrix:set nitrite_matrix 0 7 (72) matrix:set nitrite_matrix 2 7 (4.84258)
matrix:set nitrite_matrix 0 8 (78) matrix:set nitrite_matrix 2 8 (7.02693)
matrix:set nitrite_matrix 0 9 (97) matrix:set nitrite_matrix 2 9 (6.56384)
matrix:set nitrite_matrix 0 10 (99) matrix:set nitrite_matrix 2 10 (3.38517)
matrix:set nitrite_matrix 0 11 (102) matrix:set nitrite_matrix 2 11 (4.96006)
matrix:set nitrite_matrix 0 12 (120) matrix:set nitrite_matrix 2 12 (3.84764)

let column_matrix_nitrous_oxide 12 let row_matrix_nitrous_oxide 6
set nitrous_oxide_matrix matrix:make-constant row_matrix_nitrous_oxide column_matrix_nitrous_oxide 0

matrix:set nitrous_oxide_matrix 0 0 (0) matrix:set nitrous_oxide_matrix 2 0 (0.01818) matrix:set nitrous_oxide_matrix 1 0 (gn2o)
matrix:set nitrous_oxide_matrix 0 1 (6) matrix:set nitrous_oxide_matrix 2 1 (0.00771)
matrix:set nitrous_oxide_matrix 0 2 (24) matrix:set nitrous_oxide_matrix 2 2 (0.00330)
matrix:set nitrous_oxide_matrix 0 3 (24.5) matrix:set nitrous_oxide_matrix 2 3 (0.00559)
matrix:set nitrous_oxide_matrix 0 4 (30) matrix:set nitrous_oxide_matrix 2 4 (0.28329)
matrix:set nitrous_oxide_matrix 0 5 (48) matrix:set nitrous_oxide_matrix 2 5 (0.95467)
matrix:set nitrous_oxide_matrix 0 6 (54) matrix:set nitrous_oxide_matrix 2 6 (1.44598)
matrix:set nitrous_oxide_matrix 0 7 (72) matrix:set nitrous_oxide_matrix 2 7 (1.60833)
matrix:set nitrous_oxide_matrix 0 8 (78) matrix:set nitrous_oxide_matrix 2 8 (1.02808)
matrix:set nitrous_oxide_matrix 0 9 (97) matrix:set nitrous_oxide_matrix 2 9 (1.24110)
matrix:set nitrous_oxide_matrix 0 10 (99) matrix:set nitrous_oxide_matrix 2 10 (16.66431)
matrix:set nitrous_oxide_matrix 0 11 (102) matrix:set nitrous_oxide_matrix 2 11 (0.84064)

] ; end procedure succinate low A.xylosxidans

if Experiment = "Succ_High/Nitrate_Low"[

let column_matrix_biomass 15 let row_matrix_biomass 6
set biomass_matrix matrix:make-constant row_matrix_biomass column_matrix_biomass 0

matrix:set biomass_matrix 0 0 (0) matrix:set biomass_matrix 2 0 (0.07695) matrix:set biomass_matrix 1 0 (gbacteria)
matrix:set biomass_matrix 0 1 (6) matrix:set biomass_matrix 2 1 (0.20573)
matrix:set biomass_matrix 0 2 (24) matrix:set biomass_matrix 2 2 (0.58194)
matrix:set biomass_matrix 0 3 (24.5) matrix:set biomass_matrix 2 3 (0.50861)
matrix:set biomass_matrix 0 4 (27) matrix:set biomass_matrix 2 4 (0.51043)
matrix:set biomass_matrix 0 5 (31) matrix:set biomass_matrix 2 5 (0.57261)
matrix:set biomass_matrix 0 6 (48) matrix:set biomass_matrix 2 6 (0.50214)
matrix:set biomass_matrix 0 7 (52) matrix:set biomass_matrix 2 7 (0.42337)
matrix:set biomass_matrix 0 8 (73) matrix:set biomass_matrix 2 8 (0.26083)
matrix:set biomass_matrix 0 9 (75) matrix:set biomass_matrix 2 9 (0.25910)
matrix:set biomass_matrix 0 10 (79) matrix:set biomass_matrix 2 10 (0.36689)
matrix:set biomass_matrix 0 11 (97) matrix:set biomass_matrix 2 11 (0.31196)
matrix:set biomass_matrix 0 12 (99) matrix:set biomass_matrix 2 12 (0.28838)
matrix:set biomass_matrix 0 13 (102) matrix:set biomass_matrix 2 13 (0.31066)
matrix:set biomass_matrix 0 14 (120) matrix:set biomass_matrix 2 14 (0.28631)

let column_matrix_nitrate 14 let row_matrix_nitrate 6
set nitrate_matrix matrix:make-constant row_matrix_nitrate column_matrix_nitrate 0

matrix:set nitrate_matrix 0 0 (0) matrix:set nitrate_matrix 2 0 (5.15375) matrix:set nitrate_matrix 1 0 (gnitrate)
matrix:set nitrate_matrix 0 1 (4) matrix:set nitrate_matrix 2 1 (4.68250)
matrix:set nitrate_matrix 0 2 (24) matrix:set nitrate_matrix 2 2 (4.00950)
matrix:set nitrate_matrix 0 3 (24.5) matrix:set nitrate_matrix 2 3 (2.40450)
matrix:set nitrate_matrix 0 4 (27) matrix:set nitrate_matrix 2 4 (0.78025)
matrix:set nitrate_matrix 0 5 (31) matrix:set nitrate_matrix 2 5 (0.29225)
matrix:set nitrate_matrix 0 6 (49) matrix:set nitrate_matrix 2 6 (0.20075)
matrix:set nitrate_matrix 0 7 (54) matrix:set nitrate_matrix 2 7 (0.13525)
matrix:set nitrate_matrix 0 8 (73) matrix:set nitrate_matrix 2 8 (0.13325)
matrix:set nitrate_matrix 0 9 (79) matrix:set nitrate_matrix 2 9 (0.14575)
matrix:set nitrate_matrix 0 10 (97) matrix:set nitrate_matrix 2 10 (0.15725)
matrix:set nitrate_matrix 0 11 (99) matrix:set nitrate_matrix 2 11 (0.10500)
matrix:set nitrate_matrix 0 12 (102) matrix:set nitrate_matrix 2 12 (0.15638)
matrix:set nitrate_matrix 0 13 (120) matrix:set nitrate_matrix 2 13 (0.01663)

let column_matrix_nitrite 14 let row_matrix_nitrite 6
set nitrite_matrix matrix:make-constant row_matrix_nitrite column_matrix_nitrite 0

matrix:set nitrite_matrix 0 0 (0) matrix:set nitrite_matrix 2 0 (0.36863) matrix:set nitrite_matrix 1 0 (gno2-)
matrix:set nitrite_matrix 0 1 (4) matrix:set nitrite_matrix 2 1 (0.65250)
matrix:set nitrite_matrix 0 2 (24) matrix:set nitrite_matrix 2 2 (0.54525)
matrix:set nitrite_matrix 0 3 (24.5) matrix:set nitrite_matrix 2 3 (1.01600)
matrix:set nitrite_matrix 0 4 (27) matrix:set nitrite_matrix 2 4 (0.84300)
matrix:set nitrite_matrix 0 5 (31) matrix:set nitrite_matrix 2 5 (0.09875)
matrix:set nitrite_matrix 0 6 (49) matrix:set nitrite_matrix 2 6 (0.02325)
matrix:set nitrite_matrix 0 7 (54) matrix:set nitrite_matrix 2 7 (0.01525)
matrix:set nitrite_matrix 0 8 (73) matrix:set nitrite_matrix 2 8 (0.01050)
matrix:set nitrite_matrix 0 9 (79) matrix:set nitrite_matrix 2 9 (0.01400)
matrix:set nitrite_matrix 0 10 (97) matrix:set nitrite_matrix 2 10 (0.01450)
matrix:set nitrite_matrix 0 11 (99) matrix:set nitrite_matrix 2 11 (0.18683)
matrix:set nitrite_matrix 0 12 (102) matrix:set nitrite_matrix 2 12 (0.34729)
matrix:set nitrite_matrix 0 13 (120) matrix:set nitrite_matrix 2 13 (0.21405)

let column_matrix_nitrous_oxide 13 let row_matrix_nitrous_oxide 6
set nitrous_oxide_matrix matrix:make-constant row_matrix_nitrous_oxide column_matrix_nitrous_oxide 0

matrix:set nitrous_oxide_matrix 0 0 (0) matrix:set nitrous_oxide_matrix 2 0 (0.00006263) matrix:set nitrous_oxide_matrix 1 0 (gn2o)
matrix:set nitrous_oxide_matrix 0 1 (7) matrix:set nitrous_oxide_matrix 2 1 (0.00013455)
matrix:set nitrous_oxide_matrix 0 2 (24) matrix:set nitrous_oxide_matrix 2 2 (0.00019376)
matrix:set nitrous_oxide_matrix 0 3 (24.5) matrix:set nitrous_oxide_matrix 2 3 (0.00024715)
matrix:set nitrous_oxide_matrix 0 4 (28) matrix:set nitrous_oxide_matrix 2 4 (0.00715063)
matrix:set nitrous_oxide_matrix 0 5 (47) matrix:set nitrous_oxide_matrix 2 5 (0.00192994)
matrix:set nitrous_oxide_matrix 0 6 (53) matrix:set nitrous_oxide_matrix 2 6 (0.00027866)
matrix:set nitrous_oxide_matrix 0 7 (74) matrix:set nitrous_oxide_matrix 2 7 (0.00045744)
matrix:set nitrous_oxide_matrix 0 8 (77) matrix:set nitrous_oxide_matrix 2 8 (0.00025220)
matrix:set nitrous_oxide_matrix 0 9 (95) matrix:set nitrous_oxide_matrix 2 9 (0.00027134)
matrix:set nitrous_oxide_matrix 0 10 (98) matrix:set nitrous_oxide_matrix 2 10 (0.00024276)
matrix:set nitrous_oxide_matrix 0 11 (102) matrix:set nitrous_oxide_matrix 2 11 (0.00053833)
matrix:set nitrous_oxide_matrix 0 12 (119) matrix:set nitrous_oxide_matrix 2 12 (0.00026046)

] ; end procedure succinate high A.xylosxidans

] ; end procedure bacteria A.xylosxidans

end

to get_data_machine.learning

  if Denitrifying_bacterium = "P._denitrificans" [

  if Experiment = "Succ_Low/Nitrate_High"[

  if time_now = 3 [matrix:set nitrite_matrix 1 1 gno2-]
  if time_now = 4 [matrix:set biomass_matrix 1 1 gbacteria matrix:set nitrate_matrix 1 1 gnitrate]
  if time_now = 6 [matrix:set nitrite_matrix 1 2 gno2- matrix:set nitrous_oxide_matrix 1 1 gn2o]
  if time_now = 7 [matrix:set biomass_matrix 1 2 gbacteria matrix:set nitrate_matrix 1 2 gnitrate]
  if time_now = 24 [matrix:set biomass_matrix 1 3 gbacteria matrix:set nitrate_matrix 1 3 gnitrate matrix:set nitrite_matrix 1 3 gno2- matrix:set nitrous_oxide_matrix 1 2 gn2o]
  if time_now = 24.5 [matrix:set biomass_matrix 1 4 gbacteria matrix:set nitrate_matrix 1 4 gnitrate matrix:set nitrite_matrix 1 4 gno2- matrix:set nitrous_oxide_matrix 1 3 gn2o]
  if time_now = 27 [matrix:set nitrite_matrix 1 5 gno2- matrix:set nitrous_oxide_matrix 1 4 gn2o]
  if time_now = 28 [matrix:set biomass_matrix 1 5 gbacteria matrix:set nitrate_matrix 1 5 gnitrate]
  if time_now = 30 [matrix:set biomass_matrix 1 6 gbacteria matrix:set nitrate_matrix 1 6 gnitrate matrix:set nitrite_matrix 1 6 gno2-]
  if time_now = 47 [matrix:set biomass_matrix 1 7 gbacteria matrix:set nitrate_matrix 1 7 gnitrate]
  if time_now = 49 [matrix:set nitrite_matrix 1 7 gno2- matrix:set nitrous_oxide_matrix 1 5 gn2o]
  if time_now = 50 [matrix:set biomass_matrix 1 8 gbacteria matrix:set nitrate_matrix 1 8 gnitrate]
  if time_now = 52 [matrix:set nitrite_matrix 1 8 gno2- matrix:set nitrous_oxide_matrix 1 6 gn2o]
  if time_now = 53 [matrix:set biomass_matrix 1 9 gbacteria matrix:set nitrate_matrix 1 9 gnitrate]
  if time_now = 55 [matrix:set nitrite_matrix 1 9 gno2- matrix:set nitrous_oxide_matrix 1 7 gn2o]
  if time_now = 71 [matrix:set biomass_matrix 1 10 gbacteria matrix:set nitrate_matrix 1 10 gnitrate]
  if time_now = 72 [matrix:set nitrite_matrix 1 10 gno2- matrix:set nitrous_oxide_matrix 1 8 gn2o]
  if time_now = 74 [matrix:set biomass_matrix 1 11 gbacteria matrix:set nitrate_matrix 1 11 gnitrate]
  if time_now = 74.5 [matrix:set nitrite_matrix 1 11 gno2- matrix:set nitrous_oxide_matrix 1 9 gn2o]
  if time_now = 77 [matrix:set biomass_matrix 1 12 gbacteria matrix:set nitrate_matrix 1 12 gnitrate]
  if time_now = 78.5 [matrix:set nitrite_matrix 1 12 gno2- matrix:set nitrous_oxide_matrix 1 10 gn2o]
  if time_now = 95 [matrix:set biomass_matrix 1 13 gbacteria]
  if time_now = 96 [matrix:set nitrate_matrix 1 13 gnitrate]
  if time_now = 96.5 [matrix:set nitrite_matrix 1 13 gno2- matrix:set nitrous_oxide_matrix 1 11 gn2o]
  if time_now = 98 [matrix:set biomass_matrix 1 14 gbacteria]
  if time_now = 99 [matrix:set nitrate_matrix 1 14 gnitrate matrix:set nitrite_matrix 1 14 gno2- matrix:set nitrous_oxide_matrix 1 12 gn2o]
  if time_now = 102 [matrix:set biomass_matrix 1 15 gbacteria matrix:set nitrate_matrix 1 15 gnitrate matrix:set nitrite_matrix 1 15 gno2- matrix:set nitrous_oxide_matrix 1 13 gn2o]
  if time_now = 119 [matrix:set biomass_matrix 1 16 gbacteria matrix:set nitrate_matrix 1 16 gnitrate matrix:set nitrite_matrix 1 16 gno2-]
  if time_now = 119.5 [matrix:set nitrous_oxide_matrix 1 14 gn2o]

  ] ; end procedure succinate low P.denitrificans

  if Experiment = "Succ_High/Nitrate_Low"[

  if time_now = 3 [matrix:set biomass_matrix 1 1 gbacteria matrix:set nitrate_matrix 1 1 gnitrate matrix:set nitrite_matrix 1 1 gno2-]
  if time_now = 6 [matrix:set biomass_matrix 1 2 gbacteria matrix:set nitrate_matrix 1 2 gnitrate matrix:set nitrite_matrix 1 2 gno2- matrix:set nitrous_oxide_matrix 1 1 gn2o]
  if time_now = 24 [matrix:set biomass_matrix 1 3 gbacteria matrix:set nitrate_matrix 1 3 gnitrate matrix:set nitrite_matrix 1 3 gno2- matrix:set nitrous_oxide_matrix 1 2 gn2o]
  if time_now = 24.5 [matrix:set biomass_matrix 1 4 gbacteria matrix:set nitrate_matrix 1 4 gnitrate matrix:set nitrite_matrix 1 4 gno2-]
  if time_now = 27 [matrix:set biomass_matrix 1 5 gbacteria matrix:set nitrate_matrix 1 5 gnitrate matrix:set nitrite_matrix 1 5 gno2- matrix:set nitrous_oxide_matrix 1 3 gn2o]
  if time_now = 30 [matrix:set biomass_matrix 1 6 gbacteria matrix:set nitrate_matrix 1 6 gnitrate matrix:set nitrite_matrix 1 6 gno2- matrix:set nitrous_oxide_matrix 1 4 gn2o]
  if time_now = 48 [matrix:set biomass_matrix 1 7 gbacteria matrix:set nitrate_matrix 1 7 gnitrate matrix:set nitrite_matrix 1 7 gno2- matrix:set nitrous_oxide_matrix 1 5 gn2o]
  if time_now = 51 [matrix:set biomass_matrix 1 8 gbacteria matrix:set nitrate_matrix 1 8 gnitrate matrix:set nitrite_matrix 1 8 gno2-]
  if time_now = 53 [matrix:set biomass_matrix 1 9 gbacteria matrix:set nitrate_matrix 1 9 gnitrate matrix:set nitrite_matrix 1 9 gno2- matrix:set nitrous_oxide_matrix 1 6 gn2o]
  if time_now = 72 [matrix:set biomass_matrix 1 10 gbacteria matrix:set nitrate_matrix 1 10 gnitrate matrix:set nitrite_matrix 1 10 gno2- matrix:set nitrous_oxide_matrix 1 7 gn2o]
  if time_now = 75 [matrix:set biomass_matrix 1 11 gbacteria matrix:set nitrate_matrix 1 11 gnitrate matrix:set nitrite_matrix 1 11 gno2-]
  if time_now = 79 [matrix:set biomass_matrix 1 12 gbacteria matrix:set nitrate_matrix 1 12 gnitrate matrix:set nitrite_matrix 1 12 gno2- matrix:set nitrous_oxide_matrix 1 8 gn2o]
  if time_now = 96 [matrix:set biomass_matrix 1 13 gbacteria matrix:set nitrate_matrix 1 13 gnitrate matrix:set nitrite_matrix 1 13 gno2- matrix:set nitrous_oxide_matrix 1 9 gn2o]
  if time_now = 99 [matrix:set biomass_matrix 1 14 gbacteria matrix:set nitrate_matrix 1 14 gnitrate matrix:set nitrite_matrix 1 14 gno2- matrix:set nitrous_oxide_matrix 1 10 gn2o]
  if time_now = 102 [matrix:set biomass_matrix 1 15 gbacteria matrix:set nitrate_matrix 1 15 gnitrate matrix:set nitrite_matrix 1 15 gno2- matrix:set nitrous_oxide_matrix 1 11 gn2o]
  if time_now = 120 [matrix:set biomass_matrix 1 16 gbacteria matrix:set nitrate_matrix 1 16 gnitrate matrix:set nitrite_matrix 1 16 gno2- matrix:set nitrous_oxide_matrix 1 12 gn2o]

  ] ; end procedure succinate high P.denitrificans

  ] ; End procedure bacteria P.denitrificans

  if Denitrifying_bacterium = "A._xylosoxidans" [

  if Experiment = "Succ_Low/Nitrate_High"[

  if time_now = 5 [matrix:set nitrite_matrix 1 1 gno2-]
  if time_now = 6 [matrix:set biomass_matrix 1 1 gbacteria matrix:set nitrate_matrix 1 1 gnitrate matrix:set nitrous_oxide_matrix 1 1 gn2o]
  if time_now = 24 [matrix:set biomass_matrix 1 2 gbacteria matrix:set nitrate_matrix 1 2 gnitrate matrix:set nitrite_matrix 1 2 gno2- matrix:set nitrous_oxide_matrix 1 2 gn2o]
  if time_now = 24.5 [matrix:set nitrate_matrix 1 3 gnitrate matrix:set nitrous_oxide_matrix 1 3 gn2o]
  if time_now = 25 [matrix:set biomass_matrix 1 3 gbacteria matrix:set nitrite_matrix 1 3 gno2-]
  if time_now = 30 [matrix:set biomass_matrix 1 4 gbacteria matrix:set nitrate_matrix 1 4 gnitrate matrix:set nitrite_matrix 1 4 gno2- matrix:set nitrous_oxide_matrix 1 4 gn2o]
  if time_now = 48 [matrix:set biomass_matrix 1 5 gbacteria matrix:set nitrate_matrix 1 5 gnitrate matrix:set nitrite_matrix 1 5 gno2- matrix:set nitrous_oxide_matrix 1 5 gn2o]
  if time_now = 54 [matrix:set biomass_matrix 1 6 gbacteria matrix:set nitrate_matrix 1 6 gnitrate matrix:set nitrite_matrix 1 6 gno2- matrix:set nitrous_oxide_matrix 1 6 gn2o]
  if time_now = 72 [matrix:set nitrate_matrix 1 7 gnitrate matrix:set nitrite_matrix 1 7 gno2- matrix:set nitrous_oxide_matrix 1 7 gn2o]
  if time_now = 78 [matrix:set biomass_matrix 1 7 gbacteria matrix:set nitrate_matrix 1 8 gnitrate matrix:set nitrite_matrix 1 8 gno2- matrix:set nitrous_oxide_matrix 1 8 gn2o]
  if time_now = 97 [matrix:set biomass_matrix 1 8 gbacteria matrix:set nitrate_matrix 1 9 gnitrate matrix:set nitrite_matrix 1 9 gno2- matrix:set nitrous_oxide_matrix 1 9 gn2o]
  if time_now = 99 [matrix:set nitrite_matrix 1 10 gno2- matrix:set nitrous_oxide_matrix 1 10 gn2o]
  if time_now = 102 [matrix:set biomass_matrix 1 9 gbacteria matrix:set nitrate_matrix 1 10 gnitrate matrix:set nitrite_matrix 1 11 gno2- matrix:set nitrous_oxide_matrix 1 11 gn2o]
  if time_now = 120 [matrix:set biomass_matrix 1 10 gbacteria matrix:set nitrate_matrix 1 11 gnitrate matrix:set nitrite_matrix 1 12 gno2-]

  ] ; end procedure succinate low A.Xylosoxidans

  if Experiment = "Succ_High/Nitrate_Low"[

  if time_now = 4 [matrix:set nitrate_matrix 1 1 gnitrate matrix:set nitrite_matrix 1 1 gno2-]
  if time_now = 6 [matrix:set biomass_matrix 1 1 gbacteria]
  if time_now = 7 [matrix:set nitrous_oxide_matrix 1 1 gn2o]
  if time_now = 24 [matrix:set biomass_matrix 1 2 gbacteria matrix:set nitrate_matrix 1 2 gnitrate matrix:set nitrite_matrix 1 2 gno2- matrix:set nitrous_oxide_matrix 1 2 gn2o]
  if time_now = 24.5 [matrix:set biomass_matrix 1 3 gbacteria matrix:set nitrate_matrix 1 3 gnitrate matrix:set nitrite_matrix 1 3 gno2- matrix:set nitrous_oxide_matrix 1 3 gn2o]
  if time_now = 27 [matrix:set biomass_matrix 1 4 gbacteria matrix:set nitrate_matrix 1 4 gnitrate matrix:set nitrite_matrix 1 4 gno2-]
  if time_now = 28 [matrix:set nitrous_oxide_matrix 1 4 gn2o]
  if time_now = 31 [matrix:set biomass_matrix 1 5 gbacteria matrix:set nitrate_matrix 1 5 gnitrate matrix:set nitrite_matrix 1 5 gno2-]
  if time_now = 47 [matrix:set nitrous_oxide_matrix 1 5 gn2o]
  if time_now = 48 [matrix:set biomass_matrix 1 6 gbacteria]
  if time_now = 49 [matrix:set nitrate_matrix 1 6 gnitrate matrix:set nitrite_matrix 1 6 gno2-]
  if time_now = 52 [matrix:set biomass_matrix 1 7 gbacteria]
  if time_now = 53 [matrix:set nitrous_oxide_matrix 1 6 gn2o]
  if time_now = 54 [matrix:set nitrate_matrix 1 7 gnitrate matrix:set nitrite_matrix 1 7 gno2-]
  if time_now = 73 [matrix:set biomass_matrix 1 8 gbacteria matrix:set nitrate_matrix 1 8 gnitrate matrix:set nitrite_matrix 1 8 gno2-]
  if time_now = 74 [matrix:set nitrous_oxide_matrix 1 7 gn2o]
  if time_now = 75 [matrix:set biomass_matrix 1 9 gbacteria]
  if time_now = 77 [matrix:set nitrous_oxide_matrix 1 8 gn2o]
  if time_now = 79 [matrix:set biomass_matrix 1 10 gbacteria matrix:set nitrate_matrix 1 9 gnitrate matrix:set nitrite_matrix 1 9 gno2-]
  if time_now = 95 [matrix:set nitrous_oxide_matrix 1 9 gn2o]
  if time_now = 97 [matrix:set biomass_matrix 1 11 gbacteria matrix:set nitrate_matrix 1 10 gnitrate matrix:set nitrite_matrix 1 10 gno2-]
  if time_now = 98 [matrix:set nitrous_oxide_matrix 1 10 gn2o]
  if time_now = 99 [matrix:set biomass_matrix 1 12 gbacteria matrix:set nitrate_matrix 1 11 gnitrate matrix:set nitrite_matrix 1 11 gno2-]
  if time_now = 102 [matrix:set biomass_matrix 1 13 gbacteria matrix:set nitrate_matrix 1 12 gnitrate matrix:set nitrite_matrix 1 12 gno2- matrix:set nitrous_oxide_matrix 1 11 gn2o]
  if time_now = 119 [matrix:set nitrous_oxide_matrix 1 12 gn2o]
  if time_now = 120 [matrix:set biomass_matrix 1 14 gbacteria matrix:set nitrate_matrix 1 13 gnitrate matrix:set nitrite_matrix 1 13 gno2-]

  ] ; end procedure succinate high A.xylosoxidans

  ] ; end procedure bacteria A.xylosoxidans

end

to machine.learning

  if Denitrifying_bacterium = "P._denitrificans" [

  if Experiment = "Succ_Low/Nitrate_High"[

  matrix:set biomass_matrix 3 0 (matrix:get biomass_matrix 1 0 - matrix:get biomass_matrix 2 0) matrix:set biomass_matrix 4 0 (matrix:get biomass_matrix 1 0 + matrix:get biomass_matrix 2 0)
  matrix:set biomass_matrix 3 1 (matrix:get biomass_matrix 1 1 - matrix:get biomass_matrix 2 1) matrix:set biomass_matrix 4 1 (matrix:get biomass_matrix 1 1 + matrix:get biomass_matrix 2 1)
  matrix:set biomass_matrix 3 2 (matrix:get biomass_matrix 1 2 - matrix:get biomass_matrix 2 2) matrix:set biomass_matrix 4 2 (matrix:get biomass_matrix 1 2 + matrix:get biomass_matrix 2 2)
  matrix:set biomass_matrix 3 3 (matrix:get biomass_matrix 1 3 - matrix:get biomass_matrix 2 3) matrix:set biomass_matrix 4 3 (matrix:get biomass_matrix 1 3 + matrix:get biomass_matrix 2 3)
  matrix:set biomass_matrix 3 4 (matrix:get biomass_matrix 1 4 - matrix:get biomass_matrix 2 4) matrix:set biomass_matrix 4 4 (matrix:get biomass_matrix 1 4 + matrix:get biomass_matrix 2 4)
  matrix:set biomass_matrix 3 5 (matrix:get biomass_matrix 1 5 - matrix:get biomass_matrix 2 5) matrix:set biomass_matrix 4 5 (matrix:get biomass_matrix 1 5 + matrix:get biomass_matrix 2 5)
  matrix:set biomass_matrix 3 6 (matrix:get biomass_matrix 1 6 - matrix:get biomass_matrix 2 6) matrix:set biomass_matrix 4 6 (matrix:get biomass_matrix 1 6 + matrix:get biomass_matrix 2 6)
  matrix:set biomass_matrix 3 7 (matrix:get biomass_matrix 1 7 - matrix:get biomass_matrix 2 7) matrix:set biomass_matrix 4 7 (matrix:get biomass_matrix 1 7 + matrix:get biomass_matrix 2 7)
  matrix:set biomass_matrix 3 8 (matrix:get biomass_matrix 1 8 - matrix:get biomass_matrix 2 8) matrix:set biomass_matrix 4 8 (matrix:get biomass_matrix 1 8 + matrix:get biomass_matrix 2 8)
  matrix:set biomass_matrix 3 9 (matrix:get biomass_matrix 1 9 - matrix:get biomass_matrix 2 9) matrix:set biomass_matrix 4 9 (matrix:get biomass_matrix 1 9 + matrix:get biomass_matrix 2 9)
  matrix:set biomass_matrix 3 10 (matrix:get biomass_matrix 1 10 - matrix:get biomass_matrix 2 10) matrix:set biomass_matrix 4 10 (matrix:get biomass_matrix 1 10 + matrix:get biomass_matrix 2 10)
  matrix:set biomass_matrix 3 11 (matrix:get biomass_matrix 1 11 - matrix:get biomass_matrix 2 11) matrix:set biomass_matrix 4 11 (matrix:get biomass_matrix 1 11 + matrix:get biomass_matrix 2 11)
  matrix:set biomass_matrix 3 12 (matrix:get biomass_matrix 1 12 - matrix:get biomass_matrix 2 12) matrix:set biomass_matrix 4 12 (matrix:get biomass_matrix 1 12 + matrix:get biomass_matrix 2 12)
  matrix:set biomass_matrix 3 13 (matrix:get biomass_matrix 1 13 - matrix:get biomass_matrix 2 13) matrix:set biomass_matrix 4 13 (matrix:get biomass_matrix 1 13 + matrix:get biomass_matrix 2 13)
  matrix:set biomass_matrix 3 14 (matrix:get biomass_matrix 1 14 - matrix:get biomass_matrix 2 14) matrix:set biomass_matrix 4 14 (matrix:get biomass_matrix 1 14 + matrix:get biomass_matrix 2 14)
  matrix:set biomass_matrix 3 15 (matrix:get biomass_matrix 1 15 - matrix:get biomass_matrix 2 15) matrix:set biomass_matrix 4 15 (matrix:get biomass_matrix 1 15 + matrix:get biomass_matrix 2 15)
  matrix:set biomass_matrix 3 16 (matrix:get biomass_matrix 1 16 - matrix:get biomass_matrix 2 16) matrix:set biomass_matrix 4 16 (matrix:get biomass_matrix 1 16 + matrix:get biomass_matrix 2 16)


  matrix:set biomass_matrix 5 0 ((matrix:get biomass_matrix 3 0 / matrix:get biomass_matrix 4 0) ^ 2)
  matrix:set biomass_matrix 5 1 ((matrix:get biomass_matrix 3 1 / matrix:get biomass_matrix 4 1) ^ 2)
  matrix:set biomass_matrix 5 2 ((matrix:get biomass_matrix 3 2 / matrix:get biomass_matrix 4 2) ^ 2)
  matrix:set biomass_matrix 5 3 ((matrix:get biomass_matrix 3 3 / matrix:get biomass_matrix 4 3) ^ 2)
  matrix:set biomass_matrix 5 4 ((matrix:get biomass_matrix 3 4 / matrix:get biomass_matrix 4 4) ^ 2)
  matrix:set biomass_matrix 5 5 ((matrix:get biomass_matrix 3 5 / matrix:get biomass_matrix 4 5) ^ 2)
  matrix:set biomass_matrix 5 6 ((matrix:get biomass_matrix 3 6 / matrix:get biomass_matrix 4 6) ^ 2)
  matrix:set biomass_matrix 5 7 ((matrix:get biomass_matrix 3 7 / matrix:get biomass_matrix 4 7) ^ 2)
  matrix:set biomass_matrix 5 8 ((matrix:get biomass_matrix 3 8 / matrix:get biomass_matrix 4 8) ^ 2)
  matrix:set biomass_matrix 5 9 ((matrix:get biomass_matrix 3 9 / matrix:get biomass_matrix 4 9) ^ 2)
  matrix:set biomass_matrix 5 10 ((matrix:get biomass_matrix 3 10 / matrix:get biomass_matrix 4 10) ^ 2)
  matrix:set biomass_matrix 5 11 ((matrix:get biomass_matrix 3 11 / matrix:get biomass_matrix 4 11) ^ 2)
  matrix:set biomass_matrix 5 12 ((matrix:get biomass_matrix 3 12 / matrix:get biomass_matrix 4 12) ^ 2)
  matrix:set biomass_matrix 5 13 ((matrix:get biomass_matrix 3 13 / matrix:get biomass_matrix 4 13) ^ 2)
  matrix:set biomass_matrix 5 14 ((matrix:get biomass_matrix 3 14 / matrix:get biomass_matrix 4 14) ^ 2)
  matrix:set biomass_matrix 5 15 ((matrix:get biomass_matrix 3 15 / matrix:get biomass_matrix 4 15) ^ 2)
  matrix:set biomass_matrix 5 16 ((matrix:get biomass_matrix 3 16 / matrix:get biomass_matrix 4 16) ^ 2)

  let sum_biomass_square sum matrix:get-row biomass_matrix 5
  let biomass_square sqrt ( (1 / (length matrix:get-row biomass_matrix 5)) * sum_biomass_square)
  set GRI_biomass (1 + biomass_square) / (1 - biomass_square)

  matrix:set nitrate_matrix 3 0 (matrix:get nitrate_matrix 1 0 - matrix:get nitrate_matrix 2 0) matrix:set nitrate_matrix 4 0 (matrix:get nitrate_matrix 1 0 + matrix:get nitrate_matrix 2 0)
  matrix:set nitrate_matrix 3 1 (matrix:get nitrate_matrix 1 1 - matrix:get nitrate_matrix 2 1) matrix:set nitrate_matrix 4 1 (matrix:get nitrate_matrix 1 1 + matrix:get nitrate_matrix 2 1)
  matrix:set nitrate_matrix 3 2 (matrix:get nitrate_matrix 1 2 - matrix:get nitrate_matrix 2 2) matrix:set nitrate_matrix 4 2 (matrix:get nitrate_matrix 1 2 + matrix:get nitrate_matrix 2 2)
  matrix:set nitrate_matrix 3 3 (matrix:get nitrate_matrix 1 3 - matrix:get nitrate_matrix 2 3) matrix:set nitrate_matrix 4 3 (matrix:get nitrate_matrix 1 3 + matrix:get nitrate_matrix 2 3)
  matrix:set nitrate_matrix 3 4 (matrix:get nitrate_matrix 1 4 - matrix:get nitrate_matrix 2 4) matrix:set nitrate_matrix 4 4 (matrix:get nitrate_matrix 1 4 + matrix:get nitrate_matrix 2 4)
  matrix:set nitrate_matrix 3 5 (matrix:get nitrate_matrix 1 5 - matrix:get nitrate_matrix 2 5) matrix:set nitrate_matrix 4 5 (matrix:get nitrate_matrix 1 5 + matrix:get nitrate_matrix 2 5)
  matrix:set nitrate_matrix 3 6 (matrix:get nitrate_matrix 1 6 - matrix:get nitrate_matrix 2 6) matrix:set nitrate_matrix 4 6 (matrix:get nitrate_matrix 1 6 + matrix:get nitrate_matrix 2 6)
  matrix:set nitrate_matrix 3 7 (matrix:get nitrate_matrix 1 7 - matrix:get nitrate_matrix 2 7) matrix:set nitrate_matrix 4 7 (matrix:get nitrate_matrix 1 7 + matrix:get nitrate_matrix 2 7)
  matrix:set nitrate_matrix 3 8 (matrix:get nitrate_matrix 1 8 - matrix:get nitrate_matrix 2 8) matrix:set nitrate_matrix 4 8 (matrix:get nitrate_matrix 1 8 + matrix:get nitrate_matrix 2 8)
  matrix:set nitrate_matrix 3 9 (matrix:get nitrate_matrix 1 9 - matrix:get nitrate_matrix 2 9) matrix:set nitrate_matrix 4 9 (matrix:get nitrate_matrix 1 9 + matrix:get nitrate_matrix 2 9)
  matrix:set nitrate_matrix 3 10 (matrix:get nitrate_matrix 1 10 - matrix:get nitrate_matrix 2 10) matrix:set nitrate_matrix 4 10 (matrix:get nitrate_matrix 1 10 + matrix:get nitrate_matrix 2 10)
  matrix:set nitrate_matrix 3 11 (matrix:get nitrate_matrix 1 11 - matrix:get nitrate_matrix 2 11) matrix:set nitrate_matrix 4 11 (matrix:get nitrate_matrix 1 11 + matrix:get nitrate_matrix 2 11)
  matrix:set nitrate_matrix 3 12 (matrix:get nitrate_matrix 1 12 - matrix:get nitrate_matrix 2 12) matrix:set nitrate_matrix 4 12 (matrix:get nitrate_matrix 1 12 + matrix:get nitrate_matrix 2 12)
  matrix:set nitrate_matrix 3 13 (matrix:get nitrate_matrix 1 13 - matrix:get nitrate_matrix 2 13) matrix:set nitrate_matrix 4 13 (matrix:get nitrate_matrix 1 13 + matrix:get nitrate_matrix 2 13)
  matrix:set nitrate_matrix 3 14 (matrix:get nitrate_matrix 1 14 - matrix:get nitrate_matrix 2 14) matrix:set nitrate_matrix 4 14 (matrix:get nitrate_matrix 1 14 + matrix:get nitrate_matrix 2 14)
  matrix:set nitrate_matrix 3 15 (matrix:get nitrate_matrix 1 15 - matrix:get nitrate_matrix 2 15) matrix:set nitrate_matrix 4 15 (matrix:get nitrate_matrix 1 15 + matrix:get nitrate_matrix 2 15)
  matrix:set nitrate_matrix 3 16 (matrix:get nitrate_matrix 1 16 - matrix:get nitrate_matrix 2 16) matrix:set nitrate_matrix 4 16 (matrix:get nitrate_matrix 1 16 + matrix:get nitrate_matrix 2 16)

  matrix:set nitrate_matrix 5 0 ((matrix:get nitrate_matrix 3 0 / matrix:get nitrate_matrix 4 0) ^ 2)
  matrix:set nitrate_matrix 5 1 ((matrix:get nitrate_matrix 3 1 / matrix:get nitrate_matrix 4 1) ^ 2)
  matrix:set nitrate_matrix 5 2 ((matrix:get nitrate_matrix 3 2 / matrix:get nitrate_matrix 4 2) ^ 2)
  matrix:set nitrate_matrix 5 3 ((matrix:get nitrate_matrix 3 3 / matrix:get nitrate_matrix 4 3) ^ 2)
  matrix:set nitrate_matrix 5 4 ((matrix:get nitrate_matrix 3 4 / matrix:get nitrate_matrix 4 4) ^ 2)
  matrix:set nitrate_matrix 5 5 ((matrix:get nitrate_matrix 3 5 / matrix:get nitrate_matrix 4 5) ^ 2)
  matrix:set nitrate_matrix 5 6 ((matrix:get nitrate_matrix 3 6 / matrix:get nitrate_matrix 4 6) ^ 2)
  matrix:set nitrate_matrix 5 7 ((matrix:get nitrate_matrix 3 7 / matrix:get nitrate_matrix 4 7) ^ 2)
  matrix:set nitrate_matrix 5 8 ((matrix:get nitrate_matrix 3 8 / matrix:get nitrate_matrix 4 8) ^ 2)
  matrix:set nitrate_matrix 5 9 ((matrix:get nitrate_matrix 3 9 / matrix:get nitrate_matrix 4 9) ^ 2)
  matrix:set nitrate_matrix 5 10 ((matrix:get nitrate_matrix 3 10 / matrix:get nitrate_matrix 4 10) ^ 2)
  matrix:set nitrate_matrix 5 11 ((matrix:get nitrate_matrix 3 11 / matrix:get nitrate_matrix 4 11) ^ 2)
  matrix:set nitrate_matrix 5 12 ((matrix:get nitrate_matrix 3 12 / matrix:get nitrate_matrix 4 12) ^ 2)
  matrix:set nitrate_matrix 5 13 ((matrix:get nitrate_matrix 3 13 / matrix:get nitrate_matrix 4 13) ^ 2)
  matrix:set nitrate_matrix 5 14 ((matrix:get nitrate_matrix 3 14 / matrix:get nitrate_matrix 4 14) ^ 2)
  matrix:set nitrate_matrix 5 15 ((matrix:get nitrate_matrix 3 15 / matrix:get nitrate_matrix 4 15) ^ 2)
  matrix:set nitrate_matrix 5 16 ((matrix:get nitrate_matrix 3 16 / matrix:get nitrate_matrix 4 16) ^ 2)

  let sum_nitrate_square sum matrix:get-row nitrate_matrix 5
  let nitrate_square sqrt ( (1 / (length matrix:get-row nitrate_matrix 5)) * sum_nitrate_square)
  set GRI_nitrate (1 + nitrate_square) / (1 - nitrate_square)

  matrix:set nitrite_matrix 3 0 (matrix:get nitrite_matrix 1 0 - matrix:get nitrite_matrix 2 0) matrix:set nitrite_matrix 4 0 (matrix:get nitrite_matrix 1 0 + matrix:get nitrite_matrix 2 0)
  matrix:set nitrite_matrix 3 1 (matrix:get nitrite_matrix 1 1 - matrix:get nitrite_matrix 2 1) matrix:set nitrite_matrix 4 1 (matrix:get nitrite_matrix 1 1 + matrix:get nitrite_matrix 2 1)
  matrix:set nitrite_matrix 3 2 (matrix:get nitrite_matrix 1 2 - matrix:get nitrite_matrix 2 2) matrix:set nitrite_matrix 4 2 (matrix:get nitrite_matrix 1 2 + matrix:get nitrite_matrix 2 2)
  matrix:set nitrite_matrix 3 3 (matrix:get nitrite_matrix 1 3 - matrix:get nitrite_matrix 2 3) matrix:set nitrite_matrix 4 3 (matrix:get nitrite_matrix 1 3 + matrix:get nitrite_matrix 2 3)
  matrix:set nitrite_matrix 3 4 (matrix:get nitrite_matrix 1 4 - matrix:get nitrite_matrix 2 4) matrix:set nitrite_matrix 4 4 (matrix:get nitrite_matrix 1 4 + matrix:get nitrite_matrix 2 4)
  matrix:set nitrite_matrix 3 5 (matrix:get nitrite_matrix 1 5 - matrix:get nitrite_matrix 2 5) matrix:set nitrite_matrix 4 5 (matrix:get nitrite_matrix 1 5 + matrix:get nitrite_matrix 2 5)
  matrix:set nitrite_matrix 3 6 (matrix:get nitrite_matrix 1 6 - matrix:get nitrite_matrix 2 6) matrix:set nitrite_matrix 4 6 (matrix:get nitrite_matrix 1 6 + matrix:get nitrite_matrix 2 6)
  matrix:set nitrite_matrix 3 7 (matrix:get nitrite_matrix 1 7 - matrix:get nitrite_matrix 2 7) matrix:set nitrite_matrix 4 7 (matrix:get nitrite_matrix 1 7 + matrix:get nitrite_matrix 2 7)
  matrix:set nitrite_matrix 3 8 (matrix:get nitrite_matrix 1 8 - matrix:get nitrite_matrix 2 8) matrix:set nitrite_matrix 4 8 (matrix:get nitrite_matrix 1 8 + matrix:get nitrite_matrix 2 8)
  matrix:set nitrite_matrix 3 9 (matrix:get nitrite_matrix 1 9 - matrix:get nitrite_matrix 2 9) matrix:set nitrite_matrix 4 9 (matrix:get nitrite_matrix 1 9 + matrix:get nitrite_matrix 2 9)
  matrix:set nitrite_matrix 3 10 (matrix:get nitrite_matrix 1 10 - matrix:get nitrite_matrix 2 10) matrix:set nitrite_matrix 4 10 (matrix:get nitrite_matrix 1 10 + matrix:get nitrite_matrix 2 10)
  matrix:set nitrite_matrix 3 11 (matrix:get nitrite_matrix 1 11 - matrix:get nitrite_matrix 2 11) matrix:set nitrite_matrix 4 11 (matrix:get nitrite_matrix 1 11 + matrix:get nitrite_matrix 2 11)
  matrix:set nitrite_matrix 3 12 (matrix:get nitrite_matrix 1 12 - matrix:get nitrite_matrix 2 12) matrix:set nitrite_matrix 4 12 (matrix:get nitrite_matrix 1 12 + matrix:get nitrite_matrix 2 12)
  matrix:set nitrite_matrix 3 13 (matrix:get nitrite_matrix 1 13 - matrix:get nitrite_matrix 2 13) matrix:set nitrite_matrix 4 13 (matrix:get nitrite_matrix 1 13 + matrix:get nitrite_matrix 2 13)
  matrix:set nitrite_matrix 3 14 (matrix:get nitrite_matrix 1 14 - matrix:get nitrite_matrix 2 14) matrix:set nitrite_matrix 4 14 (matrix:get nitrite_matrix 1 14 + matrix:get nitrite_matrix 2 14)
  matrix:set nitrite_matrix 3 15 (matrix:get nitrite_matrix 1 15 - matrix:get nitrite_matrix 2 15) matrix:set nitrite_matrix 4 15 (matrix:get nitrite_matrix 1 15 + matrix:get nitrite_matrix 2 15)
  matrix:set nitrite_matrix 3 16 (matrix:get nitrite_matrix 1 16 - matrix:get nitrite_matrix 2 16) matrix:set nitrite_matrix 4 16 (matrix:get nitrite_matrix 1 16 + matrix:get nitrite_matrix 2 16)


  matrix:set nitrite_matrix 5 0 ((matrix:get nitrite_matrix 3 0 / matrix:get nitrite_matrix 4 0) ^ 2)
  matrix:set nitrite_matrix 5 1 ((matrix:get nitrite_matrix 3 1 / matrix:get nitrite_matrix 4 1) ^ 2)
  matrix:set nitrite_matrix 5 2 ((matrix:get nitrite_matrix 3 2 / matrix:get nitrite_matrix 4 2) ^ 2)
  matrix:set nitrite_matrix 5 3 ((matrix:get nitrite_matrix 3 3 / matrix:get nitrite_matrix 4 3) ^ 2)
  matrix:set nitrite_matrix 5 4 ((matrix:get nitrite_matrix 3 4 / matrix:get nitrite_matrix 4 4) ^ 2)
  matrix:set nitrite_matrix 5 5 ((matrix:get nitrite_matrix 3 5 / matrix:get nitrite_matrix 4 5) ^ 2)
  matrix:set nitrite_matrix 5 6 ((matrix:get nitrite_matrix 3 6 / matrix:get nitrite_matrix 4 6) ^ 2)
  matrix:set nitrite_matrix 5 7 ((matrix:get nitrite_matrix 3 7 / matrix:get nitrite_matrix 4 7) ^ 2)
  matrix:set nitrite_matrix 5 8 ((matrix:get nitrite_matrix 3 8 / matrix:get nitrite_matrix 4 8) ^ 2)
  matrix:set nitrite_matrix 5 9 ((matrix:get nitrite_matrix 3 9 / matrix:get nitrite_matrix 4 9) ^ 2)
  matrix:set nitrite_matrix 5 10 ((matrix:get nitrite_matrix 3 10 / matrix:get nitrite_matrix 4 10) ^ 2)
  matrix:set nitrite_matrix 5 11 ((matrix:get nitrite_matrix 3 11 / matrix:get nitrite_matrix 4 11) ^ 2)
  matrix:set nitrite_matrix 5 12 ((matrix:get nitrite_matrix 3 12 / matrix:get nitrite_matrix 4 12) ^ 2)
  matrix:set nitrite_matrix 5 13 ((matrix:get nitrite_matrix 3 13 / matrix:get nitrite_matrix 4 13) ^ 2)
  matrix:set nitrite_matrix 5 14 ((matrix:get nitrite_matrix 3 14 / matrix:get nitrite_matrix 4 14) ^ 2)
  matrix:set nitrite_matrix 5 15 ((matrix:get nitrite_matrix 3 15 / matrix:get nitrite_matrix 4 15) ^ 2)
  matrix:set nitrite_matrix 5 16 ((matrix:get nitrite_matrix 3 16 / matrix:get nitrite_matrix 4 16) ^ 2)

  let sum_nitrite_square sum matrix:get-row nitrite_matrix 5
  let nitrite_square sqrt ( (1 / (length matrix:get-row nitrite_matrix 5)) * sum_nitrite_square)
  set GRI_nitrite (1 + nitrite_square) / (1 - nitrite_square)

  matrix:set nitrous_oxide_matrix 3 0 (matrix:get nitrous_oxide_matrix 1 0 - matrix:get nitrous_oxide_matrix 2 0) matrix:set nitrous_oxide_matrix 4 0 (matrix:get nitrous_oxide_matrix 1 0 + matrix:get nitrous_oxide_matrix 2 0)
  matrix:set nitrous_oxide_matrix 3 1 (matrix:get nitrous_oxide_matrix 1 1 - matrix:get nitrous_oxide_matrix 2 1) matrix:set nitrous_oxide_matrix 4 1 (matrix:get nitrous_oxide_matrix 1 1 + matrix:get nitrous_oxide_matrix 2 1)
  matrix:set nitrous_oxide_matrix 3 2 (matrix:get nitrous_oxide_matrix 1 2 - matrix:get nitrous_oxide_matrix 2 2) matrix:set nitrous_oxide_matrix 4 2 (matrix:get nitrous_oxide_matrix 1 2 + matrix:get nitrous_oxide_matrix 2 2)
  matrix:set nitrous_oxide_matrix 3 3 (matrix:get nitrous_oxide_matrix 1 3 - matrix:get nitrous_oxide_matrix 2 3) matrix:set nitrous_oxide_matrix 4 3 (matrix:get nitrous_oxide_matrix 1 3 + matrix:get nitrous_oxide_matrix 2 3)
  matrix:set nitrous_oxide_matrix 3 4 (matrix:get nitrous_oxide_matrix 1 4 - matrix:get nitrous_oxide_matrix 2 4) matrix:set nitrous_oxide_matrix 4 4 (matrix:get nitrous_oxide_matrix 1 4 + matrix:get nitrous_oxide_matrix 2 4)
  matrix:set nitrous_oxide_matrix 3 5 (matrix:get nitrous_oxide_matrix 1 5 - matrix:get nitrous_oxide_matrix 2 5) matrix:set nitrous_oxide_matrix 4 5 (matrix:get nitrous_oxide_matrix 1 5 + matrix:get nitrous_oxide_matrix 2 5)
  matrix:set nitrous_oxide_matrix 3 6 (matrix:get nitrous_oxide_matrix 1 6 - matrix:get nitrous_oxide_matrix 2 6) matrix:set nitrous_oxide_matrix 4 6 (matrix:get nitrous_oxide_matrix 1 6 + matrix:get nitrous_oxide_matrix 2 6)
  matrix:set nitrous_oxide_matrix 3 7 (matrix:get nitrous_oxide_matrix 1 7 - matrix:get nitrous_oxide_matrix 2 7) matrix:set nitrous_oxide_matrix 4 7 (matrix:get nitrous_oxide_matrix 1 7 + matrix:get nitrous_oxide_matrix 2 7)
  matrix:set nitrous_oxide_matrix 3 8 (matrix:get nitrous_oxide_matrix 1 8 - matrix:get nitrous_oxide_matrix 2 8) matrix:set nitrous_oxide_matrix 4 8 (matrix:get nitrous_oxide_matrix 1 8 + matrix:get nitrous_oxide_matrix 2 8)
  matrix:set nitrous_oxide_matrix 3 9 (matrix:get nitrous_oxide_matrix 1 9 - matrix:get nitrous_oxide_matrix 2 9) matrix:set nitrous_oxide_matrix 4 9 (matrix:get nitrous_oxide_matrix 1 9 + matrix:get nitrous_oxide_matrix 2 9)
  matrix:set nitrous_oxide_matrix 3 10 (matrix:get nitrous_oxide_matrix 1 10 - matrix:get nitrous_oxide_matrix 2 10) matrix:set nitrous_oxide_matrix 4 10 (matrix:get nitrous_oxide_matrix 1 10 + matrix:get nitrous_oxide_matrix 2 10)
  matrix:set nitrous_oxide_matrix 3 11 (matrix:get nitrous_oxide_matrix 1 11 - matrix:get nitrous_oxide_matrix 2 11) matrix:set nitrous_oxide_matrix 4 11 (matrix:get nitrous_oxide_matrix 1 11 + matrix:get nitrous_oxide_matrix 2 11)
  matrix:set nitrous_oxide_matrix 3 12 (matrix:get nitrous_oxide_matrix 1 12 - matrix:get nitrous_oxide_matrix 2 12) matrix:set nitrous_oxide_matrix 4 12 (matrix:get nitrous_oxide_matrix 1 12 + matrix:get nitrous_oxide_matrix 2 12)
  matrix:set nitrous_oxide_matrix 3 13 (matrix:get nitrous_oxide_matrix 1 13 - matrix:get nitrous_oxide_matrix 2 13) matrix:set nitrous_oxide_matrix 4 13 (matrix:get nitrous_oxide_matrix 1 13 + matrix:get nitrous_oxide_matrix 2 13)
  matrix:set nitrous_oxide_matrix 3 14 (matrix:get nitrous_oxide_matrix 1 14 - matrix:get nitrous_oxide_matrix 2 14) matrix:set nitrous_oxide_matrix 4 14 (matrix:get nitrous_oxide_matrix 1 14 + matrix:get nitrous_oxide_matrix 2 14)

  matrix:set nitrous_oxide_matrix 5 0 ((matrix:get nitrous_oxide_matrix 3 0 / matrix:get nitrous_oxide_matrix 4 0) ^ 2)
  matrix:set nitrous_oxide_matrix 5 1 ((matrix:get nitrous_oxide_matrix 3 1 / matrix:get nitrous_oxide_matrix 4 1) ^ 2)
  matrix:set nitrous_oxide_matrix 5 2 ((matrix:get nitrous_oxide_matrix 3 2 / matrix:get nitrous_oxide_matrix 4 2) ^ 2)
  matrix:set nitrous_oxide_matrix 5 3 ((matrix:get nitrous_oxide_matrix 3 3 / matrix:get nitrous_oxide_matrix 4 3) ^ 2)
  matrix:set nitrous_oxide_matrix 5 4 ((matrix:get nitrous_oxide_matrix 3 4 / matrix:get nitrous_oxide_matrix 4 4) ^ 2)
  matrix:set nitrous_oxide_matrix 5 5 ((matrix:get nitrous_oxide_matrix 3 5 / matrix:get nitrous_oxide_matrix 4 5) ^ 2)
  matrix:set nitrous_oxide_matrix 5 6 ((matrix:get nitrous_oxide_matrix 3 6 / matrix:get nitrous_oxide_matrix 4 6) ^ 2)
  matrix:set nitrous_oxide_matrix 5 7 ((matrix:get nitrous_oxide_matrix 3 7 / matrix:get nitrous_oxide_matrix 4 7) ^ 2)
  matrix:set nitrous_oxide_matrix 5 8 ((matrix:get nitrous_oxide_matrix 3 8 / matrix:get nitrous_oxide_matrix 4 8) ^ 2)
  matrix:set nitrous_oxide_matrix 5 9 ((matrix:get nitrous_oxide_matrix 3 9 / matrix:get nitrous_oxide_matrix 4 9) ^ 2)
  matrix:set nitrous_oxide_matrix 5 10 ((matrix:get nitrous_oxide_matrix 3 10 / matrix:get nitrous_oxide_matrix 4 10) ^ 2)
  matrix:set nitrous_oxide_matrix 5 11 ((matrix:get nitrous_oxide_matrix 3 11 / matrix:get nitrous_oxide_matrix 4 11) ^ 2)
  matrix:set nitrous_oxide_matrix 5 12 ((matrix:get nitrous_oxide_matrix 3 12 / matrix:get nitrous_oxide_matrix 4 12) ^ 2)
  matrix:set nitrous_oxide_matrix 5 13 ((matrix:get nitrous_oxide_matrix 3 13 / matrix:get nitrous_oxide_matrix 4 13) ^ 2)
  matrix:set nitrous_oxide_matrix 5 14 ((matrix:get nitrous_oxide_matrix 3 14 / matrix:get nitrous_oxide_matrix 4 14) ^ 2)

  let sum_nitrous_oxide_square sum matrix:get-row nitrous_oxide_matrix 5
  let nitrous_oxide_square sqrt ( (1 / (length matrix:get-row nitrous_oxide_matrix 5)) * sum_nitrous_oxide_square)
  set GRI_nitrous_oxide (1 + nitrous_oxide_square) / (1 - nitrous_oxide_square)

  Fitness-Calculation

  ] ; end procedure succinate low P. denitrificans

  if Experiment = "Succ_High/Nitrate_Low"[

  matrix:set biomass_matrix 3 0 (matrix:get biomass_matrix 1 0 - matrix:get biomass_matrix 2 0) matrix:set biomass_matrix 4 0 (matrix:get biomass_matrix 1 0 + matrix:get biomass_matrix 2 0)
  matrix:set biomass_matrix 3 1 (matrix:get biomass_matrix 1 1 - matrix:get biomass_matrix 2 1) matrix:set biomass_matrix 4 1 (matrix:get biomass_matrix 1 1 + matrix:get biomass_matrix 2 1)
  matrix:set biomass_matrix 3 2 (matrix:get biomass_matrix 1 2 - matrix:get biomass_matrix 2 2) matrix:set biomass_matrix 4 2 (matrix:get biomass_matrix 1 2 + matrix:get biomass_matrix 2 2)
  matrix:set biomass_matrix 3 3 (matrix:get biomass_matrix 1 3 - matrix:get biomass_matrix 2 3) matrix:set biomass_matrix 4 3 (matrix:get biomass_matrix 1 3 + matrix:get biomass_matrix 2 3)
  matrix:set biomass_matrix 3 4 (matrix:get biomass_matrix 1 4 - matrix:get biomass_matrix 2 4) matrix:set biomass_matrix 4 4 (matrix:get biomass_matrix 1 4 + matrix:get biomass_matrix 2 4)
  matrix:set biomass_matrix 3 5 (matrix:get biomass_matrix 1 5 - matrix:get biomass_matrix 2 5) matrix:set biomass_matrix 4 5 (matrix:get biomass_matrix 1 5 + matrix:get biomass_matrix 2 5)
  matrix:set biomass_matrix 3 6 (matrix:get biomass_matrix 1 6 - matrix:get biomass_matrix 2 6) matrix:set biomass_matrix 4 6 (matrix:get biomass_matrix 1 6 + matrix:get biomass_matrix 2 6)
  matrix:set biomass_matrix 3 7 (matrix:get biomass_matrix 1 7 - matrix:get biomass_matrix 2 7) matrix:set biomass_matrix 4 7 (matrix:get biomass_matrix 1 7 + matrix:get biomass_matrix 2 7)
  matrix:set biomass_matrix 3 8 (matrix:get biomass_matrix 1 8 - matrix:get biomass_matrix 2 8) matrix:set biomass_matrix 4 8 (matrix:get biomass_matrix 1 8 + matrix:get biomass_matrix 2 8)
  matrix:set biomass_matrix 3 9 (matrix:get biomass_matrix 1 9 - matrix:get biomass_matrix 2 9) matrix:set biomass_matrix 4 9 (matrix:get biomass_matrix 1 9 + matrix:get biomass_matrix 2 9)
  matrix:set biomass_matrix 3 10 (matrix:get biomass_matrix 1 10 - matrix:get biomass_matrix 2 10) matrix:set biomass_matrix 4 10 (matrix:get biomass_matrix 1 10 + matrix:get biomass_matrix 2 10)
  matrix:set biomass_matrix 3 11 (matrix:get biomass_matrix 1 11 - matrix:get biomass_matrix 2 11) matrix:set biomass_matrix 4 11 (matrix:get biomass_matrix 1 11 + matrix:get biomass_matrix 2 11)
  matrix:set biomass_matrix 3 12 (matrix:get biomass_matrix 1 12 - matrix:get biomass_matrix 2 12) matrix:set biomass_matrix 4 12 (matrix:get biomass_matrix 1 12 + matrix:get biomass_matrix 2 12)
  matrix:set biomass_matrix 3 13 (matrix:get biomass_matrix 1 13 - matrix:get biomass_matrix 2 13) matrix:set biomass_matrix 4 13 (matrix:get biomass_matrix 1 13 + matrix:get biomass_matrix 2 13)
  matrix:set biomass_matrix 3 14 (matrix:get biomass_matrix 1 14 - matrix:get biomass_matrix 2 14) matrix:set biomass_matrix 4 14 (matrix:get biomass_matrix 1 14 + matrix:get biomass_matrix 2 14)
  matrix:set biomass_matrix 3 15 (matrix:get biomass_matrix 1 15 - matrix:get biomass_matrix 2 15) matrix:set biomass_matrix 4 15 (matrix:get biomass_matrix 1 15 + matrix:get biomass_matrix 2 15)
  matrix:set biomass_matrix 3 16 (matrix:get biomass_matrix 1 16 - matrix:get biomass_matrix 2 16) matrix:set biomass_matrix 4 16 (matrix:get biomass_matrix 1 16 + matrix:get biomass_matrix 2 16)

  matrix:set biomass_matrix 5 0 ((matrix:get biomass_matrix 3 0 / matrix:get biomass_matrix 4 0) ^ 2)
  matrix:set biomass_matrix 5 1 ((matrix:get biomass_matrix 3 1 / matrix:get biomass_matrix 4 1) ^ 2)
  matrix:set biomass_matrix 5 2 ((matrix:get biomass_matrix 3 2 / matrix:get biomass_matrix 4 2) ^ 2)
  matrix:set biomass_matrix 5 3 ((matrix:get biomass_matrix 3 3 / matrix:get biomass_matrix 4 3) ^ 2)
  matrix:set biomass_matrix 5 4 ((matrix:get biomass_matrix 3 4 / matrix:get biomass_matrix 4 4) ^ 2)
  matrix:set biomass_matrix 5 5 ((matrix:get biomass_matrix 3 5 / matrix:get biomass_matrix 4 5) ^ 2)
  matrix:set biomass_matrix 5 6 ((matrix:get biomass_matrix 3 6 / matrix:get biomass_matrix 4 6) ^ 2)
  matrix:set biomass_matrix 5 7 ((matrix:get biomass_matrix 3 7 / matrix:get biomass_matrix 4 7) ^ 2)
  matrix:set biomass_matrix 5 8 ((matrix:get biomass_matrix 3 8 / matrix:get biomass_matrix 4 8) ^ 2)
  matrix:set biomass_matrix 5 9 ((matrix:get biomass_matrix 3 9 / matrix:get biomass_matrix 4 9) ^ 2)
  matrix:set biomass_matrix 5 10 ((matrix:get biomass_matrix 3 10 / matrix:get biomass_matrix 4 10) ^ 2)
  matrix:set biomass_matrix 5 11 ((matrix:get biomass_matrix 3 11 / matrix:get biomass_matrix 4 11) ^ 2)
  matrix:set biomass_matrix 5 12 ((matrix:get biomass_matrix 3 12 / matrix:get biomass_matrix 4 12) ^ 2)
  matrix:set biomass_matrix 5 13 ((matrix:get biomass_matrix 3 13 / matrix:get biomass_matrix 4 13) ^ 2)
  matrix:set biomass_matrix 5 14 ((matrix:get biomass_matrix 3 14 / matrix:get biomass_matrix 4 14) ^ 2)
  matrix:set biomass_matrix 5 15 ((matrix:get biomass_matrix 3 15 / matrix:get biomass_matrix 4 15) ^ 2)
  matrix:set biomass_matrix 5 16 ((matrix:get biomass_matrix 3 16 / matrix:get biomass_matrix 4 16) ^ 2)

  let sum_biomass_square sum matrix:get-row biomass_matrix 5
  let biomass_square sqrt ( (1 / (length matrix:get-row biomass_matrix 5)) * sum_biomass_square)
  set GRI_biomass (1 + biomass_square) / (1 - biomass_square)

  matrix:set nitrate_matrix 3 0 (matrix:get nitrate_matrix 1 0 - matrix:get nitrate_matrix 2 0) matrix:set nitrate_matrix 4 0 (matrix:get nitrate_matrix 1 0 + matrix:get nitrate_matrix 2 0)
  matrix:set nitrate_matrix 3 1 (matrix:get nitrate_matrix 1 1 - matrix:get nitrate_matrix 2 1) matrix:set nitrate_matrix 4 1 (matrix:get nitrate_matrix 1 1 + matrix:get nitrate_matrix 2 1)
  matrix:set nitrate_matrix 3 2 (matrix:get nitrate_matrix 1 2 - matrix:get nitrate_matrix 2 2) matrix:set nitrate_matrix 4 2 (matrix:get nitrate_matrix 1 2 + matrix:get nitrate_matrix 2 2)
  matrix:set nitrate_matrix 3 3 (matrix:get nitrate_matrix 1 3 - matrix:get nitrate_matrix 2 3) matrix:set nitrate_matrix 4 3 (matrix:get nitrate_matrix 1 3 + matrix:get nitrate_matrix 2 3)
  matrix:set nitrate_matrix 3 4 (matrix:get nitrate_matrix 1 4 - matrix:get nitrate_matrix 2 4) matrix:set nitrate_matrix 4 4 (matrix:get nitrate_matrix 1 4 + matrix:get nitrate_matrix 2 4)
  matrix:set nitrate_matrix 3 5 (matrix:get nitrate_matrix 1 5 - matrix:get nitrate_matrix 2 5) matrix:set nitrate_matrix 4 5 (matrix:get nitrate_matrix 1 5 + matrix:get nitrate_matrix 2 5)
  matrix:set nitrate_matrix 3 6 (matrix:get nitrate_matrix 1 6 - matrix:get nitrate_matrix 2 6) matrix:set nitrate_matrix 4 6 (matrix:get nitrate_matrix 1 6 + matrix:get nitrate_matrix 2 6)
  matrix:set nitrate_matrix 3 7 (matrix:get nitrate_matrix 1 7 - matrix:get nitrate_matrix 2 7) matrix:set nitrate_matrix 4 7 (matrix:get nitrate_matrix 1 7 + matrix:get nitrate_matrix 2 7)
  matrix:set nitrate_matrix 3 8 (matrix:get nitrate_matrix 1 8 - matrix:get nitrate_matrix 2 8) matrix:set nitrate_matrix 4 8 (matrix:get nitrate_matrix 1 8 + matrix:get nitrate_matrix 2 8)
  matrix:set nitrate_matrix 3 9 (matrix:get nitrate_matrix 1 9 - matrix:get nitrate_matrix 2 9) matrix:set nitrate_matrix 4 9 (matrix:get nitrate_matrix 1 9 + matrix:get nitrate_matrix 2 9)
  matrix:set nitrate_matrix 3 10 (matrix:get nitrate_matrix 1 10 - matrix:get nitrate_matrix 2 10) matrix:set nitrate_matrix 4 10 (matrix:get nitrate_matrix 1 10 + matrix:get nitrate_matrix 2 10)
  matrix:set nitrate_matrix 3 11 (matrix:get nitrate_matrix 1 11 - matrix:get nitrate_matrix 2 11) matrix:set nitrate_matrix 4 11 (matrix:get nitrate_matrix 1 11 + matrix:get nitrate_matrix 2 11)
  matrix:set nitrate_matrix 3 12 (matrix:get nitrate_matrix 1 12 - matrix:get nitrate_matrix 2 12) matrix:set nitrate_matrix 4 12 (matrix:get nitrate_matrix 1 12 + matrix:get nitrate_matrix 2 12)
  matrix:set nitrate_matrix 3 13 (matrix:get nitrate_matrix 1 13 - matrix:get nitrate_matrix 2 13) matrix:set nitrate_matrix 4 13 (matrix:get nitrate_matrix 1 13 + matrix:get nitrate_matrix 2 13)
  matrix:set nitrate_matrix 3 14 (matrix:get nitrate_matrix 1 14 - matrix:get nitrate_matrix 2 14) matrix:set nitrate_matrix 4 14 (matrix:get nitrate_matrix 1 14 + matrix:get nitrate_matrix 2 14)
  matrix:set nitrate_matrix 3 15 (matrix:get nitrate_matrix 1 15 - matrix:get nitrate_matrix 2 15) matrix:set nitrate_matrix 4 15 (matrix:get nitrate_matrix 1 15 + matrix:get nitrate_matrix 2 15)
  matrix:set nitrate_matrix 3 16 (matrix:get nitrate_matrix 1 16 - matrix:get nitrate_matrix 2 16) matrix:set nitrate_matrix 4 16 (matrix:get nitrate_matrix 1 16 + matrix:get nitrate_matrix 2 16)

  matrix:set nitrate_matrix 5 0 ((matrix:get nitrate_matrix 3 0 / matrix:get nitrate_matrix 4 0) ^ 2)
  matrix:set nitrate_matrix 5 1 ((matrix:get nitrate_matrix 3 1 / matrix:get nitrate_matrix 4 1) ^ 2)
  matrix:set nitrate_matrix 5 2 ((matrix:get nitrate_matrix 3 2 / matrix:get nitrate_matrix 4 2) ^ 2)
  matrix:set nitrate_matrix 5 3 ((matrix:get nitrate_matrix 3 3 / matrix:get nitrate_matrix 4 3) ^ 2)
  matrix:set nitrate_matrix 5 4 ((matrix:get nitrate_matrix 3 4 / matrix:get nitrate_matrix 4 4) ^ 2)
  matrix:set nitrate_matrix 5 5 ((matrix:get nitrate_matrix 3 5 / matrix:get nitrate_matrix 4 5) ^ 2)
  matrix:set nitrate_matrix 5 6 ((matrix:get nitrate_matrix 3 6 / matrix:get nitrate_matrix 4 6) ^ 2)
  matrix:set nitrate_matrix 5 7 ((matrix:get nitrate_matrix 3 7 / matrix:get nitrate_matrix 4 7) ^ 2)
  matrix:set nitrate_matrix 5 8 ((matrix:get nitrate_matrix 3 8 / matrix:get nitrate_matrix 4 8) ^ 2)
  matrix:set nitrate_matrix 5 9 ((matrix:get nitrate_matrix 3 9 / matrix:get nitrate_matrix 4 9) ^ 2)
  matrix:set nitrate_matrix 5 10 ((matrix:get nitrate_matrix 3 10 / matrix:get nitrate_matrix 4 10) ^ 2)
  matrix:set nitrate_matrix 5 11 ((matrix:get nitrate_matrix 3 11 / matrix:get nitrate_matrix 4 11) ^ 2)
  matrix:set nitrate_matrix 5 12 ((matrix:get nitrate_matrix 3 12 / matrix:get nitrate_matrix 4 12) ^ 2)
  matrix:set nitrate_matrix 5 13 ((matrix:get nitrate_matrix 3 13 / matrix:get nitrate_matrix 4 13) ^ 2)
  matrix:set nitrate_matrix 5 14 ((matrix:get nitrate_matrix 3 14 / matrix:get nitrate_matrix 4 14) ^ 2)
  matrix:set nitrate_matrix 5 15 ((matrix:get nitrate_matrix 3 15 / matrix:get nitrate_matrix 4 15) ^ 2)
  matrix:set nitrate_matrix 5 16 ((matrix:get nitrate_matrix 3 16 / matrix:get nitrate_matrix 4 16) ^ 2)

  let sum_nitrate_square sum matrix:get-row nitrate_matrix 5
  let nitrate_square sqrt ( (1 / (length matrix:get-row nitrate_matrix 5)) * sum_nitrate_square)
  set GRI_nitrate (1 + nitrate_square) / (1 - nitrate_square)

  matrix:set nitrite_matrix 3 0 (matrix:get nitrite_matrix 1 0 - matrix:get nitrite_matrix 2 0) matrix:set nitrite_matrix 4 0 (matrix:get nitrite_matrix 1 0 + matrix:get nitrite_matrix 2 0)
  matrix:set nitrite_matrix 3 1 (matrix:get nitrite_matrix 1 1 - matrix:get nitrite_matrix 2 1) matrix:set nitrite_matrix 4 1 (matrix:get nitrite_matrix 1 1 + matrix:get nitrite_matrix 2 1)
  matrix:set nitrite_matrix 3 2 (matrix:get nitrite_matrix 1 2 - matrix:get nitrite_matrix 2 2) matrix:set nitrite_matrix 4 2 (matrix:get nitrite_matrix 1 2 + matrix:get nitrite_matrix 2 2)
  matrix:set nitrite_matrix 3 3 (matrix:get nitrite_matrix 1 3 - matrix:get nitrite_matrix 2 3) matrix:set nitrite_matrix 4 3 (matrix:get nitrite_matrix 1 3 + matrix:get nitrite_matrix 2 3)
  matrix:set nitrite_matrix 3 4 (matrix:get nitrite_matrix 1 4 - matrix:get nitrite_matrix 2 4) matrix:set nitrite_matrix 4 4 (matrix:get nitrite_matrix 1 4 + matrix:get nitrite_matrix 2 4)
  matrix:set nitrite_matrix 3 5 (matrix:get nitrite_matrix 1 5 - matrix:get nitrite_matrix 2 5) matrix:set nitrite_matrix 4 5 (matrix:get nitrite_matrix 1 5 + matrix:get nitrite_matrix 2 5)
  matrix:set nitrite_matrix 3 6 (matrix:get nitrite_matrix 1 6 - matrix:get nitrite_matrix 2 6) matrix:set nitrite_matrix 4 6 (matrix:get nitrite_matrix 1 6 + matrix:get nitrite_matrix 2 6)
  matrix:set nitrite_matrix 3 7 (matrix:get nitrite_matrix 1 7 - matrix:get nitrite_matrix 2 7) matrix:set nitrite_matrix 4 7 (matrix:get nitrite_matrix 1 7 + matrix:get nitrite_matrix 2 7)
  matrix:set nitrite_matrix 3 8 (matrix:get nitrite_matrix 1 8 - matrix:get nitrite_matrix 2 8) matrix:set nitrite_matrix 4 8 (matrix:get nitrite_matrix 1 8 + matrix:get nitrite_matrix 2 8)
  matrix:set nitrite_matrix 3 9 (matrix:get nitrite_matrix 1 9 - matrix:get nitrite_matrix 2 9) matrix:set nitrite_matrix 4 9 (matrix:get nitrite_matrix 1 9 + matrix:get nitrite_matrix 2 9)
  matrix:set nitrite_matrix 3 10 (matrix:get nitrite_matrix 1 10 - matrix:get nitrite_matrix 2 10) matrix:set nitrite_matrix 4 10 (matrix:get nitrite_matrix 1 10 + matrix:get nitrite_matrix 2 10)
  matrix:set nitrite_matrix 3 11 (matrix:get nitrite_matrix 1 11 - matrix:get nitrite_matrix 2 11) matrix:set nitrite_matrix 4 11 (matrix:get nitrite_matrix 1 11 + matrix:get nitrite_matrix 2 11)
  matrix:set nitrite_matrix 3 12 (matrix:get nitrite_matrix 1 12 - matrix:get nitrite_matrix 2 12) matrix:set nitrite_matrix 4 12 (matrix:get nitrite_matrix 1 12 + matrix:get nitrite_matrix 2 12)
  matrix:set nitrite_matrix 3 13 (matrix:get nitrite_matrix 1 13 - matrix:get nitrite_matrix 2 13) matrix:set nitrite_matrix 4 13 (matrix:get nitrite_matrix 1 13 + matrix:get nitrite_matrix 2 13)
  matrix:set nitrite_matrix 3 14 (matrix:get nitrite_matrix 1 14 - matrix:get nitrite_matrix 2 14) matrix:set nitrite_matrix 4 14 (matrix:get nitrite_matrix 1 14 + matrix:get nitrite_matrix 2 14)
  matrix:set nitrite_matrix 3 15 (matrix:get nitrite_matrix 1 15 - matrix:get nitrite_matrix 2 15) matrix:set nitrite_matrix 4 15 (matrix:get nitrite_matrix 1 15 + matrix:get nitrite_matrix 2 15)
  matrix:set nitrite_matrix 3 16 (matrix:get nitrite_matrix 1 16 - matrix:get nitrite_matrix 2 16) matrix:set nitrite_matrix 4 16 (matrix:get nitrite_matrix 1 16 + matrix:get nitrite_matrix 2 16)

  matrix:set nitrite_matrix 5 0 ((matrix:get nitrite_matrix 3 0 / matrix:get nitrite_matrix 4 0) ^ 2)
  matrix:set nitrite_matrix 5 1 ((matrix:get nitrite_matrix 3 1 / matrix:get nitrite_matrix 4 1) ^ 2)
  matrix:set nitrite_matrix 5 2 ((matrix:get nitrite_matrix 3 2 / matrix:get nitrite_matrix 4 2) ^ 2)
  matrix:set nitrite_matrix 5 3 ((matrix:get nitrite_matrix 3 3 / matrix:get nitrite_matrix 4 3) ^ 2)
  matrix:set nitrite_matrix 5 4 ((matrix:get nitrite_matrix 3 4 / matrix:get nitrite_matrix 4 4) ^ 2)
  matrix:set nitrite_matrix 5 5 ((matrix:get nitrite_matrix 3 5 / matrix:get nitrite_matrix 4 5) ^ 2)
  matrix:set nitrite_matrix 5 6 ((matrix:get nitrite_matrix 3 6 / matrix:get nitrite_matrix 4 6) ^ 2)
  matrix:set nitrite_matrix 5 7 ((matrix:get nitrite_matrix 3 7 / matrix:get nitrite_matrix 4 7) ^ 2)
  matrix:set nitrite_matrix 5 8 ((matrix:get nitrite_matrix 3 8 / matrix:get nitrite_matrix 4 8) ^ 2)
  matrix:set nitrite_matrix 5 9 ((matrix:get nitrite_matrix 3 9 / matrix:get nitrite_matrix 4 9) ^ 2)
  matrix:set nitrite_matrix 5 10 ((matrix:get nitrite_matrix 3 10 / matrix:get nitrite_matrix 4 10) ^ 2)
  matrix:set nitrite_matrix 5 11 ((matrix:get nitrite_matrix 3 11 / matrix:get nitrite_matrix 4 11) ^ 2)
  matrix:set nitrite_matrix 5 12 ((matrix:get nitrite_matrix 3 12 / matrix:get nitrite_matrix 4 12) ^ 2)
  matrix:set nitrite_matrix 5 13 ((matrix:get nitrite_matrix 3 13 / matrix:get nitrite_matrix 4 13) ^ 2)
  matrix:set nitrite_matrix 5 14 ((matrix:get nitrite_matrix 3 14 / matrix:get nitrite_matrix 4 14) ^ 2)
  matrix:set nitrite_matrix 5 15 ((matrix:get nitrite_matrix 3 15 / matrix:get nitrite_matrix 4 15) ^ 2)
  matrix:set nitrite_matrix 5 16 ((matrix:get nitrite_matrix 3 16 / matrix:get nitrite_matrix 4 16) ^ 2)

  let sum_nitrite_square sum matrix:get-row nitrite_matrix 5
  let nitrite_square sqrt ( (1 / (length matrix:get-row nitrite_matrix 5)) * sum_nitrite_square)
  set GRI_nitrite (1 + nitrite_square) / (1 - nitrite_square)

  matrix:set nitrous_oxide_matrix 3 0 (matrix:get nitrous_oxide_matrix 1 0 - matrix:get nitrous_oxide_matrix 2 0) matrix:set nitrous_oxide_matrix 4 0 (matrix:get nitrous_oxide_matrix 1 0 + matrix:get nitrous_oxide_matrix 2 0)
  matrix:set nitrous_oxide_matrix 3 1 (matrix:get nitrous_oxide_matrix 1 1 - matrix:get nitrous_oxide_matrix 2 1) matrix:set nitrous_oxide_matrix 4 1 (matrix:get nitrous_oxide_matrix 1 1 + matrix:get nitrous_oxide_matrix 2 1)
  matrix:set nitrous_oxide_matrix 3 2 (matrix:get nitrous_oxide_matrix 1 2 - matrix:get nitrous_oxide_matrix 2 2) matrix:set nitrous_oxide_matrix 4 2 (matrix:get nitrous_oxide_matrix 1 2 + matrix:get nitrous_oxide_matrix 2 2)
  matrix:set nitrous_oxide_matrix 3 3 (matrix:get nitrous_oxide_matrix 1 3 - matrix:get nitrous_oxide_matrix 2 3) matrix:set nitrous_oxide_matrix 4 3 (matrix:get nitrous_oxide_matrix 1 3 + matrix:get nitrous_oxide_matrix 2 3)
  matrix:set nitrous_oxide_matrix 3 4 (matrix:get nitrous_oxide_matrix 1 4 - matrix:get nitrous_oxide_matrix 2 4) matrix:set nitrous_oxide_matrix 4 4 (matrix:get nitrous_oxide_matrix 1 4 + matrix:get nitrous_oxide_matrix 2 4)
  matrix:set nitrous_oxide_matrix 3 5 (matrix:get nitrous_oxide_matrix 1 5 - matrix:get nitrous_oxide_matrix 2 5) matrix:set nitrous_oxide_matrix 4 5 (matrix:get nitrous_oxide_matrix 1 5 + matrix:get nitrous_oxide_matrix 2 5)
  matrix:set nitrous_oxide_matrix 3 6 (matrix:get nitrous_oxide_matrix 1 6 - matrix:get nitrous_oxide_matrix 2 6) matrix:set nitrous_oxide_matrix 4 6 (matrix:get nitrous_oxide_matrix 1 6 + matrix:get nitrous_oxide_matrix 2 6)
  matrix:set nitrous_oxide_matrix 3 7 (matrix:get nitrous_oxide_matrix 1 7 - matrix:get nitrous_oxide_matrix 2 7) matrix:set nitrous_oxide_matrix 4 7 (matrix:get nitrous_oxide_matrix 1 7 + matrix:get nitrous_oxide_matrix 2 7)
  matrix:set nitrous_oxide_matrix 3 8 (matrix:get nitrous_oxide_matrix 1 8 - matrix:get nitrous_oxide_matrix 2 8) matrix:set nitrous_oxide_matrix 4 8 (matrix:get nitrous_oxide_matrix 1 8 + matrix:get nitrous_oxide_matrix 2 8)
  matrix:set nitrous_oxide_matrix 3 9 (matrix:get nitrous_oxide_matrix 1 9 - matrix:get nitrous_oxide_matrix 2 9) matrix:set nitrous_oxide_matrix 4 9 (matrix:get nitrous_oxide_matrix 1 9 + matrix:get nitrous_oxide_matrix 2 9)
  matrix:set nitrous_oxide_matrix 3 10 (matrix:get nitrous_oxide_matrix 1 10 - matrix:get nitrous_oxide_matrix 2 10) matrix:set nitrous_oxide_matrix 4 10 (matrix:get nitrous_oxide_matrix 1 10 + matrix:get nitrous_oxide_matrix 2 10)
  matrix:set nitrous_oxide_matrix 3 11 (matrix:get nitrous_oxide_matrix 1 11 - matrix:get nitrous_oxide_matrix 2 11) matrix:set nitrous_oxide_matrix 4 11 (matrix:get nitrous_oxide_matrix 1 11 + matrix:get nitrous_oxide_matrix 2 11)
  matrix:set nitrous_oxide_matrix 3 12 (matrix:get nitrous_oxide_matrix 1 12 - matrix:get nitrous_oxide_matrix 2 12) matrix:set nitrous_oxide_matrix 4 12 (matrix:get nitrous_oxide_matrix 1 12 + matrix:get nitrous_oxide_matrix 2 12)

  matrix:set nitrous_oxide_matrix 5 0 ((matrix:get nitrous_oxide_matrix 3 0 / matrix:get nitrous_oxide_matrix 4 0) ^ 2)
  matrix:set nitrous_oxide_matrix 5 1 ((matrix:get nitrous_oxide_matrix 3 1 / matrix:get nitrous_oxide_matrix 4 1) ^ 2)
  matrix:set nitrous_oxide_matrix 5 2 ((matrix:get nitrous_oxide_matrix 3 2 / matrix:get nitrous_oxide_matrix 4 2) ^ 2)
  matrix:set nitrous_oxide_matrix 5 3 ((matrix:get nitrous_oxide_matrix 3 3 / matrix:get nitrous_oxide_matrix 4 3) ^ 2)
  matrix:set nitrous_oxide_matrix 5 4 ((matrix:get nitrous_oxide_matrix 3 4 / matrix:get nitrous_oxide_matrix 4 4) ^ 2)
  matrix:set nitrous_oxide_matrix 5 5 ((matrix:get nitrous_oxide_matrix 3 5 / matrix:get nitrous_oxide_matrix 4 5) ^ 2)
  matrix:set nitrous_oxide_matrix 5 6 ((matrix:get nitrous_oxide_matrix 3 6 / matrix:get nitrous_oxide_matrix 4 6) ^ 2)
  matrix:set nitrous_oxide_matrix 5 7 ((matrix:get nitrous_oxide_matrix 3 7 / matrix:get nitrous_oxide_matrix 4 7) ^ 2)
  matrix:set nitrous_oxide_matrix 5 8 ((matrix:get nitrous_oxide_matrix 3 8 / matrix:get nitrous_oxide_matrix 4 8) ^ 2)
  matrix:set nitrous_oxide_matrix 5 9 ((matrix:get nitrous_oxide_matrix 3 9 / matrix:get nitrous_oxide_matrix 4 9) ^ 2)
  matrix:set nitrous_oxide_matrix 5 10 ((matrix:get nitrous_oxide_matrix 3 10 / matrix:get nitrous_oxide_matrix 4 10) ^ 2)
  matrix:set nitrous_oxide_matrix 5 11 ((matrix:get nitrous_oxide_matrix 3 11 / matrix:get nitrous_oxide_matrix 4 11) ^ 2)
  matrix:set nitrous_oxide_matrix 5 12 ((matrix:get nitrous_oxide_matrix 3 12 / matrix:get nitrous_oxide_matrix 4 12) ^ 2)

  let sum_nitrous_oxide_square sum matrix:get-row nitrous_oxide_matrix 5
  let nitrous_oxide_square sqrt ( (1 / (length matrix:get-row nitrous_oxide_matrix 5)) * sum_nitrous_oxide_square)
  set GRI_nitrous_oxide (1 + nitrous_oxide_square) / (1 - nitrous_oxide_square)

  Fitness-Calculation

  ] ; end procedure succinate high P.denitrificans

  ] ; End procedure bacteria P. denitrificans

  if Denitrifying_bacterium = "A._xylosoxidans" [

  if Experiment = "Succ_Low/Nitrate_High"[

  matrix:set biomass_matrix 3 0 (matrix:get biomass_matrix 1 0 - matrix:get biomass_matrix 2 0) matrix:set biomass_matrix 4 0 (matrix:get biomass_matrix 1 0 + matrix:get biomass_matrix 2 0)
  matrix:set biomass_matrix 3 1 (matrix:get biomass_matrix 1 1 - matrix:get biomass_matrix 2 1) matrix:set biomass_matrix 4 1 (matrix:get biomass_matrix 1 1 + matrix:get biomass_matrix 2 1)
  matrix:set biomass_matrix 3 2 (matrix:get biomass_matrix 1 2 - matrix:get biomass_matrix 2 2) matrix:set biomass_matrix 4 2 (matrix:get biomass_matrix 1 2 + matrix:get biomass_matrix 2 2)
  matrix:set biomass_matrix 3 3 (matrix:get biomass_matrix 1 3 - matrix:get biomass_matrix 2 3) matrix:set biomass_matrix 4 3 (matrix:get biomass_matrix 1 3 + matrix:get biomass_matrix 2 3)
  matrix:set biomass_matrix 3 4 (matrix:get biomass_matrix 1 4 - matrix:get biomass_matrix 2 4) matrix:set biomass_matrix 4 4 (matrix:get biomass_matrix 1 4 + matrix:get biomass_matrix 2 4)
  matrix:set biomass_matrix 3 5 (matrix:get biomass_matrix 1 5 - matrix:get biomass_matrix 2 5) matrix:set biomass_matrix 4 5 (matrix:get biomass_matrix 1 5 + matrix:get biomass_matrix 2 5)
  matrix:set biomass_matrix 3 6 (matrix:get biomass_matrix 1 6 - matrix:get biomass_matrix 2 6) matrix:set biomass_matrix 4 6 (matrix:get biomass_matrix 1 6 + matrix:get biomass_matrix 2 6)
  matrix:set biomass_matrix 3 7 (matrix:get biomass_matrix 1 7 - matrix:get biomass_matrix 2 7) matrix:set biomass_matrix 4 7 (matrix:get biomass_matrix 1 7 + matrix:get biomass_matrix 2 7)
  matrix:set biomass_matrix 3 8 (matrix:get biomass_matrix 1 8 - matrix:get biomass_matrix 2 8) matrix:set biomass_matrix 4 8 (matrix:get biomass_matrix 1 8 + matrix:get biomass_matrix 2 8)
  matrix:set biomass_matrix 3 9 (matrix:get biomass_matrix 1 9 - matrix:get biomass_matrix 2 9) matrix:set biomass_matrix 4 9 (matrix:get biomass_matrix 1 9 + matrix:get biomass_matrix 2 9)
  matrix:set biomass_matrix 3 10 (matrix:get biomass_matrix 1 10 - matrix:get biomass_matrix 2 10) matrix:set biomass_matrix 4 10 (matrix:get biomass_matrix 1 10 + matrix:get biomass_matrix 2 10)

  matrix:set biomass_matrix 5 0 ((matrix:get biomass_matrix 3 0 / matrix:get biomass_matrix 4 0) ^ 2)
  matrix:set biomass_matrix 5 1 ((matrix:get biomass_matrix 3 1 / matrix:get biomass_matrix 4 1) ^ 2)
  matrix:set biomass_matrix 5 2 ((matrix:get biomass_matrix 3 2 / matrix:get biomass_matrix 4 2) ^ 2)
  matrix:set biomass_matrix 5 3 ((matrix:get biomass_matrix 3 3 / matrix:get biomass_matrix 4 3) ^ 2)
  matrix:set biomass_matrix 5 4 ((matrix:get biomass_matrix 3 4 / matrix:get biomass_matrix 4 4) ^ 2)
  matrix:set biomass_matrix 5 5 ((matrix:get biomass_matrix 3 5 / matrix:get biomass_matrix 4 5) ^ 2)
  matrix:set biomass_matrix 5 6 ((matrix:get biomass_matrix 3 6 / matrix:get biomass_matrix 4 6) ^ 2)
  matrix:set biomass_matrix 5 7 ((matrix:get biomass_matrix 3 7 / matrix:get biomass_matrix 4 7) ^ 2)
  matrix:set biomass_matrix 5 8 ((matrix:get biomass_matrix 3 8 / matrix:get biomass_matrix 4 8) ^ 2)
  matrix:set biomass_matrix 5 9 ((matrix:get biomass_matrix 3 9 / matrix:get biomass_matrix 4 9) ^ 2)
  matrix:set biomass_matrix 5 10 ((matrix:get biomass_matrix 3 10 / matrix:get biomass_matrix 4 10) ^ 2)

  let sum_biomass_square sum matrix:get-row biomass_matrix 5
  let biomass_square sqrt ( (1 / (length matrix:get-row biomass_matrix 5)) * sum_biomass_square)
  set GRI_biomass (1 + biomass_square) / (1 - biomass_square)

  matrix:set nitrate_matrix 3 0 (matrix:get nitrate_matrix 1 0 - matrix:get nitrate_matrix 2 0) matrix:set nitrate_matrix 4 0 (matrix:get nitrate_matrix 1 0 + matrix:get nitrate_matrix 2 0)
  matrix:set nitrate_matrix 3 1 (matrix:get nitrate_matrix 1 1 - matrix:get nitrate_matrix 2 1) matrix:set nitrate_matrix 4 1 (matrix:get nitrate_matrix 1 1 + matrix:get nitrate_matrix 2 1)
  matrix:set nitrate_matrix 3 2 (matrix:get nitrate_matrix 1 2 - matrix:get nitrate_matrix 2 2) matrix:set nitrate_matrix 4 2 (matrix:get nitrate_matrix 1 2 + matrix:get nitrate_matrix 2 2)
  matrix:set nitrate_matrix 3 3 (matrix:get nitrate_matrix 1 3 - matrix:get nitrate_matrix 2 3) matrix:set nitrate_matrix 4 3 (matrix:get nitrate_matrix 1 3 + matrix:get nitrate_matrix 2 3)
  matrix:set nitrate_matrix 3 4 (matrix:get nitrate_matrix 1 4 - matrix:get nitrate_matrix 2 4) matrix:set nitrate_matrix 4 4 (matrix:get nitrate_matrix 1 4 + matrix:get nitrate_matrix 2 4)
  matrix:set nitrate_matrix 3 5 (matrix:get nitrate_matrix 1 5 - matrix:get nitrate_matrix 2 5) matrix:set nitrate_matrix 4 5 (matrix:get nitrate_matrix 1 5 + matrix:get nitrate_matrix 2 5)
  matrix:set nitrate_matrix 3 6 (matrix:get nitrate_matrix 1 6 - matrix:get nitrate_matrix 2 6) matrix:set nitrate_matrix 4 6 (matrix:get nitrate_matrix 1 6 + matrix:get nitrate_matrix 2 6)
  matrix:set nitrate_matrix 3 7 (matrix:get nitrate_matrix 1 7 - matrix:get nitrate_matrix 2 7) matrix:set nitrate_matrix 4 7 (matrix:get nitrate_matrix 1 7 + matrix:get nitrate_matrix 2 7)
  matrix:set nitrate_matrix 3 8 (matrix:get nitrate_matrix 1 8 - matrix:get nitrate_matrix 2 8) matrix:set nitrate_matrix 4 8 (matrix:get nitrate_matrix 1 8 + matrix:get nitrate_matrix 2 8)
  matrix:set nitrate_matrix 3 9 (matrix:get nitrate_matrix 1 9 - matrix:get nitrate_matrix 2 9) matrix:set nitrate_matrix 4 9 (matrix:get nitrate_matrix 1 9 + matrix:get nitrate_matrix 2 9)
  matrix:set nitrate_matrix 3 10 (matrix:get nitrate_matrix 1 10 - matrix:get nitrate_matrix 2 10) matrix:set nitrate_matrix 4 10 (matrix:get nitrate_matrix 1 10 + matrix:get nitrate_matrix 2 10)
  matrix:set nitrate_matrix 3 11 (matrix:get nitrate_matrix 1 11 - matrix:get nitrate_matrix 2 11) matrix:set nitrate_matrix 4 11 (matrix:get nitrate_matrix 1 11 + matrix:get nitrate_matrix 2 11)

  matrix:set nitrate_matrix 5 0 ((matrix:get nitrate_matrix 3 0 / matrix:get nitrate_matrix 4 0) ^ 2)
  matrix:set nitrate_matrix 5 1 ((matrix:get nitrate_matrix 3 1 / matrix:get nitrate_matrix 4 1) ^ 2)
  matrix:set nitrate_matrix 5 2 ((matrix:get nitrate_matrix 3 2 / matrix:get nitrate_matrix 4 2) ^ 2)
  matrix:set nitrate_matrix 5 3 ((matrix:get nitrate_matrix 3 3 / matrix:get nitrate_matrix 4 3) ^ 2)
  matrix:set nitrate_matrix 5 4 ((matrix:get nitrate_matrix 3 4 / matrix:get nitrate_matrix 4 4) ^ 2)
  matrix:set nitrate_matrix 5 5 ((matrix:get nitrate_matrix 3 5 / matrix:get nitrate_matrix 4 5) ^ 2)
  matrix:set nitrate_matrix 5 6 ((matrix:get nitrate_matrix 3 6 / matrix:get nitrate_matrix 4 6) ^ 2)
  matrix:set nitrate_matrix 5 7 ((matrix:get nitrate_matrix 3 7 / matrix:get nitrate_matrix 4 7) ^ 2)
  matrix:set nitrate_matrix 5 8 ((matrix:get nitrate_matrix 3 8 / matrix:get nitrate_matrix 4 8) ^ 2)
  matrix:set nitrate_matrix 5 9 ((matrix:get nitrate_matrix 3 9 / matrix:get nitrate_matrix 4 9) ^ 2)
  matrix:set nitrate_matrix 5 10 ((matrix:get nitrate_matrix 3 10 / matrix:get nitrate_matrix 4 10) ^ 2)
  matrix:set nitrate_matrix 5 11 ((matrix:get nitrate_matrix 3 11 / matrix:get nitrate_matrix 4 11) ^ 2)

  let sum_nitrate_square sum matrix:get-row nitrate_matrix 5
  let nitrate_square sqrt ( (1 / (length matrix:get-row nitrate_matrix 5)) * sum_nitrate_square)
  set GRI_nitrate (1 + nitrate_square) / (1 - nitrate_square)

  matrix:set nitrite_matrix 3 0 (matrix:get nitrite_matrix 1 0 - matrix:get nitrite_matrix 2 0) matrix:set nitrite_matrix 4 0 (matrix:get nitrite_matrix 1 0 + matrix:get nitrite_matrix 2 0)
  matrix:set nitrite_matrix 3 1 (matrix:get nitrite_matrix 1 1 - matrix:get nitrite_matrix 2 1) matrix:set nitrite_matrix 4 1 (matrix:get nitrite_matrix 1 1 + matrix:get nitrite_matrix 2 1)
  matrix:set nitrite_matrix 3 2 (matrix:get nitrite_matrix 1 2 - matrix:get nitrite_matrix 2 2) matrix:set nitrite_matrix 4 2 (matrix:get nitrite_matrix 1 2 + matrix:get nitrite_matrix 2 2)
  matrix:set nitrite_matrix 3 3 (matrix:get nitrite_matrix 1 3 - matrix:get nitrite_matrix 2 3) matrix:set nitrite_matrix 4 3 (matrix:get nitrite_matrix 1 3 + matrix:get nitrite_matrix 2 3)
  matrix:set nitrite_matrix 3 4 (matrix:get nitrite_matrix 1 4 - matrix:get nitrite_matrix 2 4) matrix:set nitrite_matrix 4 4 (matrix:get nitrite_matrix 1 4 + matrix:get nitrite_matrix 2 4)
  matrix:set nitrite_matrix 3 5 (matrix:get nitrite_matrix 1 5 - matrix:get nitrite_matrix 2 5) matrix:set nitrite_matrix 4 5 (matrix:get nitrite_matrix 1 5 + matrix:get nitrite_matrix 2 5)
  matrix:set nitrite_matrix 3 6 (matrix:get nitrite_matrix 1 6 - matrix:get nitrite_matrix 2 6) matrix:set nitrite_matrix 4 6 (matrix:get nitrite_matrix 1 6 + matrix:get nitrite_matrix 2 6)
  matrix:set nitrite_matrix 3 7 (matrix:get nitrite_matrix 1 7 - matrix:get nitrite_matrix 2 7) matrix:set nitrite_matrix 4 7 (matrix:get nitrite_matrix 1 7 + matrix:get nitrite_matrix 2 7)
  matrix:set nitrite_matrix 3 8 (matrix:get nitrite_matrix 1 8 - matrix:get nitrite_matrix 2 8) matrix:set nitrite_matrix 4 8 (matrix:get nitrite_matrix 1 8 + matrix:get nitrite_matrix 2 8)
  matrix:set nitrite_matrix 3 9 (matrix:get nitrite_matrix 1 9 - matrix:get nitrite_matrix 2 9) matrix:set nitrite_matrix 4 9 (matrix:get nitrite_matrix 1 9 + matrix:get nitrite_matrix 2 9)
  matrix:set nitrite_matrix 3 10 (matrix:get nitrite_matrix 1 10 - matrix:get nitrite_matrix 2 10) matrix:set nitrite_matrix 4 10 (matrix:get nitrite_matrix 1 10 + matrix:get nitrite_matrix 2 10)
  matrix:set nitrite_matrix 3 11 (matrix:get nitrite_matrix 1 11 - matrix:get nitrite_matrix 2 11) matrix:set nitrite_matrix 4 11 (matrix:get nitrite_matrix 1 11 + matrix:get nitrite_matrix 2 11)
  matrix:set nitrite_matrix 3 12 (matrix:get nitrite_matrix 1 12 - matrix:get nitrite_matrix 2 12) matrix:set nitrite_matrix 4 12 (matrix:get nitrite_matrix 1 12 + matrix:get nitrite_matrix 2 12)

  matrix:set nitrite_matrix 5 0 ((matrix:get nitrite_matrix 3 0 / matrix:get nitrite_matrix 4 0) ^ 2)
  matrix:set nitrite_matrix 5 1 ((matrix:get nitrite_matrix 3 1 / matrix:get nitrite_matrix 4 1) ^ 2)
  matrix:set nitrite_matrix 5 2 ((matrix:get nitrite_matrix 3 2 / matrix:get nitrite_matrix 4 2) ^ 2)
  matrix:set nitrite_matrix 5 3 ((matrix:get nitrite_matrix 3 3 / matrix:get nitrite_matrix 4 3) ^ 2)
  matrix:set nitrite_matrix 5 4 ((matrix:get nitrite_matrix 3 4 / matrix:get nitrite_matrix 4 4) ^ 2)
  matrix:set nitrite_matrix 5 5 ((matrix:get nitrite_matrix 3 5 / matrix:get nitrite_matrix 4 5) ^ 2)
  matrix:set nitrite_matrix 5 6 ((matrix:get nitrite_matrix 3 6 / matrix:get nitrite_matrix 4 6) ^ 2)
  matrix:set nitrite_matrix 5 7 ((matrix:get nitrite_matrix 3 7 / matrix:get nitrite_matrix 4 7) ^ 2)
  matrix:set nitrite_matrix 5 8 ((matrix:get nitrite_matrix 3 8 / matrix:get nitrite_matrix 4 8) ^ 2)
  matrix:set nitrite_matrix 5 9 ((matrix:get nitrite_matrix 3 9 / matrix:get nitrite_matrix 4 9) ^ 2)
  matrix:set nitrite_matrix 5 10 ((matrix:get nitrite_matrix 3 10 / matrix:get nitrite_matrix 4 10) ^ 2)
  matrix:set nitrite_matrix 5 11 ((matrix:get nitrite_matrix 3 11 / matrix:get nitrite_matrix 4 11) ^ 2)
  matrix:set nitrite_matrix 5 12 ((matrix:get nitrite_matrix 3 12 / matrix:get nitrite_matrix 4 12) ^ 2)

  let sum_nitrite_square sum matrix:get-row nitrite_matrix 5
  let nitrite_square sqrt ( (1 / (length matrix:get-row nitrite_matrix 5)) * sum_nitrite_square)
  set GRI_nitrite (1 + nitrite_square) / (1 - nitrite_square)

  matrix:set nitrous_oxide_matrix 3 0 (matrix:get nitrous_oxide_matrix 1 0 - matrix:get nitrous_oxide_matrix 2 0) matrix:set nitrous_oxide_matrix 4 0 (matrix:get nitrous_oxide_matrix 1 0 + matrix:get nitrous_oxide_matrix 2 0)
  matrix:set nitrous_oxide_matrix 3 1 (matrix:get nitrous_oxide_matrix 1 1 - matrix:get nitrous_oxide_matrix 2 1) matrix:set nitrous_oxide_matrix 4 1 (matrix:get nitrous_oxide_matrix 1 1 + matrix:get nitrous_oxide_matrix 2 1)
  matrix:set nitrous_oxide_matrix 3 2 (matrix:get nitrous_oxide_matrix 1 2 - matrix:get nitrous_oxide_matrix 2 2) matrix:set nitrous_oxide_matrix 4 2 (matrix:get nitrous_oxide_matrix 1 2 + matrix:get nitrous_oxide_matrix 2 2)
  matrix:set nitrous_oxide_matrix 3 3 (matrix:get nitrous_oxide_matrix 1 3 - matrix:get nitrous_oxide_matrix 2 3) matrix:set nitrous_oxide_matrix 4 3 (matrix:get nitrous_oxide_matrix 1 3 + matrix:get nitrous_oxide_matrix 2 3)
  matrix:set nitrous_oxide_matrix 3 4 (matrix:get nitrous_oxide_matrix 1 4 - matrix:get nitrous_oxide_matrix 2 4) matrix:set nitrous_oxide_matrix 4 4 (matrix:get nitrous_oxide_matrix 1 4 + matrix:get nitrous_oxide_matrix 2 4)
  matrix:set nitrous_oxide_matrix 3 5 (matrix:get nitrous_oxide_matrix 1 5 - matrix:get nitrous_oxide_matrix 2 5) matrix:set nitrous_oxide_matrix 4 5 (matrix:get nitrous_oxide_matrix 1 5 + matrix:get nitrous_oxide_matrix 2 5)
  matrix:set nitrous_oxide_matrix 3 6 (matrix:get nitrous_oxide_matrix 1 6 - matrix:get nitrous_oxide_matrix 2 6) matrix:set nitrous_oxide_matrix 4 6 (matrix:get nitrous_oxide_matrix 1 6 + matrix:get nitrous_oxide_matrix 2 6)
  matrix:set nitrous_oxide_matrix 3 7 (matrix:get nitrous_oxide_matrix 1 7 - matrix:get nitrous_oxide_matrix 2 7) matrix:set nitrous_oxide_matrix 4 7 (matrix:get nitrous_oxide_matrix 1 7 + matrix:get nitrous_oxide_matrix 2 7)
  matrix:set nitrous_oxide_matrix 3 8 (matrix:get nitrous_oxide_matrix 1 8 - matrix:get nitrous_oxide_matrix 2 8) matrix:set nitrous_oxide_matrix 4 8 (matrix:get nitrous_oxide_matrix 1 8 + matrix:get nitrous_oxide_matrix 2 8)
  matrix:set nitrous_oxide_matrix 3 9 (matrix:get nitrous_oxide_matrix 1 9 - matrix:get nitrous_oxide_matrix 2 9) matrix:set nitrous_oxide_matrix 4 9 (matrix:get nitrous_oxide_matrix 1 9 + matrix:get nitrous_oxide_matrix 2 9)
  matrix:set nitrous_oxide_matrix 3 10 (matrix:get nitrous_oxide_matrix 1 10 - matrix:get nitrous_oxide_matrix 2 10) matrix:set nitrous_oxide_matrix 4 10 (matrix:get nitrous_oxide_matrix 1 10 + matrix:get nitrous_oxide_matrix 2 10)
  matrix:set nitrous_oxide_matrix 3 11 (matrix:get nitrous_oxide_matrix 1 11 - matrix:get nitrous_oxide_matrix 2 11) matrix:set nitrous_oxide_matrix 4 11 (matrix:get nitrous_oxide_matrix 1 11 + matrix:get nitrous_oxide_matrix 2 11)

  matrix:set nitrous_oxide_matrix 5 0 ((matrix:get nitrous_oxide_matrix 3 0 / matrix:get nitrous_oxide_matrix 4 0) ^ 2)
  matrix:set nitrous_oxide_matrix 5 1 ((matrix:get nitrous_oxide_matrix 3 1 / matrix:get nitrous_oxide_matrix 4 1) ^ 2)
  matrix:set nitrous_oxide_matrix 5 2 ((matrix:get nitrous_oxide_matrix 3 2 / matrix:get nitrous_oxide_matrix 4 2) ^ 2)
  matrix:set nitrous_oxide_matrix 5 3 ((matrix:get nitrous_oxide_matrix 3 3 / matrix:get nitrous_oxide_matrix 4 3) ^ 2)
  matrix:set nitrous_oxide_matrix 5 4 ((matrix:get nitrous_oxide_matrix 3 4 / matrix:get nitrous_oxide_matrix 4 4) ^ 2)
  matrix:set nitrous_oxide_matrix 5 5 ((matrix:get nitrous_oxide_matrix 3 5 / matrix:get nitrous_oxide_matrix 4 5) ^ 2)
  matrix:set nitrous_oxide_matrix 5 6 ((matrix:get nitrous_oxide_matrix 3 6 / matrix:get nitrous_oxide_matrix 4 6) ^ 2)
  matrix:set nitrous_oxide_matrix 5 7 ((matrix:get nitrous_oxide_matrix 3 7 / matrix:get nitrous_oxide_matrix 4 7) ^ 2)
  matrix:set nitrous_oxide_matrix 5 8 ((matrix:get nitrous_oxide_matrix 3 8 / matrix:get nitrous_oxide_matrix 4 8) ^ 2)
  matrix:set nitrous_oxide_matrix 5 9 ((matrix:get nitrous_oxide_matrix 3 9 / matrix:get nitrous_oxide_matrix 4 9) ^ 2)
  matrix:set nitrous_oxide_matrix 5 10 ((matrix:get nitrous_oxide_matrix 3 10 / matrix:get nitrous_oxide_matrix 4 10) ^ 2)
  matrix:set nitrous_oxide_matrix 5 11 ((matrix:get nitrous_oxide_matrix 3 11 / matrix:get nitrous_oxide_matrix 4 11) ^ 2)

  let sum_nitrous_oxide_square sum matrix:get-row nitrous_oxide_matrix 5
  let nitrous_oxide_square sqrt ( (1 / (length matrix:get-row nitrous_oxide_matrix 5)) * sum_nitrous_oxide_square)
  set GRI_nitrous_oxide (1 + nitrous_oxide_square) / (1 - nitrous_oxide_square)

  Fitness-Calculation

  ] ; end procedure succinate low A. Xylosoxidans

  if Experiment = "Succ_High/Nitrate_Low"[

  matrix:set biomass_matrix 3 0 (matrix:get biomass_matrix 1 0 - matrix:get biomass_matrix 2 0) matrix:set biomass_matrix 4 0 (matrix:get biomass_matrix 1 0 + matrix:get biomass_matrix 2 0)
  matrix:set biomass_matrix 3 1 (matrix:get biomass_matrix 1 1 - matrix:get biomass_matrix 2 1) matrix:set biomass_matrix 4 1 (matrix:get biomass_matrix 1 1 + matrix:get biomass_matrix 2 1)
  matrix:set biomass_matrix 3 2 (matrix:get biomass_matrix 1 2 - matrix:get biomass_matrix 2 2) matrix:set biomass_matrix 4 2 (matrix:get biomass_matrix 1 2 + matrix:get biomass_matrix 2 2)
  matrix:set biomass_matrix 3 3 (matrix:get biomass_matrix 1 3 - matrix:get biomass_matrix 2 3) matrix:set biomass_matrix 4 3 (matrix:get biomass_matrix 1 3 + matrix:get biomass_matrix 2 3)
  matrix:set biomass_matrix 3 4 (matrix:get biomass_matrix 1 4 - matrix:get biomass_matrix 2 4) matrix:set biomass_matrix 4 4 (matrix:get biomass_matrix 1 4 + matrix:get biomass_matrix 2 4)
  matrix:set biomass_matrix 3 5 (matrix:get biomass_matrix 1 5 - matrix:get biomass_matrix 2 5) matrix:set biomass_matrix 4 5 (matrix:get biomass_matrix 1 5 + matrix:get biomass_matrix 2 5)
  matrix:set biomass_matrix 3 6 (matrix:get biomass_matrix 1 6 - matrix:get biomass_matrix 2 6) matrix:set biomass_matrix 4 6 (matrix:get biomass_matrix 1 6 + matrix:get biomass_matrix 2 6)
  matrix:set biomass_matrix 3 7 (matrix:get biomass_matrix 1 7 - matrix:get biomass_matrix 2 7) matrix:set biomass_matrix 4 7 (matrix:get biomass_matrix 1 7 + matrix:get biomass_matrix 2 7)
  matrix:set biomass_matrix 3 8 (matrix:get biomass_matrix 1 8 - matrix:get biomass_matrix 2 8) matrix:set biomass_matrix 4 8 (matrix:get biomass_matrix 1 8 + matrix:get biomass_matrix 2 8)
  matrix:set biomass_matrix 3 9 (matrix:get biomass_matrix 1 9 - matrix:get biomass_matrix 2 9) matrix:set biomass_matrix 4 9 (matrix:get biomass_matrix 1 9 + matrix:get biomass_matrix 2 9)
  matrix:set biomass_matrix 3 10 (matrix:get biomass_matrix 1 10 - matrix:get biomass_matrix 2 10) matrix:set biomass_matrix 4 10 (matrix:get biomass_matrix 1 10 + matrix:get biomass_matrix 2 10)
  matrix:set biomass_matrix 3 11 (matrix:get biomass_matrix 1 11 - matrix:get biomass_matrix 2 11) matrix:set biomass_matrix 4 11 (matrix:get biomass_matrix 1 11 + matrix:get biomass_matrix 2 11)
  matrix:set biomass_matrix 3 12 (matrix:get biomass_matrix 1 12 - matrix:get biomass_matrix 2 12) matrix:set biomass_matrix 4 12 (matrix:get biomass_matrix 1 12 + matrix:get biomass_matrix 2 12)
  matrix:set biomass_matrix 3 13 (matrix:get biomass_matrix 1 13 - matrix:get biomass_matrix 2 13) matrix:set biomass_matrix 4 13 (matrix:get biomass_matrix 1 13 + matrix:get biomass_matrix 2 13)
  matrix:set biomass_matrix 3 14 (matrix:get biomass_matrix 1 14 - matrix:get biomass_matrix 2 14) matrix:set biomass_matrix 4 14 (matrix:get biomass_matrix 1 14 + matrix:get biomass_matrix 2 14)

  matrix:set biomass_matrix 5 0 ((matrix:get biomass_matrix 3 0 / matrix:get biomass_matrix 4 0) ^ 2)
  matrix:set biomass_matrix 5 1 ((matrix:get biomass_matrix 3 1 / matrix:get biomass_matrix 4 1) ^ 2)
  matrix:set biomass_matrix 5 2 ((matrix:get biomass_matrix 3 2 / matrix:get biomass_matrix 4 2) ^ 2)
  matrix:set biomass_matrix 5 3 ((matrix:get biomass_matrix 3 3 / matrix:get biomass_matrix 4 3) ^ 2)
  matrix:set biomass_matrix 5 4 ((matrix:get biomass_matrix 3 4 / matrix:get biomass_matrix 4 4) ^ 2)
  matrix:set biomass_matrix 5 5 ((matrix:get biomass_matrix 3 5 / matrix:get biomass_matrix 4 5) ^ 2)
  matrix:set biomass_matrix 5 6 ((matrix:get biomass_matrix 3 6 / matrix:get biomass_matrix 4 6) ^ 2)
  matrix:set biomass_matrix 5 7 ((matrix:get biomass_matrix 3 7 / matrix:get biomass_matrix 4 7) ^ 2)
  matrix:set biomass_matrix 5 8 ((matrix:get biomass_matrix 3 8 / matrix:get biomass_matrix 4 8) ^ 2)
  matrix:set biomass_matrix 5 9 ((matrix:get biomass_matrix 3 9 / matrix:get biomass_matrix 4 9) ^ 2)
  matrix:set biomass_matrix 5 10 ((matrix:get biomass_matrix 3 10 / matrix:get biomass_matrix 4 10) ^ 2)
  matrix:set biomass_matrix 5 11 ((matrix:get biomass_matrix 3 11 / matrix:get biomass_matrix 4 11) ^ 2)
  matrix:set biomass_matrix 5 12 ((matrix:get biomass_matrix 3 12 / matrix:get biomass_matrix 4 12) ^ 2)
  matrix:set biomass_matrix 5 13 ((matrix:get biomass_matrix 3 13 / matrix:get biomass_matrix 4 13) ^ 2)
  matrix:set biomass_matrix 5 14 ((matrix:get biomass_matrix 3 14 / matrix:get biomass_matrix 4 14) ^ 2)

  let sum_biomass_square sum matrix:get-row biomass_matrix 5
  let biomass_square sqrt ( (1 / (length matrix:get-row biomass_matrix 5)) * sum_biomass_square)
  set GRI_biomass (1 + biomass_square) / (1 - biomass_square)

  matrix:set nitrate_matrix 3 0 (matrix:get nitrate_matrix 1 0 - matrix:get nitrate_matrix 2 0) matrix:set nitrate_matrix 4 0 (matrix:get nitrate_matrix 1 0 + matrix:get nitrate_matrix 2 0)
  matrix:set nitrate_matrix 3 1 (matrix:get nitrate_matrix 1 1 - matrix:get nitrate_matrix 2 1) matrix:set nitrate_matrix 4 1 (matrix:get nitrate_matrix 1 1 + matrix:get nitrate_matrix 2 1)
  matrix:set nitrate_matrix 3 2 (matrix:get nitrate_matrix 1 2 - matrix:get nitrate_matrix 2 2) matrix:set nitrate_matrix 4 2 (matrix:get nitrate_matrix 1 2 + matrix:get nitrate_matrix 2 2)
  matrix:set nitrate_matrix 3 3 (matrix:get nitrate_matrix 1 3 - matrix:get nitrate_matrix 2 3) matrix:set nitrate_matrix 4 3 (matrix:get nitrate_matrix 1 3 + matrix:get nitrate_matrix 2 3)
  matrix:set nitrate_matrix 3 4 (matrix:get nitrate_matrix 1 4 - matrix:get nitrate_matrix 2 4) matrix:set nitrate_matrix 4 4 (matrix:get nitrate_matrix 1 4 + matrix:get nitrate_matrix 2 4)
  matrix:set nitrate_matrix 3 5 (matrix:get nitrate_matrix 1 5 - matrix:get nitrate_matrix 2 5) matrix:set nitrate_matrix 4 5 (matrix:get nitrate_matrix 1 5 + matrix:get nitrate_matrix 2 5)
  matrix:set nitrate_matrix 3 6 (matrix:get nitrate_matrix 1 6 - matrix:get nitrate_matrix 2 6) matrix:set nitrate_matrix 4 6 (matrix:get nitrate_matrix 1 6 + matrix:get nitrate_matrix 2 6)
  matrix:set nitrate_matrix 3 7 (matrix:get nitrate_matrix 1 7 - matrix:get nitrate_matrix 2 7) matrix:set nitrate_matrix 4 7 (matrix:get nitrate_matrix 1 7 + matrix:get nitrate_matrix 2 7)
  matrix:set nitrate_matrix 3 8 (matrix:get nitrate_matrix 1 8 - matrix:get nitrate_matrix 2 8) matrix:set nitrate_matrix 4 8 (matrix:get nitrate_matrix 1 8 + matrix:get nitrate_matrix 2 8)
  matrix:set nitrate_matrix 3 9 (matrix:get nitrate_matrix 1 9 - matrix:get nitrate_matrix 2 9) matrix:set nitrate_matrix 4 9 (matrix:get nitrate_matrix 1 9 + matrix:get nitrate_matrix 2 9)
  matrix:set nitrate_matrix 3 10 (matrix:get nitrate_matrix 1 10 - matrix:get nitrate_matrix 2 10) matrix:set nitrate_matrix 4 10 (matrix:get nitrate_matrix 1 10 + matrix:get nitrate_matrix 2 10)
  matrix:set nitrate_matrix 3 11 (matrix:get nitrate_matrix 1 11 - matrix:get nitrate_matrix 2 11) matrix:set nitrate_matrix 4 11 (matrix:get nitrate_matrix 1 11 + matrix:get nitrate_matrix 2 11)
  matrix:set nitrate_matrix 3 12 (matrix:get nitrate_matrix 1 12 - matrix:get nitrate_matrix 2 12) matrix:set nitrate_matrix 4 12 (matrix:get nitrate_matrix 1 12 + matrix:get nitrate_matrix 2 12)
  matrix:set nitrate_matrix 3 13 (matrix:get nitrate_matrix 1 13 - matrix:get nitrate_matrix 2 13) matrix:set nitrate_matrix 4 13 (matrix:get nitrate_matrix 1 13 + matrix:get nitrate_matrix 2 13)

  matrix:set nitrate_matrix 5 0 ((matrix:get nitrate_matrix 3 0 / matrix:get nitrate_matrix 4 0) ^ 2)
  matrix:set nitrate_matrix 5 1 ((matrix:get nitrate_matrix 3 1 / matrix:get nitrate_matrix 4 1) ^ 2)
  matrix:set nitrate_matrix 5 2 ((matrix:get nitrate_matrix 3 2 / matrix:get nitrate_matrix 4 2) ^ 2)
  matrix:set nitrate_matrix 5 3 ((matrix:get nitrate_matrix 3 3 / matrix:get nitrate_matrix 4 3) ^ 2)
  matrix:set nitrate_matrix 5 4 ((matrix:get nitrate_matrix 3 4 / matrix:get nitrate_matrix 4 4) ^ 2)
  matrix:set nitrate_matrix 5 5 ((matrix:get nitrate_matrix 3 5 / matrix:get nitrate_matrix 4 5) ^ 2)
  matrix:set nitrate_matrix 5 6 ((matrix:get nitrate_matrix 3 6 / matrix:get nitrate_matrix 4 6) ^ 2)
  matrix:set nitrate_matrix 5 7 ((matrix:get nitrate_matrix 3 7 / matrix:get nitrate_matrix 4 7) ^ 2)
  matrix:set nitrate_matrix 5 8 ((matrix:get nitrate_matrix 3 8 / matrix:get nitrate_matrix 4 8) ^ 2)
  matrix:set nitrate_matrix 5 9 ((matrix:get nitrate_matrix 3 9 / matrix:get nitrate_matrix 4 9) ^ 2)
  matrix:set nitrate_matrix 5 10 ((matrix:get nitrate_matrix 3 10 / matrix:get nitrate_matrix 4 10) ^ 2)
  matrix:set nitrate_matrix 5 11 ((matrix:get nitrate_matrix 3 11 / matrix:get nitrate_matrix 4 11) ^ 2)
  matrix:set nitrate_matrix 5 12 ((matrix:get nitrate_matrix 3 12 / matrix:get nitrate_matrix 4 12) ^ 2)
  matrix:set nitrate_matrix 5 13 ((matrix:get nitrate_matrix 3 13 / matrix:get nitrate_matrix 4 13) ^ 2)

  let sum_nitrate_square sum matrix:get-row nitrate_matrix 5
  let nitrate_square sqrt ( (1 / (length matrix:get-row nitrate_matrix 5)) * sum_nitrate_square)
  set GRI_nitrate (1 + nitrate_square) / (1 - nitrate_square)

  matrix:set nitrite_matrix 3 0 (matrix:get nitrite_matrix 1 0 - matrix:get nitrite_matrix 2 0) matrix:set nitrite_matrix 4 0 (matrix:get nitrite_matrix 1 0 + matrix:get nitrite_matrix 2 0)
  matrix:set nitrite_matrix 3 1 (matrix:get nitrite_matrix 1 1 - matrix:get nitrite_matrix 2 1) matrix:set nitrite_matrix 4 1 (matrix:get nitrite_matrix 1 1 + matrix:get nitrite_matrix 2 1)
  matrix:set nitrite_matrix 3 2 (matrix:get nitrite_matrix 1 2 - matrix:get nitrite_matrix 2 2) matrix:set nitrite_matrix 4 2 (matrix:get nitrite_matrix 1 2 + matrix:get nitrite_matrix 2 2)
  matrix:set nitrite_matrix 3 3 (matrix:get nitrite_matrix 1 3 - matrix:get nitrite_matrix 2 3) matrix:set nitrite_matrix 4 3 (matrix:get nitrite_matrix 1 3 + matrix:get nitrite_matrix 2 3)
  matrix:set nitrite_matrix 3 4 (matrix:get nitrite_matrix 1 4 - matrix:get nitrite_matrix 2 4) matrix:set nitrite_matrix 4 4 (matrix:get nitrite_matrix 1 4 + matrix:get nitrite_matrix 2 4)
  matrix:set nitrite_matrix 3 5 (matrix:get nitrite_matrix 1 5 - matrix:get nitrite_matrix 2 5) matrix:set nitrite_matrix 4 5 (matrix:get nitrite_matrix 1 5 + matrix:get nitrite_matrix 2 5)
  matrix:set nitrite_matrix 3 6 (matrix:get nitrite_matrix 1 6 - matrix:get nitrite_matrix 2 6) matrix:set nitrite_matrix 4 6 (matrix:get nitrite_matrix 1 6 + matrix:get nitrite_matrix 2 6)
  matrix:set nitrite_matrix 3 7 (matrix:get nitrite_matrix 1 7 - matrix:get nitrite_matrix 2 7) matrix:set nitrite_matrix 4 7 (matrix:get nitrite_matrix 1 7 + matrix:get nitrite_matrix 2 7)
  matrix:set nitrite_matrix 3 8 (matrix:get nitrite_matrix 1 8 - matrix:get nitrite_matrix 2 8) matrix:set nitrite_matrix 4 8 (matrix:get nitrite_matrix 1 8 + matrix:get nitrite_matrix 2 8)
  matrix:set nitrite_matrix 3 9 (matrix:get nitrite_matrix 1 9 - matrix:get nitrite_matrix 2 9) matrix:set nitrite_matrix 4 9 (matrix:get nitrite_matrix 1 9 + matrix:get nitrite_matrix 2 9)
  matrix:set nitrite_matrix 3 10 (matrix:get nitrite_matrix 1 10 - matrix:get nitrite_matrix 2 10) matrix:set nitrite_matrix 4 10 (matrix:get nitrite_matrix 1 10 + matrix:get nitrite_matrix 2 10)
  matrix:set nitrite_matrix 3 11 (matrix:get nitrite_matrix 1 11 - matrix:get nitrite_matrix 2 11) matrix:set nitrite_matrix 4 11 (matrix:get nitrite_matrix 1 11 + matrix:get nitrite_matrix 2 11)
  matrix:set nitrite_matrix 3 12 (matrix:get nitrite_matrix 1 12 - matrix:get nitrite_matrix 2 12) matrix:set nitrite_matrix 4 12 (matrix:get nitrite_matrix 1 12 + matrix:get nitrite_matrix 2 12)
  matrix:set nitrite_matrix 3 13 (matrix:get nitrite_matrix 1 13 - matrix:get nitrite_matrix 2 13) matrix:set nitrite_matrix 4 13 (matrix:get nitrite_matrix 1 13 + matrix:get nitrite_matrix 2 13)

  matrix:set nitrite_matrix 5 0 ((matrix:get nitrite_matrix 3 0 / matrix:get nitrite_matrix 4 0) ^ 2)
  matrix:set nitrite_matrix 5 1 ((matrix:get nitrite_matrix 3 1 / matrix:get nitrite_matrix 4 1) ^ 2)
  matrix:set nitrite_matrix 5 2 ((matrix:get nitrite_matrix 3 2 / matrix:get nitrite_matrix 4 2) ^ 2)
  matrix:set nitrite_matrix 5 3 ((matrix:get nitrite_matrix 3 3 / matrix:get nitrite_matrix 4 3) ^ 2)
  matrix:set nitrite_matrix 5 4 ((matrix:get nitrite_matrix 3 4 / matrix:get nitrite_matrix 4 4) ^ 2)
  matrix:set nitrite_matrix 5 5 ((matrix:get nitrite_matrix 3 5 / matrix:get nitrite_matrix 4 5) ^ 2)
  matrix:set nitrite_matrix 5 6 ((matrix:get nitrite_matrix 3 6 / matrix:get nitrite_matrix 4 6) ^ 2)
  matrix:set nitrite_matrix 5 7 ((matrix:get nitrite_matrix 3 7 / matrix:get nitrite_matrix 4 7) ^ 2)
  matrix:set nitrite_matrix 5 8 ((matrix:get nitrite_matrix 3 8 / matrix:get nitrite_matrix 4 8) ^ 2)
  matrix:set nitrite_matrix 5 9 ((matrix:get nitrite_matrix 3 9 / matrix:get nitrite_matrix 4 9) ^ 2)
  matrix:set nitrite_matrix 5 10 ((matrix:get nitrite_matrix 3 10 / matrix:get nitrite_matrix 4 10) ^ 2)
  matrix:set nitrite_matrix 5 11 ((matrix:get nitrite_matrix 3 11 / matrix:get nitrite_matrix 4 11) ^ 2)
  matrix:set nitrite_matrix 5 12 ((matrix:get nitrite_matrix 3 12 / matrix:get nitrite_matrix 4 12) ^ 2)
  matrix:set nitrite_matrix 5 13 ((matrix:get nitrite_matrix 3 13 / matrix:get nitrite_matrix 4 13) ^ 2)

  let sum_nitrite_square sum matrix:get-row nitrite_matrix 5
  let nitrite_square sqrt ( (1 / (length matrix:get-row nitrite_matrix 5)) * sum_nitrite_square)
  set GRI_nitrite (1 + nitrite_square) / (1 - nitrite_square)

  matrix:set nitrous_oxide_matrix 3 0 (matrix:get nitrous_oxide_matrix 1 0 - matrix:get nitrous_oxide_matrix 2 0) matrix:set nitrous_oxide_matrix 4 0 (matrix:get nitrous_oxide_matrix 1 0 + matrix:get nitrous_oxide_matrix 2 0)
  matrix:set nitrous_oxide_matrix 3 1 (matrix:get nitrous_oxide_matrix 1 1 - matrix:get nitrous_oxide_matrix 2 1) matrix:set nitrous_oxide_matrix 4 1 (matrix:get nitrous_oxide_matrix 1 1 + matrix:get nitrous_oxide_matrix 2 1)
  matrix:set nitrous_oxide_matrix 3 2 (matrix:get nitrous_oxide_matrix 1 2 - matrix:get nitrous_oxide_matrix 2 2) matrix:set nitrous_oxide_matrix 4 2 (matrix:get nitrous_oxide_matrix 1 2 + matrix:get nitrous_oxide_matrix 2 2)
  matrix:set nitrous_oxide_matrix 3 3 (matrix:get nitrous_oxide_matrix 1 3 - matrix:get nitrous_oxide_matrix 2 3) matrix:set nitrous_oxide_matrix 4 3 (matrix:get nitrous_oxide_matrix 1 3 + matrix:get nitrous_oxide_matrix 2 3)
  matrix:set nitrous_oxide_matrix 3 4 (matrix:get nitrous_oxide_matrix 1 4 - matrix:get nitrous_oxide_matrix 2 4) matrix:set nitrous_oxide_matrix 4 4 (matrix:get nitrous_oxide_matrix 1 4 + matrix:get nitrous_oxide_matrix 2 4)
  matrix:set nitrous_oxide_matrix 3 5 (matrix:get nitrous_oxide_matrix 1 5 - matrix:get nitrous_oxide_matrix 2 5) matrix:set nitrous_oxide_matrix 4 5 (matrix:get nitrous_oxide_matrix 1 5 + matrix:get nitrous_oxide_matrix 2 5)
  matrix:set nitrous_oxide_matrix 3 6 (matrix:get nitrous_oxide_matrix 1 6 - matrix:get nitrous_oxide_matrix 2 6) matrix:set nitrous_oxide_matrix 4 6 (matrix:get nitrous_oxide_matrix 1 6 + matrix:get nitrous_oxide_matrix 2 6)
  matrix:set nitrous_oxide_matrix 3 7 (matrix:get nitrous_oxide_matrix 1 7 - matrix:get nitrous_oxide_matrix 2 7) matrix:set nitrous_oxide_matrix 4 7 (matrix:get nitrous_oxide_matrix 1 7 + matrix:get nitrous_oxide_matrix 2 7)
  matrix:set nitrous_oxide_matrix 3 8 (matrix:get nitrous_oxide_matrix 1 8 - matrix:get nitrous_oxide_matrix 2 8) matrix:set nitrous_oxide_matrix 4 8 (matrix:get nitrous_oxide_matrix 1 8 + matrix:get nitrous_oxide_matrix 2 8)
  matrix:set nitrous_oxide_matrix 3 9 (matrix:get nitrous_oxide_matrix 1 9 - matrix:get nitrous_oxide_matrix 2 9) matrix:set nitrous_oxide_matrix 4 9 (matrix:get nitrous_oxide_matrix 1 9 + matrix:get nitrous_oxide_matrix 2 9)
  matrix:set nitrous_oxide_matrix 3 10 (matrix:get nitrous_oxide_matrix 1 10 - matrix:get nitrous_oxide_matrix 2 10) matrix:set nitrous_oxide_matrix 4 10 (matrix:get nitrous_oxide_matrix 1 10 + matrix:get nitrous_oxide_matrix 2 10)
  matrix:set nitrous_oxide_matrix 3 11 (matrix:get nitrous_oxide_matrix 1 11 - matrix:get nitrous_oxide_matrix 2 11) matrix:set nitrous_oxide_matrix 4 11 (matrix:get nitrous_oxide_matrix 1 11 + matrix:get nitrous_oxide_matrix 2 11)
  matrix:set nitrous_oxide_matrix 3 12 (matrix:get nitrous_oxide_matrix 1 12 - matrix:get nitrous_oxide_matrix 2 12) matrix:set nitrous_oxide_matrix 4 12 (matrix:get nitrous_oxide_matrix 1 12 + matrix:get nitrous_oxide_matrix 2 12)

  matrix:set nitrous_oxide_matrix 5 0 ((matrix:get nitrous_oxide_matrix 3 0 / matrix:get nitrous_oxide_matrix 4 0) ^ 2)
  matrix:set nitrous_oxide_matrix 5 1 ((matrix:get nitrous_oxide_matrix 3 1 / matrix:get nitrous_oxide_matrix 4 1) ^ 2)
  matrix:set nitrous_oxide_matrix 5 2 ((matrix:get nitrous_oxide_matrix 3 2 / matrix:get nitrous_oxide_matrix 4 2) ^ 2)
  matrix:set nitrous_oxide_matrix 5 3 ((matrix:get nitrous_oxide_matrix 3 3 / matrix:get nitrous_oxide_matrix 4 3) ^ 2)
  matrix:set nitrous_oxide_matrix 5 4 ((matrix:get nitrous_oxide_matrix 3 4 / matrix:get nitrous_oxide_matrix 4 4) ^ 2)
  matrix:set nitrous_oxide_matrix 5 5 ((matrix:get nitrous_oxide_matrix 3 5 / matrix:get nitrous_oxide_matrix 4 5) ^ 2)
  matrix:set nitrous_oxide_matrix 5 6 ((matrix:get nitrous_oxide_matrix 3 6 / matrix:get nitrous_oxide_matrix 4 6) ^ 2)
  matrix:set nitrous_oxide_matrix 5 7 ((matrix:get nitrous_oxide_matrix 3 7 / matrix:get nitrous_oxide_matrix 4 7) ^ 2)
  matrix:set nitrous_oxide_matrix 5 8 ((matrix:get nitrous_oxide_matrix 3 8 / matrix:get nitrous_oxide_matrix 4 8) ^ 2)
  matrix:set nitrous_oxide_matrix 5 9 ((matrix:get nitrous_oxide_matrix 3 9 / matrix:get nitrous_oxide_matrix 4 9) ^ 2)
  matrix:set nitrous_oxide_matrix 5 10 ((matrix:get nitrous_oxide_matrix 3 10 / matrix:get nitrous_oxide_matrix 4 10) ^ 2)
  matrix:set nitrous_oxide_matrix 5 11 ((matrix:get nitrous_oxide_matrix 3 11 / matrix:get nitrous_oxide_matrix 4 11) ^ 2)
  matrix:set nitrous_oxide_matrix 5 12 ((matrix:get nitrous_oxide_matrix 3 12 / matrix:get nitrous_oxide_matrix 4 12) ^ 2)

  let sum_nitrous_oxide_square sum matrix:get-row nitrous_oxide_matrix 5
  let nitrous_oxide_square sqrt ( (1 / (length matrix:get-row nitrous_oxide_matrix 5)) * sum_nitrous_oxide_square)
  set GRI_nitrous_oxide (1 + nitrous_oxide_square) / (1 - nitrous_oxide_square)

  Fitness-Calculation

  ] ; end procedure succinate high A. Xylosoxidans

  ] ; end procedure bacteria A.xylosoxidans
end

To Fitness-Calculation
  set Fitness ((GRI_nitrous_oxide - 1) ^ 2) + ((GRI_nitrite - 1) ^ 2) + ((GRI_nitrate - 1) ^ 2) + ((GRI_biomass - 1) ^ 2)
End
@#$#@#$#@
GRAPHICS-WINDOW
8
10
261
264
-1
-1
9.4231
1
10
1
1
1
0
1
1
1
0
25
0
25
1
1
1
Step time
30.0

SLIDER
8
290
232
323
Succinate
Succinate
0
20
5.0
0.1
1
[ mM ]
HORIZONTAL

SLIDER
9
323
233
356
Amonium
Amonium
0
15
10.0
0.1
1
[ mM ]
HORIZONTAL

MONITOR
4
532
98
577
Succinate [mM]
gsuccinate
6
1
11

MONITOR
97
532
169
577
NH4+ [mM]
gamonium
3
1
11

MONITOR
168
532
235
577
O2 [mg/l]
goxygen * 32
3
1
11

SLIDER
266
157
441
190
Microrganism
Microrganism
1
500
144.0
1
1
bacteria
HORIZONTAL

MONITOR
776
532
876
577
Biomass [mg/ml]
gbacteria
5
1
11

PLOT
4
827
395
1080
Products-CO2
Time [hours]
Concentration [ mM ]
0.0
120.0
0.0
0.01
true
false
"" ""
PENS
"CO2" 1.0 2 -14070903 true "" ""

SLIDER
9
356
233
389
Nitrate
Nitrate
0
25
21.60951
0.1
1
[ mM ]
HORIZONTAL

MONITOR
234
532
310
577
NO3- [mM]
gnitrate
3
1
11

PLOT
790
827
1180
1081
Culture Medium Nutrients
Time [hours]
Concentration[ mM ]
0.0
120.0
0.0
10.0
true
true
"" ""
PENS
"Succinate" 1.0 2 -14454117 true "" ""
"Amonium" 1.0 2 -15302303 true "" ""

PLOT
446
231
824
453
Products - NO2-
Time [hours]
Concentration [ mM ]
0.0
120.0
0.0
0.0
true
false
"" ""
PENS
"NO2-" 1.0 2 -7858858 true "" ""

PLOT
446
10
822
232
Biomass
Time [hours]
Biomass [mg/ml]
0.0
120.0
0.0
0.0
true
false
"" ""
PENS
"Biomass" 1.0 2 -2674135 true "" ""

SLIDER
267
256
441
289
NOX
NOX
0.01
0.3125
0.02
0.01
1
[ mM O2 ]
HORIZONTAL

MONITOR
381
532
448
577
NO [mM]
gno
6
1
11

MONITOR
309
532
382
577
NO2- [mM]
gno2-
3
1
11

MONITOR
585
532
660
577
CO2 [mM]
gdioxide
3
1
11

MONITOR
447
532
522
577
N2O [uM]
gn2o
3
1
11

MONITOR
521
532
586
577
N2 [mM]
gn2
3
1
11

PLOT
396
577
788
826
Products - NO
Time [hours]
Concentration [ mM ]
0.0
120.0
0.0
1.0E-10
true
false
"" ""
PENS
"NO" 1.0 2 -12087248 true "" ""

PLOT
822
230
1195
453
Products-N2O
Time [hours]
Concentration[ uM ]
0.0
120.0
0.0
0.0
true
false
"" ""
PENS
"N2O" 1.0 2 -13345367 true "" ""

SLIDER
234
290
441
323
Simulation_Time
Simulation_Time
0
250
120.0
1
1
[ hours ]
HORIZONTAL

PLOT
790
577
1180
825
Products - N2
Time [hours]
Concentration [ mM ]
0.0
120.0
0.0
0.001
true
false
"" ""
PENS
"N2" 1.0 2 -2674135 true "" ""

MONITOR
874
532
957
577
Time [hours]
time_now
2
1
11

SLIDER
101
388
232
421
uSc
uSc
0
u_succ
0.204
0.001
1
NIL
HORIZONTAL

SLIDER
7
388
99
421
nsc
nsc
0
1
0.28
0.01
1
NIL
HORIZONTAL

SLIDER
267
223
441
256
Min/steptime
Min/steptime
1
60
5.0
1
1
[min]
HORIZONTAL

SLIDER
234
324
441
357
Time_Shutdown_O2
Time_Shutdown_O2
1
48
24.0
1
1
[hours]
HORIZONTAL

SWITCH
266
91
441
124
Output_file?
Output_file?
1
1
-1000

MONITOR
659
532
728
577
Individuals
count bacteria
0
1
11

BUTTON
284
10
350
43
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
365
10
428
43
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
234
357
326
390
nntr
nntr
0
1
0.63
0.01
1
NIL
HORIZONTAL

SLIDER
325
356
441
389
uNO3-x
uNO3-x
0
u_no3-x
0.039
0.001
1
NIL
HORIZONTAL

SLIDER
234
390
326
423
nnr
nnr
0
1
0.77
0.01
1
NIL
HORIZONTAL

SLIDER
327
388
442
421
uNO2-
uNO2-
0
u_no2-
0.21383722098075134
0.001
1
NIL
HORIZONTAL

SLIDER
234
422
326
455
nmn
nmn
0
1
1.0
0.01
1
NIL
HORIZONTAL

SLIDER
327
420
442
453
uNO
uNO
0
u_no
0.15012175695128235
0.001
1
NIL
HORIZONTAL

PLOT
4
577
395
826
Oxygen
Time [hours]
Concentration [mg/l]
0.0
120.0
0.0
10.0
true
false
"" ""
PENS
"oxygen" 1.0 2 -955883 true "" ""

PLOT
822
10
1195
233
Nitrate
Time [hours]
Concentration [mM]
0.0
120.0
0.0
0.0
true
false
"" ""
PENS
"nitrate" 1.0 2 -10141563 true "" ""

SLIDER
446
454
693
487
energy_maintenance
energy_maintenance
0
0.004
0.002
0.0001
1
[g C/gCm h]
HORIZONTAL

MONITOR
727
532
777
577
Die Bac
bacteriaout
0
1
11

SLIDER
235
454
327
487
ndn
ndn
0
1
0.49
0.01
1
NIL
HORIZONTAL

SLIDER
329
453
440
486
uN2O
uN2O
0
u_n2o
0.104
0.001
1
NIL
HORIZONTAL

MONITOR
956
532
1013
577
NIL
d
3
1
11

SLIDER
266
190
441
223
di
di
0
0.1
0.05
0.01
1
[ 1/h ]
HORIZONTAL

SLIDER
7
454
99
487
nx
nx
0
1
0.79
0.01
1
NIL
HORIZONTAL

SLIDER
101
455
232
488
uox
uox
0
u_o2
0.125
0.001
1
NIL
HORIZONTAL

SLIDER
101
422
232
455
uam
uam
0
u_nh4+
0.1045
0.001
1
NIL
HORIZONTAL

SLIDER
7
421
99
454
nam
nam
0
1
0.84
0.01
1
NIL
HORIZONTAL

PLOT
791
1336
1180
1597
Metabolic_reactions
NIL
NIL
0.0
120.0
0.0
1.0
true
true
"" ""
PENS
"m_aero" 1.0 2 -1184463 true "" ""
"s_aero_1" 1.0 2 -14070903 true "" ""
"m_anox_1" 1.0 2 -7500403 true "" ""
"m_anox_2" 1.0 2 -2674135 true "" ""
"m_anox_3" 1.0 2 -955883 true "" ""
"m_anox_4" 1.0 2 -6459832 true "" ""
"s_anox_1" 1.0 2 -8630108 true "" ""
"s_anox_2" 1.0 2 -11221820 true "" ""
"s_anox_3" 1.0 2 -13840069 true "" ""
"s_anox_4" 1.0 2 -12186836 true "" ""
"s_aero_2" 1.0 2 -5825686 true "" ""
"R" 1.0 2 -16448764 true "" ""
"ac_mn" 1.0 0 -1191199 true "" ""
"ac_dn" 1.0 0 -2695187 true "" ""

PLOT
396
827
788
1080
Bicarbonate
Time [hours]
Concentration [mM]
0.0
120.0
0.0
0.0
true
false
"" ""
PENS
"bicarbonate" 1.0 2 -2674135 true "" ""

CHOOSER
266
46
442
91
Experiment
Experiment
"Succ_High/Nitrate_Low" "Succ_Low/Nitrate_High"
1

MONITOR
5
1086
146
1131
Succinate-ini (mmol)
isuccinate
3
1
11

MONITOR
5
1129
147
1174
Amonium-ini (mmol)
iamonium
3
1
11

MONITOR
5
1172
147
1217
oxygen-ini (mmol)
ioxygen
3
1
11

MONITOR
5
1215
147
1260
Nitrate-ini (mmol)
initrate
3
1
11

MONITOR
5
1258
147
1303
Nitrite-ini (mmol)
initrite
3
1
11

MONITOR
5
1300
147
1345
Monoxide-ini (mmol)
imonoxide
3
1
11

MONITOR
5
1342
147
1387
N2O-ini (mmol)
idinitrogen
3
1
11

MONITOR
5
1430
147
1475
CO2-ini (mmol)
idioxide
3
1
11

MONITOR
5
1516
147
1561
HCO3-ini (mmol)
ibicarbonate
3
1
11

MONITOR
5
1473
147
1518
Biomass-ini (mmol)
ibacteria
3
1
11

MONITOR
5
1385
147
1430
N2-ini (mmol)
initrogen
3
1
11

MONITOR
146
1086
284
1131
∆Succinate (mmol)
fsuccinate - isuccinate
4
1
11

MONITOR
145
1129
284
1174
∆Amonium (mmol)
famonium - iamonium
4
1
11

MONITOR
145
1172
284
1217
∆Oxygen (mmol)
foxygen - ioxygen
4
1
11

MONITOR
145
1215
284
1260
∆Nitrate (mmol)
fnitrate - initrate
4
1
11

MONITOR
145
1257
284
1302
∆Nitrite (mmol)
fnitrite - initrite
4
1
11

MONITOR
145
1300
284
1345
∆NO (mmol)
fmonoxide - imonoxide
4
1
11

MONITOR
147
1429
283
1474
∆CO2 (mmol)
fdioxide - idioxide
4
1
11

MONITOR
147
1385
283
1430
∆N2 (mmol)
fnitrogen - initrogen
4
1
11

MONITOR
146
1517
285
1562
∆HCO3- (mmol)
fbicarbonate - ibicarbonate
4
1
11

MONITOR
146
1473
284
1518
∆Biomass (mmol)
fbacteria - ibacteria
4
1
11

MONITOR
145
1342
284
1387
∆N2O (mmol)
fdinitrogen - idinitrogen
4
1
11

MONITOR
282
1086
445
1131
Succinate-inflow (mmol)
insuccinate
3
1
11

MONITOR
282
1128
445
1173
Amonium-inflow (mmol)
inamonium
3
1
11

MONITOR
283
1215
446
1260
Nitrate-inflow (mmol)
innitrate
3
1
11

MONITOR
444
1086
591
1131
Succinate-out (mmol)
outsuccinate
4
1
11

MONITOR
443
1128
591
1173
Amonium-out (mmol)
outamonium
4
1
11

MONITOR
446
1215
594
1260
Nitrate-out (mmol)
outnitrate
4
1
11

MONITOR
447
1258
594
1303
Nitrite-out (mmol)
outnitrite
4
1
11

MONITOR
447
1301
594
1346
NO-out (mmol)
outmonoxide
4
1
11

MONITOR
447
1343
594
1388
N2O-out (mmol)
outdinitrogen
4
1
11

MONITOR
447
1386
594
1431
N2-out (mmol)
outnitrogen
4
1
11

MONITOR
447
1429
594
1474
CO2-out (mmol)
outdioxide
4
1
11

MONITOR
447
1472
594
1517
Bacteria-out (mmol)
outbacteria
4
1
11

MONITOR
447
1516
594
1561
HCO3-out (mmol)
outbicarbonate
4
1
11

MONITOR
593
1473
696
1518
c-Mic [mg]
c-bacteria
2
1
11

MONITOR
695
1473
778
1518
N-bac [mg]
n-bacteria
2
1
11

MONITOR
590
1086
710
1131
C-Succinate (mg)
c-succinate
2
1
11

MONITOR
590
1128
711
1173
N-Amonium (mg)
n-amonium
2
1
11

MONITOR
594
1215
696
1260
n-Nitrate (mg)
n-nitrate
2
1
11

MONITOR
594
1258
696
1303
n-Nitrite (mg)
n-nitrite
2
1
11

MONITOR
594
1300
696
1345
n-NO (mg)
n-monoxide
2
1
11

MONITOR
594
1343
696
1388
N-N2O (mg)
n-dinitrogen
2
1
11

MONITOR
594
1386
696
1431
N-N2 (mg)
n-nitrogen
2
1
11

MONITOR
594
1429
696
1474
c-CO2 (mg)
c-co2
2
1
11

MONITOR
593
1516
698
1561
c-HCO3 (mg)
c-hco3
2
1
11

MONITOR
304
1278
417
1323
c-obtained (mg)
c-obtained
2
1
11

MONITOR
304
1323
417
1368
c-delivered (mg)
c-delivered
2
1
11

MONITOR
306
1451
420
1496
n-obtained (mg)
n-obtained
2
1
11

MONITOR
307
1496
420
1541
n-delivered (mg)
n-delivered
2
1
11

MONITOR
1011
532
1068
577
%eC
%_e_c
2
1
11

MONITOR
1066
532
1123
577
%eN
%_e_n
2
1
11

SLIDER
693
453
988
486
self_consumption_energy
self_consumption_energy
0
0.10
0.022
0.001
1
[h ^-1]
HORIZONTAL

MONITOR
1122
532
1179
577
Yc/c
yc/c
3
1
11

PLOT
791
1083
1181
1338
Bacterial-biomass-distribution
Cell-Biomass
Frequency
0.0
2.0
0.0
2.0
true
false
"" ""
PENS
"biomass-histogram" 1.0 1 -16777216 true "" ""

PLOT
792
1596
1182
1845
%eC / %eN / d
NIL
NIL
0.0
120.0
0.0
0.0
true
true
"" ""
PENS
"d" 1.0 2 -16777216 true "" "plotxy time_now d"
"%eC" 1.0 2 -2674135 true "" "plotxy time_now %_e_c"
"%eN" 1.0 2 -13345367 true "" "plotxy time_now %_e_n"

SLIDER
989
453
1100
486
uNO3-a
uNO3-a
0
u_no3-a
0.00911
0.0001
1
NIL
HORIZONTAL

SLIDER
1101
453
1194
486
k_avail
k_avail
0
1
0.3
0.01
1
NIL
HORIZONTAL

INPUTBOX
5
1564
61
1624
carbon
3.0
1
0
Number

INPUTBOX
64
1564
126
1624
hydrogen
5.4
1
0
Number

INPUTBOX
126
1564
185
1624
oxygen
1.45
1
0
Number

INPUTBOX
185
1564
257
1624
b-nitrogen
0.75
1
0
Number

MONITOR
7
1674
126
1719
P.M. [g/mol]
molecular-weight
3
1
11

BUTTON
7
1722
73
1755
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7
1758
72
1791
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
129
1674
257
1719
∆Gpc [KJ/e-eeq]
dGpc
3
1
11

SLIDER
450
1576
542
1609
e1
e1
0
1
0.76
0.01
1
NIL
HORIZONTAL

SLIDER
450
1609
542
1642
e2
e2
0
1
0.65
0.01
1
NIL
HORIZONTAL

SLIDER
450
1667
542
1700
e3
e3
0
1
0.41
0.01
1
NIL
HORIZONTAL

SLIDER
450
1700
542
1733
e4
e4
0
1
0.84
0.01
1
NIL
HORIZONTAL

SLIDER
450
1756
542
1789
e5
e5
0
1
0.66
0.01
1
NIL
HORIZONTAL

SLIDER
450
1789
542
1822
e6
e6
0
1
0.27
0.01
1
NIL
HORIZONTAL

MONITOR
542
1565
599
1610
A1
A1
3
1
11

MONITOR
600
1565
657
1610
NIL
fs1
3
1
11

MONITOR
658
1565
715
1610
NIL
fe1
3
1
11

MONITOR
716
1566
773
1611
NIL
yc/c1
3
1
11

MONITOR
542
1609
599
1654
NIL
A2
3
1
11

MONITOR
600
1610
657
1655
NIL
fs2
3
1
11

MONITOR
658
1610
715
1655
NIL
fe2
3
1
11

MONITOR
716
1611
773
1656
NIL
yc/c2
3
1
11

MONITOR
542
1654
599
1699
NIL
A3
3
1
11

MONITOR
600
1655
657
1700
NIL
fs3
3
1
11

MONITOR
658
1655
715
1700
NIL
fe3
3
1
11

MONITOR
716
1655
773
1700
NIL
yc/c3
3
1
11

MONITOR
542
1699
599
1744
NIL
A4
3
1
11

MONITOR
600
1700
657
1745
NIL
fs4
3
1
11

MONITOR
658
1700
715
1745
NIL
fe4
3
1
11

MONITOR
716
1700
773
1745
NIL
yc/c4
3
1
11

MONITOR
542
1744
599
1789
NIL
A5
3
1
11

MONITOR
600
1745
657
1790
NIL
fs5
3
1
11

MONITOR
658
1746
715
1791
NIL
fe5
3
1
11

MONITOR
716
1746
773
1791
NIL
yc/c5
3
1
11

MONITOR
543
1789
600
1834
NIL
A6
3
1
11

MONITOR
601
1790
658
1835
NIL
fs6
3
1
11

MONITOR
658
1790
715
1835
NIL
fe6
3
1
11

MONITOR
716
1791
773
1836
NIL
yc/c6
3
1
11

OUTPUT
0
1843
1181
2248
12

MONITOR
130
1722
256
1767
NIL
gamma_cells
3
1
11

SWITCH
266
124
441
157
Output-data?
Output-data?
1
1
-1000

INPUTBOX
258
1627
355
1687
u_max
0.418
1
0
Number

MONITOR
4
487
137
532
[mol succ/mol Cm h]
u_succ
3
1
11

MONITOR
138
487
271
532
[mol NH4+/molCm h]
u_nh4+
3
1
11

MONITOR
271
487
404
532
[mol O2/molCm h]
u_o2
3
1
11

MONITOR
405
487
538
532
[mol NO3-a/molCm h]
u_no3-a
4
1
11

MONITOR
539
487
672
532
[mol NO3-x/molCm h]
u_no3-x
3
1
11

MONITOR
672
487
801
532
[mol NO2-/molCm h]
u_no2-
3
1
11

MONITOR
801
487
917
532
[mol NO/molCm h]
u_no
3
1
11

MONITOR
917
487
1039
532
[mol N2O/molCm h]
u_n2o
3
1
11

MONITOR
1039
487
1111
532
u [ h-1 ]
u
3
1
11

MONITOR
1112
486
1194
531
umax [ h-1 ]
umax
3
1
11

CHOOSER
276
1714
442
1759
Denitrifying_bacterium
Denitrifying_bacterium
"P._denitrificans" "A._xylosoxidans"
0

MONITOR
6
1626
125
1671
deq small [um]
deq_small
2
1
11

MONITOR
128
1626
255
1671
deq big [um]
deq_big
2
1
11

MONITOR
8
1794
100
1839
GRI_Biomass
GRI_biomass
3
1
11

MONITOR
102
1795
184
1840
GRI_Nitrate
GRI_nitrate
3
1
11

MONITOR
187
1795
266
1840
GRI_Nitrite
GRI_nitrite
3
1
11

MONITOR
268
1795
337
1840
GRI_N2O
GRI_nitrous_oxide
3
1
11

MONITOR
337
1795
394
1840
NIL
Fitness
1
1
11

@#$#@#$#@
INDISIM-Denitrification, an individual-based model for study the denitrification process
(Araujo, Gras, Ginovart & Moulton 2019)

## WHAT IS IT?

This is the Netlogo implementation of the Model INDISIM-Denitrification. INDISIM-Denitrification represents a bioreactor to grow any denitrifying bacteria with two operating protocols (batch and CSTR conditions). The culture medium have succinate, as electron donor and C-source, ammonium (NH4+) as N-source and various electron acceptors as O2, NO3-, NO2-, NO, and N2O, in order to mimic two different experimental protocols: 1) electron-donor limited / electron acceptor sufficient and 2) electron-donor sufficient / electron acceptor limited.
INDISIM-Denitrification is the first big update of INDISIM-Paracoccus.

## HOW IT WORKS (what rules the agents use to create the overall behavior of the model)

The overall behavior of the model is created when the individuals (single cell of any denitrifying bacterium) use 5 basic behavior-rules: a) individual nutrient uptake-rate, b) cellular maintenance, c) individual mass generation, d) individual mass degradation to reduce cytotoxic products (NO and N2O), and e) reproduction cycle.
These rules have special protocols, they are:
1) individual nutrient uptake-rate, which is obtained from an input value of Umax and calculated using the stoichiometric coefficients of the metabolic reactions adjusted by TEEM2, also, the uptake-rate is accord to individual biomass and update each time step with the biomass changes.
2) Cellular maintenance value is an appropriate requirement for heterotrophic microorganisms and its different for aerobic and anaerobic phases.
3) individual mass generation is possible if the microorganism executes any of the six metabolic pathways adjusted by TEEM2.
4) individual mass degradation to reduce cytotoxic products (NO and N2O) value is a rate by which the microorganisms reduce its mass when in the culture medium exists No and/or N2O.
5) reproduction cycle is accord to binary fission of any bacterium.
The behaviour rules for the culture medium are the chemostat general procedures and they represent agitation, batch culture, continuous culture (CSTR) and dilution ratio.

Any denitrifying bacterium is the only individual in the bioreactor and only interacts with the culture medium. Simulated microorganisms do not develop aggregates; each microorganism acts uniquely. The individual identifies its biomass and uptake capacities. With this information, the agent compares their uptake capacities with the nutrient's availability and takes the lowest value as your nutrient intake.

The first decision performing by any microorganism is about the response to the O2 dissolved level in the culture medium if the O2 dissolved level is lower than NOX value the microorganism execute anaerobic metabolism and if it is higher the agent executes aerobic metabolism.
The second decision performing by any microorganism is running metabolic pathways for growth only if that reached their cellular maintenance in aerobic or anaerobic phase.
The third decision performing by any microorganism is about reproduction that means not to make bipartitions if that has not reached the minimum reproduction mass.
The fourth decision performing by any microorganism is related to its response with the culture medium concentration of the cytotoxic products NO and/or N2O. 

## HOW TO USE IT (how to use the model, including a description of each of the items in the Interface tab)

The NetLogo interface makes it easy to change initial values via sliders. The user can adjust:
1) Nutrient concentrations: [mM] for succinate, ammonium, and nitrate these values are determined from the normal distribution of which mean value is according to the experimental procedure with a standard deviation of 5% mean value for each nutrient. The initial nutrient concentration (batch culture) is the same for the reservoir medium feed (continuous culture).
2) NOX value [mM]: which is fixed by the user in the range of 0.01 to 0.31 mM O2 and is determined from the normal distribution of which means is that fixed value with a standard deviation of 5% mean value. If the O2 dissolved level is lower than NOX value the agent executes anaerobic metabolism and if it is higher the agent executes aerobic metabolism.
3) Time shutdown O2 [h]: With this slider,, the user can adjust the aerobic phase time. If the simulation time is lower than “time shutdown O2” the bioreactor works in aerobic batch culture. If the simulation time is highest than “time shutdown O2” the aeration was then switched off and the system switched to continuous culture with a dilution rate adjusted by the user.
4) Dilution ratio [1/h]: The user can adjust this value for the continuous culture.
5) Microorganism [microbes]: Initial amount of viable microorganisms.
6) Simulation time [h]: with this slider, the user can adjust the total time for simulation experiments.
7) Step time [min]: INDISIM-Denitrification is discrete in space and time, with this slider the user can establish the time scale, one full program run is equal a one-step time.
8) Nutrients availability (ni) [1/h]: Is it possible to adjust the maximum nutrient availability for the total of chemical species (nutrients and denitrification products). Succinate, ammonium, nitrate, nitrite, nitric oxide and nitrous oxide. ni a number between 0 and 1 that indicates the fraction of the total nutrient in the culture medium is available for the individuals living in the spatial unit. This parameter is directly related to the culture medium characteristics and not to the types of microorganisms involved.
9) Nutrients uptake rates (ui) [mol/mol·h]: Is it possible to adjust the maximum microbial uptake rate for each nutrient. ui is the amount of nutrient per unit of time, that a microorganism could use under non-limiting nutrient availability and is correlated with the individual biomass. This value is a microscopic parameter directly related to the type of microorganism involved.
10) Maintenance energy requirement [gC/gCm·h]: Before biomass generation, the microorganism must complete its basal metabolism to keep its structures. The user can adjust the value for the anaerobic phase
11) Input data: Normal functioning external files are not necessary for the model to run simulated procedures.
12) The graphical and numerical outputs of the model are the concentration [mM or uM] of each culture medium component (succinate, NH4+, O2, NO3- CO2, NO2-, NO, N2O and N2) and microbial biomass [mg/ml] each time step. Additionally, the user could obtain all simulated data in the output file with the extension “.txt”.
13) All of the thermodynamic calculations using TEEM2 are included in the main code, this feature gives the opportunity to test the model with any denitrifying bacterium.
14) To compare the model with experimental results, the main code uses the GRI procedure (Geometric Reliability Index) and the user has the possibility to add the own experimental data. 

## THINGS TO NOTICE (suggested things for the user to notice while running the model)

It is possible to check the biomass distribution of the bacteria, the number of times that each metabolic pathway is executed, all the balanced chemical reactions that represent the pathways and the time evolutions of all nutrients and bacteria in the bioreactor.

## THINGS TO TRY (suggested things for the user to try to do (move sliders, switches, etc.) with the model)

To study the denitrification dynamics the user can move the uptake sliders and the step time scale proposed. In batch aerobic cultures the individual could execute 2 metabolic pathways each step time. To continuos culture, the individual could execute 4 metabolic pathways each step time. Also the culture medium concentration for experiments with nutrient-limited. The overall simulation time. The maintenance energy requirement. The individual mass degradation energy.
To study the direct effect of some time-dependent variables is possible to realize any behavior-space and behavior-search experiments.

## EXTENDING THE MODEL (suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

The further works point out that make adjustments to be closer to the denitrification enzymes concentration as a response to the environmental conditions.

## NETLOGO FEATURES (interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

Our model uses the general NetLogo commands, reports and basic agent sets with the extension matrix

## RELATED MODELS (models in the NetLogo Models Library and elsewhere which are of related interest)

INDISIM-Paracoccus
Manuscript: https://doi.org/10.1016/j.jtbi.2016.05.017
Model: http://modelingcommons.org/browse/one_model/4731#model_tabs_browse_info.

## CREDITS AND REFERENCES (a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)

How to cite the paper: 
Araujo-Granda, P., Gras, A., Ginovart, M. et al. J Ind Microbiol Biotechnol (2019).
Received: 11 February 2019
Accepted: 28 October 2019
First Online: 05 November 2019
DOI: https://doi.org/10.1007/s10295-019-02245-8
Publisher Name: Springer International Publishing
Print ISSN: 1367-5435
Online ISSN: 1476-5535
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="(FA) uNO3-a y uSc (ScHigh)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO3-a">
      <value value="0.034"/>
      <value value="0.068"/>
      <value value="0.27"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uSc">
      <value value="0.065"/>
      <value value="0.13"/>
      <value value="0.52"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(FA) uNO3-a y uSc (ScLow)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO3-a">
      <value value="0.034"/>
      <value value="0.068"/>
      <value value="0.27"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uSc">
      <value value="0.065"/>
      <value value="0.13"/>
      <value value="0.52"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(Fx) uNO3-x y uSc (ScHigh)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO3-x">
      <value value="0.0119"/>
      <value value="0.119"/>
      <value value="1.19"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uSc">
      <value value="0.065"/>
      <value value="0.13"/>
      <value value="0.52"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(Fx) uNO3-x y uSc (ScLow)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO3-x">
      <value value="0.0119"/>
      <value value="0.119"/>
      <value value="1.19"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uSc">
      <value value="0.065"/>
      <value value="0.13"/>
      <value value="0.52"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(Fx) uNO3-x y uNO2- (ScLow)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO2-">
      <value value="0.0062"/>
      <value value="0.062"/>
      <value value="0.62"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uNO3-x">
      <value value="0.0119"/>
      <value value="0.119"/>
      <value value="1.19"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(Fx) uNO3-x y uNO2- (ScHigh)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO2-">
      <value value="0.0062"/>
      <value value="0.062"/>
      <value value="0.62"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uNO3-x">
      <value value="0.0119"/>
      <value value="0.119"/>
      <value value="1.19"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(Fx) uNO3-x y uNO (ScLow)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO">
      <value value="6.2E-6"/>
      <value value="6.2E-4"/>
      <value value="0.62"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uNO3-x">
      <value value="0.0119"/>
      <value value="0.119"/>
      <value value="1.19"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="(Fx) uNO3-x y uNO (ScHigh)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>gbacteria</metric>
    <metric>gnitrate</metric>
    <metric>gno2-</metric>
    <metric>gn2o</metric>
    <metric>gsuccinate</metric>
    <metric>gno</metric>
    <metric>gn2</metric>
    <enumeratedValueSet variable="uNO">
      <value value="6.2E-6"/>
      <value value="6.2E-4"/>
      <value value="0.62"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="uNO3-x">
      <value value="0.0119"/>
      <value valu                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                