;__includes[ "social-model.nls" ] ; All together for BehavioralSpace

;; CREATE WORLD
extensions [gis]

globals [ ;; recall that 'global' variables are accessed by ALL types (patches, turtles, etc.) // they all receive the same value // sliders, choosers, etc. are all globals
  ambit-dataset ;; this stores the project data
  MCSC_R-dataset
  ;; land areas percentages
  conifers quercus shrubland waterfield no-land land-area
  ;; ECOLOGICAL INTERACTIONS
  bwater-conifer bwater-quercus ET-conifer ET-quercus
  ;; global? precipitation
  dry-forests dry-conifers dry-quercus dry-shrubland forest-mass precipitation
  ; counters
  ha-multi-total ha-multi multi-ha multi-per count-multi ha-envir-total ha-envir envir-ha envir-per count-envir ha-maxim-total
  ha-tradi-total ha-tradi tradi-ha tradi-per count-tradi ha-passi-total ha-passi passi-ha count-passi passi-per ha-optim-total
  drought-total n_mfu n_env n_tra n_pas
  ; to speed the program
  conifer-multi quercus-multi conifer-envir quercus-envir
  conifer-tradi quercus-tradi conifer-passi quercus-passi
  ;territory
  ; because "if ticks n mod = 0 [" not working
  drought-list short-conifer long-conifer short-quercus long-quercus rc-bwater re-bwater
]

;; Give an identity to forest owners (Maximizers, etc.)
breed [maximizers maximizer]
breed [optimizers optimizer]
breed [traditionalists traditionalist]
breed [passives passive]
breed [multifunctionalists multifunctionalist]
breed [environmentalists environmentalist]

patches-own [
  popu         ;; species/polygons in vector data
  landcover    ;; types
  age drought bwater managed
  belong-to id-owner
  ;   land-use  management-plan ;; used for extensions
]

turtles-own [territory forestsize]

;;#######################################

to setup
  ca
  set-current-directory "..."

  load-map
  paint-world ;; add color depending on land cover
  count-species ;; keep track of the land cover composition
  base-scenario
  ;add-land-use
  add-owners
  count-ha
  ;distribute-management-plans
  tickers

  file-close
  reset-ticks
end

to load-map
  ;; credit to: https://stackoverflow.com/questions/33437711/extension-exception-1944553-while-applying-raster-netlogo-gis
  set MCSC_R-dataset gis:load-dataset "mcsc_raster.asc" ;; loading the vector data
  resize-world 0 gis:width-of MCSC_R-dataset - 1 0 gis:height-of MCSC_R-dataset - 1
  gis:set-world-envelope-ds gis:envelope-of MCSC_R-dataset
  gis:apply-raster MCSC_R-dataset popu
end

to paint-world ; see Excel "Vector-excel" in folder "GIS"
   ask patches [
    if popu = 1 [ set pcolor 64      ; "pi roig"
      set landcover "conifers" ]
    if popu = 2 [ set pcolor 56      ; "matollar"
      set landcover "shrubland" ]
    if popu = 3 [ set pcolor 62      ; "pinassa"
      set landcover "conifers" ]
    if popu = 6  [ set pcolor 76     ; "pi blanc"
      set landcover "conifers" ]
    if popu = 5 [ set pcolor 52      ; "altres: roure/alzina"
      set landcover "quercus" ]
    if popu = 4 [ set pcolor 88      ; "camp o aigua"ask territory [ ; random 2 = 1, but just in case it wants to be increased // ask n-of (int (0.30 * count turtles)) turtles [die]
      set landcover "waterfield" ]
    if pcolor = black [              ; black = no map
      set landcover "no-land" ]
  ]
end

to base-scenario
   ask patches [
     set age random (150 - 20 + 1) + 20   ;; choose a random number between 20 and 150
     set drought 0
     set managed "No" ]
   ask patches [
      if landcover = "conifers" [
        set bwater (100 - initial-ET-conifer) ]
      if landcover = "quercus" [
        set bwater (100 - initial-ET-quercus) ]
   ]
  set precipitation initial-rain
  set ET-conifer initial-ET-conifer
  set ET-quercus initial-ET-quercus
end

to count-ha
  ; initialize global agentsets of patches by landcover (to speed up the model)
  set conifer-multi patches with [ belong-to = "multifunctionalists" and landcover = "conifers" ]
  set quercus-multi patches with [ belong-to = "multifunctionalists" and landcover = "quercus" ]
  set conifer-envir patches with [ belong-to = "environmentalists" and landcover = "conifers" ]
  set quercus-envir patches with [ belong-to = "environmentalists" and landcover = "quercus" ]
  set conifer-tradi patches with [ belong-to = "traditionalists" and landcover = "conifers" ]
  set quercus-tradi patches with [ belong-to = "traditionalists" and landcover = "quercus" ]
  set conifer-passi patches with [ belong-to = "passives" and landcover = "conifers" ]
  set quercus-passi patches with [ belong-to = "passives" and landcover = "quercus" ]

  set ha-maxim-total ( count patches with [ belong-to = "maximalists" ] )
  set ha-optim-total ( count patches with [ belong-to = "optimizers" ] )
  set ha-multi-total ( count patches with [ belong-to = "multifunctionalists" ] )
  set ha-envir-total ( count patches with [ belong-to = "environmentalists" ] )
  set ha-tradi-total ( count patches with [ belong-to = "traditionalists" ] )
  set ha-passi-total ( count patches with [ belong-to = "passives" ] )

  set drought-total ( count patches with [ drought = 1 ] )
  set n_mfu ( count multifunctionalists )
  set n_mfu ( count environmentalists )
  set n_tra ( count traditionalists )
  set n_pas ( count passives )
end

to count-species
  ;; create a group based on each land to check all is fine (landcover = 100%, see observer in interface)
  set land-area (count patches with [ pcolor != black ])
  set conifers ((count patches with [ landcover = "conifers" ]/ land-area) * 100) ;; percentage of land area
  set quercus ((count patches with [ landcover = "quercus" ]/ land-area) * 100)
  set shrubland ((count patches with [ landcover = "shrubland" ]/ land-area) * 100)
  set waterfield ((count patches with [ landcover = "waterfield" ]/ land-area) * 100)
  set no-land ((count patches with [ landcover = "no-land" ]/ land-area) * 100)
  show conifers + quercus + shrubland + waterfield ;; Last check all OK ;
end

to tickers ; lists set for time finishing at 150 years
   set drought-list (list 20 40 60 80 100 120 140)
   set short-conifer (list 1 15 30 45 60 75 90 105 120 135 150)
   set short-quercus (list 1 35 70 105 140)
   set long-conifer (list 1 50 100 150)
   set long-quercus (list 1 75 150)
   set rc-bwater (list 45 44 43 42 41 40 40 39 38 37 36 35)
   set re-bwater (list 20 19 18 17 16 15)
end

to go
  ; reset-ticks --> THIS WAS THE PROBLEM, IF ON NOT RUNNING
  ;; stop model if conifers and quercus are all brown
  if ( dry-forests = 100 ) [
    stop ]
  ;; commands
  grow-old
  experience-drought
  change-blue-water
  update-landcover
  initiate-management
  update-ha
  ;update-plots // taken out for model efficiency
  tick
end

;; distribute different ages and convert land depending on the age
to grow-old
   ask patches [
   ;; grow 1 every tick
   set age age + 1
   ]
end

to experience-drought

  ask patches [
  if ( age > 100 ) and ( precipitation > 500 ) and ( ( landcover = "conifers" ) or ( landcover = "quercus" ) ) [
      let dice random 6
      if dice = 0 [ set pcolor brown ]
      if dice = 1 [ set landcover "shrubland"
                    set drought 1 ]
      if dice > 1 [ set age 20 ] ; before set age 10
    ]

  if ( age > 100 ) and ( precipitation < 500 ) and ( precipitation > 450 ) and ( ( landcover = "conifers" ) or ( landcover = "quercus" ) ) [
      let dice random 5
      if dice = 0 [ set pcolor brown ]
      if dice = 1 [ set drought 1
                    set landcover "shrubland" ] ; ideally: set landcover = "shrubland"
      if dice > 1 [ set age 20 ]
    ]

  if ( age > 100 ) and ( precipitation < 450 ) and ( ( landcover = "conifers" ) or ( landcover = "quercus" ) ) [
      let dice random 4
      if dice = 0 [ set pcolor brown ]
      if dice = 1 [ set landcover "shrubland"
                    set drought 1 ] ; ideally: set landcover = "shrubland"
      if dice > 1 [ set age 20 ]
    ]
  ]

end

to change-blue-water

  ifelse climate-change
      [ set precipitation ( precipitation - 1.15 )
        set ET-conifer ( ET-conifer + 0.03 )
        set ET-quercus ( ET-quercus + 0.03 ) ] ; amount said to decrease each year due to climate change
      [ ] ; it never changes...

  ask patches with [ landcover = "conifers" and managed = "No" ] [
    set bwater ( 100 - ET-conifer ) ]
  ask patches with [ landcover = "quercus" and managed = "No" ] [
    set bwater ( 100 - ET-quercus ) ]

end

;to add-land-use

;  ask patches [ set land-use 0 ] ; 0 = only forestry

;  ask patches with [pycor < 75] [   ; southern coordinates
;    ifelse agriculture-too?
;      [ set land-use 1] ; 1 = forestry and agriculture
;      [ set land-use 0] ; 0 = forestry
;    ]
;end

to update-landcover

  set bwater-conifer mean [bwater] of patches with [ landcover = "conifers" ]
  set bwater-quercus mean [bwater] of patches with [ landcover = "quercus" ]

  set forest-mass (count patches with [ landcover = "conifers" ] +
                   count patches with [ landcover = "quercus" ] +
                   count patches with [ landcover = "shrubland" ])

  set dry-conifers ( count patches with [ pcolor = brown and landcover = "conifers" ]
    / forest-mass ) * 100 ;/ count patches with [ landcover = "conifers" ] ) * 100
  set dry-quercus ( count patches with [ pcolor = brown and landcover = "quercus" ]
    / forest-mass ) * 100 ;/ count patches with [ landcover = "quercus" ] ) * 100
  set dry-shrubland ( count patches with [ drought = 1 and landcover = "shrubland" ]
    / forest-mass ) * 100
  set dry-forests (( dry-conifers + dry-quercus ) / 2 )
end

; ##############################################################################################################
; SOCIAL SUBMODEL
; Set number of forest owners in the interface

;; Create breeds based on the numbers and percentages in the interface
to add-owners
  ask patches [
    if pcolor != black and pcolor != 88 [ set belong-to nobody ]
  ]

  create-traditionalists (number-owners * (tradi / 100)) [
    set color 15
    set shape "dot"
    set size 5 ]
  ask traditionalists [
    move-to one-of patches with [ belong-to = nobody ]
    pd
    set territory patches in-radius (random 7) with [ belong-to = nobody ]
    ask territory [
      set belong-to "traditionalists"
      set id-owner myself
    ]
    set forestsize count territory
   ]

  create-multifunctionalists (number-owners * (multi / 100)) [
    set color 25
    set shape "dot"
    set size 5 ]
  ask multifunctionalists [
    move-to one-of patches with [ belong-to = nobody ]
    pd
    set territory patches in-radius (random 7) with [ belong-to = nobody ]
    ask territory [
      set belong-to "multifunctionalists"
      set id-owner myself
    ]
    set forestsize count territory
   ]

  create-environmentalists (number-owners * (envir / 100)) [
    set color 45
    set shape "dot"
    set size 5 ]
  ask environmentalists [
    move-to one-of patches with [ belong-to = nobody ]
    pd
    set territory patches in-radius (random 7) with [ belong-to = nobody ]
    ask territory [
      set belong-to "environmentalists"
      set id-owner myself
    ]
    set forestsize count territory
   ]

  create-passives (number-owners * (passi / 100)) [
    set color 125
    set shape "dot"
    set size 5 ]
  ask passives [
    carefully [ move-to one-of patches with [ belong-to = nobody ] ] [ user-message "Please, create world and add owners again" ]
    pd
    set territory patches in-radius (random 7) with [ belong-to = nobody ]
    ask territory [
      set belong-to "passives"
      set id-owner myself
        ]
    set forestsize count territory
   ; if any? patches in-radius 25 with [ belong-to = nobody ]
    ;   [ set territory patches in-radius 20 with [ belong-to = nobody ]
     ;    ask territory [
      ;       set belong-to "passives"
       ;      set id-owner myself ]]
   ]

  if maxim + optim + tradi + passi + multi + envir != 100 [ user-message "Please, make sure forest owners categories sum up to 100%" stop ]

end

;to distribute-management-plans

  ;ask n-of ((num-of-plans / 100) * ha-tradi-total) patches with [belong-to = "traditionalists"] [ set management-plan 1 ]
  ;ask n-of ((num-of-plans / 100) * ha-multi-total) patches with [belong-to = "multifunctionalists"] [ set management-plan 1 ]
  ;ask n-of ((num-of-plans / 100) * ha-envir-total) patches with [belong-to = "environmentalists"] [ set management-plan 1 ]
  ;ask n-of ((num-of-plans / 100) * ha-passi-total) patches with [belong-to = "passives"] [ set management-plan 1 ]

  ; ask n-of ((num-of-plans / 100) * number-owners) patches with [id-owner = (random 1000)] [ set management-plan 1 ]

;end

; ##############################################################################################################
; POLICY SUBMODEL
; Each identity responds differently to the policy design

to initiate-management

  if ( dry-forests = 100 ) [
    stop ]

  ask multifunctionalists [
  ; Multifunctionlists
  ; If below cost or more ES are on = 50 years, otherwise 15 years
    (ifelse Below-cost-coverage? [ ]; [ ask territory [ manage-long-multi ] ]
            Additional-ES-goal? [ manage-long ] ;[ ask territory [ manage-long-multi ] ]
            One-time-payment? [ attrition
                                manage-short ]
            ; else command
            [ manage-short ] )
  ]

  ask environmentalists [
  ; Multifunctionlists
  ; If below cost or more ES are on = 50 years, otherwise 15 years
    (ifelse Below-cost-coverage? [ ]; [ ask territory [ manage-long-multi ] ]
            No-intermediary? [ ]
            ;Additional-ES-goal? [ manage-long ] ; not needed
            One-time-payment? [ attrition
                                manage-long ]
            ; else command
            [ manage-long ] )
  ]

  ask traditionalists [
  ; Follows neighbours
    if any? patches in-radius 5 with [managed = "Yes" and belong-to != "traditionalists"] [
       manage-short ] ; Traditionalists manage if neighbors do
  ]

end

; For speed reasons

; COMPLETE MANAGEMENT: every 15/35 years there is forest thinning and at 50/100 there is forest cutting
to manage-short
  ; conifers are "managed" every 15-20 years / quercus every 35 (Informe DEMORGEST)
  ; old conifers are cut and new are planted (every 50 years), 30% following "Baronia Balanc Hidric"

  if member? ticks short-conifer and landcover = "conifers" [ ;  and managed = "No"
    ;ask territory with [who = myself] [ ; random 2 = 1, but just in case it wants to be increased // ask n-of (int (0.30 * count turtles)) turtles [die]
    ask n-of (int (0.3 * forestsize)) patches with [id-owner = myself] [
    set age (age - age - (50 - ticks))
    set bwater random (46 - 35 + 1) + 35 ; between 0-10% gain
    set drought 0
    set managed "Yes"
    set landcover "conifers"
    set pcolor 67 ]
  ]

  if member? ticks short-quercus and landcover = "quercus" [
    ;ask territory  [ ; random 2 = 1, but just in case it wants to be increased // ask n-of (int (0.30 * count turtles)) turtles [die]
    ask n-of (int (0.3 * forestsize)) patches with [id-owner = myself] [
    set age (age - age - (100 - ticks))
    set bwater random (21 - 15 + 1) + 15 ; between 0-5%
    set drought 0
    set managed "Yes"
    set landcover "quercus"
    set pcolor 67 ]
  ]
end

; CLOSE-TO-NATURE MANAGEMENT (long-terms)
to manage-long

  if member? ticks long-conifer and landcover = "conifers" [
    ask n-of (int (0.15 * forestsize)) patches with [id-owner = myself] [
    set age (age - age - (50 - ticks))
    set bwater random (46 - 35 + 1) + 35 ; between 0-10% gain
    set drought 0
    set managed "Yes"
    set landcover "conifers"
    set pcolor 67 ]
  ]

  if member? ticks long-quercus and landcover = "quercus" [
    ask n-of (int (0.15 * forestsize)) patches with [id-owner = myself] [
    set age (age - age - (100 - ticks))
    set bwater random (21 - 15 + 1) + 15 ; between 0-5%
    set drought 0
    set managed "Yes"
    set landcover "quercus"
    set pcolor 67 ]
  ]
end

to attrition

  if member? ticks short-quercus [
    let dice random 19
      if dice = 0 [ die ] ]

end

; OUTPUT
to update-ha

   set ha-multi ( count patches with [ managed = "Yes" and belong-to = "multifunctionalists" ] )
   set multi-per ( ha-multi / ha-multi-total ) * 100
   set multi-ha ( ha-multi * 0.01 )

   set ha-envir ( count patches with [ managed = "Yes" and belong-to = "environmentalists" ] )
   set envir-per ( ha-envir / ha-envir-total ) * 100
   set envir-ha ( ha-envir * 0.01 )

   set ha-tradi ( count patches with [ managed = "Yes" and belong-to = "traditionalists" ] )
   set tradi-per ( ha-tradi / ha-tradi-total ) * 100
   set tradi-ha ( ha-tradi * 0.01 )

   set ha-passi ( count patches with [ managed = "Yes" and belong-to = "passives" ] )
   set passi-per ( ha-passi / ha-passi-total ) * 100
   set passi-ha ( ha-passi * 0.01 )

end



;; see: https://stackoverflow.com/questions/38619289/storing-recalling-the-value-of-a-variable-in-netlogo
@#$#@#$#@
GRAPHICS-WINDOW
608
10
1080
613
-1
-1
1.8635
1
10
1
1
1
0
1
1
1
0
248
0
318
1
1
1
years
30.0

BUTTON
420
379
501
412
Create world
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
18
96
151
129
initial-rain
initial-rain
0
1000
555.0
-1.15
1
mm/y
HORIZONTAL

TEXTBOX
20
75
170
94
Environment
11
0.0
1

SLIDER
18
204
175
237
initial-ET-conifer
initial-ET-conifer
0
100
65.0
0.03
1
%
HORIZONTAL

TEXTBOX
20
49
137
67
Ecological system
12
52.0
1

TEXTBOX
209
53
333
75
Social system
12
12.0
1

TEXTBOX
20
11
219
33
MODEL PARAMETERS
16
0.0
1

PLOT
1119
211
1420
331
Evolution of blue water
time
%
0.0
200.0
25.0
50.0
false
true
"" ""
PENS
"Quercus" 1.0 1 -13345367 true "" "plot bwater-quercus"
"Conifer" 1.0 1 -11221820 true "" "plot bwater-conifer"

TEXTBOX
1204
15
1354
33
MODEL OUTPUT\n
16
0.0
1

SLIDER
17
248
176
281
initial-ET-quercus
initial-ET-quercus
0
100
85.0
0.03
1
%
HORIZONTAL

TEXTBOX
412
73
562
91
Policy options
11
0.0
1

PLOT
1118
53
1420
193
Percentage of dry areas (ha)
time
%
0.0
200.0
0.0
50.0
true
true
"" ""
PENS
"Quercus" 1.0 0 -6459832 true "" "plot dry-quercus"
"Conifer" 1.0 0 -10402772 true "" "plot dry-conifers"
"Shrubland" 1.0 0 -5207188 true "" "plot dry-shrubland"

TEXTBOX
20
186
170
204
Forest
11
0.0
1

BUTTON
420
443
475
476
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

TEXTBOX
412
49
562
67
Policy system
12
105.0
1

SWITCH
411
135
568
168
One-time-payment?
One-time-payment?
0
1
-1000

SWITCH
411
174
568
207
No-intermediary?
No-intermediary?
1
1
-1000

SWITCH
412
96
568
129
Below-cost-coverage?
Below-cost-coverage?
0
1
-1000

TEXTBOX
213
76
363
94
Forest owners
11
0.0
1

SLIDER
214
199
306
232
maxim
maxim
0
100
0.0
0
1
%
HORIZONTAL

SLIDER
214
244
306
277
optim
optim
0
100
0.0
0
1
%
HORIZONTAL

SLIDER
216
345
308
378
tradi
tradi
0
100
20.0
0
1
%
HORIZONTAL

SLIDER
217
443
309
476
passi
passi
0
100
40.0
0
1
%
HORIZONTAL

SLIDER
215
296
307
329
multi
multi
0
100
20.0
0
1
%
HORIZONTAL

SLIDER
217
395
309
428
envir
envir
0
100
20.0
0
1
%
HORIZONTAL

INPUTBOX
212
102
297
162
number-owners
1000.0
1
0
Number

TEXTBOX
214
179
311
197
% of owner types
11
0.0
1

MONITOR
1217
522
1290
567
Traditionalist
tradi-per
1
1
11

MONITOR
1386
521
1436
566
Passive
passi-per
1
1
11

MONITOR
1119
522
1214
567
Multifunctionalist
multi-per
1
1
11

MONITOR
1293
522
1383
567
Environmentalist
envir-per
1
1
11

TEXTBOX
1120
504
1362
532
Percentage of total ha managed by owner type
11
0.0
1

SWITCH
18
138
152
171
climate-change
climate-change
0
1
-1000

SWITCH
411
212
568
245
Additional-ES-goal?
Additional-ES-goal?
0
1
-1000

PLOT
1120
344
1419
494
Management
time
km2
0.0
200.0
0.0
40.0
true
true
"" ""
PENS
"Multifunctionalists" 1.0 0 -955883 true "" "plot multi-ha"
"Environmentalists" 1.0 0 -1184463 true "" "plot envir-ha"
"Traditionalists" 1.0 0 -2674135 true "" "plot tradi-ha"
"Passives" 1.0 0 -5825686 true "" "plot passi-ha"

SLIDER
23
442
165
475
num-of-plans
num-of-plans
0
100
0.0
0
1
%
HORIZONTAL

TEXTBOX
25
384
175
402
For extensions:
11
0.0
1

TEXTBOX
314
166
374
184
Km2 owned
11
0.0
1

MONITOR
310
187
367
232
Maxim
ha-maxim-total
2
1
11

MONITOR
310
234
367
279
Optim
ha-optim-total
2
1
11

MONITOR
311
284
368
329
Multi
ha-multi-total * 0.01
1
1
11

MONITOR
312
334
369
379
Tradi
ha-tradi-total * 0.01
1
1
11

MONITOR
313
433
370
478
Passi
ha-passi-total * 0.01
1
1
11

MONITOR
312
384
369
429
Envir
ha-envir-total * 0.01
1
1
11

SWITCH
23
406
166
439
agriculture-too?
agriculture-too?
1
1
-1000

TEXTBOX
418
422
568
440
Initiate the model run:
11
0.0
1

TEXTBOX
418
347
568
375
Landcover and owner distribution:
11
0.0
1

@#$#@#$#@
## Socio-ecological systems and payment for environmental systems model

The socio-ecological systems and payment for environmental systems model (SESPESModel) is an ecosystem simulation with autonomous agents (individuals) in a dynamic environment. The aim of the model is to obseve the response of these individuals, forest owners, to an incentive policy. We use the results of ecological models to establish the ecological system, and our main interest resides in observing how forest owners respond under different policy designs. We base the decision-making process of these individuals on behavioral models of forest owners and the design of the policy in a literature review.

### How to use it

#### Create world
To start the simulation, you have to create an environment first. In the documents section you can download the GIS map on your computer and change the origin of the file in the main code (ecosystem submodel). Then, click on 'Create world' on the right. The basic scenario (e.g. initial rain), is based on the characteristics of the area. You can change settings in the section in case you want to perform the model on other conditions.

#### Individuals
The second step is to add the forest owners in the landscape. You can input the number of owners, and the percentages of different owner types. The percentages have to sum up 100, otherwise an error message occurs. The different forest owners categories own a random number of hectarees, which are patches. With the original set-up, a patch is approximatedly 1,8 ha. They own a random number of ha, ranging from 0 to 25.

#### Policy settings
The policy submodel suggests different designs for the incentive policy. When you use the program with the original settings, the optimal design has been established (all switch bottons are "off"). You can change the settings before pressing the 'Go' bottom to observe how this influences the outcomes on the ecological and the social systems.

#### Starting a model run
When environment and individuals are generated, you can start the simulation by clicking on 'Go'. The observer will give you a number. This number shall be 100 and it is the sum of all land types (forest types, water). This is a check-up. 
When the model is running, it is recommended not to change the settings. The changes in the environment take some decades to be observed. These changes are: trees turn brown, indicating forest decay, and the plots evolve to show the situation.

#### Stopping a model run
The model will run forever until the user decides to stops it or the number of 'brown' trees is 100%. Output of the model can be the plots and the number of ha managed by the different stakeholder categories. You can save these data and perform further plottings in R or other statitical software.

### Author

Eulalia Baulenas (2020)
eulalia.baulenas@ifp.uni-freiburg.de

Faculty of Forest and Environmental Policy
University of Freiburg
Germany
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

campsite
false
0
Polygon -7500403 true true 150 11 30 221 270 221
Polygon -16777216 true false 151 90 92 221 212 221
Line -7500403 true 150 30 150 225

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment-optimal" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>ha-multi-total</metric>
    <metric>ha-envir-total</metric>
    <metric>ha-passi-total</metric>
    <metric>ha-tradi-total</metric>
    <metric>ha-multi</metric>
    <metric>ha-envir</metric>
    <metric>ha-tradi</metric>
    <metric>ha-passi</metric>
    <metric>bwater-conifer</metric>
    <metric>bwater-evergreen</metric>
    <metric>dry-conifers</metric>
    <metric>dry-evergreen</metric>
    <metric>dry-schrubland</metric>
    <enumeratedValueSet variable="maxim">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="No-intermediary?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="multi">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="envir">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="optim">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Below-cost-coverage?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="tradi">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-conifer">
      <value value="65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-change">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-evergreen">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-owners">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="One-time-payment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-rain">
      <value value="555"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agriculture-too?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="passi">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-of-plans">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment-active" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>ha-multi-total</metric>
    <metric>ha-envir-total</metric>
    <metric>ha-passi-total</metric>
    <metric>ha-tradi-total</metric>
    <metric>ha-multi</metric>
    <metric>ha-envir</metric>
    <metric>ha-tradi</metric>
    <metric>ha-passi</metric>
    <metric>bwater-conifer</metric>
    <metric>bwater-evergreen</metric>
    <metric>dry-conifers</metric>
    <metric>dry-evergreen</metric>
    <metric>dry-schrubland</metric>
    <enumeratedValueSet variable="maxim">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="No-intermediary?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="multi">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="envir">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="optim">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Below-cost-coverage?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="tradi">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-conifer">
      <value value="65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-change">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-evergreen">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-owners">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="One-time-payment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-rain">
      <value value="555"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agriculture-too?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="passi">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-of-plans">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Run-policy" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>ha-multi-total</metric>
    <metric>ha-envir-total</metric>
    <metric>ha-passi-total</metric>
    <metric>ha-tradi-total</metric>
    <metric>ha-multi</metric>
    <metric>ha-envir</metric>
    <metric>ha-tradi</metric>
    <metric>ha-passi</metric>
    <metric>bwater-conifer</metric>
    <metric>bwater-quercus</metric>
    <metric>dry-conifers</metric>
    <metric>dry-quercus</metric>
    <metric>dry-shrubland</metric>
    <metric>drought-total</metric>
    <metric>n_mfu</metric>
    <metric>n_env</metric>
    <metric>n_tra</metric>
    <metric>n_pas</metric>
    <enumeratedValueSet variable="Additional-ES-goal?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Below-cost-coverage?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="One-time-payment?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-owners">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="passi">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="multi">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="envir">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="tradi">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-conifer">
      <value value="65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-quercus">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-change">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-rain">
      <value value="555"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment-passive" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>ha-multi-total</metric>
    <metric>ha-envir-total</metric>
    <metric>ha-passi-total</metric>
    <metric>ha-tradi-total</metric>
    <metric>ha-multi</metric>
    <metric>ha-envir</metric>
    <metric>ha-tradi</metric>
    <metric>ha-passi</metric>
    <metric>bwater-conifer</metric>
    <metric>bwater-evergreen</metric>
    <metric>dry-conifers</metric>
    <metric>dry-evergreen</metric>
    <metric>dry-schrubland</metric>
    <enumeratedValueSet variable="maxim">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="No-intermediary?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="multi">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="envir">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="optim">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Below-cost-coverage?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="tradi">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-conifer">
      <value value="65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-change">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-ET-evergreen">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-owners">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="One-time-payment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-rain">
      <value value="555"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agriculture-too?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="passi">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-of-plans">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
