;; Metapopulation simulation v1.0 - Jon Bennie 03/08/2020
;; any lines starting with semicolons are comments and are ignored by Netlogo

;; this "globals" command just defines any variables that are used throughout the whole code and are not specified in the graphical interface

;-----------------------------------------------------------

globals [Generations ButterflyPopulation OccupiedHabitats HabitatsOccupied MyList CapacityTotal] ; MR: Added relevant things to here.


;; this section just defines the types of 'agent' or 'object' - called 'turtles' in Netlogo - that can interact in the model
;; here we just tell Netlogo that we have things called butterflies (singular butterfly) and habitats (singular habitats)
breed [butterflies butterfly]
breed [habitats habitat]

;; this defines characteristics of the agents - eg. each habitat has a capacity (size) and occupancy (occupied or not)
;; by default all 'turtles' already have some characteristics - eg. shape, colour, size etc...
habitats-own [capacity occupancy]

;; this section (from 'to setup' up to 'end') is called every time you click the setup button on the interface
;; it clears the screen, creates 10 habitat [patches], and specifies the colour, shape, capacity and location in space of each one
;; the capacity of each patch is between 1 and 5 (random 4 creates a random number between 0 and 4, then it adds one)
;; initially all patches are coloured white and occupied
;; the location is set by x and y coordinates, which are randomised
;; it also resets the count of generations and refreshes the display



to setup
  clear-turtles
  set MyList n-values Number_of_habitats [random 100000000000 + 1]  ; MR: This is where I have changed the random number to that big number rather than 10, also creating a list.
  set CapacityTotal sum(mylist)
  create-habitats Number_of_habitats [
       set capacity one-of MyList
          set mylist remove(capacity) MyList
       set shape "flower"
       set color yellow
       set size ((capacity) * (total_size_of_all_habitats / CapacityTotal))
       setxy random-xcor random-ycor
       set occupancy 1
  ]
  set Generations 0
display
end

;----------------------------------------------------

;; this section is called when you click the 'disperse button, or in continuous mode each generation
;; it update the number of generations, and 'asks' each habitat patch that is occupied to hatch new butterflies
;; the number of butterflies is the capacity (1-5) times 10
;; turns out netlogo has an inbuilt defined shape called "butterfly", so I used that and coloured them blue
;; then it calls the "move" section (details below) 100 times before killing them off

to disperse
  set Generations Generations + 1
  ask habitats with [occupancy = 1] [
    hatch-butterflies size * random 9 + 1
    [set size 1.5
    set shape "butterfly"
    set color blue
    ]
  ]
  repeat 100 [move]
  set OccupiedHabitats count habitats with [occupancy = 1]
  set HabitatsOccupied habitats with [occupancy = 1]
  ;set ButterflyPopulation ((sum [size] of HabitatsOccupied) + (count butterflies))
  set ButterflyPopulation count butterflies
  ask butterflies [die]
end

;; this section defines how butterflies move in each time step.
;; each butterfly turns right a random angle, then left a random angle
;; then moves forward one unit
;; then each butterfly 'asks' any habitat that it is on to become occupied (and turn white if it isn't already)

to move
    ask butterflies [
    rt random 18
    lt random 10
    fd 1
    ask habitats-here [
      set occupancy 1
      set color yellow
    ]
  ]
end

;; this section is triggered when you click "extinction" or in continuous mode
;; it 'asks' each occupied habitat patch to think of a random number between 0 and 100
;; if this number is lower than (100 / capacity) then the habitat patch goes extinct (occupancy set to zero) and turns blue
;; so patch size 1 goes extinct 100% of the time (every time step) and patch size 5 goes extinct 20% of the time (on average every 5 timesteps)

to extinction
  ask habitats with [occupancy = 1] [
    if random 100 < 100 / size [
      set occupancy 0
      set color brown
    ]
  ]
end

;; this section is triggered when you click the "continuous" button
;; it just checks to see if all habitat is empty - if so then it stops the simulation
;; otherwise it keeps repeating the dispersal and extinction procedure and updates the landscape forever...
;; it might be useful to put an on/off switch here somewhere if you can figure out how to do it!

to continuous
  loop [
    disperse
    extinction
    update-plots
    export-plot "Plot to show the population of butterflies over time" "plot.csv" ; MR: This is the part of the code that, as expected, exports the results of the plot. It overwrites the file 'plots.csv' each time.
    export-world "results.csv" ; MR: This is the section of code that if uncommented doesn't terminate the model once the population of the butterflies hits zero (I believe the line above to do the same).
    if all? habitats [occupancy = 0] [stop]
    ]
  reset-ticks
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
1063
864
-1
-1
13.0
1
10
1
1
1
0
0
0
1
-32
32
-32
32
0
0
1
ticks
30.0

BUTTON
103
43
187
77
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6
85
90
120
NIL
disperse
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
103
86
187
120
extinction
extinction
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
56
136
140
170
continuous
continuous
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1119
100
1315
221
Generations
Generations
17
1
30

SLIDER
0
209
205
242
Number_of_habitats
Number_of_habitats
0
15
5.0
1
1
Habitats
HORIZONTAL

SLIDER
3
252
208
285
Total_size_of_all_habitats
Total_size_of_all_habitats
10
50
50.0
2
1
NIL
HORIZONTAL

PLOT
1117
239
1919
793
Plot to show the population of butterflies over time
Generations
Pop.
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"Butterflies" 1.0 0 -14730904 true "" "plot ButterflyPopulation\nset-plot-pen-mode 0"
"Habitats" 1.0 0 -2674135 true "" "plot OccupiedHabitats\nset-plot-pen-mode 1"

MONITOR
1672
120
1957
221
Dispersed Butterflies:
count butterflies
17
1
25

MONITOR
1327
120
1659
221
Habitats Occupied:
count HabitatsOccupied
17
1
25

BUTTON
7
43
91
78
clear-all-plots
clear-all-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This is a model showing how a metapopulation system works. It shows how smaller habitat patches interact to allow a metapopulation to stay active. 

## HOW IT WORKS

When the 'setup' button is pressed, several yellow flowers will appear within the Netlogo 'World'. These represent habitat patches for the butterflies. You can alter the number of habitats and the total size of all the habitats combined using the toggle bars in the interface section. 

The simulation is started when the 'continuous' button is pressed. This hatches a random number of butterflies, calculated using the equation (size of habitat)*(random no. between 1 and 10). The details to how the butterflies disperse are in the disperse function in the code tab. 

The move function in the code tab shows how the butterflies move during one generation of a simulation. It 'asks' the butterflies to move to the left a certain number of times (a random number between 1 and 18), and then right in the same way (a random number between 1 and 10). The reason for the difference in numbers for left and right is so generally speaking the butterflies will go in a circular motion, and therefore more chance of hitting multiple other habitat patches. The move function, once the butterflies have moved left and right a certain number of times, will ask the butterflies to move one space forward. Once all of this has been done, still within the move function, the code as the butterflies whether they have come into contact with another habitat patch. If so, for clarity, the flower is changed from brown to yellow and the value relating to the occupancy is changed from 0 to 1.

The extinction function is very important, it tells the interface when to terminate the simulation. It basically asks the interface section if there are any habitats with occupancy equal to the value of 1. If not, then the simulation is terminated. There is an equation used to help determine whether or not all the habitats are unoccupied displayed in the code. 

Finally, the continuous function is used to run the simulation. You will notice that other functions are present in this loop. This is so that the 'disperse' and 'extinction' functions are run during the simulation. The continuous loop is called when one presses the continuous button in the interface tab. The continuous loop also includes lines of code that export data to a CSV value, which can be very helpful for subsequent analysis of the data produced within a simulation.

It is worth noting that the higher the number of generations is before the simulation is terminated, the better the metapopulation is at preventing a species, such as butterflies, from going extinct. 

## HOW TO USE IT

The clear-all-plots button as suggested clears the plots from the previous simulation. This will need to be pressed if a new graph is wanted to be produced when a new simulation is run. 

The setup button is used to set up the Netlogo world, and randomly spawn the desired number of selected habitats as explained below. 

The disperse button, which isn't really needed, as suggested runs the disperse section of the code. This hatches a specific number of habitats and disperses them into the Netlogo world. 

The extinction button, again like the disperse button is rarely used but runs the extinction section of the code. As stated in the 'How it works' section, asks the Netlogo world if there is at least 1 habitat occupied, if not then it terminates the simulation. 

The continuous function starts the simulation, where butterflies are dispersed, habitats are asked if they are occupied, and terminates the simulation when there are no occupied habitats. 

The 'Number_of_habitats' and 'Total_size_of_all_habitats' sliders are used as suggested to alter the total number of habitats and the total size of all the habitats created in the Netlogo world when the setup button is pressed. 


## THINGS TO NOTICE

Whether or not a habitat (or flower) changes from brown to yellow when a butterfly comes into contact with it. It is not certain that a habitats occupancy will change from 0 to 1 when a butterfly comes into contact as it is based on a probabilistic equation.

Note whether there is a correlation between the number of butterflies alive (or currently dispersing) and the number of habitats occupied. 

Why the model runs faster over time. This is due to the decreasing number of butterflies dispersing. It will appear that the simulation is running quicker when there are fewer butterflies dispersing since it will be quicker for one generation to finish if there are indeed fewer butterflies being dispersed.

Is there a general pattern to the graph produced?

## THINGS TO TRY

1) Try altering the number of habitats created, and try different values of the total size of all habitats. Sometimes the simulation will last hundreds of generations, and some less than 20. Why is this the case?

2) Explore different formations to a metapopulation. Why are some better? Why are some worse? Which formations are better and which are worse?

## EXTENDING THE MODEL

1) One thing that was discussed intensively in the creation of this model is trying to overlay plots. Resetting the model so that a new formation of the metapopulation is visible in the Netlogo world, but making it so a new plot is created for the new simulation. This would help data collection as a new column of data will be created each time in the exported CSV file. 

2) Importing a GIS shapefile was also talked about in how to extend this model. Importing a shapefile and converting these to habitats in Netlogo would make this model a very powerful tool in predicting how real-life metapopulation will develop and change over time. 

## NETLOGO FEATURES

Both the butterflies and the habitats (or flowers) are referred to as turtles in Netlogo, it is how you code them to interact with other 'turtles' is important in how to define them. 

## RELATED MODELS

N/a

## CREDITS AND REFERENCES

Copyright 2020 Jon Bennie and Matthew Reynolds 

College of Environmental sciences

University of Exeter
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
