; IADS Missile Battery Simulation
; Based on Han, Lunday & Robbins (2016)

globals [
  city-size          ; size of city patches
  initial-total-value  ; store initial total value for comparison
  total-intercepted-missiles  ; all intercepted missiles to count air defense effeciency
  attack-complete?   ; stop flag for BehaviorSpace
]

patches-own [
  coverage-intensity  ; for visualizing overlapping coverage
  damage-level        ; for visualizing city damage
]

breed [cities city]
breed [batteries battery]
breed [missiles missile]

cities-own [
  value       ; city value/population
  protected?  ; whether city is protected by battery
]

batteries-own [
  remaining-interceptors  ; number of interceptors left
]

missiles-own [
  target        ; targeted city
  intercepted?  ; whether missile has been intercepted
]

to setup
  clear-all
  setup-globals
  setup-cities
  setup-batteries
  set initial-total-value sum [value] of cities
  set total-intercepted-missiles 0
  set attack-complete? false
  update-protection-status
  update-visualizations
  reset-ticks
end

to setup-globals
  set city-size 1
  ask patches [
    set coverage-intensity 0
    set damage-level 0
  ]
end

to setup-cities
  ; Create cities with random locations and values
  create-cities num-cities [
    setxy random-xcor random-ycor
    set size city-size
    set shape "circle"
    set color blue
    set value 10 + random 90 ; Random value between 10-100
    set protected? false
    set label (value)
  ]
end

to setup-batteries
  ask batteries [die]  ; Clear any existing batteries

  (ifelse
    defender-strategy = "highest-value" [
      ; Create a list of cities sorted by decreasing value
      let sorted-cities sort-on [(- value)] cities
      create-batteries num-batteries [
        ; Place each battery at one of the highest-value cities
        let target-city item (who mod length sorted-cities) sorted-cities
        move-to target-city
        initialize-battery
      ]
    ]

    defender-strategy = "max-coverage" [
      create-batteries num-batteries [
        ; For each battery, find the position that maximizes the total value of covered cities
        let best-position max-one-of patches [
          ; Sum the values of all cities within range of this position
          sum [value] of cities in-radius battery-range
        ]
        move-to best-position
        initialize-battery

        ; If batteries overlap too much, adjust position while maintaining good coverage
        while [any? other batteries in-radius (battery-range / 3)] [
          let alternative-position max-one-of patches with [
            not any? batteries in-radius (battery-range / 3)
          ][
            sum [value] of cities in-radius battery-range
          ]
          if alternative-position != nobody [move-to alternative-position]
        ]
      ]
    ]

    defender-strategy = "choke-points" [
      ; First identify strategic points that cover multiple cities
      let strategic-points patches with [
        count cities in-radius battery-range >= 2  ; Points covering at least 2 cities
      ]

      create-batteries num-batteries [
        ifelse any? strategic-points [
          ; Find the strategic point that protects the highest total city value
          let best-point max-one-of strategic-points [
            sum [value] of cities in-radius battery-range
          ]
          move-to best-point
          ; Remove this point and surrounding area from consideration
          ask strategic-points in-radius (battery-range / 2) [
            set strategic-points strategic-points with [self != myself]
          ]
        ][
          ; If no good strategic points left, fall back to highest-value strategy
          let target-city max-one-of cities [value]
          move-to target-city
        ]
        initialize-battery
      ]
    ]

 defender-strategy = "distributed" [
  create-batteries num-batteries [
    ; First try to cover uncovered cities
    let uncovered-cities cities with [
      not any? batteries in-radius battery-range
    ]

    ifelse any? uncovered-cities [
      ; Instead of checking each uncovered city, select from top valuable ones
      let target-city max-one-of (up-to-n-of 5 uncovered-cities) [value]
      move-to target-city
    ][
      ; Instead of checking all patches, only check near cities
      let potential-positions patches with [
        any? cities in-radius battery-range and
        not any? batteries in-radius (battery-range / 2)
      ]
      ifelse any? potential-positions [
        let target-position max-one-of (up-to-n-of 10 potential-positions) [
          count cities in-radius battery-range with [
            count batteries in-radius battery-range < 2
          ]
        ]
        move-to target-position
      ][
        ; If no good positions found, just move to highest value city
        move-to max-one-of cities [value]
      ]
    ]
    initialize-battery
  ]
]

  defender-strategy = "mixed" [
      ; Split total batteries among strategies
      let remaining-batteries num-batteries
      let base-strategies ["highest-value" "max-coverage" "choke-points" "distributed"]

      foreach base-strategies [strategy ->
        if remaining-batteries > 0 [
          ; Allocate some batteries to this strategy
          let strategy-batteries ceiling (remaining-batteries / (length base-strategies))
          set remaining-batteries remaining-batteries - strategy-batteries

          if strategy = "highest-value" [
            let sorted-cities sort-on [(- value)] cities
            create-batteries strategy-batteries [
              let target-city item (who mod length sorted-cities) sorted-cities
              move-to target-city
              initialize-battery
            ]
          ]

          if strategy = "max-coverage" [
            create-batteries strategy-batteries [
              let best-position max-one-of patches [
                sum [value] of cities in-radius battery-range
              ]
              move-to best-position
              initialize-battery

              while [any? other batteries in-radius (battery-range / 3)] [
                let alternative-position max-one-of patches with [
                  not any? batteries in-radius (battery-range / 3)
                ][
                  sum [value] of cities in-radius battery-range
                ]
                if alternative-position != nobody [move-to alternative-position]
              ]
            ]
          ]

          if strategy = "choke-points" [
            let strategic-points patches with [
              count cities in-radius battery-range >= 2
            ]

            create-batteries strategy-batteries [
              ifelse any? strategic-points [
                let best-point max-one-of strategic-points [
                  sum [value] of cities in-radius battery-range
                ]
                move-to best-point
                ask strategic-points in-radius (battery-range / 2) [
                  set strategic-points strategic-points with [self != myself]
                ]
              ][
                let target-city max-one-of cities [value]
                move-to target-city
              ]
              initialize-battery
            ]
          ]

          if strategy = "distributed" [
            create-batteries strategy-batteries [
              let uncovered-cities cities with [
                not any? batteries in-radius battery-range
              ]

              ifelse any? uncovered-cities [
                let target-city max-one-of uncovered-cities [value]
                move-to target-city
              ][
                let target-position max-one-of patches [
                  count cities in-radius battery-range with [
                    count batteries in-radius battery-range < 2
                  ]
                ]
                move-to target-position
              ]
              initialize-battery
            ]
          ]
        ]
      ]
    ]
  )
end

; Helper procedure to set up individual battery properties
to initialize-battery
  set shape "triangle"
  set color red
  set size 2
  set remaining-interceptors num-interceptors
end

to update-protection-status
  ; Reset protection status for all cities
  ask cities [
    set protected? false
  ]

  ; Mark cities as protected if they are within range of any battery with interceptors
  ask cities [
    let city-batteries batteries in-radius battery-range
    if any? city-batteries [
      if sum [remaining-interceptors] of city-batteries > 0 [
        set protected? true
        set color green
      ]
    ]
  ]
end

to go
if not attack-complete? [    ; Only launch attack if not already done
    launch-attack
    while [any? missiles] [
      move-missiles
      intercept-missiles
      destroy-cities
      update-protection-status
      update-visualizations
      update-plots1
      tick
    ]
    set attack-complete? true  ; Set flag when attack is done
  ]
end

to launch-attack
 (ifelse
   attacker-strategy = "highest-value" [
     ; Target the most valuable cities first
     let high-value-targets sort-on [(- value)] cities
     let missiles-remaining num-missiles

     foreach high-value-targets [target1 ->
       ; Stop if we run out of missiles
       if missiles-remaining > 0 [
         create-missiles 1 [
           setup-missile target1
           set missiles-remaining missiles-remaining - 1
         ]
       ]
     ]
   ]

   attacker-strategy = "spread-attack" [
     ; Distribute missiles evenly across as many cities as possible
     let num-targets min list num-missiles count cities
     let target-cities n-of num-targets cities

     foreach sort target-cities [target1 ->
       create-missiles 1 [
         setup-missile target1
       ]
     ]
   ]

   attacker-strategy = "minimal-breach" [
     ; Find a valuable target and concentrate enough missiles to overwhelm its defenses
     let target1 max-one-of cities [
       value * (1 + count batteries in-radius battery-range)  ; Consider both value and defense
     ]

     if target1 != nobody [
       ; Calculate missiles needed to overwhelm defenses
       let defending-batteries batteries with [distance target1 <= battery-range]
       let total-interceptors sum [remaining-interceptors] of defending-batteries
       let missiles-needed min list (total-interceptors + 1) num-missiles

       ; Launch concentrated attack
       repeat missiles-needed [
         create-missiles 1 [
           setup-missile target1
         ]
       ]
     ]
   ]

 attacker-strategy = "concentrated" [
 let primary-target max-one-of cities [value]

 if primary-target != nobody [
   let nearby-cities cities with [
     distance primary-target <= battery-range / 2 and self != primary-target
   ]

   ; Create list of all potential targets
   let target-list (list primary-target)
   if any? nearby-cities [
     set target-list sentence target-list (sort-on [(- value)] nearby-cities)
   ]

   ; Distribute missiles among targets
   let missiles-per-target ceiling (num-missiles / length target-list)
   foreach target-list [target-city ->
     let missiles-to-create min list missiles-per-target (num-missiles - count missiles)
     repeat missiles-to-create [
       create-missiles 1 [
         setup-missile target-city
       ]
     ]
   ]
 ]
]

attacker-strategy = "mixed" [
      let strategies ["highest-value" "spread-attack" "minimal-breach" "concentrated"]
      let remaining-missiles num-missiles

      ; Randomly assign missiles to different strategies
      while [remaining-missiles > 0] [
        let current-strategy one-of strategies
        let missiles-for-strategy 1 + random (min list remaining-missiles (num-missiles / 2))
        set remaining-missiles remaining-missiles - missiles-for-strategy

        ; Apply the chosen strategy for this subset of missiles
        if current-strategy = "highest-value" [
          let high-value-targets sort-on [(- value)] cities
          foreach high-value-targets [target1 ->
            if missiles-for-strategy > 0 [
              create-missiles 1 [
                setup-missile target1
                set missiles-for-strategy missiles-for-strategy - 1
              ]
            ]
          ]
        ]

        if current-strategy = "spread-attack" [
          let num-targets min list missiles-for-strategy count cities
          let target-cities n-of num-targets cities
          foreach sort target-cities [target1 ->
            if missiles-for-strategy > 0 [
              create-missiles 1 [
                setup-missile target1
                set missiles-for-strategy missiles-for-strategy - 1
              ]
            ]
          ]
        ]

        if current-strategy = "minimal-breach" [
          let target1 max-one-of cities [
            value * (1 + count batteries in-radius battery-range)
          ]
          if target1 != nobody [
            let defending-batteries batteries with [distance target1 <= battery-range]
            let total-interceptors sum [remaining-interceptors] of defending-batteries
            let missiles-needed min list (min list (total-interceptors + 1) missiles-for-strategy) missiles-for-strategy
            repeat missiles-needed [
              create-missiles 1 [
                setup-missile target1
              ]
            ]
            set missiles-for-strategy missiles-for-strategy - missiles-needed
          ]
        ]

        if current-strategy = "concentrated" [
          let primary-target max-one-of cities [value]
          if primary-target != nobody [
            let nearby-cities cities with [
              distance primary-target <= battery-range / 2 and self != primary-target
            ]
            let target-list (list primary-target)
            if any? nearby-cities [
              set target-list sentence target-list (sort-on [(- value)] nearby-cities)
            ]
            let missiles-per-target ceiling (missiles-for-strategy / length target-list)
            foreach target-list [target-city ->
              let missiles-to-create min list missiles-per-target missiles-for-strategy
              repeat missiles-to-create [
                create-missiles 1 [
                  setup-missile target-city
                ]
              ]
              set missiles-for-strategy missiles-for-strategy - missiles-to-create
            ]
          ]
        ]
      ]
    ]
    )
end

; Helper procedure for missile setup
to setup-missile [target-city]
  setxy random-xcor max-pycor
  set target target-city
  ifelse random 2 = 0
  [ setxy random-xcor max-pycor ]  ; top of screen
  [ setxy max-pxcor random-ycor ]  ; right side
  set target target-city
  set color orange
  set size 1
  set shape "arrow"
  set intercepted? false
  set xcor xcor + random 10 - 5  ; Add some randomness to launch position
end

to move-missiles
  ask missiles [
    face target
    forward 1
    if show-damage? [
      ask patches in-radius 0.5 [
        set pcolor orange - 2
      ]
    ]
  ]
end

to intercept-missiles
  ask batteries [
    let nearby-missiles missiles in-radius battery-range
    foreach sort nearby-missiles [ m ->
      if remaining-interceptors > 0 and not [intercepted?] of m [
        if random-float 1 < intercept-prob [
          ask m [
            set intercepted? true
            set total-intercepted-missiles total-intercepted-missiles + 1  ; Track successful interceptions
            ; Create explosion effect
            ask patches in-radius 1 [
              set pcolor yellow
            ]
            die
          ]
        ]
        set remaining-interceptors remaining-interceptors - 1
      ]
    ]
  ]
end

to destroy-cities
  ask missiles [
    if distance target < 1 [  ; target is single turtle not agentset
      ask target [
        set color red
        set protected? false
        set damage-level damage-level + 1
        ask patches in-radius (city-size * 2) [
          set pcolor red
          set damage-level damage-level + 1
        ]
      ]
      die
    ]
  ]
end

to update-visualizations
  ; Clear previous visualizations
  ask patches [
    set coverage-intensity 0
    if not show-damage? [set pcolor black]
  ]

  ; Update battery coverage visualization with smoother gradients
  if show-coverage? [
    ask batteries [
      let curr-battery self
      ask patches in-radius battery-range [
        let dist distance curr-battery
        let intensity (1 - (dist / battery-range)) * 2
        set coverage-intensity coverage-intensity + intensity
        set pcolor scale-color blue coverage-intensity 2 0
      ]
    ]
  ]

  ; Update damage visualization with smoother gradients
  if show-damage? [
    ask cities [
      ifelse protected? [
        set color green
      ][
        ifelse damage-level > 0 [
          set color scale-color red damage-level 5 0
          set label (word "V:" value " D:" damage-level)
        ][
          set color blue
        ]
      ]
      ; Show damage radius with smooth gradient
      if damage-level > 0 [
        ask patches in-radius (city-size * 3) [
          let dist distance myself
          let dmg-intensity (1 - (dist / (city-size * 3))) * damage-level
          if dmg-intensity > 0 [
            set pcolor scale-color red dmg-intensity 5 0
          ]
        ]
      ]
    ]
  ]

  ; Draw range indicators for batteries with smooth edges
  ask batteries [
    ; Display remaining interceptors
    set label (word remaining-interceptors " IM")
    set label-color white
  ]

  ; Update display for missiles and effects
  ask missiles [
    set color orange
  ]
end

; Basic value reporters
to-report total-city-value
  report sum [ifelse-value (damage-level = 0) [value] [0]] of cities
end

to-report protected-value
  report sum [ifelse-value (protected?) [value] [0]] of cities
end

to-report protection-percentage
  ifelse total-city-value > 0
    [ report (protected-value / total-city-value) * 100 ]
    [ report 0 ]
end
; Strategic effectiveness reporters
to-report city-value-survival-ratio
  ifelse initial-total-value > 0
    [ report precision ((total-city-value / initial-total-value) * 100) 2 ]
    [ report 0 ]
end

to-report defense-efficiency
  ; Measures how efficiently interceptors are being used
  let total-initial-interceptors (num-batteries * num-interceptors)
  let used-interceptors total-initial-interceptors - sum [remaining-interceptors] of batteries
  ifelse used-interceptors > 0
    [ report precision ((total-intercepted-missiles / used-interceptors) * 100) 2 ]
    [ report 0 ]
end

to-report defense-coverage-ratio
  ; Measures what percentage of total city value is under SAM coverage
  let total-value sum [value] of cities
  let covered-value sum [value] of cities with [any? batteries in-radius battery-range]
  ifelse total-value > 0
    [  report precision  ( (covered-value / total-value) * 100) 2 ]
    [ report 0 ]
end

to-report value-loss-percentage
  ; Measures percentage of total value lost to successful attacks
  ifelse initial-total-value > 0
    [ report ((initial-total-value - total-city-value) / initial-total-value) * 100 ]
    [ report 0 ]
end

to-report intercept-success-rate
  ; Measures percentage of missiles successfully intercepted
  let total-launched count missiles + count missiles with [intercepted?]
  ifelse total-launched > 0
    [ report (count missiles with [intercepted?] / total-launched) * 100 ]
    [ report 0 ]
end

to-report defense-saturation
  ; Measures how close batteries are to being overwhelmed
  let avg-interceptors mean [remaining-interceptors] of batteries
  report (avg-interceptors / 20) * 100  ; 20 is initial interceptor count
end

to-report attack-concentration
  ; Measures how concentrated the attack is
  let targeted-cities length remove-duplicates [target] of missiles
  ifelse count missiles > 0
    [ report (targeted-cities / count missiles) * 100 ]
    [ report 0 ]
end

to-report high-value-targeting-ratio
  ; Measures focus on high-value targets
  let top-value max [value] of cities
  let targeted-value mean [value] of cities with [any? missiles with [target = myself]]
  ifelse top-value > 0
    [ report (targeted-value / top-value) * 100 ]
    [ report 0 ]
end

to update-plots1
  set-current-plot "Total Cities Value"
  set-current-plot-pen "Total"
  plot total-city-value
  set-current-plot-pen "Protected"
  plot protected-value

  set-current-plot "Protection Metrics"
  set-current-plot-pen "Protection %"
  plot protection-percentage
  set-current-plot-pen "Value Loss %"
  plot value-loss-percentage

  set-current-plot "Battle Status"
  set-current-plot-pen "Active Missiles"
  plot count missiles
  set-current-plot-pen "Interceptors Left"
  plot sum [remaining-interceptors] of batteries
  set-current-plot-pen "Success Rate"
  plot intercept-success-rate
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
647
448
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
11
20
74
53
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
83
20
146
53
go
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
227
131
261
num-batteries
num-batteries
1
10
10.0
1
1
NIL
HORIZONTAL

SLIDER
10
194
131
228
num-missiles
num-missiles
1
100
100.0
1
1
NIL
HORIZONTAL

SLIDER
10
379
131
413
intercept-prob
intercept-prob
0
1
0.6
0.1
1
NIL
HORIZONTAL

SWITCH
10
423
134
457
show-coverage?
show-coverage?
0
1
-1000

SWITCH
9
461
134
495
show-damage?
show-damage?
1
1
-1000

MONITOR
662
10
770
55
Total Cities Value
total-city-value
0
1
11

CHOOSER
13
61
124
106
attacker-strategy
attacker-strategy
"highest-value" "spread-attack" "minimal-breach" "concentrated" "mixed"
1

CHOOSER
13
108
125
153
defender-strategy
defender-strategy
"highest-value" "max-coverage" "choke-points" "distributed" "mixed"
4

PLOT
663
185
926
306
Total Cities Value
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Total" 1.0 0 -16777216 true "" ""
"Protected" 1.0 0 -15040220 true "" ""

PLOT
663
64
926
185
Protection Metrics
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Protection %" 1.0 0 -15040220 true "" ""
"Value Loss %" 1.0 0 -5298144 true "" ""

PLOT
664
307
926
432
Battle Status
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Active Missiles" 1.0 0 -5298144 true "" ""
"Success Rate" 1.0 0 -15040220 true "" ""
"Interceptors Left" 1.0 0 -7500403 true "" ""

SLIDER
10
265
130
299
num-interceptors
num-interceptors
1
20
10.0
1
1
NIL
HORIZONTAL

SLIDER
10
303
130
337
battery-range
battery-range
1
20
10.0
1
1
NIL
HORIZONTAL

SLIDER
10
159
131
193
num-cities
num-cities
10
100
100.0
10
1
NIL
HORIZONTAL

MONITOR
773
10
863
55
Survival Ratio
city-value-survival-ratio
3
1
11

MONITOR
865
9
969
54
NIL
defense-efficiency
3
1
11

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="IADS Strategies" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>attack-complete?</exitCondition>
    <metric>city-value-survival-ratio</metric>
    <metric>defense-efficiency</metric>
    <enumeratedValueSet variable="attacker-strategy">
      <value value="&quot;highest-value&quot;"/>
      <value value="&quot;spread-attack&quot;"/>
      <value value="&quot;minimal-breach&quot;"/>
      <value value="&quot;concentrated&quot;"/>
      <value value="&quot;mixed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defender-strategy">
      <value value="&quot;highest-value&quot;"/>
      <value value="&quot;max-coverage&quot;"/>
      <value value="&quot;choke-points&quot;"/>
      <value value="&quot;distributed&quot;"/>
      <value value="&quot;mixed&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="IADS parameters" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>attack-complete?</exitCondition>
    <metric>city-value-survival-ratio</metric>
    <metric>defense-efficiency</metric>
    <enumeratedValueSet variable="num-batteries">
      <value value="2"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-interceptors">
      <value value="5"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="battery-range">
      <value value="5"/>
      <value value="10"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attacker-strategy">
      <value value="&quot;highest-value&quot;"/>
      <value value="&quot;spread-attack&quot;"/>
      <value value="&quot;minimal-breach&quot;"/>
      <value value="&quot;concentrated&quot;"/>
      <value value="&quot;mixed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defender-strategy">
      <value value="&quot;highest-value&quot;"/>
      <value value="&quot;max-coverage&quot;"/>
      <value value="&quot;choke-points&quot;"/>
      <value value="&quot;distributed&quot;"/>
      <value value="&quot;mixed&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
