;;____________________________
;
;;   SETUP AND HOUSEKEEPING
;;____________________________

breed [ parties party ]

globals [
  cycle
  total-votes
  max-voteshare            ; largest vote share on any patch
  mean-voterx              ; mean voter x-coordinate
  mean-votery              ; mean voter y-coordinate
  mean-eccentricity        ; mean Euclidean distance of parties from (mean-voterx, mean-votery)
  largest-party            ; "who" of largest party
  voter-misery             ; mean quadratic Euclidean voter distance from closest party
  enp                      ; effective number of parties = 1/(sum of squared party vote shares)
  party-number             ; party number
  
  tickcycle                ; with aggregator runs: records cycle for the calculate-party-variance procedure
  position-list            ; with aggregator runs: list of the x & y coordinates of the party positions
  old-position-list        ; with aggregator runs: list of the x & y coordinates of the party positions from the previous tick
  tot-var                  ; with aggregator runs: total variance for the x & y coordinates between position-list & old-position-list
  var-stop                 ; with aggregator runs: stops program when 1
]
  
parties-own [
  mysize                   ; current party size
  old-mysize               ; party's size previous tick
  my-rule                  ; party's decision rule
  eccentricity             ; party's Euclidean distance from (mean-voterx, mean-votery)
]
  
patches-own [
  votes                    ; number of voters on patch
  vote-share               ; proportion of total voters on patch
  closest-party            ; party with smallest Euclidean distance from patch
]

to setup
  clear-all
  create-parties n-parties
  set party-number n-values 12 [?]
  foreach party-number [
    if ?1 >= n-parties [set party-number replace-item ?1 party-number 0]
  ]
  set position-list n-values 24 [0]
  set tickcycle 0
  set tot-var 10
  set var-stop 0
  
  ask parties [set heading random-float 360 jump random-float 30]
      ; create n parties and set their positions as a random draw from a bivariate normal with mean 0 and std. dev. 10
  ask parties [set color lime set size 1.5]
  ask parties [update-rule] 
  ask parties [set old-mysize 0]
      ; ask each party to display itself as large and lime green, and to read its decision rule from the interface
          
  ask patches [ 
      let x1 (pxcor - x-mean1) / sd-1
      let y1 (pycor - y-mean1) / sd-1      
      set votes votes1 * exp (-0.5 * ( x1 ^ 2 + y1 ^ 2)) / (2 * pi * sd-1 ^ 2)
        ;; votes1, x_mean1, y_mean1, sd_1 = votes, mean and standard deviation of subpopulation 1, read from interface
        ;; each patch's votes arising from subpopulation 1 =  votes1 * bivariate normal density with mean1, sd_1, rho = 0
          
      let x2 (pxcor - x-mean2) / sd-2
      let y2 (pycor - y-mean2) / sd-2      
      set votes votes + votes2 * exp (-0.5 * ( x2 ^ 2 + y2 ^ 2)) / (2 * pi * sd-2 ^ 2)
        ;; add votes to each patch from subpopulation 2, calculated as above
          
      let x3 (pxcor - x-mean3) / sd-3
      let y3 (pycor - y-mean3) / sd-3      
      set votes votes + votes3 * exp (-0.5 * ( x3 ^ 2 + y3 ^ 2)) / (2 * pi * sd-3 ^ 2)
        ;; add votes to each patch from subpopulation 3, calculated as above
      ]
      
  set total-votes sum [ votes ] of patches
  type "Total votes at all locations = " type round(total-votes)     
        ;; add total of votes on all patches and output this to the command window
  
  ask patches [set vote-share votes / total-votes]      
      ;calculate each patch's vote share
  
  set mean-voterx sum [ pxcor * vote-share ] of patches
  set mean-votery sum [ pycor * vote-share ] of patches      
  type "   Mean voter x = " type round(mean-voterx) 
  type "   Mean voter y = " print round(mean-votery)
      ;; calculate center (mean) of voter distribution on each dimension as sum of (patch positions weighted by vote share)
      ;; output this to the command window 
  
  set max-voteshare max [ vote-share ] of patches
  ask patches [set pcolor scale-color red vote-share 0 max-voteshare ] 
      ;; color patches red with density proportional to vote shares
    
  update-support
end  

to update-rule
  if ( who = 0 ) [set my-rule party0-rule]
  if ( who = 1 ) [set my-rule party1-rule]
  if ( who = 2 ) [set my-rule party2-rule]
  if ( who = 3 ) [set my-rule party3-rule]
  if ( who = 4 ) [set my-rule party4-rule]
  if ( who = 5 ) [set my-rule party5-rule]
  if ( who = 6 ) [set my-rule party6-rule]
  if ( who = 7 ) [set my-rule party7-rule]
  if ( who = 8 ) [set my-rule party8-rule]
  if ( who = 9 ) [set my-rule party9-rule]
  if ( who = 10 ) [set my-rule party10-rule]
  if ( who = 11 ) [set my-rule party11-rule]
    ;parties read their decision rules off choice buttons on the interface
end  

;;____________________________
;
;;   PARTY DYNAMICS
;;____________________________


to hunt
  ifelse (mysize > old-mysize) [jump 1] [set heading heading + 90 + random-float 180 jump 1]
      ;;hunter makes a move of size 1 in same direction as previous move if this increased party support
      ;;NB the NetLogo command FORWARD is limited to a maximum of a unit move per tick
      ;;JUMP is needed for larger than unit moves
      ;;else reverses direction and makes a move of size 1 in on a heading chosen from the 180 degree arc now faced
  set old-mysize mysize 
      ;;remember party size for next cycle - note that "go" calls "update-support" BEFORE asking hunters to "hunt"
end

to aggregate
   if (mysize > 0) 
   [
     set xcor (sum [votes * pxcor] of patches with [closest-party = myself] / mysize)
     set ycor (sum [votes * pycor] of patches with [closest-party = myself] / mysize)
   ]
     ;;set party x, y positions at the mean x, y positions of party members
     ;;maintain current position if zero supporters
end  

to predate
  if (mysize < [mysize] of largest-party) [face largest-party jump 1]
      ;;predator identifies the largest party and makes a move of size 1 towards it
end

;;____________________________
;
;;   MAIN CONTROL SUBROUTINES
;;____________________________

to update-support
  ask patches [set closest-party min-one-of parties [distance myself]]
      ;;patches find their closest party
  ask parties [set mysize sum [votes] of patches with [closest-party = myself]] 
      ;;each party sums the votes on patches for which it is the closest party    
      
  set largest-party max-one-of parties [mysize]
      ;;find the largest party
end

to measure-misery
 set voter-misery sum [((distance closest-party ^ 2) / 100) * votes] of patches / total-votes
end

to measure-eccentricity
  ask parties [set eccentricity sqrt ((xcor - mean-voterx) ^ 2 + (ycor - mean-votery) ^ 2) / 10] 
     ;;calculate each party's eccentricity, its Euclidean distance from the center of the voter distribution
  set mean-eccentricity sum [eccentricity] of parties / n-parties
     ;;calculate the mean eccentricity of all parties in the system
end

to adapt
  if (my-rule = "hunter") [hunt]
  if (my-rule = "aggregator") [aggregate]
  if (my-rule = "predator") [predate]
   ;;NB stickers do nothing
end

to calculate-party-variance  ;;used in all-Aggregator runs to stop the program when all parties stop moving
  
  set tickcycle tickcycle + 1
  if (tickcycle = 10) [
	  set old-position-list position-list

    (foreach n-values 12 [? * 2] party-number [
      set position-list replace-item ?1 position-list [xcor] of turtle (item ?2 party-number)
    ])

    (foreach n-values 12 [? * 2 + 1] party-number [
      set position-list replace-item ?1 position-list [ycor] of turtle (item ?2 party-number)
    ])
       
	  set tot-var 0
	  ( foreach old-position-list position-list [ set tot-var tot-var + (?2 - ?1) ^ 2 ] )
    if (tot-var < .1) [ set var-stop 1 ]
  	set tickcycle 0
  ]
end


;;____________________________
;
;;   MAIN CONTROL ROUTINE
;;____________________________

to go
ifelse (party0-rule = "aggregator")
; with aggregator runs: use the calculate-party-variance procedure and var-stop check
; the program stops when the parties stop moving and the system is in stationary state
; NB in benchmark runs all parties are set to use the same rule
  [
    ask parties [adapt]
    update-support
    set enp (total-votes ^ 2) / (sum [mysize ^ 2] of parties)
    measure-eccentricity
    measure-misery
    calculate-party-variance
    set cycle cycle + 1
    if (var-stop = 1) [stop]
  ]
  
  [
    ask parties [adapt]
    update-support
    set enp (total-votes ^ 2) / (sum [mysize ^ 2] of parties)
    measure-eccentricity
    measure-misery
    set cycle cycle + 1
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
441
10
948
538
35
35
7.0
1
10
1
1
1
0
0
0
1
-35
35
-35
35
0
0
1
ticks

BUTTON
0
10
64
52
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
0
51
114
84
votes1
votes1
0
1000000
1000000
1
1
NIL
HORIZONTAL

SLIDER
112
51
213
84
x-mean1
x-mean1
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
211
51
312
84
y-mean1
y-mean1
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
310
51
410
84
sd-1
sd-1
0
40
10
0.5
1
NIL
HORIZONTAL

SLIDER
0
83
113
116
votes2
votes2
0
1000000
0
1
1
NIL
HORIZONTAL

SLIDER
112
83
212
116
x-mean2
x-mean2
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
211
83
311
116
y-mean2
y-mean2
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
310
83
410
116
sd-2
sd-2
0
40
1
0.5
1
NIL
HORIZONTAL

SLIDER
0
114
113
147
votes3
votes3
0
1000000
0
1000
1
NIL
HORIZONTAL

SLIDER
112
114
212
147
x-mean3
x-mean3
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
211
114
311
147
y-mean3
y-mean3
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
310
114
410
147
sd-3
sd-3
0
40
1
0.5
1
NIL
HORIZONTAL

TEXTBOX
133
18
304
46
Population designer\\nEquilateral = (0 17)(-15 -9)(15 -9)
11
0.0
0

SLIDER
310
18
410
51
n-parties
n-parties
1
12
5
1
1
NIL
HORIZONTAL

CHOOSER
8
158
100
203
party0-rule
party0-rule
"sticker" "hunter" "aggregator" "predator"
1

CHOOSER
8
200
100
245
party1-rule
party1-rule
"sticker" "hunter" "aggregator" "predator"
1

CHOOSER
8
242
100
287
party2-rule
party2-rule
"sticker" "hunter" "aggregator" "predator"
1

CHOOSER
8
285
100
330
party3-rule
party3-rule
"sticker" "hunter" "aggregator" "predator"
1

CHOOSER
9
331
101
376
party4-rule
party4-rule
"sticker" "hunter" "aggregator" "predator"
1

CHOOSER
9
376
101
421
party5-rule
party5-rule
"sticker" "hunter" "aggregator" "predator"
2

CHOOSER
182
160
276
205
party6-rule
party6-rule
"sticker" "hunter" "aggregator" "predator"
2

CHOOSER
182
202
276
247
party7-rule
party7-rule
"sticker" "hunter" "aggregator" "predator"
2

BUTTON
55
10
112
52
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
101
160
164
205
p0 vote%
[100 * mysize / total-votes] of turtle 0
1
1
11

MONITOR
101
202
164
247
p1 vote%
[100 * mysize / total-votes] of turtle 1
1
1
11

MONITOR
101
245
164
290
p2 vote%
[100 * mysize / total-votes] of turtle 2
1
1
11

MONITOR
101
288
164
333
p3 vote%
[100 * mysize / total-votes] of turtle 3
1
1
11

MONITOR
100
331
164
376
p4 vote%
[100 * mysize / total-votes] of turtle 4
1
1
11

MONITOR
100
375
164
420
p5 vote%
[100 * mysize / total-votes] of turtle 5
1
1
11

MONITOR
275
160
339
205
p6 vote%
[100 * mysize / total-votes] of turtle 6
1
1
11

MONITOR
275
202
339
247
p7 vote%
[100 * mysize / total-votes] of turtle 7
1
1
11

MONITOR
349
252
426
297
Eccentricity
mean-eccentricity
1
1
11

MONITOR
350
160
425
205
Cycle
cycle
0
1
11

CHOOSER
182
247
276
292
party8-rule
party8-rule
"sticker" "hunter" "aggregator" "predator"
2

CHOOSER
182
291
275
336
party9-rule
party9-rule
"sticker" "hunter" "aggregator" "predator"
2

CHOOSER
182
335
275
380
party10-rule
party10-rule
"sticker" "hunter" "aggregator" "predator"
2

CHOOSER
182
377
275
422
party11-rule
party11-rule
"sticker" "hunter" "aggregator" "predator"
2

MONITOR
274
247
338
292
p8 vote%
[100 * mysize / total-votes] of turtle 8
1
1
11

MONITOR
274
290
339
335
p9 vote%
[100 * mysize / total-votes] of turtle 9
1
1
11

MONITOR
274
334
338
379
p10 vote%
[100 * mysize / total-votes] of turtle 10
1
1
11

MONITOR
274
379
338
424
p11 vote%
[100 * mysize / total-votes] of turtle 11
1
1
11

MONITOR
349
297
424
342
Voter Misery
voter-misery
1
1
11

MONITOR
349
208
426
253
ENP
enp
1
1
11

@#$#@#$#@
WHAT IS IT?
-----------
This implements the "benchmark" model of party competition specified in Chapter 5 of Michael Laver and Ernest Sergenti's book, Party competition: an agent based model (Princeton University Press, 2012). A full description and analysis of the model can be found in this book and is not repeated here.

Party positions and voter ideal points are defined in two dimensions of policy/ideology, directly analogous to the dimensions used in other "spatial" models of political competition.  The horizontal dimension might, for example, be seen as describing left-right economic policy positions; the vertical dimension might be seen as a liberal-conservative policy dimension on matters such as abortion, sexual orientation, euthanasia. 

VOTERS always vote for the closest party. The preference structure of the voting population can be designed as if this is an aggregate of up to three subpopulations. (Only two are investigated by Laver and Sergenti and the default setting on the interface sets the size of the third subpopulation at zero). Voters in each subpopulation have normally distributed ideal points, and each subpopulation is characterized by: the number of voters it comprises; the standard deviation of the distribution of its voters' ideal points, and the means of this distribution on the x and y dimensions. All of these parameters can be set using the sliders in the _Population Designer_ panel near the bottom of the interface.

PARTY LEADERS compete with each other by offering policies to potential supporters. They use one of three _species_ of decision rule _ Sticker, Aggregator, Hunter _ to select a party policy position. These rules are described fully in Laver and Sergenti (2012) and implemented in the _party dynamics_ section of the code.

DYNAMICS OF PARTY COMPETITION. The baseline dynamics of the model iterate forever.  (1) Voters support their closest party.  (2) Given a profile of voter support for parties, leaders adapt party policy positions using their decision rule. (3) Go to 1.

The number of political parties in the benchmark runs in Laver and Sergenti is fixed exogenously. The number of parties and the decision rule used by each party are set on the interface. (In all subsequent models, the sets of surviving parties and decision rules they use are endogenous to model dynamics.) 

HOW TO USE IT
-------------
SETUP sets up parties, supporters and system parameters as specified above.  GO starts and stops the simulation using current parameters.
(Hitting SETUP while GO is still pressed very occasionally causes an error depending where precisely the program is when setup is hit; this easily retrieved by unpressing GO and pressing SETUP again.)

RUNNING EXPERIMENTS.  Laver and Sergenti designed computational benchmarking experiments, and report results of these, in Chapter 5 of their book. Although  _production_ runs were executed on a high performance cluster, precisely equivalent smaller scale experiments can easily be run using Behavior Space. Sketch runs for all results reported in Laver and Sergenti were generated using Behavior Space on a normal laptop. Some sample behavior space experiments are included.


WHAT TO PLAY WITH
-----------
Laver and Sergenti report results from carefully controlled computational experiments and only investigate electorate with two subpopulations. There are infinitely many alternative populations for you to explore using the population designer. There are also many parameterizations of the competitive environment, and the party birth regime, not explored by Laver and Sergenti. You may, for example, want to specify a parameterization of the model you feel corresponds to some real political system that interests you.

By far the most exciting and callenging way forward is to specify and program your own decision rule for party leaders.  Just drop in your coded new rule as a procedure in the party dynamics section, add its name to the rule list, edit it in to the _adapt_ and _color-myself_ procedures, and add a reporter for your rule_s vote share to the interface. You_re good to go!

CREDITS AND REFERENCES
----------------------
Programmed by:
 
Michael Laver, Department of Politics, New York University
ml127@nyu.edu

Ernest Sergenti, The World Bank
esergenti@gmail.com
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="P5_Test_Agg" repetitions="10" runMetricsEveryStep="false">
    <setup>setup
set n-parties 5
set party0-rule "aggregator"
set party1-rule "aggregator"
set party2-rule "aggregator"
set party3-rule "aggregator"
set party4-rule "aggregator"</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <metric>party0-rule</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 0 party-number)</metric>
    <metric>[xcor] of turtle (item 0 party-number)</metric>
    <metric>[ycor] of turtle (item 0 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 1 party-number)</metric>
    <metric>[xcor] of turtle (item 1 party-number)</metric>
    <metric>[ycor] of turtle (item 1 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 2 party-number)</metric>
    <metric>[xcor] of turtle (item 2 party-number)</metric>
    <metric>[ycor] of turtle (item 2 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 3 party-number)</metric>
    <metric>[xcor] of turtle (item 3 party-number)</metric>
    <metric>[ycor] of turtle (item 3 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 4 party-number)</metric>
    <metric>[xcor] of turtle (item 4 party-number)</metric>
    <metric>[ycor] of turtle (item 4 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 5 party-number)</metric>
    <metric>[xcor] of turtle (item 5 party-number)</metric>
    <metric>[ycor] of turtle (item 5 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 6 party-number)</metric>
    <metric>[xcor] of turtle (item 6 party-number)</metric>
    <metric>[ycor] of turtle (item 6 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 7 party-number)</metric>
    <metric>[xcor] of turtle (item 7 party-number)</metric>
    <metric>[ycor] of turtle (item 7 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 8 party-number)</metric>
    <metric>[xcor] of turtle (item 8 party-number)</metric>
    <metric>[ycor] of turtle (item 8 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 9 party-number)</metric>
    <metric>[xcor] of turtle (item 9 party-number)</metric>
    <metric>[ycor] of turtle (item 9 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 10 party-number)</metric>
    <metric>[xcor] of turtle (item 10 party-number)</metric>
    <metric>[ycor] of turtle (item 10 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 11 party-number)</metric>
    <metric>[xcor] of turtle (item 11 party-number)</metric>
    <metric>[ycor] of turtle (item 11 party-number)</metric>
    <metric>voter-misery</metric>
  </experiment>
  <experiment name="P5_Test_Stick" repetitions="10" runMetricsEveryStep="false">
    <setup>setup
set n-parties 5
set party0-rule "sticker"
set party1-rule "sticker"
set party2-rule "sticker"
set party3-rule "sticker"
set party4-rule "sticker"</setup>
    <go>go</go>
    <timeLimit steps="1"/>
    <metric>party0-rule</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 0 party-number)</metric>
    <metric>[xcor] of turtle (item 0 party-number)</metric>
    <metric>[ycor] of turtle (item 0 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 1 party-number)</metric>
    <metric>[xcor] of turtle (item 1 party-number)</metric>
    <metric>[ycor] of turtle (item 1 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 2 party-number)</metric>
    <metric>[xcor] of turtle (item 2 party-number)</metric>
    <metric>[ycor] of turtle (item 2 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 3 party-number)</metric>
    <metric>[xcor] of turtle (item 3 party-number)</metric>
    <metric>[ycor] of turtle (item 3 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 4 party-number)</metric>
    <metric>[xcor] of turtle (item 4 party-number)</metric>
    <metric>[ycor] of turtle (item 4 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 5 party-number)</metric>
    <metric>[xcor] of turtle (item 5 party-number)</metric>
    <metric>[ycor] of turtle (item 5 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 6 party-number)</metric>
    <metric>[xcor] of turtle (item 6 party-number)</metric>
    <metric>[ycor] of turtle (item 6 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 7 party-number)</metric>
    <metric>[xcor] of turtle (item 7 party-number)</metric>
    <metric>[ycor] of turtle (item 7 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 8 party-number)</metric>
    <metric>[xcor] of turtle (item 8 party-number)</metric>
    <metric>[ycor] of turtle (item 8 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 9 party-number)</metric>
    <metric>[xcor] of turtle (item 9 party-number)</metric>
    <metric>[ycor] of turtle (item 9 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 10 party-number)</metric>
    <metric>[xcor] of turtle (item 10 party-number)</metric>
    <metric>[ycor] of turtle (item 10 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 11 party-number)</metric>
    <metric>[xcor] of turtle (item 11 party-number)</metric>
    <metric>[ycor] of turtle (item 11 party-number)</metric>
    <metric>voter-misery</metric>
  </experiment>
  <experiment name="P5_Test_Hunt" repetitions="1" runMetricsEveryStep="true">
    <setup>setup
set n-parties 5
set party0-rule "hunter"
set party1-rule "hunter"
set party2-rule "hunter"
set party3-rule "hunter"
set party4-rule "hunter"</setup>
    <go>go</go>
    <timeLimit steps="100"/>
    <metric>party0-rule</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 0 party-number)</metric>
    <metric>[xcor] of turtle (item 0 party-number)</metric>
    <metric>[ycor] of turtle (item 0 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 1 party-number)</metric>
    <metric>[xcor] of turtle (item 1 party-number)</metric>
    <metric>[ycor] of turtle (item 1 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 2 party-number)</metric>
    <metric>[xcor] of turtle (item 2 party-number)</metric>
    <metric>[ycor] of turtle (item 2 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 3 party-number)</metric>
    <metric>[xcor] of turtle (item 3 party-number)</metric>
    <metric>[ycor] of turtle (item 3 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 4 party-number)</metric>
    <metric>[xcor] of turtle (item 4 party-number)</metric>
    <metric>[ycor] of turtle (item 4 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 5 party-number)</metric>
    <metric>[xcor] of turtle (item 5 party-number)</metric>
    <metric>[ycor] of turtle (item 5 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 6 party-number)</metric>
    <metric>[xcor] of turtle (item 6 party-number)</metric>
    <metric>[ycor] of turtle (item 6 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 7 party-number)</metric>
    <metric>[xcor] of turtle (item 7 party-number)</metric>
    <metric>[ycor] of turtle (item 7 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 8 party-number)</metric>
    <metric>[xcor] of turtle (item 8 party-number)</metric>
    <metric>[ycor] of turtle (item 8 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 9 party-number)</metric>
    <metric>[xcor] of turtle (item 9 party-number)</metric>
    <metric>[ycor] of turtle (item 9 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 10 party-number)</metric>
    <metric>[xcor] of turtle (item 10 party-number)</metric>
    <metric>[ycor] of turtle (item 10 party-number)</metric>
    <metric>[100 * mysize / total-votes] of turtle (item 11 party-number)</metric>
    <metric>[xcor] of turtle (item 11 party-number)</metric>
    <metric>[ycor] of turtle (item 11 party-number)</metric>
    <metric>voter-misery</metric>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
