; Macro Agent Base Model (MABM)
; code written by Rubén Osuna, from the model of Assenza, Gatti, Grazzini et al. (2015) macro ABM (as coded by Elder Silva)


extensions [ matrix r stats ]                               ; calling the R extension and the matrix extension

globals [
  taylor-interest-rate                                      ; the interest rate. Conservative type: only inflation considered. Dual mandate: inflation and output gap. Three mandate: inflation, output gap and credit
  eta1                                                      ; undifferentiated interest rate applied to loans of firms with the retail banks
  eta2                                                      ; interest rate applied to loans of retail banks with the Central Bank
  eta3                                                      ; interest rate applied to deposits of firms in the accounts of the retail banks
  eta4                                                      ; interest rate applied to deposits of retail banks in the accounts of the Central Bank
  GDP                                                       ; sum of consumption, total-investment and public expenditures
  GDP_1                                                     ; GDP of the last period (tick)
  P_GDP                                                     ; potential GDP
  Growth-Rate                                               ; growth rate of the GDP
  Recession-lenght                                          ; how many periods the GDP is decreasing
  Unemployment-Rate                                         ; rate of unemployment
  Total-Consumption                                         ; sum of goods sold by the consumptions firms
  Total-Investment                                          ; sum of total capital sold by the capital production firms
  Goods-Price-Level                                         ; sum (price * qt) / Qt sold
  Goods-Price-Level_1                                       ; the same, for the previous period
  Capital-Price-Level                                       ; sum (price * qt) / Qt sold
  Capital-Price-Level_1                                     ; the same, for the previous period
  Total-Loans-Firms                                         ; sum of loans of retail banks with firms of any kind
  Total-Loans-C-Firms                                       ; sum of loans with the C-Firms
  Total-Loans-K-Firms                                       ; sum of loans with the K-Firms
  Total-Central-Bank-Loans                                  ; sum of loans of the retail banks with the Central Bank
  Total-Loans                                               ; sum of loans in this economy (adding up the previous cathegories)
  Total-Loans_1                                             ; sum of loans in this economy for the previous period
  Total-Client-Loans                                        ; this variable should be equal to Total-Loans, but calculated from banks' variables
  Total-Deposits                                            ; sum of all deposits of the retail banks
  Total-Clients-Accounts                                     ; this variable should be equal to Total-Deposits, but calculated from banks' variables
  Total-Deposits_1                                          ; sum of all deposits of the retail banks, previous period
  Total-Reserves                                            ; sum of all reserves of the retail banks
  Total-Deposits-Workers                                    ; sum of deposit at the bank of the households
  Total-Deposits-Capitalists                                ; sum of deposit at the bank of the capitalists
  Total-Deposits-C-Firms                                    ; sum of deposit at the bank of the C-Firms
  Total-Deposits-K-Firms                                    ; sum of deposit at the bank of the K-Firms
  Total-Interest-Payments                                   ; sum of all interest payments from loaners to retail banks
  Total-Installment-Payments                                ; sum of all installment payments from loaners to retail banks
  C-Firms                                                   ; group of firms producing consumption goods
  K-Firms                                                   ; group of firms producing capital goods
  Workers                                                   ; group of households that offer work time
  Capitalists                                               ; group of households that are owners of firms and banks and do not work for others
  Employed                                                  ; group of workers with an employment
  Unemployed                                                ; group of workers without an employment
  C-Broke                                                   ; number of consumption firms that went broken last period
  K-Broke                                                   ; number of capital production firms that went broken last period
  B-Broke                                                   ; number of banks that went broken last period
  Matrix-C-Firms-Bankruptcy                                 ; input with information that will be used in calculation into R (eq 23)
  Matrix-K-Firms-Bankruptcy                                 ; input with information that will be used in calculation into R (eq 23)
  Matrix-Basel-Buffer                                       ; input with information that will be used in calculation into R (eq 23)
  cpi                                                       ; consumer price index of this period
  cpi_1                                                     ; consumer price index of the previous period
  inflation                                                 ; inflation of this period
  inflation_1                                               ; inflation of the previous period
  Wage-on-GDP                                               ; total salaries as a percentage of the GDP, then wage = Wage-on-GDP * GDP / Number-of-Employees is the wage that keeps that proportion constant
  Wage-on-GDP_1                                             ; total salaries as a percentage of the GDP, then wage = Wage-on-GDP * GDP / Number-of-Employees is the wage that keeps that proportion constant
  Credit-on-GDP                                             ; this is the basic ratio used in the Counter-Cyclical-Capital-Buffer defined in Basel III rules
  Credit-on-GDP_1                                           ; the same from the previous period
  Credit-on-GDP-Gap                                         ; difference between current Credit-on-GDP and the long run trend
  State-budget                                              ; how much money the Government owns at this stick
  State-budget_1                                            ; how much money the Government had the previous period. It is related to the cycle
  Leverage-Banks                                            ; mean leverage of the Banks, the Central Bank has their numbers
  Central-Bank-Own-Capital                                  ; the own capital of the central bank, a liability in its balance
  Account-of-State-at-Central-Bank                          ; Account of the State at the Central Bank
  Monetary-Base                                             ; currency plus total reserves or public debt plus credits o the central bank to retail bank less Account-of-State-at-Central-Bank and the capital of the CB
  Currency                                                  ; bills and coins, the monetary base less total reserves
  Money-Supply                                              ; menetary base multiplied by the money multiplier, which is the inverse of RR, the reserve ratio
  Capital-Buffer                                            ; a % of the new loans acting like reserves
  QEasing?                                                  ; it is automatically set true if three periods of recession are observed
  PublicDebt                                                ; public debt titles sold to the central bank
  PublicDebt_1                                              ; public debt titles sold to the central bank the previous period
  Wage                                                      ; uniform wage paid to the workers by C-Firms and K-Firms
  Wage_1                                                    ; wage of the previous period
  Helped-Households                                         ; how many households are being helped
  Beta_1-C-Firms                                            ; parameter utilized by the banks to calculate the interest rate for the firms (eq 26)
  Beta_2-C-Firms                                            ; parameter utilized by the banks to calculate the interest rate for the firms (eq 26)
  Beta_1-K-Firms                                            ; parameter utilized by the banks to calculate the interest rate for the firms (eq 26)
  Beta_2-K-Firms                                            ; parameter utilized by the banks to calculate the interest rate for the firms (eq 26)
  Mjc                                                       ; parameter utilized in the process "SubF-Call-Matrix-formation", related to C-Firms
  Mjk                                                       ; parameter utilized in the process "SubF-Call-Matrix-formation", related to K-Firms
  Mjb                                                       ; parameter utilized in the process related to the calculation of the Capital Buffer set in Basel III rules
  Current-Number-of-C-Firms                                 ; current number of C-Firms, after losses and entries
  Current-Number-of-K-Firms                                 ; current number of K-Firms, after losses and entries
  Current-Number-of-Workers                                 ; current number of Workers, after losses and entries
  ]

breed                 [ Banks        Bank         ]         ; agent class
breed                 [ Households   Household    ]         ; agent class
breed                 [ Firms        Firm         ]         ; agent class
directed-link-breed   [ Hireds       Hired        ]         ; link used to link firms to the worker
directed-link-breed   [ Owners       Owner        ]         ; link used to link firms and banks to their owner/capitalist
directed-link-breed   [ Bankers      Banker       ]         ; link used for connecting the clients with its bank


Banks-own [
  Clients-Accounts-C-Firms                                  ; the total accounts (deposits) of C-Firms in this bank
  Clients-Accounts-K-Firms                                  ; the total accounts (deposits) of K-Firms in this bank
  Clients-Accounts-Firms                                    ; the sum of the former two variables
  Clients-Accounts-Workers                                  ; the total accounts (deposits) of Workers in this bank
  Clients-Accounts-Capitalists                              ; the total accounts (deposits) of Capitalists in this bank
  Clients-Accounts-Households                               ; the total accounts (deposits) of Households (Workers and Capitalists) in this bank
  Clients-Accounts-at-the-Bank_1                            ; the sum of all previous variables (sum of Firms and Households accounts), previous period
  Clients-Accounts-at-the-Bank                              ; the sum of all previous variables (sum of Firms and Households accounts)
  Clients-Loans-C-Firms                                     ; the total loans of C-Firms in this bank
  Clients-Loans-K-Firms                                     ; the total loans of K-Firms in this bank
  Clients-Loans-with-the-Bank_1                             ; the total loans Firms in this bank, previous period
  Clients-Loans-with-the-Bank                               ; the total loans Firms in this bank
  Available-Credit-Basel2                                   ; maximum available credit under Basel II rules
  Available-Credit-Basel31                                  ; maximum available credit under Basel III rules, Global Capital Framework
  Available-Credit-Basel32                                  ; maximum available credit under Basel III rules, Global Liquidity Requirement
  Available-Credit-Basel                                    ; maximum available credit under Basel rule
  Number-of-Firm-Clients
  Number-of-Household-Clients
  Number-of-Clients
  Own-Reserves                                              ; the reserves of retail banks are deposited in the Central Bank, and they are rewarded with the eta4
  Own-Bank-Capital                                          ; bank own capital
  Need-Cash                                                 ; need of money after production, because the bank has losses and it does not have enough assets for paying it
  Bank-Equity                                               ; Equity is the net worth, the difference between assets and liabilities
  Central-Bank-Loans                                        ; loans from the Central Bank to the Retail Banks
  Profit                                                    ; sum of interest received less the bank losses with their loans
  PBT                                                       ; profits before taxes
  I-Received                                                ; total interest payments of Firms to the Banks
  D-Received                                                ; total installment pays of Firms to the Banks
  Assets                                                    ; total assets of retail banks
  Liabilities                                               ; total liabilities of retail banks
  Overdraft                                                 ; losses of the bank so large than it cannot compensate it with own capital
  Shares                                                    ; the own-bank-capital divided by the total number of capitalists in this economy
  PBT                                                       ; profits before taxes
  ]

Households-own [
  Worker?                                                   ; property of the household, true if it is a worker, false if it is a capitalist
  Employed?                                                 ; property of the household, true if it is employed, false if it is unemployed
  Bankrupt?                                                 ; property of the household, true if the capitalist is broken and become a worker
  Human-Wealth                                              ; variable used in determining how much of the entire wealth will be used for consumption
  Current-Income                                            ; the salary if it is a worker, the profit if it is a capitalist
  Bank-Account-Household                                    ; the total of wealth of the households held with the bank
  Desired-Consumption                                       ; theorical value the household want to consume this period
  Consumption                                               ; real value the household did consumption
  C-Firm-List                                               ; list with the firms the household visited last period
  My-Nearest-Bank                                           ; the nearest bank to the household
  ]

Firms-own [
  C-Firm?                                                   ; property of the firms, yes if it is a consumption production firm, not if is a capital production firm
  Owner?                                                    ; any firm may lose its owner, so this parameter sets the existence of not of an owner at a particular moment
  Price                                                     ; last price practiced by the firm
  Demand                                                    ; last quantity demanded
  Production                                                ; actual production of the firm
  Stock                                                     ; amount available for sales (inventories)
  Delta                                                     ; parameter used to calculate new price
  Number-of-Employees                                       ; actual number of employees of this firm
  Need-of-Employees                                         ; ideal number of employees for this firm
  Vacancy                                                   ; number of opened places at this firm for new workers
  Revenue                                                   ; sum of quantity sold times the price
  Sales                                                     ; sum of quantity sold last period
  Profit                                                    ; revenue less the expenditures, considering taxes
  PBT                                                       ; profits before taxes
  Dividend                                                  ; part of the profit given to the owner, from firms and banks
  Overdraft                                                 ; difference between losses and firm's own resources
  Need-Cash                                                 ; need of money before production starts
  Bank-Account-Firm                                         ; money available at the bank owned by the firms
  Leverage                                                  ; debts of the firm divided by their assets, including the debt, ranging between 0 and 1 (0 is zero debt, and 1, all assets are debt)
  Assets                                                    ; sum deposit at bank and stock available for sales
  Liabilities                                               ; total liabilities of firms
  Equity-C-Firms                                            ; assets minus liabilities, or net worth
  Equity-K-Firms                                            ; assets minus liabilities, or net worth
  eta0                                                      ; taylor-made interest rata calculated for every firm
  Current-Retail-Loans                                      ; loans with the retail banks
  Cases-of-Bankruptcy                                       ; the total number of bankruptcies by this firm
  Probability-of-Bankruptcy                                 ; the probability of dying of the firm
  Available-Credit-Assenza                                  ; maximum available credit under internal risk calculations of the bank
  Interest-Pay                                              ; sum of interest rate payed last period
  Installment-Pay                                           ; sum of the installment payed last period
  T-expected                                                ; the expected life span for this firm
  Capital                                                   ; property of the C-firms, indicate the capital available to be used at the production process
  Desired-Investment                                        ; property of the C-Firms, ideal investment needed
  K-Firm-List                                               ; property of the C-Firms, list with the K-Firms the C-Firms visited last period trying to buy capital goods
  Investment                                                ; property of the C-Firms, capital goods bough last period
  Investment-Memory                                         ; property of the C-firms, parameter utilized to calculate te Desired-Investment
  Util-Capacity                                             ; property of the C-Firms, indicates how much of the Capital had been used
  Stock_t-1                                                 ; Stock available in t-1
  My-Nearest-Bank                                           ; the nearest bank to the firm
  ]


; TO SETUP


to Setup
  clear-all
  ;Setup-Globals
  Short-Setup-Globals
  if Do_We_Have_Banks?                [ Setup-Banks         ]             ; call the setup for banks
  if Do_We_Have_C-Firms?              [ Setup-C-Firms       ]             ; call the setup for C-Firms
  if Do_We_Have_K-Firms?              [ Setup-K-Firms       ]             ; call the setup for K-Firms
  if Do_We_Have_Workers?              [ Setup-Workers       ]             ; call the setup for workers
  if Do_We_Have_Capitalists?          [ Setup-Capitalists   ]             ; call the setup for Capitalists
  reset-ticks                                                             ; netlogo need this
end

; The setup-globals is not being called. but if it is included into the setup it is not working, because some variable is not a defined global.

to Short-Setup-Globals
  set C-Firms                        Firms      with [     C-Firm? ]                   ; building the group with all C-Firms
  set K-Firms                        Firms      with [ not C-Firm? ]                   ; building the group with all K-firms
  set Capitalists                    Households with [ not Worker?  ]                   ; building the group with all Capitalist
  set Workers                        Households with [    Worker?   ]                   ; building the group with all workers
  if not Random-Seed?               [ random-seed 12345 ]                               ; random seed may or not be used

end


to Setup-Globals                                                                      ; call the setup for globals
  set taylor-interest-rate              natural-interest-rate                         ; the interest rate set by the central bank
  set GDP                               0                                             ; initial GDP
  set GDP_1                             0                                             ; GDP of the previous period
  set Recession-lenght                  0                                             ; number of periods with GDP falls
  set Unemployment-rate                 0                                             ; initial Unemployment rate
  set Total-Consumption                 0                                             ; initial value of consumption
  set Total-Investment                  0                                             ; initial value of total investment
  set Goods-Price-Level                 1                                             ; initial goods price level, necessary minimum for taking logarithms later
  set Capital-Price-Level               1                                             ; initial capital goods price level, necessary minimum for taking logarithms later
  set Goods-Price-Level_1               0.5                                           ; initial goods price level of the previous period, necessary minimum for taking logarithms later
  set Capital-Price-Level_1             0.5                                           ; initial capital goods price level of the previous period, necessary minimum for taking logarithms later
  set Wage                              Minimum-Wage                                  ; initial value for the wage
  set Wage_1                            0                                             ; initial value for the wage of the previous period
  set cpi                               0                                             ; consumer price index of this period
  set cpi_1                             0                                             ; consumer price index of the previous period
  set inflation                         0                                             ; inflation of this period
  set inflation_1                       0                                             ; inflation of the previous period
  set Wage-on-GDP                       0                                             ; total salaries as a percentage of the GDP, then wage = Wage-on-GDP * GDP / Number-of-Employees is the wage that keeps that proportion constant
  set Wage-on-GDP_1                     0                                             ; total salaries as a percentage of the GDP, then wage = Wage-on-GDP * GDP / Number-of-Employees is the wage that keeps that proportion constant
  set Credit-on-GDP                     0                                             ; this is the basic ratio used in the Counter-Cyclical-Capital-Buffer defined in Basel III rules
  set Credit-on-GDP_1                   0                                             ; this is the basic ratio used in the Counter-Cyclical-Capital-Buffer defined in Basel III rules
  set Total-Loans-Firms                 0                                             ; initial value for total loans of retail banks and firms
  set Total-Loans-C-Firms               0                                             ; initial value for C-loans
  set Total-Loans-K-Firms               0                                             ; initial value for K-loans
  set Total-Central-Bank-Loans          0                                             ; initial value for the total loans of retail banks and central bank
  set Total-Loans                       0                                             ; initial value for the total loans in this economy
  set Total-Loans_1                     0                                             ; sum of loans in this economy for the previous period
  set Total-Deposits-Workers            0                                             ; sum of deposit at the bank of the households
  set Total-Deposits-Capitalists        0                                             ; sum of deposit at the bank of the capitalists
  set Total-Deposits-C-Firms            0                                             ; sum of deposit at the bank of the C-Firms
  set Total-Deposits-K-Firms            0                                             ; sum of deposit at the bank of the K-Firms
  set Total-Deposits                    0                                             ; sum of all deposits of retail banks
  set Total-Deposits_1                  0                                             ; sum of all deposits of retail banks, previous period
  set Total-Reserves                    0                                             ; the reserves of retail banks are deposited in the Central Bank, and they are rewarded with the eta4
  set Total-Interest-Payments           0                                             ; total interest payments of firms to reatail banks for loans received
  set Total-Installment-Payments        0                                             ; total installment payments firms to reatail banks for loans received
  ; the Available-Credit-Basel is a bank propertie, not a global. Need to be relocated to bank setup
  ;set Available-Credit-Basel            0                                             ; this is the credit limit to retail banks from Basel macroprudential rules
  ;set Available-Credit-Basel2           0                                             ; maximum available credit under Basel II rules
  ;set Available-Credit-Basel31          0                                             ; maximum available credit under Basel III rules, Global Capital Framework
  ;set Available-Credit-Basel32          0                                             ; maximum available credit under Basel III rules, Global Liquidity Requirement
  ; ^^^^^^^^ to be relocated
  set State-Budget                      0                                             ; initial value for State-Budget
  set C-Firms                           Firms      with [ C-Firm?        ]            ; building the group with all C-Firms
  set K-Firms                           Firms      with [ not C-Firm?    ]            ; building the group with all K-firms
  set Capitalists                       Households with [ not Worker?    ]            ; building the group with all Capitalist
  set Workers                           Households with [ Worker?        ]            ; building the group with all workers
  set Employed                          Workers    with [ Employed?      ]            ; building the group with all employed workers
  set Unemployed                        Workers    with [ not Employed?  ]            ; building the group with all unemployed workers
  set Helped-Households                 0                                             ; number of households helped by the State
  set Central-Bank-Own-Capital          sum [Initial-equity-of-the-bank ] of Banks    ; the own capital of the central bank, a liability in its balance
  set Account-of-State-at-Central-Bank  0                                             ; Account of the State at the Central Bank
  set Matrix-C-Firms-Bankruptcy         matrix:make-constant  2 Time-Window1 0        ; building the initial matrix which will be used by R-Netlogo later, it is a constant matrix of zeroes with 2 rows and "T-W1" columns
  set Matrix-K-Firms-Bankruptcy         matrix:make-constant  2 Time-Window1 0        ; building the initial matrix which will be used by R-Netlogo later, it is a constant matrix of zeroes with 2 rows and "T-W1" columns
  set Matrix-Basel-Buffer               matrix:make-constant  2 Time-Window2 0        ; building the initial matrix which will be used by R-Netlogo later, it is a constant matrix of zeroes with 2 rows and "T-W2" columns
  set Mjc                               0                                             ; helper with the matrix above
  set Mjk                               0                                             ; helper with the matrix above
  set Mjb                               0                                             ; helper with the matrix above
  set Probability-of-Bankruptcy         0                                             ; probability of a firm of dying, explained by the leverage of that firm
  set PublicDebt                        0                                             ; public debt titles sold to the central bank
  set PublicDebt_1                      0                                             ; public debt titles sold to the central bank the previous period
  if not Random-Seed?                 [ random-seed 12345 ]                           ; random seed may or not be used
end


to Setup-Banks
    create-Banks Number-of-Banks [                                                 ; create n banks, determined in a global variable
    setxy                                random-xcor random-ycor                   ; place them randomly
    move-to                              one-of patches with [not any? Firms-here] ; if a C-Firm or K-Firm is at the same patch, move the Bank to another empty patch in the setup process
    set shape                           "house"                                    ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set size                             2                                         ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set color                            turquoise                                 ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set Own-Bank-Capital                 Initial-equity-of-the-bank                ; explicit
    set Bank-Equity                      Own-Bank-Capital                          ; this is the initial equity of the bank
    set Clients-Accounts-C-Firms         0                                         ; the total accounts (deposits) of C-Firms in this bank
    set Clients-Accounts-K-Firms         0                                         ; the total accounts (deposits) of K-Firms in this bank
    set Clients-Accounts-Firms           0                                         ; the sum of the former two variables
    set Clients-Accounts-Workers         0                                         ; the total accounts (deposits) of Workers in this bank
    set Clients-Accounts-Capitalists     0                                         ; the total accounts (deposits) of Capitalists in this bank
    set Clients-Accounts-Households      0                                         ; the total accounts (deposits) of Households (Workers and Capitalists) in this bank
    set Clients-Accounts-at-the-Bank_1   0                                         ; the sum of all previous variables (sum of Firms and Households accounts), previous period
    set Clients-Accounts-at-the-Bank     0                                         ; the sum of all previous variables (sum of Firms and Households accounts)
    set Clients-Loans-C-Firms            0                                         ; the total loans of C-Firms in this bank
    set Clients-Loans-K-Firms            0                                         ; the total loans of K-Firms in this bank
    set Clients-Loans-with-the-Bank_1    0                                         ; the total loans Firms in this bank, previous period
    set Clients-Loans-with-the-Bank      0                                         ; the total loans Firms in this bank
    set Number-of-Firm-Clients           0
    set Number-of-Household-Clients      0
    set Number-of-Clients                0
    set Available-Credit-Basel2          0                                         ; maximum available credit under Basel II rules
    set Available-Credit-Basel31         0                                         ; maximum available credit under Basel III rules, Global Capital Framework
    set Available-Credit-Basel32         0                                         ; maximum available credit under Basel III rules, Global Liquidity Requirement
    set Available-Credit-Basel           0                                         ; maximum available credit under Basel rules
    set Own-Reserves                     0                                         ; reserves of the bank, deposits multiplied by the reserve requirement
    set Need-Cash                        0                                         ; initial value for cash needed after production, because heavy losses
    set Profit                           0                                         ; revenue less the expenditures, considering taxes
    set PBT                              0                                         ; profits before taxes
    set I-Received                       0                                         ; local variable
    set D-Received                       0                                         ; local variable
    set Overdraft                        0                                         ; this is the difference betwween own liquid assets and losses
    ; properties relocated from the setup-globals
    set Available-Credit-Basel            0                                             ; this is the credit limit to retail banks from Basel macroprudential rules
    set Available-Credit-Basel2           0                                             ; maximum available credit under Basel II rules
    set Available-Credit-Basel31          0                                             ; maximum available credit under Basel III rules, Global Capital Framework
    set Available-Credit-Basel32          0                                             ; maximum available credit under Basel III rules, Global Liquidity Requirement
    ]
end



to Setup-C-Firms
   ask n-of Number-of-C-Firms patches [                                       ; this command asks a number of patches controlled by a slider with random coordinates
    sprout-Firms 1 [                                                          ; ask to create one firm at these patches
      move-to one-of patches with [not any? Firms-here]                       ; if a C-Firm and K-Firm are at the same patch, move the K-Firm in the setup process
      set shape                      "factory"                                ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set size                       2                                        ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set color                      orange                                   ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set C-Firm?                    true                                     ; proper of the Firms, true means it is a consumption producer firm
      set Owner?                     true                                     ; this firm will have an owner
      set Price                      2                                        ; initial value for price
      set Capital                    Initial-capital                          ; initial value for capital
      set Stock                      Initial-production-C-firms               ; initial value for stock
      set Production                 Initial-production-C-firms               ; initial value for production
      set Revenue                    0                                        ; initial value for revenue
      set Sales                      0                                        ; initial value for sales
      set Profit                     0                                        ; revenue less the expenditures, considering taxes
      set PBT                        0                                        ; profits before taxes
      set Need-Cash                  0                                        ; initial value for cash needed before production, because lack of liquidity for starting it
      set Cases-of-Bankruptcy        0                                        ; the total number of bankruptcies by this firm
      set Available-Credit-Assenza   0                                        ; maximum available credit under internal risk calculations of the bank
      set Desired-Investment         0                                        ; initial value for desire investment
      set Investment                 0                                        ; initial value for investment
      set Number-of-Employees        0                                        ; initial value for number of employees
      set Need-of-Employees          0                                        ; initial value for desire employees
      set K-Firm-List               [ ]                                       ; setup for the list which will be used to colect the K-Firms visited
      set Vacancy                    0                                        ; initial value for vacancy
      set Bank-Account-Firm          Initial-liquidity-of-Firms               ; initial money deposited at the bank by the firms
      set Investment-Memory          Initial-Capital                          ; initial value for Investment-Memory, parameter utilized to calculate the desire investment
      set Util-Capacity              0                                        ; initial value for util-capacity
      create-banker-from             min-one-of banks [distance myself] [set hidden? true]      ; this create a link between the bank and the capitalist
      set My-Nearest-Bank            min-one-of banks [distance myself]       ; this bank is the same to which the C-Firms is now linked with a "banker" relationship
      ]
    ]
end

to Setup-K-Firms
   ask n-of Number-of-K-Firms patches  [                                   ; this command asks a number of patches controlled by a slider randomly selected
      sprout-Firms 1 [                                                     ; ask to create one firm in each of those patches
      move-to one-of patches with [not any? Firms-here]                    ; if a C-Firm and K-Firm are at the same patch, move the K-Firm in the setup process
      set shape                      "factory"                             ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set size                       2                                     ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set color                      green                                 ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set C-Firm?                    false                                 ; proper of Firms, false mean it is a capital producer firm
      set Owner?                     true                                  ; this firm will have an owner
      set Price                      2                                     ; initial value for price
      set Stock                      Initial-production-K-firms            ; initial value for stock
      set Production                 Initial-production-K-firms            ; initial value for production
      set Revenue                    0                                     ; initial value for revenue
      set Sales                      0                                     ; initial value for sales
      set Profit                     0                                     ; revenue less the expenditures, considering taxes
      set PBT                        0                                     ; profits before taxes
      set Need-Cash                  0                                     ; initial value for cash needed before production, because lack of liquidity for starting it
      set Cases-of-Bankruptcy        0                                     ; the total number of bankruptcies by this firm
      set Available-Credit-Assenza   0                                     ; maximum available credit under internal risk calculations of the bank
      set Desired-Investment         ""                                    ; not applied to K-Firms
      set Investment                 ""                                    ; not applied to K-Firms
      set Number-of-Employees        0                                     ; initial number for employees
      set Need-of-Employees          0                                     ; initial value for desired employees
      set Vacancy                    0                                     ; initial value for vacancy
      set Bank-Account-Firm          Initial-liquidity-of-Firms            ; initial money deposited at the banks by the firms
      set Investment-Memory          ""                                    ; not applied to K-Firms
      set Util-Capacity              ""                                    ; not applied to K-Firms
      create-banker-from             min-one-of banks [distance myself] [set hidden? true]   ; this create a link between the bank and the capitalist
      set My-Nearest-Bank            min-one-of banks [distance myself]    ; this bank is the same to which the K-Firms is now linked with a "banker" relationship
      ]
    ]
end


to Setup-Workers
   create-Households Number-of-Workers [
    setxy                            random-xcor random-ycor                   ; the initial position of the worker is random
    set shape                       "person"                                   ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set size                         1.25                                      ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set color                        yellow                                    ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set Worker?                      true                                      ; property of the household, "true" indicates it is a worker
    set Employed?                    false                                     ; all workers are unemployed at the start
    set Human-Wealth                 1                                         ; initial value for Human Wealth, utilized to calculate the consumption process
    set Current-Income               0                                         ; intitial income
    set Bank-Account-Household       Initial-households-personal-assets        ; initial value of deposits at the banks
    set Desired-Consumption          0                                         ; initial value for desired Consumption
    set Consumption                  0                                         ; initial value for Consumption
    set C-Firm-List                 [ ]                                        ; setup for the list which will be used to record the C-Firms visited
    create-banker-from               min-one-of banks [distance myself] [set hidden? true]       ; this create a link between the bank and the capitalist
    set My-Nearest-Bank              min-one-of banks [distance myself]        ; this bank is the same to which the Worker is now linked with a "banker" relationship
   ]
end

to Setup-Capitalists
   create-Households (Number-of-C-Firms + Number-of-K-Firms) [                ; one capitalist for each firm distributed ramdomly in space
    setxy                           random-xcor random-ycor                   ; the initial position of the worker is random
    set shape                      "person"                                   ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set size                        1.25                                      ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set color                       white                                     ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set Worker?                     false                                     ; proper of the household, false means it is a capitalist
    set Bankrupt?                   false
    set Human-Wealth                1                                         ; initial value for human wealth
    set Current-Income              0                                         ; initial value for current income
    set Bank-Account-Household      Initial-households-personal-assets        ; initial value for deposit at bank
    set Desired-Consumption         0                                         ; initial value for Desired-Consumption, D-C is the budget allocated to consumption
    set Consumption                 0                                         ; initial value of Consumption
    set C-Firm-List                [ ]                                        ; setup for the list which will be used to collect the price of the C-Firms visited
    create-owner-from               one-of Firms with [not any? out-link-neighbors] [set hidden? true] ; this create a link between the firm and the capitalist
    create-banker-from              min-one-of banks [distance myself] [set hidden? true]              ; this create a link between the bank and the capitalist
    set My-Nearest-Bank             min-one-of banks [distance myself]        ; this bank is the same to which the Capitalist is now linked with a "banker" relationship
   ]
end


; TO GO


to Go
   Function-Job-Market                                            ; this function plays the job market process
   Function-Consumption-Market                                    ; this function plays the consumption market process
   Function-Capital-Market                                        ; this function plays the capital market process
   Function-UpDate-Production                                     ; this function plays the production process

   Function-UpDate-Globals                                        ; this function update the globals
   Function-Cash-Flows-Dividends                                  ; this function update the cash flow and set the dividends

   Function-Credit-Market                                         ; this function update the credit market

   Function-Central-Bank                                          ; this function give loans to the banking system and set macropudential rules and monetary policy
   Function-State-Manage-Budget                                   ; the government appears getting taxes and giving subsidies to families, helping banks and during a crisis period and selling public debt

   Function-Update-System

   tick                                                           ; netlogo needs this because it controlates time

end



; FUNCTIONS


; TO FUNCTION-JOB-MARKET


To Function-Job-Market                              ; here will be called the subfunctions
  ask Firms [                                       ; reset the firms need for cash and update the desire workers number
      SubF-UpDate-Desired-Workers
      set Need-Cash                    0
  ]

  ask Workers with [Current-Income = 0] [           ; ask workers which do not have a job to look for one
      SubF-Find-a-Job
  ]
  ask Workers  [ set Bank-Account-Household    Bank-Account-Household + Current-Income         ] ; ask workers to update their income
  ask Firms    [ set Bank-Account-Firm         Bank-Account-Firm - Number-of-Employees * wage  ] ; ask firms to update their bank account
end


to SubF-UpDate-Desired-Workers                                                                                        ; this subfunction update the desired workers
   if C-Firm?  [                                                                                                      ; only C-Firms run this
     set Need-of-Employees   min list Need-of-Employees ((Capital * Productivity-of-capital) / Productivity-of-labor) ; set desire employee considering the capital production restriction (Assenza, Gatti, Grazzini et al. 2015, eq. 5.9)
     if Hiring-Rationality? [                                                                                         ; if this switch is on, the firm will hire employees limited to the Bank-Account-Firm resources
         if Need-of-Employees * Wage > Bank-Account-Firm [                                                            ; if Need-of-Employees multiplied by wage is larger than the Bank-Account, the Need-of-Employees is updated
            set Need-of-Employees      max list 0 (int (Bank-Account-Firm / Wage) + 1)                                ; update the desired employees considering the money available
         ]
     ]
   ]
   set Vacancy                         Need-of-Employees   -   Number-of-Employees     ; opening new vacancies to contract more employees
   while [Vacancy < 0 and Number-of-Employees > 0] [                                   ; if vacancy < 0 the firm needs to fire someone
    ask one-of my-out-hireds [                                                         ; choose one employee
        ask other-end [                                                                ; ask this employee to set...
            set Current-Income         0                                               ; their income as 0, it was fired (die)
            set Employed?              false                                           ; the worker is now unemployed
        ]
    die                                                                                ; "die" means this worker is fired (the link disappears)
    ]
    set Vacancy                        Vacancy               + 1                       ; record the fired worker
    set Number-of-Employees            Number-of-Employees   - 1                       ; update the number of employees after the firing process
   ]
end


to SubF-Find-a-Job                                                                                    ; this subfunction is used by the worker trying to find a job position
   let extension                             0
   repeat 3 [                                                                                         ; the worker will repeteat the process two times
      if Employed? = true [ stop ]                                                                    ; if the worker finds a job the seach process stops
      let aFirm one-of Firms in-radius (radius-for-employment-search + extension) with [Vacancy > 0 ] ; the worker looks for a firm around him with vacancies
      if aFirm != nobody [
         create-hired-from                   one-of Firms-here [set hidden? true]                     ; this link means the worker has a job in a firm, and the two know about eachother
         set Current-Income                  Wage * (1 - tax1)                                        ; update the current-income of the worker
         set State-Budget                    State-Budget + wage * tax1                               ; the tax on income goes to the State
         set Employed?                       true                                                     ; the worker is employed now
         ask aFirm [
             set Number-of-Employees      Number-of-Employees + 1                             ; update the number of employees
             set Vacancy                  Vacancy             - 1                             ; update the vacancies
         ]
      ]
      if aFirm = nobody [
         set Current-Income                 0
         set extension                      extension + 10                    ; a new search starts widening the radius
      ]
  ]
  if Do_We_Have_State? = true and Employed? = false [
     SubF-Get-Unemployment-Benefits                                ; get unemployment benefits if the worker is unemployed
  ]

end


to SubF-Get-Unemployment-Benefits
   let Unemployment-Benefit                 Wage * Unemployment-Cover                   ; the unemployment benefits is a fraction (= u-cover) of the wage
   ifelse Do_We_Have_Central-Bank? = true and QEasing? = true [                         ; if there is a Central Bank and Quantitative Easing policy, this authority will buy any amount of Public Debt, so there are no constraints
          set Current-Income                Unemployment-Benefit                        ; if the worker is unemployed and his/her Current income is zero the State pays unemployment benefits
          set State-Budget                  State-Budget - Unemployment-Benefit         ; the tax on income goes to the State
          set Helped-Households             Helped-Households + 1                       ; the total number of households helped is increased
   ] [
          if State-Budget >= Unemployment-Benefit [                                     ; if there is enough budget available the State pays the benefits, and the Central Bank help is unnecessary
             set Current-Income             Unemployment-Benefit                        ; if the worker is unemployed and its Current income is zero the State pays unemployment benefits, a tax free subsidy
             set State-Budget               State-Budget - Unemployment-Benefit         ; the tax on income goes to the State
             set Helped-Households          Helped-Households + 1                       ; the total number of households helped is increased
          ]
   ]
end



; FUNCTION-CONSUMPTION-MARKET


to Function-Consumption-Market                                           ; here will be called the subfunctions of the consumption market
   ask C-Firms [
       set Demand                      0
       set Sales                       0
       set Revenue                     0
   ]                                                                     ; updating properties of the C-Firms
   ask Households [                                                      ; explicit
       SubF-UpDate-Desired-Consumption                                   ; call subfunction for households update their desired consumption
       SubF-Visit-C-Firms                                                ; call subfunction for households visit firm trying to consume
       if Desired-Consumption > 0 [ SubF-Try-Consume ]                   ; if the household wish to consume something, it tries to buy it
   ]
end


to SubF-UpDate-Desired-Consumption
   set Human-Wealth                   (Memory-parameter-human-wealth * Human-Wealth) + (1 - Memory-parameter-human-wealth) * Current-Income                          ; update human wealth (eq 2)
   set Desired-Consumption             min list  Bank-Account-Household  ((Fraction-of-wealth-devoted-to-consumption * Bank-Account-Household) + Human-Wealth)       ; update desired consumption, it can not be higher than the money the household has
   set Desired-Consumption             max list  0 Desired-Consumption                                                                                               ; this ensures the desired consumption is not negative
end


to SubF-Visit-C-Firms                                                                                         ; this subfunction rules the shopping process, visiting the firms
  set C-Firm-List sort-by [ [?1 ?2] -> [Price] of ?1 < [Price] of ?2] C-Firms in-radius 30 with [Stock > 0]   ; we build a list with the C-Firms in the neighborhood ordered by price, assuming the household doesn't have the prices of all C-Firms because of time constraints
end


to SubF-Try-Consume                                                                              ; the household doesn't take the lowest price...
  let listposition floor random-gamma 1.2 0.66                                                   ; ... but applies a distribution to the ordered list, because information is not perfect, scale = 1/lambda, shape/lamda 1.2/0.66 CDF: 0 (0) 1 (0.39) 2 (0.66) 3 (0.81) 4 (0.90)
  if listposition >= (length C-Firm-List) [set listposition 0]                                   ; in this way we increase the probability of the lowest price, by adding up those cases in which the gamma give us a number larger than the length of the list
  print listposition
  let Selected-C-Firm                          (item listposition C-Firm-List)                   ; auxiliar variable
  let Qtt                                       Desired-Consumption / [Price] of Selected-C-Firm ; total number of consumption goods desired
  let expenses                                  0                                                ; auxiliar variable to update the deposit at bank later
  let new-cons                                  0                                                ; auxiliar variable to record the quantity consumed
  while [Qtt > 0 and empty? C-Firm-List = false] [                                               ; if there is desired-consumption not satisfied and the list of firms is not empty...
        ask Selected-C-Firm [
        ifelse [Stock] of Selected-C-Firm > Qtt [                                                ; if there are enough available goods at the first firm ...
            set Demand                          Demand + Qtt                                     ; update the demand of Selected-Firm
            set Stock                           Stock  - Qtt                                     ; updated the stock of the firm
            set Sales                           Sales  + Qtt                                     ; update the sales
            set Revenue                         Revenue + Price * Qtt                            ; update the revenue
            set Expenses                        Expenses + Price * Qtt * (1 + tax3)              ; update the auxiliar variable, used later to update the household bank account
            set State-Budget                    State-Budget + Price * Qtt * tax3                ; update State's budget considering consumption taxes (tax3)
            set new-cons                        new-cons + Qtt                                   ; update the auxiliar variable to register the qtt consumed
            set Qtt                             0                                                ; means no need to bought anymore
        ] [                                                                                      ; second situation, buying only part at the first firm
            set Sales                           Sales  +  Stock                                  ; update sales
            set Revenue                         Price  *  Stock + Revenue                        ; update revenue
            set expenses                        Price  *  Stock * (1 + tax3) + expenses          ; update auxiliar variable, later used to update the bank account of the household
            set State-Budget                    State-Budget + Price * Stock * tax3              ; update State income
            set new-cons                        new-cons  +  Stock                               ; update stock
            set Qtt                             Qtt  -  Stock                                    ; update auxiliar variable to register the Qtt consumed, Qtt = Qtt - S
            set Stock                           0                                                ; update stock
        ]
        ]
        set C-Firm-List remove-item listposition C-Firm-List                                     ; we remove the Firm with depleted Stock
        set listposition floor random-gamma 1.2 0.66                                             ; the firm don't take the lowest price, but applies a distribution to the list, because information is not perfect, scale = 1/lambda, shape/lamda 1.2/0.66 CDF: 0 (0) 1 (0.39) 2 (0.66) 3 (0.81) 4 (0.90)
        if listposition >= (length C-Firm-List) [set listposition 0]                             ; in this way we increase the probability of the lowest price, by adding up those cases in which the gamma give us a number larger than the length of the list, the maximum length of the list
        set Selected-C-Firm                    (item listposition C-Firm-List)                   ; auxiliar variable, record the quantity that can be bought, DC/Pa
  ]
  set Consumption                               new-cons                                         ; update the Consume of the households, the real value that the household did consume
  set Bank-Account-Household                    Bank-Account-Household - expenses                ; update the bank account of the households with the remaining money after the purchasing process
end



; FUNCTION-CAPITAL-MARKET


to Function-Capital-Market                                                           ; this function set the rules for the capital goods market
  ask K-Firms [                                                                      ; update K-Firms properties
    set Demand                     0
    set Revenue                    0
    set Sales                      0
  ]
  ask C-Firms [                                                                      ; updating the C-Firms properties
    set Investment                 0
    SubF-UpDate-Investment-Desire                                                    ; call sub-function to update investment
    if Desired-Investment > 0    [ SubF-UpDate-Investment ]                          ; if there is need for investment call the sub-function for investment
  ]
end


to SubF-UpDate-Investment-Desire                                                                                                                                                   ; sub-function with procedure to update the investment need of a C-Firm
  set Desired-Investment           0                                                                                                                                               ; reset variable
  let rand                         random-float 1                                                                                                                                  ; auxiliar sub-function variable
  if  rand < Probability-of-investing [                                                                                                                                            ; check if it is time to invest, because only a fraction of C-Firms (= probability) can invest
      if ticks > 1 [                                                                                                                                                               ; assure that there is no investment in time equal zero
         set Investment-Memory    (Memory-parameter-investment * Investment-Memory) + (1 - Memory-parameter-investment) * Util-Capacity * Capital                                  ; begin of equation of capital motion (eq 10 & Assenza, Gatti, Grazzini et al. 2015, p. 12)
         set Desired-Investment    max list 0 (( (1 / Desired-capacity-utilization-rate) + (Depreciation-of-capital-C / Probability-of-investing) ) * Investment-Memory - Capital) ; end of equation of capital motion (eq 12 & Assenza, Gatti, Grazzini et al. 2015 eq. 5.7)
      ]
  ]
end


to SubF-UpDate-Investment                                       ; this subfunction is called when there is need for investment (asking C-Firms)
  if Desired-Investment > 0 [                                   ; just trying to find product in two diferents K-Firms
     SubF-Visit-K-Firms                                         ; make a list with all K-Firms, take note of their prices
     SubF-Buy-Capital                                           ; sub-function used to bought capital
  ]
end


to SubF-Visit-K-Firms                                                                              ; this subfunction rules the shopping process, visiting the firms
   set K-Firm-List sort-by [ [?1 ?2] -> [Price] of ?1 < [Price] of ?2] K-Firms with [Stock > 0]    ; we build a list with the all the K-Firms ordered by price. We have the prices of all K-Firms because they are few and capital is expensive
end                                                                                                ; C-Firms have no constraits and they have a complete list and make no mistakes in ordering the prices


to SubF-Buy-Capital                                                                                ; asking C-Firms
  let Selected-K-Firm                           first K-Firm-List                                  ; auxiliar variable
  let Qtt                                       Desired-Consumption / [Price] of Selected-K-Firm   ; total number of consumption goods desired
  let expenses                                  0                                                  ; auxiliar variable to update the deposit at bank later
  let new-inv                                   0                                                  ; auxiliar variable to record the quantity of capital adquired by the C-Firm
  while [Qtt > 0 and empty? K-Firm-List = false] [
        ask Selected-K-Firm [
        ifelse [Stock] of Selected-K-Firm > Qtt [                                           ; if there is not available goods at the first firm ...
            set Demand                          Demand + Qtt                                ; update the demand of Selected-Firm
            set Stock                           Stock  - Qtt                                ; updated the stock of the firm
            set Sales                           Sales  + Qtt                                ; update the sales
            set Revenue                         Revenue + Price * Qtt                       ; update the revenue
            set Expenses                        Expenses + Price * Qtt                      ; update the auxiliar variable, used later to update the household bank account
            set new-inv                         new-inv + Qtt                               ; update the auxiliar variable to register the qtt consumed
            set Qtt                             0                                           ; means no need to bought anymore
        ] [                                                                                 ; second situation, buying only part at the first firm
           set Sales                            Sales  +  Stock                             ; update sales
           set Revenue                          Price  *  Stock + Revenue                   ; update revenue
           set expenses                         Price  *  Stock + Expenses                  ; update auxiliar variable, later used to update the bank account of the household
           set new-inv                          new-inv  +  Stock                           ; update stock
           set Qtt                              Qtt  -  Stock                               ; update auxiliar variable to register the Qtt consumed, Qtt = Qtt - S
           set Stock                            0                                           ; update stock
        ]
        ]
        set K-Firm-List                         remove-item 0 K-Firm-List                   ; we remove the Firm with depleted Stock
        set Selected-K-Firm                     first K-Firm-List                           ; auxiliar variable
  ]
  set Investment                                new-inv                                     ; update the Consumption of the households, the real value that the household did consume
  set Bank-Account-Firm                         Bank-Account-Firm - Expenses                ; update the bank account of the households with the remaining money after the purchasing process
end



; FUNCTION-UPDATE-PRODUCTION



to Function-UpDate-Production                                                                           ; this function update the production of Firms
    ask K-Firms [                                                                                       ; updating the production of K-Firms (eqs 15, 17 & 18)
        set Production                  Number-of-Employees * Productivity-of-labor                     ; updating actual production
        set Stock                       Stock * (1 - Depreciation-of-capital-K) + Production            ; updating stock, depreciation-of-capital is currently set at 2%, but K firms do not use capital so this is deterioration of unused stocks
        set Bank-Account-Firm           Bank-Account-Firm + Revenue                                     ; update bank account
        set Delta                       Production - Demand                                             ; variable used to decide new production
        ifelse K-Qtt-Function?        [ set Need-of-Employees  Report-K-Update-Need-of-Employees ]      ; if the switcher at HUD is turned on, you have a complex function below
                                      [ set Need-of-Employees  Demand / Productivity-of-labor    ]      ; ... if the switcher at HUD is turned off, you have the easy way, and the demand determines the number of employees
        ifelse K-UpDate-Price?        [ set Price              Report-K-Update-Price             ]      ; if the switcher at HUD is turned on, you have a complex function below
                                      [ set Price              Wage / Productivity-of-labor      ]      ; ... the switcher at HUD is turned off, you have the easy way
    ]
    ask C-Firms [                                                                                                                   ; updating the production of C-Firms
        set Capital                     Capital - Capital * Util-Capacity * Depreciation-of-capital-C + Investment                  ; update the capital avalible for production process (eq 5.5 of Assenza, Gatti, Grazzini et al. 2015)
        set Production                  min list (Number-of-Employees * Productivity-of-labor) (Capital * Productivity-of-capital)  ; update actual production limited by the available capital
        set Stock                       Production                                                                                  ; the C-Goods are perishable, so the actual stock are equal the production
        set Util-Capacity               Production / (Capital * Productivity-of-capital)                                            ; utilized capacity (eq 5.4 of Assenza, Gatti, Grazzini et al. 2015)
        set Bank-Account-Firm           Bank-Account-Firm + Revenue                                                                 ; update bank account
        set Delta                       Production - Demand                                                                         ; variable used to decide new production level
        ifelse C-Qtt-Function?        [ set Need-of-Employees  Report-C-Update-Need-of-Employees ]                                  ; if the switcher at HUD is turned on, you have a complex function below
                                      [ set Need-of-Employees  Demand / Productivity-of-labor    ]                                  ; ... if the switcher at HUD is turned off, you have the easy way
        ifelse C-UpDate-Price?        [ set Price              Report-C-Update-Price             ]                                  ; if the switcher at HUD is turned on, you have a complex function below
                                      [ set Price              Wage / Productivity-of-labor      ]                                  ; ... if the switcher at HUD is turned off, you have the easy way
    ]
end

to-report Report-C-Update-Price                                                                                                     ; reporting the new price for C-Firms ...
  let N-P                     Price                                                                                                 ; local auxiliar variable
  if  (delta <= 0) and       (Price <  Goods-Price-Level)  [ set N-P  Price * (1 + random-float 1 * Price-adjust-parameter)  ]      ; rule to increase price, approaching it to that of the competitors... the jump is stochastic (eq 5.2 and fig.2 of Assenza, Gatti, Grazzini et al. 2015)
  if  (delta >  0) and       (Price >  Goods-Price-Level)  [ set N-P  Price * (1 - random-float 1 * Price-adjust-parameter)  ]      ; rule to reduce price, approaching it to that of the competitors... the jump is stochastic (eq 5.2 and fig.2 of of Assenza, Gatti, Grazzini et al. 2015)
  report                      N-P                                                                                                   ; report new price
end

to-report Report-K-Update-Price                                                                                                     ; reporting the new price for K-firms
  let N-P                     Price                                                                                                 ; local auxiliar variable
  if (delta <= 0) and        (Price < Capital-Price-Level) [ set N-P Price * (1 + random-float 1 * Price-adjust-parameter )  ]      ; rule to increase price, approaching it to that of the competitors... the jump is stochastic (eq 6.5 of Assenza, Gatti, Grazzini et al. 2015)
  if (delta >  0) and        (Price > Capital-Price-Level) [ set N-P Price * (1 - random-float 1 * Price-adjust-parameter )  ]      ; rule to reduce price, approaching it to that of the competitors... the jump is stochastic (eq 6.5 of Assenza, Gatti, Grazzini et al. 2015)
  report                      N-P                                                                                                   ; report new price
end

to-report Report-C-Update-Need-of-Employees                                                                                                         ; reporting new C-Firm needs for employees (eq 14)
  let N-E                     Number-of-Employees                                                                                                   ; local auxiliar variable
  if (delta <= 0) and        (Price >= Goods-Price-Level)  [ set N-E ((Production - Quantity-adjust-parameter * delta) / productivity-of-labor) ]   ; possible condition to increase actual number of employees and production (eq 5.1 and fig.2 of of Assenza, Gatti, Grazzini et al. 2015)
  if (delta >  0) and        (Price <  Goods-Price-Level)  [ set N-E ((Production - Quantity-adjust-parameter * delta) / productivity-of-labor) ]   ; other possible condition reduce actual number of employees and production (eq 5.1 and fig.2 of of Assenza, Gatti, Grazzini et al. 2015)
  report                      N-E                                                                                                                   ; report new desired employees
end                                                                                                                                                 ; if (delta <= 0) and (Price < Goods-Price-Level) or the contrary the production does not change and the prices adjust

to-report Report-K-Update-Need-of-Employees                                                                                                                  ; reporting new need for a C-firm employees
  let N-E                     Number-of-Employees                                                                                                            ; local auxiliar variable
  if (delta <= 0) and        (Price >=  Capital-Price-Level) [ set N-E ((Production - Quantity-adjust-parameter * delta - Stock) / productivity-of-labor) ]  ; possible condition to change actual number of employees and production, but we substract the stock (eq 6.6 of of Assenza, Gatti, Grazzini et al. 2015)
  if (delta >  0) and        (Price <   Capital-Price-Level) [ set N-E ((Production - Quantity-adjust-parameter * delta - Stock) / productivity-of-labor) ]  ; other possible condition to change actual number of employees (eq 6.6 of of Assenza, Gatti, Grazzini et al. 2015)
  report                      N-E                                                                                                                            ; report new need for employees
end



; UPDATE GLOBALS


to Function-UpDate-Globals                                                                                              ; this function update the globals
  set Unemployment-Rate               1 - ((sum [Number-of-Employees] of Firms) / Number-of-Workers)                    ; update the unemployemnt rate
  set GDP_1                           GDP                                                                               ; store the previous value for the next period
  set GDP                             sum [Production] of Firms - sum [Stock] of K-Firms                                ; update the GDP, sum production - stocks
  if GDP < 0                         [set GDP 1]
  ifelse GDP_1 > GDP [
         set Recession-lenght         Recession-lenght + 1                                                              ; we define here a counter for recessions, to be used later
  ] [
         set Recession-lenght         0                                                                                 ; if there is no recession, the counter is set to zero
  ]
  if ticks > 1 [
     set Growth-Rate                 (GDP - GDP_1) / GDP_1
  ]
  if GDP_1 > GDP and Recession-lenght > 2 [
     set QEasing?   true                                                                                                ; QEasing means Quantitative Easing,and this is a policy that opens unlimited buying of Public Debt
  ]
  if GDP > GDP_1 or Recession-lenght = 0 [
     set QEasing?   false                                                                                               ; QEasing means Quantitative Easing,and this is a policy that opens unlimited buying of Public Debt
  ]
  set Total-Investment                sum [Investment] of C-Firms                                                       ; update investment realized by the C-Firms
  set Total-Consumption               sum [Consumption] of Households                                                   ; update the consumption
  set Total-Loans-Firms               sum [Current-Retail-Loans] of Firms                                               ; update the sum of retail loans, all firms
  set Total-Loans-C-Firms             sum [Current-Retail-Loans] of C-Firms                                             ; update the sum of retail loans for C-Firms
  set Total-Loans-K-Firms             sum [Current-Retail-Loans] of K-Firms                                             ; update the sum of retail loans for K-Firms
  set Total-Deposits-Workers          sum [ Bank-Account-Household ] of Workers                                         ; explicit
  set Total-Deposits-Capitalists      sum [ Bank-Account-Household ] of Capitalists                                     ; explicit
  set Total-Deposits-C-Firms          sum [ Bank-Account-Firm ] of C-firms                                              ; explicit
  set Total-Deposits-K-Firms          sum [ Bank-Account-Firm ] of K-firms                                              ; explicit
  set Total-Deposits_1                Total-Deposits                                                                    ; explicit
  set Total-Deposits                  Total-Deposits-Workers + Total-Deposits-Capitalists + Total-Deposits-C-Firms + Total-Deposits-K-Firms     ; explicit
  set Total-Interest-Payments         sum [ Interest-Pay    ] of Firms                                                  ; explicit
  set Total-Installment-Payments      sum [ Installment-Pay ] of Firms                                                  ; explicit
  set Total-Reserves                  RR * Total-Deposits                                                               ; reserves of the banks deposited at the central bank
  set Goods-Price-Level_1             Goods-Price-Level                                                                 ; store the Goods-Price-Level value for use with adaptative functions
  set Capital-Price-Level_1           Capital-Price-Level                                                               ; store the Capital-Price-Level value for use with adaptative functions
  if sum [Sales] of C-Firms != 0    [ set Goods-Price-Level   sum [ Revenue ] of C-Firms / sum [Sales] of C-Firms ]     ; update the goods price level
  if sum [Sales] of K-Firms != 0    [ set Capital-Price-Level sum [ Revenue ] of K-Firms / sum [Sales] of K-Firms ]     ; update the capital price level
  set Current-Number-of-C-Firms       count C-Firms                                                                     ; current number of C-Firms
  set Current-Number-of-K-Firms       count K-Firms                                                                     ; current number of K-Firms
  set Current-Number-of-Workers       count Workers                                                                     ; current number of Workers
end



; FUNCTION-CASH-FLOWS AND DIVIDENDS


to Function-Cash-Flows-Dividends                                                                 ; this function sets the payment of dividends and cash-flows adjustments
  ask C-Firms [                                                                                  ; Cash flow of C-Firms
      set PBT     Revenue + Bank-Account-Firm * eta3 - Number-of-Employees * Wage - Desired-Investment * Capital-Price-Level - (annuity-rate + eta0) * Current-Retail-Loans  ; PBT is profits before taxes
      let div     0                                                                              ; auxiliar variable
      if Dividends-Payments? [                                                                            ; HUD control for activating dividends payments
         if PBT > 0 [                                                                            ; if the PBT is positive, the firm pay dividends and taxes
            set Profit                              PBT * (1 - tax2)                             ; the profit is reduced by a tax
            set State-Budget                        State-Budget + PBT * tax2                    ; the State collects the taxes
            set Dividend                            Dividend-payout-ratio * Profit               ; if there was a positive profit, setup dividend, the income of capitalists
            set Bank-Account-Firm                   Bank-Account-Firm - Dividend                 ; reduce the bank account the dividend which will be payed to the capitalist
            set Div                                 Dividend                                     ; auxiliar variable bring the div to the capitalist later
            ask one-of my-out-owners [                                                           ; calling the owner of this firm, where my-out-<breed> calls "all links you can use to travel from this node"
                ask other-end [                                                                  ; at other end of the link is the capitalist
                    set Current-Income              div * (1 - tax1)                             ; update the current income, substracting the income taxes
                    set Bank-Account-Household      Bank-Account-Household + Current-Income      ; update the household bank account of the capitalist
                    set State-Budget                State-Budget + div * tax1                    ; the State collects the income taxes on this dividend
               ]                                                                                 ; end asking capitalist
           ]                                                                                     ; end asking link
       ]                                                                                         ; end PBT > 0
       if PBT <=  0 [                                                                            ; if the profit is negative (losses) the problems appear
          set Profit                                PBT                                          ; if the profit was zero or negative, the tax is zero
          ifelse Bank-Account-Firm >= abs Profit [
                 set Bank-Account-Firm              Bank-Account-Firm - abs Profit               ; the losses reduce the bank-account of the firm
          ] [                                                                                    ; if the losses are larger than the bank-account of the firm a bakruptcy process starts
             set Overdraft                          abs Profit - Bank-Account-Firm               ; overdraft are losses not covered by the money available at the bank account
             set Bank-Account-Firm                  0                                            ; the losses deplete the bank account of the firm
             ask one-of my-out-owners [                                                          ; calling the owner of this firm, where my-out-<breed> calls "all links you can use to travel from this node"
                 ask other-end [                                                                 ; at other end of the link is the capitalist
                     ifelse Bank-Account-Household >=   Overdraft [
                            set Current-Income             0                                     ; update the current income of the capitalist
                            set Bank-Account-Household     Bank-Account-Household - Overdraft    ; the capitalist of the firm has to recapitalize it using his own bank-account
                 ] [

                     set Bankrupt?                          true
                     SubF-Capitalist-Damnation
            ]                                                                                     ; end the overdraft and liquidation process
                 ]                                                                                         ; end the negative profit process
            ]                                                                                         ; end the dividends distribution process
        ]
      ]
    ]
    ]                                                                                              ; end asking C-Firms

  ask K-Firms [                                                                                  ; Cash flow of K-Firms
    set PBT     Revenue + Bank-Account-Firm * eta3 - Number-of-Employees * Wage - (annuity-rate + eta0) * Current-Retail-Loans     ; PBT is profits before taxes
    let div     0                                                                                ; auxiliar variable
    if Dividends-Payments? [                                                                              ; HUD control for activating dividends payments
       if PBT > 0 [                                                                              ; if the profit is positive, the firm pay dividends and taxes
          set Profit                             PBT * (1 - tax2)                                ; the profit is reduced by a tax
          set State-Budget                       State-Budget + PBT * tax2                       ; the State collects the taxes
          set Dividend                           Dividend-payout-ratio * Profit                  ; if there was a positive profit, setup dividend, the income of capitalists
          set Bank-Account-Firm                  Bank-Account-Firm - Dividend                    ; reduce the bank account by the dividend which will be payed to the capitalist
          set div                                Dividend                                        ; auxiliar variable bring the div to the capitalist later
          ask one-of my-out-owners [                                                             ; calling the owner of this firm, where my-out-<breed> calls "all links you can use to travel from this node".
               ask other-end [                                                                   ; at other end of the link is the capitalist
                   set Current-Income            div * (1 - tax1)                                ; update the current income, substracting the income taxes
                   set Bank-Account-Household    Bank-Account-Household + Current-Income         ; update the household bank account
                   set State-Budget              State-Budget + div * tax1                       ; the State collects the taxes
               ]
          ]
       ]
       if PBT <=  0 [                                                                            ; if the profit before taxes is negative (losses) the problems appear
          set Profit                             PBT                                             ; if the profit was zero or negative, the tax is zero
           ifelse Bank-Account-Firm >= abs Profit [
                  set Bank-Account-Firm          Bank-Account-Firm - abs Profit                  ; the losses reduce the bank-account of the firms
           ] [                                                                                   ; if the losses are larger than the bank-account of the firm
           set Overdraft                         abs Profit - Bank-Account-Firm                  ; overdraft are losses not covered by the money available at the Bank Accounts
             set Bank-Account-Firm               0                                               ; the losses deplete the bank-account of the firms
             ask one-of my-out-owners [                                                          ; calling the owner of this firm, where my-out-<breed> calls "all links you can use to travel from this node".
                 ask other-end [                                                                 ; at other end of the link is the capitalist
                     let Unassumed-Losses                  Overdraft - Bank-Account-Household    ; the losses not covered by the capitalist's money lead to the dismantlement of the firm
                     if Unassumed-Losses <= 0 [
                        set Current-Income                 0                                     ; update the current income of the capitalist
                        set Bank-Account-Household         Bank-Account-Household - Overdraft    ; the capitalist of the firm has to recapitalize it using his own bank-account
                     ]                                                                           ; end of the procedure of heavy losses in which the capitalist is called for sustaining the firm
                     if Unassumed-Losses > 0 [
                        set Bankrupt?                  true
                        SubF-Capitalist-Damnation
                      ]                                                                          ; end procedure previous to bankruptcy
                ]                                                                                ; end ask capitalist
             ]                                                                                   ; end ask one-of my-out-owners
           ]                                                                                     ; end the overdraft and liquidation process
       ]                                                                                         ; end the negative profit process
       ]                                                                                         ; end the dividends distribution process
  ]                                                                                              ; end asking K-Firms


  ask Banks [                                                                                                          ; Now lets see the cash flow of Banks
    let div                             0                                                                              ; auxiliar variable
    set Clients-Accounts-C-Firms        sum [ Bank-Account-Firm ] of out-banker-neighbors with [C-Firm? = true ]       ; bank accounts of the C-Firms at a particular bank
    set Clients-Accounts-K-Firms        sum [ Bank-Account-Firm ] of out-banker-neighbors with [C-Firm? = false]       ; bank accounts of the K-Firms at a particular bank
    set Clients-Accounts-Firms          Clients-Accounts-C-Firms  + Clients-Accounts-K-Firms                           ; bank accounts of Firms at a particular bank
    set Clients-Accounts-Workers        sum [ Bank-Account-Household ] of out-banker-neighbors with [Worker? = true ]  ; bank accounts of the Workers at a particular bank
    set Clients-Accounts-Capitalists    sum [ Bank-Account-Household ] of out-banker-neighbors with [Worker? = false]  ; bank accounts of the Capitalists at a particular bank
    set Clients-Accounts-Households     Clients-Accounts-Capitalists + Clients-Accounts-Workers                        ; bank accounts of the Households at a particular bank
    set Clients-Accounts-at-the-Bank_1  Clients-Accounts-at-the-Bank
    set Clients-Accounts-at-the-Bank    Clients-Accounts-Firms + Clients-Accounts-Households                           ; total bank accounts of firms and households at a particular bank
    set Clients-Loans-C-Firms           sum [ Current-Retail-Loans ] of out-banker-neighbors with [C-Firm? = true]     ; loans to C-Firms of a particular bank
    set Clients-Loans-K-Firms           sum [ Current-Retail-Loans ] of out-banker-neighbors with [C-Firm? = false]    ; loans to K-Firms of a particular bank
    set Clients-Loans-with-the-Bank_1   Clients-Loans-with-the-Bank
    set Clients-Loans-with-the-Bank     Clients-Loans-K-Firms + Clients-Loans-C-Firms                                  ; total loans to Firms of a particular bank
    set Number-of-Firm-Clients          count out-banker-neighbors with [shape = "factory"]                            ; total number of firm clients (with accounts or loans) with this bank
    set Number-of-Household-Clients     count out-banker-neighbors with [shape = "person"]                             ; total number of household clients (with accounts) with this bank
    set Number-of-Clients               Number-of-Firm-Clients + Number-of-Household-Clients                           ; total number of clients (any kind) with this bank
    set Shares                          Own-Bank-Capital / (Current-Number-of-C-Firms + Current-Number-of-K-Firms)     ; the shares of the bank, one for every capitalist
    set I-Received                      sum [ Interest-Pay         ] of out-banker-neighbors with [shape = "factory"]  ; total interest payments of firms to a particular bank
    set D-Received                      sum [ Installment-Pay      ] of out-banker-neighbors with [shape = "factory"]  ; total installment payments of firms to a particular bank
    set Own-Reserves                    RR * Clients-Accounts-at-the-Bank                                              ; reserves of the banks deposited at the central bank
    set PBT                             I-Received + Own-Reserves * eta4 - Central-Bank-Loans * (eta2 + annuity-rate)  ; the profit income minus expenditures
    if Dividends-Payments? [                                                                                                    ; HUD control for activating dividends payments
       ifelse Profit > 0 [                                                                                             ; if the profit is positive, the firm pay dividends and taxes
           set Profit                                 PBT * (1 - tax2)                                                 ; the profit is reduced by a tax
           set State-budget                           State-budget + Profit * tax2                                     ; the State collects the taxes
           set Dividend                               Dividend-payout-ratio * Profit                                   ; if there was a positive profit, setup dividend, the income of capitalists
           set Own-Bank-Capital                       Own-Bank-Capital + Profit * (1 - tax2) - Dividend                ; reduce the bank account the dividend which will be paid to the capitalist
           set Div                                    Dividend / (Current-Number-of-C-Firms + Current-Number-of-K-Firms) ; calculate dividend as part of the total profit divided by capitalists (everyone have a share)
           ask Capitalists [                                                                                           ; call each of one of the capitalist
               let Number-of-Capitalists              Current-Number-of-C-Firms + Current-Number-of-K-Firms            ; total number of Capitalists
               set Current-Income                     Current-Income + div * (1 - tax1)                                ; increase the income of the capitalist with this new dividend
               set Bank-Account-Household             Bank-Account-Household + div * (1 - tax1)                        ; update the bank account of the capitalist
               set State-Budget                       State-Budget + tax1 * div                                        ; the State gets the income taxes
           ]
           ]  [                                                                                                        ; if the profit is negative (losses) the problems start to show up
           ifelse Own-Bank-Capital >= abs Profit [                                                                     ; the bank has liquidity for compensating the loss
                  set Own-Bank-Capital                Own-Bank-Capital - abs Profit                                    ; the losses reduce the Own-Bank-Capital of the bank
                  ask Capitalists [
                      set Current-Income              Current-Income + 0                                               ; update the current income of the capitalist, but we have to consider this is a secondary source of income for them
                  ]
           ] [                                                                                                         ; if the losses are larger than the Own-Bank-Capital
                  set Overdraft                       abs Profit - Own-Bank-Capital                                    ; overdraft are the losses not covered by the Own-Bank-Capital
                  set Own-Bank-Capital                0                                                                ; the losses deplete the Own-Bank-Capital and then...
                  ifelse Do_We_Have_State? [                                                                           ; untenable losses... if there exist a government and taxes... (we are asking banks)
                         SubF-Call-Rescue-From-State                                                                   ; ...the State goes to rescue the retail bank
                  ] [
                         SubF-Go-Bankruptcy-Banks                                                                      ; but if the State does not exist, the retail bank starts a bankruptcy process
                  ]
           ]                                                                                                           ; end procedure previous to bankruptcy
           ]                                                                                                           ; end of the managing losses process
     ]                                                                                                                 ; end of dividends procedure for banks
   ]                                                                                                                   ; end of ask banks
end

to SubF-Capitalist-Damnation
   ask Capitalists with [Bankrupt? = true] [
       set Current-Income               0                                     ; update the current income of the capitalist/owner
       set Bank-Account-Household       Initial-households-personal-assets    ; update the bank account of the capitalist/owner
       set Worker?                      true                                  ; the capitalist becomes a worker... life is unfair
       set Employed?                    false                                 ; this worker is unemployed at the start
       set color                        yellow                                ; now you are a worker and look like one
       set size                         1.25
       set Human-Wealth                 1
       set Desired-Consumption          0
       set Consumption                  0
       set C-Firm-List                  [ ]
       ask in-owner-neighbors           [ die ]                                ; and the former firm he/she owned vanishes
       set Current-Number-of-C-Firms    Current-Number-of-C-Firms - 1          ; we correct the number of C-Firms in this economy
   ]                                                                           ; end procedure previous to bankruptcy
end

to SubF-Call-Rescue-From-State
   if Do_We_Have_Central-Bank? = true and QEasing? = true [                                                                      ; if the Central Bank exists and it buys public debt without limits, all the deficit is converted to government bonds sold to the CB
      set State-Budget                            State-Budget -  Initial-equity-of-the-bank - Overdraft                         ; in this case the government pays the recapitalization of the bank and the unpaid overdraft
      set Own-Bank-Capital                        Initial-equity-of-the-bank
      set Profit                                  0
   ]
   if State-Budget >= (Initial-equity-of-the-bank + Overdraft) and Do_We_Have_Central-Bank? = false [                            ; if the Central Bank does not exist, the State helps the bank only if the budget has enough resources
      set State-Budget                            State-Budget -  Initial-equity-of-the-bank - Overdraft                         ; in this case the government pays the recapitalization of the bank and the unpaid overdraft
      set Own-Bank-Capital                        Initial-equity-of-the-bank
      set Profit                                  0
   ]
   if State-Budget >= (Initial-equity-of-the-bank + Overdraft) and Do_We_Have_Central-Bank? = true and QEasing? = false [        ; if the Central Bank exists but the Quantitative Easing policy is not activated, the State helps the bank only if the budget has enough resources
      set State-Budget                            State-Budget -  Initial-equity-of-the-bank - Overdraft                         ; in this case the government pays the recapitalization of the bank and the unpaid overdraft
      set Own-Bank-Capital                        Initial-equity-of-the-bank
      set Profit                                  0
   ]
   if State-Budget <= (Initial-equity-of-the-bank + Overdraft) and Do_We_Have_Central-Bank? = false [                            ; if the Central Bank does not exist and the State budget has not enough resources...
      SubF-Go-Bankruptcy-Banks                                                                                                   ; ... the bankruptcy process is started
   ]
   if State-Budget <= (Initial-equity-of-the-bank + Overdraft) and Do_We_Have_Central-Bank? = true and QEasing? = false [        ; if the Central Bank exists but the Quantitative Easing policy is not activated and the State budget hasn't enough resources...
      SubF-Go-Bankruptcy-Banks                                                                                                   ; ... the bankruptcy process is started
   ]
end



to SubF-Go-Bankruptcy-Banks                                                                                               ; this subfunction rules the process for the Bank bankruptcy and rebuilding using capitalits' resources. We ask banks here.
   set B-Broke                                    B-Broke + 1                                                             ; increase the global parameter that counts the broke status
   let Original-Shares                            Initial-equity-of-the-bank / (Current-Number-of-C-Firms + Current-Number-of-K-Firms)    ; the original price of a share, before any serious problem appears
   let Bank-Owners                                Capitalists with [Bank-Account-Household >= Original-Shares]            ; this implies a bail-in, because the capitalits will be forced to buy again one share
   let Broken-Capitalists                         Capitalists with [Bank-Account-Household <  Original-Shares]            ; excluded capitalists are those unable to restitute their share in the banking sector, and their number is adjusted below
   let New-Shares                                 Original-Shares                                                         ; we prepare this instrumental variable for repeated use later
   let New-Shares_1                               0                                                                       ; we prepare this instrumental variable for repeated use later
   while [New-Shares > New-Shares_1] [
         set New-Shares_1                         New-Shares
         set New-Shares                           Initial-equity-of-the-bank / (count Bank-Owners)                        ; the capitalists unable to afford a new share are excluded, but as a consecuence, the price per share increases
         set Bank-Owners                          Capitalists with [Bank-Account-Household >= New-Shares]                 ; this implies a bail-in
         set Broken-Capitalists                   Capitalists with [Bank-Account-Household <  New-Shares]                 ; broken capitalists are unable to restitute their share in the banking sector, and they will lose their "capitalist" status
   ]
   ask  Bank-Owners [
        set Bank-Account-Household                Bank-Account-Household - New-Shares                                     ; update the bank account of the capitalist, reducing the account due to the money used to rebuild the bank
   ]
   ask  Broken-Capitalists [                                                                                              ; the capitalists unable to pay for the bail-in will be broken
        set Bank-Account-Household                0                                                                       ; the capitalist unable to pay for the new shares get broken
        set Worker?                               true                                                                    ; the capitalist is broken, and he/she becomes a poor worker
        set Number-of-Workers                     Number-of-Workers + 1
        set color                                 yellow                                                                  ; just for fun, not necessary for the model, used only for display in HUD netlogo
        set Human-Wealth                          1                                                                       ; initial value for Human Wealth, utilized to calculate the consumption process
        set Current-Income                        0                                                                       ; intitial income
        set Bank-Account-Household                Initial-households-personal-assets                                      ; initial value of deposits at the banks, a vital minimum
        set Desired-Consumption                   0                                                                       ; initial value for desire Consumption
        set Consumption                           0                                                                       ; initial value for Consumption
        set C-Firm-List                          [ ]                                                                      ; setup for the list which will be used to colect the C-Firms visited
        set Employed?                             false                                                                   ; the ex-capitalist is now an unemployed worker and he has to look for an employment... life is hard...
        ask one-of my-in-owners [                                                                                         ; we identify the firm (K or C) previously owned by the broken capitalist... my-in-breed means "all links that you can use to travel to this node".
            ask end1                            [ set Owner? false ]                                                      ; ...the firm at the other end of the link now is deprived of a proper owner
            die                                                                                                           ; the link representing ownership dies, the problem now is looking for a new owner for that orphan firm...
        ]
   ]
   let Newcapitalist                              n-of 1 Workers with-max [ Bank-Account-Household ]                      ; ...the orphan C-Firm or K-Firm needs one new owner, so we ask the worker with the highest savings to buy the Firm and become a capitalist
   ask Newcapitalist [                                                                                                    ; before we proceed to transform a worker into a capitalist, we broke his/her current labor contract with a (different) firm
       ask one-of my-in-hireds [                                                                                          ; the only link of this worker, future capitalist, is the labor contract
           ask other-end [                                                                                                ; we contact with the firm for which this worker is employed
               set Vacancy                        Vacancy + 1                                                             ; the worker leaves, so the firm adjust the vacancies adding an open job to the record
               set Number-of-Employees            Number-of-Employees - 1                                                 ; the firm losses one worker, which now is a capitalist unwilling to work
           ]
           die                                                                                                            ; the link, the labor contract, dies or disappear
       ]
       set shape                                 "person"                                                                 ; the metamorphosis of this person is compplete now... he comes to a new (better) life...
       set size                                   1.25                                                                    ; just for fun, not necessary for the model, used only for display in HUD Netlogo
       set color                                  white                                                                   ; he becomes white as well, the color of wealthy capitalists
       set Worker?                                false                                                                   ; false only means he/she is now a happy capitalist
       set Number-of-Workers                      Number-of-Workers - 1
       create-owner-from                          one-of Firms with [Owner? = false] [set hidden? true]                   ; this command creates a link between the orphan firm and the new capitalist and in this way one will know each other for all the purposes
       ask out-owner-neighbors                  [ set Owner? true ]                                                       ; we identify the (C or K) firm previously owned by the broken capitalist and now owned by the lucky new capitalist...
   ]
end


; FUNCTION-CREDIT-MARKET: CREDIT AND BANKRUPTCY PROCEDURES


to Function-Credit-Market                                                  ; this function rules the credit market system

   SubF-Call-Matrix-Formation                                              ; function to set the matrix used in R to calculate the life spam of firms

   set C-Broke                         0                                   ; global variable for counting the total number of C-firms that went broke this period
   set K-Broke                         0                                   ; global variable for counting the total number of K-firms that went broke this period

   set Total-Clients-Accounts           sum [ Clients-Accounts-at-the-Bank   ] of Banks                           ; this should be equal to Total-Deposits
   set Total-Client-Loans              sum [ Clients-Loans-with-the-Bank    ] of Banks                           ; this should be equal to Total-Loans-Firms

   ask Firms [                                                             ; firms update their status with the retail bank
       SubF-Update-Installment                                             ; to update the installment
       SubF-Update-Cash-Need                                               ; to update firm cash need
       SubF-Update-Assets                                                  ; to update assets value
       SubF-Update-Leverage                                                ; to update the leverage, all this is used to rule interest rate practiced later
   ]

   ask C-Firms [
       SubF-Ask-for-C-Credit                                               ; checks if this C-firm needs credit, it will be asked in this subfunction
   ]

   ask K-firms [
       SubF-Ask-for-K-Credit                                               ; checks if this K-Firm needs credit, it will be asked in this subfunction
   ]

   ask Banks [                                                             ; the retail bank update the balance after the new was created
       if Do_We_Have_Central-Bank? [
          SubF-Ask-for-Credit-to-Central-Bank                              ; the retail bank asks for credits to the Central Bank if reserves fall bellow the prescribed level
       ]
       set Assets                      max list 0 (Own-Reserves + Clients-Loans-C-Firms + Clients-Loans-K-Firms + Own-Bank-Capital)
       set Liabilities                 max list 0 (Clients-Accounts-Workers + Clients-Accounts-Capitalists + Clients-Accounts-C-Firms + Clients-Accounts-K-Firms + Central-Bank-Loans)
       set Bank-Equity                 Assets - Liabilities
   ]

   set Total-Central-Bank-Loans    sum [Central-Bank-Loans] of Banks       ; global variable that calculates the sum of all Central-Bank-Loans to retail banks

   if C-Broke-Procedure? [                                                 ; switch check if the user want the broke procedure
      ask C-Firms [                                                        ; this rules is for C-firm broke
          if (Cases-of-Bankruptcy > T-Ban) [                               ; check if this firm did not pay their debts with a bank, where T-Ban (= 1, 2 o 4) are the Firm's number of periods unable to make payments before going to bankrupcy
              SubF-Go-Bankruptcy-C-Firms                                   ; subfunction containing the bankruptcy process
          ]
      ]
   ]

   if K-Broke-Procedure? [                                                 ; the same as the C-firm, check above for details
      ask K-Firms [
          if (Cases-of-Bankruptcy > T-Ban) [                               ; check if this firm did not pay their debts with a bank, where T-Ban (= 1, 2 o 4) are the Firm's number of periods unable to make payments before going to bankrupcy
              SubF-Go-Bankruptcy-K-Firms                                   ; subfunction containing the bankruptcy process
          ]
      ]
   ]
end



to SubF-Call-Matrix-formation                                                                     ; subfunction that manage the R routines
  ask C-Firms [
    set Matrix-C-Firms-Bankruptcy matrix:set-and-report Matrix-C-Firms-Bankruptcy 1 Mjc Leverage  ; update the second row of the matrix with information about C-Firms: every (1, Mjc) element is replaced by Leverage
    ifelse (Cases-of-Bankruptcy < T-Ban)                                                          ; update a position with 0 if the firm is alive, and 1 if the firm went bankrupt
    [ set Matrix-C-Firms-Bankruptcy matrix:set-and-report Matrix-C-Firms-Bankruptcy 0 Mjc 0 ]     ; 0 is the first row, 1 the second row, etc. Mjc is the columns
    [ set Matrix-C-Firms-Bankruptcy matrix:set-and-report Matrix-C-Firms-Bankruptcy 0 Mjc 1 ]     ; 0 is the first row, 1 the second row, etc. Mjc is the columns
    set Mjc Mjc + 1 if Mjc > (Time-Window1 - 1) [ set Mjc 0 ]                                     ; move the matrix element to the next position, so the replacements in the matrix are sequential, and the time-window controls how many observations we'll have
  ]

  ask K-Firms [
    set Matrix-K-Firms-Bankruptcy matrix:set-and-report Matrix-K-Firms-Bankruptcy 1 Mjk Leverage  ; update the matrix with information about the K-firms
    ifelse (Cases-of-Bankruptcy < T-Ban)                                                          ; update a position with 0 if the firm is alive, and 1 if the firm went bankruptcy
    [ set Matrix-K-Firms-Bankruptcy matrix:set-and-report Matrix-K-Firms-Bankruptcy 0 Mjk 0 ]
    [ set Matrix-K-Firms-Bankruptcy matrix:set-and-report Matrix-K-Firms-Bankruptcy 0 Mjk 1 ]
    set Mjk Mjk + 1 if Mjk > (Time-Window1 - 1) [ set Mjk 0 ]                                     ; move the matrix the next position, and the time-window controls how many observations we'll have
  ]

  let A matrix:get-row Matrix-C-Firms-Bankruptcy 0                                                ; create an object with the first row of the matrix, which contains 0s and 1s, if barkrupt or not
  let B matrix:get-row Matrix-C-Firms-Bankruptcy 1                                                ; create an object with the second row of the matrix, which contains the Leverage for every C-Firm

  r:put   "a" A                                                                                   ; call the R and put there the object A
  r:put   "b" B                                                                                   ; call the R and put there the object B
  r:eval  "c <- glm(a ~ b, family = binomial)"                                                    ; call R and ask to solve the logistic function with the inputs A and B, gls stands for "generalized linear model", and binomial implies logit link between "a" and "b"

  let C  r:get "c$coefficients"                                                                   ; call R and the coefficients into object C
  set Beta_1-C-Firms  first C                                                                     ; save the beta-1 for C-firms
  if  Beta_1-C-Firms < (- 25)      [set Beta_1-C-Firms (- 25)]                                    ; assure value into a meaninful range, because the first period there is not enough information
  set Beta_2-C-Firms  last  C                                                                     ; save the beta-2 for C-firms
  if (Beta_2-C-Firms > 0) = false  [ set Beta_2-C-Firms    0 ]                                    ; assure value into a meaninful range, because the first period there is not enough information
  if  Beta_2-C-Firms > 25          [ set Beta_2-C-Firms   25 ]                                    ; assure value into a meaninful range, because the first period there is not enough information

  set A matrix:get-row Matrix-K-Firms-Bankruptcy 0                                                ; create an object with the first row of the matrix, which contains 0s and 1s, if barkrupt or not
  set B matrix:get-row Matrix-K-Firms-Bankruptcy 1                                                ; create an object with the seccond row of the matrix, which contains the Leverage for every K-Firm

  r:put   "a" A                                                                                   ; call the R and put there the object A
  r:put   "b" B                                                                                   ; call the R and put there the object B
  r:eval  "c <- glm(a ~ b, family = binomial)"                                                    ; call R and ask to solve the logistic function with the inputs A and B, gls stands for "generalized linear model", and binomial implies logit link between "a" and "b"

  set C  r:get "c$coefficients"                                                                   ; the same as for the C-Firms, look above for details
  set Beta_1-K-Firms  first C
  if  Beta_1-K-Firms < (- 25)     [ set Beta_1-K-Firms  (- 25) ]
  set Beta_2-K-Firms  last  C
  if (Beta_2-K-Firms > 0) = false [ set Beta_2-K-Firms       0 ]
  if  Beta_2-K-firms > 25         [ set Beta_2-K-Firms      25 ]

end


; SUBFUNCTIONS FOR UPDATING THE SYSTEM STATE


to SubF-Update-Installment                                                           ; this subfunction update the installment debt with retail banks (we are asking Firms)
  set  Interest-Pay                     0                                            ; reset parameter
  set  Installment-Pay                  0                                            ; reset parameter, the repayment schedule
  if Current-Retail-Loans > 0 [                                                      ; check if exist loan
     ifelse Bank-Account-Firm >= ((annuity-rate + eta0) * Current-Retail-Loans) [    ; check if there is money enough to pay the debts
           set Interest-Pay             eta0 * Current-Retail-Loans                  ; update the parameter with the value will be payed
           set Bank-Account-Firm        Bank-Account-Firm  - interest-pay            ; update parameter
           set Installment-Pay          annuity-rate * Current-Retail-Loans          ; update parameter
           set Bank-Account-Firm        Bank-Account-Firm  - Installment-Pay         ; update parameter
           set Current-Retail-Loans     Current-Retail-Loans - Installment-Pay       ; update parameter
           set Cases-of-Bankruptcy      0                                            ; as the firm pay debts this period, reset the parameter for bankruptcy
           ] [
           set Cases-of-Bankruptcy      Cases-of-Bankruptcy  +  1                    ; if the firm did not had money she will be closer and closer to bankruptcy
           ]                                                                         ; bankruptcy happens when the firm is unable to pay its debt to the bank
  ]
end


to SubF-Update-Cash-Need                                                             ; subfuntion used for updating the need for cash (loans) before production takes place (need for liquidity)... (we are asking Firms)
  ifelse C-Firm? [
         set Need-Cash         max list 0 (Need-Cash + Number-of-Employees * Wage + Desired-Investment * Capital-Price-Level + (annuity-rate + eta0) * Current-Retail-Loans - Bank-Account-Firm)
  ] [
         set Need-Cash         max list 0 (Need-Cash + Number-of-Employees * Wage + (annuity-rate + eta0) * Current-Retail-Loans - Bank-Account-Firm)
  ]
end



to SubF-Update-Assets                                                                 ; subfunction to update the assets, which differ from C-Firm to K-Firm type (we are asking Firms)
       ifelse C-Firm? [
              set Assets            max list 0 (Stock * Price  +  Bank-Account-Firm  + Capital * Capital-Price-Level)
              set Liabilities       max list 0 (Current-Retail-Loans)
              set Equity-C-Firms    Assets - Liabilities
       ] [
              set Assets            max list 0 (Stock * Price  +  Bank-Account-Firm)
              set Liabilities       max list 0 (Current-Retail-Loans)
              set Equity-K-Firms    Assets - Liabilities
       ]
end



to SubF-Update-Leverage                                                               ; updating the leverage of a firm (we are asking Firms)
   if Need-Cash > 0                  [ set Leverage   (Current-Retail-Loans + Need-Cash) / (Assets + Current-Retail-Loans + Need-Cash) ]
   if (Current-Retail-Loans = 0)     [ set Leverage 0 ]
end



; CREDIT PROCESS



to SubF-Ask-for-C-Credit                                                                                                           ; this subfunction is called when there is need for a loan, we ask C-Firms

   ; how much money the nearest bank may offer as a loan, see eq. 8.11 in Assenza, Gatti, Grazzini et al. 2015
   let Available-Credit-Assenza1  (Bank-loss-parameter * ([Bank-Equity] of My-Nearest-Bank - Probability-of-Bankruptcy * [Clients-Loans-with-the-Bank] of My-Nearest-Bank)) / Probability-of-Bankruptcy
   ; how much money the nearest bank may offer as a loan, see eq. 27 of Elder Silva's document
   let Available-Credit-Assenza2   Bank-loss-parameter * ([Bank-Equity] of My-Nearest-Bank * B-Lev * Risk - [Clients-Loans-with-the-Bank] of My-Nearest-Bank)
   set Available-Credit-Assenza    min (list Available-Credit-Assenza1 Available-Credit-Assenza2)
   let Available-Credit            Available-Credit-Assenza

   if (Need-Cash > 0) and ([Available-Credit-Basel] of My-Nearest-Bank > 0) and (Available-Credit >= Need-Cash) and (Cases-of-Bankruptcy < T-Ban) [                                      ; check if the need of a firm is lower then the available credit
      set Current-Retail-Loans           Current-Retail-Loans  +  Need-Cash                                                        ; updating the loans parameter
      set Bank-Account-Firm              Bank-Account-Firm +  Need-Cash                                                            ; updating the bank account with the money received
      ask My-Nearest-Bank [
          set Available-Credit-Basel     Available-Credit-Basel - Need-Cash
      ]
      set Need-Cash                      0
   ]
   if (Need-Cash > 0) and ([Available-Credit-Basel] of My-Nearest-Bank > 0) and (Available-Credit < Need-Cash) and (Cases-of-Bankruptcy < T-Ban) [                                       ; check if the need of a firm is lower then the available credit
      set Current-Retail-Loans           Current-Retail-Loans + Available-Credit                                                   ; the nearest bank offer a loan up to the limit
      set Bank-Account-Firm              Bank-Account-Firm +  Available-Credit                                                     ; updating the bank account with the money received
      ask My-Nearest-Bank [
          set Available-Credit-Basel     Available-Credit-Basel - Available-Credit
      ]
      set Need-Cash                      Need-Cash - Available-Credit
   ]
   ifelse Update-Interest?             [ SubF-Update-Interest-Retail-C ] [ set eta0  taylor-interest-rate ]                        ; call a subfunction to update the interest rate of this firm
end


to SubF-Update-Interest-Retail-C                                                                                                   ; update the interest rate of a C-firm
   set Probability-of-Bankruptcy      (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * leverage))))                            ; save the probability of dying of this C-Firm, and for the Betas see above
   set T-expected                     (1 / Probability-of-Bankruptcy)                                                              ; calculate the life expectation of this firm (eq 24)
   let Numerator                      (1 + (taylor-interest-rate / annuity-rate))                                                  ; local auxiliar variable, first part to calculate the new interest rate
   let Denominator                    ((1 - (1 - annuity-rate) ^ (T-expected + 1)) / annuity-rate)                                 ; // // second part of the new interest rate
   let new-int                        Bank-gross-mark-up * ((numerator / denominator) - annuity-rate)                              ; calculating the new interest rate (eq 26 or 8.8 in Assenza, Gatti, Grazzini et al. 2015)
   set eta0                          (eta0  * (Current-Retail-Loans - Need-Cash) + new-int * Need-Cash) / Current-Retail-Loans     ; update the interest rate as mean of the new and the later interest rate
end




to SubF-Ask-for-K-Credit                                                                                                                                 ; this subfunction is called when there is need for a loan, C-Firm type

  ; how much money the nearest bank may offer as a loan, see eq. 8.11 in Assenza, Gatti, Grazzini et al. 2015
   let Available-Credit-Assenza1  (Bank-loss-parameter * ([Bank-Equity] of My-Nearest-Bank - Probability-of-Bankruptcy * [Clients-Loans-with-the-Bank] of My-Nearest-Bank)) / Probability-of-Bankruptcy
   ; how much money the nearest bank may offer as a loan, see eq. 27 of Elder Silva's document
   let Available-Credit-Assenza2   Bank-loss-parameter * ([Bank-Equity] of My-Nearest-Bank * B-Lev * Risk - [Clients-Loans-with-the-Bank] of My-Nearest-Bank)
   set Available-Credit-Assenza    min (list Available-Credit-Assenza1 Available-Credit-Assenza2)
   let Available-Credit            Available-Credit-Assenza

   if (Need-Cash > 0) and ([Available-Credit-Basel] of My-Nearest-Bank > 0) and (Available-Credit >= Need-Cash) and (Cases-of-Bankruptcy < T-Ban) [      ; check if the need of a firm is lower then the available credit
      set Current-Retail-Loans           Current-Retail-Loans  +  Need-Cash                                                                              ; updating the loans parameter
      set Bank-Account-Firm              Bank-Account-Firm +  Need-Cash                                                                                  ; updating the bank account with the money received
      ask My-Nearest-Bank [
          set Available-Credit-Basel     Available-Credit-Basel - Need-Cash
      ]
      set Need-Cash                      0
   ]
   if (Need-Cash > 0) and ([Available-Credit-Basel] of My-Nearest-Bank > 0) and (Available-Credit < Need-Cash) and (Cases-of-Bankruptcy < T-Ban) [       ; check if the need of a firm is lower then the available credit
      set Current-Retail-Loans           Current-Retail-Loans + Available-Credit                                                                         ; the nearest bank offer a loan up to the limit
      set Bank-Account-Firm              Bank-Account-Firm + Available-Credit                                                                            ; updating the bank account with the money received
      ask My-Nearest-Bank [
          set Available-Credit-Basel     Available-Credit-Basel - Available-Credit
      ]
      set Need-Cash                      Need-Cash - Available-Credit
   ]
   ifelse Update-Interest?             [ SubF-Update-Interest-Retail-K ] [ set eta0  taylor-interest-rate ]                         ; call a subfunction to update the interest rate of this firm
end



to SubF-Update-Interest-Retail-K                                                                                                    ; update the interest rate of a K-firm
   set Probability-of-Bankruptcy      (1 / (1 + e ^ ( - Beta_1-K-Firms - (Beta_2-K-Firms * leverage))))                             ; save the probability of dying of this K-Firm, and for the Betas see above
   set T-expected                     (1 / Probability-of-Bankruptcy)                                                               ; calculate the life expectation of this firm (eq 24)
   let Numerator                      (1 + (taylor-interest-rate / annuity-rate))
   let Denominator                   ((1 - (1 - annuity-rate) ^ (T-expected + 1)) / annuity-rate)
   let new-int                         Bank-gross-mark-up * ((numerator / denominator) - annuity-rate)                              ; calculating the new interest rate (eq 26 or 8.9 in Assenza, Gatti, Grazzini et al. 2015)
   set eta0                           (eta0 * (Current-Retail-Loans - Need-Cash) + new-int * Need-Cash) / Current-Retail-Loans
end



to SubF-Go-Bankruptcy-C-Firms                                                                           ; this subfunction rules the process for the C-firm bankruptcy
  set C-Broke                                      C-Broke + 1                                          ; increase the global parameter which count the broke status
  set Cases-of-Bankruptcy                          0                                                    ; reset the parameter
  ask out-owner-neighbors [                                                                             ; find out who is the owner of this firm
          ifelse Bank-Account-Household >= Initial-liquidity-of-Firms [                                 ; if the owner has enough money for paying the initial capital of the new firm
                 set Bank-Account-Household        Bank-Account-Household - Initial-liquidity-of-Firms  ; update the bank account of the capitalist, reduce the money used to rebuild the firm
                 ask in-owner-neighbors [                                                               ; the owner asks the firm to reset the following variables
                     set Current-Retail-Loans      0                                                    ; reset the parameter
                     if Capital-Destruction?     [ set Capital       Initial-Capital ]                  ; reset the capital of the C-Firm, the input applied to the production of C goods
                     set Bank-Account-Firm         Initial-liquidity-of-Firms                           ; updated the bank account with the money from the capitalist
                 ]
          ] [                                                                                           ; if the owner does not have enough money for paying the initial capital of the firm with problems...
                 set Bank-Account-Household        0                                                    ; ...the owner pay the debts with all his/her current assets
                 set Worker?                       true                                                 ; the capitalist is broken, and he/she becomes a poor worker
                 set Number-of-Workers             Number-of-Workers + 1
                 set color                         yellow                                               ; just for fun, not necessary for the model, used only for display in HUD netlogo
                 set Human-Wealth                  1                                                    ; initial value for Human Wealth, utilized to calculate the consumption process
                 set Current-Income                0                                                    ; intitial income
                 set Bank-Account-Household        Initial-households-personal-assets                   ; initial value of deposits at the banks, a vital minimum
                 set Desired-Consumption           0                                                    ; initial value for desire consume
                 set Consumption                   0                                                    ; initial value for Consumption
                 set C-Firm-List                  [ ]                                                   ; setup for the list which will be used to colect the C-Firms visited
                 set Employed?                     false                                                ; this worker is unemployed for the moment
                 ask in-owner-neighbors [                                                               ; the owner asks the firm to reset the following variables
                     set Current-Retail-Loans      0                                                    ; reset the parameter... in fact this is unnecesary because the firm will die soon...
                     die                                                                                ; ... I warned you about it... the C-Firm dies, and all its links
                     set Number-of-C-Firms         Number-of-C-Firms - 1
                 ]
         ]
  ]                                                                                                     ; end asking the owner
end


to SubF-Go-Bankruptcy-K-Firms                                                                           ; this subfunction rules the process for the K-firm bankruptcy
  set K-Broke                                     K-Broke + 1                                           ; increase the global parameter which count the broke status
  set Cases-of-Bankruptcy                         0                                                     ; reset the parameter
  ask out-owner-neighbors [                                                                             ; find out who is the owner of this firm... there is only one
          ifelse Bank-Account-Household >= Initial-liquidity-of-Firms [                                 ; if the owner has enough money for paying the initial capital of the new firm
                 set Bank-Account-Household       Bank-Account-Household - Initial-liquidity-of-Firms   ; update the bank account of the capitalist, reduce the money used to rebuild the firm
                 ask in-owner-neighbors [                                                               ; the owner asks the firm to reset these variables
                     set Current-Retail-Loans     0                                                     ; reset the parameter
                     set Bank-Account-Firm        Initial-liquidity-of-Firms                            ; updated the bank account with the money from the capitalist
                 ]
          ] [                                                                                           ; if the owner does not have enough money for paying the initial capital of the firm with problems...
                 set Bank-Account-Household       0                                                     ; ...the owner pay the debts with all his/her current assets
                 set Worker?                      true                                                  ; the capitalist is broken, and he/she becomes a poor worker
                 set Number-of-Workers            Number-of-Workers + 1
                 set color                        yellow                                                ; just for fun, not necessary for the model, used only for display in HUD netlogo
                 set Human-Wealth                 1                                                     ; initial value for Human Wealth, utilized to calculate the consumption process
                 set Current-Income               0                                                     ; intitial income
                 set Bank-Account-Household       Initial-households-personal-assets                    ; initial value of deposits at the banks, a vital minimum
                 set Desired-Consumption          0                                                     ; initial value for desire Consumption
                 set Consumption                  0                                                     ; initial value for Consumption
                 set C-Firm-List                 [ ]                                                    ; setup for the list which will be used to colect the C-Firms visited
                 set Employed?                    false                                                 ; this worker is unemployed for the moment
                 ask in-owner-neighbors [                                                               ; the owner asks the firm to reset these variables
                     set Current-Retail-Loans     0                                                     ; reset the parameter... in fact this is unnecesary because the K-Firm will die soon...
                     die                                                                                ; ... I warned you about it... the K-Firm dies, and all its links
                     set Current-Number-of-K-Firms        Current-Number-of-K-Firms - 1
                 ]
         ]
  ]                                                                                                     ; end asking the owner
end


to SubF-Ask-for-Credit-to-Central-Bank                                                                  ; this subfunction is called when the retail bank needs for a loan due to liquidity problems
   if Own-Reserves / Clients-Accounts-at-the-Bank_1 < RR [
      let Needed-Liquidity                        Clients-Accounts-at-the-Bank_1 * RR - Own-Reserves
      set Central-Bank-Loans                      Central-Bank-Loans + Needed-Liquidity
   ]
end




; FUNCTION-CENTRAL-BANK: SETTING INTEREST RATE AND CONDITIONS FOR THE LOANS TO THE RETAIL BANKS


to Function-Central-Bank
   SubF-Set-Interest-Rates                           ; different interest rates are calculated by the central bank
   SubF-Set-Macroprudential-Rules                    ; a set of macroprudential rules included in the Basel III catalogue
end


to SubF-Set-Interest-Rates
   set Total-Loans_1             Total-Loans
   set Total-Loans               Total-Loans-Firms + Total-Central-Bank-Loans
   set inflation_1               inflation                                                                                                                                                              ; store the inflation value for use with adaptative functions
   set inflation                 ((ln Goods-Price-Level - ln Goods-Price-Level_1) + (ln Capital-Price-Level - ln Capital-Price-Level_1))                                                                ; this approximation with logarithms is good only if the rate is close to zero
   set P_GDP                     Productivity-of-labor * Number-of-Workers                                                                                                                              ; P_GDP is potential GDP, and this is the maximum production, by using all resources (Unemployment = 0)
   let sensity_inflation         Taylor-rule-parameter-for-inflation * (inflation - desired-inflation)                                                                                                  ; Taylor-rule-parameter-for-inflation & Desire-inflation-for-the-monetary-authority
   let sensity_product           Taylor-rule-parameter-for-product   * (ln GDP - ln P_GDP)                                                                                                              ; Taylor-rule-parameter-for-product
   let sensity_credit            Taylor-rule-parameter-for-credit    * (ln Total-Loans - ln Total-Loans_1)                                                                                              ; the parameter for the credit component of the Taylor rule
   if taylor-rule = "conservative" [
   set taylor-interest-rate    max list 0.01 ( (inflation + natural-interest-rate + sensity_inflation) * (Slow_taylor) + (1 - slow_taylor) * taylor-interest-rate )                                     ; conservative Taylor rule: inflation (eq 18 of Popoyan et al)
   ]
   if taylor-rule = "dual-mandate" [
   set taylor-interest-rate    max list 0.01 ( (inflation + natural-interest-rate + sensity_inflation + sensity_product) * (Slow_taylor) + (1 - slow_taylor) * taylor-interest-rate )                   ; dual mandate Taylor rule: inflation and output gap (eq 30 of Silva or eq 19 of Popoyan et al)
   ]
   if taylor-rule = "three-mandate" [
   set taylor-interest-rate    max list 0.01 ( (inflation + natural-interest-rate + sensity_inflation + sensity_product + sensity_credit) * (Slow_taylor) + (1 - slow_taylor) * taylor-interest-rate )  ; three-mandate Taylor rule: inflation, output gap and credit (eq 20 of Popoyan et al)
   ]
end


to SubF-Set-Macroprudential-Rules                                                                     ; we follow here Popoyan et al (2017), pp. 120-121 and 125-127

   ; First, we prepare the global variables needed below

   set Credit-on-GDP_1                             Credit-on-GDP                                      ; we store the value of the previous period
   set Credit-on-GDP                               Total-Loans-Firms / GDP                            ; this is the basic ratio used in the Counter-Cyclical-Capital-Buffer defined in Basel III rules
   let Credit-on-GDP_bis                           Total-Client-Loans / GDP                           ; this is a different way for the same result, and we should see Credit-on-GDP = Credit-on-GDP_bis
   set Matrix-Basel-Buffer matrix:set-and-report Matrix-Basel-Buffer 0 Mjb Credit-on-GDP              ; update the first row of the matrix with information about the explained variable: every (0, Mjb) element is replaced by Credit-on-GDP of the current period
   set Matrix-Basel-Buffer matrix:set-and-report Matrix-Basel-Buffer 1 Mjb ticks                      ; update the second row of the matrix with information about the explicative variable: every (1, Mjb) element is replaced by Credit-on-GDP of the previous period
   set Mjb Mjb + 1 if Mjb > (Time-Window2 - 1) [ set Mjb 0 ]                                          ; move the matrix element to the next position, so the replacements in the matrix are sequential, and the time-window controls how many observations we'll have

   let A matrix:get-row Matrix-Basel-Buffer 0                                                         ; create an object with the first row (0) of the matrix, which contains current Credit-on-GDP
   let B matrix:get-row Matrix-Basel-Buffer 1                                                         ; create an object with the second row (1) of the matrix, which contains Credit-on-GDP of the previous period

   r:put   "a" A                                                                                      ; call R and put there the object A
   r:put   "b" B                                                                                      ; call R and put there the object B
   r:eval  "c <- lm(a ~ b)"                                                                           ; call R and ask to solve the OLS regression with the inputs A and B, lm stands for "linear model". The OLS regression is Credit-on-GDP = a + b * ticks + e

   let C    r:get "c$fitted.values"                                                                   ; call R and put the fitted values into an object C, which is a list
   let long-Run-Credit-on-GDP  last C                                                                 ; save the last fitted value as a new variable used in the calculation of the gap
   set Credit-on-GDP-Gap                           Credit-on-GDP - long-Run-Credit-on-GDP             ; we create the gap variable here... and then we proceeed to the calculation of the capital buffer


   ; Now we start with the definition of the macroprudential rules and their implications for the available credit of banks

   ask Banks [                                                                                                                ; this is important because the rules are applied to a macro scale but also to a micro scale, for every bank
       if Macroprudential-Rules = "basel2" [                                                                                  ; we have created a chooser called "macroprudential-rules" with several possibilities: basel2, basel31, basel32, basel312
          set Available-Credit-Basel2              max (list 0 (Own-Bank-Capital * 12.5 - Clients-Loans-with-the-Bank))       ; available-credit = Own-Bank-Capital * 12.5 - Total-Loans-Firms, the maximum limit minus the current level of loans
          let Minimum-Capital-Requirement-Basel2   Own-Bank-Capital / (Clients-Loans-with-the-Bank + available-credit-basel2) ; Basel II sets this ratio as 8% as a minimum
          set Available-Credit-Basel               Available-Credit-Basel2                                                    ; Available-Credit-Assenza are the particular constraints of the bank to new loans, and it is combined with Available-Credit-Basel determined by monetary authorities
       ]

       if Macroprudential-Rules = "basel3_1" [
          let Available-Credit-Basel31a            max (list 0 (Own-Bank-Capital * 22.2 - Clients-Loans-with-the-Bank))          ; available-credit = Own-Bank-Capital * 22.2 - Total-Loans-Firms, the maximum limit minus the current level of loans
          let Minimum-Capital-Requirement-Basel3   Own-Bank-Capital / (Clients-Loans-with-the-Bank + Available-Credit-Basel31a)  ; Basel II sets this ratio as 4,5% as a minimum

          let Available-Credit-Basel31b            0
          set Capital-Buffer                       0
          let Buffer-Add-On                        0

          if Credit-on-GDP-Gap < 2 [
             set Capital-Buffer                    0                                                                                   ; there is no buffer here, and therefore there are no specific constraits on new available-credits. The capital Buffer is a percentage on Total-Loans-Firms.
             set Buffer-Add-On                     0                                                                                   ; the Capital Buffer is a % of the Total-Loans-Firms not available for firms, this is, for each $ loaned this % must be reserved as a buffer...
          ]                                                                                                                            ; end of first case
          if Credit-on-GDP-Gap <= 10 and Credit-on-GDP-Gap >= 2 [
             set Capital-Buffer                    max (list 0 ((Credit-on-GDP-Gap - 2) * 0.003125))                                   ; 0.003125 = 0.025 / 8 and from here we have a range of values for the Capital-Buffer... (Total-Loans-Firms + Available-Credit-Basel32)
             set Available-Credit-Basel31b        (Clients-Accounts-at-the-Bank + Own-Bank-Capital + Central-Bank-Loans - Own-Reserves - Clients-Loans-with-the-Bank - Clients-Loans-with-the-Bank * Capital-Buffer) / (1 + Capital-Buffer)          ; Deposits-Total includes Own-Bank-Capital
             set Buffer-Add-On                     Capital-Buffer * (Clients-Loans-with-the-Bank + Available-Credit-Basel31b)          ; the aggregate constraint is (Deposits-Total + Central-Bank-Loans + Own-Capital-Bank - Reserves - Total-Loans-Firms - Buffer-Add-On) = Available-Credit-Basel31b
          ]                                                                                                                            ; and aggregate Buffer-Add-On = Capital-Buffer * (Total-Loans-Firms + Available-Credit-Basel31)...
          if Credit-on-GDP-Gap > 10 [
             set Capital-Buffer                    0.025
             set Available-Credit-Basel31b         max (list 0 ((Clients-Accounts-at-the-Bank + Own-Bank-Capital + Central-Bank-Loans - Own-Reserves - Clients-Loans-with-the-Bank - Clients-Loans-with-the-Bank * Capital-Buffer) / (1 + Capital-Buffer)))     ; Deposits-Total includes Own-Bank-Capital
             set Buffer-Add-On                     Capital-Buffer * (Clients-Loans-with-the-Bank + Available-Credit-Basel31b)          ; the aggregate constraint is (Deposits-Total + Total-Central-Bank-Loans + sum (Own-Capital-Bank) - Reserves - Total-Loans-Firms - Buffer-Add-On) = Available-Credit-Basel31b
          ]                                                                                                                            ; end of third case


          let Leverage-Requirement-Basel3          Own-Bank-Capital / Assets                                                           ; this is the third ratio of the first macroprudential rule of Basel III, and the ratio has to be equal to or higher than 3%
          let Available-Credit-Basel31c            max (list 0 ((Own-Bank-Capital - 0.03 * Assets) / 0.03))                            ; 0.03 = Own-Bank-Capital / (Assets of Banks + Available-Credit-Basel31c) and therefore...

          set Available-Credit-Basel31             min (list Available-Credit-Basel31a Available-Credit-Basel31b Available-Credit-Basel31c)
          set Available-Credit-Basel               Available-Credit-Basel31
       ]

       if Macroprudential-Rules = "basel3_2" [
           let High-Quality-Liquid-Assets         (Own-Reserves + Own-Bank-Capital) * 1.75
           set Available-Credit-Basel32           (High-Quality-Liquid-Assets - (taylor-interest-rate + annuity-rate) * Central-Bank-Loans - 0.1 * Clients-Accounts-at-the-Bank - 0.25 * Central-Bank-Loans + I-Received + D-Received - 0.5 * Clients-Loans-with-the-Bank) * 2
           let Expected-Net-Cash-Outflows         (taylor-interest-rate + annuity-rate) * Central-Bank-Loans + 0.1 * Clients-Accounts-at-the-Bank + 0.25 * Central-Bank-Loans - I-Received - D-Received + 0.5 * (Clients-Loans-with-the-Bank + Available-Credit-Basel32)
           let Liquidity-Coverage-Ratio            High-Quality-Liquid-Assets / Expected-Net-Cash-Outflows                             ; Liquidity-Coverage-Ratio >= 1
           set Available-Credit-Basel              Available-Credit-Basel32
       ]

       if Macroprudential-Rules = "basel3_12" [
           set Available-Credit-Basel              min (list Available-Credit-Basel31 Available-Credit-Basel32)
       ]
   ]
end







; FUNCTION-CALL-STATE


to Function-State-Manage-Budget                                                                                         ; the State get taxes, sells PublicDebt to the Central Bank if there is a deficit and buys PublicDebt to the Central Bank if there is a superavit and pays unemployment benefits
   set State-Budget                               State-Budget - PublicDebt * (taylor-interest-rate * annuity-rate)     ; the unemploymet benefits was substracted from the State Budget when the State paid for it
   let State-Budget-Balance                       State-Budget                                                          ; we record the budget balance
   let State-Budget-Balance_1                     State-Budget-Balance                                                  ; we store the surplus or deficit of this period
   if State-Budget-Balance < 0 and Do_We_Have_Central-Bank? = true and QEasing? = true [                                ; if the Central Bank exists and it buys public debt without limits, all the deficit is converted to government bonds sold to the CB
      set PublicDebt                              PublicDebt + abs State-Budget                                         ; new debt is added to the existing
      set State-Budget                            0                                                                     ; when the deficit is covered, the State budget is in equilibrium
   ]                                                                                                                    ; if the Quantitative Easing is not activated the State Budget cannot be negative (we might consider a constraint of 3% of deficit instead of 0%)
   if State-Budget-Balance > 0 and PublicDebt > 0 and PublicDebt >= State-Budget-Balance and Do_We_Have_Central-Bank? = true [                         ; if the Central Bank exists and there is a surplus...
      set PublicDebt                              PublicDebt - State-Budget                                             ; new debt is added to the existing
      set State-Budget                            0                                                                     ; when the deficit is covered, the State budget is in equilibrium
   ]                                                                                                                    ; end of this case
   if State-Budget-Balance > 0 and PublicDebt >= 0 and PublicDebt < State-Budget-Balance and Do_We_Have_Central-Bank? = true [                         ; if the Central Bank exists and there is a surplus...
      set State-Budget                            State-Budget-Balance - PublicDebt                                     ; new debt is added to the existing
      set PublicDebt                              0                                                                     ; the public debt is cancelled
      set Account-of-State-at-Central-Bank        Account-of-State-at-Central-Bank + State-Budget                       ; the surplus goes to the account of the State at the Central Bank, and this reduces the amount of money in the system
      set State-Budget                            0                                                                     ; when the surplus is applied, the State budget is in equilibrium
   ]                                                                                                                    ; end of this case
end


to Function-Update-System
   SubF-Update-Wages
   SubF-Update-Monetary-Variables
   SubF-Entry-New-Firms
end




 To SubF-Update-Wages                                                                           ; this subfunction adjusts wages depending on unempleyment rate, and this wage is applied to the next period
   set cpi_1           cpi                                                                      ; store the cpi value for use with adaptative functions... it is a global variable
   set cpi             ln Goods-Price-Level - ln Goods-Price-Level_1                            ; cpi is the consumer prince index, user for indexing salaries
   set Wage-on-GDP_1   Wage-on-GDP                                                              ; store the Wage-on-GDP value for use with adaptative functions
   set Wage-on-GDP    (sum [Number-of-Employees] of Firms * Wage) / GDP                             ; total salaries as a percentage of the GDP, then Wage = (Wage-on-GDP * GDP) / Number-of-Employees is the wage that keeps that proportion constant
   set Wage_1          Wage                                                                     ; store the Wage value for use with adaptative functions
   ifelse Unemployment-Rate <= frictional-unemployment-rate + 3 [                               ; if the unemployment rate approaches the frictional unemployment rate (of about 5%), then trade unions get stronger and wages rise
          set Wage  min list ((1 + cpi_1) * Wage) ((Wage-on-GDP_1 * GDP) / sum [Number-of-Employees] of Firms) ; two options: the previous wage increased by the consumer price index or the wage which keeps Wage-on-GDP constant
   ] [                                                                                          ; the second option for wage comes from finding the value of wage in (Wage-on-GDP_1 = Wage-on-GDP)
          set Wage  minimum-wage                                                                ; if the unemployment rate is higher than the frictional unemployment rate plus a margin, the wages cannot be lower than a legal minimum
   ]                                                                                            ; trade unions try to keep purchasing power of salaries or weight of salaries on GDP constant, but they loss power if unemployment increases
end


to SubF-Update-Monetary-Variables
   set Monetary-Base                             Total-Central-Bank-Loans + PublicDebt - Account-of-State-at-Central-Bank - Central-Bank-Own-Capital    ; monetary base is "outside money", and that difference is "coins and bills" money
   set Currency                                  Monetary-Base - Total-Reserves                                                                         ; Currency is Currency, and this plus Total-Reserves is the Monetary-Base
   set Money-Supply                              Monetary-Base / RR                                                                                     ; Monetary-Base multiplied by the money multiplier (1/RR) is the Money-Supply
end



to SubF-Entry-New-Firms                                                                                         ; if bakruptcy of firms occurs, a procedure for regenerating firms is activated
   let C-Firms-newcomers%                        0                                                              ; local variable, a probability following a Beta distribution
   ifelse GDP > GDP_1 [ set C-Firms-newcomers%  stats:beta 6 3 ] [ set C-Firms-newcomers%  stats:beta 3 6 ]     ; the probability of new firms depends on the GDP growth, and the Beta distribution is asymmetrical
   if Number-of-C-Firms > Current-Number-of-C-Firms [                                                           ; if the number of C-Firms is lower than the maximun, there is room for newcomers
      if random-float 1 < C-Firms-newcomers% [                                                                  ; with a probability that depends on the GDP growth, new C-Firms appear in the economy
         create-firms 1 [
                 move-to one-of patches with [not any? Firms-here]                                              ; if a C-Firm and K-Firm are at the same patch, move the new firm in the setup process
                 set shape                      "factory"                                                        ; just for fun, not necessary for the model, used only for display in HUD netlogo
                 set size                        2                                                               ; just for fun, not necessary for the model, used only for display in HUD netlogo
                 set color                       orange                                                          ; just for fun, not necessary for the model, used only for display in HUD netlogo
                 set C-Firm?                     true                                                            ; proper of the Firms, true means it is a consumption producer firm
                 set Owner?                      false                                                           ; this firm will have an owner, but at this moment does not have it
                 set Price                       random-float 2                                                  ; initial value for price
                 set Capital                     Initial-capital                                                 ; initial value for capital
                 set Stock                       Initial-production-C-firms                                      ; initial value for stock
                 set Production                  Initial-production-C-firms                                      ; initial value for production
                 set Revenue                     0                                                               ; initial value for revenue
                 set Sales                       0                                                               ; initial value for sales
                 set Need-Cash                   0                                                               ; initial value for cash needed before production, because lack of liquidity for starting it
                 set Cases-of-Bankruptcy         0                                                               ; the total number of bankruptcies by this firm
                 set Desired-Investment          0                                                               ; initial value for desire investment
                 set Investment                  0                                                               ; initial value for investment
                 set Number-of-Employees         0                                                               ; initial value for number of employees
                 set Need-of-Employees           0                                                               ; initial value for desire employees
                 set K-Firm-List                [ ]                                                              ; setup for the list which will be used to colect the K-Firms visited
                 set Vacancy                     0                                                               ; initial value for vacancy
                 set Bank-Account-Firm           Initial-liquidity-of-Firms                                      ; initial money deposited at the bank by the firms
                 set Investment-Memory           Initial-Capital                                                 ; initial value for Investment-Memory, parameter utilized to calculate the desire investment
                 set Util-Capacity               0                                                               ; initial value for util-capacity
                 create-banker-from              min-one-of banks [distance myself] [set hidden? true]           ; this create a link between the bank and the capitalist
                 set My-Nearest-Bank             min-one-of banks [distance myself]                              ; this bank is the same to which the C-Firms is now linked with a "banker" relationship
         ]                                                                                                      ; end new firm commands
      ]                                                                                                         ; end if random 100
   ]                                                                                                            ; end condition about room for new firms

   let Newcapitalist-C                           n-of 1 Workers with-max [ Bank-Account-Household ]              ; the new firm needs a new capitalist-owner
   ask Newcapitalist-C [                                                                                        ; we ask the worker with the highest savings to buy the Firm and become a capitalist
       ask one-of my-in-hireds [                                                                                ; the new capitalist is a worker yet and we take its labor contract and...
           ask other-end [                                                                                      ; ... at the other end of that contract is a firm
                set Vacancy                      Vacancy               + 1                                       ; the worker leaves, so the firm adjust the vacancies
                set Number-of-Employees          Number-of-Employees   - 1                                       ; the firm losses one worker
           ]
       die                                                                                                       ; the link, the labor relationship, dies
       ]
       set shape                                "person"                                                         ; this is unnecessary, because the worker also has a person shape
       set size                                  1.25                                                            ; the same goes for the size
       set color                                 white                                                           ; the color changes, because workers are yellow and capitalists are white
       set Worker?                               false                                                           ; he/she is now a capitalist
       set Current-Number-of-Workers             Current-Number-of-Workers - 1
       create-owner-from                         one-of Firms with [Owner? = false] [set hidden? true]           ; this creates a link between the firm and the capitalist
   ]


                                                     ; local variable with the total number of K-Firms alive in this period
   let K-Firms-newcomers%                        0                                                              ; local variable, a probability following a Beta distribution
   ifelse GDP > GDP_1 [ set K-Firms-newcomers%  stats:beta 6 3 ] [ set K-Firms-newcomers%  stats:beta 3 6 ]     ; the probability of new firms appearing depends on the GDP growth, and the Beta distribution is asymmetrical
       if Number-of-K-Firms > Current-Number-of-K-Firms [                                                       ; if the number of K-Firms is lower than the maximun, there is room for newcomers
          if random-float 1 < K-Firms-newcomers% [                                                              ; with a probability that depends on the GDP growth, new K-Firms appear in the economy
             create-firms 1 [
                  move-to one-of patches with [not any? Firms-here]                                             ; if a C-Firm and K-Firm are at the same patch, move the K-Firm in the setup process
                  set shape                     "factory"                                                       ; just for fun, not necessary for the model, used only for display in HUD netlogo
                  set size                       2                                                              ; just for fun, not necessary for the model, used only for display in HUD netlogo
                  set color                      green                                                          ; just for fun, not necessary for the model, used only for display in HUD netlogo
                  set C-Firm?                    false                                                          ; proper of the Firms, true means it is a consumption producer firm
                  set Owner?                     false                                                          ; this firm will have an owner, but at this moment does not have it
                  set Price                      random-float 2                                                 ; initial value for price
                  set Capital                    Initial-capital                                                ; initial value for capital
                  set Stock                      Initial-production-K-firms                                     ; initial value for stock
                  set Production                 Initial-production-K-firms                                     ; initial value for production
                  set Revenue                    0                                                              ; initial value for revenue
                  set Sales                      0                                                              ; initial value for sales
                  set Need-Cash                  0                                                              ; initial value for cash needed before production, because lack of liquidity for starting it
                  set Cases-of-Bankruptcy        0                                                              ; the total number of bankruptcies by this firm
                  set Desired-Investment         0                                                              ; initial value for desire investment
                  set Investment                 0                                                              ; initial value for investment
                  set Number-of-Employees        0                                                              ; initial value for number of employees
                  set Need-of-Employees          0                                                              ; initial value for desire employees
                  set Vacancy                    0                                                              ; initial value for vacancy
                  set Bank-Account-Firm          Initial-liquidity-of-Firms                                     ; initial money deposited at the bank by the firms
                  set Investment-Memory          Initial-Capital                                                ; initial value for Investment-Memory, parameter utilized to calculate the desire investment
                  set Util-Capacity              0                                                              ; initial value for util-capacity
                  create-banker-from             min-one-of banks [distance myself] [set hidden? true]          ; this create a link between the bank and the capitalist
                  set My-Nearest-Bank            min-one-of banks [distance myself]                             ; this bank is the same to which the C-Firms is now linked with a "banker" relationship
             ]                                                                                                  ; end sprout
          ]                                                                                                     ; end if random 100
   ]                                                                                                            ; end condition about room for new firms

   let Newcapitalist-K                           n-of 1 Workers with-max [ Bank-Account-Household ]             ; the new firm needs a new capitalist-owner
   ask Newcapitalist-K [                                                                                        ; we ask the worker with the highest savings to buy the Firm and become a capitalist
          ask one-of my-in-hireds [                                                                             ; the new capitalist is a worker yet and we take its labor contract
              ask other-end [                                                                                   ; at the other end of that contract is a firm
                  set Vacancy                    Vacancy               + 1                                      ; the worker leaves, so the firm adjust the vacancies
                  set Number-of-Employees        Number-of-Employees   - 1                                      ; the firm losses one worker
              ]
          die                                                                                                   ; the link, the labor relationship, dies
          ]
          set shape                            "person"                                                         ; this is unnecessary, because the worker also has a person shape
          set size                              1.25                                                            ; the same goes for the size
          set color                             white                                                           ; the color changes, because workers are yellow and capitalists are white
          set Worker?                           false                                                           ; he/she is now a capitalist
          set Current-Number-of-Workers         Current-Number-of-Workers - 1
          create-owner-from                     one-of Firms with [Owner? = false] [set hidden? true]           ; this create a link between the firm and the capitalist
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
-5
10
508
524
-1
-1
5.0
1
10
1
1
1
0
1
1
1
0
100
0
100
1
1
1
ticks
24.0

SLIDER
865
215
1085
248
Memory-parameter-human-wealth
Memory-parameter-human-wealth
0
1
0.15
0.01
1
NIL
HORIZONTAL

SLIDER
865
10
1085
43
Dividend-payout-ratio
Dividend-payout-ratio
0
1
0.04
.01
1
NIL
HORIZONTAL

SLIDER
865
45
1085
78
Fraction-of-wealth-devoted-to-consumption
Fraction-of-wealth-devoted-to-consumption
0
1
0.03
.01
1
NIL
HORIZONTAL

SLIDER
865
80
1085
113
Initial-risk-free-interest-rate
Initial-risk-free-interest-rate
0
0.1
0.02
.005
1
NIL
HORIZONTAL

SLIDER
1305
10
1530
43
Quantity-adjust-parameter
Quantity-adjust-parameter
0
1
0.19
0.01
1
NIL
HORIZONTAL

SLIDER
1530
40
1753
73
Price-adjust-parameter
Price-adjust-parameter
0
0.2
0.01
.01
1
NIL
HORIZONTAL

SLIDER
1530
110
1754
143
Bank-gross-mark-up
Bank-gross-mark-up
1
2
1.04
0.01
1
NIL
HORIZONTAL

SLIDER
1080
10
1304
43
Productivity-of-labor
Productivity-of-labor
0
1
0.15
.01
1
NIL
HORIZONTAL

SLIDER
1080
42
1304
75
Productivity-of-capital
Productivity-of-capital
0
1
0.2
.01
1
NIL
HORIZONTAL

SLIDER
1080
75
1304
108
Probability-of-investing
Probability-of-investing
0
1
0.15
.01
1
NIL
HORIZONTAL

SLIDER
1080
110
1304
143
Bank-loss-parameter
Bank-loss-parameter
0
.01
0.002
0.001
1
NIL
HORIZONTAL

SLIDER
1080
145
1304
178
annuity-rate
annuity-rate
0
.2
0.05
0.01
1
NIL
HORIZONTAL

SLIDER
1755
145
1980
178
Depreciation-of-Capital-C
Depreciation-of-Capital-C
0
.1
0.03
0.01
1
NIL
HORIZONTAL

SLIDER
1755
75
1979
108
Memory-parameter-investment
Memory-parameter-investment
0
1
0.2
.1
1
NIL
HORIZONTAL

SLIDER
1530
10
1787
43
Desired-capacity-utilization-rate
Desired-capacity-utilization-rate
.5
1
0.52
.01
1
NIL
HORIZONTAL

SLIDER
1080
180
1304
213
Minimum-Wage
Minimum-Wage
.25
2
0.4
.01
1
NIL
HORIZONTAL

SLIDER
864
114
1084
147
Initial-liquidity-of-Firms
Initial-liquidity-of-Firms
5
30
9.0
1
1
NIL
HORIZONTAL

SLIDER
864
146
1084
179
Initial-capital
Initial-capital
5
30
9.0
1
1
NIL
HORIZONTAL

SLIDER
1530
74
1753
107
Initial-production-C-firms
Initial-production-C-firms
1
20
3.0
1
1
NIL
HORIZONTAL

SLIDER
1530
146
1754
179
Initial-production-K-firms
Initial-production-K-firms
1
10
3.0
1
1
NIL
HORIZONTAL

SLIDER
1530
180
1754
213
Initial-equity-of-the-bank
Initial-equity-of-the-bank
0
1000
150.0
50
1
NIL
HORIZONTAL

SLIDER
865
180
1085
213
Initial-households-personal-assets
Initial-households-personal-assets
1
10
2.0
1
1
NIL
HORIZONTAL

SLIDER
1305
80
1529
113
Natural-interest-rate
Natural-interest-rate
0
.02
0.005
.005
1
NIL
HORIZONTAL

SLIDER
1305
110
1530
143
Desired-inflation
Desired-inflation
0
1
0.06
.015
1
NIL
HORIZONTAL

SLIDER
1305
145
1529
178
Taylor-rule-parameter-for-inflation
Taylor-rule-parameter-for-inflation
0
1
0.2
0.05
1
NIL
HORIZONTAL

SLIDER
1305
180
1530
213
Taylor-rule-parameter-for-product
Taylor-rule-parameter-for-product
0.01
0.99
0.16
0.05
1
NIL
HORIZONTAL

BUTTON
520
75
580
108
Setup
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
510
350
720
383
Do_We_Have_Workers?
Do_We_Have_Workers?
0
1
-1000

SWITCH
510
385
720
418
Do_We_Have_Capitalists?
Do_We_Have_Capitalists?
0
1
-1000

SWITCH
510
315
720
348
Do_We_Have_C-Firms?
Do_We_Have_C-Firms?
0
1
-1000

SWITCH
510
280
720
313
Do_We_Have_K-Firms?
Do_We_Have_K-Firms?
0
1
-1000

TEXTBOX
515
250
661
270
Setup Conditions:
16
0.0
1

BUTTON
520
110
580
143
Go
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
520
145
580
178
Go
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
600
1370
965
1520
Unemployment Rate
NIL
NIL
0.0
1.0
1.0
0.0
true
true
"" ""
PENS
"Rate" 1.0 0 -13791810 true "if ticks > 100 [plot Unemployment-rate]" "if ticks > 100 [plot Unemployment-rate]"

MONITOR
890
1450
965
1495
Unemp
Unemployment-Rate
1
1
11

PLOT
290
1220
600
1370
Stock K-Firms
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [stock] of K-Firms]"

PLOT
2
1222
291
1372
Capital C-sector
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 300 [plot sum [Capital] of C-Firms]"

PLOT
600
1070
964
1220
Investment
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"Investment" 1.0 0 -13345367 true "" "if ticks > 100 [plot Total-Investment]"
"K-Firms Sales" 1.0 0 -955883 true "" "if ticks > 100 [plot sum [Sales] of K-Firms]"

PLOT
290
1070
600
1220
Production K-Firms
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Production] of K-Firms]"

PLOT
2
768
292
918
Workers C-Sector
NIL
NIL
0.0
10.0
10000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Number-of-Employees] of C-Firms]"

PLOT
293
770
603
920
Workers K-Sector
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Number-of-Employees] of K-Firms]"

MONITOR
528
838
599
883
K-Workers
sum [number-of-employees] of K-Firms
2
1
11

MONITOR
222
840
292
885
C-Workers
sum [number-of-employees] of C-Firms
2
1
11

MONITOR
542
1173
599
1218
K-Prod
sum [production] of k-firms
1
1
11

PLOT
601
919
962
1069
Desired-Investment
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"des-invest" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [desired-investment] of C-firms]"
"invest" 1.0 0 -13791810 true "" "if ticks > 100 [plot total-investment]"

MONITOR
540
1300
597
1345
K-Stock
sum [stock] of k-firms
2
1
11

MONITOR
905
1171
962
1216
Inves
total-investment
1
1
11

MONITOR
1970
570
2027
615
K-Sale
sum [Sales] of K-firms
2
1
11

MONITOR
906
1126
963
1171
C-Inv
sum [Investment] of C-Firms
1
1
11

PLOT
1
1371
600
1521
GDP
NIL
NIL
0.0
10.0
10000.0
10.0
true
true
"" ""
PENS
"GDP" 1.0 0 -16777216 true "" "if ticks > 100 [plot GDP]"
"pen-1" 1.0 0 -2674135 true "plot 0" "plot 0"

MONITOR
2030
525
2088
570
C-Rev
sum [Revenue] of C-Firms
1
1
11

MONITOR
2030
570
2088
615
K-Rev
sum [revenue] of K-Firms
1
1
11

MONITOR
2089
525
2147
570
C-Prof
sum [profit] of C-Firms
1
1
11

MONITOR
2088
570
2146
615
K-Prof
sum [profit] of K-firms
1
1
11

MONITOR
1660
570
1740
615
C-Accounts
sum [bank-account-firm] of C-Firms
0
1
11

MONITOR
1580
570
1660
615
K-Accounts
sum [bank-account-firm] of K-Firms
0
1
11

MONITOR
1580
525
1740
570
F-Accounts
sum [bank-account-firm] of Firms
0
1
11

MONITOR
1670
1290
1806
1335
Liq-C-Sec
sum [profit] of C-Firms - sum [revenue] of K-Firms
0
1
11

SWITCH
735
575
885
608
K-UpDate-Price?
K-UpDate-Price?
0
1
-1000

SWITCH
735
680
885
713
K-Qtt-Function?
K-Qtt-Function?
0
1
-1000

SWITCH
735
646
885
679
C-Qtt-Function?
C-Qtt-Function?
0
1
-1000

SWITCH
735
610
885
643
C-UpDate-Price?
C-UpDate-Price?
0
1
-1000

PLOT
603
770
963
919
Prices
NIL
NIL
0.0
10.0
10.0
0.0
true
true
"" ""
PENS
"C-Price" 1.0 0 -16777216 true "" "if ticks > 5 [plot Goods-Price-Level]"
"K-Price" 1.0 0 -5298144 true "" "if ticks > 5 [plot Capital-Price-Level]"

PLOT
990
1370
1552
1520
Monetary Variables
NIL
NIL
0.0
10.0
8900.0
9100.0
true
false
"" ""
PENS
"Bank Accounts" 1.0 0 -16777216 true "" "plot (sum [bank-account-household] of households + sum [bank-account-firm] of firms)"
"Money-Supply" 1.0 0 -7500403 true "" "plot (Money-Supply)"
"Monetary-Base" 1.0 0 -2674135 true "" "plot (Monetary-Base)"
"Bills-and-Coins" 1.0 0 -955883 true "" "plot (Currency)"

PLOT
988
1219
1272
1369
Households Bank Accounts
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"Workers" 1.0 0 -5298144 true "" "plot sum [bank-account-household] of Workers"
"Capitalist" 1.0 0 -14070903 true "" "plot sum [Bank-Account-Household] of Capitalists"

PLOT
1274
1220
1551
1370
Firm-Money
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"C-Firms" 1.0 0 -5298144 true "" "plot Total-Deposits-C-Firms"
"K-Firms" 1.0 0 -14070903 true "" "plot Total-Deposits-K-Firms"

SWITCH
720
385
900
418
Hiring-Rationality?
Hiring-Rationality?
0
1
-1000

PLOT
600
1222
964
1372
Consumption
NIL
NIL
0.0
10.0
10000.0
10.0
true
true
"" ""
PENS
"Consumption" 1.0 0 -16777216 true "" "if ticks > 100 [plot Total-Consumption]"
"Income" 1.0 0 -7500403 true "" "if ticks > 100 [plot (sum [Current-Income] of households)]"

PLOT
985
915
1267
1065
C-Loans
NIL
NIL
0.0
10.0
10000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Total-Loans-C-Firms"

PLOT
1268
914
1551
1064
K-Loans
NIL
NIL
0.0
10.0
10000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Total-Loans-K-Firms"

SWITCH
910
340
1060
373
Update-Interest?
Update-Interest?
0
1
-1000

PLOT
986
1067
1551
1217
Total-Loans
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot total-loans"

MONITOR
1490
1140
1550
1185
T-Loans
Total-loans
1
1
11

CHOOSER
1985
185
2157
230
B-Lev
B-Lev
1 5 10
2

CHOOSER
1985
140
2157
185
T-Ban
T-Ban
1 2 3 4
1

PLOT
2
919
291
1069
Hist C-Production
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "if ticks > 20 [histogram [production] of C-Firms]"

PLOT
293
919
602
1069
Hist K-Production
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "if ticks > 20 [histogram [production] of K-Firms]"

SWITCH
720
280
902
313
C-Broke-Procedure?
C-Broke-Procedure?
0
1
-1000

SWITCH
720
314
902
347
K-Broke-Procedure?
K-Broke-Procedure?
0
1
-1000

PLOT
985
766
1265
914
C-Broke
NIL
NIL
0.0
10.0
100.0
1.0
true
true
"" ""
PENS
"default" 1.0 1 -16777216 true "" "plot c-broke"

MONITOR
1205
866
1263
911
C-Broke
C-Broke
0
1
11

PLOT
1265
766
1550
916
K-Broke
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"default" 1.0 1 -16777216 true "" "plot K-Broke"

MONITOR
1205
995
1267
1040
C-Loans
Total-Loans-C-Firms
0
1
11

MONITOR
1490
990
1552
1035
K-Loans
Total-Loans-K-Firms
0
1
11

SWITCH
720
420
900
453
Dividends-Payments?
Dividends-Payments?
0
1
-1000

SWITCH
720
350
900
383
Capital-Destruction?
Capital-Destruction?
0
1
-1000

SWITCH
910
305
1040
338
Prof-K-Firms?
Prof-K-Firms?
0
1
-1000

PLOT
1580
1065
1865
1215
C-Leverage
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean [leverage] of C-Firms"

PLOT
1865
1065
2145
1215
K-Leverage
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean [leverage] of K-Firms"

PLOT
1580
1215
1865
1365
Credit-Restriction-C
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot  ( 1 / exp(- ( - 5.5 + 12 * mean [leverage] of C-Firms) ))"

PLOT
1865
1215
2147
1365
Credit-Restriction-K
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot  ( 1 / exp(- ( - 10.3 + 12 * mean [leverage] of K-firms) ))"

PLOT
1580
915
1865
1065
Beta_1-C
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Beta_1-C-Firms"

PLOT
1865
915
2145
1065
Beta_2-C
NIL
NIL
0.0
10.0
10.0
0.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Beta_2-C-Firms"

MONITOR
1805
980
1865
1025
Beta_1-C
Beta_1-C-Firms
1
1
11

MONITOR
2085
980
2145
1025
Beta_2-C
Beta_2-C-Firms
1
1
11

PLOT
1580
1365
1865
1515
C-Time-Life
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 1000 [ plot 1 / (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * mean [leverage] of C-Firms)))) ]"

SWITCH
911
273
1041
306
Random-Seed?
Random-Seed?
0
1
-1000

SWITCH
510
455
720
488
Do_We_Have_Central-Bank?
Do_We_Have_Central-Bank?
0
1
-1000

PLOT
1265
646
1550
766
Free interest rate
NIL
NIL
0.0
0.0
10.0
0.0
true
true
"" ""
PENS
"FIR" 1.0 0 -16777216 true "" "plot Taylor-Interest-Rate"

MONITOR
1490
690
1550
735
TIR
taylor-interest-rate
1
1
11

SLIDER
1304
44
1529
77
Slow_taylor
Slow_taylor
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
1755
110
1980
143
Depreciation-of-Capital-K
Depreciation-of-Capital-K
0
0.1
0.02
0.01
1
NIL
HORIZONTAL

TEXTBOX
535
15
630
51
Parameters:
16
0.0
1

PLOT
1
1072
291
1222
Production C-Firms
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Production] of C-Firms]"

PLOT
1866
1365
2148
1513
K-Time-Life
NIL
NIL
0.0
10.0
0.0
100.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 1000 [ plot 1 / (1 / (1 + e ^ ( - Beta_1-K-Firms - (Beta_2-K-Firms * mean [leverage] of K-Firms)))) ]"

SLIDER
1755
180
1980
213
Risk
Risk
0
100
1.0
1
1
NIL
HORIZONTAL

MONITOR
545
1445
600
1490
GDP
GDP
1
1
11

MONITOR
905
870
962
915
K-Price
Capital-Price-Level
1
1
11

MONITOR
905
825
960
870
C-Price
Goods-Price-Level
1
1
11

MONITOR
235
1175
292
1220
C-Prod
sum [production] of c-firms
1
1
11

MONITOR
880
1325
965
1370
Income
sum [Current-Income] of households
1
1
11

MONITOR
880
1280
965
1325
Consumption
Total-Consumption
1
1
11

PLOT
1580
765
1865
915
Beta_1-K
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Beta_1-K-Firms"

MONITOR
1805
835
1865
880
Beta_1-K
Beta_1-K-Firms
1
1
11

PLOT
1865
765
2145
915
Beta_2-K
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Beta_2-K-Firms"

MONITOR
2085
835
2145
880
Beta_2-K
Beta_2-K-Firms
1
1
11

MONITOR
1775
1465
1832
1510
C-Time
1 / (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * mean [leverage] of C-Firms))))
1
1
11

MONITOR
1810
1165
1865
1210
C-Lev
mean [leverage] of C-Firms
2
1
11

MONITOR
2090
1165
2145
1210
K-Lev
mean [leverage] of K-Firms
2
1
11

PLOT
985
646
1265
766
Interest Retail (eta1)
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"IR-C" 1.0 0 -16777216 true "plot mean [eta0] of C-Firms" "plot mean [eta0] of C-Firms"
"IR-K" 1.0 0 -7500403 true "plot mean [eta0] of K-Firms" "plot mean [eta0] of K-Firms"

MONITOR
1495
870
1550
915
NIL
K-Broke
1
1
11

MONITOR
1865
525
1922
570
Nº Cs
Number-of-C-Firms
0
1
11

MONITOR
1865
570
1922
615
Nº Ks
Number-of-K-Firms
0
1
11

PLOT
1580
615
2145
765
Total Deposits in Retail Banks
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Total-Deposits"
"pen-1" 1.0 0 -7500403 true "" "plot Total-Clients-Accounts"

CHOOSER
1985
60
2160
105
Time-Window1
Time-Window1
600 800 1600 3200 6400 10000
3

SWITCH
510
490
720
523
Do_We_Have_State?
Do_We_Have_State?
1
1
-1000

SWITCH
510
420
720
453
Do_We_Have_Banks?
Do_We_Have_Banks?
0
1
-1000

SLIDER
1755
10
1970
43
Frictional-Unemployment-Rate
Frictional-Unemployment-Rate
0.1
0.20
0.12
0.01
1
NIL
HORIZONTAL

SLIDER
1220
395
1530
428
radius-for-employment-search
radius-for-employment-search
0
50
0.0
10
1
NIL
HORIZONTAL

SLIDER
1305
215
1530
248
Taylor-rule-parameter-for-credit
Taylor-rule-parameter-for-credit
0
1
0.1
0.1
1
NIL
HORIZONTAL

CHOOSER
1305
250
1530
295
taylor-rule
taylor-rule
"conservative" "dual-mandate" "three-mandate"
0

SLIDER
1080
215
1305
248
Unemployment-Cover
Unemployment-Cover
0
1
0.4
0.1
1
NIL
HORIZONTAL

CHOOSER
1090
275
1252
320
Macroprudential-Rules
Macroprudential-Rules
"basel2" "basel3_1" "basel3_2" "basel3_12"
2

CHOOSER
1985
10
2160
55
Time-Window2
Time-Window2
60
0

SLIDER
1540
290
1712
323
tax1
tax1
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
1540
395
1712
428
RR
RR
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
1540
325
1712
358
tax2
tax2
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
1540
360
1712
393
tax3
tax3
0
1
0.0
0.01
1
NIL
HORIZONTAL

CHOOSER
1090
330
1228
375
Number-of-Banks
Number-of-Banks
1 3 5 10 15 20 30
2

SLIDER
675
45
862
78
Number-of-C-Firms
Number-of-C-Firms
0
1000
300.0
100
1
NIL
HORIZONTAL

SLIDER
685
95
862
128
Number-of-K-Firms
Number-of-K-Firms
0
500
50.0
50
1
NIL
HORIZONTAL

SLIDER
675
140
862
173
Number-of-Workers
Number-of-Workers
0
5000
2500.0
500
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

The financial-economic crisis that started in 2008 has questioned all the previous equilibrium-based macroeconomic models (Dynamic Stochastic General Equilibrium models). 
The Agent-Based Models allow a more flexible and complex kind of models, where heterogeneity are easily incorporated and equilibrium is not a pre-imposed feature of the model. In ABMs individuals are not forced to solve any complex intertemporal optimization under the rational expectations hypothesis. Now we can assume more realistic assumptions and consider that the information is limited, there are trasaction costs, rationality is constrained, etc. This model is a specific implementaion of the model outlined in Assenza et al. (2015).

## HOW IT WORKS

We simulate an economy with two kinds of Firms: (i) Firms that will produce
Capital; and (ii) Firms which will produce final goods. There is some numbers of
workers, which also are the consumers in this model, the workers offer worker force
to the firms and receive a wage for this. If the workers has no job he try to find
one visiting some amount of Firms. The productivity of capital and Labor are taken
as constant in this model, as well as the wage; the wage are the same for the both
kind Firms, so the workers will accept the first job offer he receive.

## HOW TO USE IT

You just need to click in "Setup", and then click "Go Once" or "Go Forever". Each
house represent one Firm, and each person we see represent one worker.

## THINGS TO NOTICE

Look at the series of the Unemployment rate, note they are workers which are able
to work and don't find a job, this reflect in the series of the GDP, that means,
the production in this economy is not the full, there is some gap between the
potential GDP and the actual GDP. The other graphics show the series of prices, the
level price in the Capital goods, and the level price in the consumer goods.

## THINGS TO TRY

We can see some interesting things emerging in the simulations from this model. For example, try to change a little the productivity of Labor - the first slider option - and take a look what happen with the series of the Unemployment rate below.

## EXTENDING THE MODEL

The next steps will probably be to incorporate a stock market in which households invest their savings and to consider families, with the heterogeneity associated with age, retirement, child care, etc.

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

Details about this framework you may contact us: rosuna@cee.uned.es
Also, there are some papers which could be very helpful, here are some:
Assenza, Tiziana, Domenico Delli Gatti, and Jakob Grazzini. "Emergent dynamics of a
macroeconomic agent based model with capital and credit." Journal of Economic
Dynamics and Control 50 (2015): 5-28.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dollar bill
false
0
Rectangle -7500403 true true 15 90 285 210
Rectangle -1 true false 30 105 270 195
Circle -7500403 true true 120 120 60
Circle -7500403 true true 120 135 60
Circle -7500403 true true 254 178 26
Circle -7500403 true true 248 98 26
Circle -7500403 true true 18 97 36
Circle -7500403 true true 21 178 26
Circle -7500403 true true 66 135 28
Circle -1 true false 72 141 16
Circle -7500403 true true 201 138 32
Circle -1 true false 209 146 16
Rectangle -16777216 true false 64 112 86 118
Rectangle -16777216 true false 90 112 124 118
Rectangle -16777216 true false 128 112 188 118
Rectangle -16777216 true false 191 112 237 118
Rectangle -1 true false 106 199 128 205
Rectangle -1 true false 90 96 209 98
Rectangle -7500403 true true 60 168 103 176
Rectangle -7500403 true true 199 127 230 133
Line -7500403 true 59 184 104 184
Line -7500403 true 241 189 196 189
Line -7500403 true 59 189 104 189
Line -16777216 false 116 124 71 124
Polygon -1 true false 127 179 142 167 142 160 130 150 126 148 142 132 158 132 173 152 167 156 164 167 174 176 161 193 135 192
Rectangle -1 true false 134 199 184 205

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

factory
false
0
Rectangle -7500403 true true 76 194 285 270
Rectangle -7500403 true true 36 95 59 231
Rectangle -16777216 true false 90 210 270 240
Line -7500403 true 90 195 90 255
Line -7500403 true 120 195 120 255
Line -7500403 true 150 195 150 240
Line -7500403 true 180 195 180 255
Line -7500403 true 210 210 210 240
Line -7500403 true 240 210 240 240
Line -7500403 true 90 225 270 225
Circle -1 true false 37 73 32
Circle -1 true false 55 38 54
Circle -1 true false 96 21 42
Circle -1 true false 105 40 32
Circle -1 true false 129 19 42
Rectangle -7500403 true true 14 228 78 270

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="hp_filters_test" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="1"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="4"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Logistic-Regression-r-Extension" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
      <value value="800"/>
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Logistic-Regression-r-Extension_3" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <metric>sum [production] of K-Firms</metric>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <metric>sum [production] of K-Firms</metric>
    <metric>sum [Current-Retail-Loan] of firms</metric>
    <metric>sum [current-wholesale-loan] of firms</metric>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Wholesales-2" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <metric>sum [production] of K-Firms</metric>
    <metric>sum [Current-Retail-Loan] of firms</metric>
    <metric>sum [current-wholesale-loan] of firms</metric>
    <metric>sum [Intrabank-Loan] of Wholesale-Banks</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Goods-Price-Level</metric>
    <metric>sum [Current-Retail-Loan] of C-Firms</metric>
    <metric>sum [Current-Retail-Loan] of K-Firms</metric>
    <metric>sum [Current-Wholesale-Loan] of C-Firms</metric>
    <metric>sum [Current-Wholesale-Loan] of K-Firms</metric>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Central-Bank" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>Free-Interest-Rate</metric>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>Goods-Price-Level</metric>
    <metric>Goods-Price-Level_t-1</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Capital-Price-Level_t-1</metric>
    <metric>Total-Loans</metric>
    <metric>Total-Loans-C-Firms</metric>
    <metric>Total-Loans-K-Firms</metric>
    <metric>C-Firms</metric>
    <metric>K-Firms</metric>
    <metric>Workers</metric>
    <metric>Capitalists</metric>
    <metric>Retail-Banks</metric>
    <metric>Wholesale-Banks</metric>
    <metric>C-Broke</metric>
    <metric>K-Broke</metric>
    <metric>Beta_1-C-Firms</metric>
    <metric>Beta_2-C-Firms</metric>
    <metric>Beta_1-K-Firms</metric>
    <metric>Beta_2-K-Firms</metric>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-product" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-inflation" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Slow_taylor" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Central-Bank?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="1"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
      <value value="0.01"/>
      <value value="0.1"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Central-Bank-B" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>Free-Interest-Rate</metric>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>Goods-Price-Level</metric>
    <metric>Goods-Price-Level_t-1</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Capital-Price-Level_t-1</metric>
    <metric>Total-Loans</metric>
    <metric>Total-Loans-C-Firms</metric>
    <metric>Total-Loans-K-Firms</metric>
    <metric>C-Broke</metric>
    <metric>K-Broke</metric>
    <metric>Beta_1-C-Firms</metric>
    <metric>Beta_2-C-Firms</metric>
    <metric>Beta_1-K-Firms</metric>
    <metric>Beta_2-K-Firms</metric>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-product" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-inflation" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Slow_taylor">
      <value value="0"/>
      <value value="0.33"/>
      <value value="0.66"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Central-Bank?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
      <value value="0.01"/>
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Central-Bank-C" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>Free-Interest-Rate</metric>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>Goods-Price-Level</metric>
    <metric>Goods-Price-Level_t-1</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Capital-Price-Level_t-1</metric>
    <metric>Total-Loans</metric>
    <metric>Total-Loans-C-Firms</metric>
    <metric>Total-Loans-K-Firms</metric>
    <metric>C-Broke</metric>
    <metric>K-Broke</metric>
    <metric>Beta_1-C-Firms</metric>
    <metric>Beta_2-C-Firms</metric>
    <metric>Beta_1-K-Firms</metric>
    <metric>Beta_2-K-Firms</metric>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-product" first="0" step="0.33" last="1"/>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-inflation" first="0" step="0.33" last="1"/>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Slow_taylor">
      <value value="0"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Central-Bank?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Na                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  