; Revisions:
; 2013-08-02
;   - accuracy now reports ( 100% - per-capita-mean-error )
;   - diffuse-individual moved into .nls file
;   - output-wrap moved into .nls file
;   - sigfigs moved into .nls file
; 2013-07-29
;   - line-breaking procedure to replace output-prints
;   - moved some generic code into includes file
;   - events are defined at beginning and applied behind the scenes (in included file)

; design choices:
; * the (full) model is agent-based
; * bacteria (wilds and cheaters) are mobile (Brownian)
; * their products (food, toxin, and bacteriocide) are stationary
; * interactions are local (per patch)
; * time steps forward in increments large enough for multiple events to occur
; * the full model fixes the fast (food, toxin, & bacteriocide) production rates
; * the System Dynamics Modeler contains a numeric approximation
; * the numeric model applies the QSSA to reduce the system, removing the products
; * the numeric model approximates the equations in Brown et al., 2009

__includes [
  "diffuse-individual.nls"
  "output-wrap.nls"
  "per-capita-tau-leaping.nls"
  "sigfigs.nls"
]


; the possible "breeds" of turtles
breed [wilds wild]
breed [cheaters cheater]


; turtles own event rates
patches-own [
  food
  toxin
  bacteriocide
]


globals [ 
  count-patches  ; assign to variable to improve speed; used in setup and plots
  beta-food
  delta-toxin
  gamma-bacteriocide
  fast-rate      ; = beta = delta = gamma (fast consumption rates for food, toxin, & bacteriocide)
  wild-virulence
]


to startup
  setup
end

 
to setup
  clear-all
  set fast-rate 5 ; TO DO: how big does this need to be to satisfy QSSA?
  set wild-virulence 2 ; assume wild type twice as virulent as cheaters (as per Brown et al.)
  set beta-food fast-rate
  set delta-toxin fast-rate
  set gamma-bacteriocide fast-rate
  set count-patches count patches
  system-dynamics-setup ; setup system-dynamics-modeler for numeric comparison
  ; set breed shapes
  set-default-shape turtles "circle"
  ; create numbers of each breed to get the desired average density (per patch)
  create-wilds       ( W-num * count-patches ) [ set color green ]
  create-cheaters    ( C-num * count-patches ) [ set color red   ]
  ; shuffle them 
  ask turtles [ setxy random-xcor random-ycor ]
  ask patches [
    ; start with QSSA densities of food, toxin, & bacteriocide
    let count-turtles-here  count turtles-here
    let count-cheaters-here count cheaters-here
    if ( count-turtles-here > 0 ) [
      ; Poisson-distributed random variates
      set food random-poisson ( b-shared-benefit * count wilds-here / ( beta-food * count-turtles-here ) )
      set toxin random-poisson ( a-toxin-production * count-cheaters-here / ( delta-toxin * count-turtles-here ) )
      set bacteriocide random-poisson ( e-bacteriocinogen * count-cheaters-here / ( gamma-bacteriocide * count-turtles-here ) )
    ]
  ]
  reset-ticks
  ; hint
  output-wrap 26 ; set wrapping column
  output-wrap "An agent-based model of 'Social evolution in micro-organisms and a Trojan horse approach to medical intervention strategies' by Brown et al. (2009)."
  output-wrap "Press go to get started.\n"
  
  ; merge all events into one master list [2013-07-29]
  add-per-capita-event turtles                  ; N -> 2 N @ rate 1
    task [ 1 ] 
    task [ hatch 1 ]
  add-per-capita-event turtles                  ; N + N -> N @ rate 1
    task [ count other turtles-here ] 
    task [ die ]
  add-per-capita-event turtles                  ; N + F -> 2 N @ rate beta
    task [ beta-food * food ]
    task [ set food food - 1
           hatch 1           ]
  add-per-capita-event turtles                  ; N + T -> 0 @ rate delta
    task [ delta-toxin * toxin ]
    task [ set toxin toxin - 1    
           die                 ]
  add-per-capita-event wilds                    ; W -> 0 @ rate x
    task [ x-wild-cost ]
    task [ die ]
  add-per-capita-event wilds                    ; W -> W + F @ rate b
    task [ b-shared-benefit ]
    task [ set food food + 1 ]
  add-per-capita-event wilds                    ; W + B -> 0 @ rate gamma
    task [ gamma-bacteriocide * bacteriocide ]
    task [ set bacteriocide bacteriocide - 1    
           die                               ]
  add-per-capita-event cheaters                 ; C -> 0 @ rate q
    task [ q-cheater-cost ]
    task [ die ]
  add-per-capita-event cheaters                 ; C -> C + T @ rate a
    task [ a-toxin-production ]
    task [ set toxin toxin + 1 ]
  add-per-capita-event cheaters                 ; C -> C + B @ rate e
    task [ e-bacteriocinogen ]
    task [ set bacteriocide bacteriocide + 1 ]
  add-per-capita-event cheaters                 ; C + B -> C @ rate gamma
    task [ gamma-bacteriocide * bacteriocide ]
    task [ set bacteriocide bacteriocide - 1 ]
end


to go
  if ticks = 0 [ 
    reset-timer 
    output-wrap "You may adjust all parameters while the simulation runs."
    output-wrap "Compare the numeric, non-spatial (faint) vs. agent-based (bold) dynamics in the graphs."
    output-wrap ( word "It is assumed that the wild type is " wild-virulence " times as virulent as the cheater.\n" )
  ]
  system-dynamics-go ; numeric model
  set dt per-capita-tau-leap inaccuracy dt
  diffuse-individual turtles diffusion-const dt
  do-plots
  if (dt = 0) [  ; simulation over
    output-wrap (word "Finished in " timer " seconds.")
    stop 
  ]
end


to do-plots
  set-current-plot "dynamics"
  system-dynamics-do-plot
  update-plots ; update other plots ("Edit" the plots to see code)
end


to-report diffusion-const
  report ifelse-value well-mixed [ 0 ][ 10 ^ log-diffusion ]
end


to-report inaccuracy
; complement of accuracy, depends on err-tolerance
  report 10 ^ ( err-tolerance )
end
@#$#@#$#@
GRAPHICS-WINDOW
203
189
537
472
-1
-1
12.0
1
10
1
1
1
0
1
1
1
0
26
0
20
0
0
1
ticks
30.0

SLIDER
21
341
193
374
q-cheater-cost
q-cheater-cost
0
5
5
0.01
1
NIL
HORIZONTAL

SLIDER
21
374
193
407
b-shared-benefit
b-shared-benefit
0
15
15
0.1
1
NIL
HORIZONTAL

SLIDER
21
440
193
473
e-bacteriocinogen
e-bacteriocinogen
0
1
0
0.1
1
NIL
HORIZONTAL

SLIDER
21
407
193
440
a-toxin-production
a-toxin-production
-0.5
1
0
0.1
1
NIL
HORIZONTAL

BUTTON
35
265
102
298
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
113
265
176
298
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
203
15
504
179
dynamics
time
density
0.0
0.0
0.0
0.0
true
true
"" ""
PENS
"wilds" 1.0 2 -10899396 true "" "plotxy ticks ( count wilds / count-patches )"
"W-num" 1.0 0 -4399183 true "" ""
"cheaters" 1.0 2 -2674135 true "" "plotxy ticks ( count cheaters / count-patches )"
"C-num" 1.0 0 -1069655 true "" ""
"virulence" 1.0 2 -16777216 true "" "plotxy ticks ( wild-virulence * count wilds + count cheaters ) / count-patches"
"vir-num" 1.0 0 -3026479 true "" "plotxy ticks vir-num"

PLOT
503
15
672
179
phase
wilds
cheaters
0.0
0.0
0.0
0.0
true
false
"clear-plot" ""
PENS
"agent" 1.0 2 -16777216 true "" "plotxy ( count wilds / count-patches ) ( count cheaters / count-patches )"
"num" 1.0 0 -3026479 true "" "plotxy W-num C-num"

SLIDER
21
190
193
223
initial-W-density
initial-W-density
0
2
0.5
0.1
1
NIL
HORIZONTAL

SLIDER
21
222
193
255
initial-C-density
initial-C-density
0
0.2
0.05
0.01
1
NIL
HORIZONTAL

SWITCH
548
189
673
222
well-mixed
well-mixed
1
1
-1000

SLIDER
548
221
673
254
log-diffusion
log-diffusion
-2
4
0.3
0.1
1
NIL
HORIZONTAL

MONITOR
548
253
673
298
diff-const [patches^2/tick]
diffusion-const
2
1
11

MONITOR
548
430
673
475
ticks
;precision ticks (2 - ceiling log dt 10)\nsigfigs-other ticks dt 2
10
1
11

MONITOR
548
386
673
431
dt
;precision dt (2 - ceiling log dt 10)\nsigfigs dt 2
10
1
11

SLIDER
548
309
673
342
err-tolerance
err-tolerance
-2
0
-1
0.2
1
NIL
HORIZONTAL

MONITOR
548
342
673
387
accuracy (%)
;100 * ( 1 - precision per-capita-mean-error (1 - ceiling log per-capita-mean-error 10) )\n100 * ( 1 - sigfigs per-capita-mean-error 1 )
10
1
11

SLIDER
21
309
193
342
x-wild-cost
x-wild-cost
0
10
10
0.01
1
NIL
HORIZONTAL

OUTPUT
12
15
204
179
10

@#$#@#$#@
## A Trojan horse approach to medical intervention strategies

A [NetLogo] model by Rik Blok.

[http://www.zoology.ubc.ca/~rikblok/wiki/doku.php?id=science:popmod:brown2009:start](http://www.zoology.ubc.ca/~rikblok/wiki/doku.php?id=science:popmod:brown2009:start)

This agent-based model represents a bacterial infection as described in [[Brown2009]].  It is assumed the wild type (W) produces a public good (F) which benefits all.  To combat the infection the population is innoculated with a "cheater" strain (C).  Changes are simply represented as elementary reactions between (local) agents:

  * N &rarr; 2 N    @ rate 1 (growth)
  * W + N &rarr; N    @ rate 1 (competition)  
    C + N &rarr; N    @ rate 1
  * W &rarr; W + F    @ rate b (public good)  
    W + F &rarr; 2 W    @ rate &beta;
  * W &rarr; &empty;    @ rate x (wild cost)
  * C &rarr; &empty;    @ rate q (cheater cost)

where N indicates an individual of either type and &empty; indicates an absence of products.

Additionally, the cheater strain has one or more of the following traits:

  * It consumes but does not produce the public good (F),  
    C + F &rarr; 2 C    @ rate &beta;
  * It produces a toxin (T) that harms all,  
    C &rarr; C + T    @ rate a  
    N + T &rarr; &empty;    @ rate &delta;
  * It produces a bacteriocinogen (B) it is immune to but harms the wild type,  
    C &rarr; C + B    @ rate e  
    C + B &rarr; C    @ rate &gamma;  
    W + B &rarr; &empty;    @ rate &gamma;.


The target and infected cells are fixed whereas the virions move randomly with diffusion constant diff-const.  If 'well-mixed' is on then they are shuffled randomly in space with each timestep.

Finally, it is assumed that the wild type is more "virulent" (harmful to the host) than the cheater strain.

## How it works

The simulation approximates a Poisson process for each of the above events.  The best known technique would be the Gillespie algorithm [[Gibson2000]] but it isn't well suited to NetLogo's strengths.  Instead, time proceeds in steps with multiple events occurring in each timestep.  

The step size is adaptive, chosen to achieve a desired error tolerance, compared with the Gillespie algorithm.  When the error tolerance is near zero the likelihood of each event is small and we may expect just a few events to occur per timestep.  Then we're accurately -- but inefficiently -- mimicking the Gillespie algorithm.  As the tolerance increases we have more simultaneous events, lowering accuracy but increasing performance.

## References

[[Brown2009]] Brown, Sam P., West, Stuart A., Diggle, Stephen P., and Griffin, Ashleigh S. [Social evolution in micro-organisms and a Trojan horse approach to medical intervention strategies](http://rstb.royalsocietypublishing.org/content/364/1533/3157). Philosophical Transactions of the Royal Society B: Biological Sciences, 364: 3157-3168. doi:[10.1098/rstb.2009.0055](http://dx.doi.org/10.1098/rstb.2009.0055). 2009.

[[Gibson2000]] Gibson, Michael A. and Bruck, Jehoshua. [Efficient Exact Stochastic Simulation of Chemical Systems with Many Species and Many Channels](http://dx.doi.org/10.1021/jp993732q). J. Phys. Chem. A, 104(9): 1876-1889. doi:[10.1021/jp993732q](http://dx.doi.org/10.1021/jp993732q). 2000. 

[[NetLogo]] Wilensky, U. NetLogo. [http://ccl.northwestern.edu/netlogo/](http://ccl.northwestern.edu/netlogo/). Center for Connected Learning and Computer-Based Modeling, Northwestern University. Evanston, IL. 1999.

[Brown2009]: http://dx.doi.org/10.1098/rstb.2009.0055
[Gibson2000]: http://dx.doi.org/10.1021/jp993732q
[NetLogo]: http://ccl.northwestern.edu/netlogo/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
1.0E-8 
    org.nlogo.sdm.gui.AggregateDrawing 29 
        org.nlogo.sdm.gui.StockFigure "attributes" "attributes" 1 "FillColor" "Color" 225 225 182 165 242 60 40 
            org.nlogo.sdm.gui.WrappedStock "W-num" "initial-W-density" 0   
        org.nlogo.sdm.gui.StockFigure "attributes" "attributes" 1 "FillColor" "Color" 225 225 182 451 231 60 40 
            org.nlogo.sdm.gui.WrappedStock "C-num" "initial-C-density" 0   
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 213 120 30 30  
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 491 125 30 30  
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 286 -3 30 30  
        org.nlogo.sdm.gui.RateConnection 3 178 294 123 398 69 502 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 47 499 30 30   
            org.nlogo.sdm.gui.WrappedRate "W-num * N-num" "W-competition" REF 2 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 11 
        org.nlogo.sdm.gui.RateConnection 3 185 294 156 397 128 501 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 109 499 30 30   
            org.nlogo.sdm.gui.WrappedRate "W-num * x-wild-cost" "wild-cost" REF 2 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 17 
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 228 22 30 30  
        org.nlogo.sdm.gui.RateConnection 3 241 52 221 141 201 230 NULL NULL 0 0 0 
            org.jhotdraw.figures.ChopEllipseConnector REF 20  
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.nlogo.sdm.gui.WrappedRate "W-num * b-shared-benefit * W-num / N-num" "W-benefit" 
                org.nlogo.sdm.gui.WrappedReservoir  REF 2 0   
        org.nlogo.sdm.gui.RateConnection 3 201 294 222 398 244 502 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 231 501 30 30   
            org.nlogo.sdm.gui.WrappedRate "W-num * a-toxin-production * C-num / N-num" "w-toxin" REF 2 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 29 
        org.nlogo.sdm.gui.ConverterFigure "attributes" "attributes" 1 "FillColor" "Color" 130 188 183 309 165 50 50 
            org.nlogo.sdm.gui.WrappedConverter "W-num + C-num" "N-num"   
        org.nlogo.sdm.gui.RateConnection 3 210 294 260 398 311 502 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 301 500 30 30   
            org.nlogo.sdm.gui.WrappedRate "W-num * e-bacteriocinogen * C-num / N-num" "bacteriocinogen" REF 2 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 37 
        org.nlogo.sdm.gui.RateConnection 3 465 283 420 381 376 479 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 355 477 30 30   
            org.nlogo.sdm.gui.WrappedRate "C-num * N-num" "C-competition" REF 4 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 43 
        org.nlogo.sdm.gui.RateConnection 3 482 283 486 383 491 484 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 476 483 30 30   
            org.nlogo.sdm.gui.WrappedRate "C-num * q-cheater-cost" "cheater-cost" REF 4 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 49 
        org.nlogo.sdm.gui.RateConnection 3 498 283 553 383 609 483 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.jhotdraw.figures.ChopEllipseConnector 
                org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 600 480 30 30   
            org.nlogo.sdm.gui.WrappedRate "C-num * a-toxin-production * C-num / N-num" "C-toxin" REF 4 
                org.nlogo.sdm.gui.WrappedReservoir  0   REF 55 
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 523 33 30 30  
        org.nlogo.sdm.gui.RateConnection 3 535 62 512 140 489 219 NULL NULL 0 0 0 
            org.jhotdraw.figures.ChopEllipseConnector REF 58  
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.nlogo.sdm.gui.WrappedRate "C-num * b-shared-benefit * W-num / N-num" "C-benefit" 
                org.nlogo.sdm.gui.WrappedReservoir  REF 4 0   
        org.nlogo.sdm.gui.ConverterFigure "attributes" "attributes" 1 "FillColor" "Color" 130 188 183 314 82 50 50 
            org.nlogo.sdm.gui.WrappedConverter "wild-virulence * W-num + C-num" "vir-num"   
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 134 24 30 30  
        org.nlogo.sdm.gui.RateConnection 3 152 54 170 142 188 230 NULL NULL 0 0 0 
            org.jhotdraw.figures.ChopEllipseConnector REF 66  
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.nlogo.sdm.gui.WrappedRate "W-num" "W-growth" 
                org.nlogo.sdm.gui.WrappedReservoir  REF 2 0   
        org.nlogo.sdm.gui.ReservoirFigure "attributes" "attributes" 1 "FillColor" "Color" 192 192 192 417 35 30 30  
        org.nlogo.sdm.gui.RateConnection 3 435 65 453 142 472 219 NULL NULL 0 0 0 
            org.jhotdraw.figures.ChopEllipseConnector REF 72  
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.nlogo.sdm.gui.WrappedRate "C-num" "C-growth" 
                org.nlogo.sdm.gui.WrappedReservoir  REF 4 0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
