globals [
  seed

 max-infected              ;; Maximum number of agents infected at a single time point
 max-infected-time         ;; time step at which the above occurs
 end-time                  ;; time step at which the number of infected = 0 (DC: redundant due to step counter already output with csv file?)
 last-susceptible          ;; number of agents who were never infected. Subtracting from population size would give number of recovered at end.
 final-epi-size            ;; number of total people who got infected throughout the epidemic (N - last-susceptible)

 total-helper-infected     ;; total number of helpers that got infected throughout epidemic
 total-normal-infected     ;; total number of non-helpers infected throughout epidemic

 total-contacts             ;; interactions for every agent
 total-contact-list         ;; list of total contacts for each time step

 r0                        ;; r0 calculated by averaging number of people one indiv infects throughout time as infected
 length-infection          ;; average length of infection for initially infected agents

]

turtles-own [
initial-infected?          ;; turtle owned marker indicating if one of the initial infected agents
help                       ;; each helper's contribution depending on surroundings and maxhelp set by user, calc at each time step for each helper, normal setting = 0
totalInfected              ;; total number agents that a turtle infected thus far
helper-surrounding         ;; keeps track of how many helpers are around each sick individual at each time step
infected-surrounding       ;; keeps track of how many infected are around each helper at each time step
finish-infected            ;; (initial infected agents) how long initial agents were sick
]


to setup
  clear-all
  set seed new-seed
  random-seed seed
  resize-world 0 world-size 0 world-size

  set final-epi-size 0
  set total-helper-infected 0
  set total-normal-infected 0
  set total-contacts 0
  set total-contact-list []
  set max-infected 0
  set max-infected-time 0
  set end-time 0
  set last-susceptible 0
  turtle-creation

  ask patches [
    set pcolor 8
    if pxcor mod 2 = 1 and pycor mod 2 = 0 [set pcolor 9.9] if pxcor mod 2 = 0 and pycor mod 2 = 1 [set pcolor 9.9]]
  ask turtles [ set pen-mode "up"]

  reset-ticks
end

to turtle-creation
  create-turtles N-H  [                               ;; create helpers
    set shape "person"
    set color black
    setxy random-xcor random-ycor
    set initial-infected? false
    ]
  if count turtles < N [                              ;; if helpers do not make up entire population, fill rest with default
    create-turtles (N - count turtles)[               ;; could theoretically put in more helpers than N, population size would be # helpers rather than N
      set color black
      setxy random-xcor random-ycor
      set initial-infected? false
    ]
  ]
  ask n-of N-I turtles [
    set initial-infected? true                        ;; initial infected
    set color red
  ]
end

to go
  if count turtles with [color = red] = 0 [                         ;; check to see if epidemic over, do ending measurements
    end-stats
    calculate-r0
    stop]

  tracking
  contacts
  targets-calc
  move
  infection-transmission
  help-calc
  recover
  update-plots
  tick-advance 1

end

to tracking
  if count turtles with [color = red] > max-infected [              ;; keep writing over largest number of infected
    set max-infected count turtles with [color = red]               ;; record and write over time at which maximum number of infected achieved
    set max-infected-time ticks
  ]
end

to targets-calc
  ask turtles with [color = red] [
    set helper-surrounding count turtles with [shape = "person" and color != red] in-radius hrad      ;;if sick, count number of helpers in help radius
  ]
  ask turtles with [color != red] [                                                                   ;; don't care about number of helpers surrounding the healthy or recovered
    set helper-surrounding 0
  ]
  ask turtles with [shape = "person" and color != red] [
    set infected-surrounding count turtles with [color = red] in-radius hrad
  ]
  ask turtles with [shape != "person"] [
    set infected-surrounding 0
  ]
  ask turtles with [shape = "person" and color = red] [
    set infected-surrounding 0
  ]
end

to contacts
  ask turtles [
    set total-contacts total-contacts + count turtles in-radius infrad
  ]
set total-contact-list lput total-contacts total-contact-list

end

to move


 if interaction-type = "random-move" [                                                      ;; all turtles move randomly
   ask turtles [
     right (random-float 2 * 90 ) - 90
    fd 1
   ]
 ]

 if interaction-type = "helper-seek" [
   ask turtles with [shape = "person" and color = red] [                                    ;; sick helpers move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]

   ask turtles with [shape = "default"][                                                    ;; non-helpers move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [shape = "person" and color != red][                                    ;; non-sick helpers seek the sick
    ifelse any? turtles with [color = red][
      let target min-one-of turtles with [color = red] [distance myself]                    ;; move toward nearest sick person
      face target
      fd 1
    ]
    [
      right (random-float 2 * 90 ) - 90                                                     ;; if no one is sick, move normally
      fd 1
    ]
   ]
 ]

 if interaction-type = "helper-smart-seek" [                                                ;; non-helpers move randomly
   ask turtles with [shape = "default"][
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [shape = "person" and color = red] [                                    ;; sick helpers move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [shape = "person" and color != red][                                    ;; non-sick helpers seek...
     ifelse any? turtles with [color = red and (helper-surrounding = 0)][
       let possible-targets turtles with [color = red]
       let targets possible-targets with [helper-surrounding = 0]
       let target min-one-of targets [distance myself]                                      ;; nearest sick agent without a helper (helper-surrounding = 0)
       face target
       fd 1
     ]
     [
       right (random-float 2 * 90 ) - 90                                                    ;; if all helpers busy or all are non-sick, move randomly
       fd 1
     ]
   ]
 ]

 if interaction-type = "infected-seek"[
   ask turtles with [shape = "default" and color != red] [                                 ;; ask normal agents to move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [shape = "person" and color != red] [                                  ;; ask non-sick helpers to move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [color = red] [
     ifelse any? turtles with [shape = "person" and color != red] [                        ;; ask infected to seek nearest non-sick helper
       let target min-one-of turtles with [shape = "person" and color != red] [distance myself]
       face target
       fd 1
     ]
     [
       right (random-float 2 * 90 ) - 90                                                   ;; if no non-sick helpers, move randomly
       fd 1
     ]
   ]
 ]

 if interaction-type = "infected-smart-seek" [
   ask turtles with [shape = "default" and color != red] [                                ;; ask normal non-sick agents to move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [shape = "person" and color != red] [                                 ;; ask helper non-sick agents to move randomly
     right (random-float 2 * 90 ) - 90
     fd 1
   ]
   ask turtles with [color = red] [                                                       ;; ask infected to seek nearest helper who isn't already helping someone else
     ifelse any? turtles with [shape = "person" and color != red and (infected-surrounding = 0)] [
       let possible-targets turtles with [shape = "person" and color != red]
       let targets possible-targets with [infected-surrounding = 0]
       let target min-one-of targets [distance myself]
       face target
       fd 1
     ]
     [
       right (random-float 2 * 90 ) - 90                                                 ;; if no non-sick helpers free, move randomly
       fd 1
     ]
   ]
 ]

if interaction-type = "dual-smart-seek" [
  ask turtles with [shape = "default" and color != red] [                                ;; ask normal non-sick agents to move randomly
    right (random-float 2 * 90 ) - 90
    fd 1
  ]
  ask turtles with [shape = "person" and color != red][                                    ;; non-sick helpers seek...
    ifelse any? turtles with [color = red and (helper-surrounding = 0)][
      let possible-targets turtles with [color = red]
      let targets possible-targets with [helper-surrounding = 0]
      let target min-one-of targets [distance myself]                                      ;; nearest sick agent without a helper (helper-surrounding = 0)
      face target
      fd 1
    ]
    [
      right (random-float 2 * 90 ) - 90                                                    ;; if all helpers busy or all are non-sick, move randomly
      fd 1
    ]
  ]
  ask turtles with [color = red] [                                                       ;; ask infected to seek nearest helper who isn't already helping someone else
    ifelse any? turtles with [shape = "person" and color != red and (infected-surrounding = 0)] [
      let possible-targets turtles with [shape = "person" and color != red]
      let targets possible-targets with [infected-surrounding = 0]
      let target min-one-of targets [distance myself]
      face target
      fd 1
    ]
    [
      right (random-float 2 * 90 ) - 90                                                 ;; if no non-sick helpers free, move randomly
      fd 1
    ]
  ]
]

end

to infection-transmission
  ask turtles with [color = black and shape != "person"] [                                ;; normal suceptible agents calculate prob infection based on frac infected neighbors and infect-prob
    if (random-float 1) < infect-prob * count turtles with [color = red] in-radius infrad / count turtles in-radius infrad [
        set color red
        ask turtles with [color = red] in-radius infrad [
          set totalInfected totalInfected + 1                                             ;; infected how many agents across entire infective period
        ]
    ]
  ]
  ask turtles with [color = black and shape = "person"] [                                 ;; helpers get decreased chance of infection.
    if (random-float 1) < (1 - helper-immunity) * infect-prob * count turtles with [color = red] in-radius infrad / count turtles in-radius infrad [
        set color red
        ask turtles with [color = red] in-radius infrad [
          set totalInfected totalInfected + 1                                             ;; infected how many agents across entire infective period
        ]
    ]
  ]
end

to help-calc
  ask turtles with [shape = "person" and color != red]
  [
    ifelse any? turtles with [color = red] in-radius hrad [
       set help maxhelp / count turtles with [color = red] in-radius hrad                 ;; ideal "maxhelp" from slider, helper splits effort between sick in radius
    ]
    [
       set help 0                                                                         ;; no one to help, set help 0
    ]
  ]

end

to recover
  ask turtles with [color = red][
    ifelse any? turtles with [shape = "person" and color != red] in-radius hrad [        ;; ask helpers for help
      let totalhelp 0
      ask turtles with [shape = "person" and color != red] in-radius hrad [
        set totalhelp totalhelp + help
        if totalhelp > maxhelp [set totalhelp maxhelp]                                   ;; if aggregate help exceeds maxhelp, set total help back to max
      ]
      if (random-float 1) < recover-prob + totalhelp [                                   ;; increased probability of recovery with help
        set color blue
        if initial-infected? [set finish-infected ticks]
      ]
    ]
    [ if (random-float 1) < recover-prob [                                               ;; probability of recovery without help
        set color blue
        if initial-infected? [set finish-infected ticks]
      ]
      ]
  ]
end


to calculate-r0
  set r0 mean [totalInfected] of turtles with [initial-infected?]                       ;; mean number infected by individual during entire infective state
  set length-infection mean [finish-infected] of turtles with [initial-infected?]       ;; average length of infection for initially infected agents
end

to end-stats
  set end-time ticks
  set last-susceptible count turtles with [color = black]
  set final-epi-size (N - last-susceptible)
  set total-helper-infected count turtles with [color = blue and shape = "person"]
  set total-normal-infected count turtles with [color = blue and shape = "default"]
end
@#$#@#$#@
GRAPHICS-WINDOW
195
65
478
369
-1
-1
5.81
1
10
1
1
1
0
1
1
1
0
46
0
46
1
1
1
ticks
30.0

SLIDER
5
65
177
98
N
N
N-H
1000
1000
1
1
NIL
HORIZONTAL

BUTTON
16
20
82
53
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
99
20
162
53
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
180
20
243
53
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
100
177
133
N-H
N-H
0
1000
6
1
1
NIL
HORIZONTAL

SLIDER
5
180
177
213
hrad
hrad
0
2.5
0.5
.1
1
NIL
HORIZONTAL

SLIDER
5
215
177
248
infrad
infrad
0
2.5
0.5
.1
1
NIL
HORIZONTAL

SLIDER
5
260
177
293
infect-prob
infect-prob
0
1
0.26
.01
1
NIL
HORIZONTAL

SLIDER
5
295
177
328
recover-prob
recover-prob
0
1
0.01
.001
1
NIL
HORIZONTAL

SLIDER
5
135
177
168
N-I
N-I
0
200
6
1
1
NIL
HORIZONTAL

PLOT
565
25
1165
325
Turtle Count
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Susceptibles" 1.0 0 -16777216 true "" "plot count turtles with [color = black]"
"Infected" 1.0 0 -2674135 true "" "plot count turtles with [color = red]"
"Recovered" 1.0 0 -13345367 true "" "plot count turtles with [color = blue]"

SLIDER
5
345
177
378
maxhelp
maxhelp
0
1
0.05
.01
1
NIL
HORIZONTAL

MONITOR
565
340
622
385
NIL
r0
3
1
11

SLIDER
285
25
457
58
world-size
world-size
0
200
46
1
1
NIL
HORIZONTAL

SLIDER
5
380
177
413
helper-immunity
helper-immunity
0
1
0
1
1
NIL
HORIZONTAL

CHOOSER
5
420
162
465
interaction-type
interaction-type
"random-move" "helper-seek" "helper-smart-seek" "infected-seek" "infected-smart-seek" "dual-smart-seek"
1

PLOT
670
340
870
490
Total Contacts
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"" 1.0 0 -16777216 true "" "plot total-contacts"

@#$#@#$#@

## WHAT IS IT?
 This model illustrates the impact of "helping" agents on the outcome of epidemics.

## HOW IT WORKS
At the beginning of a simulation, N agents are placed on the map. Some of these are assigned as helpers, while the others are default agents. A small number begin as infected. As the simulation progresses, agents move according to the movement strategy selected by the "interaction-type" chooser. These movement strategies include random movement, infected seeking, helper seeking, infected smart seeking, helper smart seeking, and dual smart seeking. These strategies are discussed in more detail in the How to Use It section below. When an infected agent and susceptible agent are in contact, there is a chance of disease transmission. When a helping agent is in contact with an infected agent, the helping agent raises the probability of the infected agent's recovery at that time step. When no more infected agents remain, the simulation ends.

## HOW TO USE IT
Run the model a few times with the default values. The Turtle Count plot keeps track of the number of agents of each color (black: susceptible, red: infected, blue: recovered). The Total Contacts plot keeps track of how many times agents have been within each other's infrad. The r0 display shows the r0 at the end of the simulation. R0 is the average number of agents infected by the initially infected agents before they recover.

world-size: Specifies the length and width of the world (world-size + 1 by world-size + 1)

N: specifies the total number of agents in the simulation.

N-H: specifies the number of helping agents in the simulation (must be smaller than N).

N-I: specifies the number of infected agents at the beginning of the simulation. Helping agents can be chosen as initially infected.

infrad: specifies a radius around an infected individual. Agents within this radius 			interact with the infected agent, and disease can be transmitted.

hrad: specifies a radius around a helper agent. Infected agents within this radius			receive an increase in probability of recovery. If there are multiple agents 			within a single helper's radius, the total help a helper can provide is split 			evenly between them.

infectprob: Given an interaction with an infected individual, the probability that an 			agent will become sick (modified by the fraction of infected agents within the 		infrad).

recoverprob: Each infected agent recovers with this probability at each time step.

maxhelp: The maximum amount a helper agent can increase the recoverprob of a single 			agent.

helper-immunity: At 0, the helper agents are just as susceptible as the default agents.			 At 1, the helper agents are immune to disease. Intermediate values modify the 			probability of a helper agent getting infected.

interaction-type: Specifies how each type of agent will move.

	Random Movement: Each agent moves randomly.

	Helper Seek: Helping agents move toward the nearest infected.

	Infected Seek: Infected agents move toward the nearest helping agent (who is not also infected).

	Helper Smart Seek: Helper agents move as in Helper Seek, but only move toward infected agents without a helper already nearby.

	Infected Smart Seek: Infected agents move as in Infected Seek, but only move toward helping agents that are not already nearby an infected.

	Dual Smart Seek: Both Infected and Helper agents move according to Helper Smart Seek and Infected Smart Seek.


## THINGS TO NOTICE
Notice how the counts for each agent type vary during the epidemic. Do the curves change shape if you run the model multiple times with the same parameters? Does R0 change? Multiple measured variables are assigned throughout the simulation. Consider plotting these or displaying these: max-infected, max-infected-time, end-time, final-epi-size.


## THINGS TO TRY

Change the probabilities of infection and recovery to extreme values. Do your observations match what you expected? How does the number of agents initially infected affect the final epidemic size? How many do you need to get at least 10% of the agents infected every time? What is the optimal number of helpers? Should they be immune? What level of immunity is realistic?


## EXTENDING THE MODEL
Consider creating more interaction-types. Is there an optimal way to get the infected to the helpers? How could the existing interaction types be modified to make them more realistic?

Consider adding another stage of the disease. What if agents did not know they were sick for a specified number of ticks? Would the epidemic spread faster?


## CREDITS AND REFERENCES
Creator: Dorothy Catey.

Thanks to advisor Bert Baumgaertner

Support came from the 2015 Brian and Gayle Hill Undergraduate Research Fellowship and NIH Award Number P20GM104420.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -7500403 true true 135 285 195 285 270 90 30 90 105 285
Polygon -7500403 true true 270 90 225 15 180 90
Polygon -7500403 true true 30 90 75 15 120 90
Circle -1 true false 183 138 24
Circle -1 true false 93 138 24

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="HradNoSight" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>last-susceptible</metric>
    <metric>seed</metric>
    <steppedValueSet variable="hrad" first="0.5" step="0.25" last="2.5"/>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="H-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="N-H-min" first="0" step="25" last="500"/>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sight-rad">
      <value value="2.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-A">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rate-H-N">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H-max">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="AvoidersAffectDense25" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>last-susceptible</metric>
    <metric>seed</metric>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="H-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H-min">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="50"/>
      <value value="36"/>
      <value value="30"/>
      <value value="24"/>
      <value value="22"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sight-rad">
      <value value="2.5"/>
    </enumeratedValueSet>
    <steppedValueSet variable="N-A" first="0" step="25" last="500"/>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rate-H-N">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H-max">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="AvoiderAffectSmall" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>last-susceptible</metric>
    <metric>seed</metric>
    <steppedValueSet variable="N-A" first="0" step="50" last="500"/>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H-max">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H-min">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sight-rad">
      <value value="2.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="H-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rate-H-N">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="HelpersAffectDense25" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>last-susceptible</metric>
    <metric>seed</metric>
    <metric>N</metric>
    <steppedValueSet variable="N-H-min" first="0" step="25" last="500"/>
    <enumeratedValueSet variable="world-size">
      <value value="50"/>
      <value value="36"/>
      <value value="30"/>
      <value value="24"/>
      <value value="22"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rate-H-N">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-A">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="helper-immunity">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="H-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sight-rad">
      <value value="2.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H-max">
      <value value="80"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="interaction-type-100-reps" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000000"/>
    <metric>max-infected</metric>
    <metric>max-infected-time</metric>
    <metric>end-time</metric>
    <metric>last-susceptible</metric>
    <metric>final-epi-size</metric>
    <metric>total-helper-infected</metric>
    <metric>total-normal-infected</metric>
    <metric>total-SI-contacts-v1</metric>
    <metric>total-SI-contacts-v2</metric>
    <metric>total-HI-contacts-v1</metric>
    <metric>total-HI-contacts-v2</metric>
    <metric>r0</metric>
    <metric>SI-contact-list</metric>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H">
      <value value="0"/>
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
      <value value="125"/>
      <value value="150"/>
      <value value="200"/>
      <value value="300"/>
      <value value="400"/>
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="helper-immunity">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interaction-type">
      <value value="&quot;random-move&quot;"/>
      <value value="&quot;helper-seek&quot;"/>
      <value value="&quot;helper-smart-seek&quot;"/>
      <value value="&quot;infected-seek&quot;"/>
      <value value="&quot;infected-smart-seek&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="46"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="case6-resist-30reps" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000000"/>
    <metric>max-infected</metric>
    <metric>max-infected-time</metric>
    <metric>end-time</metric>
    <metric>last-susceptible</metric>
    <metric>final-epi-size</metric>
    <metric>total-helper-infected</metric>
    <metric>total-normal-infected</metric>
    <metric>total-SI-contacts-v1</metric>
    <metric>total-SI-contacts-v2</metric>
    <metric>total-HI-contacts-v1</metric>
    <metric>total-HI-contacts-v2</metric>
    <metric>r0</metric>
    <metric>SI-contact-list</metric>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H">
      <value value="0"/>
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
      <value value="125"/>
      <value value="150"/>
      <value value="200"/>
      <value value="300"/>
      <value value="400"/>
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="helper-immunity">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interaction-type">
      <value value="&quot;dual-smart-seek&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="46"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="contact-count-simple" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000000"/>
    <metric>max-infected</metric>
    <metric>max-infected-time</metric>
    <metric>last-susceptible</metric>
    <metric>final-epi-size</metric>
    <metric>r0</metric>
    <metric>SI-contact-list</metric>
    <metric>num-inf-list</metric>
    <metric>total-contact-list</metric>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H">
      <value value="0"/>
      <value value="2"/>
      <value value="6"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="30"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
      <value value="150"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="helper-immunity">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interaction-type">
      <value value="&quot;random-move&quot;"/>
      <value value="&quot;helper-seek&quot;"/>
      <value value="&quot;helper-smart-seek&quot;"/>
      <value value="&quot;infected-seek&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="20"/>
      <value value="40"/>
      <value value="46"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="contact-count-simple2" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000000"/>
    <metric>max-infected</metric>
    <metric>max-infected-time</metric>
    <metric>last-susceptible</metric>
    <metric>final-epi-size</metric>
    <metric>r0</metric>
    <metric>SI-contact-list</metric>
    <metric>num-inf-list</metric>
    <metric>total-contact-list</metric>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H">
      <value value="0"/>
      <value value="2"/>
      <value value="6"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="30"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
      <value value="150"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="helper-immunity">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interaction-type">
      <value value="&quot;random-move&quot;"/>
      <value value="&quot;helper-seek&quot;"/>
      <value value="&quot;helper-smart-seek&quot;"/>
      <value value="&quot;infected-seek&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="500"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Big-int-resist-dense" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000000"/>
    <metric>max-infected</metric>
    <metric>max-infected-time</metric>
    <metric>end-time</metric>
    <metric>last-susceptible</metric>
    <metric>final-epi-size</metric>
    <metric>total-helper-infected</metric>
    <metric>total-normal-infected</metric>
    <metric>r0</metric>
    <metric>total-contact-list</metric>
    <metric>length-infection</metric>
    <enumeratedValueSet variable="hrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-H">
      <value value="0"/>
      <value value="4"/>
      <value value="8"/>
      <value value="12"/>
      <value value="16"/>
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
      <value value="60"/>
      <value value="80"/>
      <value value="100"/>
      <value value="150"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="helper-immunity">
      <value value="0"/>
      <value value="0.5"/>
      <value value="0.75"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interaction-type">
      <value value="&quot;random-move&quot;"/>
      <value value="&quot;helper-seek&quot;"/>
      <value value="&quot;helper-smart-seek&quot;"/>
      <value value="&quot;infected-seek&quot;"/>
      <value value="&quot;infected-smart-seek&quot;"/>
      <value value="&quot;dual-smart-seek&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="world-size">
      <value value="28"/>
      <value value="42"/>
      <value value="57"/>
      <value value="65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recover-prob">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maxhelp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N-I">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infrad">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="infect-prob">
      <value value="0.26"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="1000"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
