;; Capture the Flag, by Paul Franz
;; EDUC 390X

;; Development History
;; Version 0.1 - January 27, 2012
;; - Creates flags, players, and playing surface
;; - Supports two players on a single computer
;; - Checks for winner, and ends "go" if condition met
;; - Moves players to edge of screen if captured; returns flag
;; - Change player color with flag capture

;; Version 0.2a - January 30, 2012
;; - Added hubnet base code
;; - Created player movement and monitoring code

;; Version 0.2b - January 31, 2012
;; - Sorted players into teams.
;; - Made players 1.5 size, and all the same shape (labeled for differentiation).
;; - Removed single-player movement code and repurposed capturing and flag-grabbing code.
;; - Made gameworld persistent after game end, so that players don't have to log back in.

;; Version 0.3 - February 1, 2012
;; - Made flags 1.5 size.
;; - Randomized grass color and adjusted neutral zone color.
;; - Added capture and rescue protocols.

;; Version 0.4 - February 1, 2012
;; - Fixed "located at:" display for players.
;; - Cleaned up "go" and "setup" codes by moving code to subroutines.
;; - Allowed for reorganizing teams when players leave.

;; Version 0.5 - February 1, 2012
;; - Added scorekeeping and option to set winning score.
;; - Reset players and flags after each score.
;; - Rebalanced teams after each score in case of quitters.
;; - Improved hubnet client so players can see playing field (theoretically) and current score.

breed [ players player ]
breed [ redflags redflag ]
breed [ blueflags blueflag ]
players-own [ user-id ]

globals [ winner team-counter team-balance blue-score red-score score ]

to startup
  hubnet-reset
  set team-counter 1
end

to setup
  zero-variables
  reset-ticks
  reset-players
  build-field
  set-flags
  balance-teams
end

to zero-variables
  set winner 0
  set score 0
  set blue-score 0
  set red-score 0
end

to reset-players    ;; Resets players to starting position and conditions
  ask turtles with [color = orange ]
    [ set color red ]
  ask turtles with [color = violet ]
    [ set color blue ]
  ask turtles with [shape = "face sad"]
    [ set shape "person"]
  ask players with [color = red ]
    [ setxy (max-pxcor - 2) random-pycor ]
  ask players with [color = blue ]
    [ setxy (min-pxcor + 2) random-pycor ]
end

to build-field  ;; Builds the fields
  ask patches [ set pcolor ((random 3) + 54) ] ;; grassifies the field
  ask patches with [pxcor = 0] ;; creates central "neutral zone"
    [ set pcolor gray + 1]
  ask patches with [pycor < min-pycor + 2 and pxcor > 7 and pxcor < 12]  ;; Creates prison areas
    [ set pcolor gray - 1]
  ask patches with [pycor < min-pycor + 2 and pxcor < -7 and pxcor > -12]
    [ set pcolor gray - 1]
end

to set-flags  ;; (re)places flags
  ask redflags [die]
  ask blueflags [die]
  set-default-shape redflags "flag"
  set-default-shape blueflags "flag"
  create-redflags 1
    [ setxy (max-pxcor - 1) 0 
      set color red
      set size 1.5]
  create-blueflags 1
    [ setxy (min-pxcor + 1) 0
      set color blue
      set size 1.5]
end

to balance-teams  ;; Ensures that both teams have the same number of players
  set team-balance (count players with [color = red] - count players with [color = blue])
    if team-balance > 1 [
      ask one-of players with [color = red] [set color blue
      reset-players ]]
    if team-balance < -1 [
      ask one-of players with [color = blue] [set color red
      reset-players ]]
end

to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-player ]
    [
      ifelse hubnet-exit-message?
      [ remove-player ]
      [ ask players with [user-id = hubnet-message-source]
        [execute-command hubnet-message-tag ]
      ]]]
end

to execute-command [command]
  if command = "Up"
  [ set heading 0
    fd 1
    stop ]
  if command = "Right"
  [ set heading 90
    fd 1
    stop ]
  if command = "Down"
  [ set heading 180
    fd 1
    stop ]
  if command = "Left"
  [ set heading 270
    fd 1
    stop ]
end

to create-new-player
  create-players 1
  [ set user-id hubnet-message-source
    set label user-id
    assign-players
    hubnet-send user-id "Located at:" (word "(" pxcor "," pycor ")")
    hubnet-send user-id "Red" (red-score)
    hubnet-send user-id "Blue" (blue-score)
  ]
end

to assign-players  ;; distributes new players to teams
  if team-counter = 1
    [ set color red
      set shape "person"
      setxy (max-pxcor - 2) random-pycor ]
  if team-counter = 0
    [ set color blue
      set shape "person" 
      setxy (min-pxcor + 2) random-pycor ]
  set size 1.5
  ifelse team-counter = 1 [set team-counter 0] [set team-counter 1 ]
end  
  
to remove-player
  ask players with [ user-id = hubnet-message-source ] [die]
end

to go
  listen-clients
  every 0.1 [ tick ]
  if winner = 1 [
    stop ]
  return-flag
  ask players with [color = red] [
    check-capture-blue
    grab-flag-red
    liberate-red ]
  ask players with [color = blue] [
    check-capture-red
    grab-flag-blue
    liberate-blue ]
  ask players [
    check-victory
    hubnet-send user-id "Located at:" (word "(" pxcor "," pycor ")")
    hubnet-send user-id "Red" (red-score)
    hubnet-send user-id "Blue" (blue-score) ]
  prison-guard
  if score = 1 [reset]
end

to prison-guard ;; Keeps captured players in jail
  ask players with [shape = "face sad" and color = blue and pcolor != gray - 1]
    [ setxy 8 min-pycor ]
  ask players with [shape = "face sad" and color = red and pcolor != gray - 1]
    [ setxy -8 min-pycor ]
end

to return-flag    ;; Returns flags if players carrying them are captured
  if count turtles with [color = orange] = 0 and count blueflags = 0
     [create-blueflags 1
      [ setxy (min-pxcor + 1) 0
      set color blue]]
  if count turtles with [color = violet] = 0 and count redflags = 0
     [create-redflags 1
      [ setxy (max-pxcor - 1) 0 
      set color red]]
end

to check-capture-blue ;; allows red players to capture and imprison opponents
  let bcaptured one-of players-here with [color = blue]
  if bcaptured != nobody and xcor > 0
    [ ask bcaptured [setxy 8 min-pycor]
      ask bcaptured [set color blue]
      ask bcaptured [set shape "face sad"] ]
end

to check-capture-red  ;; allows blue players to capture and imprison opponents
  let rcaptured one-of players-here with [color = red]
  if rcaptured != nobody and xcor < 0
    [ ask rcaptured [setxy max-pxcor max-pycor]
      ask rcaptured [set color red]
      ask rcaptured [set shape "face sad"] ]
end  

to grab-flag-red  ;; allows players to capture the flag
  let flagcaptured one-of blueflags-here
  if flagcaptured != nobody
    [ ask flagcaptured [die]
      set color orange]   ;; changes player color to reflect captured flag
end

to grab-flag-blue  ;; allows blue players to capture the red flag
  let flagcaptured one-of redflags-here
  if flagcaptured != nobody
    [ ask flagcaptured [die]
      set color violet]  ;; changes player color to reflect captured flag
end

to liberate-red  ;; allows red players to rescue captured teammates
  if shape != "face sad"
  [ let rliberated one-of players-here with [color = red and pcolor = gray - 1]
  if rliberated != nobody
    [ ask rliberated [set shape "person"]]]
end

to liberate-blue ;; allows blue players to rescue captured teammates
  if shape != "face sad"
  [ let bliberated one-of players-here with [color = blue and pcolor = gray - 1]
  if bliberated != nobody
    [ ask bliberated [set shape "person"]]]
end

to check-victory  ;; Adds a point to a team when it captures a flag
  if color = violet and xcor = -1
    [ set blue-score blue-score + 1 
      set score 1 ]
  if color = orange and xcor = 1
    [ set red-score red-score + 1 
      set score 1 ]
  if blue-score = winning-score [blue-victory]  ;; checks for victory
  if red-score = winning-score [red-victory]
end

to reset
   reset-players
   set-flags
   balance-teams
   set score 0
end

to blue-victory
  print "Blue Wins"
  set winner 1
end

to red-victory
  print "Red Wins"
  set winner 1
end
@#$#@#$#@
GRAPHICS-WINDOW
242
19
756
554
16
16
15.3
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
57
108
126
141
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
145
108
208
141
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
48
72
220
105
winning-score
winning-score
1
10
2
1
1
NIL
HORIZONTAL

MONITOR
80
23
137
68
Red
red-score
17
1
11

MONITOR
147
24
204
69
Blue
blue-score
17
1
11

@#$#@#$#@
## WHAT IS IT?

This model is a multi-player game of Capture the Flag for use on Hubnet.  This version is 0.5.  I have plans to develop the model further.

## HOW IT WORKS

Students log in to hubnet and each control one player, either on the blue or red team.  The objective is to capture the other team's flag and to carry it back across the center line.

## HOW TO USE IT

Allow students to log in, press setup to balance teams and create the field, and go to begin the competition.  Before the game starts the user can set the "winning score," so the game need not end after the first capture.  Students will be reset and teams rebalanced (in the event of a dropout or login) after the each score.

## THINGS TO NOTICE

It is interesting to see how teams manage their strategies.  Do they work together and distribute tasks (guarding the prison, guarding the flag, attacking the opponent flag, and saving imprisoned allies) or do they engage in a free for all?

## THINGS TO TRY

N/A

## EXTENDING THE MODEL

I have plans to extend the model to include AI players.  An easier task might be further dividing the playing field and allowing for more teams.

## NETLOGO FEATURES

Uses hubnet.  As a beginner in NetLogo, I suspect much of my code is not as efficient as it could be.

## RELATED MODELS

Uri Wilensky also has authored a very different (and more polished) Capture the Flag model available on the NetLogo commons.

## CREDITS AND REFERENCES

N/A
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC9
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
BUTTON
53
67
116
100
Up
NIL
NIL
1
T
OBSERVER
NIL
W

BUTTON
8
106
71
139
Left
NIL
NIL
1
T
OBSERVER
NIL
A

BUTTON
90
106
159
139
Right
NIL
NIL
1
T
OBSERVER
NIL
D

BUTTON
47
146
118
179
Down
NIL
NIL
1
T
OBSERVER
NIL
S

MONITOR
8
10
103
59
Located at:
NIL
3
1

VIEW
211
51
716
556
0
0
0
1
1
1
1
1
0
1
1
1
-16
16
-16
16

MONITOR
73
194
130
243
Blue
NIL
3
1

MONITOR
12
194
69
243
Red
NIL
3
1

@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
