globals [
  guest-count
  host-count
  choice
  gini-index-reserve
  lorenz-points
  base-ratio
  aff-homo
]

;extensions [ import-a ]
;extensions [ rnd ]


breed [hosts host]
breed [guests guest]
breed [trees tree]

hosts-own [
  h-race
  w-ratio
  b-ratio
  a-ratio
  w-cnt ;;expected number of white on the front page
  b-cnt ;;expected number of black on the front page
  a-cnt ;;expected number of asian on the front page
  num-guest
]

guests-own [
  g-race
  cate

]


to setup
  clear-all
  ask patches [set pcolor green - random-float 0.5]

  ;ask patches [set pcolor 56 ]
  ;import-drawing "map.jpg"
  set-default-shape guests "person"
  set-default-shape hosts "house"

  set host-count (white-hosts + black-hosts + asian-hosts)
  ;;set guest-count (white-hosts + black-hosts + asian-hosts)


  ;;add number of hosts according to the input
  create-hosts white-hosts [
    set size 2
    set h-race "White"
    set color 105 ;; white = 105
    set xcor -20 + random 40
    set ycor -20 + random 50
  ]

  create-hosts black-hosts [
    set size 2
    set h-race "Black"
    set color 3 ;; grey = 3
    set xcor -20 + random 40
    set ycor -20 + random 50
  ]

  create-hosts asian-hosts [
    set size 2
    set h-race "Asian"
    set color 25 ;; yellow = 45
    set xcor -20 + random 40
    set ycor -20 + random 50
  ]
  ;;set hosts apart for visual purpose
  ask hosts [move-to one-of patches with [not any? other hosts in-radius 2]]
  update-lorenz-and-gini
  reset-ticks
end


to go
  ;; update the front page reviewer counts for all hosts based on the guest race
  ask hosts [
    if count link-neighbors > 0 [
      set w-ratio count (link-neighbors with [g-race = "White"])/ count link-neighbors
      set b-ratio count (link-neighbors with [g-race = "Black"])/ count link-neighbors
      set a-ratio count (link-neighbors with [g-race = "Asian"])/ count link-neighbors
      if count link-neighbors > front-page [
        set w-cnt w-ratio * front-page ;count (link-neighbors with [g-race = "White"])
        set b-cnt b-ratio * front-page ;count (link-neighbors with [g-race = "Black"])
        set a-cnt a-ratio * front-page ;count (link-neighbors with [g-race = "Asian"])
      ]
      if count link-neighbors <= front-page [
        set w-cnt count (link-neighbors with [g-race = "White"])
        set b-cnt count (link-neighbors with [g-race = "Black"])
        set a-cnt count (link-neighbors with [g-race = "Asian"])
      ]
    ]
    if count link-neighbors = 0 [
    set w-ratio 0
    set b-ratio 0
    set a-ratio 0

    set w-cnt 0
    set b-cnt 0
    set a-cnt 0
    ]
    set num-guest count (link-neighbors)
  ]

  ask links [ set color gray ]
  set guest-count (guest-count + 1)

  create-guest
  make-decision find-partner
  update-lorenz-and-gini

  tick
end


;; used for creating a new guest
to create-guest
  create-guests 1 [
      ;set new? true
      let a-g-prob 1 - w-g-prob - b-g-prob
      let probs ( list (w-g-prob) (b-g-prob) (a-g-prob) )
      set g-race (random-pick-guest-race probs)
      if g-race = "White" [set color 105 set cate 1]
      if g-race = "Black" [set color 3 set cate 2]
      if g-race = "Asian" [set color 25 set cate 3]
    ]

end


to make-decision [listing]
  ask (max-one-of guests [who]) [
        if listing != nobody
        [ create-link-with listing [ set color grey ]
        ;; position the new node near its partner
          move-to listing
          fd 3 + (random 6)
    ]
  ]

end


to-report find-partner

  let temp-race ([g-race] of (max-one-of guests [who]))
  ;; update the attractiveness of hosts based on the guest race --> attractiveness is a list of probabilities
  ifelse gr_homo? [
    ;;###########################
    ;;###########################
    ;;if switch is ON, pick host based on the coefficient and intercept
    let w-pref reduce sentence (map [[a] -> [w-cnt] of hosts with [who = a]] (range 0 (host-count)))
    let b-pref reduce sentence (map [[a] -> [b-cnt] of hosts with [who = a]] (range 0 (host-count)))
    let a-pref reduce sentence (map [[a] -> [a-cnt] of hosts with [who = a]] (range 0 (host-count)))

    if temp-race = "White" [
      let coef (sentence (n-values white-hosts [w-w-coef]) (n-values black-hosts [w-b-coef]) (n-values asian-hosts [w-a-coef]))
      let ic (sentence (n-values white-hosts [w-w-ic]) (n-values black-hosts [w-b-ic]) (n-values asian-hosts [w-a-ic]))
      let att (map [[a b c] -> (a * b + c) ] coef w-pref ic)
      set choice (random-pick-host att)
      ;set choice first rnd:weighted-one-of-list (map list host-order att) last
      report host choice
    ]

    if temp-race = "Black" [
      let coef (sentence (n-values white-hosts [b-w-coef]) (n-values black-hosts [b-b-coef]) (n-values asian-hosts [b-a-coef]))
      let ic (sentence (n-values white-hosts [b-w-ic]) (n-values black-hosts [b-b-ic]) (n-values asian-hosts [b-a-ic]))
      let att (map [[a b c] -> (a * b + c) ] coef b-pref ic)
      set choice (random-pick-host att)
      ;set choice first rnd:weighted-one-of-list (map list host-order att) last
      report host choice
    ]

    if temp-race = "Asian" [
      let coef (sentence (n-values white-hosts [a-w-coef]) (n-values black-hosts [a-b-coef]) (n-values asian-hosts [a-a-coef]))
      let ic (sentence (n-values white-hosts [a-w-ic]) (n-values black-hosts [a-b-ic]) (n-values asian-hosts [a-a-ic]))
      let att (map [[a b c] -> (a * b + c) ] coef a-pref ic)
      set choice (random-pick-host att)
      ;set choice first rnd:weighted-one-of-list (map list host-order att) last
      report host choice
    ]
    ] [
    ;;###########################
    ;;###########################
    ;;if switch is off, pick host without considering guest-reviewer homophily

    let w-a-1 1 - w-w-1 - w-b-1
    let b-a-1 1 - b-w-1 - b-b-1
    let a-w-1 1 - a-a-1 - a-b-1

    if temp-race = "White" [
      let gh-pref (sentence (n-values white-hosts [w-w-1]) (n-values black-hosts [w-b-1]) (n-values asian-hosts [w-a-1]))
      set gh-pref (map [[a] -> a * 1] gh-pref )
      set choice (random-pick-host gh-pref)
      report host choice
    ]

    if temp-race = "Black" [
      let gh-pref (sentence (n-values white-hosts [b-w-1]) (n-values black-hosts [b-b-1]) (n-values asian-hosts [b-a-1]))
      set gh-pref (map [[a] -> a * 1] gh-pref )
      set choice (random-pick-host gh-pref)
      report host choice
    ]

    if temp-race = "Asian" [
      let gh-pref (sentence (n-values white-hosts [a-w-1]) (n-values black-hosts [a-b-1]) (n-values asian-hosts [a-a-1]))
      set gh-pref (map [[a] -> a * 1] gh-pref )
      set choice (random-pick-host gh-pref)
      report host choice
    ]

   ]


end

to-report random-pick-guest-race [tmp]
  let var (1 / sum tmp)
  let _ps map [[a] -> a * var] tmp  ;[0.1 0.2 0.1 0.4 0.1 0.01]
  let _r random-float 1
  let _lst [ "White" "Black" "Asian" ]
  let _i 0
  while [_r >= item _i _ps] [
   set _r (_r - item _i _ps)
   set _i (_i + 1) ]
  report item _i _lst
end


to-report random-pick-host [tmp]
  let var (1 / sum tmp)
  let _ps map [[a] -> a * var] tmp  ;[0.1 0.2 0.1 0.4 0.1 0.01]
  let _r random-float 1
  let _lst (range 0 host-count)
  let _i 0
  while [_r >= item _i _ps] [
   set _r (_r - item _i _ps)
   set _i (_i + 1) ]
  report item _i _lst
end


;;;;;;Gini coefficient
to update-lorenz-and-gini
  let sorted-wealths sort [num-guest] of hosts
  let total-wealth sum sorted-wealths
  let wealth-sum-so-far 0
  let index 0
  set gini-index-reserve 0
  set lorenz-points []
  ;; now actually plot the Lorenz curve -- along the way, we also
  ;; calculate the Gini index.
  ;; (see the Info tab for a description of the curve and measure)
  if total-wealth > 0 [
  repeat host-count [
    set wealth-sum-so-far (wealth-sum-so-far + item index sorted-wealths)
    set lorenz-points lput ((wealth-sum-so-far / total-wealth) * 100) lorenz-points
    set index (index + 1)
    set gini-index-reserve
      gini-index-reserve +
      (index / host-count) -
      (wealth-sum-so-far / total-wealth)
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
530
19
888
609
-1
-1
5.7525
1
10
1
1
1
0
0
0
1
-30
30
-50
50
0
0
1
ticks
30.0

BUTTON
198
132
259
165
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
198
167
259
200
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
5
41
91
101
white-hosts
2.0
1
0
Number

INPUTBOX
180
41
260
101
asian-hosts
2.0
1
0
Number

BUTTON
124
207
259
251
go foever
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
132
116
165
w-g-prob
w-g-prob
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
5
170
116
203
b-g-prob
b-g-prob
0
1 - w-g-prob
0.25
0.01
1
NIL
HORIZONTAL

SLIDER
3
299
100
332
w-w-1
w-w-1
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
104
298
200
331
w-b-1
w-b-1
0
1 - w-w-1
0.25
0.01
1
NIL
HORIZONTAL

SLIDER
105
345
200
378
b-w-1
b-w-1
0
1 - b-b-1
0.25
0.01
1
NIL
HORIZONTAL

SLIDER
3
346
101
379
b-b-1
b-b-1
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
106
393
200
426
a-b-1
a-b-1
0
1 - a-a-1
0.25
0.01
1
NIL
HORIZONTAL

SLIDER
3
393
101
426
a-a-1
a-a-1
0
1
0.5
0.01
1
NIL
HORIZONTAL

INPUTBOX
94
41
176
101
black-hosts
2.0
1
0
Number

TEXTBOX
5
264
184
294
Guest-host homophily \nproportion (row sum=1)
12
0.0
1

TEXTBOX
1
106
201
136
Guest race proportion (sum = 1)
12
0.0
1

SLIDER
6
478
105
511
w-w-2
w-w-2
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
108
478
200
511
w-b-2
w-b-2
0
1 - w-w-2
0.25
0.01
1
NIL
HORIZONTAL

TEXTBOX
6
443
145
473
Guest-reviewer \nhomophily (row sum=1)
12
0.0
1

SLIDER
108
523
201
556
b-w-2
b-w-2
0
1 - b-b-2
0.25
0.01
1
NIL
HORIZONTAL

SLIDER
5
523
105
556
b-b-2
b-b-2
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
108
568
202
601
a-b-2
a-b-2
0
1 - a-a-2
0.25
0.01
1
NIL
HORIZONTAL

TEXTBOX
5
16
249
34
Set the # of hosts by race BEFORE run
13
14.0
1

PLOT
269
20
526
252
Mean # of guests by host race
Time
# of guests
0.0
10.0
0.0
5.0
true
true
"" ""
PENS
"White" 1.0 0 -13345367 true "" "plot (mean ( map [[a] -> (count [link-neighbors] of host a)] (range 0 white-hosts) ) ) "
"Black" 1.0 0 -16449023 true "" "plot (mean ( map [[a] -> (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)) ) ) \n"
"Asian" 1.0 0 -955883 true "" "plot (mean ( map [[a] -> (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count)  ) ) \n"

MONITOR
6
207
117
252
a-g-prob
(1 - w-g-prob - b-g-prob)
2
1
11

MONITOR
205
295
265
340
w-a-1
1 - w-w-1 - w-b-1
2
1
11

MONITOR
205
339
265
384
b-a-1
1 - b-b-1 - b-w-1
2
1
11

MONITOR
205
384
265
429
a-w-1
1 - a-a-1 - a-b-1
2
1
11

SLIDER
6
568
105
601
a-a-2
a-a-2
0
1
0.5
0.01
1
NIL
HORIZONTAL

MONITOR
204
564
263
609
a-w-2
1 - a-a-2 - a-b-2
2
1
11

MONITOR
204
520
263
565
b-a-2
1 - b-w-2 - b-b-2
2
1
11

MONITOR
204
476
263
521
w-a-2
1 - w-w-2 - w-b-2
2
1
11

PLOT
270
261
526
429
Lorenz Curve
Host Pop%
# of Guest %
0.0
100.0
0.0
100.0
false
true
"" ""
PENS
"lorenz" 1.0 0 -2674135 true "" "plot-pen-reset\nset-plot-pen-interval 100 / host-count\nplot 0\nforeach lorenz-points plot"
"equal" 100.0 0 -16777216 true "plot 0\nplot 100" ""

PLOT
270
438
526
609
Gini-Index (# of guests) 
Time
Gini
0.0
50.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot (gini-index-reserve / host-count) / 0.5"

SWITCH
158
439
264
472
gr_homo?
gr_homo?
0
1
-1000

INPUTBOX
123
132
192
199
front-page
6.0
1
0
Number

INPUTBOX
902
18
968
78
w-w-coef
0.058
1
0
Number

INPUTBOX
902
83
967
143
w-w-ic
0.484
1
0
Number

INPUTBOX
903
167
969
227
b-w-coef
-0.112
1
0
Number

INPUTBOX
904
231
969
291
b-w-ic
0.737
1
0
Number

INPUTBOX
1057
312
1125
372
a-a-coef
0.123
1
0
Number

INPUTBOX
1055
231
1120
291
b-a-ic
0.088
1
0
Number

INPUTBOX
976
16
1038
76
w-b-coef
-0.05
1
0
Number

INPUTBOX
1048
16
1109
76
w-a-coef
-0.008
1
0
Number

INPUTBOX
976
84
1038
144
w-b-ic
0.414
1
0
Number

INPUTBOX
1048
83
1110
143
w-a-ic
0.1
1
0
Number

INPUTBOX
979
167
1046
227
b-b-coef
0.124
1
0
Number

INPUTBOX
1054
167
1121
227
b-a-coef
-0.011
1
0
Number

INPUTBOX
980
231
1047
291
b-b-ic
0.173
1
0
Number

INPUTBOX
902
312
970
372
a-w-coef
-0.078
1
0
Number

INPUTBOX
980
313
1050
373
a-b-coef
-0.043
1
0
Number

INPUTBOX
902
379
974
439
a-w-ic
0.741
1
0
Number

INPUTBOX
980
380
1051
440
a-b-ic
0.282
1
0
Number

INPUTBOX
1057
380
1126
440
a-a-ic
-0.024
1
0
Number

@#$#@#$#@
## WHAT IS IT?

This model simulates the influence of racial homophily in sharing economy accommodations (like AirBnB). In this simulation, we can see how how guests' racial homophily (guests' preference for same-race hosts and guests' preference for same-race reviewers who previously stayed) influence their choices of hosts. 

Racial groups and their icon colors: White = Blue; Black = Grey; Asian = Orange.

## HOW IT WORKS

First, we can set the proportion of hosts (house icon) by race on the market.

Second, the guests (person icon) enter in the market by probability (can be adjusted during the model run). The probabilities of guests of different races entering the market sum up to 1. We only need to adjust the first two probabilities and the third one will be displayed automatically. 

Third, the guests' choices of hosts are based on 1) guest-host homopihily and 2) guest-reviewer homophily. 

1) Guest-host homopihilies are displayed under "Guest-host homophily proportion (sum=1)". "w" = "Whhite", "b" = "Black", and "a" = "Asian". The first "w/b/a" means the racial groups for guests, while the second "w/b/a" means the racial groups for hosts. The last digit means this it the first type of homophily preference. Thus, "w-w-1" means "White guests' preference for White hosts".

For the guest-host homopihily, we can set how a guest of a specific race will choose a host based on their racial groups. For instance, the default setting is that the probability of guests' choosing same-race hosts is double (0.5) as the probability of choosing different-race hosts (0.25). 

2) Guest-reviewer homophily variables are displayed under "Guest-reviewer homophily proportion (sum=1)". Similarly, "w" = "Whhite", "b" = "Black", and "a" = "Asian". The first "w/b/a" means the racial groups for guests, while the second "w/b/a" means the racial groups for hosts. The last digit means this it the second type of homophily preference. Thus, "w-w-2" means "White guests' preference for the ratio of previous White reviewers".

For the guest-reviewer homophily, we can set how a guest of a racial group will be influenced by the ratio of previous reviewers from a specific racial group. Our assumption is that the more a guest sees a place being stayed by people from their own racial groups, the more likely they are going to choose this place (host). In other words this type of homophily is the extent to which White/Black/Asian guests prefer to stay with hosts that have already been stayed by other White/Black/Asian guests. In our default setting, the extent is double for same-race reviewers as different-race reviewers. 

Both types of homophily preferences can be adjusted while the model is running. 

The three plots from the top to the bottom are: the average number of guests by hosts' racial groups; the Lorenz Curve in terms of the distribution of the number of guests; the Gini-index in terms of the distribution of the number of guests. All of the three plots will change over ticks/time. 

## HOW TO USE IT

Before running the model, we need to set up the number of hosts by race on the market. This should be done BEFORE running the model and cannot be adjusted when the model is running. By default, we set the number of White, Black, Asian hosts as 25, 25, 25.

Next, we can adjust the proportion of guests of a specific racial group that enter into the market. By default, we set proportion of White:Black:Asian guests as 0.5: 0.25: 0.25

Then, we can adjust the 1) guest-host homopihily and 2) guest-reviewer homophily. By default, we set guests' preference for same-race hosts and reviewers double as their preference for different-race hosts and reviewers. 

After the above settings, we can choose to "go once" for trial run or "go forever" to see the simulation over ticks.


## THINGS TO NOTICE

It is important to know that we should make sure the proportions in each category add up to 1. In most cases, Netlogo will automatically make this adjustment, but in some cases, it will not and needs to be refreshed by moving the sliders.

## THINGS TO TRY

We can tweak the model with different combinations of four groups of variables: host number/distribution, guest distribution, guest-host homopihily and guest-reviewer homophily. Again, host number/distribution cannot be adjusted during the model run but others can. 

To what extent the population distribution by race lead to uneuqal performance of hosts. Specifically, how will the hosts from the majority group benefit solely by this population distribution? 

We can make the two types of homophilies become racial biases by setting guests' preference for a specific group to a very small value (e.g., 0.8:0.19:0.01). We thus can explore the influence of racial discrimination on the performance (number of guests received) of hosts over time. 


## EXTENDING THE MODEL

We can extend this model by adding other variables that influence people's homophily preferences such as room types and prices. However, here we solely want to focus on the choices people would make based on racial homophily, assuming other things being equal.

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

house bungalow
false
0
Rectangle -7500403 true true 210 75 225 255
Rectangle -7500403 true true 90 135 210 255
Rectangle -16777216 true false 165 195 195 255
Line -16777216 false 210 135 210 255
Rectangle -16777216 true false 105 202 135 240
Polygon -7500403 true true 225 150 75 150 150 75
Line -16777216 false 75 150 225 150
Line -16777216 false 195 120 225 150
Polygon -16777216 false false 165 195 150 195 180 165 210 195
Rectangle -16777216 true false 135 105 165 135

house colonial
false
0
Rectangle -7500403 true true 270 75 285 255
Rectangle -7500403 true true 45 135 270 255
Rectangle -16777216 true false 124 195 187 256
Rectangle -16777216 true false 60 195 105 240
Rectangle -16777216 true false 60 150 105 180
Rectangle -16777216 true false 210 150 255 180
Line -16777216 false 270 135 270 255
Polygon -7500403 true true 30 135 285 135 240 90 75 90
Line -16777216 false 30 135 285 135
Line -16777216 false 255 105 285 135
Line -7500403 true 154 195 154 255
Rectangle -16777216 true false 210 195 255 240
Rectangle -16777216 true false 135 150 180 180

house efficiency
false
0
Rectangle -7500403 true true 180 90 195 195
Rectangle -7500403 true true 90 165 210 255
Rectangle -16777216 true false 165 195 195 255
Rectangle -16777216 true false 105 202 135 240
Polygon -7500403 true true 225 165 75 165 150 90
Line -16777216 false 75 165 225 165

house ranch
false
0
Rectangle -7500403 true true 270 120 285 255
Rectangle -7500403 true true 15 180 270 255
Polygon -7500403 true true 0 180 300 180 240 135 60 135 0 180
Rectangle -16777216 true false 120 195 180 255
Line -7500403 true 150 195 150 255
Rectangle -16777216 true false 45 195 105 240
Rectangle -16777216 true false 195 195 255 240
Line -7500403 true 75 195 75 240
Line -7500403 true 225 195 225 240
Line -16777216 false 270 180 270 255
Line -16777216 false 0 180 300 180

house two story
false
0
Polygon -7500403 true true 2 180 227 180 152 150 32 150
Rectangle -7500403 true true 270 75 285 255
Rectangle -7500403 true true 75 135 270 255
Rectangle -16777216 true false 124 195 187 256
Rectangle -16777216 true false 210 195 255 240
Rectangle -16777216 true false 90 150 135 180
Rectangle -16777216 true false 210 150 255 180
Line -16777216 false 270 135 270 255
Rectangle -7500403 true true 15 180 75 255
Polygon -7500403 true true 60 135 285 135 240 90 105 90
Line -16777216 false 75 135 75 180
Rectangle -16777216 true false 30 195 93 240
Line -16777216 false 60 135 285 135
Line -16777216 false 255 105 285 135
Line -16777216 false 0 180 75 180
Line -7500403 true 60 195 60 240
Line -7500403 true 154 195 154 255

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tile log
false
0
Rectangle -7500403 true true 0 0 300 300
Line -16777216 false 0 30 45 15
Line -16777216 false 45 15 120 30
Line -16777216 false 120 30 180 45
Line -16777216 false 180 45 225 45
Line -16777216 false 225 45 165 60
Line -16777216 false 165 60 120 75
Line -16777216 false 120 75 30 60
Line -16777216 false 30 60 0 60
Line -16777216 false 300 30 270 45
Line -16777216 false 270 45 255 60
Line -16777216 false 255 60 300 60
Polygon -16777216 false false 15 120 90 90 136 95 210 75 270 90 300 120 270 150 195 165 150 150 60 150 30 135
Polygon -16777216 false false 63 134 166 135 230 142 270 120 210 105 116 120 88 122
Polygon -16777216 false false 22 45 84 53 144 49 50 31
Line -16777216 false 0 180 15 180
Line -16777216 false 15 180 105 195
Line -16777216 false 105 195 180 195
Line -16777216 false 225 210 165 225
Line -16777216 false 165 225 60 225
Line -16777216 false 60 225 0 210
Line -16777216 false 300 180 264 191
Line -16777216 false 255 225 300 210
Line -16777216 false 16 196 116 211
Line -16777216 false 180 300 105 285
Line -16777216 false 135 255 240 240
Line -16777216 false 240 240 300 255
Line -16777216 false 135 255 105 285
Line -16777216 false 180 0 240 15
Line -16777216 false 240 15 300 0
Line -16777216 false 0 300 45 285
Line -16777216 false 45 285 45 270
Line -16777216 false 45 270 0 255
Polygon -16777216 false false 150 270 225 300 300 285 228 264
Line -16777216 false 223 209 255 225
Line -16777216 false 179 196 227 183
Line -16777216 false 228 183 266 192

tile stones
false
0
Polygon -7500403 true true 0 240 45 195 75 180 90 165 90 135 45 120 0 135
Polygon -7500403 true true 300 240 285 210 270 180 270 150 300 135 300 225
Polygon -7500403 true true 225 300 240 270 270 255 285 255 300 285 300 300
Polygon -7500403 true true 0 285 30 300 0 300
Polygon -7500403 true true 225 0 210 15 210 30 255 60 285 45 300 30 300 0
Polygon -7500403 true true 0 30 30 0 0 0
Polygon -7500403 true true 15 30 75 0 180 0 195 30 225 60 210 90 135 60 45 60
Polygon -7500403 true true 0 105 30 105 75 120 105 105 90 75 45 75 0 60
Polygon -7500403 true true 300 60 240 75 255 105 285 120 300 105
Polygon -7500403 true true 120 75 120 105 105 135 105 165 165 150 240 150 255 135 240 105 210 105 180 90 150 75
Polygon -7500403 true true 75 300 135 285 195 300
Polygon -7500403 true true 30 285 75 285 120 270 150 270 150 210 90 195 60 210 15 255
Polygon -7500403 true true 180 285 240 255 255 225 255 195 240 165 195 165 150 165 135 195 165 210 165 255

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

tree pine
false
0
Rectangle -6459832 true false 120 225 180 300
Polygon -7500403 true true 150 240 240 270 150 135 60 270
Polygon -7500403 true true 150 75 75 210 150 195 225 210
Polygon -7500403 true true 150 7 90 157 150 142 210 157 150 7

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment1.1" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment2.1" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment3.1" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment4.1" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment1.2" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment2.2" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.333"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment3.2" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment4.2" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range 0 white-hosts) )</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range white-hosts (white-hosts + black-hosts)))</metric>
    <metric>mean ( map [[a] -&gt; (count [link-neighbors] of host a)] (range (white-hosts + black-hosts) host-count))</metric>
    <enumeratedValueSet variable="white-hosts">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="black-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="asian-hosts">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-g-prob">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-g-prob">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-1">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-1">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-w-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="w-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-w-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="b-b-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-a-2">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="a-b-2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gr_homo?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="front-page">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
