;; local population is always static
;; volunteers arrive and depart as a function of migrant arrivals and departures

;; agents wander randomly through the world
;; random numbers (user-defined) of the population of migrants and volunteers
;;     commit norms violations that the locals don't like. (not liking isn't in code yet)
;; locals witness these violations
;; violations and witnessing are reset at each tick (instances are fleeting)

;; I added an experience variable to count the number of events witnessed, but it's not used right now
;; added networks

globals
[
  ;; Local-Population ;; population is static and defined by user
  ;; Migrant-Arrival-Rate ;; is a slider that can be adjusted for number of migrants arriving per day (tick)
  ;; Volunteer-Arrival-Rate ;; is a slider adjusted by user defining a percentage of volunteers based on refugee arrivals
  ;; Vision ;; controls how far away the locals can see a transgression (measured in patches)
  ;; Migrant-Weight ;; how much weight a local places on witnessing a migrant transgression to change opinion of them
  ;; Volunteer-Weight ;; how much weight a local places on witnessing a volunteer transgression to change opinion of them
  ;; Influence ;; controls how far away locals look to neighbors for a rumor
  ;; Networks ;; networks on or off
  Volunteer-incidents ;; infractions by volunteers
  Migrant-incidents ;; infractsions by migrants
]

breed [ migrants migrant ]
breed [ volunteers volunteer ]
breed [ locals local ]

locals-own [
  migrant-feelings ;; heterogeneous, normal, (0,1) feelings about refugees
  volunteer-feelings ;; heterogeneous, normal, (0,1) feelings about volunteers
  migrant-witness? ;; have witnessed a norm violation by refugee in this tick
  volunteer-witness? ;; have witnessed a norm violation by volunteer in this tick
  experience ;; a count of how many events the local has witnessed
  migrant-rumor-holder? ;; the equivalent of "infected" in SIR model
  volunteer-rumor-holder? ;; the equivalent of "infected" in SIR model -- like there are two related diseases
  friends ;; number of friends in network
]

turtles-own [
 transgressor?
]

to setup
  ca
  reset-ticks
  create-locals Local-Population
  [
    set color blue
    set shape "circle 2"
    ;; feelings along a normal distribution with mean 0, s.d. 1
    set migrant-feelings random-normal 0 1
    set volunteer-feelings random-normal 0 1
    set migrant-rumor-holder? 0
    set volunteer-rumor-holder? 0
    set friends 0
    setxy random-xcor random-ycor ;; distribute around the environment
  ]

  if Networks = True [
    ask locals [
      set friends random-normal ( Local-Population / 5) ( Local-Population * .05 ) ;; version of normal dist
      create-links-to up-to-n-of friends other locals
      set friends count my-links
    ]
    ask links [ hide-link ]
  ]
end


to go
  if ticks = 300 [ stop ]
  reset-violations
  reset-witness
  get-over-it

  outsiders-arrive
  violate-norm
  update-global-counters
  wander
  witness
  spread-rumor
  outsiders-leave

  tick
end

to update-global-counters
  set Volunteer-incidents Volunteer-incidents + count volunteers with [transgressor? = 1 ]
  set Migrant-incidents Migrant-incidents + count migrants with [ transgressor? = 1 ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; population arrival and departure code ;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to outsiders-arrive
  create-migrants Migrant-Arrival-Rate [ set color red ]
  ask migrants [ setxy random-xcor random-ycor ]

  ;; this makes volunteers arrive at a proportion of refugee arrivals
  let volunteer-draw int (Migrant-Arrival-Rate * (Volunteer-Arrival-Rate / 100) )
  create-volunteers volunteer-draw [ set color green set shape "square 2"]

  ask volunteers [ setxy random-xcor random-ycor ]
end

to outsiders-leave
  ask up-to-n-of Migrant-Departure-Rate migrants [ die ]

  let volunteer-sad int (Migrant-Departure-Rate * (Volunteer-Departure-Rate / 100))
  ask up-to-n-of volunteer-sad volunteers [ die ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; norm violation code ;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to violate-norm
  let integer random Transgression-Rate

  ask up-to-n-of integer migrants [ set shape "x" set size 2 set color yellow set transgressor? 1 ]
  ask up-to-n-of (Transgression-Rate - integer) volunteers [ set shape "x" set size 2 set color yellow set transgressor? 1 ]
end

to reset-violations
  ask migrants with [transgressor? = 1] [ set shape "default" set size 1 set color red set transgressor? 0 ]
  ask volunteers with [transgressor? = 1] [ set shape "square 2" set size 1 set color green set transgressor? 0 ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; population movement code ;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to wander
  ask turtles [ left random 180 fd 1 ]
end

to witness
  ask locals [
    if count migrants with [ transgressor? = 1 ] in-radius Vision >= 1
    [
      set migrant-witness? 1
      set experience experience + 1
      set migrant-rumor-holder? 1
      update-migrant-feelings
      if Networks = True [ ask out-link-neighbors [ catch-migrant-rumor ] ] ;; it's not a direct spread, they "catch" it through SIR
    ]
    if count volunteers with [ transgressor? = 1 ] in-radius Vision >= 1
    [
      set volunteer-witness? 1
      set experience experience + 1
      set volunteer-rumor-holder? 1
      update-volunteer-feelings
      if Networks = True [ ask out-link-neighbors [ catch-volunteer-rumor ] ] ;; it's not a direct spread, they "catch" it through SIR
    ]

  ]
end

to reset-witness
  ask locals with [migrant-witness? = 1 ] [ set migrant-witness? migrant-witness? = 0 ]
  ask locals with [volunteer-witness? = 1 ] [ set volunteer-witness? volunteer-witness? = 0 ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; updating locals' feelings code ;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;note here that feelings only update in the negative for norm violations
;; adjusts differently for migrants and volunteers to try to capture locals' empathy for certain groups
to update-migrant-feelings
  set migrant-feelings migrant-feelings - 0.1 * Migrant-Weight
end

to update-volunteer-feelings
  set volunteer-feelings volunteer-feelings - 0.1 * Volunteer-Weight
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; SIR/ Rumor Spread code ;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to spread-rumor
  let susceptibles locals with [ migrant-rumor-holder? = 0 OR volunteer-rumor-holder? = 0 ]
  ask susceptibles [
    if count locals with [ migrant-rumor-holder? = 1 ] in-radius Influence >= 1 [ catch-migrant-rumor ]
    if count locals with [ volunteer-rumor-holder? = 1 ] in-radius Influence >= 1 [ catch-volunteer-rumor ]
    ]
end

to catch-migrant-rumor
  let coin random 100 ;; select a random number, uniform distribution
  if coin <= Probability-Spread [
    set shape "circle"
    set migrant-rumor-holder? 1
    update-migrant-feelings
  ]
end

to catch-volunteer-rumor
  let coin random 100 ;; select a random number, uniform distribution
  if coin <= Probability-Spread [
    set shape "circle"
    set volunteer-rumor-holder? 1
    update-volunteer-feelings
  ]
end

to get-over-it
  let infected locals with [ migrant-rumor-holder? = 1 OR volunteer-rumor-holder? = 1 ]
  ask infected [
    let coin random 100 ;; select a random number, uniform distribution
    if coin <= Probability-Recover [
      ifelse migrant-rumor-holder? = 1
      [ set migrant-rumor-holder? 0 ]
      [ set volunteer-rumor-holder? 0 ] ;;NOTE THAT THERE IS A PRIORITIZATION HERE AND THAT SOME AGENTS WILL HAVE BOTH TRUE -- NEED TO UPDATE
      set shape "circle 2"
      ;;user-message "got better!"
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
351
10
1369
529
-1
-1
10.0
1
10
1
1
1
0
1
1
1
-50
50
-25
25
0
0
1
ticks
30.0

BUTTON
13
10
84
43
Set Up
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
95
50
343
83
Local-Population
Local-Population
0
2000
352.0
1
1
people
HORIZONTAL

SLIDER
94
95
342
128
Migrant-Arrival-Rate
Migrant-Arrival-Rate
0
1000
318.0
1
1
people/day
HORIZONTAL

SLIDER
93
178
342
211
Volunteer-Arrival-Rate
Volunteer-Arrival-Rate
0
100
15.0
1
1
% of Migrants
HORIZONTAL

SLIDER
93
131
342
164
Migrant-Departure-Rate
Migrant-Departure-Rate
0
1000
295.0
1
1
people/day
HORIZONTAL

SLIDER
93
215
342
248
Volunteer-Departure-Rate
Volunteer-Departure-Rate
0
100
15.0
1
1
% of Mig. Dep.
HORIZONTAL

BUTTON
12
61
83
94
Go Once!
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
12
111
84
144
Go!! 
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
12
361
346
519
Migrant and Volunteer Numbers
Time (day?)
People
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Migrants" 1.0 0 -16777216 true "" "plot count migrants"
"Volunteers" 1.0 0 -7500403 true "" "plot count volunteers"

SLIDER
93
262
341
295
Transgression-Rate
Transgression-Rate
0
100
7.0
1
1
people
HORIZONTAL

TEXTBOX
12
645
162
663
SIR Model Inputs
14
0.0
1

SLIDER
13
672
200
705
Probability-Spread
Probability-Spread
0
100
50.0
1
1
%
HORIZONTAL

SLIDER
13
708
200
741
Probability-Recover
Probability-Recover
0
100
40.0
1
1
%
HORIZONTAL

TEXTBOX
15
527
198
561
Locals' Feelings Inputs
14
0.0
1

SLIDER
14
551
186
584
Migrant-Weight
Migrant-Weight
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
13
588
185
621
Volunteer-Weight
Volunteer-Weight
0
100
50.0
1
1
NIL
HORIZONTAL

PLOT
353
533
865
697
Feelings towards Outsiders
Time (day?)
Average Feelings
0.0
10.0
-4.0
4.0
true
true
"" ""
PENS
"Migrant Feelings" 1.0 0 -16777216 true "" "plot mean [ migrant-feelings ] of locals"
"Volunteer Feelings" 1.0 0 -7500403 true "" "plot mean [ volunteer-feelings ] of locals"

SLIDER
190
551
334
584
Vision
Vision
0
10
1.0
1
1
patch
HORIZONTAL

SLIDER
203
672
338
705
Influence
Influence
0
10
1.0
1
1
patch
HORIZONTAL

PLOT
872
534
1369
693
People Spreading Rumors
Time (day?)
Rumor Spreaders (people)
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Migrant Spread" 1.0 0 -16777216 true "" "plot count locals with [ migrant-rumor-holder? = 1]"
"Volunteer Spread" 1.0 0 -7500403 true "" "plot count locals with [ volunteer-rumor-holder? = 1]"

SWITCH
96
10
210
43
Networks
Networks
0
1
-1000

MONITOR
13
310
179
355
Volunteer Transgressions
Volunteer-incidents
0
1
11

MONITOR
186
310
347
355
Migrant Transgressions
Migrant-incidents
0
1
11

@#$#@#$#@
## WHAT IS IT?

This model is about the encounters of local citizens with migrants and volunteers coming to help migrants, based on field observations in Lesvos, Greece during the ongoing European Migration Situation. The model helps to understand how chance encounters by locals with outsiders (migrants or volunteers) can change the sentiment of the population towards outsiders for the negative. 

Note that this model is about negative encounters. An extension would look at positive encounters as well, since not all encounters with migrants and locals would be negative ones. Here, the model focuses on norm transgressions. 

## HOW IT WORKS



## HOW TO USE IT

There are three populations: locals, migrants, and volunteers. Locals' population is always the same (set by the user). Migrants arrive and depart at a user-defined number of people per day. Volunteers arrive and depart as a user-defined percentage of migrant arrivals and departures per day to capture the idea that volunteers come when there are many migrants and leave when there is less need. 

All locals have a starting "opinion" of migrants and of volunteers ollowing a normal distribution with mean 0 and standard deviation 1. In other words, most locals fall somewhere in the -1 to 1 range of the bell-curve, where negative values denote negative feelings and positive denote positive feelings. Very few people would be beyond -3 or +3.  

All agents wander randomly through the world to enable chance encounters with one another. 

The user can set a "Transgression-Rate" which is the number of people per tick (day?) that commits some sort of norm violation that the locals do not like. That rate represents a random selection from the populations of migrants and volunteers to commit a norm violation. These are minor infractions that reset at the next tick. 

## THINGS TO NOTICE



## THINGS TO TRY



## EXTENDING THE MODEL



## NETLOGO FEATURES



## RELATED MODELS


## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
