globals [
  bison-stride                ;; how much a bison moves in each simulation step
  bison-size                  ;; the size of the shape for a bison
  bison-reproduce-age         ;; age at which bison reproduces
  max-bison-age
  min-reproduce-energy-bison  ;; how much energy, at minimum, bison needs to reproduce
  max-bison-offspring        ;; max offspring a bison can have

  cattle-stride                ;; how much a bison moves in each simulation step
  cattle-size                  ;; the size of the shape for a bison
  cattle-reproduce-age         ;; age at which bison reproduces
  max-cattle-age
  min-reproduce-energy-cattle  ;; how much energy, at minimum, bison needs to reproduce
  max-cattle-offspring           ;; max offspring a bison can have


  max-grasses-energy          ;; the maximum amount of energy a plant in a patch can accumulate
  grass-level               ;; a measure of the amount of grass currently in the ecosystem
  grass-energy

  food-bison-eat
  food-cattle-eat

  rain-timer
  where-rain-band-is

  bison-color
  cattle-color
  grass-color
  dirt-color
  disease-age
  disease-color
  grazers
  rain-band-width
  coefficient-a
  sprout-delay-time
  region-boundaries
  speed-seasons
  rock-color
  bison-migrate?

  constant-simulation-length?
  cattle-migrate?    ;;we changed this PS
    ;;We changed this
]

breed [rocks rock]
breed [grasses grass]
breed [dividers divider]                 ;; dummy turtles that separate the view in regions
breed [disease-markers disease-marker]   ;; visual cue, red "X" that bison has a disease and will die
breed [bison a-bison]
breed [cattle a-cattle]

breed [embers ember]                     ;; visual cue that a grass patch is on fire
breed [disruptions disruption]           ;; dummy turtles that record the disruption events

breed [clouds cloud]
breed [rain-lines rain-line]
breed [rain-drops rain-drop]

bison-own [
  energy
  current-age
  max-age
  female?
  #-offspring
]

cattle-own [
  energy
  current-age
  max-age
  female?
  #-offspring
]

patches-own [
  fertile?
  plant-energy
  countdown
  region
  rain-level
]

grasses-own [

]

embers-own [
 current-age
]

disease-markers-own [
  current-age
]

rain-drops-own [current-age]

disruptions-own [
  disruption-region         ;; the region in which the disruption happened
  disruption-tick           ;; the tick at which the disruption happened
  disruption-description    ;; "infected bison", "added more bison" or "started a wild fire"
]

to startup
  setup-regions 3
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; setup procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup

  clear-all

  set constant-simulation-length? false

  set bison-size 1.4
  set bison-stride 0.25
  set bison-reproduce-age 100
  set min-reproduce-energy-bison 200
  set max-bison-offspring 2
  set max-bison-age 500
  set bison-color [180 90 90]
  set-default-shape bison "buffalo"

 ; set rock-color [216 212 196]
  set rock-color [216 212 196]
  set bison-migrate? false
  set cattle-migrate? false

  set cattle-size 1.4
  set cattle-stride 0.25
  set cattle-reproduce-age 100
  set min-reproduce-energy-cattle 160
  set max-cattle-offspring 2
  set max-cattle-age 400
  set cattle-color [110 110 140]
  set-default-shape cattle "wildebeest"

  set-default-shape rain-lines "line"

  set speed-seasons 4

  set grass-level 0
  set disease-age 20
  set grass-energy 5

  set rain-timer 0

  set max-grasses-energy 200

  set food-bison-eat 4.0
  set food-cattle-eat 4.0

  set rain-band-width 5
  set coefficient-a 0.003

  set grass-color green
  set dirt-color white
  set disease-color [0 0 0 120]

  set-default-shape disease-markers "new-circle"
  set-default-shape embers "fire"
  set-default-shape disruptions "empty"

  set sprout-delay-time 5

  setup-regions 3
  add-starting-grasses

  add-bison region-number-from-name add-bison-to-region
  add-cattle region-number-from-name add-cattle-to-region
  set grazers turtles with  [breed = bison or breed = cattle]

  create-rain-lines 1 [ set color red set shape "rain-line" set size 35 set hidden? true]

   update-bison-fence

  reset-ticks
  set-rain-line-position
  ask patches  [color-water-levels]
end


to add-starting-grasses
  ask patches [set fertile? false
    if region = 1 [set rain-level 1]
    if region = 2 [set rain-level 1]
    if region = 3 [set rain-level 1]
  ]
  add-grass-in-region 1 %-grassland-southern-region
  add-grass-in-region 2 %-grassland-central-region
  add-grass-in-region 3 %-grassland-northern-region
  ask grasses [visualize-grass]
  add-rocks
end

to add-grass-in-region [ which-region %-grassland ]
  let region-patches patches with [ region = which-region ]
  let n floor ((%-grassland / 100) * count region-patches)
  ask n-of n region-patches [
    set fertile? true
    sprout 1 [
      set breed grasses
      set color [10 100 10 75]
      set shape "square"
      set plant-energy max-grasses-energy / 2
    ]
  ]
  ask patches [color-water-levels]
end

to add-more-bison [which-region]
  ifelse any? disruptions with [ disruption-region = which-region ] [
    inform-user-about-max-disruptions which-region
  ]
  [
    if bison-to-add > 0 [
      add-bison which-region
      record-disruption which-region "added more bison"
    ]
  ]
end

to add-more-cattle [which-region]
  ifelse any? disruptions with [ disruption-region = which-region ] [
    inform-user-about-max-disruptions which-region
  ]
  [
    if cattle-to-add > 0 [
      add-cattle which-region
      record-disruption which-region "added more cattle"
    ]
  ]
end

to add-bison [which-region]
  let region-patches patches with [ region = which-region ]
  create-bison bison-to-add [                  ;; create the bison, then initialize their variables
    set color bison-color
    set size bison-size
    set energy (20 + (random (50))  )  ;; randomize starting energies
    set current-age 0  + random (max-bison-age) ;; start out bison at different ages
    set max-age max-bison-age
    set #-offspring 0
    move-to one-of region-patches
  ]
end

to add-cattle [which-region]
  let region-patches patches with [ region = which-region ]
  create-cattle cattle-to-add [                  ;; create the cattle, then initialize their variables
    set color cattle-color
    set size cattle-size
    set energy (20 + (random (50))  )    ;; randomize starting energies
    set current-age 0  + random  (max-cattle-age)  ;; start out bison at different ages
    set max-age max-cattle-age
    set #-offspring 0
    move-to one-of region-patches
  ]
end

to visualize-rocks
  ask rocks [set hidden? not show-rocky-ground?]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; runtime procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  if ticks >= 999 and constant-simulation-length? [ stop ]

  ask bison [
    bison-live
    reproduce-bison
  ]

  ask cattle [
    cattle-live
    reproduce-cattle
  ]

  set-rain-line-position
  if rainfall? [set rain-timer rain-timer + 1]

  ask patches [
    color-water-levels
    if visualize-rain-drops? [add-rain-drops]
    grow-grass
  ]
  visualize-rocks
  ask grasses [visualize-grass]
  age-move-rain-drops
  age-and-spread-embers
  update-disease
  update-bison-fence
  tick
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; disruption procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to record-disruption [ which-region description ]
  create-disruptions 1 [
    set disruption-region which-region
    set disruption-description description
    set disruption-tick ticks
  ]
  update-plots
end

to inform-user-about-max-disruptions [which-region]
  ask one-of disruptions with [ disruption-region = which-region ] [
    let region-side item (which-region - 1) ["left" "right"]
    user-message (word
      "You are allowed only one disruption in each ecosystem every time you run the model.\n\n"
      "You have already " disruption-description " in the " region-side " ecosystem.")
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; disease procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to update-disease
  ask disease-markers [
    set current-age (current-age + 1)
    if (ticks = 999 and constant-simulation-length?) [
      ask in-link-neighbors [ die ]
      die
    ]
  ]
  infect-other-grazers
end

to infect-other-grazers

 if not vaccine? [
  ask (turtle-set bison cattle) with [ has-disease? ] [
    ask turtle-set one-of other bison-here with [ not has-disease? ] [
      infect-this-grazer
    ]
    ask turtle-set one-of other cattle-here with [ not has-disease? ] [
      infect-this-grazer
    ]
  ]

  ]

end

to infect-a-%-of-bison
  ;; procedure for removing a percentage of bison (when button is clicked)
 ; ifelse any? disruptions with [ disruption-region = which-region ] [
 ;   inform-user-about-max-disruptions which-region
 ; ]
 ; [

    let number-bison count bison
    ask n-of floor (number-bison * %-bison-to-infect / 100) bison [infect-this-grazer]
 ;   record-disruption which-region "infected bison"
;  ]
end

to infect-this-grazer
  hatch-disease-markers 1 [
    set current-age 0
    set size 1.5
    set color disease-color
    create-link-from myself [ tie ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; fire procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to start-fire

    let current-grass-patches patches with [ fertile? and rain-level <= 1 and pycor = min-pycor and plant-energy >= (.25 * max-grasses-energy)]
    ask current-grass-patches [
      set countdown sprout-delay-time
      set plant-energy 0

      create-ember
    ]
   ; record-disruption which-region "started a wild fire"

end

to create-ember ;; patch procedure
  sprout 1 [
    set breed embers
    set current-age (sprout-delay-time / 2)
    set color [255 0 0 255]
    set size 1.5
  ]
end

to age-move-rain-drops


  ask rain-drops [
    set heading 210
    fd 0.05
    set current-age (current-age - 1)
    if current-age <= 0 [ die ]

    let time-to-age-peak abs (current-age - 10)
    let trans-val (150 - round (time-to-age-peak * 15))
   ; if trans-val > 255 or trans-val < 0 [set trans-val 255]
   ; show trans-val
  let color-string []
  set color-string lput 0 color-string
  set color-string lput 0 color-string
  set color-string lput 150 color-string
  set color-string lput trans-val color-string
    set color color-string
  ]
end

to age-and-spread-embers
  ask embers [
    set plant-energy 0
    set countdown sprout-delay-time
    set current-age (current-age - 1)
    if current-age <= 0 [ die ]
    let my-region region
    ask neighbors4 with [ region = [ region ] of myself and fertile? and grass-energy > (max-grasses-energy / 4) and not any? embers-here ] [
      create-ember
    ]
    let transparency round floor current-age * 255 / sprout-delay-time
    set color lput transparency [255 155 0]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; grazers procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to bison-live
  move-bison
  ifelse has-disease?  [set energy (energy - 1.5)][set energy (energy - 1)]
  set current-age (current-age + 1) ;; age by 1.5
  grazers-eat-grass food-bison-eat
  grazer-death
end

to cattle-live
  move-cattle
  ifelse has-disease?  [set energy (energy - 1.5)][set energy (energy - 1)]
  set current-age (current-age + 1)
  grazers-eat-grass food-cattle-eat
  grazer-death
end

to move-bison
  head-towards-rains
  rt random 44 - random 44
  let current-region region
  keep-in-region current-region bison-stride
  fd bison-stride
end

to move-cattle
  head-towards-rains
  rt random 44 - random 44
  let current-region region
  keep-in-region current-region cattle-stride
  fd cattle-stride
end

to grazers-eat-grass  [food-grazer-eats];; grazers procedurebisonbisonbisonbison
  ;; if there is enough grass to eat at this patch, the grazer eats it
  ;; and then gains energy from it.
    if plant-energy > food-grazer-eats [
      ;; plants lose ten times as much energy as the  gains (trophic level assumption)
      set plant-energy (plant-energy - (food-grazer-eats * 10))
      set energy energy + food-grazer-eats * 1.25 ;;  gain energy by eating
    ]
    ;; if plant-energy is negative, make it positive
    if plant-energy <= food-grazer-eats [ set countdown 5 ]
end

to grazer-death   ;; a grazer dies when energy dips below zero (starvation), or it gets too old
  if (current-age > max-age) or (energy < 0) [ask out-link-neighbors [ die ] die]
end


to reproduce-bison  ;; bison procedure
  let number-new-offspring (random (max-bison-offspring + 1)) ;; set number of potential offpsring from 1 to (max-bison-offspring)
  if (energy > ((number-new-offspring + 1) * min-reproduce-energy-bison) and current-age > bison-reproduce-age) [
    set energy (energy - (number-new-offspring * min-reproduce-energy-bison)) ;;lose energy when reproducing --- given to children
    set #-offspring #-offspring + number-new-offspring
    ;; hatch an offspring set it heading off in a a random direction and move it forward a step
    hatch number-new-offspring [
      set energy min-reproduce-energy-bison ;; split remaining half of energy amongst litter
      set current-age 0
      set #-offspring 0
      move-bison
    ]
  ]
end

to reproduce-cattle  ;; bison procedure
  let number-new-offspring (random (max-cattle-offspring + 1)) ;; set number of potential offpsring from 1 to (max-bison-offspring)
  if (energy > ((number-new-offspring + 1) * min-reproduce-energy-cattle) and current-age > cattle-reproduce-age) [
    set energy (energy - (number-new-offspring * min-reproduce-energy-cattle)) ;;lose energy when reproducing --- given to children
    set #-offspring #-offspring + number-new-offspring
    ;; hatch an offspring set it heading off in a a random direction and move it forward a step
    hatch number-new-offspring [
      set energy min-reproduce-energy-cattle ;; split remaining half of energy amongst litter
      set current-age 0
      set #-offspring 0
      move-cattle
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; grass procedures ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; adjust the size of the grass to reflect how much food is in that patch


to grow-grass ;; patch procedure
  set countdown (countdown - 1)
  ;; fertile patches gain 1 energy unit per turn, up to a maximum max-grasses-energy threshold
  if fertile? and countdown <= 0 [
    set plant-energy (plant-energy + grass-growth-rate)
    if plant-energy > max-grasses-energy [
      set plant-energy max-grasses-energy
    ]
  ]
  if not fertile? [
    set plant-energy 0
  ]
  if plant-energy < 0 [
    set plant-energy 0
    set countdown sprout-delay-time
  ]
end


to visualize-grass
      ifelse plant-energy >= 5
        [ set size (plant-energy / max-grasses-energy) ]
        [ set size ((5 / max-grasses-energy) )]
end

to add-rocks
  ask patches with [ not fertile?  ]  [
    sprout 1 [
      set breed rocks
      set size 1
      set shape "no-grass"
      set color [10 0 0 25]
      stamp
      die
    ]
  ]
end

to-report random-rock-color
   let r-min-color 150
   let g-min-color 130
   let b-min-color 130
   let brighten random 20
   let r-offset random 10
   let g-offset random 10
   let b-offset random 10
   let color-list []
   set color-list fput (r-min-color + brighten + r-offset) color-list
   set color-list fput (g-min-color + brighten + g-offset) color-list
   set color-list fput (b-min-color + brighten + b-offset) color-list
   report color-list
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; water / rain procedures ;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to set-rain-line-position
  ask rain-lines [set xcor lat-sin]
  update-water-levels
end


to update-water-levels
    let rain-line-pxcor [xcor] of one-of rain-lines
    let low-rain-value 0.5
    let a .25
    ask patches  [
      let distance-from-rain-line abs (pxcor - rain-line-pxcor)
      let rain-band-spread 9

      let distance-to-edge-of-rain ((rain-band-width / 2) + rain-band-spread)
      let xd (distance-from-rain-line - (rain-band-width / 2))
      let rain-at-xd (1 + (3 - (a * xd)))
      if (distance-from-rain-line <= (rain-band-width / 2)) [ set rain-level (4)]
      if (distance-from-rain-line >  (rain-band-width / 2)) and (distance-from-rain-line <=  (distance-to-edge-of-rain)) [ set rain-level rain-at-xd]
      if (distance-from-rain-line >  (distance-to-edge-of-rain)) and (distance-from-rain-line <=  (3 * distance-to-edge-of-rain))
          [ set rain-level (1.4 - (xd - 11) * (1.4 / 22 ))  ]
      if (distance-from-rain-line >  (3 * distance-to-edge-of-rain)) [ set rain-level 0 ]
    ]
end


to color-water-levels
  let red-channel (255 - (rain-level * (255 - 150) / 4))
  let green-channel (255 - (rain-level * (255 - 175) / 4))
  let blue-channel (255 - (rain-level * (255 - 235) / 4))
  let color-string []
  set color-string lput red-channel color-string
  set color-string lput green-channel color-string
  set color-string lput blue-channel color-string
  set pcolor color-string
end

to add-rain-drops
  let non-linear-age 20
  if random 100 < (round (rain-level) ^ 1.5)  [
    sprout 1 [set breed rain-drops set shape "drop" set color [0 0 150 70] set heading 0 bk 0.5 fd random-float 1 set heading 90 bk 0.5 fd random-float 1 set current-age 20 set size 0.4]
  ]

end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; REGION MANAGEMENT CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup-regions [ num-regions ]
  ; Store our region definitions globally for faster access:
  set region-boundaries calculate-region-boundaries num-regions
  ; Set the `region` variable for all patches included in regions:
  let region-numbers n-values num-regions [ n -> n + 1 ]
  (foreach region-boundaries region-numbers [ [boundaries region-number] ->
    ask patches with [ pxcor >= first boundaries and pxcor <= last boundaries ] [
      set region region-number
    ]
  ])
  add-dividers
end

to add-dividers
  set-default-shape dividers "block"
  ask patches with [ region = 0 ] [
    sprout-dividers 1 [
      set color gray + 2
      set size 1.2
    ]
  ]
end

to-report calculate-region-boundaries [ num-regions ]
  ; The region definitions are built from the region divisions:
  let divisions region-divisions num-regions
  ; Each region definition lists the min-pxcor and max-pxcor of the region.
  ; To get those, we use `map` on two "shifted" copies of the division list,
  ; which allow us to scan through all pairs of dividers
  ; and built our list of definitions from those pairs:
  report (map [ [d1 d2] -> list (d1 + 1) (d2 - 1) ] (but-last divisions) (but-first divisions))
end

to-report region-divisions [ num-regions ]
  ; This procedure reports a list of pxcor that should be outside every region.
  ; Patches with these pxcor will act as "dividers" between regions.
  report n-values (num-regions + 1) [ n ->
    [ pxcor ] of patch (min-pxcor + (n * ((max-pxcor - min-pxcor) / num-regions))) 0
  ]
end

to keep-in-region [ which-region grazer-stride] ; turtle procedure
  ; This is the procedure that make sure that turtles don't leave the region they're
  ; supposed to be in. It is your responsibility to call this whenever a turtle moves.

  if [region] of patch-ahead grazer-stride = 0 and which-region != 0 and bison-fences? and
   ((breed = bison and not bison-migrate?) or (breed = cattle and not cattle-migrate?)) [
   let region-min-pxcor first item (which-region - 1) region-boundaries
   let region-max-pxcor last item  (which-region - 1) region-boundaries

    if [pxcor] of patch-ahead grazer-stride = (region-min-pxcor - 1) or [pxcor] of patch-ahead grazer-stride = (region-max-pxcor + 1)
    ; if so, reflect heading around x axis
    [set heading (- heading) ]
    ]
  if pxcor = min-pxcor [set heading 90]
  if pxcor = max-pxcor [set heading -90]
end

to head-towards-rains
  if ((abs (pxcor - where-rain-band-is) )  > 10.5  and ((cattle-migrate? and breed = cattle) or (bison-migrate? and breed = bison)) ) [
    if (pxcor < where-rain-band-is) [set heading 90]
    if (pxcor > where-rain-band-is) [set heading -90]
]
end

to update-bison-fence
  ask dividers with [pxcor != min-pxcor and pxcor != max-pxcor] [set hidden? not bison-fences?]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GRAPHING;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to draw-vert-line [ xval ]
  plotxy xval plot-y-min
  plot-pen-down
  plotxy xval plot-y-max
  plot-pen-up
end

to-report max-plot-y-max
  ;; Reports the highest plot-y-max of the two plots.
  ;; Used for maintaining the same y-scale in both plots.
  let name plot-name ;; save the name of the current plot
  set-current-plot "Population Size vs. Time"
  let ys (list plot-y-max)
  set-current-plot name ;; restore current plot to what it was
  report max ys
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;REPORTERS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to-report has-disease?
  report any? out-link-neighbors
end


to-report lat-sin

  let rain-band-spread 20
 ; ifelse rainfall?
    set where-rain-band-is  (sin  ( 90  -  (rain-timer * speed-seasons / 10 ) ) * ( max-pxcor - (rain-band-spread / 2)))
   ; [set where-rain-band-is max-pxcor - (rain-band-width / 2)]
  report where-rain-band-is
end

to-report region-number-from-name [this-region-name]
 let #-region 0

  if this-region-name = "southern region" [set #-region 1]
  if this-region-name = "central region" [set #-region 2]
  if this-region-name = "northern region" [set #-region 3]
  report #-region
end

to-report grass-growth-rate
  ifelse rain-level > 0 [report   sqrt (rain-level * rain-level )] [report 0]
end

; Copyright 2015 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
375
110
1120
372
-1
-1
11.0
1
10
1
1
1
0
0
1
1
-33
33
-11
11
1
1
1
ticks
30.0

BUTTON
21
10
123
43
setup/reset
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
125
10
216
43
go/pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
15
165
370
408
Population Size vs. Time
time
population size
0.0
1000.0
0.0
100.0
true
true
"" "set-plot-y-range 0 max-plot-y-max"
PENS
"bison" 1.0 0 -2674135 true "" "plot count bison"
"cattle" 1.0 0 -14070903 true "" "plot count cattle"
"disrupt" 1.0 0 -7500403 true "" "foreach [ disruption-tick ] of disruptions with [ disruption-region = 1 ] draw-vert-line"

SLIDER
565
65
737
98
cattle-to-add
cattle-to-add
0
100
1.0
1
1
NIL
HORIZONTAL

BUTTON
890
20
995
53
infect bison
infect-a-%-of-bison
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
890
60
1110
93
%-bison-to-infect
%-bison-to-infect
1
100
25.0
1
1
NIL
HORIZONTAL

SLIDER
565
15
725
48
bison-to-add
bison-to-add
0
100
10.0
1
1
NIL
HORIZONTAL

BUTTON
730
10
865
51
add more bison
add-more-bison region-number-from-name add-bison-to-region
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
15
118
75
163
bison
count bison
0
1
11

MONITOR
75
118
152
163
cattle
count cattle
17
1
11

MONITOR
155
118
215
163
time
ticks
17
1
11

SWITCH
220
95
370
128
bison-fences?
bison-fences?
0
1
-1000

SLIDER
380
375
617
408
%-grassland-southern-region
%-grassland-southern-region
0
100
100.0
1
1
NIL
HORIZONTAL

SLIDER
640
375
862
408
%-grassland-central-region
%-grassland-central-region
0
100
100.0
1
1
NIL
HORIZONTAL

SLIDER
880
375
1110
408
%-grassland-northern-region
%-grassland-northern-region
0
100
100.0
1
1
NIL
HORIZONTAL

SWITCH
1000
20
1110
53
vaccine?
vaccine?
0
1
-1000

SWITCH
220
130
370
163
rainfall?
rainfall?
0
1
-1000

CHOOSER
380
10
555
55
add-bison-to-region
add-bison-to-region
"southern region" "central region" "northern region"
0

CHOOSER
380
60
555
105
add-cattle-to-region
add-cattle-to-region
"southern region" "central region" "northern region"
0

BUTTON
730
60
865
105
add more cattle
add-more-cattle region-number-from-name add-cattle-to-region
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
275
40
355
58
S --+-->N
12
0.0
1

TEXTBOX
300
10
320
90
W\n |\n\n |\n E
12
0.0
1

SWITCH
15
48
195
81
show-rocky-ground?
show-rocky-ground?
0
1
-1000

SWITCH
180
415
367
448
visualize-rain-drops?
visualize-rain-drops?
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

This model explores the stability of the consumer producer ecosystems in the Serengetti related to two primary consumers (Buffalo and Wildebeest) that have an overlapping range they graze in, for part of the year. 

## HOW IT WORKS



## HOW TO USE IT

1. Adjust the slider parameters (see below), or use the default settings.

2. Press the SETUP button.

4. Press the GO button to begin the model run.



## THINGS TO NOTICE


## THINGS TO TRY



## EXTENDING THE MODEL


## NETLOGO FEATURES


## RELATED MODELS

Wolf Sheep Predation and Rabbits Weeds Grass are other examples of interacting predator/prey populations with different rules.



## CREDITS AND REFERENCES

Based on ihub model in How Do Small Changes Make Big Impacts on Ecosystems? 

This model is part of HS Ecosystems storyline, sponsored by funding from the Moore Foundation.

modified by Luisa Garcia, Pam Lucas and Peggy Schultz

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Novak, M. and Penuel, W. (2017).  Serengetti Consumers model.  

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2015 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

<!-- 2015 Cite: Novak, M. -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

block
false
1
Rectangle -7500403 true false 135 0 150 315
Polygon -7500403 true false 90 180 90 210 210 270 210 240
Polygon -7500403 true false 210 30 210 60 90 120 90 90

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

buffalo
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 95 15 161 9 185 11 213 20 185 28 115
Polygon -1 true false 244 57 252 66 249 75 231 91 208 95 184 85 178 67 181 53 193 45 217 40 196 56 196 72 209 79 226 70 231 57

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

bug-infected
true
10
Circle -7500403 true false 96 182 108
Circle -7500403 true false 110 127 80
Circle -7500403 true false 110 75 80
Line -7500403 false 150 100 80 30
Line -7500403 false 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
14
Circle -16777216 true true 0 0 300
Circle -1 false false 29 29 242

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cloud
true
0
Circle -7500403 true true 13 118 94
Circle -7500403 true true 86 101 127
Circle -7500403 true true 51 51 108
Circle -7500403 true true 118 43 95
Circle -7500403 true true 158 68 134

compass
false
0
Circle -1 false false 88 88 122
Line -1 false 300 165 240 165
Line -1 false 240 135 300 135
Line -1 false 240 165 300 135
Line -1 false 30 165 30 135
Line -1 false 15 135 0 150
Line -1 false 60 150 45 165
Line -1 false 60 150 45 135
Line -1 false 15 165 0 150
Line -1 false 180 0 120 15
Line -1 false 164 22 120 29
Line -1 false 166 22 120 15
Line -1 false 180 45 120 30
Line -1 false 120 270 180 270
Line -1 false 120 270 120 300
Line -1 false 180 270 180 300
Line -1 false 150 270 150 300
Line -1 false 30 165 15 165
Line -1 false 45 135 30 135
Polygon -16777216 true false 150 150 150 165 90 150 150 135
Polygon -2674135 true false 150 150 150 165 210 150 150 135

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

drop
true
0
Rectangle -7500403 true true 135 90 165 210

empty
true
0

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fire
false
0
Polygon -7500403 true true 151 286 134 282 103 282 59 248 40 210 32 157 37 108 68 146 71 109 83 72 111 27 127 55 148 11 167 41 180 112 195 57 217 91 226 126 227 203 256 156 256 201 238 263 213 278 183 281
Polygon -955883 true false 126 284 91 251 85 212 91 168 103 132 118 153 125 181 135 141 151 96 185 161 195 203 193 253 164 286
Polygon -1184463 true false 155 284 172 268 172 243 162 224 148 201 130 233 131 260 135 282

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

new-circle
false
0
Circle -7500403 false true 0 0 300

no-grass
false
14
Rectangle -16777216 true true 0 0 300 300
Rectangle -7500403 false false 0 0 300 300
Line -7500403 false 0 0 300 300
Line -7500403 false 300 0 0 300

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

rain-line
false
0
Line -7500403 true 150 0 150 300

rock-0
false
0
Circle -7500403 true true 105 0 60
Circle -7500403 true true 174 114 42
Circle -7500403 true true 120 240 60
Circle -7500403 true true 45 90 60
Circle -7500403 true true 204 39 42
Circle -7500403 true true 24 174 42
Circle -7500403 true true 69 234 42
Circle -7500403 true true 234 129 42
Circle -7500403 true true 249 9 42
Circle -7500403 true true 249 234 42
Circle -7500403 true true -6 69 42
Circle -7500403 true true 204 219 42

rock-1
false
0
Polygon -7500403 true true 0 300 60 300 45 270 30 255 15 255
Polygon -7500403 true true 0 30 0 75 45 90 60 60 45 15 30 0
Polygon -7500403 true true 300 255 285 285 285 300 255 300 240 270 270 255
Polygon -7500403 true true 300 60 240 90 210 60 240 15 255 0

rock-2
true
0
Polygon -7500403 true true 0 105 -15 150 0 180 30 195 75 195 105 165 105 135 75 105 45 90 15 90
Polygon -7500403 true true 120 0 165 0 195 30 195 45 180 60 150 60 120 45 105 15
Polygon -7500403 true true 180 75 135 75 105 105 105 120 120 135 150 135 180 120 195 90
Polygon -7500403 true true 270 75 225 90 210 180 240 225 270 225 300 180 300 105
Polygon -7500403 true true 60 240 75 285 165 300 210 270 210 240 165 210 90 210

rock-3
true
0
Polygon -7500403 true true 180 165 135 150 105 165 90 195 75 240 105 285 165 285 195 255 195 210 195 180
Polygon -7500403 true true 30 60 75 60 105 90 105 105 90 120 60 120 30 105 15 75
Polygon -7500403 true true 120 150 120 105 90 75 75 75 60 90 60 120 75 150 105 165
Polygon -7500403 true true 240 180 195 165 165 105 180 45 240 30 270 75 270 150

rock-4
true
0
Polygon -7500403 true true 15 120 0 165 15 195 45 210 75 195 75 180 90 150 90 120 60 105 30 105
Polygon -7500403 true true 120 0 165 0 195 30 195 45 180 60 150 60 120 45 105 15
Polygon -7500403 true true 270 105 225 105 195 135 195 150 210 165 240 165 270 150 285 120
Polygon -7500403 true true 135 75 180 90 195 180 165 225 135 225 105 180 105 105
Polygon -7500403 true true 105 240 120 285 210 300 255 270 255 240 210 210 135 225

rock-5
true
0
Polygon -7500403 true true 30 60 15 105 30 135 60 150 105 150 135 120 135 90 105 60 75 45 45 45
Polygon -7500403 true true 120 165 75 165 45 195 45 210 60 225 90 225 120 210 135 180
Polygon -7500403 true true 225 30 180 45 165 135 195 180 225 180 255 135 255 60
Polygon -7500403 true true 120 210 135 255 225 270 270 240 270 210 225 180 150 180

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 0 0 300 300

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wildebeest
false
0
Polygon -7500403 true true 182 181 177 279 165 278 167 217 160 177 128 172 81 180 68 165 46 210 33 210 32 177 21 150 9 121 15 92 42 84 108 91 149 54 225 65 275 68 299 137 284 139 268 149 248 120 226 119 200 180
Polygon -7500403 true true 45 209 61 277 49 278 33 208
Polygon -7500403 true true 12 117 9 159 9 204 14 232 23 201 17 133
Polygon -6459832 true false 258 57 261 72 244 73 231 79 221 72 221 62 220 49 230 32 229 52 235 59 248 54
Line -16777216 false 110 86 81 116
Circle -16777216 true false 251 79 18

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
