;; A parameterized model of the Capital Hill Baby-Sitting Co-op
;; Copyright (C) 2015 Russ Abbott
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful, but 
;; WITHOUT ANY WARRANTY; without even the implied warranty of 
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License (http://www.gnu.org/licenses/) for more details.

turtles-own [ 
  scrip 
  
  scrip-needed
  want-sitter? 
  
  want-to-go-out? 
  enough-scrip? 
  going-out? 
  
  want-to-sit? 
  sitting? 
]

globals [ 
  aggregated-hours-out cycle-pos frustrated-want-sitter frustrated-want-to-sit going-out# 
  max-gini max-scrip my-ticks no-sitter# no-sitting-jobs# not-enough-scrip# quiet-evening# 
  sitting# sitting-as-1st-choice# sitting-as-2nd-choice# stop-run? total-hours-out 
  total-scrip want-sitter# want-to-go-out# want-to-sit# 
]

extensions [ array table ]

;;;;;;;;;;;;;;;;;;;;;

to setup
  clear-all
  create-co-op
  set aggregated-hours-out table:make
  set total-hours-out 0
  set max-scrip 0
  ;; set monthly-dues 1
  set frustrated-want-sitter no-turtles
  set frustrated-want-to-sit no-turtles
  set stop-run? false
  reset-ticks  
end

;;;;;;;;;;;;;;;;;;;;;

to go
  reset-turtles
  ;; my-ticks is used as a settable proxy for ticks for testing
  set my-ticks ticks
  set cycle-pos (sin ( my-ticks mod 360 )) * cycle-amplitude ;; Uses a 360 day year. 
  plan-evening
  match-sitees-and-sitters
  if clear-sitter-market? [ clear-sitter-market ]
  check-consistency
  set total-scrip sum [scrip] of turtles
  let max-scrip-now max [scrip] of turtles
  set max-scrip ceiling maxi max-scrip max-scrip-now
  ;; a gross approximation of the GINI constant. It is always very low and not very useful
  let gini ifelse-value (total-scrip = 0) [0] [(max-scrip-now / total-scrip) - (1 / co-op-size)]
  set max-gini maxi max-gini gini
  tick
  ;; should the central bank adjust the monthy dues?
  if auto-adjust [ set-monthly-dues ]
  if ticks mod 12 = 0 [ ask turtles [ set scrip scrip - monthly-dues ] ]
  ;; stop-run is set to true if an internal inconsistency is noticed
  if ticks >= 365 * years or stop-run? [stop]
end

;;;;;;;;;;;;;;;;;;;;; 

;; Performs no model function. Checks the consistency of the values computed. 
;; Essentially a continual regression test.
to check-consistency
  set frustrated-want-sitter turtles with [ want-sitter? and not going-out? and not sitting? ]
  set frustrated-want-to-sit turtles with [ want-to-sit? and not sitting? ]
  if count frustrated-want-sitter > 0 and count frustrated-want-to-sit > 0 [
    show word "No sitters: " frustrated-want-sitter 
    show word "Available sitters: " frustrated-want-to-sit 
    set stop-run? true
  ]

  set want-sitter# count turtles with [want-sitter?]
  set want-to-sit# count turtles with [want-to-sit?]
  set quiet-evening# count turtles with [not want-sitter? and not want-to-sit?]
  if want-sitter# + want-to-sit# + quiet-evening# != co-op-size [
    show "want-sitter# + want-to-sit# + quiet-evening# != co-op-size failed"
    set stop-run? true
  ]

  set want-to-go-out# count turtles with [want-to-go-out?]
  set not-enough-scrip# count turtles with [want-to-go-out? and not enough-scrip?]
  if  want-sitter# + not-enough-scrip# != want-to-go-out# [
    show "want-sitter# + not-enough-scrip# != want-to-go-out# failed"
    set stop-run? true
  ]
  
  set going-out# count turtles with [ going-out? ]
  set no-sitter# count frustrated-want-sitter
  set sitting-as-2nd-choice# count turtles with [ want-sitter? and sitting? ]
  
  if going-out# + no-sitter# + sitting-as-2nd-choice# != want-sitter# [
    show "want-to-go-out# - not-enough-scrip# != want-sitter# failed"
    set stop-run? true
  ]
  
  set sitting-as-1st-choice# count turtles with [ want-to-sit? and sitting? ]
  set no-sitting-jobs# count frustrated-want-to-sit
  if sitting-as-1st-choice# + no-sitting-jobs# != want-to-sit# [
    show "sitting-as-first-choice# + no-sitting-jobs# != want-to-sit# failed"
    set stop-run? true
  ]
  
  set sitting# count turtles with [ sitting? ]
  if not multiple-sitees? and going-out# != sitting# [
    show "going-out# = sitting# failed"
    set stop-run? true
  ]
end

;; If more members want to go out than sit, the extras sit for each other if they are willing.
to clear-sitter-market 
  ;; Don't look at turtles that just want-to-go-out. They may not have enough scrip, 
  ;; in which case they were already asked if they want to sit.
  ;; uncleared-turtles are those who can't find sitters through the first step of the planning process.
  let uncleared-turtles turtles with [ want-sitter? and not going-out? ] 
  
  ;; sitters-set are willing to sit
  let sitters-set uncleared-turtles with [ random-probability < willing-to-sit? ]
  
  ;; sittees is a list unwilling to sit
  let sittees [ self ] of uncleared-turtles with [ not member? self sitters-set ]
  
  ;; sitters is a list willing to sit. It's just a copy of sitter-set.
  let sitters [ self ] of sitters-set    ;; [self] of <AgentSet> turns the AgentSet into a List

  if-else length sitters > length sittees [
    set sitters n-of length sittees sitters 
  ] [
    set sittees n-of length sitters sittees 
  ]
  (foreach sittees sitters [ pay-sitter ?1 ?2 "clear-sitters1" ] )
  
  ;; Any sitters left over can sit for each other.
  let extra-sitters [ self ] of sitters-set with [ not sitting? ]
  set sittees n-of ((length extra-sitters) / 2) extra-sitters
  set sitters n-of length sittees filter [not member? ? sittees] extra-sitters 
  (foreach sittees sitters [ pay-sitter ?1 ?2 "clear-sitters2" ] )  
end

to create-co-op
  create-turtles co-op-size [ 
    set scrip initial-scrip 
    set hidden? true
  ] 
  reset-turtles
end

;; Determine the time this member will go out this day.
to-report get-event-length  
  if random-probability > get-want-to-go-out-probability [ report 0 ]
  if hours-out-distribution = "uniform"      [ report (random (mean-hours-out + mean-hours-out / 2)) + 1 ]
  if hours-out-distribution = "poisson"      [ report (random-poisson (mean-hours-out - 1)) + 1 ] 
  if hours-out-distribution = "none"         [ report mean-hours-out ] 
end

;; Determine the probability that this member wants to go out
to-report get-want-to-go-out-probability
  report seasonally-adjusted-prob (mini percent-of-desired-times-out-reserve want-to-go-out-probability) cycle-pos
end

;; Determine the probability that this member wants to sit
to-report get-want-to-sit-probability
  report seasonally-adjusted-prob (maxi (1 - percent-of-desired-times-out-reserve) want-to-sit-probability) ((-1) * cycle-pos)
end

;; Used to keep track of the distribution of event lengths
to increment-table [ table instance ]
  let current-value ifelse-value table:has-key? table instance [table:get table instance] [0]
  table:put table instance current-value + 1
end

;; By this time the members have want-sitter or want-to-sit set of they want to go out or want to sit
to match-sitees-and-sitters
  let want-sitters [self] of turtles with [ want-sitter? ]
  let available-sitters [self] of turtles with [ want-to-sit? ]
  if length want-sitters < length available-sitters [ set available-sitters n-of length want-sitters available-sitters ]
  if length want-sitters > length available-sitters [ 
    if-else multiple-sitees? and length available-sitters > 0 [
      set available-sitters repeat-to available-sitters length want-sitters
    ]
    [ set want-sitters n-of length available-sitters want-sitters ]
  ]
  ( foreach want-sitters available-sitters [ pay-sitter ?1 ?2  "match-sitees-and-sitters" ] )
end

;; max with 2 parameters
to-report maxi [a b]
  report max list a b
end

;; min with 2 parameters
to-report mini [a b]
  report min list a b
end

;; transfer scrip from sittee to sitter. 
to pay-sitter [sittee sitter source] 
  if [going-out?] of sittee [ show (sentence source " " sittee " sittee going out twice")   set stop-run? true ]  
  if [sitting?]   of sittee [ show (sentence source " " sittee " sittee already sitting")   set stop-run? true]  
  if [going-out?] of sitter [ show (sentence source " " sitter " sitter already going out") set stop-run? true ]  
  if not multiple-sitees? and 
     [sitting?]   of sitter [ show (sentence source " " sitter " sitter sitting twice ")    set stop-run? true ] 
     
  let payment [scrip-needed] of sittee
  ask sittee [
    set scrip scrip - payment
    set going-out? true
  ]
  ask sitter [ 
    set scrip scrip + payment
    set sitting? true
    set total-hours-out total-hours-out + payment
  ]
end

;; Append the list xs to itself until the total length is = n
to-report repeat-to [xs n]
  if-else length xs < n [ report repeat-to sentence xs xs n ] [ report n-of n xs ]
end

;; Returns the percent of the desired times-out reserve this member has in scrip
;; The desired times-out reserve is the number of times the member can afford to 
;; go out without additional sitting
to-report percent-of-desired-times-out-reserve  
  report ifelse-value (desired-times-out-reserve = 0)  [ 1 ]
;    ifelse-value (scrip <= 0) [0] [1] 
;  ] 
  [ percent-of-limit scrip mean-hours-out * desired-times-out-reserve ]  ;;[ mini 1 maxi 0 ( scrip / ( mean-hours-out * desired-times-out-reserve ) ) ]
end

;; Amount is what percent of limit. Constrain result to be between 0 and 1.
to-report percent-of-limit [amount limit]
  report mini 1 maxi 0 ( amount / limit ) 
end

;; Determine what, if anything each member wants to do and set their flags accordingly
to plan-evening
  ask turtles [
    let hours-out-wanted get-event-length
    if hours-out-wanted > 0 [ increment-table aggregated-hours-out hours-out-wanted ]
    ;; In this model each unit of scrip buys a full hour of sitting. In the original
    ;; each unit bought half an hour.
    set scrip-needed hours-out-wanted   
    ;; scrip-needed = 0 indicates that the member doesn't want to go out. 
    if scrip-needed > 0 [ 
      set want-to-go-out? true
      ;; can go out only if member posesses enough scrip
      if scrip >= scrip-needed [ set want-sitter? true set enough-scrip? true ]
    ] 
    ;; Member who don't want to go out or don't have enough scrip are asked if they want to sit.
    set want-to-sit? not want-sitter? and random-probability < get-want-to-sit-probability  
  ]
end

;; Plots the hours-out histogram
to plot-table-as-percentages [tab]
  let pct-list table-as-pct-list tab
  if length pct-list > 0 [ 
    plot-pen-reset  ;; erase what we plotted before   
    let keys table:keys tab
    set-plot-x-range min keys - 1 (max keys + 1)
    set-plot-y-range 0 ceiling max pct-list + 1
    ;; Not clear why fput 0 is needed. If not done, first value is not plotted.
    ;; With it, the extra initial zero is not plotted.
    foreach fput 0 pct-list plot
  ]
end

;; All NetLogo arithmetic is done if floating point.
to-report random-probability
  report (random 100) / 100
end

;; Unset all flags
to reset-turtles
  ask turtles [ 
    set going-out? false
    set enough-scrip? false
    set sitting? false
    set want-sitter? false
    set want-to-go-out? false  
    set want-to-sit? false 
  ]
end

;; Adjust want-to-go-out and want-to-sit for the season.
;; If aren't using seasons, cycle-p will be 0 and no change will occur.
;; Otherwise increase toward 1 or decrease toward 0 the input probability
;; based on the cycle value.
to-report seasonally-adjusted-prob [p cycle-p]
  let diff ifelse-value (cycle-p > 0) [1 - p] [
    ifelse-value (cycle-p = 0) [0] [p]]
  report p + diff * cycle-p
end

;; Sets the monthly dues to control the money supply
to set-monthly-dues
  if no-sitting-jobs# > max-no-sitting [set monthly-dues -0.3]
  if no-sitting-jobs# < min-no-sitting [set monthly-dues 0.6]
end

;; Converts a table into an array of percentages to make a histogram.
;; Requires that the keys of the table be integers. 
;; They become the indices of an array in which the corresponding values are stored
;; Then a list is created and returned with each value replaced by its pct (0 to 100) of the sum of all values
to-report table-as-pct-list [tab]
  let keys table:keys tab   ;; The keys are the indices of the array/list (zero-based). (max keys) is the index of the last element.
  if length keys = 0 [ report [ ] ]
  let min-key min keys
  let arr array:from-list n-values (1 + max keys - min-key) [0]  
  foreach keys [array:set arr (? - min-key) table:get tab ?]
  let list-arr array:to-list arr
  let sum-of-elements sum list-arr  ;; Compute the sum only once
  report map [ round ( 100 * ? / sum-of-elements ) ] list-arr
end

@#$#@#$#@
GRAPHICS-WINDOW
381
128
626
313
16
16
4.67
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
17
12
80
45
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
84
12
147
45
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
18
47
198
80
want-to-go-out-probability
want-to-go-out-probability
0
1
0.5
0.1
1
NIL
HORIZONTAL

SLIDER
228
11
320
44
co-op-size
co-op-size
0
300
100
1
1
NIL
HORIZONTAL

SLIDER
287
117
379
150
initial-scrip
initial-scrip
0
100
25
1
1
NIL
HORIZONTAL

SLIDER
177
117
285
150
mean-hours-out
mean-hours-out
0
6
4
1
1
NIL
HORIZONTAL

PLOT
17
153
851
310
want to go out
NIL
NIL
0.0
3650.0
0.0
1.0
false
true
"" ";; set-plot-x-range 0 365 * years\n;; set-plot-y-range 0 want-to-go-out-probability + 0.15"
PENS
"want to go out" 1.0 0 -16777216 true "" "if ticks > 0 [\n  plot (count turtles with [ want-to-go-out? ] / co-op-size)\n]"
"not enough scrip" 1.0 0 -13791810 true "" "if ticks > 0 [\n  plot (count turtles with [ want-to-go-out? and not enough-scrip? ] / co-op-size)\n]"
"no sitter" 1.0 0 -2674135 true "" "if ticks > 0 [\n  plot (count frustrated-want-sitter / co-op-size)\n]"
"going out" 1.0 0 -13840069 true "" "if ticks > 0 [\n  plot (count turtles with [ want-to-go-out? and going-out? ] / co-op-size)\n]"

SWITCH
700
484
840
517
multiple-sitees?
multiple-sitees?
1
1
-1000

PLOT
601
10
957
150
scrip histogram
Scrip amount
Members
0.0
10.0
0.0
10.0
true
false
"set-plot-x-range min [ scrip ] of turtles - 1 max [ scrip ] of turtles + 1\nset-plot-y-range 0 count turtles\nset-histogram-num-bars 10" "let scrips [ scrip ] of turtles\n  let mode first modes scrips \n  let mode-instances filter [ ? = mode ] scrips\n  set-plot-x-range floor (min [ scrip ] of turtles - 1) ceiling (max [ scrip ] of turtles + 1)\n  set-plot-y-range 0 length mode-instances + 1\n  set-histogram-num-bars (max [ scrip ] of turtles + 1) - (min [ scrip ] of turtles - 1)"
PENS
"default" 1.0 1 -16777216 true "" "histogram [scrip] of turtles"

PLOT
959
10
1159
149
hours out distribution
NIL
NIL
0.0
15.0
0.0
25.0
false
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "plot-table-as-percentages aggregated-hours-out"

CHOOSER
381
82
498
127
hours-out-distribution
hours-out-distribution
"uniform" "poisson" "none"
1

SLIDER
200
47
353
80
want-to-sit-probability
want-to-sit-probability
0
1
0.5
0.1
1
NIL
HORIZONTAL

SLIDER
323
10
444
43
monthly-dues
monthly-dues
-1
1
0.6
0.1
1
NIL
HORIZONTAL

SLIDER
19
117
176
150
desired-times-out-reserve
desired-times-out-reserve
0
10
0
1
1
NIL
HORIZONTAL

BUTTON
150
12
225
45
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
17
309
851
460
want to sit
NIL
NIL
0.0
3650.0
0.0
1.0
false
true
"" ";; set-plot-x-range 0 365 * years\n;; set-plot-y-range 0 (1 - want-to-go-out-probability) * want-to-sit-probability  + 0.15\n\n; if ticks > 0 [\n;   set-plot-y-range 0 ceiling (10 * (0.1 + count turtles with [ want-to-sit? ] / co-op-size) ) / 10\n; ]"
PENS
"want to sit" 1.0 0 -16777216 true "" "if ticks > 0 [\n  plot (count turtles with [ want-to-sit? ] / co-op-size)\n]"
"no sitting jobs" 1.0 0 -2674135 true "" "if ticks > 0 [\n  plot (count frustrated-want-to-sit / co-op-size)\n]"
"sitting" 1.0 0 -13840069 true "" "if ticks > 0 [\n  plot (count turtles with [ want-to-sit? and sitting? ] / co-op-size)\n]"

SWITCH
236
82
379
115
clear-sitter-market?
clear-sitter-market?
0
1
-1000

SLIDER
355
47
473
80
willing-to-sit?
willing-to-sit?
0
1
0
0.1
1
NIL
HORIZONTAL

MONITOR
890
10
957
55
total scrip
total-scrip
0
1
11

MONITOR
852
153
909
198
turtles
co-op-size
17
1
11

MONITOR
853
197
948
242
want to go out
want-to-go-out#
17
1
11

MONITOR
852
336
926
381
want-to-sit
want-to-sit#
17
1
11

MONITOR
989
404
1065
449
quiet evening
quiet-evening#
17
1
11

MONITOR
949
197
1054
242
not enough scrip
not-enough-scrip#
17
1
11

MONITOR
853
243
927
288
want-sitter
want-sitter#
17
1
11

MONITOR
928
337
1049
382
sitting as 1st choice
sitting-as-1st-choice#
17
1
11

MONITOR
1052
337
1142
382
no sitting jobs
no-sitting-jobs#
17
1
11

MONITOR
994
243
1053
288
no sitter
no-sitter#
17
1
11

MONITOR
928
243
993
288
going out
going-out#
17
1
11

MONITOR
928
290
1053
335
sitting as 2nd choice
sitting-as-2nd-choice#
17
1
11

MONITOR
930
384
987
429
sitting
sitting#
17
1
11

SLIDER
446
10
538
43
years
years
0
10
10
1
1
NIL
HORIZONTAL

SLIDER
19
81
129
114
cycle-amplitude
cycle-amplitude
0
1
0
0.1
1
NIL
HORIZONTAL

PLOT
855
479
1149
629
seas-adj
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -13791810 true "" "plot (sum ([get-want-to-go-out-probability] of turtles) / count turtles)"
"pen-1" 1.0 0 -2064490 true "" "plot (sum ([get-want-to-sit-probability] of turtles) / count turtles)"

MONITOR
1040
150
1135
195
total hours out
total-hours-out / 1000
0
1
11

MONITOR
1056
209
1122
254
max-scrip
max-scrip
17
1
11

MONITOR
1056
256
1114
301
max gini
max-gini
3
1
11

SWITCH
130
81
235
114
auto-adjust
auto-adjust
0
1
-1000

SLIDER
474
47
583
80
max-no-sitting
max-no-sitting
0
110
10
01
1
NIL
HORIZONTAL

SLIDER
499
82
598
115
min-no-sitting
min-no-sitting
-10
100
5
01
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
