globals [num-adopted num-implementers]
turtles-own [power attitude norm control intent last-attitude last-norm last-control last-intent implementer avg-advice-attitude avg-influence-attitude avg-interveners-influence]
directed-link-breed [advice-ties advice-tie]
directed-link-breed [influencer-ties influencer-tie]
directed-link-breed [intervener-ties intervener-tie]


to setup
  ca
   
  ;; make sure the input number of turtles is integer
  if num-turtles != floor(num-turtles) [set num-turtles floor(num-turtles)]
  
  ;; create turtles and distribute them
  crt num-turtles [
    ;; color turtles blue - not adopted
    set color blue    
    setxy random-xcor random-ycor
    set shape "person"
    
    set size 0.75 ;; this makes it easier to see
    
    ;; set intent levels based on survey data
    let u1 random-float 1
    if u1 < 0.09 [set intent 1]
    if u1 >= 0.09 and u1 < 0.11 [set intent 2]
    if u1 >= 0.11 and u1 < 0.18 [set intent 3]
    if u1 >= 0.18 and u1 < 0.24 [set intent 4]
    if u1 >= 0.24 and u1 < 0.36 [set intent 5]
    if u1 >= 0.36 and u1 < 0.60 [set intent 6]
    if u1 >= 0.60 [set intent 7]
    
    ;; set attitude levels based on survey data
    let u2 random-float 1
    if u2 < 0.002 [set attitude 1]
    if u2 >= 0.002 and u2 < 0.008 [set attitude 2]
    if u2 >= 0.008 and u2 < 0.03 [set attitude 3]
    if u2 >= 0.03 and u2 < 0.1 [set attitude 4]
    if u2 >= 0.1 and u2 < 0.25 [set attitude 5]
    if u2 >= 0.25 and u2 < 0.54 [set attitude 6]
    if u2 >= 0.54 [set attitude 7]
    
    ;; set norm levels based on survey data
    let u3 random-float 1
    if u3 < 0.02 [set norm 1]
    if u3 >= 0.02 and u3 < 0.04 [set norm 2]
    if u3 >= 0.04 and u3 < 0.08 [set norm 3]
    if u3 >= 0.08 and u3 < 0.18 [set norm 4]
    if u3 >= 0.18 and u3 < 0.34 [set norm 5]
    if u3 >= 0.34 and u3 < 0.63 [set norm 6]
    if u3 >= 0.63 [set norm 7]
    
    ;; set control levels based on survey data
    let u4 random-float 1
    if u4 < 0.11 [set control 1]
    if u4 >= 0.11 and u4 < 0.19 [set control 2]
    if u4 >= 0.19 and u4 < 0.27 [set control 3]
    if u4 >= 0.27 and u4 < 0.40 [set control 4]
    if u4 >= 0.40 and u4 < 0.57 [set control 5]
    if u4 >= 0.57 and u4 < 0.80 [set control 6]
    if u4 >= 0.80 [set control 7]
    
    ;; set implementer status based on survey data
    let u5 random-float 1
    ifelse u5 < 0.47 [set implementer true set color green] [set implementer false]
    set num-implementers count turtles with [color = green]
    
    
    
  ]
  
  ask turtles [
    ;; create the two types of ties randomly, according to the density of the original network
    ;; use a poisson distribution to represent a power-law type distribution of degree
    ;; this is done in a separate loop so all ties are possible
    create-advice-ties-to n-of (random-poisson (10.52 * num-turtles / 9799)) other turtles [set color yellow set thickness -0.5]    
    create-influencer-ties-to n-of (random-poisson (2.12 * num-turtles / 9799)) other turtles [set color pink set thickness -0.5]  
  ]
  set num-adopted 0
  
  ;; create interveners who can influence the government agents. these new turtles will exert their influence
  ;; on the other turtles to try to create change
  crt num-interveners [
    set color violet
    set shape "person"
    setxy random-xcor random-ycor
    ;; influencers own a "power variable," which effectively measures their ability to influence someone
    set power 1 + random 6
    ;; they only have ties to potential implementers
    create-intervener-ties-to n-of num-intervener-ties turtles with [color = green]
  ]
  
 
  reset-ticks
end

to go
  if not any? turtles with [implementer = true and color = green] [stop]
  if ticks > 200 [stop]
  ask turtles [
    ;; create the mean values of their neighbors attitudes; we differentiate between
    ;; advice ties and influencer ties
    
    ;; by initializing this way, the values won't change when turtles update their
    ;; properties
    let advice-attitude []
    ask out-advice-tie-neighbors [set advice-attitude lput attitude advice-attitude]  
    if not empty? advice-attitude [set avg-advice-attitude mean(advice-attitude)] 
    
    
    let influence-attitude []
    ask out-influencer-tie-neighbors [set influence-attitude lput attitude influence-attitude]    
    if not empty? influence-attitude [set avg-influence-attitude mean(influence-attitude)]
    
    ;; save the average influence power of any turtle that is an interventionist
    let interveners-influence []
    ask out-intervener-tie-neighbors [set interveners-influence lput power interveners-influence]    
    if not empty? interveners-influence [set avg-interveners-influence mean(interveners-influence)]
    
    ;; save the current status of beliefs for use later
    set last-attitude attitude
    set last-norm norm
    set last-control control
    set last-intent intent 
  ]
  
  ask turtles [
    ;; if they have not already adopted, adjust their belief systems
    if (color = blue) or (color = green) [
      change-attitude   ;; update attitudes
      change-norm       ;; update normative beliefs
      change-control    ;; update perceived control
      change-intent     ;; update intent
      
      ;; if they have the ability to implement, make change if intent and ability
      ;; are present
      if implementer = true [make-change]  
    ] 
  ]
  
  ;; interventionists can change their influence ties if their target has adopted
  ask turtles with [color = violet] [
    
  ]
  
  set num-adopted count turtles with [color = red]
  tick
end

to change-attitude
  ;; attitude is positively influenced by norms and perception of control
  ;; this is programmed so that if norms or controls are more than attitude, attitude will increase, and vice-versa
  ;; I divided by 12 so that the max change is 1 unit, and multiplied by a random number so that the increase is further surpressed 
  ;; (change will range from 0 to 1 (plus/minus)
  ;; I also cap the values at 1 to 7
   set attitude (last-attitude + (random-float 1.0) * (norm/control-attitude * (last-norm - last-attitude) + (2 - norm/control-attitude) * (last-control - last-attitude)) / 12)
   if attitude > 7 [set attitude 7]
   if attitude < 1 [set attitude 1]
  
end

to change-norm
  ;; norm is positively influenced by attitudes and perception of control
  ;; this is programmed so that if attitudes or controls are more than norms, norms will increase, and vice-versa
  ;; perception of norms is also influenced by the attitudes of those in the advice/influence network. in other words, if people you
  ;; get advice from have a positive attitude, that should positive influence your attitude and vice-versa
  ;; I divided by 24 so that the max change is 1 unit, and multiplied by a random number so that the increase is further surpressed 
  ;; (change will range from 0 to 1 (plus/minus)
  ;; I also cap the values at 1 to 7
   set norm (last-norm + (random-float 1.0) * (control/attitude-norm * (last-attitude - last-norm) + (2 - control/attitude-norm) * (last-control - last-norm) + advice/influence-norm * (avg-advice-attitude - last-norm) + (2 - advice/influence-norm) * (avg-influence-attitude - last-norm)) / 24)
   if norm > 7 [set norm 7]
   if norm < 1 [set norm 1]
end

to change-control
  ;; perception of control is positively influenced by norms and attitudes
  ;; this is programmed so that if norms or attitudes are more than perceived control, control will increase, and vice-versa
  ;; I divided by 12 so that the max change is 1 unit, and multiplied by a random number so that the increase is further surpressed 
  ;; (change will range from 0 to 1 (plus/minus)
  ;; I also cap the values at 1 to 7
   set control (last-control + (random-float 1.0) * (influencer-control * avg-interveners-influence / 7) + (random-float 1.0) * (attitude/norm-control * (last-attitude - last-control) + (2 - attitude/norm-control) * (last-norm - last-control)) / 12)
   if control > 7 [set control 7]
   if control < 1 [set control 1]
end

to change-intent
  ;; intent is positively influenced by all of attitudes, norms, and perceived control
  ;; I divided by 18 so that the max change is 1 unit, and multiplied by a random number so that the increase is further surpressed 
  ;; (change will range from 0 to 1 (plus/minus)
  ;; I also cap the values at 1 to 7
   set intent (last-intent + (random-float 1.0) * ((2 - influencer-control) * avg-interveners-influence / 7) + (random-float 1.0) * ( attitude-intent * (last-attitude - last-intent) + norm-intent * (last-norm - last-intent) + (3 - attitude-intent - norm-intent) * (last-control - last-intent)) / 18)
   if intent > 7 [set intent 7]
   if intent < 1 [set intent 1]
end

to make-change
  ;; to make adoption "hard", I incorporated what effectively functions as a threshold
  ;; in other words, change can only occur if someone perceives themselves to have high control, and have high intent
  ;; then, there is still a random component; the number 0.5 is meant to create randomness
  if control >= control-threshold [
    if intent >= intent-threshold [
      if random-float 1 < 0.5 [set color red]  ;; indicate adoption via changing the color
    ]
  ] 

end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
10
12
73
45
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
6
52
161
112
num-turtles
1000
1
0
Number

BUTTON
80
13
143
46
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
6
123
179
156
control/attitude-norm
control/attitude-norm
0
2
0.01
.01
1
NIL
HORIZONTAL

SLIDER
5
162
178
195
norm/control-attitude
norm/control-attitude
0
2
0.01
.01
1
NIL
HORIZONTAL

SLIDER
4
201
181
234
advice/influence-norm
advice/influence-norm
0
2
0.01
.01
1
NIL
HORIZONTAL

SLIDER
4
241
177
274
attitude/norm-control
attitude/norm-control
0
2
0.01
.01
1
NIL
HORIZONTAL

SLIDER
4
280
176
313
attitude-intent
attitude-intent
0
1.5
0.01
.01
1
NIL
HORIZONTAL

SLIDER
5
318
177
351
norm-intent
norm-intent
0
1.5
0.01
.01
1
NIL
HORIZONTAL

SLIDER
5
360
177
393
num-interveners
num-interveners
0
100
5
1
1
NIL
HORIZONTAL

SLIDER
5
401
177
434
num-intervener-ties
num-intervener-ties
1
5
5
1
1
NIL
HORIZONTAL

SLIDER
5
442
177
475
influencer-control
influencer-control
0
2
1
.01
1
NIL
HORIZONTAL

SLIDER
4
482
176
515
control-threshold
control-threshold
1
7
5
.1
1
NIL
HORIZONTAL

SLIDER
2
523
174
556
intent-threshold
intent-threshold
1
7
5
.1
1
NIL
HORIZONTAL

PLOT
921
242
1121
392
Adoption Rate
Time
Number Adopted
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot count turtles with [color = red]"

MONITOR
989
407
1076
452
NIL
num-adopted
17
1
11

@#$#@#$#@
## WHAT IS IT?

This model is an extension to the base model "Adoption of Critical Health Practices in Rural India." The additional complexity lies in the introduction of an intervener, which is an agent who leverages their persuasive power to improve the feelings of control and relative intent in the model.

## HOW IT WORKS

The rules that agents follow are based on the Theory of Planned Behavior (see Ajzen, 1991), which maps the relationships between attitudes, subjective norms, perceived behavior control, intentions, and the final performance of a behavior. 

At every clock tick, each agent will adjust their attitudes, norms, perceived control, and intent. Because each of these factors are interrelated, previous values are stored and used for computation. The hypothesized relationships are all positive, however the exact functional forms are not known. As a result, the model effectively moves each factor towards the other factors. In other words, if attitude > norms, then at the next tick norms will either increase or stay the same. The amount of increase is random and determined by a set of model parameters. These values are set by the user via a set of sliders.

In addition to the internal forces, each agent also has a set of links to other turtles. These represent the advice and influence ties reported by agents in the survey. Change is operationalized as a reaction to the attitudes of those in the neighborhood. At each time step, turtles calculate the average attitude of each agent in the advice network and in the influence network. Then, their subjective norms are positively influenced by the attitudes of the neighborhood.

The new model rules also include the effect of an intervention. These agents posess a "power" variable, which captures their ability to affect change. They form links to a set number of implementer turtles, and at each clock tick they attempt to positively increase a turtles feelings of control and intent.

## HOW TO USE IT

To use the model, first initialize the number of agents of interest. Upon clicking setup, that number of turtles will be generated, along with their advice and influence ties. The number of ties is generated randomly; yellow ties are advice and pink ties are influence. 47% of turtles will be green - these the potential implementers, who are capable of actually making change. The others will be blue - these turtles can influence attitudes and norms, but are not capable of actually making change.

A new layer of turtles - colored purple - will also be created. These are the intervention turtles (number specified by slider). They will create a set of grey ties to implementer turtles (also set by slider).

Upon clicking go, the turtles will begin to change their internal factors, as well as potentially adopt. When the turtles adopt, they will change their color to red. The model stops either after 200 ticks or if all green turtles have become red.

There are a number of sliders in the interface; these control the relative influence of each factor on the other factors. These sliders range from 0 to 2, and usually control the effect of two factors on a resulting other. For example, attitude/norm-control fixes the effect of attitude on control, relative to norms. If attitude/norm-control>1, then attitude has a greater impact than norms on changing control values, and vice versa. There is also a slider for the tie influence.

Finally, there is a slider for the control and intent threshold. This ranges from 1 to 7 and sets the level of each factor that is required for a turtle to make change. In other words, if the control threshold is 5, then only turtles whose perceived control is at least 5 may make change at that tick.

## THINGS TO NOTICE

Of particular interest is the overall rate of adoption, compared to a model with no intervention. Is it higher or lower, and why? 

## THINGS TO TRY

Varying the sliders will result in much different adoption rates. In particular, changing the thresholds can have a big impact on final adoption. Try decreasing the threshold until 100% adoption is achieved.

Additionally, increase or decrease the number of interveners, as well as the number of ties they can make.


## RELATED MODELS

This is an extension to the base model.

## CREDITS AND REFERENCES

Ajzen, I. (1991). The theory of planned behavior. Organizational behavior and human decision processes, 50(2), 179-211. 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="param8">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param10">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param1">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param9">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param12">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param11">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param2">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param5">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param3">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param4">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param7">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="param6">
      <value value="0.01"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
