;; Global variables.
globals [ vision_area bugs_dead p_food]

;; Set two breeds of agents, one for the bugs and one for food.
breed [ food a-food ]
breed [ bugs bug ]

bugs-own
[ energy ;; The resource bugs will be depleting and refilling by moving and eating.
  speed ;; The "gene" of movement speed for each bug.
  vision_r ;; The "gene" that decides how far a bug can see.
  vision_a ;; The angle of vision that the bug has.
  age ;; The ticks at which they were created.
]

to setup
  clear-all
  reset-ticks
  setup-world
  setup-bugs
  grow-food
end

to setup-world
  ask patches [ set pcolor green + 2]
end

to setup-bugs
  ;; Initialize global variables:
  set vision_area 1.5 ;; vision_area: area of a circular sector with angle = 60 deg and radius = 3, wo considering PI.
  set bugs_dead 0 ;; Number of dead bugs.
  set p_food -1 ;; Reference for the probability with which the food will grow.

  ;; Creating the bugs and setting their initial states to numbers that, after trials, suits the model.
  create-bugs 100
  [ set color 0
    set speed .5
    set energy (25 + random 50)
    set vision_r 3
    set vision_a 60
    set age (-50 - random 100)]
  ask bugs
  [ set shape "bug"
    setxy random-xcor random-ycor]
end

to go
  ;; Two criteria for stop: One if no bugs are alive and one if any setting results in uncontrolled reproduction.
  if not any? bugs [ stop ]
  if count bugs >= 2000 [ stop ]
  ;; Otherwise:
  move-eat ;; Bugs move forward, change direction towards food/prey within their vision, and then eat it when under reach.
  reproduce ;; Bugs check if their energy is at or above the reproduction threshold, if yes they reproduce.
  check-age ;; Bugs do not live forever!
  grow-food ;; Food grows at random places at a rate set in the interface.

  tick
end

to move-eat
  ;; Reduce the activity of feeble bugs:
  ask bugs with [energy <= 5]
  [ set speed min list speed 0.1
    set color 5]

  ask bugs with [energy < 1]
  [ set speed 0
    set color 9.9]

  ;; Move all, in relation to their energy/speed:
  ask bugs with [energy > 0]
  [ ;; Make bugs consider as prey the nearest bug within their size-range:
    let prey_b min-one-of other ((bugs in-cone vision_r vision_a) with [ (size * size_ratio) < [size] of myself ]) [distance myself]
    ;; Let bugs also consider as prey the dead bugs:
    let prey_d min-one-of other ((bugs in-cone vision_r vision_a) with [ energy < 1 ]) [distance myself]
    ;; Finally, bugs also consider the food within their vision:
    let prey_f min-one-of (food in-cone vision_r vision_a)  [distance myself]

    ;; Target the nearest prey (bug, dead or alive, or food):
    let prey min-one-of (turtles with [member? self (list prey_b prey_d prey_f)]) [distance myself]
    ;; Go for the prey:
    ifelse prey != nobody
    [ face prey
      if ([breed] of prey = food) [ask prey [set color 45]] ;; Mark the food.
      ifelse distance prey < speed
      [ ;; Jump to the prey:
        setxy  [xcor] of prey [ycor] of prey
        ;; Eat:
        ifelse ([breed] of prey = food)
        [ set energy energy + food-value]
        [ ifelse [energy] of prey > 1
          [ set energy energy + ([size] of prey) * prey-value]
          [ set energy energy + ([size] of prey) * prey-value * 0.5] ;; Assume dead corpses are worse than live bodies!
          set bugs_dead bugs_dead + 1]
        ask prey [ die ]
      ]
      [ fd speed]
    ]
    [ ;; If no prey within vision, move forward at respective speed with a 20% chance of changing direction.
      let rnd random 100
      if rnd < 20 [ifelse rnd < 10 [right random vision_a] [right random (-1 * vision_a)]]
      fd speed
    ]

    ;; Set energy to decrease relative to current size and speed as well as the specified cost of each.
    set energy energy - ((cost-of-size * size ^ 0.9) + (cost-of-speed * size * speed ^ 1.2))
  ]
end

to reproduce
  ask bugs
  [ ;; If energy is above 100, decrease it by 50 and hatch a bug identical to self.
    if energy > 100
    [ set energy energy - 50
      hatch 1
      [ set age ticks
        ;; Each "gene" has a set chance of increasing or decreasing a set amount when being passed on to the offspring.
        if random 100 < mutation-rate
        [ ifelse random 100 < 50
          [ set speed speed * (1 + mutation-amount / 100) ]
          [ set speed speed * (1 - mutation-amount / 100) ]]

        if random 100 < mutation-rate
        [ ifelse random 100 < 50
          [ set size size * (1 + mutation-amount / 100) ]
          [ set size size * (1 - mutation-amount / 100) ]]

        if random 100 < mutation-rate
        [ ifelse random 100 < 50
          [ set vision_r vision_r * (1 + mutation-amount / 100) ]
          [ set vision_r vision_r * (1 - mutation-amount / 100) ]
          set vision_a min list (vision_area * 360. / (vision_r ^ 2)) 360]
      ]
    ]
  ]
end

to check-age
  ask bugs with [age > 0]
  [ if (ticks - age > bug_age - 10)
    [ set speed min list speed 0.5
      set energy 5
      set color 85]
    if (ticks - age > bug_age)
    [ set speed min list speed 0
      set energy 0
      set color 105]
  ]
end

to grow-food
  ;; Create food at the rate of food-growth at random places.
  if random 3 > p_food
  [ create-food food-growth
    [ setxy random-xcor random-ycor
      set shape "circle"
      set color red
      set size 0.5]
  ]
  set p_food 1 ;; for the first call p_food = -1, for the rest 1.
end
@#$#@#$#@
GRAPHICS-WINDOW
335
10
833
509
-1
-1
14.85
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
20
175
85
208
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
95
210
160
243
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
20
50
160
83
food-value
food-value
0
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
170
10
310
43
food-growth
food-growth
0
20
10.0
1
1
NIL
HORIZONTAL

PLOT
20
325
310
510
Population
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Bugs_act" 1.0 0 -16777216 true "" "plot count bugs - (count bugs with [energy < 1])"
"Bugs_all" 1.0 0 -7500403 true "" "plot count bugs"
"Food" 1.0 0 -2674135 true "" "plot count food"

SLIDER
20
90
160
123
prey-value
prey-value
0
100
40.0
1
1
NIL
HORIZONTAL

SLIDER
170
50
310
83
cost-of-speed
cost-of-speed
0
5
1.4
0.1
1
NIL
HORIZONTAL

SLIDER
170
90
310
123
cost-of-size
cost-of-size
0
5
1.4
0.1
1
NIL
HORIZONTAL

PLOT
1015
135
1175
255
Mean age
NIL
NIL
0.0
2.0
0.0
2.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot ifelse-value (count bugs = 0) \n[ 0 ]\n[ ticks -  mean [age] of bugs]"

PLOT
850
135
1010
255
Mean speed & size
NIL
NIL
0.0
1.0
0.0
1.0
true
false
"" ""
PENS
"speed" 1.0 0 -16777216 true "" "plot ifelse-value (count bugs = 0) \n[ 0 ]\n[ mean [speed] of bugs ]"
"size" 1.0 0 -2674135 true "" "plot ifelse-value (count bugs = 0) \n[ 0 ]\n[ mean [size] of bugs ]"

PLOT
850
305
1116
509
Bugs by size and speed
Speed
Size
0.0
1.0
0.0
1.0
true
false
"ask bugs [ plotxy speed size ]" "\n\n"
PENS
"pen-0" 0.01 2 -7500403 true "ask bugs [ plotxy speed size ]" "clear-plot\nask bugs [ plotxy speed size ]\n"

PLOT
850
10
1010
130
Mean vision
NIL
NIL
0.0
2.0
0.0
2.0
true
false
"" ""
PENS
"vision_r" 1.0 0 -16777216 true "" "plot ifelse-value (count bugs = 0) \n[ 0 ]\n[ mean [vision_r] of bugs ]"
"vision_a" 1.0 0 -2674135 true "" "plot ifelse-value (count bugs = 0) \n[ 0 ]\n[ (mean [vision_a] of bugs) * pi / 180]"

MONITOR
220
270
312
315
bugs_active
count bugs - (count bugs with [energy < 1])
0
1
11

SLIDER
20
130
160
163
mutation-rate
mutation-rate
0
100
20.0
1
1
%
HORIZONTAL

SLIDER
170
130
310
163
mutation-amount
mutation-amount
1
100
20.0
1
1
%
HORIZONTAL

BUTTON
95
175
160
208
go
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
1015
10
1175
130
Mean energy
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot ifelse-value (count bugs = 0) \n[ 0 ]\n[ mean [energy] of bugs ]"

MONITOR
120
270
210
315
bugs_dead
bugs_dead
17
1
11

MONITOR
20
270
110
315
bugs_inact.
count bugs with [energy < 1]
17
1
11

SLIDER
20
10
160
43
size_ratio
size_ratio
1
2.5
1.2
.1
1
NIL
HORIZONTAL

SLIDER
170
175
310
208
bug_age
bug_age
50
500
200.0
10
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

[This is a **child version** of the original model. Go below for a description of the changes: "Description of Version R1". Other than this, the info tab is as per the original].

A model of natural/artificial selection with bugs who compete for food. By controlling different aspects of the environment and gene mutation one can observe how the population of bugs adapt and the selection of genes differ.

The model shows adaption of a population through natural selection. There is no adaption on the agent level, rather it is probability that drives mutation of the genes which manifests as a battle of who can get more food and reproduce.


## HOW IT WORKS

Bugs wander around the area looking for food. Moving consumes energy and if energy reaches zero the bug dies. Eating refills energy. If energy reaches 100 the bug will reproduce and lower it's energy to 50. 

When reproducing, an exact copy of the bug will hatch. However, there is a chance set by sliders that each "gene" of the bug will mutate. Speed, size and vision can mutate higher or lower.

- Higher speed can help a bug reach more food in shorter time but comes at a higher cost of energy.

- Larger size allows bugs to eat smaller sized bugs (possible when double the size of another bug) which effectively supplies the environment with more possible food. Larger size also comes with a higher cost of energy.

- Greater vision allows bugs to see food at a further distance and move towards it and has no higher cost of energy.
 

## HOW TO USE IT


# Sliders

FOOD-VALUE and PREY-VALUE controls how much energy a bug receives from eating a piece of food and another bug (prey) respectively.

COST-OF-SPEED and COST-OF-SIZE controls how the cost of energy co-varies with speed and size. Meaning, how much more energy should be consumed by increasing size/speed.

MUTATION-RATE controls how likely it is that a gene will mutate between two generations. 

MUTATION-AMOUNT controls how much a gene can change between two generations.

FOOD-GROWTH controls the regrowth of food. This could be said to indicate if the bugs live in an environment of abundance or scarcity.


# Plots and monitors

The POPULATION plot on the left side shows population of bugs and food.

The SPEED, SIZE and VISION histograms show the distribution of values of the three genes.

The MEAN plots beneath them plots the mean value in the population of each gene. It also has a line for food-growth to visualize where changes were made during a run.

The BUGS PLOTTED BY SPEED AND SIZE plot each bug as a dot on a xy-plane with speed and size on the axes.


## THINGS TO NOTICE

Remember, evolution takes a very long time. Thus, there is no end to the model. It is exploratory. For each run of the model, let it run for at least a couple thousands, or even tens of thousands of ticks to see the long term effect on the population.

There is a tradeoff between size and speed which makes it energy expensive to be both.

Predators appear as the difference in size between bugs reaches a ratio of 2 when the larger bugs can eat the smaller. When this happens, the large "family" of bugs tends to reproduce quickly and take over parts or the whole environment.

Notice how vision, without having a cost of energy, is still subject to natural selection and is not always optimal to increase. Why?

When reaching extreme states the population takes longer to adapt to a change in FOOD-GROWTH. Why?


## THINGS TO TRY

- Investigate how the population evolves at different levels of food-growth.

- See if you can produce two different species co-existing e.g. one big slow and one small fast type of bug.

- Let the population adapt to one level of food growth then set it to a new level and see how the population adapts. Note that this can take many thousands of ticks.

- Move between the extreme values of food-growth.

- Change the cost-of-speed/size and prey/food-value to see how this affects the propensity of adaptation and predation.

- Explore how mutation rate/amount affects evolution.


## EXTENDING THE MODEL

The bugs have no flight behaviour in this model. This could be added to make the predator/prey relationship more authentic. 

A cone of vision could be used instead of 360 which would be more realistic and decrease the risk of bugs changing direction too often.

Further sliders for starting energy and reproduction thresholds could be added.


## DESCRIPTION OF VERSION R1

# Vision
In the original model, the bugs can see within a radius (vision) from theirselves, at full 360 deg.
In version R1 the bugs can see within a circular sector in front of them (in-cone instead of in-radius). 
The view sector is defined by an angle and a radius whose original values are introduced with a slider.
The radius is subject to genetic evolution (as in the original model), but the area of the view sector (PI*radius^2*angle/360) is constant. Therefore, an increase of the view radius by genetic evolution implies a decrease of the view angle.

# Prey
In the original model the prey is the nearest bug with at most half one's size; if there is no such bug, then the attention falls to the nearest food.
In version R1 the hunter bug selects the nearest target among three prey types: bugs_alive (of appropriate size), bugs_dead (of all sizes), or food.
When a bug-eats-bug, the energy that the eater obtains is proportional to the size of the prey.
The ratio between prey and hunter sizes, which is defined as 2 in the original model, is now a global variable through a slider.

# Eat
The original mode allows eating if the prey and the hunter are on the same patch.
This is changed in version R1: they must be within a predefined given distance (which is set as speed (speed*inter-tick time), but could be different.
Food preys are identified by yellow color.

# Changes in direction
In the original model the bugs with no prey in sight change directions with a chance of 20%, and the new heading can be any (0-360 deg).
In version R1 the bugs retain the same probability of changing direction, but they can only move within 2 times of their vision angle, not 360 deg.

# Cost of size and speed
In the original model the energy costs of size and speed are directly proportional to size and speed.
In version R1 the relationship is not linear, but follows a power expression a bit more realistic: more energy efficient wrt size, less energy efficient wrt speed.
The relationship between size and speed energy cost is now not multiplicative but additive.

# Deaths
In the original model bugs that reach an energy of 0 die.
In this version a bug that is "feeble" reduces its activity (speed) and then may more easily become a prey, but only "dies" (dissapears from the model) when eaten. 
Therefore, the original check-death procedure is cancelled. The move and eat procedures are also reorganized.
The level of "feebleness" could be related to the size, but it is fixed so far.
Two levels of energy wrt activity are defined: one which reduces the speed (and sets color to grey), and another which halts the bug's activity (and sets color to white).

# Age
In the original model age is not considered.
In version R1 the bugs have age measured from the ticks at which they were created.
At a given age, they reduce their mobility, and finally they cease moving. They only "die" (dissapear from the model) when eaten.

# Plots and monitors
New buttons, sliders, plots and monitors are introduced, or modify the original ones.

## RELATED MODELS

- Vision evolution
- Bug Hunt Coevolution
- Wolf sheep predation

## CREDITS AND REFERENCES

This model was created as a part of the Introduction to Agent-Based Modeling course on Complexity Explorer: https://www.complexityexplorer.org/courses/96-introduction-to-agent-based-modeling.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
