;; --------------------------------------------------------------------------
;; A multi-leader swarm model for bound-constrained numerical optimization
;; based on follow-the-leader and random walk (FLW) operators.
;;
;; This agent-based model aims to find the patch with the global minimum cost
;; value within the search space (landscape) of the chosen problem.
;;
;; A model by Sergio Rojas-Galeano
;; v1.5, April 2022 Copyright (c) The author
;; Correspondance email: srojas@udistrital.edu.co
;; Universidad Distrital Francisco Jose de Caldas, Bogota, Colombia
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License (GPLv3)
;; (see license at: https://www.gnu.org/licenses/gpl-3.0.txt)
;;
;; The model is made publicly available in the hope that it will be useful
;; to modelers, but WITHOUT ANY WARRANTY whatsoever (see license for details).
;; --------------------------------------------------------------------------

globals [
  ;; FLW algorithm globals
  top-leader          ; best agent in current iteration
  top-leader-fitness  ; fitness value of current top leader
  flw-best-patch      ; best solution (patch) discovered by FLW
  flw-runtime         ; runtime (ms) until optimum discovery or max. ticks
  flw-optimum-tick    ; tick where optimum was discovered (if found)
  avg-cohesion        ; average cohesion of followers to their leaders

  ;; Problem globals
  true-best-patch     ; ground truth optimum patch for a given landscape
  agent-size          ; agent size depending on lanscape's grid size
]

patches-own [
  x                   ; simulated pxcor, depending on bound constraints
  y                   ; simulated pycor, depending on bound constraints
  value               ; value of cost function at these simulated coordinates
]

turtles-own [
  my-leader           ; leader that this agent is following
]

;; FLW breeds
breed [leaders leader]
breed [followers follower]
breed [walkers walker]

;; Initialise problem landscape and swarm of agents
to setup

  ;; compute problem landscape (cost function) and set initial best patch randomly
  setup-search-landscape
  set flw-best-patch one-of patches
  reset-ticks

  ;; create walker agents
  create-walkers swarm-size * walkers-rate [
    set color gray set size agent-size                    ; assing walker color & size
    move-to one-of patches                                ; assign initial random location
  ]

  ;; create leader agents and their followers
  let colors shuffle [ 15 25 65 75 95 115 125 135 ]        ; colors to distinguish leaders
  create-leaders n-leaders [
    set color first colors set colors remove color colors ; assing random color
    set shape "bird" set size agent-size                  ; assign shape and size
    set my-leader self
    move-to one-of patches                                ; assign initial random location

    ;; create an equal number of followers for this leader and spread them around
    hatch-followers (swarm-size - count walkers - n-leaders) / n-leaders [
      rt random-normal 0 360 fd random world-width
    ]
  ]

  ;; initialise FLW globals
  update-best
end

;; Execute one iteration of FLW algorithm
to go
  reset-timer

  ;; apply search operators
  do-elitism
  follow-leaders
  random-walks
  track-clashes
  track-cohesion
  track-leadership


  ;; update FLW globals
  update-best
  update-runtime
  show-spotlight
  tick

  ;; termination condition: max ticks, or optimum discovered
  if (ticks > max-ticks) or ((flw-optimum-tick > 0) ) [ stop ]
end

;; Explotiation operator: follow-the-leader
to follow-leaders
  ask leaders [ rt random-normal 0 30 fd random-float 1 ]
  ask followers [ follow-move ]
end

;; Move a follower towards its leader
to follow-move
  face my-leader fd (distance my-leader) * (random-float 2) ; move towards leader...
  rt random-normal 0 60 fd random-float 2  ; ... with a small local perturbation
end

;; Exploration operator: random walks
to random-walks
  ask walkers [
    walk-move
    if value < top-leader-fitness [        ; if walker is better, drag top-leader
      ask top-leader [ move-to myself ]
    ]
  ]
end

;; Move a walker around
to walk-move
  rt random-normal 0 30 jump (random-float 20)
end

;; Move walker to best solution so far
to do-elitism
  ask walker 0 [
    if elitism? [ move-to flw-best-patch walk-move ]
    set shape ifelse-value elitism? [ "star" ][ "default" ]
  ]
end

;; Spread away clashing leaders and their followers
to track-clashes
  ask leaders [
    ask other leaders with [ distance myself < (world-width - 1) / 25 ] [
      move-to one-of patches
      let me my-leader
      ask followers with [ my-leader = me ] [ follow-move ]
    ]
  ]
end

;; Verify early convergence to local minima (low cohesion)
to track-cohesion
  if cohesion? [
    ;; compute average cohesion measure
    let total 0
    ask leaders [
      set total total + sum [distance myself] of followers with [ my-leader = myself ]
    ]
    set avg-cohesion total / swarm-size
  ]

  ;; spread out if early convergence to local minima
  if (cohesion? and avg-cohesion < .75) or (not cohesion? and ticks mod 30 = 0)[
    ask turtles [ rt 360 fd random 200 ]  ; radial dispersion
  ]
end

;; Allow best followers to claim leadership
to track-leadership
  ask leaders [
    ;; swap places with best follower (including itself)
    let my-old-patch patch-here
    let my-best-follower min-one-of turtles with [ my-leader = myself ] [value]
    move-to my-best-follower
    ask my-best-follower [ move-to my-old-patch ]
  ]
end

;; Update best solution discovered so far
to update-best
  ;; best solution so far would be top-leader
  ask min-one-of leaders [value] [
    set top-leader self
    set top-leader-fitness value
    if top-leader-fitness < [value] of flw-best-patch [
        set flw-best-patch patch-here

        ;; check if optimum found (termination condition)
        if flw-best-patch = true-best-patch [ set flw-optimum-tick ticks ]
    ]
  ]
end

;; Update current algorithm runtime
to update-runtime
  set flw-runtime flw-runtime + timer
end

;; Reset simulation (without setting up world and landscape from scratch)
to reset
  ask turtles [ move-to one-of patches ]
  set flw-best-patch one-of patches
  set top-leader one-of leaders
  set flw-optimum-tick 0
  set flw-runtime 0
  set avg-cohesion 0
  update-best
  show-spotlight
  clear-all-plots
  reset-ticks
end

;; Create the fitness landscape depending on optimisation problem
to setup-search-landscape
  clear-all

  ;; set view (world) size and cell (patch) size
  (ifelse
    grid-size = "100x100" [ resize-world -50 50 -50 50  set agent-size 4 ]
    grid-size = "200x200" [ resize-world -100 100 -100 100 set agent-size 8 ]
    grid-size = "400x400" [ resize-world -200 200 -200 200 set agent-size 16 ]
    grid-size = "800x800" [ resize-world -400 400 -400 400 set agent-size 18 ]
  )
  set-patch-size  ifelse-value grid-size = "800x800" [ 1 ] [ 400 / (world-width - 1) ]

  ;; make a landscape with hills and valleys according to chosen cost function
  ask patches [
    set x pxcor * (xy-bounds / max-pxcor)
    set y pycor * (xy-bounds / max-pycor)

    set value (ifelse-value
      landscape = "Sphere" [
        x ^ 2 + y ^ 2
      ]
    	landscape = "Sphere-offset" [
        (x - 50 * (xy-bounds / max-pxcor) ) ^ 2  + (y + 50 * (xy-bounds / max-pxcor) ) ^ 2
      ]
      landscape = "Rastrigin" [ ; note that degrees, not radians, are needed for cos function
        20 + ((x ^ 2) -  10 * cos ( (180 / pi) * (2 * pi) * x )) + ((y ^ 2) -  10 * cos ( (180 / pi) * (2 * pi) * y ))
    	]
      landscape = "Rosenbrock" [
        100 * (y - (x ^ 2))^ 2 + (1 - x)^ 2
      ]
      landscape = "Himmelblau" [
        ((x ^ 2) + y - 11) ^ 2 + (x + (y ^ 2) - 7)^ 2
      ]
      landscape = "Eggholder" [ ; note that degrees, not radians, are needed for sin function
        ( (- x) * sin ( (180 / pi) * sqrt (abs (x - (y + 47))))) - (y + 47) * sin ( (180 / pi) * sqrt (abs ((x / 2) + (y + 47))))
      ]
      [ random-normal 0 500  ] ; the last case is a random landscape
;      [ random-float 500  ] ; the last case is a random landscape
    )
  ]

  if landscape = "Random" [
    ask min-n-of 4 patches [value] [ ask patches in-radius 30 [ set value value - random-float 300 ] ]
    repeat 20 [ diffuse value 1 ]
  ]

  ;; find the true best value
  ask min-one-of patches [value][ set true-best-patch self ]

	;; scale patches color within values limits
  let min-val min [value] of patches
  let max-val max [value] of patches
	ask patches [ set pcolor scale-color yellow value min-val log abs max-val 1.05 ]
end

;; Turn on the spotlight on the chosen agent
to show-spotlight
  ifelse spotlight = "Best patch found"
  [ watch flw-best-patch]
  [ ifelse spotlight = "Ground truth patch"
    [  watch true-best-patch ]
    [ ifelse spotlight = "Top leader now"
      [  watch top-leader ]
      [  reset-perspective ]
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
608
12
1018
423
-1
-1
2.0
1
10
1
1
1
0
1
1
1
-100
100
-100
100
0
0
1
ticks
30.0

CHOOSER
6
10
115
55
landscape
landscape
"Sphere" "Sphere-offset" "Rastrigin" "Rosenbrock" "Himmelblau" "Eggholder" "Random"
4

BUTTON
129
194
184
227
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
188
237
243
270
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
6
110
116
143
swarm-size
swarm-size
5
40
20.0
5
1
NIL
HORIZONTAL

CHOOSER
124
60
246
105
spotlight
spotlight
"Off" "Best patch found" "Top leader now" "Ground truth patch"
3

MONITOR
13
336
237
381
best solution found
word \"f ( \" \nprecision [x] of flw-best-patch 2 \", \" \nprecision [y] of flw-best-patch 2 \" ) = \"\nprecision [value] of flw-best-patch 6
5
1
11

MONITOR
13
282
237
327
ground truth solution
word \"f ( \" \nprecision [x] of true-best-patch 2 \", \" \nprecision [y] of true-best-patch 2 \" ) = \"\nprecision [value] of true-best-patch 6
5
1
11

SLIDER
124
110
247
143
walkers-rate
walkers-rate
0.2
0.8
0.2
0.1
1
NIL
HORIZONTAL

PLOT
257
10
596
130
top leader fitness
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -955883 true "" "plot top-leader-fitness"

BUTTON
129
237
184
270
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
13
391
237
436
current top leader
word \"f ( \" \nprecision [x] of top-leader 2 \", \" \nprecision [y] of top-leader 2 \" ) = \"\nprecision [value] of top-leader 6
17
1
11

MONITOR
256
391
365
436
runtime (ms)
ifelse-value (true-best-patch = flw-best-patch) \n[ (word precision flw-runtime 3 \"!!!\") ] \n[ flw-runtime ]
2
1
11

PLOT
256
136
596
256
best solution fitness
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot [value] of flw-best-patch"

PLOT
256
262
596
382
average group cohesion
NIL
NIL
0.0
10.0
0.0
2.0
true
false
"" ""
PENS
"cohesion" 1.0 0 -5825686 true "" "plot avg-cohesion"

SLIDER
124
150
247
183
max-ticks
max-ticks
1000
4000
1000.0
1000
1
NIL
HORIZONTAL

MONITOR
373
391
481
436
optimum tick
ifelse-value (true-best-patch = flw-best-patch) \n[ (word flw-optimum-tick  \"!!!\") ] \n[ flw-optimum-tick ]
17
1
11

CHOOSER
124
10
246
55
xy-bounds
xy-bounds
6 512
0

SLIDER
6
150
116
183
n-leaders
n-leaders
1
8
4.0
1
1
NIL
HORIZONTAL

BUTTON
188
194
243
227
NIL
reset
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
6
60
115
105
grid-size
grid-size
"100x100" "200x200" "400x400" "800x800"
1

MONITOR
489
391
596
436
NIL
avg-cohesion
2
1
11

SWITCH
7
236
117
269
cohesion?
cohesion?
0
1
-1000

SWITCH
7
193
117
226
elitism?
elitism?
0
1
-1000

@#$#@#$#@
## WHAT IS IT? 

This is an agent-based model that implements a swarm search algorithm for numerical optimisation of real-valued bound-constraint functions, based mainly on two search operators: Follow the Leader and Random Walk (FLW). The goal of the swarm is to discover the best patch within a discretised landscape for a given optimisation problem, that is, to find the optimal (minimum) value of the discretised cost function for said problem, in a more efficient way than traversing the entire landscape (exhaustive search).


The complete set of operations performed by the algorithm are: 

- Follow the leader: there are groups of agents that follow a leader. 

- Random walk: there are scout agents moving on their own.

- Single- or multi-leaders mode: there can be multiple leaders in the swarm.

- Leadership relay: leadership is constantly updated based on fitness.

- Clash safeguard: leader agents must stay away from each other to avoid collisions.

- Monitor premature convergence: when cohesion is too low, agents disperse.

- Enable elitism: an agent stays near the best patch discovered so far.



## HOW IT WORKS 

The optimisation problem defined by the LANDSCAPE chooser (from SPHERE, RASTRIGIN, ROSENBROCK, HIMMELBLAU, EGGHOLDER, or RANDOM), is discretised and projected onto the 2D grid of cells (patches) that make up the view area of the simulation. Therefore, the swarm of agents aims to discover the optimal patch in said projected landscape, that is, the coordinates where the cost function is the lowest.

Each agent is characterised by a location (x,y) in the landscape and the value of the  cost function evaluated at that patch, which also indicates its fitness as a candidate solution for the chosen problem. Now each agent can have one of three possible roles (or breeds): LEADER (an agent located in a promising patch), FOLLOWER (an agent following a leader), or WALKER (a scout agent).

In the simulation, LEADERS are shown as a "bird" shape (reminiscent of earlier models of swarm intelligence inspired by the behavior of flocks of birds) and a different color, while FOLLOWERS are shown as a triangle and the color of his LEADER. WALKERs, in turn, are shown as a gray "triangle" shape. As a final solution to the problem, the algorithm maintains a FLW-BEST-PATCH ever discovered during the simulation run. Similarly, the TOP-LEADER represents the best solution during a given iteration.

LEADERs and WALKERs move randomly around their current positions, however LEADERs move within a small range of distance (local exploitation) while WALKERS move within a larger range (global exploration). When a WALKER accidentally discovers a more promising patch than those of the current leaders, the TOP-LEADER is asked to move to that patch.

Furthermore, every LEADER has an (equally larger) group of FOLLOWERs. FOLLOWERs move in pursuit of their leaders: they face their corresponding leader and jump in that direction with a random step length. When a FOLLOWER accidentally discovers a more promising patch, he switches roles to become the LEADER of their group (in practice, since all agents are homogeneous, they simply trade places).

In each iteration of the algorithm, all the agents of the swarm traverse the landscape moving according to their roles. Additionally, three operators were introduced to improve search efficacy:

- Monitor premature convergence: When the average cohesion of groups within the swarm falls below a certain level, agents disperse by performing a warm restart to random locations.

- Clash safeguard: Prevents two leaders with their groups from colliding in the same landscape region.

- Elitism: Recalls the best discovered patch (the best solution found so far) by moving a WALKER to that position.

The simulation ends after a maximum number of steps (MAX-TICKS) is reached, or when the GROUND-TRUTH-SOLUTION is discovered first.

## HOW TO USE IT

First, choose an optimisation problem to solve from the LANDSCAPE dropdown list and select the corresponding XY-BOUNDS (recommended values are [-6, 6] for SPHERE, RASTRIGIN, ROSENBROCK, and HIMMELBLAU, and [-512, 512] for EGGHOLDER and RANDOM). The GRID-SIZE chooser allows defining the sampling resolution of the 2D grid of patches where the cost function will be evaluated (the larger this size, the harder it is to discover the best patch).

Then choose the number of agents to simulate with the SWARM-SIZE slider, the percentage of walkers in the swarm with the WALKERS-RATE slider, and the number of N-LEADERS within the swarm. Also, choose the termination condition with the MAX-TICKS slider and use the ELITISM? and COHESION? switches to enable or disable the corresponding operators. A typical configuration of these parameters would be (to simulate a population of 20 agents, including 4 leaders with 3 followers each, plus 4 walkers):

- SWARM-SIZE: 20

- N-LEADERS: 4

- WALKERS-RATE: 0.2

- ELITISM?: On

- COHESION?: On

- MAX-TICKS: 1000

Now you can run the simulation using the buttons in the control panel:

- SETUP: Computes the problem LANDSCAPE and displays it in the view grid as a 2D discretised surface plot. In addition, it creates the swarm, scatters the agents randomly within the boundaries of the landscape, and initialises the algorithm global variables (depending on the GRID-SIZE this setup may take a while).

- RESET: Relocates the agents randomly in the 2D view and initialises the algorithm's global variables (this is much faster than SETUP, as it doesn't recalculate the problem landscape, which might be useful for multiple simulation runs).

- GO: Iteratively execute the main procedure of the algorithm until finding the optimum or reaching the MAX-TICKS.

- STEP: Execute an iteration (a single tick) of the main procedure of the algorithm.

- SPOTLIGHT: A useful chooser to keep track of the optimal solution, the best solution found so far, or the current top leader within the display area.

Also remember that the SPEED slider at the top of the window can be used to control how fast the simulation runs in the view grid.

Once you start running the simulation, the swarm behavior will emerge in the view grid, as the agents try to discover the promising regions within the landscape and hopefully find the optimal patch before MAX-TICKS. Besides, the output monitors allow you to inspect such emerging behaviour as the model evolves:


- GROUND-TRUTH-SOLUTION: Coordinates and cost function value of the optimum.

- BEST-SOLUTION-FOUND: Coordinates and cost function value of the FLW-BEST-PATCH discovered so far.

- CURRENT-TOP-LEADER: Coordinates and cost function value of the leader at the best patch in the current iteration.

- TOP-LEADER-FITNESS: Historical plot of the leader with best fitness (value of cost function) throughout the simulation.

- BEST-SOLUTION-FITNESS: Historical plot of best solutions discovered throughout the simulation.

- AVG-COHESION: The average measure of cohesion of the groups within the swarm (only if COHESION? is activated).

- AVERAGE-GROUP-COHESION: Historical plot of the average cohesion measurement throughout the simulation (only if COHESION? is enabled).

- RUNTIME: Total execution time of the simulation run.

- OPTIMUM-TICK: The tick number where the optimum was discovered (if it happens). Notice that if the algorithm is able to do so, these last two monitors will display a "!!!" sign in their text box.


## THINGS TO NOTICE

Each LANDSCAPE may exhibit different properties: multimodality, convexity, separability, multiple global or local minima, plateaus, ridges, valleys, etc. The advantage of swarm optimisation methods is their ability to explore several regions of the search space simultaneously so they can discover these properties without getting stuck in local minima, as opposed to single point metaheuristics. However, it is also known that these swarm-based methods may suffer from premature convergence where the entire population collapses to a single suboptimal patch or small region. For reference, each landscape exhibits the following properties (the actual mathematical expressions and 2D landscapes can be seen in the Appendix of [1]):

- SPHERE, SPHERE-OFFSET: unimodal, convex.

- RASTRIGIN: multimodal, multiple local minima, ridges.

- ROSENBROCK: multiple local minima, valleys.

- HIMMELBLAU: multimodal, multiple global minima, valleys.

- EGGHOLDER: multimodal, multiple local minima, valleys.

- RANDOM: multimodal, multiple local minima, sampled from a scaled normal distribution.

The FLW algorithm attempts to circumvent such swarm-based drawbacks by using two features. First, it allows for multiple leaders, meaning you can choose between having a single swarm of agents with an unique leader, or having multiple leaders with different subgroups of followers (distinguished by different colors). When choosing the multi-leader mode, you can notice how the subgroups spread out looking for minima in separate regions of the LANDSCAPE, which is partly due to the anti-clash mechanism that the algorithm implements to avoid collisions between leaders. You can test this emergent behaviour (leaders dominating separate regions of the search space) by experimenting with different values for the N-LEADERS and SWARM-SIZE sliders.

The second feature is intended to prevent premature convergence by dispersing agents by means of a warm restart of agent locations every time AVG-COHESION drops below a certain threshold (if COHESION? is enabled), or periodically once a predetermined number of ticks is reached (if COHESION? is disabled) during the execution of the GO loop. Since relocated leaders can start exploring newer unknown regions, an ELITISM-like learning mechanism allows the swarm to keep track of the most promising solution found so far in previous restarts by moving an arbitrary WALKER to the BEST-SOLUTION-FOUND patch; this way, eventually a leader would be attracted to that patch ("the elite") if no better solutions are found in the new restart.

You can test the above behaviour by experimenting with the COHESION? and ELITISM? switches, whose effect is depicted in the view area and in the periodic saw-like patterns that appear in AVERAGE-GROUP-COHESION and TOP-LEADER-FITNESS plots. Use the SPEED slider at the top to slow down the display of the warm restart mechanism. You can use the SPOTLIGHT chooser to highlight in the view area the location of the LANDSCAPE's GROUND-TRUTH-SOLUTION or the current BEST-SOLUTION-FOUND patch or the current TOP-LEADER.  
## THINGS TO TRY

The following are interesting simulation scenarios:

- Solve each LANDSCAPE with a single-leader swarm. Repeat the experiment many times (use the SETUP button once, then RESET and GO as many times as you like). What is the success rate for each problem? (i.e. what percentage of the total number of repetitions was the FLW algorithm able to discover the ground-truth optimum?)

- Is the multi-leader strategy better than the single-leader strategy? Which one achieves a higher success rate? Do you get better results as the number of leaders increases? You can test the last hypothesis by trying different combinations of pair values (N-LEADERS, SWARM-SIZE): (1, 5), (2, 10), (4, 20), (8, 40).

- Try to solve each LANDSCAPE with different resolutions (GRID-SIZE): 100x100, 200x200, 400x400, 800x800. This parameter affects the sampling rate of the problem cost function during quantisation and thus changes the size of the search space (the larger this size, the more difficult the search for the best patch). How does this affect the success rate of the algorithm? How does this affect the speed of convergence (that is, the average tick where the optimum is discovered)?

- Notice that unlike the other problems, the RANDOM problem produces a different landscape each time you press SETUP. It would be interesting to investigate if nonetheless the FLW algorithm is able to solve it or if some instances present more difficulty than others.

As a side note, we remark that the resolution level can induce quantisation errors during the cost function sampling, therefore the optimum patch coordinates of a given LANDSCAPE can differ depending on the GRID-SIZE. For example, the optimum of ROSENBROCK'S problem for 100x100 and 200x200 resolutions is at *f(1.2, 1.44) = 0.04*, but for 400x400 resolution it is at *f(0.93, 0.87) = 0.007501*, whereas for 800x800 resolution it is at *f(1, 1) = 0.00255*.

Another interesting idea to experiment with is to assess the performance of the model under different swarm configurations, in terms of efficiency. Since the worst case for a single-agent algorithm would be to perform an exhaustive search of the entire LANDSCAPE, i.e., evaluate the cost function on the entire grid of NxN patches (where N depends on the GRID-SIZE), the efficiency of each algorithm configuration can be measured as the percentage of evaluations of the cost function needed to find the optimum, with respect to NxN, in other words, the average rate OPTIMUM-TICK / NxN. The following swarm configurations can be tested:

- *Simultaneous Global Random Search*: A swarm of WALKERs (SWARM-SIZE: 5, N-LEADERS: 1, WALKERS-RATE: 0.8, COHESION?: OFF, ELITISM?: OFF)

- *Simultaneous Local Random Search*: A swarm of singleton LEADERs (SWARM-SIZE: 5, N-LEADERS: 5, WALKERS-RATE: 0.2, COHESION?: OFF, ELITISM?: OFF)

- *Single-leader Swarm Search*: A swarm of agents with an unique LEADER, FOLLOWERs and WALKERs (SWARM-SIZE: 5, N-LEADERS: 1, WALKERS-RATE: 0.2, COHESION?: OFF, ELITISM?: OFF).

- *Multi-leader Swarm Search*: A swarm of agents with multiple LEADERs, FOLLOWERs and WALKERs (SWARM-SIZE: 20, N-LEADERS: 4, WALKERS-RATE: 0.2, COHESION?: OFF, ELITISM?: OFF)

- *FLW swarm search*: A swarm of agents with multiple LEADERs, FOLLOWERs and WALKERs, plus cohesion tracking and elitism (SWARM-SIZE: 20, N-LEADERS: 4, WALKER-RATE: 0.2, COHESION?: ON, ELITISM?: ON)

Can you identify which configuration (hence which search strategy/operators) improves the optimisation effectiveness/efficiency of the FLW algorithm? Is there a minimum follower base size for the multi-leader setup to work properly?

## EXTENDING THE MODEL

Some possible paths for model extensions are:

- Expand the landscape repertoire with additional problem cost functions.

- Extend the model to solve problems in higher dimensions (not necessarily using a 2D grid view).

- Extend the model to discrete (binary) problem domains.

- Extend the model to solve non-stationary problems, that is, problems where the landscape can vary with time during the course of a run. In this sense, we believe that the mechanisms to prevent premature convergence implemented in the FLW algorithm can be useful to also adapt to this type of dynamic changes in the landscape.


## RELATED MODELS

Modeling Commons -> Urban Pigeon-inspired Model for Unconstraint Optimisation, see [1].

(http://modelingcommons.org/browse/one_model/6390)

NetLogo Models Library -> Particle Swarm Optimization (PSO), see [2].

(https://ccl.northwestern.edu/netlogo/models/ParticleSwarmOptimization)

## CREDITS AND REFERENCES

Authors:  

Sergio Rojas-Galeano 

Copyright (c) April 2022

email: srojas@udistrital.edu.co

Version 1.5

Licenses:

- The model code is licensed as GNU General Public License (GPLv3) 
  (see https://www.gnu.org/licenses/gpl-3.0.txt)

- This Info Tab document is licensed as CC BY-NC-ND 
  (see https://creativecommons.org/licenses/by-nc-nd/4.0/)

References:

[1] Garzon, M., and Rojas-Galeano, S. (2019, November). An Agent-Based Model of Urban   	    Pigeon Swarm Optimisation. In 2019 IEEE Latin American Conference on Computational        Intelligence (LA-CCI) (pp. 1-6). IEEE.  doi: 10.1109/LA-CCI47412.2019.9036758.
    https://ieeexplore.ieee.org/document/9036758

[2] Stonedahl, F. and Wilensky, U. (2008). NetLogo Particle Swarm Optimization model.         http://ccl.northwestern.edu/netlogo/models/ParticleSwarmOptimization. Center for          Connected Learning and Computer-Based Modeling, Northwestern Institute on Complex         Systems, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bird
false
0
Polygon -7500403 true true 135 165 90 270 120 300 180 300 210 270 165 165
Rectangle -7500403 true true 120 105 180 237
Polygon -7500403 true true 135 105 120 75 105 45 121 6 167 8 207 25 257 46 180 75 165 105
Circle -16777216 true false 128 21 42
Polygon -7500403 true true 163 116 194 92 212 86 230 86 250 90 265 98 279 111 290 126 296 143 298 158 298 166 296 183 286 204 272 219 259 227 235 240 241 223 250 207 251 192 245 180 232 168 216 162 200 162 186 166 175 173 171 180
Polygon -7500403 true true 137 116 106 92 88 86 70 86 50 90 35 98 21 111 10 126 4 143 2 158 2 166 4 183 14 204 28 219 41 227 65 240 59 223 50 207 49 192 55 180 68 168 84 162 100 162 114 166 125 173 129 180

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="xy-bounds">
      <value value="512"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Eggholder&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-alpha" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.01"/>
      <value value="0.1"/>
      <value value="1"/>
      <value value="10"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-sigma" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="0.01"/>
      <value value="0.1"/>
      <value value="1"/>
      <value value="10"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-popsize" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="20"/>
      <value value="40"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-walkrate" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-maxticks" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="2000"/>
      <value value="5000"/>
      <value value="10000"/>
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
