undirected-link-breed [friends friend]
undirected-link-breed [locals local]

globals [
  average_smoking_attitude
  average_security_desire
  average_conformity_desire
  average_hedonism_desire
  count_change_environment
  count_maladaptive_behavior
  count_adaptive_behavior
  count_adjust
  count_do_nothing
  objective_severity
  n_smokers
  n_non_smokers
  n_casuals
]

turtles-own [
  smoking_attitude
  health_status
  security_desire
  conformity_desire
  hedonism_desire
  people_nearby
  avg_peer_attitude
  avg_local_attitude
  avg_perceived_attitude
  intrinsic_rewards
  extrinsic_rewards
  severity
  vulnerability
  response_efficacy
  self_efficacy
  response_cost
  threat_appraisal
  coping_appraisal
  days_since_env_change
]
friends-own [connection_strength]
locals-own [connection_strength]


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                 ;;
;;                                              SETUP                                              ;;
;;                                                                                                 ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; Creates agents, initiates their desires, friends and pmt variables.
to setup
  ca
  random-seed seed
  set objective_severity 0.64 ;relative value calculated from statista data
  set count_change_environment 0
  set count_maladaptive_behavior 0
  set count_adaptive_behavior 0
  set count_do_nothing 0
  set count_adjust 0
  let total_people_percent  conformity_people + hedonism_people + security_people

  let n_conformity_people int( (conformity_people / total_people_percent) * n_people ) + 1
  let n_hedonism_people int( (hedonism_people / total_people_percent) * n_people ) + 1
  let n_security_people int( (security_people / total_people_percent) * n_people ) + 1
  if (n_security_people + n_conformity_people + n_hedonism_people < n_people) [
    let dif  n_people - ( n_security_people + n_conformity_people + n_hedonism_people )
    set n_conformity_people n_conformity_people + dif
  ]
  crt n_conformity_people [
    setxy random-xcor random-ycor
    set shape "face neutral"
    set conformity_desire (70 + (random (100 - 70))) / 100
    set security_desire (40 + (random (60 - 40))) / 100
    set hedonism_desire (40 + (random (60 - 40))) / 100
  ]

    crt n_hedonism_people [
    setxy random-xcor random-ycor
    set shape "face happy"
    set conformity_desire (40 + (random (60 - 40))) / 100
    set security_desire (0 + (random (30 - 0))) / 100
    set hedonism_desire (70 + (random (100 - 70))) / 100
  ]

    crt n_security_people [
    setxy random-xcor random-ycor
    set shape "face sad"
    set conformity_desire (40 + (random (60 - 40))) / 100
    set security_desire  (70 + (random (100 - 70))) / 100
    set hedonism_desire (0 + (random (30 - 0))) / 100
  ]

  ask turtles [
    set avg_peer_attitude 0
    set intrinsic_rewards 0
    set extrinsic_rewards 0
    set severity 0
    set vulnerability 0
    set response_efficacy 0
    set self_efficacy 0
    set response_cost 0
    set threat_appraisal 0
    set coping_appraisal 0
    set days_since_env_change 0
    set people_nearby nobody
    create-friends-with n-of number_of_friends other turtles
    init_health_status
    init_smoking_attitude ; initializes smoking_attitude
    set_color_for_turtles
  ]

  ask friends [
   init_friend
  ]
  calculate_average_smoking_attitude
  show-label
  watch-one
  reset-ticks
end


; Initiates the health status of an agent as a random value [0,100]
to init_health_status
  set health_status (1 + (random (100 - 1))) / 100
end


; Initiates smoking attitude of an agent depending on their security and hedonism desire
to init_smoking_attitude
    set smoking_attitude ((1 - security_desire) * hedonism_desire)
  ifelse (smoking_attitude > 0.5)[
    set n_smokers n_smokers + 1
  ]
  [set n_non_smokers n_non_smokers + 1]
end


; Sets the color of a turtle based on their smoking attitude
to set_color_for_turtles
  (ifelse (   smoking_attitude >= 0.75) [set color red ]
    (smoking_attitude < 0.75 and smoking_attitude > 0.25) [set color orange]
    [set color green]
    )
end


; Sets the connection strength between agents
to init_friend
    ifelse static? = true [
    set connection_strength 1
  ]
  [
    set connection_strength ( 80 + (random (120 - 80))) / 100
  ]
    set color red
    set thickness connection_strength / 4
end


; Shows labels on links in interface
to show-label
  if label? = true [
    ask turtles [set label who]
    ask links [ set label precision connection_strength 3 ]]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                 ;;
;;                                         EXPERIMENT                                              ;;
;;                                                                                                 ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; Executes the model for one tick
to go
  ;print (word "================================== tick # "ticks" ==================================")
  ask turtles [
    move
    connect_to_locals
  ]
  ask locals[
    set color green
    set connection_strength  1
    set thickness connection_strength / 4
  ]

    ask turtles [
      calculate_local_pressure
      calculate_social_pressure

      let friends_count count friend-neighbors
      let locals_count count local-neighbors
      let neighbors_count (friends_count + locals_count)
    ifelse (neighbors_count < 1 ) [
      set avg_perceived_attitude smoking_attitude
    ]
    [

      set avg_perceived_attitude ((avg_peer_attitude * (friends_count / neighbors_count)) + (avg_local_attitude * (locals_count / neighbors_count)))
      set avg_perceived_attitude avg_perceived_attitude * pressure_modifier
    ]

      protection_motivation
    ]

  calculate_average_smoking_attitude
  calculate_average_security_desire
  calculate_average_conformity_desire
  calculate_average_hedonism_desire
  show-label
  watch-one
  behavior
  tick
end


; Runs the model for "ticks_to_run" times
to start
  loop [
    if ticks >= ticks_to_run [stop]
    go
  ]
end


; Executes the agent behavior depending on the calculated threat and coping appraisal
to behavior
  ask turtles [
    ; raucher
    (ifelse ( smoking_attitude >= 0.5)[
      ; raucherfreunde
      (ifelse ( abs( smoking_attitude -   avg_perceived_attitude) <= change_rate  or smoking_attitude >= 0.98) [
        set count_do_nothing count_do_nothing + 1
      ]
        ;nichtraucherfreunde
      (coping_appraisal > threat_appraisal  ) [
        set count_adaptive_behavior count_adaptive_behavior + 1
        update_desires
      ]
      (threat_appraisal > coping_appraisal) [
      set count_maladaptive_behavior count_maladaptive_behavior + 1
          ifelse (days_since_env_change >= min_stay and abs( smoking_attitude -   avg_perceived_attitude) >= contact_termination ) [
            change_environment
            set days_since_env_change 0
          ]
          [
          set days_since_env_change days_since_env_change + 1
          update_desires
          ]
      ]
        )
    ]
     ; nichtraucher
    (smoking_attitude <= 0.5) [
      (ifelse ( abs( smoking_attitude -   avg_perceived_attitude) <= change_rate  or smoking_attitude <= 0.02) [
        set count_do_nothing count_do_nothing + 1
        ]
      (coping_appraisal > threat_appraisal ) [
       ifelse (days_since_env_change >= min_stay and abs( smoking_attitude -   avg_perceived_attitude) >= contact_termination)  [
            change_environment
            set days_since_env_change 0
          ]
          [
          set days_since_env_change days_since_env_change + 1
          update_desires
          ]
      ]
      (threat_appraisal > coping_appraisal ) [
        set count_maladaptive_behavior count_maladaptive_behavior + 1
        update_desires
      ]
          )

    ]
    )
    let oldcolor color
    set_color_for_turtles
		if (color != oldcolor and color = red) [
      ifelse oldcolor = orange [set n_casuals n_casuals - 1][ set n_non_smokers n_non_smokers - 1]
      ;print (word who " turned red")
      set n_smokers n_smokers + 1

    ]
		if (color != oldcolor and color = green) [
      ifelse oldcolor = orange [set n_casuals n_casuals - 1][ set n_smokers n_smokers - 1]
      ;print (word who " turned green")
      set n_non_smokers n_non_smokers + 1
    ]

    if (color != oldcolor and color = orange) [
      ifelse oldcolor = red [set n_smokers n_smokers - 1][ set n_non_smokers n_non_smokers - 1]
    set n_casuals n_casuals + 1
    ]
	]
  show-label
  watch-one
end


; Moves the agent for the specified distance
to move
  fd moving_distance
  rt random 360
end


; Connects the agent to locals when distance is < 6
to connect_to_locals
  let turt_self self
  let turt_local self
  if (people_nearby != nobody) [
    ask people_nearby [
      if distance turt_self >= 6 [
        if local-with turt_self != nobody [
          ask local-with turt_self[die]
        ]
        set turt_local self
        let turt_local_set nobody
        if people_nearby != nobody [
          ask people_nearby [if self != turt_self [set turt_local_set (turtle-set turt_local_set self)]]
          set people_nearby turt_local_set
        ]
      ]
    ]
  ]
  let people_near_agent nobody
  if turt_local != turt_self [
    ask people_nearby [
      if self != turt_local [set people_near_agent (turtle-set people_near_agent self)]
    ]
    set people_nearby people_near_agent
  ]
  let friendships friend-neighbors
  let near_turtles other turtles with [distance turt_self < 6 and not member? self friendships]
  ask near_turtles [create-local-with turt_self]
  set people_nearby (turtle-set people_nearby near_turtles)
end


; Calculates social pressure based on smoking attitude of friends
to calculate_social_pressure
  let my_friends friend-neighbors
  let turt self
  let avg_smoking_attitude 0
  let connections count my_friends
  ifelse connections > 0 [
    ask my_friends [
      let connectivity 0
      ask friend-with turt [set connectivity connection_strength]
      set avg_smoking_attitude (avg_smoking_attitude + (smoking_attitude * connectivity))
    ]
    set avg_peer_attitude (avg_smoking_attitude / connections)
    if avg_peer_attitude > 1 [set avg_peer_attitude 1]
  ]
    [set avg_peer_attitude 0]
end


; Calculates social pressure based on smoking attitude of locals
to calculate_local_pressure
  let my_locals local-neighbors
  let turt self
  let avg_smoking_attitude 0
  let connections count my_locals
  ifelse connections > 0 [
    ask my_locals [
      set avg_smoking_attitude (avg_smoking_attitude + smoking_attitude)
    ]
    set avg_local_attitude (avg_smoking_attitude / connections)
  ]
    [set avg_local_attitude 0]
end


; Calculates values for adaptive and maladaptive response
to protection_motivation
  calculate_threat_appraisal
  calculate_coping_appraisal
end


; Calculates values for maladaptive response
to calculate_threat_appraisal
  calculate_intrinsic_rewards
  calculate_extrinsic_rewards
  init_severity
  init_vulnerability
  set threat_appraisal ((intrinsic_rewards + extrinsic_rewards) / 2) - ((severity + vulnerability) / 2)
end


; Calculates values for adaptive response
to calculate_coping_appraisal
  calculate_response_efficacy
  calculate_self_efficacy
  calculate_response_cost
  set coping_appraisal ((response_efficacy + self_efficacy) / 2) - response_cost
end


; Sets the intrinsic rewards (agents smoking attitude)
to calculate_intrinsic_rewards
  set intrinsic_rewards smoking_attitude
end


; Sets the extrinsic rewards (social pressure)
to calculate_extrinsic_rewards
  set extrinsic_rewards avg_perceived_attitude
end


; Initiates severity componentn based on conformity desire, security desire and objective severity
to init_severity
  set severity (objective_severity + ((1 - conformity_desire) * (security_desire - objective_severity)))
end


; Initiates vulnerability based on health status and security desire
to init_vulnerability
  set vulnerability 1 - (health_status + (security_desire - health_status) * health_status)
end


; Calculates the response efficacy based on smoking attitude, vulnerability and severity
to calculate_response_efficacy
  set response_efficacy smoking_attitude * ((vulnerability + severity) / 2)
end


; Calculates self efficacy (case: smoking attitude and social pressure both high or low --> higher self efficacy than if only one high)
to calculate_self_efficacy
  (ifelse
    ((smoking_attitude > 0.5) and (avg_perceived_attitude > 0.5)) or ((smoking_attitude <= 0.5) and (avg_perceived_attitude <= 0.5)) [
     set self_efficacy 1 - ((smoking_attitude + avg_perceived_attitude) / 2)
    ]
    (smoking_attitude > 0.5) and (avg_perceived_attitude <= 0.5) [
     set self_efficacy 1 - smoking_attitude
    ]
    (smoking_attitude <= 0.5) and (avg_perceived_attitude > 0.5) [
     set self_efficacy 1 - avg_perceived_attitude
    ]
  )
end


; Calculates response cost based on smoking attitude and a by conformity desire discounted social pressure
to calculate_response_cost
  set response_cost (smoking_attitude + (conformity_desire * avg_perceived_attitude)) / 2
end



; Adjusts desires of an agent towards the group mean
to update_desires
  set count_adjust count_adjust + 1
  let group_mean_hedonism 0
  let group_mean_security 0
  let group_mean_smoking_attitude 0
  let friend_turtles friend-neighbors
  let local_turtles local-neighbors
  ask friend_turtles [
    set group_mean_hedonism group_mean_hedonism + hedonism_desire
    set group_mean_security group_mean_security + security_desire
    set group_mean_smoking_attitude group_mean_smoking_attitude + smoking_attitude
  ]
  ask local_turtles [
    set group_mean_hedonism group_mean_hedonism + hedonism_desire
    set group_mean_security group_mean_security + security_desire
    set group_mean_smoking_attitude group_mean_smoking_attitude + smoking_attitude
  ]

  ; assumption: desires are more resistent to change than attitudes, thus for desires we take change_rate^2
  ifelse (count friend_turtles + count local_turtles) > 0 [
    set group_mean_hedonism group_mean_hedonism / (count friend_turtles + count local_turtles)
    set group_mean_security group_mean_security / (count friend_turtles + count local_turtles)
    set group_mean_smoking_attitude group_mean_smoking_attitude / (count friend_turtles + count local_turtles)
    set hedonism_desire hedonism_desire + (change_rate * (group_mean_hedonism - hedonism_desire))
    set security_desire security_desire + (change_rate * (group_mean_security - security_desire))
    set smoking_attitude smoking_attitude + (change_rate * (group_mean_smoking_attitude - smoking_attitude))
  ]
  [
    set hedonism_desire hedonism_desire
    set security_desire security_desire
    set smoking_attitude smoking_attitude
  ]
end


; Lets an agent change environment by cutting ties with friends or locals with a too different smoking attitude
to change_environment
  set count_change_environment count_change_environment + 1
  let friend_turtles friend-neighbors
  ifelse count friend_turtles > 0 [
    let turt self
    let smoking_attitude_of_turt smoking_attitude
    let local_turtles local-neighbors
    let move_away false
    ask local_turtles [
      if abs(smoking_attitude_of_turt - smoking_attitude) > contact_termination [
        set move_away true
      ]
    ]
    if move_away = true [
      ask my-locals [die]
      fd 12
      rt random 360
    ]
    let friends_deleted nobody
    ask friend_turtles [
      if abs(smoking_attitude_of_turt - smoking_attitude) > contact_termination [
        ask friend-with turt [die]
        set friends_deleted (turtle-set friends_deleted self)
      ]
    ]
    if friends_deleted != nobody [
      let other_friends other turtles with [not member? self friends_deleted]
      let number_of_deleted_friends count friends_deleted
      let number_of_other_friends count other_friends
      ifelse number_of_other_friends >= number_of_deleted_friends [
        ask n-of number_of_deleted_friends other_friends [
          create-friend-with turt
          ask friend-with turt [init_friend]
        ]
      ]
      [
        ask n-of number_of_other_friends other_friends [
          create-friend-with turt
          ask friend-with turt [init_friend]
        ]
      ]
        let current_locals local-neighbors
        ask friend-neighbors [
          if member? self current_locals [ask local-with turt [die]]
        ]
      ]
    ]
    [
    create-friends-with n-of number_of_friends other turtles
    ask my-friends [
      init_friend
    ]
  ]
  set count_change_environment (count_change_environment + 1)
end


; Calculates the average smoking attitude of all turtles
to calculate_average_smoking_attitude
  let sum_of_attitudes 0
  let num_turtles count turtles
  ask turtles [
    set sum_of_attitudes (sum_of_attitudes + smoking_attitude)
  ]
  set average_smoking_attitude (sum_of_attitudes / num_turtles)
end


; Calculates the average security desire of all turtles
to calculate_average_security_desire
  let security_sum 0
  let num_turtles count turtles
  ask turtles [
    set security_sum (security_sum + security_desire)
  ]
  set average_security_desire (security_sum / num_turtles)
end


; Calculates the average conformity desire of all turtles
to calculate_average_conformity_desire
  let conformity_sum 0
  let num_turtles count turtles
  ask turtles [
    set conformity_sum (conformity_sum + conformity_desire)
  ]
  set average_conformity_desire (conformity_sum / num_turtles)
end


; Calculates the average hedonism desire of all turtles
to calculate_average_hedonism_desire
  let hedonism_sum 0
  let num_turtles count turtles
  ask turtles [
    set hedonism_sum (hedonism_sum + hedonism_desire)
  ]
  set average_hedonism_desire (hedonism_sum / num_turtles)
end


; Highlights one agent in the interface
to watch-one
  if watch? = true [
    watch turtle 0
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
549
10
1059
521
-1
-1
15.212121212121213
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
549
565
612
598
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
728
568
791
601
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
66
180
99
conformity_people
conformity_people
0
1
0.23
0.01
1
NIL
HORIZONTAL

TEXTBOX
23
24
299
54
Step 1:\nconfigure the model with the following parameters
12
0.0
1

SLIDER
180
66
352
99
hedonism_people
hedonism_people
0
1
0.64
0.01
1
NIL
HORIZONTAL

SLIDER
361
66
533
99
security_people
security_people
0
1
0.18
0.01
1
NIL
HORIZONTAL

TEXTBOX
551
531
701
559
Step 2:\nSetup the model
11
0.0
1

TEXTBOX
1066
10
1216
47
Results
30
0.0
1

SWITCH
40
425
143
458
watch?
watch?
1
1
-1000

SWITCH
217
423
320
456
label?
label?
1
1
-1000

PLOT
1065
79
1265
229
Smoking Distribution
Ticks
Turtles
0.0
15.0
0.0
50.0
true
true
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot count turtles with [color = red]"
"pen-1" 1.0 0 -10899396 true "" "plot count turtles with [color = green]"
"pen-2" 1.0 0 -817084 true "" "plot count turtles with [color = orange]"

PLOT
1266
79
1466
229
Individual smoking attitude
Ticks
Smoking attitude
0.0
15.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "if watch? = true [plot [smoking_attitude] of turtle 0]"

SLIDER
5
181
175
214
contact_termination
contact_termination
0
1
0.4
0.1
1
NIL
HORIZONTAL

SLIDER
6
319
178
352
number_of_friends
number_of_friends
0
8
2.0
1
1
NIL
HORIZONTAL

SWITCH
398
422
501
455
static?
static?
1
1
-1000

SLIDER
364
181
537
214
moving_distance
moving_distance
0
5
2.0
1
1
NIL
HORIZONTAL

TEXTBOX
43
462
159
489
Switch on to watch an individual
9
0.0
1

TEXTBOX
220
459
326
496
Show agent ID and connection strength as label
9
0.0
1

TEXTBOX
401
457
513
500
Switch on to watch model behavior with dynamic social pressure
9
0.0
1

TEXTBOX
8
215
182
259
Threshold indicating allowed difference between smoking attitude and social pressure towards agent before changing environment
9
0.0
1

TEXTBOX
11
354
173
376
Number of social connections for each agent  
9
0.0
1

TEXTBOX
366
216
542
238
Agent moves forward for a provided number of units
9
0.0
1

TEXTBOX
192
355
342
377
Indicates how strong the rate of change of the smoke attitude is
9
0.0
1

PLOT
1065
228
1265
378
Mean Smoking Attitude
Ticks
Smoking attitude
0.0
15.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean([smoking_attitude] of turtles)"

PLOT
1266
228
1466
378
Cognitive Dissonance
Ticks
Strategy Usage
0.0
15.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -13345367 true "" "plot count_change_environment"
"pen-2" 1.0 0 -10899396 true "" "plot count_adjust"

SLIDER
189
320
361
353
change_rate
change_rate
0
0.1
0.01
0.01
1
NIL
HORIZONTAL

PLOT
1065
377
1265
527
Mean Desires
Ticks
Needs
0.0
1.0
0.0
1.0
true
false
"" ""
PENS
"pen-1" 1.0 0 -13345367 true "" "plot average_security_desire"
"pen-2" 1.0 0 -13840069 true "" "plot average_conformity_desire"
"pen-3" 1.0 0 -2674135 true "" "plot average_hedonism_desire"

SLIDER
370
320
543
353
ticks_to_run
ticks_to_run
0
180
145.0
1
1
NIL
HORIZONTAL

BUTTON
831
568
894
601
start
start
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
729
532
879
562
Step 3:\nExecute the model
12
0.0
1

TEXTBOX
834
602
984
620
Run \"ticks_to_run\" times
9
0.0
1

TEXTBOX
1068
46
1218
76
Step 4:\nanalyze the resulting data
12
0.0
1

TEXTBOX
375
354
525
376
Amount of ticks the model will run when \"start\" is pressed
9
0.0
1

TEXTBOX
365
103
515
125
Amount of agents with archetype \"security\"
9
0.0
1

TEXTBOX
183
101
333
123
Amount of agents with archetype \"hedonism\"
9
0.0
1

TEXTBOX
11
103
161
125
Amount of agents with archetype \"conformity\"
9
0.0
1

TEXTBOX
732
602
882
620
Execute one tick
9
0.0
1

SLIDER
187
179
359
212
min_stay
min_stay
0
20
3.0
1
1
days
HORIZONTAL

TEXTBOX
194
218
344
251
minimum number of days agents must stay in their environment before being able to change 
9
0.0
1

SLIDER
360
24
532
57
n_people
n_people
0
100
100.0
1
1
people
HORIZONTAL

INPUTBOX
405
520
510
580
seed
8597968.0
1
0
Number

SLIDER
13
513
185
546
pressure_modifier
pressure_modifier
0
1.5
1.25
0.01
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?
This model implements a dynamic social pressure component integrated in the Protection Motivation Theory (PMT) schema. Furthermore it visualy compares two model variants: one as a dynamic version where social pressure is calculated as the mean attitude of an agents friends and local persons he's surrounded with and one where social pressure is constant and regulated through the interface.

## HOW IT WORKS
Each agent calculates its threat and coping appraisal through various components of the PMT (response costs, rewards, response-efficacy, etc.) to then decide whether they act according to the pressure from their environment, decide to cut ties with certain friends or locals, or just do nothing when the pressure is not strong enough.

## HOW TO USE IT
Setup the model parameters in the frontend and press "go" to observe the model behavior tick by tick. Press "start" to observe model behavior for a pre-defined number of ticks (ticks_to_run). Press "go" to run the model for one tick. 
"conformity_people", "hedonism_people" and "security_people": define the number of agents from each archetype
"contact_termination": allowed difference between agents smoking attitude and social pressure before agent tries to change environment
"static_social_pressure": social pressure value for the static model variant
"moving_distance": distance each agent travels per tick
"number_of_friends": minimal number of friends each agent has after intialization 
"change_rate": defines how fast agent desires change
"watch?": if on highlights an individual agent
"label?": if on shows connection strength on links
"static?": if on model runs in static social pressure mode 

## THINGS TO NOTICE
The result plots show how relevant system measures change over time or with different model parameter settings.

## THINGS TO TRY
Although model parameters have been calibrated with Netlogo's Behaviorsearch, it can still be interesting to modify number of agents or other parameters to observe how that influences model behavior.

## NETLOGO FEATURES
This model makes use of the Netlogo turtles and link breeds.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pencil
false
0
Polygon -7500403 true true 255 60 255 90 105 240 90 225
Polygon -7500403 true true 60 195 75 210 240 45 210 45
Polygon -7500403 true true 90 195 105 210 255 60 240 45
Polygon -6459832 true false 90 195 60 195 45 255 105 240 105 210
Polygon -16777216 true false 45 255 74 248 75 240 60 225 51 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="dynamic" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt;= ticks_to_run</exitCondition>
    <metric>mean([smoking_attitude] of turtles)</metric>
    <enumeratedValueSet variable="security_people">
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="watch?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="static?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_rate">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="conformity_people">
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact_termination">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hedonism_people">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks_to_run">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moving_distance">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="label?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="static_social_pressure">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number_of_friends">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="static" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt;= ticks_to_run</exitCondition>
    <metric>mean([smoking_attitude] of turtles)</metric>
    <enumeratedValueSet variable="security_people">
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="watch?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="static?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_rate">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="conformity_people">
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact_termination">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hedonism_people">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks_to_run">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moving_distance">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="label?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="static_social_pressure">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number_of_friends">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_static_dynamic" repetitions="50" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt;= ticks_to_run</exitCondition>
    <metric>mean([smoking_attitude] of turtles)</metric>
    <metric>count_change_environment</metric>
    <metric>count_do_nothing</metric>
    <metric>count_adjust</metric>
    <metric>average_security_need</metric>
    <metric>average_conformity_need</metric>
    <metric>average_hedonism_need</metric>
    <enumeratedValueSet variable="security_people">
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="watch?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="static?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="change_rate">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="conformity_people">
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="contact_termination">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hedonism_people">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks_to_run">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="label?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moving_distance">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="static_social_pressure">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number_of_friends">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
