globals [          ; variables for spin groups a (blue) and spin group b (pink)
  spin_diff_a      ; difference between spin ups(e two's) and spin downs(e one's) in spin group a
  spin_diff_b      ; difference between spin ups and spin downs in spin group b
  n_two_a          ; excited state for spin group a (dark blue)
  n_two_b          ; excited state for spin group b (dark pink)
  n_one_a          ; ground state for spin group a (light blue)
  n_one_b          ; ground state for spin group b (light pink)
  total_energy_a   ; spin group a's total energy
  total_energy_b   ; spin group b's total energy
  e_one_a          ; spin downs for spin group a (blue) when negative
  e_one_b          ; spin downs for spin group b (pink) when negative
  e_two_a          ; spin ups for spin group a (blue) when positive
  e_two_b          ; spin ups for spin group b (pink) when positve
  n_total_a        ; total number of spins in group a
  n_total_b        ; total number of spins in group b
  entropy_a
  entropy_b
  temperature2_a
  temperature2_b
  free_energy_a
  free_energy_b
  delta_e_b
  spin_group_a     ; left side of the world in blue
  spin_group_b     ; right side of the world in pink
  temp_hold_a      ; compares previous step of spin group a and stops temperatures from crossing and flipping
  temp_hold_b      ; compares previous step of spin group b and stops temperatures from crossing and flipping
  spin_flip_a      ; keeps track on whether or not more than 50% of a spin group is exctied
  spin_flip_b
  spin_flip_hold_a ; compares previous step of spin group a and stops temperatures from crossing and flipping
  spin_flip_hold_b ; compares previous step of spin group b and stops temperatures from crossing and flipping
  ;
]

patches-own [spin        ; holds 1 or 2
             orientation ; -1 or 1 for spin 1 or 2 respectively
             spin_group  ; spin group a = 1 is blue and spin group b = 2 is pink
             ]

to setup
   ca                                                         ; resets all global variables to zero
  set e_one_a 100                                             ; ground state energy for spin group a (blue)
  set e_one_b 100                                             ; ground state energy for spin group b (pink)
  set delta_e_b delta_e_a                                     ; set the same enregy difference for both spin groups
  set e_two_a (e_one_a + delta_e_a)                           ; excited state energy for spin group a
  set e_two_b (e_one_b + delta_e_b)                           ; excited state energy for spin group b
  ask patches [set spin 1                                     ; sets all to ground state (spin down)
               set orientation -1]                            ; sets all to ground state (spin down)
  if dense-setup = true and world-division = true [setup1]
  if dense-setup = false and world-division = false [setup2]
  if dense-setup = true and world-division = false [setup3]
  if dense-setup = false and world-division = true [setup4]
    reset-ticks                                               ; resets ticks to zero and resets plots
end

to setup1                                                                                    ; when both switches are on nuclei are contained and are distrubuted                                                                                               ; and world-division = true
  if world-division = true [                                                                 ; into rectangular blocks based on the amount of percent excited set
    set n_two_a ceiling ((percent_excited_a / 100) * (0.75 * world-width * world-height))    ; sets spin group a larger
    set n_two_b ceiling ((percent_excited_b / 100) * (0.25 * world-width * world-height))]   ; sets spin group b smaller
      set n_one_a floor ((world-width * world-height) - n_two_a)
      set n_one_b floor ((world-width * world-height) - n_two_b)
      set n_total_a floor (n_one_a + n_two_a)
      set n_total_b floor (n_one_b + n_two_b)
  ask patches [set spin_group 1
               set pcolor blue]
  ask patches [if pxcor >= world-width / 2 and pycor <= (world-height / 2) [set spin_group 2
                                                                            set pcolor pink]
    ]

 if dense-setup = true [
    set n_two_a ceiling ((percent_excited_a / 100) * (0.75 * world-width * world-height))
    set n_two_b ceiling ((percent_excited_b / 100) * (0.25 * world-width * world-height))

    let bw (world-width / 4)                                       ; sets block width where world is divided into 4 x 4 rectangles
    let bh (world-height / 4)                                      ; sets block height where world is divided into 4 x 4 rectangles
    let block_area floor ((world-width / 4) * (world-height / 4))  ; area of one rectanlge block
    let number_b_blocks ceiling (n_two_b / block_area)             ; determines number of blocks for spin group b
    let number_a_blocks ceiling (n_two_a / block_area)             ; determines number of blocks for spin group a

    ; sets rectangular sections for spin group a
    if number_a_blocks = 1 [ask n-of n_two_a patches with [pxcor < (bw) and pycor <= (bh)] [set spin 2 set orientation 1]]
    if number_a_blocks = 2 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh)) or (pxcor < (bw) and pycor <= ((bh * 2) ))]
      [set spin 2 set orientation 1]]
    if number_a_blocks = 3 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh)) or (pxcor < (bw) and pycor <= ((bh * 2))) or
        (pxcor < (bw) and pycor <= ((bh * 3)))] [set spin 2 set orientation 1]]
    if number_a_blocks = 4 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh)) or (pxcor < (bw) and pycor <= ((bh * 2))) or
        (pxcor < (bw) and pycor <= ((bh * 3))) or (pxcor < (bw) and pycor <= ((bh * 4)))] [set spin 2 set orientation 1]]
    if number_a_blocks = 5 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (2 * bw) and pycor > (3 * bh))]
        [set spin 2 set orientation 1]]
    if number_a_blocks = 6 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (3 * bw) and pycor >= (3 * bh))]
        [set spin 2 set orientation 1]]
    if number_a_blocks = 7 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (4 * bw) and pycor >= (3 * bh))]
        [set spin 2 set orientation 1]]
    if number_a_blocks = 8 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (4 * bw) and pycor >= (3 * bh)) or
        (pxcor < (bw * 2) and pycor <= (bh))] [set spin 2 set orientation 1]]
    if number_a_blocks = 9 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (4 * bw) and pycor >= (3 * bh)) or
        (pxcor < (bw * 2) and pycor <= (2 * bh))] [set spin 2 set orientation 1]]
    if number_a_blocks = 10 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (4 * bw) and pycor >= (3 * bh)) or
        (pxcor < (bw * 2) and pycor <= (3 * bh))] [set spin 2 set orientation 1]]
    if number_a_blocks = 11 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (4 * bw) and pycor >= (3 * bh)) or
        (pxcor < (bw * 2) and pycor < (3 * bh)) or (pxcor < (3 * bw) and pycor >= (2 * bh))] [set spin 2 set orientation 1]]
    if number_a_blocks = 12 [ask n-of n_two_a patches with [(pxcor < (bw) and pycor <= (bh * 4)) or (pxcor < (4 * bw) and pycor >= (3 * bh)) or
        (pxcor < (bw * 2) and pycor <= (3 * bh)) or (pxcor < (4 * bw) and pycor > (2 * bh))] [set spin 2 set orientation 1]]

    ; sets rectangular sections for spin group b
    if number_b_blocks = 1 [ask n-of n_two_b patches with [pxcor > ((3 * bw) - 1) and pycor <= (bh - 1)] [set spin 2 set orientation 1]]
    if number_b_blocks = 2 [ask n-of n_two_b patches with [(pxcor > ((3 * bw) - 1) and pycor <= (bh - 1)) or
        (pxcor > ((2 * bw) - 1) and pycor <= (bh - 1))] [set spin 2 set orientation 1]]
    if number_b_blocks = 3 [ask n-of n_two_b patches with [(pxcor > ((3 * bw) - 1) and pycor <= (bh - 1)) or
        (pxcor > ((2 * bw) - 1) and pycor <= (bh - 1)) or (pxcor > ((3 * bw) - 1) and pycor <= (2 * bh))] [set spin 2 set orientation 1]]
    if number_b_blocks = 4 [ask n-of n_two_b patches with [(pxcor > ((3 * bw) - 1) and pycor <= (bh - 1)) or
        (pxcor > ((2 * bw) - 1) and pycor <= (bh - 1)) or (pxcor > ((3 * bw) - 1) and pycor <= (2 * bh)) or (pxcor >= ((2 * bw)) and pycor <= (2 * bh))]
        [set spin 2 set orientation 1]]

      set spin_diff_a count patches with [spin = 2 and spin_group = 1] - count patches with [spin = 1 and spin_group = 1]
      set spin_diff_b count patches with [spin = 2 and spin_group = 2] - count patches with [spin = 1 and spin_group = 2]
        set temperature2_a  temp2_a
        set temperature2_b  temp2_b
        set entropy_a       S_a
        set entropy_b       S_b
        set total_energy_a  U_a
        set total_energy_b  U_b
        set free_energy_a   F_a
        set free_energy_b   F_b
        set temp_hold_a     temp3_a
        set temp_hold_b     temp3_b

  ; if most of neighbor patches are spin up (greater than 0) then set spint_flip 1 and if not se spin_flip 0
  ifelse (spin_diff_a > 0) [set spin_flip_a 1] [ifelse (spin_diff_a < 0) [set spin_flip_a -1][set spin_flip_a 0]
        ]

  ifelse (spin_diff_b > 0) [set spin_flip_b 1] [ifelse (spin_diff_b < 0) [set spin_flip_b -1][set spin_flip_b 0]
        ]
      ask patches [recolor]
  ]
end

to setup2                                                                                ; when both switches are fasle world is
  if world-division = false [                                                            ; cut in half and nuclei are spread out
    set n_two_a floor ((percent_excited_a / 100) * (0.5 * world-width * world-height))
    set n_two_b floor ((percent_excited_b / 100) * (0.5 * world-width * world-height))]
      set n_one_a floor ((world-width * world-height) - n_two_a)
      set n_one_b floor ((world-width * world-height) - n_two_b)
      set n_total_a floor (n_one_a + n_two_a)
      set n_total_b floor (n_one_b + n_two_b)
 ask patches [set spin_group 1
              set pcolor blue]
 ask patches [if pxcor > (world-width / 2 - 1) [set pcolor pink
                                                set spin_group 2]
  ]
  if dense-setup = false [
    ask n-of n_two_a patches with [spin_group = 1] [set spin 2                         ; excited states in spin group a (dark blue)
                                                    set orientation 1]
    ask n-of n_two_b patches with [spin_group = 2] [set spin 2                         ; excited states in spin group b (dark pink)
                                                    set orientation 1]
    ask patches [recolor]                                                              ; sets color of the spins that determine excited or ground states
      set spin_diff_a count patches with [spin = 2 and spin_group = 1] - count patches with [spin = 1 and spin_group = 1]
      set spin_diff_b count patches with [spin = 2 and spin_group = 2] - count patches with [spin = 1 and spin_group = 2]
        set temperature2_a  temp2_a
        set temperature2_b  temp2_b
        set entropy_a       S_a
        set entropy_b       S_b
        set total_energy_a  U_a
        set total_energy_b  U_b
        set free_energy_a   F_a
        set free_energy_b   F_b
        set temp_hold_a     temp3_a
        set temp_hold_b     temp3_b
  ifelse (spin_diff_a > 0) [set spin_flip_a 1] [ifelse (spin_diff_a < 0) [set spin_flip_a -1][set spin_flip_a 0]
        ]
  ifelse (spin_diff_b > 0) [set spin_flip_b 1] [ifelse (spin_diff_b < 0) [set spin_flip_b -1][set spin_flip_b 0]
    ]
  ]
end

to setup3                                                                                 ; when world division switch is
  if world-division = false [                                                             ; false the world is cut in half
    set n_two_a floor ((percent_excited_a / 100) * (0.5 * world-width * world-height))
      set n_two_b floor ((percent_excited_b / 100) * (0.5 * world-width * world-height))]
      set n_one_a floor ((world-width * world-height) - n_two_a)
      set n_one_b floor ((world-width * world-height) - n_two_b)
      set n_total_a floor (n_one_a + n_two_a)
      set n_total_b floor (n_one_b + n_two_b)
 ask patches [set spin_group 1
              set pcolor blue]
 ask patches [if pxcor > (world-width / 2 - 1) [set pcolor pink
                                                set spin_group 2]
  ]
  if dense-setup = true [
    let x_excited_a floor ((percent_excited_a / 100) * (world-width / 2))
    let x_excited_b ceiling (world-width - (percent_excited_b / 100) * (world-width / 2))
  ask n-of n_two_a patches with [pxcor <= x_excited_a] [set spin 2
                                                        set orientation 1]
  ask n-of n_two_b patches with [ pxcor >= x_excited_b - 1 ][set spin 2
                                                             set orientation 1]
      set spin_diff_a count patches with [spin = 2 and spin_group = 1] - count patches with [spin = 1 and spin_group = 1]
      set spin_diff_b count patches with [spin = 2 and spin_group = 2] - count patches with [spin = 1 and spin_group = 2]
        set temperature2_a  temp2_a
        set temperature2_b  temp2_b
        set entropy_a       S_a
        set entropy_b       S_b
        set total_energy_a  U_a
        set total_energy_b  U_b
        set free_energy_a   F_a
        set free_energy_b   F_b
        set temp_hold_a     temp3_a
        set temp_hold_b     temp3_b
  ifelse (spin_diff_a > 0) [set spin_flip_a 1] [ifelse (spin_diff_a < 0) [set spin_flip_a -1][set spin_flip_a 0]
        ]
  ifelse (spin_diff_b > 0) [set spin_flip_b 1] [ifelse (spin_diff_b < 0) [set spin_flip_b -1][set spin_flip_b 0]
        ]
      ask patches [recolor]
  ]
end

to setup4
  if world-division = true [
    set n_two_a floor ((percent_excited_a / 100) * (0.75 * world-width * world-height))
    set n_two_b floor ((percent_excited_b / 100) * (0.25 * world-width * world-height))]
      set n_one_a floor ((world-width * world-height) - n_two_a)
      set n_one_b floor ((world-width * world-height) - n_two_b)
      set n_total_a floor (n_one_a + n_two_a)
      set n_total_b floor (n_one_b + n_two_b)
  ask patches [set spin_group 1
               set pcolor blue]
  ask patches [if (pxcor >= world-width / 2) and (pycor <= world-height / 2) [set spin_group 2
                                                                              set pcolor pink]
  ]
  if dense-setup = false [
   ask n-of n_two_a patches with [spin_group = 1] [set spin 2
                                                   set orientation 1]
   ask n-of n_two_b patches with [spin_group = 2] [set spin 2
                                                   set orientation 1]
     ask patches [recolor]
       set spin_diff_a count patches with [spin = 2 and spin_group = 1] - count patches with [spin = 1 and spin_group = 1]
       set spin_diff_b count patches with [spin = 2 and spin_group = 2] - count patches with [spin = 1 and spin_group = 2]
         set temperature2_a  temp2_a
         set temperature2_b  temp2_b
         set entropy_a       S_a
         set entropy_b       S_b
         set total_energy_a  U_a
         set total_energy_b  U_b
         set free_energy_a   F_a
         set free_energy_b   F_b
         set temp_hold_a     temp3_a
         set temp_hold_b     temp3_b
    ifelse (spin_diff_a > 0) [set spin_flip_a 1] [ifelse (spin_diff_a < 0) [set spin_flip_a -1][set spin_flip_a 0]
        ]
    ifelse (spin_diff_b > 0) [set spin_flip_b 1] [ifelse (spin_diff_b < 0) [set spin_flip_b -1][set spin_flip_b 0]
    ]
  ]
end

to recolor                ; colors patches according to spin
  ifelse spin_group = 1 [
  ifelse spin = 1
    [set pcolor blue + 2] ; excited state for spin group a
    [set pcolor blue - 2] ; ground state for spin group a
  ]
  [ifelse spin = 1
    [set pcolor pink + 2] ; excited state for spin group b
    [set pcolor pink - 2] ; ground state for spin group b
  ]
end

to go
  repeat 1000 [                                 ; updates 1000 patches at a time
    ask one-of patches [update]
  ]
 set spin_diff_a count patches with [spin = 2 and spin_group = 1] - count patches with [spin = 1 and spin_group = 1]
 set spin_diff_b count patches with [spin = 2 and spin_group = 2] - count patches with [spin = 1 and spin_group = 2]
   tick                                         ; updating 1000 patches at a time only
 update-plots                                   ; must update plots separately
   if floor temp3_a = floor temp3_b [stop]      ; rounding values to
   if floor temp3_a = ceiling temp3_b [stop]    ; show that equilibrium
   if ceiling temp3_a = floor temp3_b [stop]    ; has been reached
   if ticks > 50000 [stop]                      ; if equilibrium has not been reached yet
 ifelse (spin_diff_a > 0) [set spin_flip_a 1 ] [ifelse (spin_diff_a < 0) [set spin_flip_a -1][set spin_flip_a 0]
        ]
 ifelse (spin_diff_b > 0) [set spin_flip_b 1] [ifelse (spin_diff_b < 0) [set spin_flip_b -1][set spin_flip_b 0]
        ]
  if (spin_flip_hold_a = spin_flip_a and spin_flip_hold_b = spin_flip_b) [ ; when temp3_a and temp3_b cross
  if (temp3_a < temp3_b and temp_hold_a > temp_hold_b) [stop]
  if (temp3_a > temp3_b and temp_hold_a < temp_hold_b) [stop]
  ]
   set temp_hold_a      temp3_a
   set temp_hold_b      temp3_b
   set spin_flip_hold_a spin_flip_a
   set spin_flip_hold_b spin_flip_b
end

to update                                                           ; updating spins of one patch
  let Ediff (2 * orientation * sum [orientation] of neighbors)
  if (Ediff <= 0) [
    let hold_orientation orientation
      ask n-of 1 neighbors with [orientation != hold_orientation] [ ; if neighboring spins are different then changes to ground state
      set orientation (- orientation)
  ifelse orientation = -1 [set spin 1] [set spin 2]                 ; if true then ground state and if not assign to excited state
      recolor
      ]
      set orientation (- orientation)
      ifelse orientation = -1 [set spin 1] [set spin 2]
      recolor
      ]
end

to-report S_a ; spin group a initial entropy
  report (1) * (n_total_a * ln (n_total_a) - n_two_a * ln (n_two_a) - n_one_a * ln (n_one_a))
end
to-report S_b ; spin group b initial entropy
  report (1) * (n_total_b * ln (n_total_b) - n_two_b * ln (n_two_b) - n_one_b * ln (n_one_b))
end

to-report entropy3_a ; updates monitor
  report (1) * (n_total_a * ln (n_total_a) - count patches with [spin = 2 and spin_group = 1] * ln (count patches with [spin = 2 and spin_group = 1]) -
    count patches with [spin = 1 and spin_group = 1] * ln (count patches with [spin = 1 and spin_group = 1]))
end
to-report entropy3_b ; updates monitor
  report (1) * (n_total_b * ln (n_total_b) - count patches with [spin = 2 and spin_group = 2] * ln (count patches with [spin = 2 and spin_group = 2]) -
    count patches with [spin = 1 and spin_group = 2] * ln (count patches with [spin = 1 and spin_group = 2]))
end
to-report total_entropy3 ; updates monitor
  report (1) * (n_total_a * ln (n_total_a) - count patches with [spin = 2 and spin_group = 1] * ln (count patches with [spin = 2 and spin_group = 1]) -
    count patches with [spin = 1 and spin_group = 1] * ln (count patches with [spin = 1 and spin_group = 1])) + (1) * (n_total_b * ln (n_total_b) -
    count patches with [spin = 2 and spin_group = 2] * ln (count patches with [spin = 2 and spin_group = 2]) - count patches with [spin = 1 and spin_group = 2] *
    ln (count patches with [spin = 1 and spin_group = 2]))
end

to-report U_a ; initial total_energy
  report ((n_two_a * e_two_a) + (n_one_a * e_one_a))
end
to-report U_b ; initial total_energy
  report ((n_two_b * e_two_b) + (n_one_b * e_one_b))
end

to-report total_energy3_a ; updates monitor
  report ((count patches with [spin = 2 and spin_group = 1] * e_two_a) + (count patches with [spin = 2 and spin_group = 2] * e_one_a))
end
to-report total_energy3_b ; updates monitor
  report ((count patches with [spin = 2 and spin_group = 2] * e_two_b) + (count patches with [spin = 1 and spin_group = 2] * e_one_b))
end
to-report total_total_energy3 ; updates monitor
  report ((count patches with [spin = 2 and spin_group = 1] * e_two_a) + (count patches with [spin = 2 and spin_group = 2] * e_one_a)) +
  ((count patches with [spin = 2 and spin_group = 2] * e_two_b) + (count patches with [spin = 1 and spin_group = 2] * e_one_b))
end

to-report temp2_a ; initial temperature
  ifelse (n_two_a / n_one_a) = 1 [report 0] [
    report ( -1 * (e_two_a - e_one_a) / (ln (n_two_a / n_one_a)))]
end
to-report temp2_b ; initial temperature
   ifelse (n_two_b / n_one_b) = 1 [report 0] [
     report ( -1 * (e_two_b - e_one_b) / (ln (n_two_b / n_one_b)))]
end

to-report temp3_a ; updates monitor
   ifelse (count patches with [spin = 2 and spin_group = 1] / count patches with [spin = 1 and spin_group = 1]) = 1 [report 0] [
     report ( -1 * (e_two_a - e_one_a) / (ln (count patches with [spin = 2 and spin_group = 1] / count patches with [spin = 1 and spin_group = 1])))]
end
to-report temp3_b ; updates monitor
   ifelse (count patches with [spin = 2 and spin_group = 2] / count patches with [spin = 1 and spin_group = 2]) = 1 [report 0] [
     report ( -1 * (e_two_b - e_one_b) / (ln (count patches with [spin = 2 and spin_group = 2] / count patches with [spin = 1 and spin_group = 2])))]
end

to-report F_a ; initial free_energy
  report (total_energy_a - (temperature2_a * entropy_a))
end
to-report F_b ; initial free_energy
  report (total_energy_b - (temperature2_b * entropy_b))
end
@#$#@#$#@
GRAPHICS-WINDOW
883
10
1248
734
-1
-1
8.94
1
10
1
1
1
0
0
0
1
0
39
0
79
0
0
1
ticks
1000.0

BUTTON
8
10
95
44
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
96
10
186
44
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
8
44
186
89
NIL
temp3_a
17
1
11

MONITOR
9
90
187
135
NIL
temp3_b
17
1
11

MONITOR
9
136
187
181
NIL
entropy3_a
17
1
11

MONITOR
9
182
187
227
NIL
entropy3_b
17
1
11

MONITOR
9
274
187
319
NIL
total_energy3_a
17
1
11

MONITOR
9
320
187
365
NIL
total_energy3_b
17
1
11

SLIDER
9
412
187
445
percent_excited_a
percent_excited_a
1
99
4.0
1
1
%
HORIZONTAL

SLIDER
9
446
187
479
percent_excited_b
percent_excited_b
1
99
99.0
1
1
%
HORIZONTAL

SLIDER
9
480
187
513
delta_e_a
delta_e_a
0
100
94.0
1
1
NIL
HORIZONTAL

SWITCH
9
514
187
547
dense-setup
dense-setup
0
1
-1000

PLOT
689
10
881
191
temp3_a
NIL
NIL
0.0
10.0
0.0
300.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot temp3_a"

PLOT
689
192
881
372
temp3_b
NIL
NIL
0.0
10.0
0.0
50.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot temp3_b"

PLOT
188
10
348
130
spin_diff_a
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot spin_diff_a"

PLOT
348
10
508
130
spin_diff_b
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot spin_diff_b"

PLOT
508
10
689
130
total spin diff
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot spin_diff_a + spin_diff_b"

PLOT
188
131
348
251
entropy3_a
NIL
NIL
0.0
10.0
15000.0
30000.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot entropy3_a"

PLOT
348
131
508
251
entropy3_b
NIL
NIL
0.0
10.0
15000.0
30000.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot entropy3_b"

PLOT
508
131
690
251
total_entropy3
NIL
NIL
0.0
10.0
39000.0
41000.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot entropy3_a + entropy3_b"

PLOT
188
374
882
581
temp3_a and temp3_b
NIL
NIL
0.0
10.0
0.0
300.0
true
false
"" ""
PENS
"temp_a" 1.0 0 -14454117 true "" "plot temp3_a"
"temp_b" 1.0 0 -5825686 true "" "plot temp3_b"

PLOT
188
252
348
372
total_energy3_a
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot total_energy3_a"

PLOT
348
252
508
372
total_energy3_b
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot total_energy3_b"

PLOT
508
252
689
372
total_total_energy3
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot total_energy3_a + total_energy3_b"

MONITOR
9
366
187
411
NIL
total_total_energy3
17
1
11

MONITOR
9
228
187
273
NIL
total_entropy3
17
1
11

SWITCH
8
548
187
581
world-division
world-division
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

This is a model of a binary spin lattice thermodynamic system that measures properties such as temperature, entropy, total energy. The spins in the binary spin lattice system can either be up representing an excited state or can be down representing the ground state. These spins can be influenced and flipped by neighboring spins.  

## HOW IT WORKS

In this model, the patches represent nuclei that are in different energy states. These energy states can change when influenced by neighboring patches. If most of neighboring patches are spin up or greater than zero, then the patch being influenced will flip up to the excited state and if this is not the case, then the patch will flip down to the ground state.

The model stops when the temperatures of the two spin groups have reached thermal equilibrium. 

The switches will affect the outcome of thermodynamic properties, the world division switch makes one spin group large and the other spin group smaller.

The sliders will also affect the outcome of the simulation. Any percent extied slider above fifty will produce a negative temperature and any below will produce a positive temperature.

## HOW TO USE IT

SET UP: a button that resets the model and updates any changes made to the sliders and/or switches.

GO: a button that continually runs the simulation. 

MONITORS: temp3 a, temp3 b, entropy3 a, entropy3 b, total entropy3, total energy3 a, total energy3 b, and total total energy3 are monitors in the interface that are updated while the simulation is running. The final measurements on the monitors reflect when both spin groups are at equilibrium with each other. 

PERCENT EXCITED A: a slider that allows the percent of excited nuclei to vary for spin group a (blue group).

PERCENT EXCITED B: a slider that allows the percent of excited nuclei to vary for spin group b (pink group).

DENSE SETUP: a switch that splits the world in half creating spin group a and spin group b and allows the percent of excited nuclei of each spin group to be contained on either side of the world.

WORLD DIVISION: a switch that isolates spin group b in the bottom right corner which results in a larger spin group a.

PLOTS: these plots will continue to update throughout the simulation. 

## THINGS TO NOTICE

When the spin group temperatures are transitioning from a positive temperature to a negative temperature or vice versa, the temp3 a and temp3 b plot will fluctuate between the boundary of positive infinity to negative infinity. The plot is discontinuous between the positive and negative infinites during this time because the program reads these values as extremely positive or extremely negative contrary to the kelvin scale where they are near each other.

## THINGS TO TRY

Analyze how the temperatures react when both switches are on and percent excited a is much greater then percent excited b as well as when percent excited a is much less than percent excited b. Compare and contrast the final data with different switch scenarios. 

When both switches are on, try setting percent excited b two percents higher than percent excited a and determine why it only took less than five ticks to reach equilibrium. Also, vary delta e to determine if this slider increases or decreases the amount of ticks.

Next, try setting the percent excited's three values away from each other and manipulate delta e to compare the number of ticks it takes to reach equilibrium when both switches are on.

Observe the differences between having different switches on or off and how it affects the number of ticks to reach equilibrium.


## EXTENDING THE MODEL

* Creating a three-level system
* Changing the world division to occupy a larger or smaller space within the world
  thus enabling one group to act as a heat reservoir
* Modifying the world size of the spin groups
* Including the entropy of mixing 

## RELATED MODELS

Ising Model (in the Chemistry & Physics - GasLab Section of Models Library)

## CREDITS AND REFERENCES

This model was created by Jennifer Campos and Kaitlyn Moore as part of Palm Beach Atlantic University's Summer UnderGraduate Academic Research (SUGAR) Program.
 
Credit to: Dr. Fred Browning for advising this modeling project

Wilensky, U. (2003). NetLogo Ising model. http://ccl.northwestern.edu/netlogo/models/Ising. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.


###Additional resources that were used to complete this model:

Kelly, J. (1996-2002). Ensembles.
http://www.physics.umd.edu/courses/Phys603/kelly/Notes/Ensembles.pdf.
Statistical Physics using Mathmatica.

Kittel C. & Kroemer H. (1980). Thermal Physics, Second Edition. University of California, San Francisco. W. H. Freeman and Company, NY. 

Railsback, S. F., & Grimm, V. (2012). Agent-based and individual-based modeling: A practical introduction. Princeton, N.J: Princeton University Press. 

Klein, M. (1956) Thermodynamics and Statistical Mechanics at Negative Absolute Temperatures. Physical Review, 103, 20-28.

Section 2: Introduction to Statistical Mechanics. (1997). 
http://personal.rhul.ac.uk/uhap/027/ph2610/ph2610_files/sect2.pdf.

Wikipedia Contributors. (3 June 2017). Population Inversion.
https://en.wikipedia.org/wiki/Population_inversion. Wikipedia, The Free Encyclopedia.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="new code test" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>temp2_a</metric>
    <metric>temp2_b</metric>
    <metric>temp3_a</metric>
    <metric>temp3_b</metric>
    <metric>entropy3_a</metric>
    <metric>entropy3_b</metric>
    <metric>total_entropy3</metric>
    <metric>total_energy3_a</metric>
    <metric>total_energy3_b</metric>
    <metric>total_total_energy3</metric>
    <metric>ticks</metric>
    <steppedValueSet variable="percent_excited_a" first="30" step="1" last="90"/>
    <enumeratedValueSet variable="delta_e_a">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent_excited_b">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dense-setup">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="pe_a 30 pe_b 31 90 d_e 25 ds_f wd_t" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>temp2_a</metric>
    <metric>temp2_b</metric>
    <metric>temp3_a</metric>
    <metric>temp3_b</metric>
    <metric>entropy3_a</metric>
    <metric>entropy3_b</metric>
    <metric>total_entropy3</metric>
    <metric>total_energy3_a</metric>
    <metric>total_energy3_b</metric>
    <metric>total_total_energy3</metric>
    <enumeratedValueSet variable="percent_excited_a">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="delta_e_a">
      <value value="25"/>
    </enumeratedValueSet>
    <steppedValueSet variable="percent_excited_b" first="31" step="1" last="90"/>
    <enumeratedValueSet variable="world-division">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dense-setup">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
